/* -*- C++ -*-
   (c) 2008 Peter Rockai <me@mornfall.net>
   (c) 2008 Yuriy Kozlov <yuriy.kozlov@gmail.com> */

#include <KVBox>
#include <KHBox>
#include <KUrl>
#include <KLocale>
#include <KTabWidget>
#include <KDirModel>
#include <KDirLister>
#include <KFileItemDelegate>

#include <QtGui/QLabel>
#include <QtGui/QGridLayout>
#include <QtGui/QHeaderView>
#include <QtGui/QTextBrowser>
#include <QtGui/QSizePolicy>
#include <QtGui/QTreeView>
#include <QtGui/QSortFilterProxyModel>
#include <QtGui/QProgressBar>

#include <ept/core/apt.h>
#include <ept/core/apt/action.h>

#include <adept/util.h>
#include <adept/packageinfo.h>
#include <adept/tokenmodel.h>
#include <adept/packagelist.h>
#include <adept/processevents.h>

#ifndef ADEPT_PACKGEDETAILS_H
#define ADEPT_PACKGEDETAILS_H

namespace adept {

using namespace ept::core;

class UrlListProxy : public QSortFilterProxyModel {
    Q_OBJECT
    QSet< KUrl > visible;
    KDirModel *dir;
public:
    bool filterAcceptsRow( int src, const QModelIndex &parent ) const
    {
        QModelIndex idx = sourceModel()->index( src, 0, parent );
        KUrl url = dir->itemForIndex( idx ).url();
        if ( url == KUrl( "file:///" ) )
            return true;
        return visible.contains( url );
    }

    void setSourceModel( QAbstractItemModel *m ) {
        QSortFilterProxyModel::setSourceModel( m );
        dir = qobject_cast< KDirModel * >( m );
        dir->dirLister()->openUrl( KUrl( "file:///" ) );
        while ( !dir->dirLister()->isFinished() )
            ProcessEvents::process();
    }

    void addUrl( KUrl url ) {
        visible.insert( url );
        dir->expandToUrl( url );
        while ( !dir->dirLister()->isFinished() )
            ProcessEvents::process();
    }

    void clear() {
        visible.clear();
        invalidateFilter();
    }

    void apply() {
        invalidateFilter();
    }
};

class PackageDetails : public PackageInfo
{
    Q_OBJECT
private:
    QLabel *m_versionInfo, *m_pkgNameLabel, *m_shortDesc;
    QLabel *m_section, *m_installedSize, *m_maintainer, *m_sourcePackage,
         *m_architecture, *m_filename, *m_md5;
    QTextEdit *m_desc;
    QVBoxLayout *m_layout;
    KVBox *m_infoBox;
    
    KTabWidget *m_bottom;
    PackageListView *m_packageRelationshipsView;
    TokenModel *m_packageRelationshipsModel;
    QTreeView *m_tags;
    QTextEdit *m_changelog;
    QTreeView *m_fileList;
    QProgressBar *m_fileLoadingProgress;
    KDirModel *m_fileModel;
    UrlListProxy *m_fileProxy;

    bool m_needExpensiveRefresh;

public:
    PackageDetails( PackageData &, QWidget *parent = 0 );

public Q_SLOTS:
    void setPackage( ept::Token _t ) {
        t = _t;
        refresh();
    }
    
    virtual void refresh();
    
    void fillRelationshipsList() {
        /*m_packageRelationshipsModel->fillFrom(
        xap.query( m_searchLine->text().toLocal8Bit().data() ) );*/
    }

protected:
    void setupInfo() {
        QWidget *top = new KHBox( this );
        QSizePolicy pol( QSizePolicy::Preferred,
                         QSizePolicy::Preferred );
        pol.setVerticalStretch( 2 );
        top->setSizePolicy( pol );

        top->layout()->setSpacing( 5 );
        m_layout->setMargin( 5 );
        m_layout->addWidget( top );

        KVBox *infoWrap = new KVBox( top );
        m_infoBox = new KVBox( infoWrap );
        QWidget *spacer = new QWidget( infoWrap );
        QWidget *descBox = new KVBox( top );

        m_shortDesc = new QLabel( descBox );
        m_desc = new QTextBrowser( descBox );

        pol.setHorizontalStretch( 3 );
        descBox->setSizePolicy( pol );
        pol.setHorizontalStretch( 2 );

        infoWrap->setSizePolicy( pol );
        pol.setVerticalPolicy( QSizePolicy::Fixed );
        m_infoBox->setSizePolicy( pol );
        pol.setVerticalPolicy( QSizePolicy::Expanding );
        spacer->setSizePolicy( pol );
    
        m_pkgNameLabel = new QLabel( m_infoBox );
        m_versionInfo = new QLabel( m_infoBox );
        m_maintainer = new QLabel( m_infoBox );
        m_section = new QLabel( m_infoBox );
        m_installedSize = new QLabel( m_infoBox );
        m_sourcePackage = new QLabel( m_infoBox );
        m_architecture = new QLabel( m_infoBox );

        m_desc->setFrameStyle( QFrame::NoFrame );
        m_desc->viewport()->setAutoFillBackground( false );
        m_versionInfo->setWordWrap( true );
        m_maintainer->setWordWrap( true );
        m_shortDesc->setWordWrap( true );
    }
    
    void setupTabWidget() {
        QSizePolicy pol( QSizePolicy::Preferred,
                         QSizePolicy::Preferred );
        pol.setVerticalStretch( 3 );
        m_bottom = new KTabWidget( this );
        m_bottom->setSizePolicy( pol );
        m_layout->addWidget( m_bottom );

        /* m_packageRelationshipsModel = new TokenModel();
        m_packageRelationshipsView = setupList( this,
                   m_bottom,
                   m_packageRelationshipsModel,
                   d );
        
        m_bottom->addTab(m_packageRelationshipsView,
           i18n("Package Relationships") ); */
        
        /* m_tags = new QTreeView();
           m_bottom->addTab(m_tags, i18n("Tags")); */

        /* m_changelog = new QTextEdit();
        m_changelog->setReadOnly(true);
        m_bottom->addTab(m_changelog, i18n("Developer Changelog") ); */

        m_fileModel = new KDirModel();
        KVBox *fileListBox = new KVBox;
        m_fileList = new QTreeView(fileListBox);
        m_fileLoadingProgress = new QProgressBar( fileListBox );
        m_fileLoadingProgress->setFormat(i18n("Loading File List: %p%"));
        m_fileLoadingProgress->hide();
        m_bottom->addTab( fileListBox, i18n( "Installed Files" ) );
        m_fileProxy = new UrlListProxy();
        m_fileProxy->setSourceModel( m_fileModel );
        m_fileList->setModel( m_fileProxy );
        m_fileList->setItemDelegate( new KFileItemDelegate() );
    }

    void expensiveRefresh() {
        if ( !m_needExpensiveRefresh )
            return;
        m_needExpensiveRefresh = false;

        QTimer::singleShot( 0, this, SLOT( fillRelationshipsList() ) );
        QTimer::singleShot( 0, this, SLOT( loadFileList() ) );
    }
    
protected:
    void showEvent( QShowEvent * event) {
        PackageInfo::showEvent( event );
        expensiveRefresh();
    }

protected Q_SLOTS:
    void loadFileList();
};

}

#endif
