// tasks.cc
//
//  Copyright 2001 Daniel Burrows

#include "tasks.h"
#include "apt.h"

#include "../aptitude.h"

#include <apt-pkg/error.h>
#include <apt-pkg/tagfile.h>

#include <assert.h>
#include <errno.h>

#include <ctype.h>

// Don't bother with hashing.
#include <map>
#include <vector>
#include <algorithm>
#include <sstream>

using namespace std;

map<string, task> *task_list=new map<string, task>;

// This is an array indexed by package ID, managed by load_tasks.
// (as usual, it's initialized to NULL)
list<string> *tasks_by_package;

// A pair (veriterator,verfile) -- used for building a list of versions
// sorted by file location.
typedef pair<pkgCache::PkgIterator, pkgCache::VerFileIterator> loc_pair;

// Used to compare two version files based on their location
struct location_compare
{
  bool operator()(loc_pair a, loc_pair b)
  {
    if(a.second->File==b.second->File)
      return a.second->Offset<b.second->Offset;
    else
      return a.second->File<b.second->File;
  }
};

// Now this is just a wrapper, as you can see..
std::list<std::string> *get_tasks(const pkgCache::PkgIterator &pkg)
{
  if(!tasks_by_package)
    return NULL;

  return tasks_by_package+pkg->ID;
}

// Thanks to Jason for pointing me to the methods that are necessary
// to do this.
static void update_tasks(const pkgCache::PkgIterator &pkg,
			 const pkgCache::VerFileIterator &verfile)
{
  // This should never be called before load_tasks has initialized the
  // tasks structure.
  assert(tasks_by_package);

  list<string> &lst=tasks_by_package[pkg->ID];

  lst.clear();

  if(apt_package_records)
    {
      const char *start,*stop;
      pkgTagSection sec;

      // Pull out pointers to the underlying record.
      apt_package_records->Lookup(verfile).GetRec(start, stop);

      // Parse it as a section.
      sec.Scan(start, stop-start+1);

      string tasks=sec.FindS("Task");

      string::size_type loc=0, firstcomma=0;

      // Strip leading whitespace
      while(loc<tasks.size() && isspace(tasks[loc]))
	++loc;

      while( (firstcomma=tasks.find(',', loc))!=tasks.npos)
	{
	  // Strip trailing whitespace
	  string::size_type loc2=firstcomma-1;
	  while(isspace(tasks[loc2]))
	    --loc2;
	  ++loc2;

	  lst.push_back(string(tasks, loc, loc2-loc));
	  loc=firstcomma+1;

	  // Strip leading whitespace
	  while(loc<tasks.size() && isspace(tasks[loc]))
	    ++loc;
	}

      if(loc!=tasks.size())
	lst.push_back(string(tasks, loc));
    }
}

bool task::keys_present()
{
  if(!keys_present_cache_stale)
    return keys_present_cache;

  keys_present_cache_stale=false;

  for(list<string>::const_iterator i=keys.begin(); i!=keys.end(); ++i)
    {
      pkgCache::PkgIterator pkg=(*apt_cache_file)->FindPkg(*i);

      if(pkg.end())
	{
	  keys_present_cache=false;
	  return false;
	}
      else
	// Here it is assumed that all the tasks are loaded, because
	// we're going to look them up.
	{
	  list<string> *tasks=get_tasks(pkg);

	  if(!tasks)
	    {
	      keys_present_cache=false;
	      return false;
	    }

	  bool present=false;

	  for(list<string>::const_iterator j=tasks->begin();
	      j!=tasks->end(); ++j)
	    if(*j==name)
	      {
		present=true;
		break;
	      }

	  if(!present)
	    {
	      keys_present_cache=false;
	      return false;
	      break;
	    }
	}
    }

  keys_present_cache=true;
  return true;
}

void load_tasks(OpProgress &progress)
{
  // Build a list for each package of the tasks that package belongs to.
  //
  // Sorting by location on disk is *critical* -- otherwise, this operation
  // will take ages.

  // This is done prior to loading the task descriptions so that I can just
  // bail if that fails.

  vector<loc_pair> versionfiles;

  for(pkgCache::PkgIterator pkg=(*apt_cache_file)->PkgBegin();
      !pkg.end(); ++pkg)
    if(!pkg.VersionList().end() && !pkg.VersionList().FileList().end())
      versionfiles.push_back(loc_pair(pkg,  pkg.VersionList().FileList()));

  sort(versionfiles.begin(), versionfiles.end(), location_compare());

  // Allocate and set up the table of task information.
  delete[] tasks_by_package;
  tasks_by_package=new list<string>[(*apt_cache_file)->Head().PackageCount];

  for(vector<loc_pair>::iterator i=versionfiles.begin();
      i!=versionfiles.end();
      ++i)
    update_tasks(i->first, i->second);

  FileFd task_file;

  // Load the task descriptions:
  task_file.Open("/usr/share/tasksel/debian-tasks.desc", FileFd::ReadOnly);

  if(!task_file.IsOpen())
    {
      _error->Discard();

      // Allow the task file not to exist (eg, the user might not have
      // tasksel installed)
      if(errno!=ENOENT)
	_error->Errno("load_tasks",
		      _("Unable to open /usr/share/tasksel/debian-tasks.desc"));

      return;
    }

  int file_size=task_file.Size();
  int amt=0;
  progress.OverallProgress(0, file_size, 1, _("Reading task descriptions"));

  pkgTagFile tagfile(&task_file);
  pkgTagSection section;

  while(tagfile.Step(section))
    {
      task newtask;
      string desc;
      string taskname=section.FindS("Task");

      if(!taskname.empty())
	{
	  istringstream keystr(section.FindS("Key"));

	  while(!keystr.eof())
	    {
	      string s;

	      keystr >> ws >> s >> ws;

	      newtask.keys.push_back(s);
	    }

	  newtask.name=taskname;
	  newtask.section=section.FindS("Section");
	  newtask.relevance=section.FindI("Relevance", 5);

	  desc=section.FindS("Description");

	  string::size_type newline=desc.find('\n');
	  newtask.shortdesc=string(desc, 0, newline);
	  newtask.longdesc=string(desc, newline);

	  (*task_list)[taskname]=newtask;
	}

      amt+=section.size();
      progress.OverallProgress(amt, file_size, 1, _("Reading task descriptions"));
    }
  progress.OverallProgress(file_size, file_size, 1, _("Reading task descriptions"));

  progress.Done();
}

void reset_tasks()
{
  task_list->clear();
  delete[] tasks_by_package;
  tasks_by_package=NULL;
}
