// keybindings.h, -*-c++-*-
//
//  Copyright 1999,2000 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  A global repository for keybindings.

#include "keybindings.h"

#ifdef HAVE_CONFIG_H
#include "../../../config.h"
#endif

#ifdef HAVE_LIBAPT_PKG
#include <apt-pkg/error.h>
#endif

#include <ctype.h>

using namespace std;
using namespace __gnu_cxx;

keybindings global_bindings;

hash_map<string, chtype> keynames;
hash_map<string, chtype> s_keynames;
// For simplicity, we use the convention that the names are stored in
// lowercase; however, the routines to parse keys take this into account and
// convert the input to lowercase before checking it.
// FIXME: Function keys (F0-Fx) really ought to be handled specially
hash_map<chtype, string> rev_keynames;

bool key_tables_initialized=false;

void init_key_tables()
{
  if(key_tables_initialized)
    return;

  key_tables_initialized=true;

  keynames["tab"]='\t';
  rev_keynames['\t']="tab";
  keynames["space"]=' ';
  rev_keynames[' ']="space";

  keynames["break"]=KEY_BREAK;
  rev_keynames[KEY_BREAK]="break";
  keynames["down"]=KEY_DOWN;
  rev_keynames[KEY_DOWN]="down";
  keynames["up"]=KEY_UP;
  rev_keynames[KEY_UP]="up";
  keynames["left"]=KEY_LEFT;
  rev_keynames[KEY_LEFT]="left";
  keynames["right"]=KEY_RIGHT;
  rev_keynames[KEY_RIGHT]="right";
  keynames["home"]=KEY_HOME;
  rev_keynames[KEY_HOME]="home";
  keynames["backspace"]=KEY_BACKSPACE;
  rev_keynames[KEY_BACKSPACE]="backspace";
  keynames["f0"]=KEY_F(0);
  rev_keynames[KEY_F(0)]="f0";
  keynames["f1"]=KEY_F(1);
  rev_keynames[KEY_F(1)]="f1";
  keynames["f2"]=KEY_F(2);
  rev_keynames[KEY_F(2)]="f2";
  keynames["f3"]=KEY_F(3);
  rev_keynames[KEY_F(3)]="f3";
  keynames["f4"]=KEY_F(4);
  rev_keynames[KEY_F(4)]="f4";
  keynames["f5"]=KEY_F(5);
  rev_keynames[KEY_F(5)]="f5";
  keynames["f6"]=KEY_F(6);
  rev_keynames[KEY_F(6)]="f6";
  keynames["f7"]=KEY_F(7);
  rev_keynames[KEY_F(7)]="f7";
  keynames["f8"]=KEY_F(8);
  rev_keynames[KEY_F(8)]="f8";
  keynames["f9"]=KEY_F(9);
  rev_keynames[KEY_F(9)]="f9";
  keynames["f10"]=KEY_F(10);
  rev_keynames[KEY_F(10)]="f10";
  keynames["delete_line"]=KEY_DL;
  rev_keynames[KEY_DL]="delete_line";
  keynames["insert_line"]=KEY_IL;
  rev_keynames[KEY_IL]="insert_line";
  keynames["delete"]=KEY_DC;
  rev_keynames[KEY_DC]="delete";
  keynames["insert"]=KEY_IC;
  rev_keynames[KEY_IC]="insert";
  keynames["insert_exit"]=KEY_EIC;
  rev_keynames[KEY_EIC]="insert_exit";
  keynames["clear"]=KEY_CLEAR;
  rev_keynames[KEY_CLEAR]="clear";
  keynames["clear_eos"]=KEY_EOS;
  rev_keynames[KEY_EOS]="clear_eos";
  keynames["clear_eol"]=KEY_EOL;
  rev_keynames[KEY_EOL]="clear_eol";
  keynames["scrollf"]=KEY_SF;
  rev_keynames[KEY_SF]="scrollf";
  keynames["scrollr"]=KEY_SR;
  rev_keynames[KEY_SR]="scrollr";
  keynames["pagedown"]=KEY_NPAGE;
  rev_keynames[KEY_NPAGE]="pagedown";
  keynames["pageup"]=KEY_PPAGE;
  rev_keynames[KEY_PPAGE]="pageup";
  keynames["enter"]=KEY_ENTER;
  rev_keynames[KEY_ENTER]="enter";
  keynames["return"]=KEY_ENTER;
  keynames["print"]=KEY_PRINT;
  rev_keynames[KEY_PRINT]="print";
  keynames["a1"]=KEY_A1;
  rev_keynames[KEY_A1]="a1";
  keynames["a3"]=KEY_A3;
  rev_keynames[KEY_A3]="a3";
  keynames["b2"]=KEY_B2;
  rev_keynames[KEY_B2]="b2";
  keynames["c1"]=KEY_C1;
  rev_keynames[KEY_C1]="c1";
  keynames["c3"]=KEY_C3;
  rev_keynames[KEY_C3]="c3";
  keynames["backtab"]=KEY_BTAB;
  rev_keynames[KEY_BTAB]="backtab";
  keynames["begin"]=KEY_BEG;
  rev_keynames[KEY_BEG]="begin";
  keynames["cancel"]=KEY_CANCEL;
  rev_keynames[KEY_CANCEL]="cancel";
  keynames["close"]=KEY_CLOSE;
  rev_keynames[KEY_CLOSE]="close";
  keynames["command"]=KEY_COMMAND;
  rev_keynames[KEY_COMMAND]="command";
  keynames["copy"]=KEY_COPY;
  rev_keynames[KEY_COPY]="copy";
  keynames["create"]=KEY_CREATE;
  rev_keynames[KEY_CREATE]="create";
  keynames["end"]=KEY_END;
  rev_keynames[KEY_END]="end";
  keynames["exit"]=KEY_EXIT;
  rev_keynames[KEY_EXIT]="exit";
  keynames["find"]=KEY_FIND;
  rev_keynames[KEY_FIND]="find";
  keynames["help"]=KEY_HELP;
  rev_keynames[KEY_HELP]="help";
  keynames["mark"]=KEY_MARK;
  rev_keynames[KEY_MARK]="mark";
  keynames["message"]=KEY_MESSAGE;
  rev_keynames[KEY_MESSAGE]="message";
  keynames["move"]=KEY_MOVE;
  rev_keynames[KEY_MOVE]="move";
  keynames["next"]=KEY_NEXT;
  rev_keynames[KEY_NEXT]="next";
  keynames["open"]=KEY_OPEN;
  rev_keynames[KEY_OPEN]="open";
  keynames["options"]=KEY_OPTIONS;
  rev_keynames[KEY_OPTIONS]="options";
  keynames["previous"]=KEY_PREVIOUS;
  rev_keynames[KEY_PREVIOUS]="previous";
  keynames["redo"]=KEY_REDO;
  rev_keynames[KEY_REDO]="redo";
  keynames["reference"]=KEY_REFERENCE;
  rev_keynames[KEY_REFERENCE]="reference";
  keynames["refresh"]=KEY_REFRESH;
  rev_keynames[KEY_REFRESH]="refresh";
  keynames["replace"]=KEY_REPLACE;
  rev_keynames[KEY_REPLACE]="replace";
  keynames["restart"]=KEY_RESTART;
  rev_keynames[KEY_RESTART]="restart";
  keynames["resume"]=KEY_RESUME;
  rev_keynames[KEY_RESUME]="resume";
  keynames["save"]=KEY_SAVE;
  rev_keynames[KEY_SAVE]="save";
  keynames["select"]=KEY_SELECT;
  rev_keynames[KEY_SELECT]="select";
  keynames["suspend"]=KEY_SUSPEND;
  rev_keynames[KEY_SUSPEND]="suspend";
  keynames["undo"]=KEY_UNDO;
  rev_keynames[KEY_UNDO]="undo";

  s_keynames["begin"]=KEY_SBEG;
  rev_keynames[KEY_SBEG]="begin";
  s_keynames["cancel"]=KEY_SCANCEL;
  rev_keynames[KEY_SCANCEL]="cancel";
  s_keynames["command"]=KEY_SCOMMAND;
  rev_keynames[KEY_SCOMMAND]="command";
  s_keynames["copy"]=KEY_SCOPY;
  rev_keynames[KEY_SCOPY]="copy";
  s_keynames["create"]=KEY_SCREATE;
  rev_keynames[KEY_SCREATE]="create";
  s_keynames["delete"]=KEY_SDC;
  rev_keynames[KEY_SDC]="delete";
  s_keynames["delete_line"]=KEY_SDL;
  rev_keynames[KEY_SDL]="delete_line";
  s_keynames["end"]=KEY_SEND;
  rev_keynames[KEY_SEND]="end";
  s_keynames["clear_eol"]=KEY_SEOL;
  rev_keynames[KEY_SEOL]="clear_eol";
  s_keynames["exit"]=KEY_SEXIT;
  rev_keynames[KEY_SEXIT]="exit";
  s_keynames["find"]=KEY_SFIND;
  rev_keynames[KEY_SFIND]="find";
  s_keynames["help"]=KEY_SHELP;
  rev_keynames[KEY_SHELP]="help";
  s_keynames["home"]=KEY_SHOME;
  rev_keynames[KEY_SHOME]="home";
  s_keynames["insert"]=KEY_SIC;
  rev_keynames[KEY_SIC]="insert";
  s_keynames["left"]=KEY_SLEFT;
  rev_keynames[KEY_SLEFT]="left";
  s_keynames["message"]=KEY_SMESSAGE;
  rev_keynames[KEY_SMESSAGE]="message";
  s_keynames["move"]=KEY_SMOVE;
  rev_keynames[KEY_SMOVE]="move";
  s_keynames["next"]=KEY_SNEXT;
  rev_keynames[KEY_SNEXT]="next";
  s_keynames["options"]=KEY_SOPTIONS;
  rev_keynames[KEY_SOPTIONS]="options";
  s_keynames["previous"]=KEY_SPREVIOUS;
  rev_keynames[KEY_SPREVIOUS]="previous";
  s_keynames["print"]=KEY_SPRINT;
  rev_keynames[KEY_SPRINT]="print";
  s_keynames["redo"]=KEY_SREDO;
  rev_keynames[KEY_SREDO]="redo";
  s_keynames["replace"]=KEY_SREPLACE;
  rev_keynames[KEY_SREPLACE]="replace";
  s_keynames["right"]=KEY_SRIGHT;
  rev_keynames[KEY_SRIGHT]="right";
  s_keynames["resume"]=KEY_SRSUME;
  rev_keynames[KEY_SRSUME]="resume";
  s_keynames["save"]=KEY_SSAVE;
  rev_keynames[KEY_SSAVE]="save";
  s_keynames["suspend"]=KEY_SSUSPEND;
  rev_keynames[KEY_SSUSPEND]="suspend";
  s_keynames["undo"]=KEY_SUNDO;
  rev_keynames[KEY_SUNDO]="undo";
}

void keybindings::set(string tag, keybinding strokes)
{
  keymap[tag]=strokes;
}

bool keybindings::key_matches(chtype ch, string tag)
{
  hash_map<string, keybinding>::iterator found=keymap.find(tag);
  if(found==keymap.end())
    return parent?parent->key_matches(ch, tag):false;
  else
    {
      for(keybinding::iterator i=found->second.begin(); i!=found->second.end(); i++)
	{
	  if(*i==KEY_ENTER)
	    {
	      if(ch==KEY_ENTER || ch=='\r' || ch=='\n')
		return true;
	    }
	  else if(ch==*i)
	    return true;
	}
      return false;
    }
}

chtype parse_key(string keystr)
{
  bool sfound=false,cfound=false,afound=false;
  string tmpstr=keystr;
  chtype rval=(chtype) ERR;

  init_key_tables();

  while(tmpstr.size()>2 && tmpstr[1]=='-')
    {
      switch(tmpstr[0])
	{
	case 's':
	case 'S':
	  sfound=true;
	  break;
	case 'a':
	case 'A':
	case 'm':
	case 'M':
	  afound=true;
	  break;
	case 'c':
	case 'C':
	  cfound=true;
	  break;
	default:
#ifdef HAVE_LIBAPT_PKG
	  _error->Error("Cannot parse key description: %s", keystr.c_str());
#endif
	  return (chtype) ERR;
	}
      tmpstr=string(tmpstr,2);
    }

  if(tmpstr.size()==0)
    {
#ifdef HAVE_LIBAPT_PKG
      _error->Error("Invalid null keybinding");
#endif
      return (chtype) ERR;
    }

  if(cfound && tmpstr.size()>1)
    {
#ifdef HAVE_LIBAPT_PKG
      _error->Error("Sorry, control modifiers may not be used with unprintable characters");
#endif
      return (chtype) ERR;
    }

  if(tmpstr.size()==1)
    {
      rval=sfound?toupper(tmpstr[0]):tmpstr[0];
      if(cfound)
	rval=KEY_CTRL(rval);
      if(afound)
	rval=KEY_ALT(rval);
      return rval;
    }
  else
    {
      for(unsigned int i=0; i<tmpstr.size(); i++)
	tmpstr[i]=tolower(tmpstr[i]);
      hash_map<string,chtype>::iterator found=(sfound?s_keynames:keynames).find(tmpstr);
      if(found==(sfound?s_keynames:keynames).end())
	return (chtype) ERR;
      else
	{
	  rval=found->second;
	  if(afound)
	    rval=KEY_ALT(rval);

	  return rval;
	}
    }
}

string keyname(chtype ch)
{
  init_key_tables();

  // This is a nasty special-case..all of this stuff is nasty special-cases..
  // someday I need to learn the underlying logic, if there is any..
  if(ch==31)
    return "C-_";
  // <31 seem to be control-characters?  doh...is 32 the amount to add?
  // Daniel is confused..
  if((ch&(~31))==0)
    return "C-"+keyname(ch|64|32);
  // and 200 seems to be the ALT-character?
  else if(ch&0x200)
    return "A-"+keyname(ch&~0x200);
  else
    {
      hash_map<chtype, string>::iterator found=rev_keynames.find(ch);

      if(found!=rev_keynames.end())
	return found->second;
      else
	{
	  char tmp[2];
	  tmp[0]=ch;
	  tmp[1]=0;
	  return string(tmp);
	}
    }
}

// Doesn't return all available bindings as that gets way too long.
string keybindings::keyname(string tag)
{
  hash_map<string, keybinding>::iterator found=keymap.find(tag);

  if(found!=keymap.end())
    return ::keyname(found->second.front());
  else
    return "";
}
