// vs_pager.cc
//
//  Copyright 2000 Daniel Burrows

#include "vs_pager.h"

#include "vscreen.h"
#include "vs_minibuf_win.h"
#include "vs_editline.h"
#include "config/keybindings.h"

#include <unistd.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/fcntl.h>

using namespace std;

keybindings *vs_pager::bindings=NULL;

vs_pager::vs_pager(const char *_text)
  :vscreen_widget(), text(_text), len(text?strlen(_text):0), start(0), horizontal_shift(0)
{
  calc_bounds();
}

vs_pager::vs_pager(const char *_text, int _len)
  :vscreen_widget(), text(_text), len(_len), start(0), horizontal_shift(0)
{
  calc_bounds();
}

vs_pager::~vs_pager() {}

void vs_pager::set_text(const char *_text, int _len)
{
  text=_text;
  len=_len;

  calc_bounds();

  do_signal();

  vscreen_queuelayout();
}

void vs_pager::calc_bounds()
{
  int x=0, y=0;
  maxx=0;
  maxy=0;
  cury=0;
  start=0;

  int curr=0;
  while(curr<len)
    {
      switch(text[curr])
	{
	case '\n':
	  y++;
	  x=0;
	  break;
	case '\t':
	  x+=8-(x%8);
	  break;
	default:
	  x++;
	  break;
	}
      curr++;

      maxx=max<int>(x, maxx);
      maxy=max<int>(y, maxy);
    }
}

void vs_pager::do_signal()
{
  int realmax=max<int>(maxy-getmaxy()+1, 0);
  location_changed(cury, realmax);
}

void vs_pager::scroll_up(int nlines)
{
  while(nlines-->0 && start!=0)
    // Note that if start is 0, the following calculation has no effect
    {
      start-=2;
      while(start>0 && text[start]!='\n')
	start--;
      if(start<0)
	start=0;
      else if(start!=0)
	start++;
      cury--;
    }

  do_signal();
}

void vs_pager::scroll_down(int nlines)
{
  while(nlines-->0 && cury<maxy-getmaxy())
    {
      while(start<len && text[start]!='\n')
	start++;
      if(start<len)
	start++;
      cury++;
    }

  do_signal();
}

void vs_pager::scroll_top()
{
  start=0;
  cury=0;

  do_signal();
}

void vs_pager::scroll_bottom()
{
  int width,height;
  getmaxyx(height,width);
  start=len;
  cury=maxy;
  scroll_up(height);

  do_signal();
}

void vs_pager::scroll_page(bool dir)
{
  if(dir)
    scroll_up(getmaxy());
  else
    scroll_down(getmaxy());
}

void vs_pager::search_for(string s)
{
  if(s!="")
    last_search=s;
  else if(last_search=="")
    {
      beep();
      return;
    }

  // Skip the current line;

  unsigned int i=start;
  while(i<len-last_search.size() && text[i]!='\n')
    i++;
  i++;

  for( ; i<len-last_search.size(); i++)
    {
      if(!strncmp(last_search.c_str(), text+i, last_search.size()))
	{
	  start=i;
	  scroll_up(1);
	  vscreen_update();
	  return;
	}
    }
  beep();
}

bool vs_pager::handle_char(chtype ch)
{
  if(bindings->key_matches(ch, "Up"))
    {
      scroll_up(1);
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "Down"))
    {
      scroll_down(1);
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "Left"))
    {
      horizontal_shift-=1;
      if(horizontal_shift<0)
	horizontal_shift=0;
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "Right"))
    {
      horizontal_shift+=1;
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "PrevPage"))
    {
      int width,height;
      getmaxyx(height,width);
      scroll_up(height);
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "NextPage"))
    {
      int width,height;
      getmaxyx(height,width);
      scroll_down(height);
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "Begin"))
    {
      scroll_top();
      vscreen_update();
    }
  else if(bindings->key_matches(ch, "End"))
    {
      scroll_bottom();
      vscreen_update();
    }
  /*else if(bindings->key_matches(ch, "Search"))
    {
      vs_statusedit *ed=new vs_statusedit("Search for: ");
      ed->entered.connect(slot(this, &vs_pager::search_for));
      add_widget(ed);
    }
  else if(bindings->key_matches(ch, "ReSearch"))
  search_for("");*/
  else
    return vscreen_widget::handle_char(ch);

  return true;
}

void vs_pager::paint()
{
  int x=0, y=0;

  int width,height;
  getmaxyx(height, width);

  int curr=start;
  while(curr<len && y<height)
    {
      switch(text[curr])
	{
	case '\n':
	  y++;
	  x=0;
	  break;
	case '\t':
	  x+=8-(x%8);
	  break;
	default:
	  if(x>=horizontal_shift && x-horizontal_shift<width)
	    // Converting to an unsigned char here is easier than fighting
	    // strlen et al.
	    mvaddch(y, x-horizontal_shift, (unsigned char) text[curr]);
	  x++;
	  break;
	}
      curr++;
    }
}

size vs_pager::size_request()
{
  return size(maxx, maxy);
}

void vs_pager::init_bindings()
{
  bindings=new keybindings(&global_bindings);
}

vs_file_pager::vs_file_pager():vs_pager("", -1), fd(-1)
{
}

vs_file_pager::vs_file_pager(string filename)
  :vs_pager("", -1), fd(-1)
{
  load_file(filename);
}

vs_file_pager::vs_file_pager(const char *text, int size)
  :vs_pager(text, size), fd(-1)
{
}

void vs_file_pager::load_file(string filename)
{
  if(fd!=-1)
    dispose_mapping();

  fd=open(filename.c_str(), O_RDONLY, 0644);

  if(fd==-1)
    error="open: "+filename+": "+strerror(errno);
  else
    {
      struct stat buf;
      if(fstat(fd, &buf)<0)
	{
	  close(fd);
	  fd=-1;
	  error="fstat: "+filename+": "+strerror(errno);
	}
      else
	{
	  const char *contents=(const char *) mmap(NULL,
						   buf.st_size,
						   PROT_READ,
						   MAP_SHARED,
						   fd,
						   0);

	  if(contents==((const char *) -1))
	    {
	      close(fd);
	      fd=-1;
	      error="mmap: "+filename+": "+strerror(errno);
	      // FIXME: just display something in the widget itself?
	    }
	  else
	    vs_pager::set_text(contents, buf.st_size);
	}
    }
}

void vs_file_pager::dispose_mapping()
{
  if(fd!=-1)
    {
      munmap((void *) get_text(), get_len());
      close(fd);
    }
  fd=-1;
  vs_pager::set_text(NULL, -1);
  error.erase();
}
