// vs_multiplex.cc
//
//  Copyright 1999,2000 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.

#include "vscreen.h"
#include "vs_multiplex.h"

#include <sigc++/bind.h>
#include <sigc++/object_slot.h>

#include <assert.h>

using namespace std;

vs_multiplex::vs_multiplex()
  :vs_passthrough(), visible_child(children.end())
{
  do_layout.connect(slot(*this, &vs_multiplex::layout_me));

  focussed.connect(slot(*this, &vs_multiplex::got_focus));
  unfocussed.connect(slot(*this, &vs_multiplex::lost_focus));
}

vs_multiplex::~vs_multiplex()
{
  for(list<vscreen_widget *>::iterator i=children.begin();
      i!=children.end();
      i++)
    (*i)->set_owner(NULL);
}

size vs_multiplex::size_request()
{
  if(visible_child!=children.end())
    return (*visible_child)->size_request();
  else
    return size(0,0);
}

void vs_multiplex::layout_me()
{
  if(visible_child!=children.end())
    (*visible_child)->alloc_size(0, 0, getmaxx(), getmaxy());
}

void vs_multiplex::paint()
{
  if(visible_child!=children.end())
    (*visible_child)->display();
}

void vs_multiplex::dispatch_mouse(short id, int x, int y, int z,
				  mmask_t bstate)
{
  if(visible_child!=children.end())
    (*visible_child)->dispatch_mouse(id,
				     x-(*visible_child)->get_startx(),
				     y-(*visible_child)->get_starty(),
				     z, bstate);
}

void vs_multiplex::got_focus()
{
  if(visible_child!=children.end())
    (*visible_child)->focussed();
}

void vs_multiplex::lost_focus()
{
  if(visible_child!=children.end())
    (*visible_child)->unfocussed();
}

void vs_multiplex::show_all()
{
  show();

  if(visible_child!=children.end())
    (*visible_child)->show_all();
}

void vs_multiplex::show_widget(vscreen_widget *w)
{
  assert(!children.empty());

  list<vscreen_widget *>::iterator new_visible=visible_child;

  if(new_visible!=children.end())
    ++new_visible;
  else
    new_visible=children.begin();

  while(new_visible!=visible_child)
    {
      if(new_visible==children.end())
	new_visible=children.begin();
      else if(*new_visible==w)
	break;
      else
	++new_visible;
    }

  if(visible_child!=children.end() && get_isfocussed())
    (*visible_child)->unfocussed();

  visible_child=new_visible;

  if(visible_child!=children.end() && get_isfocussed())
    (*visible_child)->focussed();

  vscreen_queuelayout();
  vscreen_update();
}

void vs_multiplex::hide_widget(vscreen_widget *w)
{
  assert(!children.empty());

  if(visible_child!=children.end() && *visible_child==w)
    {
      list<vscreen_widget *>::iterator new_visible=visible_child;

      if(new_visible!=children.begin())
	--new_visible;
      else
	{
	  new_visible=children.end();
	  --new_visible;
	}

      while(new_visible!=visible_child)
	{
	  if((*new_visible)->get_visible())
	    break;
	  else if(new_visible==children.begin())
	    {
	      new_visible=children.end();
	      --new_visible;
	    }
	  else
	    --new_visible;
	}

      if(visible_child!=children.end() && get_isfocussed())
	(*visible_child)->unfocussed();

      if(new_visible==visible_child)
	visible_child=children.end();
      else
	visible_child=new_visible;

      if(visible_child!=children.end() && get_isfocussed())
	(*visible_child)->focussed();
    }

  vscreen_queuelayout();
  vscreen_update();
}

vscreen_widget *vs_multiplex::get_focus()
{
  return visible_child==children.end()?NULL:*visible_child;
}

void vs_multiplex::rem_widget(vscreen_widget *w)
{
  hide_widget(w);

  for(list<vscreen_widget *>::iterator i=children.begin(); i!=children.end(); i++)
    {
      if(*i==w)
	{
	  assert(w->get_owner()==this);

	  w->set_owner(NULL);

	  break;
	}
    }

  children.remove(w);

  vscreen_queuelayout();
  vscreen_update();
}

void vs_multiplex::add_widget(vscreen_widget *w)
{
  w->shown_sig.connect(bind(slot(*this, &vs_multiplex::show_widget), w));
  w->hidden_sig.connect(bind(slot(*this, &vs_multiplex::hide_widget), w));

  children.push_back(w);
  w->set_owner(this);

  if(w->get_visible())
    show_widget(w);
}

void vs_multiplex::add_widget_after(vscreen_widget *w, vscreen_widget *after)
{
  for(list<vscreen_widget *>::iterator i=children.begin();
      i!=children.end();
      i++)
    {
      if(*i==after)
	{
	  ++i;

	  w->shown_sig.connect(bind(slot(*this, &vs_multiplex::show_widget), w));
	  w->hidden_sig.connect(bind(slot(*this, &vs_multiplex::hide_widget), w));

	  children.insert(i, w);
	  w->set_owner(this);

	  if(w->get_visible())
	    show_widget(w);

	  return;
	}
    }

  // Fallback to something reasonable.  (or just abort? dunno)
  add_widget(w);
}

void vs_multiplex::cycle_forward()
{
  if(!children.empty())
    {
      list<vscreen_widget *>::iterator new_visible=visible_child;

      if(new_visible!=children.end())
	++new_visible;
      else
	new_visible=children.begin();

      while(new_visible!=visible_child)
	{
	  if(new_visible==children.end())
	    new_visible=children.begin();
	  else if((*new_visible)->get_visible())
	    break;
	  else
	    ++new_visible;
	}

      if(visible_child!=new_visible)
	{
	  vscreen_queuelayout();
	  vscreen_update();
	}

      if(visible_child!=children.end() && get_isfocussed())
	(*visible_child)->unfocussed();

      visible_child=new_visible;

      if(visible_child!=children.end() && get_isfocussed())
	(*visible_child)->focussed();
    }
}

void vs_multiplex::cycle_backward()
{
  if(!children.empty())
    {
      list<vscreen_widget *>::iterator new_visible=visible_child;

      if(new_visible!=children.begin())
	--new_visible;
      else
	{
	  new_visible=children.end();
	  --new_visible;
	}

      while(new_visible!=visible_child)
	{
	  if((*new_visible)->get_visible())
	    break;
	  else if(new_visible==children.begin())
	    {
	      new_visible=children.end();
	      --new_visible;
	    }
	  else
	    --new_visible;
	}

      if(visible_child!=new_visible)
	{
	  vscreen_queuelayout();
	  vscreen_update();
	}

      if(visible_child!=children.end() && get_isfocussed())
	(*visible_child)->unfocussed();

      visible_child=new_visible;

      if(visible_child!=children.end() && get_isfocussed())
	(*visible_child)->focussed();
    }
}

vscreen_widget *vs_multiplex::visible_widget()
{
  if(visible_child!=children.end())
    return *visible_child;
  else
    return NULL;
}

unsigned int vs_multiplex::num_children()
{
  return children.size();
}

unsigned int vs_multiplex::num_visible()
{
  unsigned int n=0;

  for(list<vscreen_widget *>::iterator i=children.begin();
      i!=children.end(); ++i)
    if((*i)->get_visible())
      ++n;

  return n;
}
