/* cmd-tree-id.c
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2004 Canonical Limited
 *       Authors: Robert Collins <robert.collins@canonical.com>
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "commands/tree-id.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [dir]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit."))

t_uchar arch_cmd_tree_id_help[] = N_("Print the tree identifier for a project tree.\nPrint the patchset identifier that represents the\ncurrent tree. I.e. foo@example.com/demo--1.0--patch-4.\nDefaults to the tree the current directory is in.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_tree_id (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  char * dir;
  arch_project_tree_t tree;

  dir = ".";

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_tree_id_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;
        }
    }

  if (argc > 2)
    goto usage_error;

  if (argc == 2)
    dir = argv[1];

  arch_project_tree_init (&tree, dir);

  if (!tree.root)
    {
      safe_printfmt (2, "%s: directory (%s) is not in a project tree\n", argv[0], dir);
      exit (1);
    }

  if (!tree.fqrevision)
    {
      safe_printfmt (2, "%s: unable to determine project tree identifier.\n  tree: %s\n",
                     argv[0], tree.root);
      exit (1);
    }

  safe_printfmt (1, "%s\n", tree.fqrevision);

  arch_project_tree_finalise (&tree);

  return 0;
}
