/* inode-sig.c: inode sig manipulation
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2005 Canonical Limited
 *      Authors: Robert Collins <robert.collins@canonical.com>
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/vu/safe.h"
#include "hackerlab/char/pika-escaping-utils.h"
#include "hackerlab/os/errno-to-string.h"
#include "libarch/project-tree.h"
#include "libarch/inode-sig.h"
#include "commands/cmd.h"
#include "commands/inode-sig.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [dir]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("display help")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("display version info\n")) 


t_uchar arch_cmd_inode_sig_help[] = N_("output the inode signature for a source tree.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};


int
arch_cmd_inode_sig (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  struct arch_inventory_options inv_options;
  t_uchar * maybe_root = NULL;

  option = 0;
  mem_set0 ((t_uchar *)&inv_options, sizeof (inv_options));

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv,
                        program_name, usage, libarch_version_string, arch_cmd_inode_sig_help,
                        opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

        /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        }
    }

  lim_free (lim_use_must_malloc, option);

  if (argc == 1)
      maybe_root = str_save (0, ".");
  else if (argc == 2)
      maybe_root = str_save (0, argv[1]);
  else
      goto usage_error;

  {
    arch_project_tree_t * tree = arch_project_tree_new (talloc_context, maybe_root);
    inode_sig inode_sig = arch_tree_inode_sig (tree);
    rel_print_pika_escape_iso8859_1_table (1, arch_escape_classes, inode_sig.ids);
    arch_inode_sig_free (inode_sig);
    arch_project_tree_delete (tree);
  }
  return 0;
}
