/* tag.c:
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 * Copyright (C) 2005 Canonical Limited.
 *        Authors: Robert Collins <robert.collins@canonical.com>
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "hackerlab/bugs/panic.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/os/time.h"
#include "hackerlab/char/str.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/vu/safe.h"
#include "libdate/date-string.h"
#include "libfsutils/string-files.h"
#include "libfsutils/copy-file.h"
#include "libfsutils/tmp-files.h"
#include "libfsutils/rmrf.h"
#include "libarch/ancestry.h"
#include "libarch/arch.h"
#include "libarch/diffs.h"
#include "libarch/patch-logs.h"
#include "libarch/invent.h"
#include "libarch/my.h"
#include "libarch/hooks.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/make-changeset.h"
#include "libarch/changeset-report.h"
#include "libarch/local-cache.h"
#include "libarch/inv-ids.h"
#include "libarch/changelogs.h"
#include "libarch/apply-changeset.h"
#include "libarch/merge-points.h"
#include "libarch/tag.h"


/* __STDC__ prototypes for static functions */
static void arch_prepare_for_tag (tag_method_t * tag,
                                  struct arch_archive * from_arch, t_uchar * from_revision,
                                  t_uchar * raw_log);
static t_uchar * tag_temp_dir (tag_method_t * tag);



tag_method_t *
arch_tag (void * context, int chatter_fd,
          struct arch_archive * arch,
          t_uchar * revision,
          struct arch_archive * from_arch,
          t_uchar * from_revision,
          t_uchar * raw_log)
{
  t_uchar * errstr;
  t_uchar * version = 0;
  t_uchar * this_level = 0;
  arch_patch_id * prev_revision;
  t_uchar * continuation = 0;
  t_uchar * my_uid = 0;
  t_uchar * txn_id = 0;
  tag_method_t * result = talloc (talloc_context, tag_method_t);
  result->arch = talloc_reference (result, arch);
  result->revision = talloc_strdup (result, revision);
  result->temp_dir = NULL;
  arch_patch_id * patch_id = arch_patch_id_new_archive (result->arch->official_name, result->revision);

  version = arch_parse_package_name (arch_ret_package_version, 0, revision);
  this_level = arch_parse_package_name (arch_ret_patch_level, 0, revision);
  prev_revision = arch_previous_revision (arch, patch_id);

  continuation = arch_fully_qualify (from_arch->official_name, from_revision);

  my_uid = arch_my_id_uid ();
  txn_id = arch_generate_txn_id ();

  arch_prepare_for_tag (result, from_arch, from_revision, raw_log);

  if (arch_archive_lock_revision (&errstr, arch, version, prev_revision ? arch_patch_id_patchlevel (prev_revision) : NULL, my_uid, txn_id, this_level))
    {
      safe_printfmt (2, "arch_tag: unable to acquire revision lock (%s)\n    revision: %s/%s\n    url: %s\n",
                     errstr, arch->official_name, revision, arch->location);
      exit (2);
    }

  if (arch_archive_put_log (&errstr, arch, version, prev_revision ? arch_patch_id_patchlevel (prev_revision) : NULL, my_uid, txn_id, result->cooked_log))
    {
      safe_printfmt (2, "arch_tag: unable to send log message to archive (%s)\n    revision: %s/%s\n    url: %s\n",
                     errstr, arch->official_name, version, arch->location);
      exit (2);
    }

  if (arch_archive_put_changeset (&errstr, arch, version, prev_revision ? arch_patch_id_patchlevel (prev_revision) : NULL, my_uid, txn_id, this_level, result->changeset))
    {
      safe_printfmt (2, "arch_tag: unable to send log message to archive (%s)\n    revision: %s/%s\n    url: %s\n",
                     errstr, arch->official_name, version, arch->location);
      exit (2);
    }

  if (arch_archive_put_continuation (&errstr, arch, version, prev_revision ? arch_patch_id_patchlevel (prev_revision) : NULL, my_uid, txn_id, continuation))
    {
      safe_printfmt (2, "arch_tag: unable to send tagged revision id to archive (%s)\n    revision: %s/%s\n    url: %s\n",
                     errstr, arch->official_name, revision, arch->location);
      exit (2);
    }

  if (arch_revision_ready (&errstr, arch, version, prev_revision ? arch_patch_id_patchlevel (prev_revision) : NULL, my_uid, txn_id, this_level))
    {
      safe_printfmt (2, "arch_tag: error sending tag to archive (%s)\n    revision: %s/%s\n    url: %s\n",
                     errstr, arch->official_name, revision, arch->location);
      exit (2);
    }
  
  if (arch_archive_finish_revision (&errstr, arch, version, prev_revision ? arch_patch_id_patchlevel (prev_revision) : NULL, my_uid, txn_id, this_level))
    {
      safe_printfmt (2, "arch_tag: unable to complete commit transaction (%s)\n    revision: %s/%s\n    url: %s\n",
                     errstr, arch->official_name, version, arch->location);
      exit (2);
    }

    {
      /* FIXME upload limited history and update a current dir in the archive */
      ancestry_upload_patch (arch, patch_id, -1);
    }


  {
    t_uchar * revision = 0;

    revision = str_alloc_cat_many (0, version, "--", this_level, str_end);

    arch_run_hook ("tag", "ARCH_ARCHIVE", arch->official_name, "ARCH_REVISION", revision, "ARCH_TAGGED_ARCHIVE", from_arch->official_name, "ARCH_TAGGED_REVISION", from_revision, 0);

    lim_free (0, revision);
  }

  lim_free (0, version);
  lim_free (0, this_level);
  talloc_free (prev_revision);
  lim_free (0, continuation);
  lim_free (0, my_uid);
  lim_free (0, txn_id);
  talloc_free (patch_id);
  return talloc_steal (context, result);
}



t_uchar *
tag_temp_dir (tag_method_t * tag)
{
  if (tag->temp_dir)
    return tag->temp_dir;
  /*    tmp_output = tmp_file_name_in_tmp (".#baz-branch"); */
  tag->temp_dir = talloc_tmp_file_name (tag, ".", ",,tag-working");
  rmrf_file (tag->temp_dir);
  safe_mkdir (tag->temp_dir, 0777);
  talloc_set_destructor (tag->temp_dir, rmrf_on_free);
  return tag->temp_dir;
}

void
arch_prepare_for_tag (tag_method_t * tag, 
                      struct arch_archive * from_arch, t_uchar * from_revision,
                      t_uchar * raw_log)
{
  t_uchar * my_id = 0;
  time_t now;
  t_uchar * std_date = 0;
  t_uchar * human_date = 0;
  t_uchar * from_version = 0;
  int out_fd;
  rel_table merge_points = 0;
  t_uchar * fqrevision = 0;
  t_uchar * changeset_tail = 0;
  struct arch_make_changeset_report make_report = {0, };
  struct arch_changeset_report report = {0, };
  int log_fd = -1;
  t_uchar * log_loc = 0;
  t_uchar * log_id = 0;

  my_id = arch_my_id ();
  now = time(0);
  std_date = standard_date (now);
  human_date = pretty_date (now);

  from_version = arch_parse_package_name (arch_ret_package_version, 0, from_revision);

  tag->current_tree = arch_project_tree_new (tag, ".");
  if (tag->current_tree->root)
    {
      tag->reference_is_tmp = 0;
      tag->reference_tree_root = arch_find_or_make_local_copy (-1, tag->current_tree, 
							  0, from_arch, from_arch->official_name, from_revision);
    }
  else
    {
      tag->reference_is_tmp = 1;
      tag->reference_tree_root = arch_find_or_make_tmp_local_copy (-1, tag_temp_dir (tag), tag->current_tree, 0, from_arch, from_arch->official_name, from_revision);
    }
  tag->reference_tree_root = str_replace (tag->reference_tree_root, talloc_strdup (tag, tag->reference_tree_root));

  merge_points = arch_all_logs (tag->reference_tree_root);
  fqrevision = arch_fully_qualify (tag->arch->official_name, tag->revision);
  rel_add_records (&merge_points, rel_make_record (fqrevision, 0), 0);
#define merge_col 0
  rel_sort_table_by_field (0, merge_points, merge_col);
  rel_uniq_by_field (&merge_points, merge_col);
  arch_sort_table_by_name_field (0, merge_points, merge_col);

  out_fd = make_output_to_string_fd ();

  safe_printfmt (out_fd, "Revision: %s\n", tag->revision);
  safe_printfmt (out_fd, "Archive: %s\n", tag->arch->official_name);
  safe_printfmt (out_fd, "Creator: %s\n", my_id);
  safe_printfmt (out_fd, "Date: %s\n", human_date);
  safe_printfmt (out_fd, "Standard-date: %s\n", std_date);
  arch_print_log_list_header (out_fd, "New-patches", merge_points, merge_col);
  safe_printfmt (out_fd, "Continuation-of: %s/%s\n", from_arch->official_name, from_revision);
  if (raw_log)
    safe_printfmt (out_fd, "%s", raw_log);
  else
    safe_printfmt (out_fd, "Summary: tag of %s/%s\n\n(automatically generated log message)\n", from_arch->official_name, from_revision);

  tag->cooked_log = string_fd_close (out_fd);
  tag->cooked_log = str_replace (tag->cooked_log, talloc_strdup (tag, tag->cooked_log));

  changeset_tail = str_alloc_cat (0, tag->revision, ".patches");
  tag->changeset = talloc_file_name_in_vicinity (tag, tag_temp_dir (tag), changeset_tail);
  arch_make_empty_changeset (&make_report, &report, tag->changeset);
  log_loc = arch_log_file (".", tag->arch->official_name, tag->revision);
  log_id = arch_log_file_id (tag->arch->official_name, tag->revision);
  log_fd = arch_changeset_add_file (0, &report, &make_report, tag->changeset, log_loc, log_id);
  safe_printfmt (log_fd, "%s", tag->cooked_log);
  safe_close (log_fd);
  arch_changeset_rewrite_indexes (tag->changeset, &report);

  lim_free (0, my_id);
  lim_free (0, std_date);
  lim_free (0, human_date);
  lim_free (0, fqrevision);
  rel_free_table (merge_points);
  lim_free (0, changeset_tail);
  lim_free (0, log_loc);
  lim_free (0, log_id);
  arch_free_make_changeset_report_data (&make_report);
  arch_free_changeset_report_data (&report);
#undef merge_col
}

void
arch_tag_cacherev (tag_method_t * tag)
{
  safe_printfmt (1, "* Archive caching revision\n");
  safe_flush (1);

  if (tag->reference_is_tmp)
    {
      /* recycle reference */
      t_uchar * inventory_path;
      struct arch_apply_changeset_report * report;
      arch_project_tree_t * recycled_tree;

      t_uchar * recycled_tmp = talloc_tmp_file_name (talloc_context, tag_temp_dir (tag), ",,tag_to_pristine");
      t_uchar * final_name = talloc_file_name_in_vicinity (recycled_tmp, tag_temp_dir (tag), tag->revision);
      safe_rename (tag->reference_tree_root, recycled_tmp);
      talloc_free (tag->reference_tree_root);
      tag->reference_tree_root = NULL;
      
      inventory_path = talloc_file_name_in_vicinity (recycled_tmp, recycled_tmp, ",,index");
      rmrf_file (inventory_path);
      recycled_tree = arch_project_tree_new (recycled_tmp, recycled_tmp);

      report = arch_apply_changeset (tag->changeset, recycled_tmp, recycled_tree, arch_unspecified_id_tagging, arch_inventory_unrecognized, 0, 0, arch_escape_classes, NULL, NULL);
      invariant (!arch_conflicts_occurred (report));

      safe_rename (recycled_tmp, final_name);

      tag->tagged_tree_root = talloc_steal (tag, final_name);
      talloc_free (recycled_tmp);
    }
  else
    {
      /* cant touch reference tree, but can make new one and it should use the
       * cache as thats how we got it in the first place
       */
      tag->tagged_tree_root = arch_find_or_make_local_copy (-1, tag->current_tree, 
							  0, tag->arch, tag->arch->official_name, tag->revision);
    }
    {
      t_uchar * errstr;
      
      if (arch_archive_put_cached (&errstr, tag->arch, tag->revision, tag->tagged_tree_root))
        {
          safe_printfmt (2, "Warning:  was unable to cache revision %s/%s (%s)\n",
                         tag->arch->official_name, tag->revision, errstr);
        }
      else
        {
          safe_printfmt (1, "* Made cached revision of %s/%s\n", tag->arch->official_name, tag->revision);
        }
    }
}




/* tag: Tom Lord Tue May 27 17:37:04 2003 (tag.c)
 */
