/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <netdb.h>
#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <gtk/gtk.h>

#include "discover.h"

static unsigned char ping_data[] = { 'p', 'i', 'n', 'g' };
static unsigned char pong_data[] = { 'p', 'o', 'n', 'g' };

static inline void send_ping(int sk)
{
	struct sockaddr_in addr;
	ssize_t len;

	memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_port = htons(10839);
	addr.sin_addr.s_addr = htonl(INADDR_BROADCAST);

	len = sendto(sk, ping_data, sizeof(ping_data), 0,
				(struct sockaddr *) &addr, sizeof(addr));
}

static int create_socket(void)
{
	struct sockaddr_in addr;
	int sk, opt = 1;

	sk = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (sk < 0)
		return -1;

	if (setsockopt(sk, SOL_SOCKET, SO_BROADCAST, &opt, sizeof(opt)) < 0) {
		close(sk);
		return -1;
	}

	memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_port = htons(0);
	addr.sin_addr.s_addr = htonl(INADDR_ANY);

	if (bind(sk, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
		close(sk);
		return -1;
	}

	return sk;
}

static gboolean socket_callback(GIOChannel *chan,
					GIOCondition cond, gpointer user_data)
{
	GtkListStore *store = user_data;
	GtkTreeIter iter;
	struct sockaddr_in addr;
	socklen_t addr_len = sizeof(addr);
	unsigned char buf[64];
	ssize_t len;
	char name[128];
	int sk;

	if (cond & (G_IO_NVAL | G_IO_ERR | G_IO_HUP))
		return TRUE;

	sk = g_io_channel_unix_get_fd(chan);

	len = recvfrom(sk, buf, sizeof(buf), MSG_DONTWAIT,
				(struct sockaddr *) &addr, &addr_len);

	if (len < 0)
		return TRUE;

	if (len != sizeof(pong_data))
		return TRUE;

	if (memcmp(buf, pong_data, sizeof(pong_data)) != 0)
		return TRUE;

	if (getnameinfo((struct sockaddr *) &addr, addr_len,
					name, sizeof(name), NULL, 0, 0) < 0)
		return TRUE;

	gtk_list_store_insert_with_values(store, &iter, -1, 0, name, -1);

	return TRUE;
}

GtkTreeModel *create_discover(const gchar *recent)
{
	GtkListStore *store;
	GtkTreeIter iter;
	GIOChannel *io;
	int sk;

	store = gtk_list_store_new(1, G_TYPE_STRING);

	if (recent != NULL) {
		gtk_list_store_insert_with_values(store, &iter, -1, 0,
								recent, -1);
		gtk_list_store_insert_with_values(store, &iter, -1, 0,
								NULL, -1);
	}

	sk = create_socket();
	if (sk < 0)
		return GTK_TREE_MODEL(store);

	io = g_io_channel_unix_new(sk);
	if (io == NULL) {
		close(sk);
		return GTK_TREE_MODEL(store);
	}

	g_io_channel_set_close_on_unref(io, TRUE);
	g_io_channel_set_encoding(io, NULL, NULL);
	g_io_channel_set_buffered(io, FALSE);
	g_io_channel_set_flags(io, G_IO_FLAG_NONBLOCK, NULL);

	g_object_set_data(G_OBJECT(store), "io", io);

	g_io_add_watch_full(io, G_PRIORITY_DEFAULT,
				G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
						socket_callback, store, NULL);

	send_ping(sk);

	return GTK_TREE_MODEL(store);
}

void destroy_discover(GtkTreeModel *model)
{
	GIOChannel *io;

	if (model == NULL)
		return;

	io = g_object_get_data(G_OBJECT(model), "io");
	if (io != NULL)
		g_io_channel_unref(io);

	g_object_unref(model);
}
