/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>

#include "hci.h"
#include "baseband.h"

static struct hci_error_entry {
	guint8 code;
	gchar *label;
} error_map[] = {
	{ 0x01, "Unknown HCI Command"                         },
	{ 0x02, "Unknown Connection Identifier"               },
	{ 0x03, "Hardware Failure"                            },
	{ 0x04, "Page Timeout"                                },
	{ 0x05, "Authentication Failure"                      },
	{ 0x06, "PIN or key Missing"                          },
	{ 0x07, "Memory Capacity Exceeded"                    },
	{ 0x08, "Connection Timeout"                          },
	{ 0x09, "Connection Limit Exceeded"                   },
	{ 0x0c, "Command Disallowed"                          },
	{ 0x11, "Unsupported Feature or Parameter Value"      },
	{ 0x12, "Invalid HCI Command Parameters"              },
	{ 0x25, "Encryption Mode Not Acceptable"              },
	{ 0x26, "Link Key Can Not be Changed"                 },
	{ 0x27, "Requested QoS Not Supported"                 },
	{ 0x28, "Instant Passed"                              },
	{ 0x29, "Pairing With Unit Key Not Supported"         },
	{ 0x2a, "Different Transaction Collision"             },
	{ 0x2c, "QoS Unacceptable Parameter"                  },
	{ 0x2d, "QoS Rejected"                                },
	{ 0x2e, "Channel Classification Not Supported"        },
	{ 0x2f, "Insufficient Security"                       },
	{ 0x30, "Parameter Out Of Mandatory Range"            },
	{ 0x32, "Role Switch Pending"                         },
	{ 0x34, "Reserved Slot Violation"                     },
	{ 0x35, "Role Switch Failed"                          },
	{ 0x36, "Extended Inquiry Response Too Large"         },
	{ 0x37, "Secure Simple Pairing Not Supported By Host" },
	{ 0x38, "Host Busy - Pairing"                         },
	{ 0xff, "Unknown"                                     },
};

static gchar *status2str(guint8 code)
{
	struct hci_error_entry *entry = error_map;

	while (1) {
		if (entry->code == code || entry->code == 0xff)
			return entry->label;
		entry++;
	}

	return NULL;
}

static gchar *create_status(gchar *string, guint8 status)
{
	gchar *temp = string, *error = NULL;

	if (status != 0x00)
		error = g_strdup_printf("\n<span foreground=\"red\">%s</span>",
							status2str(status));

	string = g_strdup_printf("%s status 0x%02x%s", temp ? temp : "",
						status, error ? error : "");

	g_free(error);

	g_free(temp);

	return string;
}

static struct hci_command_entry {
	gint number;
	guint16 opcode;
	gchar *label;
	gboolean noevent;
	gchar *(*func)(gpointer data, guint size);
} command_map[] = {
	{    0, 0x0401, "Inquiry",                                  FALSE },
	{    1, 0x0402, "Inquiry Cancel",                           TRUE  },
	{    2, 0x0403, "Periodic Inquiry Mode",                    TRUE  },
	{    3, 0x0404, "Exit Periodic Inquiry Mode",               TRUE  },
	{    4, 0x0405, "Create Connection",                        FALSE },
	{    5, 0x0406, "Disconnect",                               FALSE },
	{    6, 0x0407, "Add SCO Connection",                       FALSE },
	{    7, 0x0408, "Create Connection Cancel",                 TRUE  },
	{    8, 0x0409, "Accept Connection Request",                FALSE },
	{    9, 0x040a, "Reject Connection Request",                FALSE },
	{   10, 0x040b, "Link Key Request Reply",                   TRUE  },
	{   11, 0x040c, "Link Key Request Negative Reply",          TRUE  },
	{   12, 0x040d, "PIN Code Request Reply",                   TRUE  },
	{   13, 0x040e, "PIN Code Request Negative Reply",          TRUE  },
	{   14, 0x040f, "Change Connection Packet Type",            FALSE },
	{   15, 0x0411, "Authentication Requested",                 FALSE },
	{   16, 0x0413, "Set Connection Encryption",                FALSE },
	{   17, 0x0415, "Change Connection Link Key",               FALSE },
	{   18, 0x0417, "Master Link Key",                          FALSE },
	{   19, 0x0419, "Remote Name Request",                      FALSE },
	{   20, 0x041a, "Cancel Remote Name Request",               TRUE  },
	{   21, 0x041b, "Read Remote Supported Features",           FALSE },
	{   22, 0x041c, "Read Remote Extended Features",            FALSE },
	{   23, 0x041d, "Read Remote Version Information",          FALSE },
	{   24, 0x041f, "Read Clock Offset",                        FALSE },
	{   25, 0x0420, "Read LMP Handle",                          TRUE  },
	{   33, 0x0801, "Hold Mode",                                FALSE },
	{   34, 0x0803, "Sniff Mode",                               FALSE },
	{   35, 0x0804, "Exit Sniff Mode",                          FALSE },
	{   36, 0x0805, "Park State",                               FALSE },
	{   37, 0x0806, "Exit Park State",                          FALSE },
	{   38, 0x0807, "QoS Setup",                                FALSE },
	{   39, 0x0809, "Role Discovery",                           TRUE  },
	{   40, 0x080b, "Switch Role",                              FALSE },
	{   41, 0x080c, "Read Link Policy Settings",                TRUE  },
	{   42, 0x080d, "Write Link Policy Settings",               TRUE  },
	{   43, 0x080e, "Read Default Link Policy Settings",        TRUE  },
	{   44, 0x080f, "Write Default Link Policy Settings",       TRUE  },
	{   45, 0x0810, "Flow Specification",                       FALSE },
	{   46, 0x0c01, "Set Event Mask",                           TRUE  },
	{   47, 0x0c03, "Reset",                                    TRUE  },
	{   48, 0x0c05, "Set Event Filter",                         TRUE  },
	{   49, 0x0c08, "Flush",                                    TRUE  },
	{   50, 0x0c09, "Read PIN Type",                            TRUE  },
	{   51, 0x0c0a, "Write PIN Type",                           TRUE  },
	{   52, 0x0c0b, "Create New Unit Key",                      TRUE  },
	{   53, 0x0c0d, "Read Stored Link Key",                     TRUE  },
	{   54, 0x0c11, "Write Stored Link Key",                    TRUE  },
	{   55, 0x0c12, "Delete Stored Link Key",                   TRUE  },
	{   56, 0x0c13, "Write Local Name",                         TRUE  },
	{   57, 0x0c14, "Read Local Name",                          TRUE  },
	{   58, 0x0c15, "Read Connection Accept Timeout",           TRUE  },
	{   59, 0x0c16, "Write Connection Accept Timeout",          TRUE  },
	{   60, 0x0c17, "Read Page Timeout",                        TRUE  },
	{   61, 0x0c18, "Write Page Timeout",                       TRUE  },
	{   62, 0x0c19, "Read Scan Enable",                         TRUE  },
	{   63, 0x0c1a, "Write Scan Enable",                        TRUE  },
	{   64, 0x0c1b, "Read Page Scan Activity",                  TRUE  },
	{   65, 0x0c1c, "Write Page Scan Activity",                 TRUE  },
	{   66, 0x0c1d, "Read Inquiry Scan Activity",               TRUE  },
	{   67, 0x0c1e, "Write Inquiry Scan Activity",              TRUE  },
	{   68, 0x0c1f, "Read Authentication Enable",               TRUE  },
	{   69, 0x0c20, "Write Authentication Enable",              TRUE  },
	{   70, 0x0c21, "Read Encryption Mode",                     TRUE  },
	{   71, 0x0c22, "Write Encryption Mode",                    TRUE  },
	{   72, 0x0c23, "Read Class Of Device",                     TRUE  },
	{   73, 0x0c24, "Write Class Of Device",                    TRUE  },
	{   74, 0x0c25, "Read Voice Setting",                       TRUE  },
	{   75, 0x0c26, "Write Voice Setting",                      TRUE  },
	{   76, 0x0c27, "Read Automatic Flush Timeout",             TRUE  },
	{   77, 0x0c28, "Write Automatic Flush Timeout",            TRUE  },
	{   78, 0x0c29, "Read Num Broadcast Retransmissions",       TRUE  },
	{   79, 0x0c2a, "Write Num Broadcast Retransmissions",      TRUE  },
	{   80, 0x0c2b, "Read Hold Mode Activity",                  TRUE  },
	{   81, 0x0c2c, "Write Hold Mode Activity",                 TRUE  },
	{   82, 0x0c2d, "Read Transmit Power Level",                TRUE  },
	{   83, 0x0c2e, "Read Synchronous Flow Control Enable",     TRUE  },
	{   84, 0x0c2f, "Write Synchronous Flow Control Enable",    TRUE  },
	{   85, 0x0c31, "Set Host Controller To Host Flow Control", TRUE  },
	{   86, 0x0c33, "Host Buffer Size",                         TRUE  },
	{   87, 0x0c35, "Host Number Of Completed Packets",         FALSE },
	{   88, 0x0c36, "Read Link Supervision Timeout",            TRUE  },
	{   89, 0x0c37, "Write Link Supervision Timeout",           TRUE  },
	{   90, 0x0c38, "Read Number of Supported IAC",             TRUE  },
	{   91, 0x0c39, "Read Current IAC LAP",                     TRUE  },
	{   92, 0x0c3a, "Write Current IAC LAP",                    TRUE  },
	{   93, 0x0c3b, "Read Page Scan Period Mode",               TRUE  },
	{   94, 0x0c3c, "Write Page Scan Period Mode",              TRUE  },
	{   95, 0x0c3d, "Read Page Scan Mode",                      TRUE  },
	{   96, 0x0c3e, "Write Page Scan Mode",                     TRUE  },
	{   97, 0x0c3f, "Set AFH Channel Classification",           TRUE  },
	{  100, 0x0c42, "Read Inquiry Scan Type",                   TRUE  },
	{  101, 0x0c43, "Write Inquiry Scan Type",                  TRUE  },
	{  102, 0x0c44, "Read Inquiry Mode",                        TRUE  },
	{  103, 0x0c45, "Write Inquiry Mode",                       TRUE  },
	{  104, 0x0c46, "Read Page Scan Type",                      TRUE  },
	{  105, 0x0c47, "Write Page Scan Type",                     TRUE  },
	{  106, 0x0c48, "Read AFH Channel Assessment Mode",         TRUE  },
	{  107, 0x0c49, "Write AFH Channel Assessment Mode",        TRUE  },
	{  115, 0x1001, "Read Local Version Information",           TRUE  },
	{  116, 0x1002, "Read Local Supported Commands",            TRUE  },
	{  117, 0x1003, "Read Local Supported Features",            TRUE  },
	{  118, 0x1004, "Read Local Extended Features",             TRUE  },
	{  119, 0x1005, "Read Buffer Size",                         TRUE  },
	{  120, 0x1007, "Read Country Code",                        TRUE  },
	{  121, 0x1009, "Read BD ADDR",                             TRUE  },
	{  122, 0x1401, "Read Failed Contact Counter",              TRUE  },
	{  123, 0x1402, "Reset Failed Contact Counter",             TRUE  },
	{  124, 0x1403, "Get Link Quality",                         TRUE  },
	{  125, 0x1405, "Read RSSI",                                TRUE  },
	{  126, 0x1406, "Read AFH Channel Map",                     TRUE  },
	{  127, 0x1407, "Read Clock",                               TRUE  },
	{  128, 0x1801, "Read Loopback Mode",                       TRUE  },
	{  129, 0x1802, "Write Loopback Mode",                      TRUE  },
	{  130, 0x1803, "Enable Device Under Test Mode",            TRUE  },
	{  131, 0x0428, "Setup Synchronous Connection",             FALSE },
	{  132, 0x0429, "Accept Synchronous Connection",            FALSE },
	{  133, 0x042a, "Reject Synchronous Connection",            FALSE },
	{  136, 0x0c51, "Read Extended Inquiry Response",           TRUE  },
	{  137, 0x0c52, "Write Extended Inquiry Response",          TRUE  },
	{  138, 0x0c53, "Refresh Encryption Key",                   TRUE  },
	{  140, 0x0811, "Sniff Subrating",                          TRUE  },
	{  141, 0x0c55, "Read Simple Pairing Mode",                 TRUE  },
	{  142, 0x0c56, "Write Simple Pairing Mode",                TRUE  },
	{  143, 0x0c57, "Read Local OOB Data",                      TRUE  },
	{  144, 0x0c58, "Read Inquiry Transmit Power Level",        TRUE  },
	{  145, 0x0c59, "Write Inquiry Transmit Power Level",       TRUE  },
	{  146, 0x0c5a, "Read Default Erroneous Data Reporting",    TRUE  },
	{  147, 0x0c5b, "Write Default Erroneous Data Reporting",   TRUE  },
	{  151, 0x042b, "IO Capability Request Reply",              TRUE  },
	{  152, 0x042c, "User Confirmation Request Reply",          TRUE  },
	{  153, 0x042d, "User Confirmation Request Negative Reply", TRUE  },
	{  154, 0x042e, "User Passkey Request Reply",               TRUE  },
	{  155, 0x042f, "User Passkey Request Negative Reply",      TRUE  },
	{  156, 0x0430, "Remote OOB Data Request Reply",            TRUE  },
	{  157, 0x1804, "Write Simple Pairing Debug Mode",          TRUE  },
	{  158, 0x0c5f, "Enhanced Flush",                           FALSE },
	{  159, 0x0433, "Remote OOB Data Request Negative Reply",   TRUE  },
	{  162, 0x0c60, "Send Keypress Notification",               TRUE  },
	{  163, 0x0434, "IO Capabilities Response Negative Reply",  TRUE  },
	{   -1, 0x0000, "Unknown",                                  FALSE },
};

static struct hci_command_entry *get_command_entry(guint16 opcode)
{
	struct hci_command_entry *entry = command_map;

	while (1) {
		if (entry->opcode == opcode || entry->number == -1)
			return entry;
		entry++;
	}

	return NULL;
}

static gchar *inquiry_complete(gpointer data, guint size)
{
	guint8 status = *((guint8 *) data);

	return create_status(NULL, status);
}

static gchar *inquiry_result(gpointer data, guint size)
{
	guint8 num = *((guint8 *) data);

	return create_hex(g_strdup_printf("num %d", num),
						data + 1, size -1);
}

static gchar *command_complete(gpointer data, guint size)
{
	struct hci_command_complete_evt *evt = data;
	struct hci_command_entry *entry;
	guint16 opcode = GUINT16_FROM_LE(evt->opcode);
	guint16 ogf = opcode >> 10;
	guint16 ocf = opcode & 0x03ff;
	guint8 status;
	guint offset = 3;
	gchar *label, *text;

	if (ogf != 0x3f) {
		entry = get_command_entry(opcode);
		label = entry->label;
		if (entry->noevent == TRUE) {
			status = ((guint8 *) data)[offset];
			offset++;
		} else
			label = "<span foreground=\"red\">[command status]</span>";
	} else
		label = "Vendor";

	text = g_strdup_printf(" cmds %d\n%s (0x%02x|0x%04x) plen %d",
				evt->num_cmds, label, ogf, ocf, size - offset);

	if (offset > 3)
		text = create_status(text, status);

	return create_hex(text, data + offset, size - offset);
}

static gchar *command_status(gpointer data, guint size)
{
	struct hci_command_status_evt *evt = data;
	struct hci_command_entry *entry;
	guint16 opcode = GUINT16_FROM_LE(evt->opcode);
	guint16 ogf = opcode >> 10;
	guint16 ocf = opcode & 0x03ff;
	gchar *label, *text;

	if (ogf != 0x3f) {
		entry = get_command_entry(opcode);
		label = entry->label;
		if (entry->noevent == TRUE)
			label = "<span foreground=\"red\">[command complete]</span>";
	} else
		label = "Vendor";

	text = g_strdup_printf(" cmds %d\n%s (0x%02x|0x%04x)",
					evt->num_cmds, label, ogf, ocf);

	return create_status(text, evt->status);
}

static struct hci_event_entry {
	guint8 event;
	gchar *label;
	gchar *(*func)(gpointer data, guint size);
} event_map[] = {
	{ 0x01, "Inquiry Complete",                          inquiry_complete },
	{ 0x02, "Inquiry Result",                            inquiry_result   },
	{ 0x03, "Connection Complete"                         },
	{ 0x04, "Connection Request"                          },
	{ 0x05, "Disconnection Complete"                      },
	{ 0x06, "Authentication Complete"                     },
	{ 0x07, "Remote Name Request Complete"                },
	{ 0x08, "Encryption Change"                           },
	{ 0x09, "Change Connection Link Key Complete"         },
	{ 0x0a, "Master Link Key Complete"                    },
	{ 0x0b, "Read Remote Supported Features Complete"     },
	{ 0x0c, "Read Remote Version Information Complete"    },
	{ 0x0d, "QoS Setup Complete"                          },
	{ 0x0e, "Command Complete",                          command_complete },
	{ 0x0f, "Command Status",                            command_status   },
	{ 0x10, "Hardware Error"                              },
	{ 0x11, "Flush Occurred"                              },
	{ 0x12, "Role Change"                                 },
	{ 0x13, "Number Of Completed Packets"                 },
	{ 0x14, "Mode Change"                                 },
	{ 0x15, "Return Link Keys"                            },
	{ 0x16, "PIN Code Request"                            },
	{ 0x17, "Link Key Request"                            },
	{ 0x18, "Link Key Notification"                       },
	{ 0x19, "Loopback Command"                            },
	{ 0x1a, "Data Buffer Overflow"                        },
	{ 0x1b, "Max Slots Change"                            },
	{ 0x1c, "Read Clock Offset Complete"                  },
	{ 0x1d, "Connection Packet Type Changed"              },
	{ 0x1e, "QoS Violation"                               },
	{ 0x1f, "Page Scan Mode Change"                       },
	{ 0x20, "Page Scan Repetition Mode Change"            },
	{ 0x21, "Flow Specification Complete"                 },
	{ 0x22, "Inquiry Result with RSSI"                    },
	{ 0x23, "Read Remote Extended Features Complete"      },
	{ 0x2c, "Synchronous Connection Complete"             },
	{ 0x2d, "Synchronous Connection Changed"              },
	{ 0x2e, "Sniff Subrating"                             },
	{ 0x2f, "Extended Inquiry Result"                     },
	{ 0x30, "Encryption Key Refresh Complete"             },
	{ 0x31, "IO Capability Request"                       },
	{ 0x32, "IO Capability Response"                      },
	{ 0x33, "User Confirmation Request"                   },
	{ 0x34, "User Passkey Request"                        },
	{ 0x35, "Remote OOB Data Request"                     },
	{ 0x36, "Simple Pairing Complete"                     },
	{ 0x38, "Link Supervision Timeout Changed"            },
	{ 0x39, "Enhanced Flush Occurred"                     },
	{ 0x3b, "User Passkey Notification"                   },
	{ 0x3c, "Keypress Notification"                       },
	{ 0x3d, "Remote Host Supported Features Notification" },
	{ 0xff, "Vendor"                                      },
	{ 0x00, "Unknown"                                     },
};

static struct hci_event_entry *get_event_entry(guint8 event)
{
	struct hci_event_entry *entry = event_map;

	while (1) {
		if (entry->event == event || entry->event == 0x00)
			return entry;
		entry++;
	}

	return NULL;
}

static gchar *decode_command(gpointer data, guint size)
{
	struct hci_command_hdr *hdr = data;
	struct hci_command_entry *entry;
	guint16 opcode = GUINT16_FROM_LE(hdr->opcode);
	guint16 ogf = opcode >> 10;
	guint16 ocf = opcode & 0x03ff;
	gchar *text, *label;

	if (ogf != 0x3f) {
		entry = get_command_entry(opcode);
		label = entry->label;
	} else
		label = "Vendor";

	text = g_strdup_printf("%s (0x%02x|0x%04x) plen %d", label,
							ogf, ocf, hdr->plen);

	return create_hex(text, data + HCI_COMMAND_HDR_SIZE,
						size - HCI_COMMAND_HDR_SIZE);
}

static gchar *decode_event(gpointer data, guint size)
{
	struct hci_event_hdr *hdr = data;
	struct hci_event_entry *entry;
	gchar *text;

	entry = get_event_entry(hdr->evt);

	text = g_strdup_printf("%s (0x%02x) plen %d", entry->label,
							hdr->evt, hdr->plen);

	if (entry->func) {
		gchar *tmp = text;
		gchar *str = entry->func(data + HCI_EVENT_HDR_SIZE,
						size - HCI_EVENT_HDR_SIZE);
		if (str) {
			text = g_strconcat(tmp, str, NULL);
			g_free(str);
			g_free(tmp);
		}
	} else if (hdr->plen > 0)
		text = create_hex(text, data + HCI_EVENT_HDR_SIZE,
						size - HCI_EVENT_HDR_SIZE);

	return text;
}

static gchar *pb2str(guint8 pb)
{
	switch (pb) {
	case 0x00:
		return "First non-flushable";
	case 0x01:
		return "Continuation";
	case 0x02:
		return "First flushable";
	case 0x03:
		return "Reserved";
	default:
		return "Unknown";
	}
}

static gchar *decode_acldata(gpointer data, guint size)
{
	struct hci_acldata_hdr *hdr = data;
	guint16 handle = GUINT16_FROM_LE(hdr->handle);
	guint16 dlen = GUINT16_FROM_LE(hdr->dlen);
	guint8 pb = (handle & 0x3000) >> 12;
	guint8 bc = (handle & 0xc000) >> 14;
	gchar *text;

	handle &= 0x0fff;

	text = g_strdup_printf("%s (0x%x) handle %d broadcast 0x%x dlen %d",
					pb2str(pb), pb, handle, bc, dlen);

	text = check_packet_size(text, dlen, size - HCI_ACLDATA_HDR_SIZE);

	return create_hex(text, data + HCI_ACLDATA_HDR_SIZE,
						size - HCI_ACLDATA_HDR_SIZE);
}

gchar *decode_hci(guint type, gpointer data, guint size)
{
	gchar *text;

	switch (type) {
	case 0x00:
		text = decode_command(data, size);
		break;
	case 0x01:
		text = decode_event(data, size);
		break;
	case 0x02:
	case 0x03:
		text = decode_acldata(data, size);
		break;
	case 0xff:
		text = g_strndup(data, size);
		break;
	default:
		text = create_hex(NULL, data, size);
		break;
	}

	return text;
}
