/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *  Copyright (C) 2006-2007  Bastien Nocera <hadess@hadess.net>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <dbus/dbus-glib.h>

#ifdef HAVE_HAL
#include <dbus/dbus-glib-lowlevel.h>
#include <hal/libhal.h>
#endif

#include <gconf/gconf-client.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#ifdef HAVE_LIBNOTIFY
#include <libnotify/notify.h>
#endif

#include "bluetooth-device-selection.h"

#define PASSKEY_AGENT_PATH	"/org/bluez/passkey"
#define AUTH_AGENT_PATH		"/org/bluez/auth"

static int volatile registered_passkey	= 0;
static int volatile registered_auth	= 0;

static DBusGConnection *conn;

static gboolean use_hal = FALSE;

static GList *adapter_list = NULL;

struct adapter_data {
	char *path;
	int attached;
	char *old_mode;
};

typedef enum {
	ICON_POLICY_NEVER,
	ICON_POLICY_ALWAYS,
	ICON_POLICY_PRESENT,
} EnumIconPolicy;

static int icon_policy = ICON_POLICY_PRESENT;

static gboolean auto_authorize = FALSE;

#define PREF_DIR		"/apps/bluetooth-manager"
#define PREF_USE_HAL		PREF_DIR "/use_hal"
#define PREF_ICON_POLICY	PREF_DIR "/icon_policy"
#define PREF_AUTO_AUTHORIZE	PREF_DIR "/auto_authorize"

static GConfClient* gconf;

static GtkStatusIcon *statusicon = NULL;

static GtkWidget *menuitem_browse = NULL;

typedef enum {
	AGENT_ERROR_REJECT
} AgentError;

#define AGENT_ERROR (agent_error_quark())

#define AGENT_ERROR_TYPE (agent_error_get_type()) 

static GQuark agent_error_quark(void)
{
	static GQuark quark = 0;
	if (!quark)
		quark = g_quark_from_static_string("agent");

	return quark;
}

#define ENUM_ENTRY(NAME, DESC) { NAME, "" #NAME "", DESC }

static GType agent_error_get_type(void)
{
	static GType etype = 0;
	if (etype == 0) {
		static const GEnumValue values[] = {
			ENUM_ENTRY(AGENT_ERROR_REJECT, "Rejected"),
			{ 0, 0, 0 }
		};

		etype = g_enum_register_static("agent", values);
	}

	return etype;
}

static GList *input_list = NULL;

struct input_data {
	char *path;
	char *address;
	char *service;
	char *uuid;
	DBusGMethodInvocation *context;
	GtkWidget *dialog;
	GtkWidget *button;
	GtkWidget *entry;
};

static gint input_compare(gconstpointer a, gconstpointer b)
{
	struct input_data *a_data = (struct input_data *) a;
	struct input_data *b_data = (struct input_data *) b;

	return strcmp(a_data->address, b_data->address);
}

static void input_free(struct input_data *input)
{
	gtk_widget_destroy(input->dialog);

	input_list = g_list_remove(input_list, input);

	g_free(input->uuid);
	g_free(input->service);
	g_free(input->address);
	g_free(input->path);
	g_free(input);

	if (g_list_length(input_list) == 0)
		gtk_status_icon_set_blinking(statusicon, FALSE);
}

static void passkey_callback(GtkWidget *dialog,
				gint response, gpointer user_data)
{
	struct input_data *input = user_data;

	if (response == GTK_RESPONSE_ACCEPT) {
		const char *passkey;
		passkey = gtk_entry_get_text(GTK_ENTRY(input->entry));
		dbus_g_method_return(input->context, passkey);
	} else {
		GError *error;
		error = g_error_new(AGENT_ERROR, AGENT_ERROR_REJECT,
						"Pairing request rejected");
		dbus_g_method_return_error(input->context, error);
	}

	input_free(input);
}

static void confirm_callback(GtkWidget *dialog,
				gint response, gpointer user_data)
{
	struct input_data *input = user_data;

	if (response != GTK_RESPONSE_YES) {
		GError *error;
		error = g_error_new(AGENT_ERROR, AGENT_ERROR_REJECT,
					"Confirmation request rejected");
		dbus_g_method_return_error(input->context, error);
	} else
		dbus_g_method_return(input->context);

	input_free(input);
}

static void set_trusted(struct input_data *input)
{
	DBusGProxy *object;
	gboolean active;

	active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(input->button));
	if (active == FALSE)
		return;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					input->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "SetTrusted", NULL,
				G_TYPE_STRING, input->address, G_TYPE_INVALID,
								G_TYPE_INVALID);
}

static void auth_callback(GtkWidget *dialog,
				gint response, gpointer user_data)
{
	struct input_data *input = user_data;

	if (response == GTK_RESPONSE_YES) {
		set_trusted(input);
		dbus_g_method_return(input->context);
	} else {
		GError *error;
		error = g_error_new(AGENT_ERROR, AGENT_ERROR_REJECT,
					"Authorization request rejected");
		dbus_g_method_return_error(input->context, error);
	}

	input_free(input);
}

static void changed_callback(GtkWidget *editable, gpointer user_data)
{
	struct input_data *input = user_data;
	const gchar *text;

	text = gtk_entry_get_text(GTK_ENTRY(input->entry));

	gtk_widget_set_sensitive(input->button, strlen(text) ? TRUE : FALSE);
}

static void toggled_callback(GtkWidget *button, gpointer user_data)
{
	struct input_data *input = user_data;
	gboolean mode;

	mode = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));

	gtk_entry_set_visibility(GTK_ENTRY(input->entry), mode);
}

static void passkey_dialog(const char *path, const char *address,
			const gchar *device, DBusGMethodInvocation *context)
{
	GtkWidget *dialog;
	GtkWidget *button;
	GtkWidget *image;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *table;
	GtkWidget *vbox;
	struct input_data *input;
	gchar *markup;

	input = g_try_malloc0(sizeof(*input));
	if (!input)
		return;

	input->path = g_strdup(path);
	input->address = g_strdup(address);

	input->context = context;

	dialog = gtk_dialog_new();

	gtk_window_set_title(GTK_WINDOW(dialog), _("Authentication request"));

	gtk_window_set_icon_name(GTK_WINDOW(dialog), "stock_bluetooth");

	gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);

	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);

	gtk_window_set_urgency_hint(GTK_WINDOW(dialog), TRUE);

	gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);

	input->dialog = dialog;

	button = gtk_dialog_add_button(GTK_DIALOG(dialog),
				GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT);

	button = gtk_dialog_add_button(GTK_DIALOG(dialog),
					GTK_STOCK_OK, GTK_RESPONSE_ACCEPT);

	gtk_widget_grab_default(button);

	gtk_widget_set_sensitive(button, FALSE);

	input->button = button;

	table = gtk_table_new(5, 2, FALSE);

	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 20);

	gtk_container_set_border_width(GTK_CONTAINER(table), 12);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);

	image = gtk_image_new_from_icon_name(GTK_STOCK_DIALOG_AUTHENTICATION,
							GTK_ICON_SIZE_DIALOG);

	gtk_misc_set_alignment(GTK_MISC(image), 0.0, 0.0);

	gtk_table_attach(GTK_TABLE(table), image, 0, 1, 0, 5,
						GTK_SHRINK, GTK_FILL, 0, 0);

	vbox = gtk_vbox_new(FALSE, 6);

	label = gtk_label_new(_("Pairing request for device:"));

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_container_add(GTK_CONTAINER(vbox), label);

	gtk_table_attach(GTK_TABLE(table), vbox, 1, 2, 0, 1,
				GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	label = gtk_label_new(NULL);

	markup = g_strdup_printf("<b>%s</b>", device);
	gtk_label_set_markup(GTK_LABEL(label), markup);
	g_free(markup);

	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

	gtk_label_set_selectable(GTK_LABEL(label), TRUE);

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_widget_set_size_request(GTK_WIDGET(label), 280, -1);

	gtk_container_add(GTK_CONTAINER(vbox), label);

	vbox = gtk_vbox_new(FALSE, 6);

	label = gtk_label_new(_("Enter passkey for authentication:"));

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_container_add(GTK_CONTAINER(vbox), label);

	gtk_table_attach(GTK_TABLE(table), vbox, 1, 2, 2, 3,
				GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	entry = gtk_entry_new();

	gtk_entry_set_max_length(GTK_ENTRY(entry), 16);

	gtk_entry_set_activates_default(GTK_ENTRY(entry), TRUE);

	gtk_entry_set_visibility(GTK_ENTRY(entry), FALSE);

	input->entry = entry;

	g_signal_connect(G_OBJECT(entry), "changed",
				G_CALLBACK(changed_callback), input);

	gtk_container_add(GTK_CONTAINER(vbox), entry);

	button = gtk_check_button_new_with_label(_("Show input"));

	g_signal_connect(G_OBJECT(button), "toggled",
				G_CALLBACK(toggled_callback), input);

	gtk_container_add(GTK_CONTAINER(vbox), button);

	input_list = g_list_append(input_list, input);

	g_signal_connect(G_OBJECT(dialog), "response",
				G_CALLBACK(passkey_callback), input);

	gtk_status_icon_set_blinking(statusicon, TRUE);
}

static void confirm_dialog(const char *path, const char *address,
				const char *value, const gchar *device,
					DBusGMethodInvocation *context)
{
	GtkWidget *dialog;
	GtkWidget *button;
	GtkWidget *image;
	GtkWidget *label;
	GtkWidget *table;
	GtkWidget *vbox;
	gchar *markup;
	struct input_data *input;

	input = g_try_malloc0(sizeof(*input));
	if (!input)
		return;

	input->path = g_strdup(path);
	input->address = g_strdup(address);

	input->context = context;

	dialog = gtk_dialog_new();

	gtk_window_set_title(GTK_WINDOW(dialog), _("Confirmation request"));

	gtk_window_set_icon_name(GTK_WINDOW(dialog), "stock_bluetooth");

	gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);

	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);

	gtk_window_set_urgency_hint(GTK_WINDOW(dialog), TRUE);

	gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);

	input->dialog = dialog;

	button = gtk_dialog_add_button(GTK_DIALOG(dialog),
					GTK_STOCK_NO, GTK_RESPONSE_NO);
 
	button = gtk_dialog_add_button(GTK_DIALOG(dialog),
					GTK_STOCK_YES, GTK_RESPONSE_YES);

	table = gtk_table_new(5, 2, FALSE);

	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 20);

	gtk_container_set_border_width(GTK_CONTAINER(table), 12);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);

	image = gtk_image_new_from_icon_name(GTK_STOCK_DIALOG_AUTHENTICATION,
							GTK_ICON_SIZE_DIALOG);

	gtk_misc_set_alignment(GTK_MISC(image), 0.0, 0.0);

	gtk_table_attach(GTK_TABLE(table), image, 0, 1, 0, 5,
						GTK_SHRINK, GTK_FILL, 0, 0);

	vbox = gtk_vbox_new(FALSE, 6);
	label = gtk_label_new(_("Pairing request for device:"));

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_container_add(GTK_CONTAINER(vbox), label);

	gtk_table_attach(GTK_TABLE(table), vbox, 1, 2, 0, 1,
				GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	label = gtk_label_new(NULL);

	markup = g_strdup_printf("<b>%s</b>", device);
	gtk_label_set_markup(GTK_LABEL(label), markup);
	g_free(markup);

	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

	gtk_label_set_selectable(GTK_LABEL(label), TRUE);

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_widget_set_size_request(GTK_WIDGET(label), 280, -1);

	gtk_container_add(GTK_CONTAINER(vbox), label);

	vbox = gtk_vbox_new(FALSE, 6);

	label = gtk_label_new(_("Confirm value for authentication:"));

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_container_add(GTK_CONTAINER(vbox), label);

	gtk_table_attach(GTK_TABLE(table), vbox, 1, 2, 2, 3,
				GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	label = gtk_label_new(NULL);

	markup = g_strdup_printf("<b>%s</b>\n", value);

	gtk_label_set_markup(GTK_LABEL(label), markup);

	g_free(markup);

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_container_add(GTK_CONTAINER(vbox), label);

	input_list = g_list_append(input_list, input);

	g_signal_connect(G_OBJECT(dialog), "response",
				G_CALLBACK(confirm_callback), input);

	gtk_status_icon_set_blinking(statusicon, TRUE);
}

static void auth_dialog(const char *path, const char *address,
		const char *service, const char *uuid, const gchar *device,
		const gchar *profile, DBusGMethodInvocation *context)
{
	GtkWidget *dialog;
	GtkWidget *button;
	GtkWidget *image;
	GtkWidget *label;
	GtkWidget *table;
	GtkWidget *vbox;
	gchar *markup, *text;
	struct input_data *input;

	input = g_try_malloc0(sizeof(*input));
	if (!input)
		return;

	input->path = g_strdup(path);
	input->address = g_strdup(address);
	input->service = g_strdup(service);
	input->uuid = g_strdup(uuid);

	input->context = context;

	dialog = gtk_dialog_new();

	gtk_window_set_title(GTK_WINDOW(dialog), _("Authorization request"));

	gtk_window_set_icon_name(GTK_WINDOW(dialog), "stock_bluetooth");

	gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);

	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);

	gtk_window_set_urgency_hint(GTK_WINDOW(dialog), TRUE);

	gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);

	input->dialog = dialog;

	button = gtk_dialog_add_button(GTK_DIALOG(dialog),
					GTK_STOCK_NO, GTK_RESPONSE_NO);
 
	button = gtk_dialog_add_button(GTK_DIALOG(dialog),
					GTK_STOCK_YES, GTK_RESPONSE_YES);

	table = gtk_table_new(5, 2, FALSE);

	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 20);

	gtk_container_set_border_width(GTK_CONTAINER(table), 12);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);

	image = gtk_image_new_from_icon_name(GTK_STOCK_DIALOG_AUTHENTICATION,
							GTK_ICON_SIZE_DIALOG);

	gtk_misc_set_alignment(GTK_MISC(image), 0.0, 0.0);

	gtk_table_attach(GTK_TABLE(table), image, 0, 1, 0, 5,
						GTK_SHRINK, GTK_FILL, 0, 0);

	vbox = gtk_vbox_new(FALSE, 6);

	label = gtk_label_new(_("Authorization request for device:"));

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1,
				GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	label = gtk_label_new(NULL);

	markup = g_strdup_printf("<b>%s</b>", device);
	gtk_label_set_markup(GTK_LABEL(label), markup);
	g_free(markup);

	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

	gtk_label_set_selectable(GTK_LABEL(label), TRUE);

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_widget_set_size_request(GTK_WIDGET(label), 280, -1);

	gtk_container_add(GTK_CONTAINER(vbox), label);

	gtk_table_attach(GTK_TABLE(table), vbox, 1, 2, 2, 3,
				GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	label = gtk_label_new(NULL);

	/* translators: Whether to grant access to a particular service
	 * to the device mentioned */
	markup = g_strdup_printf("<i>%s</i>", profile);
	text = g_strdup_printf(_("Grant access to %s?"), markup);
	g_free(markup);

	markup = g_strdup_printf("%s\n", text);
	gtk_label_set_markup(GTK_LABEL(label), markup);
	g_free(markup);

	g_free(text);

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_container_add(GTK_CONTAINER(vbox), label);

	button = gtk_check_button_new_with_label(_("Always grant access"));

	input->button = button;

	gtk_container_add(GTK_CONTAINER(vbox), button);

	input_list = g_list_append(input_list, input);

	g_signal_connect(G_OBJECT(dialog), "response",
				G_CALLBACK(auth_callback), input);

	gtk_status_icon_set_blinking(statusicon, TRUE);
}

static void show_dialog(gpointer data, gpointer user_data)
{
	struct input_data *input = data;

	gtk_widget_show_all(input->dialog);

	gtk_window_present(GTK_WINDOW(input->dialog));
}

#ifdef HAVE_LIBNOTIFY
static NotifyNotification *notify = NULL;

static void notify_action(NotifyNotification *notify,
					gchar *action, gpointer user_data)
{
}
#endif

static void show_notification(const gchar *summary, const gchar *message,
					gint timeout, GCallback handler)
{
#ifdef HAVE_LIBNOTIFY
	NotifyActionCallback callback;
	GdkScreen *screen;
	GdkRectangle area;

	if (notify) {
		g_signal_handlers_destroy(notify);
		notify_notification_close(notify, NULL);
	}

	notify = notify_notification_new(summary, message,
						"stock_bluetooth", NULL);

	notify_notification_set_timeout(notify, timeout);

	if (gtk_status_icon_get_visible(statusicon) == TRUE) {
		gtk_status_icon_get_geometry(statusicon, &screen, &area, NULL);

		notify_notification_set_hint_int32(notify,
					"x", area.x + area.width / 2);
		notify_notification_set_hint_int32(notify,
					"y", area.y + area.height / 2);
	}

	notify_notification_set_urgency(notify, NOTIFY_URGENCY_NORMAL);

	callback = handler ? NOTIFY_ACTION_CALLBACK(handler) : notify_action;

	notify_notification_add_action(notify, "default", "action",
						callback, NULL, NULL);

	notify_notification_show(notify, NULL);
#endif
}

static void close_notification(void)
{
#ifdef HAVE_LIBNOTIFY
	if (notify) {
		g_signal_handlers_destroy(notify);
		notify_notification_close(notify, NULL);
		notify = NULL;
	}
#endif
}

typedef struct {
	GObject parent;
} PasskeyAgent;

typedef struct {
	GObjectClass parent;
} PasskeyAgentClass;

static GObjectClass *passkey_agent_parent;

G_DEFINE_TYPE(PasskeyAgent, passkey_agent, G_TYPE_OBJECT)

#define PASSKEY_AGENT_OBJECT_TYPE (passkey_agent_get_type())

#define PASSKEY_AGENT(obj) (G_TYPE_CHECK_INSTANCE_CAST((obj), \
				PASSKEY_AGENT_OBJECT_TYPE, PasskeyAgent))

static void passkey_agent_finalize(GObject *obj)
{
	passkey_agent_parent->finalize(obj);
}

static void passkey_agent_init(PasskeyAgent *obj)
{
}

static void passkey_agent_class_init(PasskeyAgentClass *klass)
{
	GObjectClass *gobject_class;

	passkey_agent_parent = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	gobject_class->finalize = passkey_agent_finalize;
}

static PasskeyAgent *passkey_agent_new(const char *path)
{
	PasskeyAgent *agent;

	agent = g_object_new(PASSKEY_AGENT_OBJECT_TYPE, NULL);

	dbus_g_connection_register_g_object(conn, path, G_OBJECT(agent));

	return agent;
}

static void notification_closed(GObject *object, gpointer user_data)
{
	g_list_foreach(input_list, show_dialog, NULL);

	gtk_status_icon_set_blinking(statusicon, FALSE);
}

static gboolean passkey_agent_request(PasskeyAgent *agent,
				const char *path, const char *address,
					DBusGMethodInvocation *context)
{
	DBusGProxy *object;
	const char *adapter = NULL, *name = NULL;
	gchar *device, *text, *line;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
						path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetRemoteName", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	if (name) {
		if (g_strrstr(name, address))
			device = g_strdup(name);
		else
			device = g_strdup_printf("%s (%s)", name, address);
	} else
		device = g_strdup(address);

	passkey_dialog(path, address, device, context);

	/* translators: this is a popup telling you a particular device
	 * has asked for pairing */
	line = g_strdup_printf(_("Pairing request for '%s'"), device);
	text = g_strconcat(line, "\n\n",
			_("Click to open passkey entry dialog"), NULL);
	g_free(line);
	g_free(device);

	show_notification(adapter ? adapter : _("Bluetooth device"),
				text, 0, G_CALLBACK(notification_closed));

	g_free(text);

	return TRUE;
}

static gboolean passkey_agent_confirm(PasskeyAgent *agent,
			const char *path, const char *address,
			const char *value, DBusGMethodInvocation *context)
{
	DBusGProxy *object;
	const char *adapter = NULL, *name = NULL;
	gchar *device, *text, *line;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
						path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetRemoteName", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	if (name) {
		if (g_strrstr(name, address))
			device = g_strdup(name);
		else
			device = g_strdup_printf("%s (%s)", name, address);
	} else
		device = g_strdup(address);

	confirm_dialog(path, address, value, device, context);

	line = g_strdup_printf(_("Pairing request for '%s'"), device);
	text = g_strconcat(line, "\n\n",
			_("Click to open confirmation dialog"), NULL);
	g_free(line);
	g_free(device);

	show_notification(adapter ? adapter : _("Bluetooth device"),
				text, 0, G_CALLBACK(notification_closed));

	g_free(text);

	return TRUE;
}

static gboolean passkey_agent_cancel(PasskeyAgent *agent,
			const char *path, const char *address, GError **error)
{
	GList *list;
	GError *result;
	struct input_data *input;

	input = g_try_malloc0(sizeof(*input));
	if (!input)
		return FALSE;

	input->path = g_strdup(path);
	input->address = g_strdup(address);

	list = g_list_find_custom(input_list, input, input_compare);

	g_free(input->address);
	g_free(input->path);
	g_free(input);

	if (!list || !list->data)
		return FALSE;

	input = list->data;

	close_notification();

	result = g_error_new(AGENT_ERROR, AGENT_ERROR_REJECT,
						"Agent callback canceled");

	dbus_g_method_return_error(input->context, result);

	input_free(input);

	return TRUE;
}

static gboolean passkey_agent_release(PasskeyAgent *agent, GError **error)
{
	registered_passkey = 0;

	return TRUE;
}

#include "passkey-agent-glue.h"

typedef struct {
	GObject parent;
} AuthAgent;

typedef struct {
	GObjectClass parent;
} AuthAgentClass;

static GObjectClass *auth_agent_parent;

G_DEFINE_TYPE(AuthAgent, auth_agent, G_TYPE_OBJECT)

#define AUTH_AGENT_OBJECT_TYPE (auth_agent_get_type())

#define AUTH_AGENT(obj) (G_TYPE_CHECK_INSTANCE_CAST((obj), \
				AUTH_AGENT_OBJECT_TYPE, AuthAgent))

static void auth_agent_finalize(GObject *obj)
{
	auth_agent_parent->finalize(obj);
}

static void auth_agent_init(AuthAgent *obj)
{
}

static void auth_agent_class_init(AuthAgentClass *klass)
{
	GObjectClass *gobject_class;

	auth_agent_parent = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	gobject_class->finalize = auth_agent_finalize;
}

static AuthAgent *auth_agent_new(const char *path)
{
	AuthAgent *agent;

	agent = g_object_new(AUTH_AGENT_OBJECT_TYPE, NULL);

	dbus_g_connection_register_g_object(conn, path, G_OBJECT(agent));

	return agent;
}

static gboolean auth_agent_authorize(PasskeyAgent *agent,
		const char *path, const char *address, const char *service,
			const char *uuid, DBusGMethodInvocation *context)
{
	DBusGProxy *object;
	const char *adapter = NULL, *name = NULL;
	gchar *device, *profile, *text, *line;

	if (auto_authorize == TRUE) {
		dbus_g_method_return(context);
		return TRUE;
	}

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
						path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetRemoteName", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
						service, "org.bluez.Service");

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &profile, G_TYPE_INVALID);

	if (name) {
		if (g_strrstr(name, address))
			device = g_strdup(name);
		else
			device = g_strdup_printf("%s (%s)", name, address);
	} else
		device = g_strdup(address);

	auth_dialog(path, address, service, uuid, device, profile, context);

	line = g_strdup_printf(_("Authorization request for %s"), device);
	text = g_strconcat(line, "\n\n",
			_("Click to open authorization dialog"), NULL);
	g_free(line);
	g_free(device);

	show_notification(adapter ? adapter : _("Bluetooth device"),
				text, 0, G_CALLBACK(notification_closed));

	g_free(text);

	return TRUE;
}

static gboolean auth_agent_cancel(PasskeyAgent *agent,
		const char *path, const char *address, const char *service,
			const char *uuid, DBusGMethodInvocation *context)
{
	GList *list;
	GError *result;
	struct input_data *input;

	input = g_try_malloc0(sizeof(*input));
	if (!input)
		return FALSE;

	input->path = g_strdup(path);
	input->address = g_strdup(address);
	input->service = g_strdup(service);
	input->uuid = g_strdup(uuid);

	list = g_list_find_custom(input_list, input, input_compare);

	g_free(input->uuid);
	g_free(input->service);
	g_free(input->address);
	g_free(input->path);
	g_free(input);

	if (!list || !list->data)
		return FALSE;

	input = list->data;

	close_notification();

	result = g_error_new(AGENT_ERROR, AGENT_ERROR_REJECT,
						"Agent callback canceled");

	dbus_g_method_return_error(input->context, result);

	input_free(input);

	return TRUE;
}

static gboolean auth_agent_release(PasskeyAgent *agent, GError **error)
{
	registered_auth = 0;

	return TRUE;
}

#include "auth-agent-glue.h"

static int register_agents(void)
{
	DBusGProxy *object;
	GError *error = NULL;

	if (registered_passkey && registered_auth)
		return 0;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					"/org/bluez", "org.bluez.Security");

	if (!registered_passkey) {

		dbus_g_proxy_call(object, "RegisterDefaultPasskeyAgent",
				&error, G_TYPE_STRING, PASSKEY_AGENT_PATH,
					G_TYPE_INVALID, G_TYPE_INVALID);

		if (error != NULL) {
			g_error_free(error);
			return -1;
		}

		registered_passkey = 1;
	}

	if (!registered_auth) {
		dbus_g_proxy_call(object, "RegisterDefaultAuthorizationAgent",
				&error,	G_TYPE_STRING, AUTH_AGENT_PATH,
					G_TYPE_INVALID, G_TYPE_INVALID);

		if (error != NULL) {
			g_error_free(error);
			return -1;
		}

		registered_auth = 1;
	}

	return 0;
}

static void bonding_created(DBusGProxy *object,
				const char *address, gpointer user_data)
{
	const char *adapter = NULL, *name = NULL;
	gchar *device, *text;

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetRemoteName", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	if (name) {
		if (g_strrstr(name, address))
			device = g_strdup(name);
		else
			device = g_strdup_printf("%s (%s)", name, address);
	} else
		device = g_strdup(address);

	text = g_strdup_printf(_("Created bonding with %s"), device);

	g_free(device);

	show_notification(adapter ? adapter : _("Bluetooth device"),
							text, 6000, NULL);

	g_free(text);
}

static void bonding_removed(DBusGProxy *object,
				const char *address, gpointer user_data)
{
	const char *adapter = NULL, *name = NULL;
	gchar *device, *text;

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetRemoteName", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	if (name) {
		if (g_strrstr(name, address))
			device = g_strdup(name);
		else
			device = g_strdup_printf("%s (%s)", name, address);
	} else
		device = g_strdup(address);

	text = g_strdup_printf(_("Removed bonding with %s"), device);

	g_free(device);

	show_notification(adapter ? adapter : _("Bluetooth device"),
							text, 6000, NULL);

	g_free(text);
}

static void trust_added(DBusGProxy *object,
				const char *address, gpointer user_data)
{
}

static void trust_removed(DBusGProxy *object,
				const char *address, gpointer user_data)
{
}

static void set_new_mode(struct adapter_data *adapter, const char *mode)
{
	g_free(adapter->old_mode);

	adapter->old_mode = g_strdup(mode);
}

static void mode_changed(DBusGProxy *object,
				const char *mode, gpointer user_data)
{
	struct adapter_data *adapter = (struct adapter_data *) user_data;
	const char *adapter_name = NULL;
	const char *text;

	if (icon_policy == ICON_POLICY_PRESENT) {
		if (g_str_equal(mode, "off") == TRUE) {
			set_new_mode(adapter, mode);
			return;
		}
		if (g_str_equal(adapter->old_mode, "off")
				&& g_str_equal(mode, "connectable")) {
			set_new_mode(adapter, mode);
			return;
		}
	}

	if (g_str_equal(mode, "off") != FALSE) {
		text = N_("Device has been switched off");
	} else if (g_str_equal(mode, "connectable") != FALSE
		   && g_str_equal(adapter->old_mode, "discoverable") != FALSE) {
		text = N_("Device has been made non-discoverable");
	} else if (g_str_equal(mode, "connectable") != FALSE) {
		text = N_("Device has been made connectable");
	} else if (g_str_equal (mode, "discoverable") != FALSE) {
		text = N_("Device has been made discoverable");
	} else if (g_str_equal(mode, "limited") != FALSE) {
		text = N_("Device has been made limited discoverable");
	} else if (g_str_equal(mode, "pairing") != FALSE) {
		text = N_("Device has been switched into pairing mode");
	} else {
		set_new_mode(adapter, mode);
		return;
	}

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter_name, G_TYPE_INVALID);

	show_notification(adapter_name ? adapter_name : _("Bluetooth device"),
							_(text), 3000, NULL);

	set_new_mode(adapter, mode);
}

static void adapter_free(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	adapter_list = g_list_remove(adapter_list, adapter);

	g_free(adapter->path);
	g_free(adapter->old_mode);
	g_free(adapter);
}

static void adapter_disable(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	adapter->attached = 0;
}

static gint adapter_compare(gconstpointer a, gconstpointer b)
{
	const struct adapter_data *adapter = a;
	const char *path = b;

	return strcmp(adapter->path, path);
}

static void adapter_count(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;
	int *count = user_data;

	if (adapter->attached)
		(*count)++;
}

static int attached_adapters(void)
{
	int count = 0;

	g_list_foreach(adapter_list, adapter_count, &count);

	return count;
}

#ifdef HAVE_HAL
static char *get_form_factor(void)
{
	LibHalContext *ctx;
	char *formfactor;

	ctx = libhal_ctx_new();
	if (ctx == NULL)
		return NULL;

	if (libhal_ctx_set_dbus_connection(ctx,
			dbus_g_connection_get_connection(conn)) == FALSE) {
		libhal_ctx_free(ctx);
		return NULL;
	}

	if (libhal_ctx_init(ctx, NULL) == FALSE) {
		g_warning("Unable to init HAL context");
		libhal_ctx_free(ctx);
		return NULL;
	}

	formfactor = libhal_device_get_property_string(ctx,
				"/org/freedesktop/Hal/devices/computer",
						"system.formfactor", NULL);

	libhal_ctx_shutdown(ctx, NULL);
	libhal_ctx_free(ctx);

	if (strcmp(formfactor, "laptop") && strcmp(formfactor, "desktop")) {
		/* We don't know anything else */
		g_free(formfactor);
		formfactor = g_strdup("uncategorized");
	}

	return formfactor;
}

static void adapter_setup(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;
	const char *formfactor = user_data;
	DBusGProxy *object;

	if (formfactor == NULL)
		return;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					adapter->path, "org.bluez.Adapter");
	if (!object)
		return;

	dbus_g_proxy_call(object, "SetMajorClass", NULL,
		G_TYPE_STRING, "computer", G_TYPE_INVALID, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "SetMinorClass", NULL,
		G_TYPE_STRING, formfactor, G_TYPE_INVALID, G_TYPE_INVALID);
}
#endif

static void add_adapter(const char *path)
{
	GList *list;
	DBusGProxy *object;
	struct adapter_data *adapter;
	const char *old_mode;

	if (icon_policy != ICON_POLICY_NEVER)
		gtk_status_icon_set_visible(statusicon, TRUE);

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;

		adapter->attached = 1;
		return;
	}

	adapter = g_try_malloc0(sizeof(*adapter));
	if (!adapter)
		return;

	adapter->path = g_strdup(path);
	adapter->attached = 1;

	adapter_list = g_list_append(adapter_list, adapter);

#ifdef HAVE_HAL
	if (use_hal == TRUE) {
		char *formfactor;

		formfactor = get_form_factor();
		adapter_setup(adapter, formfactor);
		g_free(formfactor);
	}
#endif

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
						path, "org.bluez.Adapter");

	dbus_g_proxy_add_signal(object, "ModeChanged",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "ModeChanged",
				G_CALLBACK(mode_changed), adapter, NULL);

	dbus_g_proxy_add_signal(object, "BondingCreated",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "BondingCreated",
				G_CALLBACK(bonding_created), NULL, NULL);

	dbus_g_proxy_add_signal(object, "BondingRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "BondingRemoved",
				G_CALLBACK(bonding_removed), NULL, NULL);

	dbus_g_proxy_add_signal(object, "TrustAdded",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "TrustAdded",
				G_CALLBACK(trust_added), NULL, NULL);

	dbus_g_proxy_add_signal(object, "TrustRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "TrustRemoved",
				G_CALLBACK(trust_removed), NULL, NULL);

	old_mode = NULL;
	dbus_g_proxy_call(object, "GetMode", NULL,
			  G_TYPE_INVALID, G_TYPE_STRING,
			  &old_mode, G_TYPE_INVALID);
	if (old_mode != NULL)
		set_new_mode(adapter, old_mode);
}

static void adapter_added(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	register_agents();

	add_adapter(path);
}

static void adapter_removed(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	GList *list;

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;

		adapter->attached = 0;
	}

	if (icon_policy != ICON_POLICY_ALWAYS && attached_adapters() == 0)
		gtk_status_icon_set_visible(statusicon, FALSE);
}

static int setup_manager(void)
{
	DBusGProxy *object;
	GError *error = NULL;
	const gchar **array = NULL;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					"/org/bluez", "org.bluez.Manager");

	dbus_g_proxy_add_signal(object, "AdapterAdded",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "AdapterAdded",
				G_CALLBACK(adapter_added), NULL, NULL);

	dbus_g_proxy_add_signal(object, "AdapterRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "AdapterRemoved",
				G_CALLBACK(adapter_removed), NULL, NULL);

	dbus_g_proxy_call(object, "ListAdapters", &error,
			G_TYPE_INVALID,	G_TYPE_STRV, &array, G_TYPE_INVALID);

	if (error == NULL) {
		while (*array) {
			add_adapter(*array);
			array++;
		}
	} else
		g_error_free(error);

	return 0;
}

static void name_owner_changed(DBusGProxy *object, const char *name,
			const char *prev, const char *new, gpointer user_data)
{
	if (!strcmp(name, "org.bluez") && *new == '\0') {
		registered_passkey = 0;
		registered_auth = 0;

		g_list_foreach(adapter_list, adapter_disable, NULL);

		if (icon_policy != ICON_POLICY_ALWAYS)
			gtk_status_icon_set_visible(statusicon, FALSE);
	}
}

static gboolean obexftp_available(void)
{
	gchar *path;

	path = g_find_program_in_path("nautilus");
	if (path == NULL)
		return FALSE;

	g_free(path);

	return TRUE;
}

static int setup_dbus(void)
{
	DBusGProxy *object;
	void *agent;

	object = dbus_g_proxy_new_for_name(conn, DBUS_SERVICE_DBUS,
					DBUS_PATH_DBUS, DBUS_INTERFACE_DBUS);

	dbus_g_proxy_add_signal(object, "NameOwnerChanged",
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "NameOwnerChanged",
				G_CALLBACK(name_owner_changed), NULL, NULL);

	dbus_g_object_type_install_info(PASSKEY_AGENT_OBJECT_TYPE,
					&dbus_glib_passkey_agent_object_info);

	dbus_g_object_type_install_info(AUTH_AGENT_OBJECT_TYPE,
					&dbus_glib_auth_agent_object_info);

	dbus_g_error_domain_register(AGENT_ERROR, "org.bluez.Error",
							AGENT_ERROR_TYPE);

	agent = passkey_agent_new(PASSKEY_AGENT_PATH);

	agent = auth_agent_new(AUTH_AGENT_PATH);

	return 0;
}

static void close_callback(GtkWidget *dialog, gpointer user_data)
{
	gtk_widget_destroy(dialog);
}

static void about_callback(GtkWidget *item, gpointer user_data)
{
	const gchar *authors[] = {
		"Marcel Holtmann <marcel@holtmann.org>",
		"Bastien Nocera <hadess@hadess.net>",
		NULL
	};
	GtkWidget *dialog;

	dialog = gtk_about_dialog_new();

	gtk_window_set_icon_name(GTK_WINDOW(dialog), "stock_bluetooth");

	gtk_about_dialog_set_name(GTK_ABOUT_DIALOG(dialog),
						_("Bluetooth Applet"));

	gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(dialog), VERSION);

	gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(dialog),
			"Copyright \xc2\xa9 2005-2007 Marcel Holtmann");

	gtk_about_dialog_set_comments(GTK_ABOUT_DIALOG(dialog),
			_("A Bluetooth manager for the GNOME desktop"));

	gtk_about_dialog_set_logo_icon_name(GTK_ABOUT_DIALOG(dialog),
							"stock_bluetooth");

	gtk_about_dialog_set_website(GTK_ABOUT_DIALOG(dialog),
							"www.bluez.org");

	gtk_about_dialog_set_authors(GTK_ABOUT_DIALOG(dialog), authors);

	gtk_about_dialog_set_translator_credits(GTK_ABOUT_DIALOG(dialog),
						_("translator-credits"));

	g_signal_connect(dialog, "close",
				G_CALLBACK(close_callback), NULL);

	g_signal_connect(dialog, "response",
				G_CALLBACK(close_callback), NULL);

	gtk_widget_show_all(dialog);
}

static void settings_callback(GObject *widget, gpointer user_data)
{
	const char *command = "bluetooth-properties";

	if (!g_spawn_command_line_async(command, NULL))
		g_printerr("Couldn't execute command: %s\n", command);
}

static void response_callback(GtkWidget *dialog,
					gint response, gpointer user_data)
{
	GtkWidget *selector = user_data;
	gchar *command, *address = NULL;

	if (response != GTK_RESPONSE_ACCEPT) {
		gtk_widget_destroy(dialog);
		return;
	}

	g_object_get(selector, "device-selected", &address, NULL);

	gtk_widget_destroy(dialog);

	command = g_strdup_printf("%s --no-default-window \"obex://[%s]\"",
							"nautilus", address);

	g_free(address);

	if (!g_spawn_command_line_async(command, NULL))
		g_printerr(_("Couldn't execute command: %s\n"), command);

	g_free(command);
}

static void selected_device_changed(BluetoothDeviceSelection *selector,
					gchar *address, gpointer user_data)
{
	GtkWidget *dialog = user_data;

	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
				GTK_RESPONSE_ACCEPT, address != NULL);
}

static void browse_callback(GObject *widget, gpointer user_data)
{
	GtkWidget *dialog;
	GtkWidget *selector;

	dialog = gtk_dialog_new_with_buttons(_("Browse Devices"),
				NULL, GTK_DIALOG_NO_SEPARATOR,
				GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
				GTK_STOCK_CONNECT, GTK_RESPONSE_ACCEPT, NULL);

	gtk_window_set_icon_name(GTK_WINDOW(dialog), "stock_bluetooth");

	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
						GTK_RESPONSE_ACCEPT, FALSE);

	gtk_window_set_default_size(GTK_WINDOW(dialog), 450, 300);

	gtk_container_set_border_width(GTK_CONTAINER(dialog), 6);

	selector = bluetooth_device_selection_new(_("Select device to browse"));

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), selector);

	g_signal_connect(selector, "selected-device-changed",
				G_CALLBACK(selected_device_changed), dialog);

	g_signal_connect(dialog, "close",
				G_CALLBACK(close_callback), NULL);

	g_signal_connect(dialog, "response",
				G_CALLBACK(response_callback), selector);

	gtk_widget_show_all(dialog);
}

#if 0
static void sendto_callback(GObject *widget, gpointer user_data)
{
	const char *command = "bluetooth-sendto";

	if (!g_spawn_command_line_async(command, NULL))
		g_printerr("Couldn't execute command: %s\n", command);
}
#endif

#if 0
static void wizard_callback(GObject *widget, gpointer user_data)
{
	const char *command = "bluetooth-wizard";

	if (!g_spawn_command_line_async(command, NULL))
		g_printerr("Couldn't execute command: %s\n", command);
}
#endif

static void activate_callback(GObject *widget, gpointer user_data)
{
	close_notification();

	g_list_foreach(input_list, show_dialog, NULL);

	gtk_status_icon_set_blinking(statusicon, FALSE);
}

static void popup_callback(GObject *widget, guint button,
				guint activate_time, gpointer user_data)
{
	GtkWidget *menu = user_data;

	gtk_widget_set_sensitive(menuitem_browse, obexftp_available() &&
						attached_adapters() > 0);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL,
			gtk_status_icon_position_menu,
			GTK_STATUS_ICON(widget), button, activate_time);
}

static GtkWidget *create_popupmenu(void)
{
	GtkWidget *menu;
	GtkWidget *item;

	menu = gtk_menu_new();

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_PREFERENCES, NULL);
	g_signal_connect(item, "activate",
				G_CALLBACK(settings_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_ABOUT, NULL);
	g_signal_connect(item, "activate",
				G_CALLBACK(about_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	item = gtk_separator_menu_item_new();
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

#if 0
	item = gtk_image_menu_item_new_with_label(_("Send File..."));
	g_signal_connect(item, "activate",
				G_CALLBACK(sendto_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
#endif

	item = gtk_image_menu_item_new_with_label(_("Browse Device..."));
	g_signal_connect(item, "activate",
				G_CALLBACK(browse_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	menuitem_browse = item;

#if 0
	item = gtk_separator_menu_item_new();
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	item = gtk_menu_item_new_with_label(_("Setup New Device"));
	g_signal_connect(item, "activate",
				G_CALLBACK(wizard_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
#endif

	return menu;
}

static GConfEnumStringPair icon_policy_enum_map [] = {
	{ ICON_POLICY_NEVER,	"never"		},
	{ ICON_POLICY_ALWAYS,	"always"	},
	{ ICON_POLICY_PRESENT,	"present"	},
	{ ICON_POLICY_PRESENT,	NULL		},
};

static void gconf_callback(GConfClient *client, guint cnxn_id,
					GConfEntry *entry, gpointer user_data)
{
	GConfValue *value;

	value = gconf_entry_get_value(entry);
	if (value == NULL)
		return;

	if (strcmp(entry->key, PREF_USE_HAL) == 0) {
#ifdef HAVE_HAL
		use_hal = gconf_value_get_bool(value);

		if (use_hal == TRUE) {
			gchar *formfactor;

			formfactor = get_form_factor();
			g_list_foreach(adapter_list, adapter_setup, formfactor);
			g_free(formfactor);
		}
#endif
		return;
	}

	if (strcmp(entry->key, PREF_ICON_POLICY) == 0) {
		gboolean visible;
		const char *str;

		str = gconf_value_get_string(value);
		if (!str)
			return;

		gconf_string_to_enum(icon_policy_enum_map, str, &icon_policy);

		visible = gtk_status_icon_get_visible(statusicon);

		if (icon_policy == ICON_POLICY_NEVER)
			visible = FALSE;
		else if (icon_policy == ICON_POLICY_ALWAYS)
			visible = TRUE;
		else if (icon_policy == ICON_POLICY_PRESENT)
			visible = attached_adapters() > 0 ? TRUE : FALSE;

		gtk_status_icon_set_visible(statusicon, visible);
		return;
	}

	if (strcmp(entry->key, PREF_AUTO_AUTHORIZE) == 0)
		auto_authorize = gconf_value_get_bool(value);
}

int main(int argc, char *argv[])
{
	GtkWidget *menu;
	GError *error = NULL;
	char *str;

	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	gtk_init(&argc, &argv);

#ifdef HAVE_LIBNOTIFY
	notify_init("bluetooth-manager");
#endif

	conn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	if (error != NULL) {
		g_printerr("Connecting to system bus failed: %s\n",
							error->message);
		g_error_free(error);
		exit(EXIT_FAILURE);
	}

	gconf = gconf_client_get_default();

#ifdef HAVE_HAL
	use_hal = gconf_client_get_bool(gconf, PREF_USE_HAL, NULL);
#endif

	str = gconf_client_get_string(gconf, PREF_ICON_POLICY, NULL);
	if (str) {
		gconf_string_to_enum(icon_policy_enum_map, str, &icon_policy);
		g_free(str);
	}

	auto_authorize = gconf_client_get_bool(gconf,
					PREF_AUTO_AUTHORIZE, NULL);

	gconf_client_add_dir(gconf, PREF_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);

	gconf_client_notify_add(gconf, PREF_DIR,
					gconf_callback, NULL, NULL, NULL);

	statusicon = gtk_status_icon_new_from_icon_name("stock_bluetooth");

	gtk_status_icon_set_tooltip(statusicon, _("Bluetooth Manager"));

	if (icon_policy != ICON_POLICY_ALWAYS)
		gtk_status_icon_set_visible(statusicon, FALSE);

	g_signal_connect(statusicon, "activate",
				G_CALLBACK(activate_callback), NULL);

	menu = create_popupmenu();

	g_signal_connect(statusicon, "popup-menu",
				G_CALLBACK(popup_callback), menu);

	setup_dbus();

	setup_manager();

	register_agents();

	//passkey_dialog("/org/bluez/hci0", "00:11:22:33:44:55", "Test", NULL);
	//confirm_dialog("/org/bluez/hci0", "00:11:22:33:44:55", "123456", "Test", NULL);
	//auth_dialog("/org/bluez/hci0", "00:11:22:33:44:55", "/org/bluez/echo", "", "Test", "Echo service", NULL);

	gtk_main();

	gtk_widget_destroy(menu);

	g_object_unref(gconf);

	close_notification();

	g_list_foreach(adapter_list, adapter_free, NULL);

	dbus_g_connection_unref(conn);

	return 0;
}
