/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dbus/dbus-glib.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include "bluetooth-device-selection.h"

#include "marshal.h"

static DBusGConnection *conn = NULL;

static GtkWidget *dialog;
static GtkWidget *label_filename;
static GtkWidget *label_status;
static GtkWidget *progress;

static gint filesize = -1;

static gchar *open_file_dialog(void)
{
	GtkWidget *dialog;
	gchar *filename = NULL;

	dialog = gtk_file_chooser_dialog_new(_("Select File"), NULL,
				GTK_FILE_CHOOSER_ACTION_OPEN,
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);

	gtk_window_set_icon_name(GTK_WINDOW(dialog), "stock_bluetooth");

	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
		filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));

	gtk_widget_destroy(dialog);

	return filename;
}

static void selected_device_changed(BluetoothDeviceSelection *selector,
					gchar *address, gpointer user_data)
{
	GtkWidget *dialog = user_data;

	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
				GTK_RESPONSE_ACCEPT, address != NULL);
}

static gchar *browse_device_dialog(void)
{
	GtkWidget *dialog;
	GtkWidget *selector;
	gchar *address = NULL;

	dialog = gtk_dialog_new_with_buttons(_("Select Device"),
				NULL, GTK_DIALOG_NO_SEPARATOR,
				GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
				GTK_STOCK_CONNECT, GTK_RESPONSE_ACCEPT, NULL);

	gtk_window_set_icon_name(GTK_WINDOW(dialog), "stock_bluetooth");

	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
						GTK_RESPONSE_ACCEPT, FALSE);

	gtk_window_set_default_size(GTK_WINDOW(dialog), 450, 300);

	gtk_container_set_border_width(GTK_CONTAINER(dialog), 6);

	selector = bluetooth_device_selection_new(_("Select destination device"));

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), selector);

	g_signal_connect(selector, "selected-device-changed",
				G_CALLBACK(selected_device_changed), dialog);

	gtk_widget_show_all(selector);

	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
		g_object_get(selector, "device-selected", &address, NULL);

	gtk_widget_destroy(dialog);

	return address;
}

static void response_callback(GtkWidget *dialog,
					gint response, gpointer user_data)
{
	gtk_widget_destroy(dialog);

	gtk_main_quit();
}

static void create_window(const gchar *filename)
{
	GtkWidget *vbox;
	GtkWidget *label;
	gchar *text;

	dialog = gtk_dialog_new_with_buttons(_("File Transfer"), NULL,
				GTK_DIALOG_NO_SEPARATOR,
				GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);

	gtk_window_set_icon_name(GTK_WINDOW(dialog), "stock_bluetooth");
	gtk_window_set_default_size(GTK_WINDOW(dialog), 400, -1);

	gtk_container_set_border_width(GTK_CONTAINER(dialog), 6);

	vbox = gtk_vbox_new(FALSE, 6);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 12);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), vbox);

	label = gtk_label_new(NULL);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	label_filename = label;

	progress = gtk_progress_bar_new();
	gtk_box_pack_start(GTK_BOX(vbox), progress, FALSE, FALSE, 0);

	label = gtk_label_new(NULL);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 6);

	label_status = label;

	text = g_strdup_printf("<b>%s</b>", filename);
	gtk_label_set_markup(GTK_LABEL(label_filename), text);
	g_free(text);

	gtk_label_set_markup(GTK_LABEL(label_status), _("Connecting..."));

	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
						GTK_RESPONSE_CLOSE, FALSE);

	g_signal_connect(G_OBJECT(dialog), "response",
				G_CALLBACK(response_callback), NULL);

	gtk_widget_show_all(dialog);
}

static void transfer_started(DBusGProxy *proxy, gchar *a, gchar *b,
						gint size, gpointer user_data)
{
	gchar *text;

	filesize = size;

	text = g_strdup_printf(_("Starting transfer of %d bytes"), size);
	gtk_label_set_markup(GTK_LABEL(label_status), text);
	g_free(text);

	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progress), 0.0);
}

static void transfer_progress(DBusGProxy *proxy, guint bytes, gpointer user_data)
{
	gchar *text;
	gdouble fraction;

	text = g_strdup_printf(_("Transfered %d of %d bytes"), bytes, filesize);
	gtk_label_set_markup(GTK_LABEL(label_status), text);
	g_free(text);

	fraction = (gdouble) bytes / (gdouble) filesize;
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progress), fraction);
}

static void transfer_completed(DBusGProxy *proxy, gpointer user_data)
{
	gtk_label_set_markup(GTK_LABEL(label_status), _("Completed"));

	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progress), 1.0);

	dbus_g_proxy_call(proxy, "Disconnect", NULL, G_TYPE_INVALID,
							G_TYPE_INVALID);

	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
						GTK_RESPONSE_CLOSE, TRUE);
}

static void session_connected(DBusGProxy *proxy, gpointer user_data)
{
	gchar *filename = user_data;
	GError *error = NULL;

	gtk_label_set_markup(GTK_LABEL(label_status), _("Connected"));

	dbus_g_proxy_call(proxy, "SendFile", &error,
				G_TYPE_STRING, filename, G_TYPE_INVALID,
							G_TYPE_INVALID);

	if (error != NULL) {
		g_printerr("Sending of file %s failed: %s\n", filename,
							error->message);
		g_error_free(error);
		gtk_main_quit();
	}
}

static void create_notify(DBusGProxy *proxy,
				DBusGProxyCall *call, void *user_data)
{
	GError *error = NULL;
	const gchar *path = NULL;

	if (dbus_g_proxy_end_call(proxy, call, &error,
			G_TYPE_STRING, &path, G_TYPE_INVALID) == FALSE) {
		gchar *text, *message;

		if (error != NULL) {
			message = g_strdup(error->message);
			g_error_free(error);
		} else
			message = g_strdup(_("An unknown error occured"));

		text = g_strdup_printf("<span foreground=\"red\">%s</span>",
								message);
		gtk_label_set_markup(GTK_LABEL(label_status), text);
		g_free(text);

		g_free(message);

		gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
						GTK_RESPONSE_CLOSE, TRUE);
		return;
	}

	proxy = dbus_g_proxy_new_for_name(conn, "org.openobex",
						path, "org.openobex.Session");

	dbus_g_proxy_add_signal(proxy, "Connected", G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(proxy, "Connected",
				G_CALLBACK(session_connected), user_data, NULL);

	dbus_g_proxy_add_signal(proxy, "TransferStarted", G_TYPE_STRING,
				G_TYPE_STRING, G_TYPE_INT, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(proxy, "TransferStarted",
				G_CALLBACK(transfer_started), NULL, NULL);

	dbus_g_proxy_add_signal(proxy, "TransferProgress",
						G_TYPE_UINT, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(proxy, "TransferProgress",
				G_CALLBACK(transfer_progress), NULL, NULL);

	dbus_g_proxy_add_signal(proxy, "TransferCompleted", G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(proxy, "TransferCompleted",
				G_CALLBACK(transfer_completed), NULL, NULL);

	dbus_g_proxy_call(proxy, "Connect", NULL, G_TYPE_INVALID,
							G_TYPE_INVALID);
}

static gchar *option_device = NULL;

static GOptionEntry options[] = {
	{ "device", 0, 0, G_OPTION_ARG_STRING, &option_device,
				N_("Remote device to use"), "ADDRESS" },
	{ NULL },
};

int main(int argc, char *argv[])
{
	DBusGProxy *proxy;
	GError *error;
	gchar *filename, *address;

	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	error = NULL;

	if (gtk_init_with_args(&argc, &argv, "[FILE...]",
				options, GETTEXT_PACKAGE, &error) == FALSE) {
		if (error != NULL) {
			g_printerr("%s\n", error->message);
			g_error_free(error);
		} else
			g_printerr("An unknown error occurred\n");

		gtk_exit(1);
	}

	if (argc < 2) {
		filename = open_file_dialog();
		if (filename == NULL)
			gtk_exit(1);
	} else
		filename = g_strdup(argv[1]);

	if (option_device == NULL) {
		address = browse_device_dialog();
		if (address == NULL) {
			g_free(filename);
			gtk_exit(1);
		}
	} else
		address = g_strdup(option_device);

	dbus_g_object_register_marshaller(marshal_VOID__STRING_STRING_INT,
				G_TYPE_NONE, G_TYPE_STRING,
				G_TYPE_STRING, G_TYPE_INT, G_TYPE_INVALID);

	error = NULL;

	conn = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
	if (conn == NULL) {
		if (error != NULL) {
			g_printerr("Connecting to session bus failed: %s\n",
							error->message);
			g_error_free(error);
		} else
			g_print("An unknown error occured\n");

		gtk_exit(1);
	}

	create_window(filename);

	proxy = dbus_g_proxy_new_for_name(conn, "org.openobex",
				"/org/openobex", "org.openobex.Manager");

	dbus_g_proxy_begin_call(proxy, "CreateBluetoothSession",
				create_notify, filename, NULL,
				G_TYPE_STRING, address, G_TYPE_STRING, "opp",
				G_TYPE_BOOLEAN, FALSE, G_TYPE_INVALID);

	gtk_main();

	dbus_g_connection_unref(conn);

	g_free(address);
	g_free(filename);

	return 0;
}
