/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2006  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <dbus/dbus-glib.h>

#include <gconf/gconf-client.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#ifdef HAVE_LIBNOTIFY
#include <libnotify/notify.h>
#endif

#ifndef HAVE_GTK_STATUS_ICON
#include "gtkstatusicon.h"
#endif

#define AGENT_PATH "/org/bluez/applet"

static int volatile registered = 0;

static int always_show_icon = 1;

static GtkStatusIcon *statusicon = NULL;

static GConfClient* gconf;

typedef enum {
	PASSKEY_AGENT_ERROR_REJECT
} PasskeyAgentError;

#define PASSKEY_AGENT_ERROR (passkey_agent_error_quark())

#define PASSKEY_AGENT_ERROR_TYPE (passkey_agent_error_get_type()) 

static GQuark passkey_agent_error_quark(void)
{
	static GQuark quark = 0;
	if (!quark)
		quark = g_quark_from_static_string("passkey_agent");

	return quark;
}

#define ENUM_ENTRY(NAME, DESC) { NAME, "" #NAME "", DESC }

static GType passkey_agent_error_get_type(void)
{
	static GType etype = 0;
	if (etype == 0) {
		static const GEnumValue values[] = {
			ENUM_ENTRY(PASSKEY_AGENT_ERROR_REJECT, "Rejected"),
			{ 0, 0, 0 }
		};

		etype = g_enum_register_static("passkey_agent", values);
	}

	return etype;
}

static GList *input_list = NULL;

struct input_data {
	char *path;
	char *address;
	DBusGMethodInvocation *context;
	GtkWidget *dialog;
	GtkWidget *button;
	GtkWidget *entry;
};

static gint input_compare(gconstpointer a, gconstpointer b)
{
	struct input_data *a_data = (struct input_data *) a;
	struct input_data *b_data = (struct input_data *) b;

	return strcmp(a_data->address, b_data->address);
}

static void response_callback(GtkWidget *dialog,
					gint response, gpointer user_data)
{
	struct input_data *input = user_data;

	if (response == GTK_RESPONSE_ACCEPT) {
		const char *passkey;
		passkey = gtk_entry_get_text(GTK_ENTRY(input->entry));
		dbus_g_method_return(input->context, passkey);
	} else {
		GError *error;
		error = g_error_new(PASSKEY_AGENT_ERROR,
					PASSKEY_AGENT_ERROR_REJECT,
						"Pairing request rejected");
		dbus_g_method_return_error(input->context, error);
	}

	gtk_widget_destroy(dialog);

	input_list = g_list_remove(input_list, input);

	g_free(input->address);
	g_free(input->path);
	g_free(input);

	if (g_list_length(input_list) == 0)
		gtk_status_icon_set_blinking(statusicon, FALSE);
}

static void changed_callback(GtkWidget *editable, gpointer user_data)
{
	struct input_data *input = user_data;
	const gchar *text;

	text = gtk_entry_get_text(GTK_ENTRY(input->entry));

	gtk_widget_set_sensitive(input->button, strlen(text) ? TRUE : FALSE);
}

static void toggled_callback(GtkWidget *button, gpointer user_data)
{
	struct input_data *input = user_data;
	gboolean mode;

	mode = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));

	gtk_entry_set_visibility(GTK_ENTRY(input->entry), mode);
}

static void create_dialog(const char *path, const char *address,
			gchar *device, DBusGMethodInvocation *context)
{
	GtkWidget *dialog;
	GtkWidget *button;
	GtkWidget *image;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *table;
	struct input_data *input;

	input = g_try_malloc0(sizeof(*input));
	if (!input)
		return;

	input->path = g_strdup(path);
	input->address = g_strdup(address);

	input->context = context;

	dialog = gtk_dialog_new();

	gtk_window_set_title(GTK_WINDOW(dialog), _("Authentication request"));

	gtk_window_set_icon_name(GTK_WINDOW(dialog), "stock_bluetooth");

	gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);

	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

	gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);

	gtk_window_set_urgency_hint(GTK_WINDOW(dialog), TRUE);

	gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);

	input->dialog = dialog;

	button = gtk_dialog_add_button(GTK_DIALOG(dialog),
				GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT);
 
	button = gtk_dialog_add_button(GTK_DIALOG(dialog),
					GTK_STOCK_OK, GTK_RESPONSE_ACCEPT);
  
	gtk_widget_grab_default(button);

	gtk_widget_set_sensitive(button, FALSE);

	input->button = button;

	table = gtk_table_new(5, 2, FALSE);

	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 20);

	gtk_container_set_border_width(GTK_CONTAINER(table), 12);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);

	image = gtk_image_new_from_icon_name(GTK_STOCK_DIALOG_AUTHENTICATION,
							GTK_ICON_SIZE_DIALOG);

	gtk_misc_set_alignment(GTK_MISC(image), 0.0, 0.0);

	gtk_table_attach(GTK_TABLE(table), image, 0, 1, 0, 5,
						GTK_SHRINK, GTK_FILL, 0, 0);

	label = gtk_label_new(_("Pairing request for device:"));

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_table_attach(GTK_TABLE(table), label, 1, 2, 0, 1,
				GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	label = gtk_label_new(device);

	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

	gtk_label_set_selectable(GTK_LABEL(label), TRUE);

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_widget_set_size_request(GTK_WIDGET(label), 280, -1);

	gtk_table_attach(GTK_TABLE(table), label, 1, 2, 1, 2,
			GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 4);

	label = gtk_label_new(_("\nEnter passkey for authentication:"));

	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);

	gtk_table_attach(GTK_TABLE(table), label, 1, 2, 2, 3,
				GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	entry = gtk_entry_new();

	gtk_entry_set_max_length(GTK_ENTRY(entry), 16);

	gtk_entry_set_activates_default(GTK_ENTRY(entry), TRUE);

	gtk_entry_set_visibility(GTK_ENTRY(entry), FALSE);

	input->entry = entry;

	g_signal_connect(G_OBJECT(entry), "changed",
				G_CALLBACK(changed_callback), input);

	gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 3, 4,
				GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	button = gtk_check_button_new_with_label(_("Show input"));

	g_signal_connect(G_OBJECT(button), "toggled",
				G_CALLBACK(toggled_callback), input);

	gtk_table_attach(GTK_TABLE(table), button, 1, 2, 4, 5,
				GTK_EXPAND | GTK_FILL, GTK_SHRINK, 0, 0);

	input_list = g_list_append(input_list, input);

	g_signal_connect(G_OBJECT(dialog), "response",
				G_CALLBACK(response_callback), input);

	gtk_status_icon_set_blinking(statusicon, TRUE);
}

static void show_dialog(gpointer data, gpointer user_data)
{
	struct input_data *input = data;

	gtk_widget_show_all(input->dialog);

	gtk_window_present(GTK_WINDOW(input->dialog));
}

#ifdef HAVE_LIBNOTIFY
static NotifyNotification *notify = NULL;

static void notify_action(NotifyNotification *notify,
					gchar *action, gpointer user_data)
{
}
#endif

static void show_notification(const gchar *summary, const gchar *message,
					gint timeout, GCallback handler)
{
#ifdef HAVE_LIBNOTIFY
	NotifyActionCallback callback;
	GdkScreen *screen;
	GdkRectangle area;

	if (notify) {
		g_signal_handlers_destroy(notify);
		notify_notification_close(notify, NULL);
	}

	notify = notify_notification_new(summary, message,
						"stock_bluetooth", NULL);

	notify_notification_set_timeout(notify, timeout);

	if (gtk_status_icon_get_visible(statusicon) == TRUE) {
#ifdef HAVE_GTK_STATUS_ICON
		gtk_status_icon_get_geometry(statusicon, &screen, &area, NULL);
#else
		gtk_status_icon_get_screen_geometry(statusicon, &screen,
				&area.x, &area.y, &area.width, &area.height);
#endif

		notify_notification_set_hint_int32(notify,
					"x", area.x + area.width / 2);
		notify_notification_set_hint_int32(notify,
					"y", area.y + area.height / 2);
	}

	notify_notification_set_urgency(notify, NOTIFY_URGENCY_NORMAL);

	callback = handler ? NOTIFY_ACTION_CALLBACK(handler) : notify_action;

	notify_notification_add_action(notify, "default", "action",
						callback, NULL, NULL);

	notify_notification_show(notify, NULL);
#endif
}

static void close_notification(void)
{
#ifdef HAVE_LIBNOTIFY
	if (notify) {
		g_signal_handlers_destroy(notify);
		notify_notification_close(notify, NULL);
		notify = NULL;
	}
#endif
}

typedef struct {
	GObject parent;
	DBusGConnection *conn;
} PasskeyAgent;

typedef struct {
	GObjectClass parent;
} PasskeyAgentClass;

static GObjectClass *parent_class;

G_DEFINE_TYPE(PasskeyAgent, passkey_agent, G_TYPE_OBJECT)

#define PASSKEY_AGENT_OBJECT_TYPE (passkey_agent_get_type())

#define PASSKEY_AGENT(obj) (G_TYPE_CHECK_INSTANCE_CAST((obj), \
				PASSKEY_AGENT_OBJECT_TYPE, PasskeyAgent))

static void passkey_agent_finalize(GObject *obj)
{
	PasskeyAgent *agent = PASSKEY_AGENT(obj);

	dbus_g_connection_unref(agent->conn);

	parent_class->finalize(obj);
}

static void passkey_agent_init(PasskeyAgent *obj)
{
}

static void passkey_agent_class_init(PasskeyAgentClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	gobject_class->finalize = passkey_agent_finalize;
}

static PasskeyAgent *passkey_agent_new(DBusGConnection *conn, const char *path)
{
	PasskeyAgent *agent;

	agent = g_object_new(PASSKEY_AGENT_OBJECT_TYPE, NULL);

	agent->conn = dbus_g_connection_ref(conn);

	dbus_g_connection_register_g_object(conn, path, G_OBJECT(agent));

	return agent;
}

static void notification_closed(GObject *object, gpointer user_data)
{
	g_list_foreach(input_list, show_dialog, NULL);

	gtk_status_icon_set_blinking(statusicon, FALSE);
}

static gboolean passkey_agent_request(PasskeyAgent *agent,
				const char *path, const char *address,
					DBusGMethodInvocation *context)
{
	DBusGProxy *object;
	const char *adapter = NULL, *name = NULL;
	gchar *device, *text;

	object = dbus_g_proxy_new_for_name(agent->conn, "org.bluez",
						path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetRemoteName", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	if (name) {
		if (g_strrstr(name, address))
			device = g_strdup(name);
		else
			device = g_strdup_printf("%s (%s)", name, address);
	} else
		device = g_strdup(address);

	create_dialog(path, address, device, context);

	text = g_strdup_printf(_("Pairing request for %s"), device);

	g_free(device);

	show_notification(adapter ? adapter : _("Bluetooth device"),
				text, 0, G_CALLBACK(notification_closed));

	g_free(text);

	return TRUE;
}

static gboolean passkey_agent_cancel(PasskeyAgent *agent,
			const char *path, const char *address, GError **error)
{
	GList *list;
	struct input_data *input;

	input = g_try_malloc0(sizeof(*input));
	if (!input)
		return FALSE;

	input->path = g_strdup(path);
	input->address = g_strdup(address);

	list = g_list_find_custom(input_list, input, input_compare);

	g_free(input->address);
	g_free(input->path);
	g_free(input);

	if (!list || !list->data)
		return FALSE;

	input = list->data;

	close_notification();

	response_callback(input->dialog, GTK_RESPONSE_DELETE_EVENT, input);

	return TRUE;
}

static gboolean passkey_agent_release(PasskeyAgent *agent, GError **error)
{
	registered = 0;

	return TRUE;
}

#include "passkey-agent-glue.h"

static int register_passkey_agent(DBusGConnection *conn)
{
	DBusGProxy *object;
	GError *error = NULL;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					"/org/bluez", "org.bluez.Security");

	dbus_g_proxy_call(object, "RegisterDefaultPasskeyAgent", &error,
				G_TYPE_STRING, AGENT_PATH,
				G_TYPE_INVALID, G_TYPE_INVALID);

	if (error != NULL) {
		g_printerr(_("Passkey agent registration failed: %s\n"),
							error->message);
		g_error_free(error);
		return -1;
	}

	registered = 1;

	return 0;
}

static void bonding_created(DBusGProxy *object,
				const char *address, gpointer user_data)
{
	const char *adapter = NULL, *name = NULL;
	gchar *device, *text;

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetRemoteName", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	if (name) {
		if (g_strrstr(name, address))
			device = g_strdup(name);
		else
			device = g_strdup_printf("%s (%s)", name, address);
	} else
		device = g_strdup(address);

	text = g_strdup_printf(_("Created bonding with %s"), device);

	g_free(device);

	show_notification(adapter ? adapter : _("Bluetooth device"),
							text, 6000, NULL);

	g_free(text);
}

static void bonding_removed(DBusGProxy *object,
				const char *address, gpointer user_data)
{
	const char *adapter = NULL, *name = NULL;
	gchar *device, *text;

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	dbus_g_proxy_call(object, "GetRemoteName", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);

	if (name) {
		if (g_strrstr(name, address))
			device = g_strdup(name);
		else
			device = g_strdup_printf("%s (%s)", name, address);
	} else
		device = g_strdup(address);

	text = g_strdup_printf(_("Removed bonding with %s"), device);

	g_free(device);

	show_notification(adapter ? adapter : _("Bluetooth device"),
							text, 6000, NULL);

	g_free(text);
}

static void mode_changed(DBusGProxy *object,
				const char *mode, gpointer user_data)
{
	const char *adapter = NULL;
	gchar *text;

	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &adapter, G_TYPE_INVALID);

	text = g_strdup_printf(_("Switched device into\n%s mode"), mode);

	show_notification(adapter ? adapter : _("Bluetooth device"),
							text, 3000, NULL);

	g_free(text);
}

static GList *adapter_list = NULL;

struct adapter_data {
	char *path;
	int attached;
};

static void adapter_free(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	adapter_list = g_list_remove(adapter_list, adapter);

	g_free(adapter->path);
	g_free(adapter);
}

static void adapter_disable(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	adapter->attached = 0;
}

static gint adapter_compare(gconstpointer a, gconstpointer b)
{
	const struct adapter_data *adapter = a;
	const char *path = b;

	return strcmp(adapter->path, path);
}

static void add_adapter(DBusGConnection *conn, const char *path)
{
	GList *list;
	DBusGProxy *object;
	struct adapter_data *adapter;
	
	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;

		adapter->attached = 1;
		return;
	}

        adapter = g_try_malloc0(sizeof(*adapter));
        if (!adapter)
		return;

	adapter->path = g_strdup(path);

	adapter_list = g_list_append(adapter_list, adapter);

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
						path, "org.bluez.Adapter");

	dbus_g_proxy_add_signal(object, "ModeChanged",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "ModeChanged",
				G_CALLBACK(mode_changed), conn, NULL);

	dbus_g_proxy_add_signal(object, "BondingCreated",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "BondingCreated",
				G_CALLBACK(bonding_created), conn, NULL);

	dbus_g_proxy_add_signal(object, "BondingRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "BondingRemoved",
				G_CALLBACK(bonding_removed), conn, NULL);
}

static void adapter_added(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	DBusGConnection *conn = user_data;

	if (!registered)
		register_passkey_agent(conn);

	add_adapter(conn, path);

	if (!always_show_icon) {
		if (g_list_length(adapter_list) > 0)
			gtk_status_icon_set_visible(statusicon, TRUE);
	}
}

static void adapter_removed(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	GList *list;

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;

		adapter->attached = 0;
	}

	if (!always_show_icon) {
		if (g_list_length(adapter_list) == 0)
			gtk_status_icon_set_visible(statusicon, FALSE);
	}
}

static int setup_manager(DBusGConnection *conn)
{
	DBusGProxy *object;
	GError *error = NULL;
	const gchar **array = NULL;

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					"/org/bluez", "org.bluez.Manager");

	dbus_g_proxy_add_signal(object, "AdapterAdded",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "AdapterAdded",
				G_CALLBACK(adapter_added), conn, NULL);

	dbus_g_proxy_add_signal(object, "AdapterRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "AdapterRemoved",
				G_CALLBACK(adapter_removed), conn, NULL);

	dbus_g_proxy_call(object, "ListAdapters", &error,
			G_TYPE_INVALID,	G_TYPE_STRV, &array, G_TYPE_INVALID);

	if (error != NULL) {
		g_printerr(_("Getting adapter list failed: %s\n"),
							error->message);
		g_error_free(error);
	} else {
		while (*array) {
			add_adapter(conn, *array);
			array++;
		}
	}

	return 0;
}

static void name_owner_changed(DBusGProxy *object, const char *name,
			const char *prev, const char *new, gpointer user_data)
{
	if (strcmp(name, "org.bluez") && *new == '\0') {
		registered = 0;

		g_list_foreach(adapter_list, adapter_disable, NULL);

		if (!always_show_icon)
			gtk_status_icon_set_visible(statusicon, FALSE);
	}
}

static int setup_dbus(DBusGConnection *conn)
{
	DBusGProxy *object;
	void *agent;

	object = dbus_g_proxy_new_for_name(conn, DBUS_SERVICE_DBUS,
					DBUS_PATH_DBUS, DBUS_INTERFACE_DBUS);

	dbus_g_proxy_add_signal(object, "NameOwnerChanged",
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "NameOwnerChanged",
				G_CALLBACK(name_owner_changed), NULL, NULL);

	dbus_g_object_type_install_info(PASSKEY_AGENT_OBJECT_TYPE,
					&dbus_glib_passkey_agent_object_info);

	dbus_g_error_domain_register(PASSKEY_AGENT_ERROR, "org.bluez.Error",
						PASSKEY_AGENT_ERROR_TYPE);

	agent = passkey_agent_new(conn, AGENT_PATH);

	return 0;
}

static void close_callback(GtkWidget *dialog, gpointer user_data)
{
	gtk_widget_destroy(dialog);
}

static void about_callback(GtkWidget *item, gpointer user_data)
{
	const gchar *authors[] = {
		"Marcel Holtmann <marcel@holtmann.org>",
		NULL
	};
	GtkWidget *dialog;

	dialog = gtk_about_dialog_new();

	gtk_window_set_icon_name(GTK_WINDOW(dialog), "stock_bluetooth");

	gtk_about_dialog_set_name(GTK_ABOUT_DIALOG(dialog),
						"Bluetooth Applet");

	gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(dialog), VERSION);

	gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(dialog),
			"Copyright \xc2\xa9 2005-2006 Marcel Holtmann");

	gtk_about_dialog_set_comments(GTK_ABOUT_DIALOG(dialog),
			_("A Bluetooth manager for the GNOME desktop"));

	gtk_about_dialog_set_logo_icon_name(GTK_ABOUT_DIALOG(dialog),
							"stock_bluetooth");

	gtk_about_dialog_set_website(GTK_ABOUT_DIALOG(dialog),
							"www.bluez.org");

	gtk_about_dialog_set_authors(GTK_ABOUT_DIALOG(dialog), authors);

	g_signal_connect(dialog, "close",
				G_CALLBACK(close_callback), NULL);

	g_signal_connect(dialog, "response",
				G_CALLBACK(close_callback), NULL);

	gtk_widget_show_all(dialog);
}

static void activate_callback(GObject *object, gpointer user_data)
{
	close_notification();

	g_list_foreach(input_list, show_dialog, NULL);

	gtk_status_icon_set_blinking(statusicon, FALSE);
}

static void popup_callback(GObject *object, guint button,
				guint activate_time, gpointer user_data)
{
	GtkWidget *menu;
	GtkWidget *item;

	menu = gtk_menu_new();

	item = gtk_image_menu_item_new_from_stock(GTK_STOCK_ABOUT, NULL);
	g_signal_connect(item, "activate",
				G_CALLBACK(about_callback), NULL);
	gtk_widget_show(item);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL,
			gtk_status_icon_position_menu,
			GTK_STATUS_ICON(object), button, activate_time);
}

static void gconf_callback(GConfClient *client, guint cnxn_id,
					GConfEntry *entry, gpointer user_data)
{
}

int main(int argc, char *argv[])
{
	DBusGConnection *conn;
	GError *error = NULL;

	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	gtk_init(&argc, &argv);

#ifdef HAVE_LIBNOTIFY
	notify_init("bt-applet");
#endif

	conn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	if (error != NULL) {
		g_printerr(_("Connecting to system bus failed: %s\n"),
							error->message);
		g_error_free(error);
		exit(EXIT_FAILURE);
	}

	gconf = gconf_client_get_default();

	gconf_client_add_dir(gconf, "/apps/bt-applet",
					GCONF_CLIENT_PRELOAD_NONE, NULL);

	gconf_client_notify_add(gconf, "/apps/bt-applet",
					gconf_callback, NULL, NULL, NULL);

	statusicon = gtk_status_icon_new_from_icon_name("stock_bluetooth");

	gtk_status_icon_set_tooltip(statusicon, "Bluetooth Manager");

	g_signal_connect(statusicon, "activate",
				G_CALLBACK(activate_callback), NULL);

	g_signal_connect(statusicon, "popup-menu",
				G_CALLBACK(popup_callback), NULL);

	setup_dbus(conn);

	setup_manager(conn);

	if (!always_show_icon) {
		gtk_status_icon_set_visible(statusicon,
			g_list_length(adapter_list) > 0 ? TRUE : FALSE);
	}

	register_passkey_agent(conn);

	gtk_main();

	g_object_unref(gconf);

	close_notification();

	g_list_foreach(adapter_list, adapter_free, NULL);

	return 0;
}
