/*******************************************************************************
  Copyright (c) 2009-2019, Intel Corporation

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
  * Neither the name of Intel Corporation nor the names of its contributors
    may be used to endorse or promote products derived from this software
    without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include "snow3g_internal.h"

/*--------------------------------------------------------------------
 *
 * An implementation of SNOW 3G, the core algorithm for the
 * 3GPP Confidentiality and Integrity algorithms.
 *
 *--------------------------------------------------------------------*/

IMB_DLL_LOCAL
DECLARE_ALIGNED(const int snow3g_table_A_mul[256], 32) = {
        0x00000000, 0xe19fcf13, 0x6b973726, 0x8a08f835, 0xd6876e4c, 0x3718a15f,
        0xbd10596a, 0x5c8f9679, 0x05a7dc98, 0xe438138b, 0x6e30ebbe, 0x8faf24ad,
        0xd320b2d4, 0x32bf7dc7, 0xb8b785f2, 0x59284ae1, 0x0ae71199, 0xeb78de8a,
        0x617026bf, 0x80efe9ac, 0xdc607fd5, 0x3dffb0c6, 0xb7f748f3, 0x566887e0,
        0x0f40cd01, 0xeedf0212, 0x64d7fa27, 0x85483534, 0xd9c7a34d, 0x38586c5e,
        0xb250946b, 0x53cf5b78, 0x1467229b, 0xf5f8ed88, 0x7ff015bd, 0x9e6fdaae,
        0xc2e04cd7, 0x237f83c4, 0xa9777bf1, 0x48e8b4e2, 0x11c0fe03, 0xf05f3110,
        0x7a57c925, 0x9bc80636, 0xc747904f, 0x26d85f5c, 0xacd0a769, 0x4d4f687a,
        0x1e803302, 0xff1ffc11, 0x75170424, 0x9488cb37, 0xc8075d4e, 0x2998925d,
        0xa3906a68, 0x420fa57b, 0x1b27ef9a, 0xfab82089, 0x70b0d8bc, 0x912f17af,
        0xcda081d6, 0x2c3f4ec5, 0xa637b6f0, 0x47a879e3, 0x28ce449f, 0xc9518b8c,
        0x435973b9, 0xa2c6bcaa, 0xfe492ad3, 0x1fd6e5c0, 0x95de1df5, 0x7441d2e6,
        0x2d699807, 0xccf65714, 0x46feaf21, 0xa7616032, 0xfbeef64b, 0x1a713958,
        0x9079c16d, 0x71e60e7e, 0x22295506, 0xc3b69a15, 0x49be6220, 0xa821ad33,
        0xf4ae3b4a, 0x1531f459, 0x9f390c6c, 0x7ea6c37f, 0x278e899e, 0xc611468d,
        0x4c19beb8, 0xad8671ab, 0xf109e7d2, 0x109628c1, 0x9a9ed0f4, 0x7b011fe7,
        0x3ca96604, 0xdd36a917, 0x573e5122, 0xb6a19e31, 0xea2e0848, 0x0bb1c75b,
        0x81b93f6e, 0x6026f07d, 0x390eba9c, 0xd891758f, 0x52998dba, 0xb30642a9,
        0xef89d4d0, 0x0e161bc3, 0x841ee3f6, 0x65812ce5, 0x364e779d, 0xd7d1b88e,
        0x5dd940bb, 0xbc468fa8, 0xe0c919d1, 0x0156d6c2, 0x8b5e2ef7, 0x6ac1e1e4,
        0x33e9ab05, 0xd2766416, 0x587e9c23, 0xb9e15330, 0xe56ec549, 0x04f10a5a,
        0x8ef9f26f, 0x6f663d7c, 0x50358897, 0xb1aa4784, 0x3ba2bfb1, 0xda3d70a2,
        0x86b2e6db, 0x672d29c8, 0xed25d1fd, 0x0cba1eee, 0x5592540f, 0xb40d9b1c,
        0x3e056329, 0xdf9aac3a, 0x83153a43, 0x628af550, 0xe8820d65, 0x091dc276,
        0x5ad2990e, 0xbb4d561d, 0x3145ae28, 0xd0da613b, 0x8c55f742, 0x6dca3851,
        0xe7c2c064, 0x065d0f77, 0x5f754596, 0xbeea8a85, 0x34e272b0, 0xd57dbda3,
        0x89f22bda, 0x686de4c9, 0xe2651cfc, 0x03fad3ef, 0x4452aa0c, 0xa5cd651f,
        0x2fc59d2a, 0xce5a5239, 0x92d5c440, 0x734a0b53, 0xf942f366, 0x18dd3c75,
        0x41f57694, 0xa06ab987, 0x2a6241b2, 0xcbfd8ea1, 0x977218d8, 0x76edd7cb,
        0xfce52ffe, 0x1d7ae0ed, 0x4eb5bb95, 0xaf2a7486, 0x25228cb3, 0xc4bd43a0,
        0x9832d5d9, 0x79ad1aca, 0xf3a5e2ff, 0x123a2dec, 0x4b12670d, 0xaa8da81e,
        0x2085502b, 0xc11a9f38, 0x9d950941, 0x7c0ac652, 0xf6023e67, 0x179df174,
        0x78fbcc08, 0x9964031b, 0x136cfb2e, 0xf2f3343d, 0xae7ca244, 0x4fe36d57,
        0xc5eb9562, 0x24745a71, 0x7d5c1090, 0x9cc3df83, 0x16cb27b6, 0xf754e8a5,
        0xabdb7edc, 0x4a44b1cf, 0xc04c49fa, 0x21d386e9, 0x721cdd91, 0x93831282,
        0x198beab7, 0xf81425a4, 0xa49bb3dd, 0x45047cce, 0xcf0c84fb, 0x2e934be8,
        0x77bb0109, 0x9624ce1a, 0x1c2c362f, 0xfdb3f93c, 0xa13c6f45, 0x40a3a056,
        0xcaab5863, 0x2b349770, 0x6c9cee93, 0x8d032180, 0x070bd9b5, 0xe69416a6,
        0xba1b80df, 0x5b844fcc, 0xd18cb7f9, 0x301378ea, 0x693b320b, 0x88a4fd18,
        0x02ac052d, 0xe333ca3e, 0xbfbc5c47, 0x5e239354, 0xd42b6b61, 0x35b4a472,
        0x667bff0a, 0x87e43019, 0x0decc82c, 0xec73073f, 0xb0fc9146, 0x51635e55,
        0xdb6ba660, 0x3af46973, 0x63dc2392, 0x8243ec81, 0x084b14b4, 0xe9d4dba7,
        0xb55b4dde, 0x54c482cd, 0xdecc7af8, 0x3f53b5eb};

IMB_DLL_LOCAL
DECLARE_ALIGNED(const int snow3g_table_A_div[256],32) = {
        0x00000000, 0x180f40cd, 0x301e8033, 0x2811c0fe, 0x603ca966, 0x7833e9ab,
        0x50222955, 0x482d6998, 0xc078fbcc, 0xd877bb01, 0xf0667bff, 0xe8693b32,
        0xa04452aa, 0xb84b1267, 0x905ad299, 0x88559254, 0x29f05f31, 0x31ff1ffc,
        0x19eedf02, 0x01e19fcf, 0x49ccf657, 0x51c3b69a, 0x79d27664, 0x61dd36a9,
        0xe988a4fd, 0xf187e430, 0xd99624ce, 0xc1996403, 0x89b40d9b, 0x91bb4d56,
        0xb9aa8da8, 0xa1a5cd65, 0x5249be62, 0x4a46feaf, 0x62573e51, 0x7a587e9c,
        0x32751704, 0x2a7a57c9, 0x026b9737, 0x1a64d7fa, 0x923145ae, 0x8a3e0563,
        0xa22fc59d, 0xba208550, 0xf20decc8, 0xea02ac05, 0xc2136cfb, 0xda1c2c36,
        0x7bb9e153, 0x63b6a19e, 0x4ba76160, 0x53a821ad, 0x1b854835, 0x038a08f8,
        0x2b9bc806, 0x339488cb, 0xbbc11a9f, 0xa3ce5a52, 0x8bdf9aac, 0x93d0da61,
        0xdbfdb3f9, 0xc3f2f334, 0xebe333ca, 0xf3ec7307, 0xa492d5c4, 0xbc9d9509,
        0x948c55f7, 0x8c83153a, 0xc4ae7ca2, 0xdca13c6f, 0xf4b0fc91, 0xecbfbc5c,
        0x64ea2e08, 0x7ce56ec5, 0x54f4ae3b, 0x4cfbeef6, 0x04d6876e, 0x1cd9c7a3,
        0x34c8075d, 0x2cc74790, 0x8d628af5, 0x956dca38, 0xbd7c0ac6, 0xa5734a0b,
        0xed5e2393, 0xf551635e, 0xdd40a3a0, 0xc54fe36d, 0x4d1a7139, 0x551531f4,
        0x7d04f10a, 0x650bb1c7, 0x2d26d85f, 0x35299892, 0x1d38586c, 0x053718a1,
        0xf6db6ba6, 0xeed42b6b, 0xc6c5eb95, 0xdecaab58, 0x96e7c2c0, 0x8ee8820d,
        0xa6f942f3, 0xbef6023e, 0x36a3906a, 0x2eacd0a7, 0x06bd1059, 0x1eb25094,
        0x569f390c, 0x4e9079c1, 0x6681b93f, 0x7e8ef9f2, 0xdf2b3497, 0xc724745a,
        0xef35b4a4, 0xf73af469, 0xbf179df1, 0xa718dd3c, 0x8f091dc2, 0x97065d0f,
        0x1f53cf5b, 0x075c8f96, 0x2f4d4f68, 0x37420fa5, 0x7f6f663d, 0x676026f0,
        0x4f71e60e, 0x577ea6c3, 0xe18d0321, 0xf98243ec, 0xd1938312, 0xc99cc3df,
        0x81b1aa47, 0x99beea8a, 0xb1af2a74, 0xa9a06ab9, 0x21f5f8ed, 0x39fab820,
        0x11eb78de, 0x09e43813, 0x41c9518b, 0x59c61146, 0x71d7d1b8, 0x69d89175,
        0xc87d5c10, 0xd0721cdd, 0xf863dc23, 0xe06c9cee, 0xa841f576, 0xb04eb5bb,
        0x985f7545, 0x80503588, 0x0805a7dc, 0x100ae711, 0x381b27ef, 0x20146722,
        0x68390eba, 0x70364e77, 0x58278e89, 0x4028ce44, 0xb3c4bd43, 0xabcbfd8e,
        0x83da3d70, 0x9bd57dbd, 0xd3f81425, 0xcbf754e8, 0xe3e69416, 0xfbe9d4db,
        0x73bc468f, 0x6bb30642, 0x43a2c6bc, 0x5bad8671, 0x1380efe9, 0x0b8faf24,
        0x239e6fda, 0x3b912f17, 0x9a34e272, 0x823ba2bf, 0xaa2a6241, 0xb225228c,
        0xfa084b14, 0xe2070bd9, 0xca16cb27, 0xd2198bea, 0x5a4c19be, 0x42435973,
        0x6a52998d, 0x725dd940, 0x3a70b0d8, 0x227ff015, 0x0a6e30eb, 0x12617026,
        0x451fd6e5, 0x5d109628, 0x750156d6, 0x6d0e161b, 0x25237f83, 0x3d2c3f4e,
        0x153dffb0, 0x0d32bf7d, 0x85672d29, 0x9d686de4, 0xb579ad1a, 0xad76edd7,
        0xe55b844f, 0xfd54c482, 0xd545047c, 0xcd4a44b1, 0x6cef89d4, 0x74e0c919,
        0x5cf109e7, 0x44fe492a, 0x0cd320b2, 0x14dc607f, 0x3ccda081, 0x24c2e04c,
        0xac977218, 0xb49832d5, 0x9c89f22b, 0x8486b2e6, 0xccabdb7e, 0xd4a49bb3,
        0xfcb55b4d, 0xe4ba1b80, 0x17566887, 0x0f59284a, 0x2748e8b4, 0x3f47a879,
        0x776ac1e1, 0x6f65812c, 0x477441d2, 0x5f7b011f, 0xd72e934b, 0xcf21d386,
        0xe7301378, 0xff3f53b5, 0xb7123a2d, 0xaf1d7ae0, 0x870cba1e, 0x9f03fad3,
        0x3ea637b6, 0x26a9777b, 0x0eb8b785, 0x16b7f748, 0x5e9a9ed0, 0x4695de1d,
        0x6e841ee3, 0x768b5e2e, 0xfedecc7a, 0xe6d18cb7, 0xcec04c49, 0xd6cf0c84,
        0x9ee2651c, 0x86ed25d1, 0xaefce52f, 0xb6f3a5e2};

IMB_DLL_LOCAL
snow3gTableEntry_t snow3g_table_S1[256] = {
        {.v = 0xc6a56363c6a56363ULL}, {.v = 0xf8847c7cf8847c7cULL},
        {.v = 0xee997777ee997777ULL}, {.v = 0xf68d7b7bf68d7b7bULL},
        {.v = 0xff0df2f2ff0df2f2ULL}, {.v = 0xd6bd6b6bd6bd6b6bULL},
        {.v = 0xdeb16f6fdeb16f6fULL}, {.v = 0x9154c5c59154c5c5ULL},
        {.v = 0x6050303060503030ULL}, {.v = 0x0203010102030101ULL},
        {.v = 0xcea96767cea96767ULL}, {.v = 0x567d2b2b567d2b2bULL},
        {.v = 0xe719fefee719fefeULL}, {.v = 0xb562d7d7b562d7d7ULL},
        {.v = 0x4de6abab4de6ababULL}, {.v = 0xec9a7676ec9a7676ULL},
        {.v = 0x8f45caca8f45cacaULL}, {.v = 0x1f9d82821f9d8282ULL},
        {.v = 0x8940c9c98940c9c9ULL}, {.v = 0xfa877d7dfa877d7dULL},
        {.v = 0xef15fafaef15fafaULL}, {.v = 0xb2eb5959b2eb5959ULL},
        {.v = 0x8ec947478ec94747ULL}, {.v = 0xfb0bf0f0fb0bf0f0ULL},
        {.v = 0x41ecadad41ecadadULL}, {.v = 0xb367d4d4b367d4d4ULL},
        {.v = 0x5ffda2a25ffda2a2ULL}, {.v = 0x45eaafaf45eaafafULL},
        {.v = 0x23bf9c9c23bf9c9cULL}, {.v = 0x53f7a4a453f7a4a4ULL},
        {.v = 0xe4967272e4967272ULL}, {.v = 0x9b5bc0c09b5bc0c0ULL},
        {.v = 0x75c2b7b775c2b7b7ULL}, {.v = 0xe11cfdfde11cfdfdULL},
        {.v = 0x3dae93933dae9393ULL}, {.v = 0x4c6a26264c6a2626ULL},
        {.v = 0x6c5a36366c5a3636ULL}, {.v = 0x7e413f3f7e413f3fULL},
        {.v = 0xf502f7f7f502f7f7ULL}, {.v = 0x834fcccc834fccccULL},
        {.v = 0x685c3434685c3434ULL}, {.v = 0x51f4a5a551f4a5a5ULL},
        {.v = 0xd134e5e5d134e5e5ULL}, {.v = 0xf908f1f1f908f1f1ULL},
        {.v = 0xe2937171e2937171ULL}, {.v = 0xab73d8d8ab73d8d8ULL},
        {.v = 0x6253313162533131ULL}, {.v = 0x2a3f15152a3f1515ULL},
        {.v = 0x080c0404080c0404ULL}, {.v = 0x9552c7c79552c7c7ULL},
        {.v = 0x4665232346652323ULL}, {.v = 0x9d5ec3c39d5ec3c3ULL},
        {.v = 0x3028181830281818ULL}, {.v = 0x37a1969637a19696ULL},
        {.v = 0x0a0f05050a0f0505ULL}, {.v = 0x2fb59a9a2fb59a9aULL},
        {.v = 0x0e0907070e090707ULL}, {.v = 0x2436121224361212ULL},
        {.v = 0x1b9b80801b9b8080ULL}, {.v = 0xdf3de2e2df3de2e2ULL},
        {.v = 0xcd26ebebcd26ebebULL}, {.v = 0x4e6927274e692727ULL},
        {.v = 0x7fcdb2b27fcdb2b2ULL}, {.v = 0xea9f7575ea9f7575ULL},
        {.v = 0x121b0909121b0909ULL}, {.v = 0x1d9e83831d9e8383ULL},
        {.v = 0x58742c2c58742c2cULL}, {.v = 0x342e1a1a342e1a1aULL},
        {.v = 0x362d1b1b362d1b1bULL}, {.v = 0xdcb26e6edcb26e6eULL},
        {.v = 0xb4ee5a5ab4ee5a5aULL}, {.v = 0x5bfba0a05bfba0a0ULL},
        {.v = 0xa4f65252a4f65252ULL}, {.v = 0x764d3b3b764d3b3bULL},
        {.v = 0xb761d6d6b761d6d6ULL}, {.v = 0x7dceb3b37dceb3b3ULL},
        {.v = 0x527b2929527b2929ULL}, {.v = 0xdd3ee3e3dd3ee3e3ULL},
        {.v = 0x5e712f2f5e712f2fULL}, {.v = 0x1397848413978484ULL},
        {.v = 0xa6f55353a6f55353ULL}, {.v = 0xb968d1d1b968d1d1ULL},
        {.v = 0x0000000000000000ULL}, {.v = 0xc12cededc12cededULL},
        {.v = 0x4060202040602020ULL}, {.v = 0xe31ffcfce31ffcfcULL},
        {.v = 0x79c8b1b179c8b1b1ULL}, {.v = 0xb6ed5b5bb6ed5b5bULL},
        {.v = 0xd4be6a6ad4be6a6aULL}, {.v = 0x8d46cbcb8d46cbcbULL},
        {.v = 0x67d9bebe67d9bebeULL}, {.v = 0x724b3939724b3939ULL},
        {.v = 0x94de4a4a94de4a4aULL}, {.v = 0x98d44c4c98d44c4cULL},
        {.v = 0xb0e85858b0e85858ULL}, {.v = 0x854acfcf854acfcfULL},
        {.v = 0xbb6bd0d0bb6bd0d0ULL}, {.v = 0xc52aefefc52aefefULL},
        {.v = 0x4fe5aaaa4fe5aaaaULL}, {.v = 0xed16fbfbed16fbfbULL},
        {.v = 0x86c5434386c54343ULL}, {.v = 0x9ad74d4d9ad74d4dULL},
        {.v = 0x6655333366553333ULL}, {.v = 0x1194858511948585ULL},
        {.v = 0x8acf45458acf4545ULL}, {.v = 0xe910f9f9e910f9f9ULL},
        {.v = 0x0406020204060202ULL}, {.v = 0xfe817f7ffe817f7fULL},
        {.v = 0xa0f05050a0f05050ULL}, {.v = 0x78443c3c78443c3cULL},
        {.v = 0x25ba9f9f25ba9f9fULL}, {.v = 0x4be3a8a84be3a8a8ULL},
        {.v = 0xa2f35151a2f35151ULL}, {.v = 0x5dfea3a35dfea3a3ULL},
        {.v = 0x80c0404080c04040ULL}, {.v = 0x058a8f8f058a8f8fULL},
        {.v = 0x3fad92923fad9292ULL}, {.v = 0x21bc9d9d21bc9d9dULL},
        {.v = 0x7048383870483838ULL}, {.v = 0xf104f5f5f104f5f5ULL},
        {.v = 0x63dfbcbc63dfbcbcULL}, {.v = 0x77c1b6b677c1b6b6ULL},
        {.v = 0xaf75dadaaf75dadaULL}, {.v = 0x4263212142632121ULL},
        {.v = 0x2030101020301010ULL}, {.v = 0xe51affffe51affffULL},
        {.v = 0xfd0ef3f3fd0ef3f3ULL}, {.v = 0xbf6dd2d2bf6dd2d2ULL},
        {.v = 0x814ccdcd814ccdcdULL}, {.v = 0x18140c0c18140c0cULL},
        {.v = 0x2635131326351313ULL}, {.v = 0xc32fececc32fececULL},
        {.v = 0xbee15f5fbee15f5fULL}, {.v = 0x35a2979735a29797ULL},
        {.v = 0x88cc444488cc4444ULL}, {.v = 0x2e3917172e391717ULL},
        {.v = 0x9357c4c49357c4c4ULL}, {.v = 0x55f2a7a755f2a7a7ULL},
        {.v = 0xfc827e7efc827e7eULL}, {.v = 0x7a473d3d7a473d3dULL},
        {.v = 0xc8ac6464c8ac6464ULL}, {.v = 0xbae75d5dbae75d5dULL},
        {.v = 0x322b1919322b1919ULL}, {.v = 0xe6957373e6957373ULL},
        {.v = 0xc0a06060c0a06060ULL}, {.v = 0x1998818119988181ULL},
        {.v = 0x9ed14f4f9ed14f4fULL}, {.v = 0xa37fdcdca37fdcdcULL},
        {.v = 0x4466222244662222ULL}, {.v = 0x547e2a2a547e2a2aULL},
        {.v = 0x3bab90903bab9090ULL}, {.v = 0x0b8388880b838888ULL},
        {.v = 0x8cca46468cca4646ULL}, {.v = 0xc729eeeec729eeeeULL},
        {.v = 0x6bd3b8b86bd3b8b8ULL}, {.v = 0x283c1414283c1414ULL},
        {.v = 0xa779dedea779dedeULL}, {.v = 0xbce25e5ebce25e5eULL},
        {.v = 0x161d0b0b161d0b0bULL}, {.v = 0xad76dbdbad76dbdbULL},
        {.v = 0xdb3be0e0db3be0e0ULL}, {.v = 0x6456323264563232ULL},
        {.v = 0x744e3a3a744e3a3aULL}, {.v = 0x141e0a0a141e0a0aULL},
        {.v = 0x92db494992db4949ULL}, {.v = 0x0c0a06060c0a0606ULL},
        {.v = 0x486c2424486c2424ULL}, {.v = 0xb8e45c5cb8e45c5cULL},
        {.v = 0x9f5dc2c29f5dc2c2ULL}, {.v = 0xbd6ed3d3bd6ed3d3ULL},
        {.v = 0x43efacac43efacacULL}, {.v = 0xc4a66262c4a66262ULL},
        {.v = 0x39a8919139a89191ULL}, {.v = 0x31a4959531a49595ULL},
        {.v = 0xd337e4e4d337e4e4ULL}, {.v = 0xf28b7979f28b7979ULL},
        {.v = 0xd532e7e7d532e7e7ULL}, {.v = 0x8b43c8c88b43c8c8ULL},
        {.v = 0x6e5937376e593737ULL}, {.v = 0xdab76d6ddab76d6dULL},
        {.v = 0x018c8d8d018c8d8dULL}, {.v = 0xb164d5d5b164d5d5ULL},
        {.v = 0x9cd24e4e9cd24e4eULL}, {.v = 0x49e0a9a949e0a9a9ULL},
        {.v = 0xd8b46c6cd8b46c6cULL}, {.v = 0xacfa5656acfa5656ULL},
        {.v = 0xf307f4f4f307f4f4ULL}, {.v = 0xcf25eaeacf25eaeaULL},
        {.v = 0xcaaf6565caaf6565ULL}, {.v = 0xf48e7a7af48e7a7aULL},
        {.v = 0x47e9aeae47e9aeaeULL}, {.v = 0x1018080810180808ULL},
        {.v = 0x6fd5baba6fd5babaULL}, {.v = 0xf0887878f0887878ULL},
        {.v = 0x4a6f25254a6f2525ULL}, {.v = 0x5c722e2e5c722e2eULL},
        {.v = 0x38241c1c38241c1cULL}, {.v = 0x57f1a6a657f1a6a6ULL},
        {.v = 0x73c7b4b473c7b4b4ULL}, {.v = 0x9751c6c69751c6c6ULL},
        {.v = 0xcb23e8e8cb23e8e8ULL}, {.v = 0xa17cdddda17cddddULL},
        {.v = 0xe89c7474e89c7474ULL}, {.v = 0x3e211f1f3e211f1fULL},
        {.v = 0x96dd4b4b96dd4b4bULL}, {.v = 0x61dcbdbd61dcbdbdULL},
        {.v = 0x0d868b8b0d868b8bULL}, {.v = 0x0f858a8a0f858a8aULL},
        {.v = 0xe0907070e0907070ULL}, {.v = 0x7c423e3e7c423e3eULL},
        {.v = 0x71c4b5b571c4b5b5ULL}, {.v = 0xccaa6666ccaa6666ULL},
        {.v = 0x90d8484890d84848ULL}, {.v = 0x0605030306050303ULL},
        {.v = 0xf701f6f6f701f6f6ULL}, {.v = 0x1c120e0e1c120e0eULL},
        {.v = 0xc2a36161c2a36161ULL}, {.v = 0x6a5f35356a5f3535ULL},
        {.v = 0xaef95757aef95757ULL}, {.v = 0x69d0b9b969d0b9b9ULL},
        {.v = 0x1791868617918686ULL}, {.v = 0x9958c1c19958c1c1ULL},
        {.v = 0x3a271d1d3a271d1dULL}, {.v = 0x27b99e9e27b99e9eULL},
        {.v = 0xd938e1e1d938e1e1ULL}, {.v = 0xeb13f8f8eb13f8f8ULL},
        {.v = 0x2bb398982bb39898ULL}, {.v = 0x2233111122331111ULL},
        {.v = 0xd2bb6969d2bb6969ULL}, {.v = 0xa970d9d9a970d9d9ULL},
        {.v = 0x07898e8e07898e8eULL}, {.v = 0x33a7949433a79494ULL},
        {.v = 0x2db69b9b2db69b9bULL}, {.v = 0x3c221e1e3c221e1eULL},
        {.v = 0x1592878715928787ULL}, {.v = 0xc920e9e9c920e9e9ULL},
        {.v = 0x8749cece8749ceceULL}, {.v = 0xaaff5555aaff5555ULL},
        {.v = 0x5078282850782828ULL}, {.v = 0xa57adfdfa57adfdfULL},
        {.v = 0x038f8c8c038f8c8cULL}, {.v = 0x59f8a1a159f8a1a1ULL},
        {.v = 0x0980898909808989ULL}, {.v = 0x1a170d0d1a170d0dULL},
        {.v = 0x65dabfbf65dabfbfULL}, {.v = 0xd731e6e6d731e6e6ULL},
        {.v = 0x84c6424284c64242ULL}, {.v = 0xd0b86868d0b86868ULL},
        {.v = 0x82c3414182c34141ULL}, {.v = 0x29b0999929b09999ULL},
        {.v = 0x5a772d2d5a772d2dULL}, {.v = 0x1e110f0f1e110f0fULL},
        {.v = 0x7bcbb0b07bcbb0b0ULL}, {.v = 0xa8fc5454a8fc5454ULL},
        {.v = 0x6dd6bbbb6dd6bbbbULL}, {.v = 0x2c3a16162c3a1616ULL}};

IMB_DLL_LOCAL
snow3gTableEntry_t snow3g_table_S2[256] = {
        {.v = 0x4a6f25254a6f2525ULL}, {.v = 0x486c2424486c2424ULL},
        {.v = 0xe6957373e6957373ULL}, {.v = 0xcea96767cea96767ULL},
        {.v = 0xc710d7d7c710d7d7ULL}, {.v = 0x359baeae359baeaeULL},
        {.v = 0xb8e45c5cb8e45c5cULL}, {.v = 0x6050303060503030ULL},
        {.v = 0x2185a4a42185a4a4ULL}, {.v = 0xb55beeeeb55beeeeULL},
        {.v = 0xdcb26e6edcb26e6eULL}, {.v = 0xff34cbcbff34cbcbULL},
        {.v = 0xfa877d7dfa877d7dULL}, {.v = 0x03b6b5b503b6b5b5ULL},
        {.v = 0x6def82826def8282ULL}, {.v = 0xdf04dbdbdf04dbdbULL},
        {.v = 0xa145e4e4a145e4e4ULL}, {.v = 0x75fb8e8e75fb8e8eULL},
        {.v = 0x90d8484890d84848ULL}, {.v = 0x92db494992db4949ULL},
        {.v = 0x9ed14f4f9ed14f4fULL}, {.v = 0xbae75d5dbae75d5dULL},
        {.v = 0xd4be6a6ad4be6a6aULL}, {.v = 0xf0887878f0887878ULL},
        {.v = 0xe0907070e0907070ULL}, {.v = 0x79f1888879f18888ULL},
        {.v = 0xb951e8e8b951e8e8ULL}, {.v = 0xbee15f5fbee15f5fULL},
        {.v = 0xbce25e5ebce25e5eULL}, {.v = 0x61e5848461e58484ULL},
        {.v = 0xcaaf6565caaf6565ULL}, {.v = 0xad4fe2e2ad4fe2e2ULL},
        {.v = 0xd901d8d8d901d8d8ULL}, {.v = 0xbb52e9e9bb52e9e9ULL},
        {.v = 0xf13dccccf13dccccULL}, {.v = 0xb35eededb35eededULL},
        {.v = 0x80c0404080c04040ULL}, {.v = 0x5e712f2f5e712f2fULL},
        {.v = 0x2233111122331111ULL}, {.v = 0x5078282850782828ULL},
        {.v = 0xaef95757aef95757ULL}, {.v = 0xcd1fd2d2cd1fd2d2ULL},
        {.v = 0x319dacac319dacacULL}, {.v = 0xaf4ce3e3af4ce3e3ULL},
        {.v = 0x94de4a4a94de4a4aULL}, {.v = 0x2a3f15152a3f1515ULL},
        {.v = 0x362d1b1b362d1b1bULL}, {.v = 0x1ba2b9b91ba2b9b9ULL},
        {.v = 0x0dbfb2b20dbfb2b2ULL}, {.v = 0x69e9808069e98080ULL},
        {.v = 0x63e6858563e68585ULL}, {.v = 0x2583a6a62583a6a6ULL},
        {.v = 0x5c722e2e5c722e2eULL}, {.v = 0x0406020204060202ULL},
        {.v = 0x8ec947478ec94747ULL}, {.v = 0x527b2929527b2929ULL},
        {.v = 0x0e0907070e090707ULL}, {.v = 0x96dd4b4b96dd4b4bULL},
        {.v = 0x1c120e0e1c120e0eULL}, {.v = 0xeb2ac1c1eb2ac1c1ULL},
        {.v = 0xa2f35151a2f35151ULL}, {.v = 0x3d97aaaa3d97aaaaULL},
        {.v = 0x7bf289897bf28989ULL}, {.v = 0xc115d4d4c115d4d4ULL},
        {.v = 0xfd37cacafd37cacaULL}, {.v = 0x0203010102030101ULL},
        {.v = 0x8cca46468cca4646ULL}, {.v = 0x0fbcb3b30fbcb3b3ULL},
        {.v = 0xb758efefb758efefULL}, {.v = 0xd30eddddd30eddddULL},
        {.v = 0x88cc444488cc4444ULL}, {.v = 0xf68d7b7bf68d7b7bULL},
        {.v = 0xed2fc2c2ed2fc2c2ULL}, {.v = 0xfe817f7ffe817f7fULL},
        {.v = 0x15abbebe15abbebeULL}, {.v = 0xef2cc3c3ef2cc3c3ULL},
        {.v = 0x57c89f9f57c89f9fULL}, {.v = 0x4060202040602020ULL},
        {.v = 0x98d44c4c98d44c4cULL}, {.v = 0xc8ac6464c8ac6464ULL},
        {.v = 0x6fec83836fec8383ULL}, {.v = 0x2d8fa2a22d8fa2a2ULL},
        {.v = 0xd0b86868d0b86868ULL}, {.v = 0x84c6424284c64242ULL},
        {.v = 0x2635131326351313ULL}, {.v = 0x01b5b4b401b5b4b4ULL},
        {.v = 0x82c3414182c34141ULL}, {.v = 0xf33ecdcdf33ecdcdULL},
        {.v = 0x1da7baba1da7babaULL}, {.v = 0xe523c6c6e523c6c6ULL},
        {.v = 0x1fa4bbbb1fa4bbbbULL}, {.v = 0xdab76d6ddab76d6dULL},
        {.v = 0x9ad74d4d9ad74d4dULL}, {.v = 0xe2937171e2937171ULL},
        {.v = 0x4263212142632121ULL}, {.v = 0x8175f4f48175f4f4ULL},
        {.v = 0x73fe8d8d73fe8d8dULL}, {.v = 0x09b9b0b009b9b0b0ULL},
        {.v = 0xa346e5e5a346e5e5ULL}, {.v = 0x4fdc93934fdc9393ULL},
        {.v = 0x956bfefe956bfefeULL}, {.v = 0x77f88f8f77f88f8fULL},
        {.v = 0xa543e6e6a543e6e6ULL}, {.v = 0xf738cfcff738cfcfULL},
        {.v = 0x86c5434386c54343ULL}, {.v = 0x8acf45458acf4545ULL},
        {.v = 0x6253313162533131ULL}, {.v = 0x4466222244662222ULL},
        {.v = 0x6e5937376e593737ULL}, {.v = 0x6c5a36366c5a3636ULL},
        {.v = 0x45d3969645d39696ULL}, {.v = 0x9d67fafa9d67fafaULL},
        {.v = 0x11adbcbc11adbcbcULL}, {.v = 0x1e110f0f1e110f0fULL},
        {.v = 0x1018080810180808ULL}, {.v = 0xa4f65252a4f65252ULL},
        {.v = 0x3a271d1d3a271d1dULL}, {.v = 0xaaff5555aaff5555ULL},
        {.v = 0x342e1a1a342e1a1aULL}, {.v = 0xe326c5c5e326c5c5ULL},
        {.v = 0x9cd24e4e9cd24e4eULL}, {.v = 0x4665232346652323ULL},
        {.v = 0xd2bb6969d2bb6969ULL}, {.v = 0xf48e7a7af48e7a7aULL},
        {.v = 0x4ddf92924ddf9292ULL}, {.v = 0x9768ffff9768ffffULL},
        {.v = 0xb6ed5b5bb6ed5b5bULL}, {.v = 0xb4ee5a5ab4ee5a5aULL},
        {.v = 0xbf54ebebbf54ebebULL}, {.v = 0x5dc79a9a5dc79a9aULL},
        {.v = 0x38241c1c38241c1cULL}, {.v = 0x3b92a9a93b92a9a9ULL},
        {.v = 0xcb1ad1d1cb1ad1d1ULL}, {.v = 0xfc827e7efc827e7eULL},
        {.v = 0x1a170d0d1a170d0dULL}, {.v = 0x916dfcfc916dfcfcULL},
        {.v = 0xa0f05050a0f05050ULL}, {.v = 0x7df78a8a7df78a8aULL},
        {.v = 0x05b3b6b605b3b6b6ULL}, {.v = 0xc4a66262c4a66262ULL},
        {.v = 0x8376f5f58376f5f5ULL}, {.v = 0x141e0a0a141e0a0aULL},
        {.v = 0x9961f8f89961f8f8ULL}, {.v = 0xd10ddcdcd10ddcdcULL},
        {.v = 0x0605030306050303ULL}, {.v = 0x78443c3c78443c3cULL},
        {.v = 0x18140c0c18140c0cULL}, {.v = 0x724b3939724b3939ULL},
        {.v = 0x8b7af1f18b7af1f1ULL}, {.v = 0x19a1b8b819a1b8b8ULL},
        {.v = 0x8f7cf3f38f7cf3f3ULL}, {.v = 0x7a473d3d7a473d3dULL},
        {.v = 0x8d7ff2f28d7ff2f2ULL}, {.v = 0xc316d5d5c316d5d5ULL},
        {.v = 0x47d0979747d09797ULL}, {.v = 0xccaa6666ccaa6666ULL},
        {.v = 0x6bea81816bea8181ULL}, {.v = 0x6456323264563232ULL},
        {.v = 0x2989a0a02989a0a0ULL}, {.v = 0x0000000000000000ULL},
        {.v = 0x0c0a06060c0a0606ULL}, {.v = 0xf53bcecef53bceceULL},
        {.v = 0x8573f6f68573f6f6ULL}, {.v = 0xbd57eaeabd57eaeaULL},
        {.v = 0x07b0b7b707b0b7b7ULL}, {.v = 0x2e3917172e391717ULL},
        {.v = 0x8770f7f78770f7f7ULL}, {.v = 0x71fd8c8c71fd8c8cULL},
        {.v = 0xf28b7979f28b7979ULL}, {.v = 0xc513d6d6c513d6d6ULL},
        {.v = 0x2780a7a72780a7a7ULL}, {.v = 0x17a8bfbf17a8bfbfULL},
        {.v = 0x7ff48b8b7ff48b8bULL}, {.v = 0x7e413f3f7e413f3fULL},
        {.v = 0x3e211f1f3e211f1fULL}, {.v = 0xa6f55353a6f55353ULL},
        {.v = 0xc6a56363c6a56363ULL}, {.v = 0xea9f7575ea9f7575ULL},
        {.v = 0x6a5f35356a5f3535ULL}, {.v = 0x58742c2c58742c2cULL},
        {.v = 0xc0a06060c0a06060ULL}, {.v = 0x936efdfd936efdfdULL},
        {.v = 0x4e6927274e692727ULL}, {.v = 0xcf1cd3d3cf1cd3d3ULL},
        {.v = 0x41d5949441d59494ULL}, {.v = 0x2386a5a52386a5a5ULL},
        {.v = 0xf8847c7cf8847c7cULL}, {.v = 0x2b8aa1a12b8aa1a1ULL},
        {.v = 0x0a0f05050a0f0505ULL}, {.v = 0xb0e85858b0e85858ULL},
        {.v = 0x5a772d2d5a772d2dULL}, {.v = 0x13aebdbd13aebdbdULL},
        {.v = 0xdb02d9d9db02d9d9ULL}, {.v = 0xe720c7c7e720c7c7ULL},
        {.v = 0x3798afaf3798afafULL}, {.v = 0xd6bd6b6bd6bd6b6bULL},
        {.v = 0xa8fc5454a8fc5454ULL}, {.v = 0x161d0b0b161d0b0bULL},
        {.v = 0xa949e0e0a949e0e0ULL}, {.v = 0x7048383870483838ULL},
        {.v = 0x080c0404080c0404ULL}, {.v = 0xf931c8c8f931c8c8ULL},
        {.v = 0x53ce9d9d53ce9d9dULL}, {.v = 0xa740e7e7a740e7e7ULL},
        {.v = 0x283c1414283c1414ULL}, {.v = 0x0bbab1b10bbab1b1ULL},
        {.v = 0x67e0878767e08787ULL}, {.v = 0x51cd9c9c51cd9c9cULL},
        {.v = 0xd708dfdfd708dfdfULL}, {.v = 0xdeb16f6fdeb16f6fULL},
        {.v = 0x9b62f9f99b62f9f9ULL}, {.v = 0xdd07dadadd07dadaULL},
        {.v = 0x547e2a2a547e2a2aULL}, {.v = 0xe125c4c4e125c4c4ULL},
        {.v = 0xb2eb5959b2eb5959ULL}, {.v = 0x2c3a16162c3a1616ULL},
        {.v = 0xe89c7474e89c7474ULL}, {.v = 0x4bda91914bda9191ULL},
        {.v = 0x3f94abab3f94ababULL}, {.v = 0x4c6a26264c6a2626ULL},
        {.v = 0xc2a36161c2a36161ULL}, {.v = 0xec9a7676ec9a7676ULL},
        {.v = 0x685c3434685c3434ULL}, {.v = 0x567d2b2b567d2b2bULL},
        {.v = 0x339eadad339eadadULL}, {.v = 0x5bc299995bc29999ULL},
        {.v = 0x9f64fbfb9f64fbfbULL}, {.v = 0xe4967272e4967272ULL},
        {.v = 0xb15dececb15dececULL}, {.v = 0x6655333366553333ULL},
        {.v = 0x2436121224361212ULL}, {.v = 0xd50bdeded50bdedeULL},
        {.v = 0x59c1989859c19898ULL}, {.v = 0x764d3b3b764d3b3bULL},
        {.v = 0xe929c0c0e929c0c0ULL}, {.v = 0x5fc49b9b5fc49b9bULL},
        {.v = 0x7c423e3e7c423e3eULL}, {.v = 0x3028181830281818ULL},
        {.v = 0x2030101020301010ULL}, {.v = 0x744e3a3a744e3a3aULL},
        {.v = 0xacfa5656acfa5656ULL}, {.v = 0xab4ae1e1ab4ae1e1ULL},
        {.v = 0xee997777ee997777ULL}, {.v = 0xfb32c9c9fb32c9c9ULL},
        {.v = 0x3c221e1e3c221e1eULL}, {.v = 0x55cb9e9e55cb9e9eULL},
        {.v = 0x43d6959543d69595ULL}, {.v = 0x2f8ca3a32f8ca3a3ULL},
        {.v = 0x49d9909049d99090ULL}, {.v = 0x322b1919322b1919ULL},
        {.v = 0x3991a8a83991a8a8ULL}, {.v = 0xd8b46c6cd8b46c6cULL},
        {.v = 0x121b0909121b0909ULL}, {.v = 0xc919d0d0c919d0d0ULL},
        {.v = 0x8979f0f08979f0f0ULL}, {.v = 0x65e3868665e38686ULL}};

IMB_DLL_LOCAL
DECLARE_ALIGNED(const int S1_T0[256], 32) = {
        0xA56363C6, 0x847C7CF8, 0x997777EE, 0x8D7B7BF6, 0xDF2F2FF,  0xBD6B6BD6,
        0xB16F6FDE, 0x54C5C591, 0x50303060, 0x3010102,  0xA96767CE, 0x7D2B2B56,
        0x19FEFEE7, 0x62D7D7B5, 0xE6ABAB4D, 0x9A7676EC, 0x45CACA8F, 0x9D82821F,
        0x40C9C989, 0x877D7DFA, 0x15FAFAEF, 0xEB5959B2, 0xC947478E, 0xBF0F0FB,
        0xECADAD41, 0x67D4D4B3, 0xFDA2A25F, 0xEAAFAF45, 0xBF9C9C23, 0xF7A4A453,
        0x967272E4, 0x5BC0C09B, 0xC2B7B775, 0x1CFDFDE1, 0xAE93933D, 0x6A26264C,
        0x5A36366C, 0x413F3F7E, 0x2F7F7F5,  0x4FCCCC83, 0x5C343468, 0xF4A5A551,
        0x34E5E5D1, 0x8F1F1F9,  0x937171E2, 0x73D8D8AB, 0x53313162, 0x3F15152A,
        0xC040408,  0x52C7C795, 0x65232346, 0x5EC3C39D, 0x28181830, 0xA1969637,
        0xF05050A,  0xB59A9A2F, 0x907070E,  0x36121224, 0x9B80801B, 0x3DE2E2DF,
        0x26EBEBCD, 0x6927274E, 0xCDB2B27F, 0x9F7575EA, 0x1B090912, 0x9E83831D,
        0x742C2C58, 0x2E1A1A34, 0x2D1B1B36, 0xB26E6EDC, 0xEE5A5AB4, 0xFBA0A05B,
        0xF65252A4, 0x4D3B3B76, 0x61D6D6B7, 0xCEB3B37D, 0x7B292952, 0x3EE3E3DD,
        0x712F2F5E, 0x97848413, 0xF55353A6, 0x68D1D1B9, 0x0,        0x2CEDEDC1,
        0x60202040, 0x1FFCFCE3, 0xC8B1B179, 0xED5B5BB6, 0xBE6A6AD4, 0x46CBCB8D,
        0xD9BEBE67, 0x4B393972, 0xDE4A4A94, 0xD44C4C98, 0xE85858B0, 0x4ACFCF85,
        0x6BD0D0BB, 0x2AEFEFC5, 0xE5AAAA4F, 0x16FBFBED, 0xC5434386, 0xD74D4D9A,
        0x55333366, 0x94858511, 0xCF45458A, 0x10F9F9E9, 0x6020204,  0x817F7FFE,
        0xF05050A0, 0x443C3C78, 0xBA9F9F25, 0xE3A8A84B, 0xF35151A2, 0xFEA3A35D,
        0xC0404080, 0x8A8F8F05, 0xAD92923F, 0xBC9D9D21, 0x48383870, 0x4F5F5F1,
        0xDFBCBC63, 0xC1B6B677, 0x75DADAAF, 0x63212142, 0x30101020, 0x1AFFFFE5,
        0xEF3F3FD,  0x6DD2D2BF, 0x4CCDCD81, 0x140C0C18, 0x35131326, 0x2FECECC3,
        0xE15F5FBE, 0xA2979735, 0xCC444488, 0x3917172E, 0x57C4C493, 0xF2A7A755,
        0x827E7EFC, 0x473D3D7A, 0xAC6464C8, 0xE75D5DBA, 0x2B191932, 0x957373E6,
        0xA06060C0, 0x98818119, 0xD14F4F9E, 0x7FDCDCA3, 0x66222244, 0x7E2A2A54,
        0xAB90903B, 0x8388880B, 0xCA46468C, 0x29EEEEC7, 0xD3B8B86B, 0x3C141428,
        0x79DEDEA7, 0xE25E5EBC, 0x1D0B0B16, 0x76DBDBAD, 0x3BE0E0DB, 0x56323264,
        0x4E3A3A74, 0x1E0A0A14, 0xDB494992, 0xA06060C,  0x6C242448, 0xE45C5CB8,
        0x5DC2C29F, 0x6ED3D3BD, 0xEFACAC43, 0xA66262C4, 0xA8919139, 0xA4959531,
        0x37E4E4D3, 0x8B7979F2, 0x32E7E7D5, 0x43C8C88B, 0x5937376E, 0xB76D6DDA,
        0x8C8D8D01, 0x64D5D5B1, 0xD24E4E9C, 0xE0A9A949, 0xB46C6CD8, 0xFA5656AC,
        0x7F4F4F3,  0x25EAEACF, 0xAF6565CA, 0x8E7A7AF4, 0xE9AEAE47, 0x18080810,
        0xD5BABA6F, 0x887878F0, 0x6F25254A, 0x722E2E5C, 0x241C1C38, 0xF1A6A657,
        0xC7B4B473, 0x51C6C697, 0x23E8E8CB, 0x7CDDDDA1, 0x9C7474E8, 0x211F1F3E,
        0xDD4B4B96, 0xDCBDBD61, 0x868B8B0D, 0x858A8A0F, 0x907070E0, 0x423E3E7C,
        0xC4B5B571, 0xAA6666CC, 0xD8484890, 0x5030306,  0x1F6F6F7,  0x120E0E1C,
        0xA36161C2, 0x5F35356A, 0xF95757AE, 0xD0B9B969, 0x91868617, 0x58C1C199,
        0x271D1D3A, 0xB99E9E27, 0x38E1E1D9, 0x13F8F8EB, 0xB398982B, 0x33111122,
        0xBB6969D2, 0x70D9D9A9, 0x898E8E07, 0xA7949433, 0xB69B9B2D, 0x221E1E3C,
        0x92878715, 0x20E9E9C9, 0x49CECE87, 0xFF5555AA, 0x78282850, 0x7ADFDFA5,
        0x8F8C8C03, 0xF8A1A159, 0x80898909, 0x170D0D1A, 0xDABFBF65, 0x31E6E6D7,
        0xC6424284, 0xB86868D0, 0xC3414182, 0xB0999929, 0x772D2D5A, 0x110F0F1E,
        0xCBB0B07B, 0xFC5454A8, 0xD6BBBB6D, 0x3A16162C};

IMB_DLL_LOCAL
DECLARE_ALIGNED(const int S1_T1[256], 32) = {
        0x6363C6A5, 0x7C7CF884, 0x7777EE99, 0x7B7BF68D, 0xF2F2FF0D, 0x6B6BD6BD,
        0x6F6FDEB1, 0xC5C59154, 0x30306050, 0x1010203,  0x6767CEA9, 0x2B2B567D,
        0xFEFEE719, 0xD7D7B562, 0xABAB4DE6, 0x7676EC9A, 0xCACA8F45, 0x82821F9D,
        0xC9C98940, 0x7D7DFA87, 0xFAFAEF15, 0x5959B2EB, 0x47478EC9, 0xF0F0FB0B,
        0xADAD41EC, 0xD4D4B367, 0xA2A25FFD, 0xAFAF45EA, 0x9C9C23BF, 0xA4A453F7,
        0x7272E496, 0xC0C09B5B, 0xB7B775C2, 0xFDFDE11C, 0x93933DAE, 0x26264C6A,
        0x36366C5A, 0x3F3F7E41, 0xF7F7F502, 0xCCCC834F, 0x3434685C, 0xA5A551F4,
        0xE5E5D134, 0xF1F1F908, 0x7171E293, 0xD8D8AB73, 0x31316253, 0x15152A3F,
        0x404080C,  0xC7C79552, 0x23234665, 0xC3C39D5E, 0x18183028, 0x969637A1,
        0x5050A0F,  0x9A9A2FB5, 0x7070E09,  0x12122436, 0x80801B9B, 0xE2E2DF3D,
        0xEBEBCD26, 0x27274E69, 0xB2B27FCD, 0x7575EA9F, 0x909121B,  0x83831D9E,
        0x2C2C5874, 0x1A1A342E, 0x1B1B362D, 0x6E6EDCB2, 0x5A5AB4EE, 0xA0A05BFB,
        0x5252A4F6, 0x3B3B764D, 0xD6D6B761, 0xB3B37DCE, 0x2929527B, 0xE3E3DD3E,
        0x2F2F5E71, 0x84841397, 0x5353A6F5, 0xD1D1B968, 0x0,        0xEDEDC12C,
        0x20204060, 0xFCFCE31F, 0xB1B179C8, 0x5B5BB6ED, 0x6A6AD4BE, 0xCBCB8D46,
        0xBEBE67D9, 0x3939724B, 0x4A4A94DE, 0x4C4C98D4, 0x5858B0E8, 0xCFCF854A,
        0xD0D0BB6B, 0xEFEFC52A, 0xAAAA4FE5, 0xFBFBED16, 0x434386C5, 0x4D4D9AD7,
        0x33336655, 0x85851194, 0x45458ACF, 0xF9F9E910, 0x2020406,  0x7F7FFE81,
        0x5050A0F0, 0x3C3C7844, 0x9F9F25BA, 0xA8A84BE3, 0x5151A2F3, 0xA3A35DFE,
        0x404080C0, 0x8F8F058A, 0x92923FAD, 0x9D9D21BC, 0x38387048, 0xF5F5F104,
        0xBCBC63DF, 0xB6B677C1, 0xDADAAF75, 0x21214263, 0x10102030, 0xFFFFE51A,
        0xF3F3FD0E, 0xD2D2BF6D, 0xCDCD814C, 0xC0C1814,  0x13132635, 0xECECC32F,
        0x5F5FBEE1, 0x979735A2, 0x444488CC, 0x17172E39, 0xC4C49357, 0xA7A755F2,
        0x7E7EFC82, 0x3D3D7A47, 0x6464C8AC, 0x5D5DBAE7, 0x1919322B, 0x7373E695,
        0x6060C0A0, 0x81811998, 0x4F4F9ED1, 0xDCDCA37F, 0x22224466, 0x2A2A547E,
        0x90903BAB, 0x88880B83, 0x46468CCA, 0xEEEEC729, 0xB8B86BD3, 0x1414283C,
        0xDEDEA779, 0x5E5EBCE2, 0xB0B161D,  0xDBDBAD76, 0xE0E0DB3B, 0x32326456,
        0x3A3A744E, 0xA0A141E,  0x494992DB, 0x6060C0A,  0x2424486C, 0x5C5CB8E4,
        0xC2C29F5D, 0xD3D3BD6E, 0xACAC43EF, 0x6262C4A6, 0x919139A8, 0x959531A4,
        0xE4E4D337, 0x7979F28B, 0xE7E7D532, 0xC8C88B43, 0x37376E59, 0x6D6DDAB7,
        0x8D8D018C, 0xD5D5B164, 0x4E4E9CD2, 0xA9A949E0, 0x6C6CD8B4, 0x5656ACFA,
        0xF4F4F307, 0xEAEACF25, 0x6565CAAF, 0x7A7AF48E, 0xAEAE47E9, 0x8081018,
        0xBABA6FD5, 0x7878F088, 0x25254A6F, 0x2E2E5C72, 0x1C1C3824, 0xA6A657F1,
        0xB4B473C7, 0xC6C69751, 0xE8E8CB23, 0xDDDDA17C, 0x7474E89C, 0x1F1F3E21,
        0x4B4B96DD, 0xBDBD61DC, 0x8B8B0D86, 0x8A8A0F85, 0x7070E090, 0x3E3E7C42,
        0xB5B571C4, 0x6666CCAA, 0x484890D8, 0x3030605,  0xF6F6F701, 0xE0E1C12,
        0x6161C2A3, 0x35356A5F, 0x5757AEF9, 0xB9B969D0, 0x86861791, 0xC1C19958,
        0x1D1D3A27, 0x9E9E27B9, 0xE1E1D938, 0xF8F8EB13, 0x98982BB3, 0x11112233,
        0x6969D2BB, 0xD9D9A970, 0x8E8E0789, 0x949433A7, 0x9B9B2DB6, 0x1E1E3C22,
        0x87871592, 0xE9E9C920, 0xCECE8749, 0x5555AAFF, 0x28285078, 0xDFDFA57A,
        0x8C8C038F, 0xA1A159F8, 0x89890980, 0xD0D1A17,  0xBFBF65DA, 0xE6E6D731,
        0x424284C6, 0x6868D0B8, 0x414182C3, 0x999929B0, 0x2D2D5A77, 0xF0F1E11,
        0xB0B07BCB, 0x5454A8FC, 0xBBBB6DD6, 0x16162C3A};

IMB_DLL_LOCAL
DECLARE_ALIGNED(const int S1_T2[256], 32) = {
        0x63C6A563, 0x7CF8847C, 0x77EE9977, 0x7BF68D7B, 0xF2FF0DF2, 0x6BD6BD6B,
        0x6FDEB16F, 0xC59154C5, 0x30605030, 0x1020301,  0x67CEA967, 0x2B567D2B,
        0xFEE719FE, 0xD7B562D7, 0xAB4DE6AB, 0x76EC9A76, 0xCA8F45CA, 0x821F9D82,
        0xC98940C9, 0x7DFA877D, 0xFAEF15FA, 0x59B2EB59, 0x478EC947, 0xF0FB0BF0,
        0xAD41ECAD, 0xD4B367D4, 0xA25FFDA2, 0xAF45EAAF, 0x9C23BF9C, 0xA453F7A4,
        0x72E49672, 0xC09B5BC0, 0xB775C2B7, 0xFDE11CFD, 0x933DAE93, 0x264C6A26,
        0x366C5A36, 0x3F7E413F, 0xF7F502F7, 0xCC834FCC, 0x34685C34, 0xA551F4A5,
        0xE5D134E5, 0xF1F908F1, 0x71E29371, 0xD8AB73D8, 0x31625331, 0x152A3F15,
        0x4080C04,  0xC79552C7, 0x23466523, 0xC39D5EC3, 0x18302818, 0x9637A196,
        0x50A0F05,  0x9A2FB59A, 0x70E0907,  0x12243612, 0x801B9B80, 0xE2DF3DE2,
        0xEBCD26EB, 0x274E6927, 0xB27FCDB2, 0x75EA9F75, 0x9121B09,  0x831D9E83,
        0x2C58742C, 0x1A342E1A, 0x1B362D1B, 0x6EDCB26E, 0x5AB4EE5A, 0xA05BFBA0,
        0x52A4F652, 0x3B764D3B, 0xD6B761D6, 0xB37DCEB3, 0x29527B29, 0xE3DD3EE3,
        0x2F5E712F, 0x84139784, 0x53A6F553, 0xD1B968D1, 0x0,        0xEDC12CED,
        0x20406020, 0xFCE31FFC, 0xB179C8B1, 0x5BB6ED5B, 0x6AD4BE6A, 0xCB8D46CB,
        0xBE67D9BE, 0x39724B39, 0x4A94DE4A, 0x4C98D44C, 0x58B0E858, 0xCF854ACF,
        0xD0BB6BD0, 0xEFC52AEF, 0xAA4FE5AA, 0xFBED16FB, 0x4386C543, 0x4D9AD74D,
        0x33665533, 0x85119485, 0x458ACF45, 0xF9E910F9, 0x2040602,  0x7FFE817F,
        0x50A0F050, 0x3C78443C, 0x9F25BA9F, 0xA84BE3A8, 0x51A2F351, 0xA35DFEA3,
        0x4080C040, 0x8F058A8F, 0x923FAD92, 0x9D21BC9D, 0x38704838, 0xF5F104F5,
        0xBC63DFBC, 0xB677C1B6, 0xDAAF75DA, 0x21426321, 0x10203010, 0xFFE51AFF,
        0xF3FD0EF3, 0xD2BF6DD2, 0xCD814CCD, 0xC18140C,  0x13263513, 0xECC32FEC,
        0x5FBEE15F, 0x9735A297, 0x4488CC44, 0x172E3917, 0xC49357C4, 0xA755F2A7,
        0x7EFC827E, 0x3D7A473D, 0x64C8AC64, 0x5DBAE75D, 0x19322B19, 0x73E69573,
        0x60C0A060, 0x81199881, 0x4F9ED14F, 0xDCA37FDC, 0x22446622, 0x2A547E2A,
        0x903BAB90, 0x880B8388, 0x468CCA46, 0xEEC729EE, 0xB86BD3B8, 0x14283C14,
        0xDEA779DE, 0x5EBCE25E, 0xB161D0B,  0xDBAD76DB, 0xE0DB3BE0, 0x32645632,
        0x3A744E3A, 0xA141E0A,  0x4992DB49, 0x60C0A06,  0x24486C24, 0x5CB8E45C,
        0xC29F5DC2, 0xD3BD6ED3, 0xAC43EFAC, 0x62C4A662, 0x9139A891, 0x9531A495,
        0xE4D337E4, 0x79F28B79, 0xE7D532E7, 0xC88B43C8, 0x376E5937, 0x6DDAB76D,
        0x8D018C8D, 0xD5B164D5, 0x4E9CD24E, 0xA949E0A9, 0x6CD8B46C, 0x56ACFA56,
        0xF4F307F4, 0xEACF25EA, 0x65CAAF65, 0x7AF48E7A, 0xAE47E9AE, 0x8101808,
        0xBA6FD5BA, 0x78F08878, 0x254A6F25, 0x2E5C722E, 0x1C38241C, 0xA657F1A6,
        0xB473C7B4, 0xC69751C6, 0xE8CB23E8, 0xDDA17CDD, 0x74E89C74, 0x1F3E211F,
        0x4B96DD4B, 0xBD61DCBD, 0x8B0D868B, 0x8A0F858A, 0x70E09070, 0x3E7C423E,
        0xB571C4B5, 0x66CCAA66, 0x4890D848, 0x3060503,  0xF6F701F6, 0xE1C120E,
        0x61C2A361, 0x356A5F35, 0x57AEF957, 0xB969D0B9, 0x86179186, 0xC19958C1,
        0x1D3A271D, 0x9E27B99E, 0xE1D938E1, 0xF8EB13F8, 0x982BB398, 0x11223311,
        0x69D2BB69, 0xD9A970D9, 0x8E07898E, 0x9433A794, 0x9B2DB69B, 0x1E3C221E,
        0x87159287, 0xE9C920E9, 0xCE8749CE, 0x55AAFF55, 0x28507828, 0xDFA57ADF,
        0x8C038F8C, 0xA159F8A1, 0x89098089, 0xD1A170D,  0xBF65DABF, 0xE6D731E6,
        0x4284C642, 0x68D0B868, 0x4182C341, 0x9929B099, 0x2D5A772D, 0xF1E110F,
        0xB07BCBB0, 0x54A8FC54, 0xBB6DD6BB, 0x162C3A16};

IMB_DLL_LOCAL
DECLARE_ALIGNED(const int S1_T3[256], 32) = {
        0xC6A56363, 0xF8847C7C, 0xEE997777, 0xF68D7B7B, 0xFF0DF2F2, 0xD6BD6B6B,
        0xDEB16F6F, 0x9154C5C5, 0x60503030, 0x2030101,  0xCEA96767, 0x567D2B2B,
        0xE719FEFE, 0xB562D7D7, 0x4DE6ABAB, 0xEC9A7676, 0x8F45CACA, 0x1F9D8282,
        0x8940C9C9, 0xFA877D7D, 0xEF15FAFA, 0xB2EB5959, 0x8EC94747, 0xFB0BF0F0,
        0x41ECADAD, 0xB367D4D4, 0x5FFDA2A2, 0x45EAAFAF, 0x23BF9C9C, 0x53F7A4A4,
        0xE4967272, 0x9B5BC0C0, 0x75C2B7B7, 0xE11CFDFD, 0x3DAE9393, 0x4C6A2626,
        0x6C5A3636, 0x7E413F3F, 0xF502F7F7, 0x834FCCCC, 0x685C3434, 0x51F4A5A5,
        0xD134E5E5, 0xF908F1F1, 0xE2937171, 0xAB73D8D8, 0x62533131, 0x2A3F1515,
        0x80C0404,  0x9552C7C7, 0x46652323, 0x9D5EC3C3, 0x30281818, 0x37A19696,
        0xA0F0505,  0x2FB59A9A, 0xE090707,  0x24361212, 0x1B9B8080, 0xDF3DE2E2,
        0xCD26EBEB, 0x4E692727, 0x7FCDB2B2, 0xEA9F7575, 0x121B0909, 0x1D9E8383,
        0x58742C2C, 0x342E1A1A, 0x362D1B1B, 0xDCB26E6E, 0xB4EE5A5A, 0x5BFBA0A0,
        0xA4F65252, 0x764D3B3B, 0xB761D6D6, 0x7DCEB3B3, 0x527B2929, 0xDD3EE3E3,
        0x5E712F2F, 0x13978484, 0xA6F55353, 0xB968D1D1, 0x0,        0xC12CEDED,
        0x40602020, 0xE31FFCFC, 0x79C8B1B1, 0xB6ED5B5B, 0xD4BE6A6A, 0x8D46CBCB,
        0x67D9BEBE, 0x724B3939, 0x94DE4A4A, 0x98D44C4C, 0xB0E85858, 0x854ACFCF,
        0xBB6BD0D0, 0xC52AEFEF, 0x4FE5AAAA, 0xED16FBFB, 0x86C54343, 0x9AD74D4D,
        0x66553333, 0x11948585, 0x8ACF4545, 0xE910F9F9, 0x4060202,  0xFE817F7F,
        0xA0F05050, 0x78443C3C, 0x25BA9F9F, 0x4BE3A8A8, 0xA2F35151, 0x5DFEA3A3,
        0x80C04040, 0x58A8F8F,  0x3FAD9292, 0x21BC9D9D, 0x70483838, 0xF104F5F5,
        0x63DFBCBC, 0x77C1B6B6, 0xAF75DADA, 0x42632121, 0x20301010, 0xE51AFFFF,
        0xFD0EF3F3, 0xBF6DD2D2, 0x814CCDCD, 0x18140C0C, 0x26351313, 0xC32FECEC,
        0xBEE15F5F, 0x35A29797, 0x88CC4444, 0x2E391717, 0x9357C4C4, 0x55F2A7A7,
        0xFC827E7E, 0x7A473D3D, 0xC8AC6464, 0xBAE75D5D, 0x322B1919, 0xE6957373,
        0xC0A06060, 0x19988181, 0x9ED14F4F, 0xA37FDCDC, 0x44662222, 0x547E2A2A,
        0x3BAB9090, 0xB838888,  0x8CCA4646, 0xC729EEEE, 0x6BD3B8B8, 0x283C1414,
        0xA779DEDE, 0xBCE25E5E, 0x161D0B0B, 0xAD76DBDB, 0xDB3BE0E0, 0x64563232,
        0x744E3A3A, 0x141E0A0A, 0x92DB4949, 0xC0A0606,  0x486C2424, 0xB8E45C5C,
        0x9F5DC2C2, 0xBD6ED3D3, 0x43EFACAC, 0xC4A66262, 0x39A89191, 0x31A49595,
        0xD337E4E4, 0xF28B7979, 0xD532E7E7, 0x8B43C8C8, 0x6E593737, 0xDAB76D6D,
        0x18C8D8D,  0xB164D5D5, 0x9CD24E4E, 0x49E0A9A9, 0xD8B46C6C, 0xACFA5656,
        0xF307F4F4, 0xCF25EAEA, 0xCAAF6565, 0xF48E7A7A, 0x47E9AEAE, 0x10180808,
        0x6FD5BABA, 0xF0887878, 0x4A6F2525, 0x5C722E2E, 0x38241C1C, 0x57F1A6A6,
        0x73C7B4B4, 0x9751C6C6, 0xCB23E8E8, 0xA17CDDDD, 0xE89C7474, 0x3E211F1F,
        0x96DD4B4B, 0x61DCBDBD, 0xD868B8B,  0xF858A8A,  0xE0907070, 0x7C423E3E,
        0x71C4B5B5, 0xCCAA6666, 0x90D84848, 0x6050303,  0xF701F6F6, 0x1C120E0E,
        0xC2A36161, 0x6A5F3535, 0xAEF95757, 0x69D0B9B9, 0x17918686, 0x9958C1C1,
        0x3A271D1D, 0x27B99E9E, 0xD938E1E1, 0xEB13F8F8, 0x2BB39898, 0x22331111,
        0xD2BB6969, 0xA970D9D9, 0x7898E8E,  0x33A79494, 0x2DB69B9B, 0x3C221E1E,
        0x15928787, 0xC920E9E9, 0x8749CECE, 0xAAFF5555, 0x50782828, 0xA57ADFDF,
        0x38F8C8C,  0x59F8A1A1, 0x9808989,  0x1A170D0D, 0x65DABFBF, 0xD731E6E6,
        0x84C64242, 0xD0B86868, 0x82C34141, 0x29B09999, 0x5A772D2D, 0x1E110F0F,
        0x7BCBB0B0, 0xA8FC5454, 0x6DD6BBBB, 0x2C3A1616};

IMB_DLL_LOCAL
DECLARE_ALIGNED(const int S2_T0[256], 32) = {
        0x6f25254a, 0x6c242448, 0x957373e6, 0xa96767ce, 0x10d7d7c7, 0x9baeae35,
        0xe45c5cb8, 0x50303060, 0x85a4a421, 0x5beeeeb5, 0xb26e6edc, 0x34cbcbff,
        0x877d7dfa, 0xb6b5b503, 0xef82826d, 0x04dbdbdf, 0x45e4e4a1, 0xfb8e8e75,
        0xd8484890, 0xdb494992, 0xd14f4f9e, 0xe75d5dba, 0xbe6a6ad4, 0x887878f0,
        0x907070e0, 0xf1888879, 0x51e8e8b9, 0xe15f5fbe, 0xe25e5ebc, 0xe5848461,
        0xaf6565ca, 0x4fe2e2ad, 0x01d8d8d9, 0x52e9e9bb, 0x3dccccf1, 0x5eededb3,
        0xc0404080, 0x712f2f5e, 0x33111122, 0x78282850, 0xf95757ae, 0x1fd2d2cd,
        0x9dacac31, 0x4ce3e3af, 0xde4a4a94, 0x3f15152a, 0x2d1b1b36, 0xa2b9b91b,
        0xbfb2b20d, 0xe9808069, 0xe6858563, 0x83a6a625, 0x722e2e5c, 0x06020204,
        0xc947478e, 0x7b292952, 0x0907070e, 0xdd4b4b96, 0x120e0e1c, 0x2ac1c1eb,
        0xf35151a2, 0x97aaaa3d, 0xf289897b, 0x15d4d4c1, 0x37cacafd, 0x03010102,
        0xca46468c, 0xbcb3b30f, 0x58efefb7, 0x0eddddd3, 0xcc444488, 0x8d7b7bf6,
        0x2fc2c2ed, 0x817f7ffe, 0xabbebe15, 0x2cc3c3ef, 0xc89f9f57, 0x60202040,
        0xd44c4c98, 0xac6464c8, 0xec83836f, 0x8fa2a22d, 0xb86868d0, 0xc6424284,
        0x35131326, 0xb5b4b401, 0xc3414182, 0x3ecdcdf3, 0xa7baba1d, 0x23c6c6e5,
        0xa4bbbb1f, 0xb76d6dda, 0xd74d4d9a, 0x937171e2, 0x63212142, 0x75f4f481,
        0xfe8d8d73, 0xb9b0b009, 0x46e5e5a3, 0xdc93934f, 0x6bfefe95, 0xf88f8f77,
        0x43e6e6a5, 0x38cfcff7, 0xc5434386, 0xcf45458a, 0x53313162, 0x66222244,
        0x5937376e, 0x5a36366c, 0xd3969645, 0x67fafa9d, 0xadbcbc11, 0x110f0f1e,
        0x18080810, 0xf65252a4, 0x271d1d3a, 0xff5555aa, 0x2e1a1a34, 0x26c5c5e3,
        0xd24e4e9c, 0x65232346, 0xbb6969d2, 0x8e7a7af4, 0xdf92924d, 0x68ffff97,
        0xed5b5bb6, 0xee5a5ab4, 0x54ebebbf, 0xc79a9a5d, 0x241c1c38, 0x92a9a93b,
        0x1ad1d1cb, 0x827e7efc, 0x170d0d1a, 0x6dfcfc91, 0xf05050a0, 0xf78a8a7d,
        0xb3b6b605, 0xa66262c4, 0x76f5f583, 0x1e0a0a14, 0x61f8f899, 0x0ddcdcd1,
        0x05030306, 0x443c3c78, 0x140c0c18, 0x4b393972, 0x7af1f18b, 0xa1b8b819,
        0x7cf3f38f, 0x473d3d7a, 0x7ff2f28d, 0x16d5d5c3, 0xd0979747, 0xaa6666cc,
        0xea81816b, 0x56323264, 0x89a0a029, 0x00000000, 0x0a06060c, 0x3bcecef5,
        0x73f6f685, 0x57eaeabd, 0xb0b7b707, 0x3917172e, 0x70f7f787, 0xfd8c8c71,
        0x8b7979f2, 0x13d6d6c5, 0x80a7a727, 0xa8bfbf17, 0xf48b8b7f, 0x413f3f7e,
        0x211f1f3e, 0xf55353a6, 0xa56363c6, 0x9f7575ea, 0x5f35356a, 0x742c2c58,
        0xa06060c0, 0x6efdfd93, 0x6927274e, 0x1cd3d3cf, 0xd5949441, 0x86a5a523,
        0x847c7cf8, 0x8aa1a12b, 0x0f05050a, 0xe85858b0, 0x772d2d5a, 0xaebdbd13,
        0x02d9d9db, 0x20c7c7e7, 0x98afaf37, 0xbd6b6bd6, 0xfc5454a8, 0x1d0b0b16,
        0x49e0e0a9, 0x48383870, 0x0c040408, 0x31c8c8f9, 0xce9d9d53, 0x40e7e7a7,
        0x3c141428, 0xbab1b10b, 0xe0878767, 0xcd9c9c51, 0x08dfdfd7, 0xb16f6fde,
        0x62f9f99b, 0x07dadadd, 0x7e2a2a54, 0x25c4c4e1, 0xeb5959b2, 0x3a16162c,
        0x9c7474e8, 0xda91914b, 0x94abab3f, 0x6a26264c, 0xa36161c2, 0x9a7676ec,
        0x5c343468, 0x7d2b2b56, 0x9eadad33, 0xc299995b, 0x64fbfb9f, 0x967272e4,
        0x5dececb1, 0x55333366, 0x36121224, 0x0bdeded5, 0xc1989859, 0x4d3b3b76,
        0x29c0c0e9, 0xc49b9b5f, 0x423e3e7c, 0x28181830, 0x30101020, 0x4e3a3a74,
        0xfa5656ac, 0x4ae1e1ab, 0x997777ee, 0x32c9c9fb, 0x221e1e3c, 0xcb9e9e55,
        0xd6959543, 0x8ca3a32f, 0xd9909049, 0x2b191932, 0x91a8a839, 0xb46c6cd8,
        0x1b090912, 0x19d0d0c9, 0x79f0f089, 0xe3868665};

IMB_DLL_LOCAL
DECLARE_ALIGNED(const int S2_T1[256], 32) = {
        0x25254a6f, 0x2424486c, 0x7373e695, 0x6767cea9, 0xd7d7c710, 0xaeae359b,
        0x5c5cb8e4, 0x30306050, 0xa4a42185, 0xeeeeb55b, 0x6e6edcb2, 0xcbcbff34,
        0x7d7dfa87, 0xb5b503b6, 0x82826def, 0xdbdbdf04, 0xe4e4a145, 0x8e8e75fb,
        0x484890d8, 0x494992db, 0x4f4f9ed1, 0x5d5dbae7, 0x6a6ad4be, 0x7878f088,
        0x7070e090, 0x888879f1, 0xe8e8b951, 0x5f5fbee1, 0x5e5ebce2, 0x848461e5,
        0x6565caaf, 0xe2e2ad4f, 0xd8d8d901, 0xe9e9bb52, 0xccccf13d, 0xededb35e,
        0x404080c0, 0x2f2f5e71, 0x11112233, 0x28285078, 0x5757aef9, 0xd2d2cd1f,
        0xacac319d, 0xe3e3af4c, 0x4a4a94de, 0x15152a3f, 0x1b1b362d, 0xb9b91ba2,
        0xb2b20dbf, 0x808069e9, 0x858563e6, 0xa6a62583, 0x2e2e5c72, 0x02020406,
        0x47478ec9, 0x2929527b, 0x07070e09, 0x4b4b96dd, 0x0e0e1c12, 0xc1c1eb2a,
        0x5151a2f3, 0xaaaa3d97, 0x89897bf2, 0xd4d4c115, 0xcacafd37, 0x01010203,
        0x46468cca, 0xb3b30fbc, 0xefefb758, 0xddddd30e, 0x444488cc, 0x7b7bf68d,
        0xc2c2ed2f, 0x7f7ffe81, 0xbebe15ab, 0xc3c3ef2c, 0x9f9f57c8, 0x20204060,
        0x4c4c98d4, 0x6464c8ac, 0x83836fec, 0xa2a22d8f, 0x6868d0b8, 0x424284c6,
        0x13132635, 0xb4b401b5, 0x414182c3, 0xcdcdf33e, 0xbaba1da7, 0xc6c6e523,
        0xbbbb1fa4, 0x6d6ddab7, 0x4d4d9ad7, 0x7171e293, 0x21214263, 0xf4f48175,
        0x8d8d73fe, 0xb0b009b9, 0xe5e5a346, 0x93934fdc, 0xfefe956b, 0x8f8f77f8,
        0xe6e6a543, 0xcfcff738, 0x434386c5, 0x45458acf, 0x31316253, 0x22224466,
        0x37376e59, 0x36366c5a, 0x969645d3, 0xfafa9d67, 0xbcbc11ad, 0x0f0f1e11,
        0x08081018, 0x5252a4f6, 0x1d1d3a27, 0x5555aaff, 0x1a1a342e, 0xc5c5e326,
        0x4e4e9cd2, 0x23234665, 0x6969d2bb, 0x7a7af48e, 0x92924ddf, 0xffff9768,
        0x5b5bb6ed, 0x5a5ab4ee, 0xebebbf54, 0x9a9a5dc7, 0x1c1c3824, 0xa9a93b92,
        0xd1d1cb1a, 0x7e7efc82, 0x0d0d1a17, 0xfcfc916d, 0x5050a0f0, 0x8a8a7df7,
        0xb6b605b3, 0x6262c4a6, 0xf5f58376, 0x0a0a141e, 0xf8f89961, 0xdcdcd10d,
        0x03030605, 0x3c3c7844, 0x0c0c1814, 0x3939724b, 0xf1f18b7a, 0xb8b819a1,
        0xf3f38f7c, 0x3d3d7a47, 0xf2f28d7f, 0xd5d5c316, 0x979747d0, 0x6666ccaa,
        0x81816bea, 0x32326456, 0xa0a02989, 0x00000000, 0x06060c0a, 0xcecef53b,
        0xf6f68573, 0xeaeabd57, 0xb7b707b0, 0x17172e39, 0xf7f78770, 0x8c8c71fd,
        0x7979f28b, 0xd6d6c513, 0xa7a72780, 0xbfbf17a8, 0x8b8b7ff4, 0x3f3f7e41,
        0x1f1f3e21, 0x5353a6f5, 0x6363c6a5, 0x7575ea9f, 0x35356a5f, 0x2c2c5874,
        0x6060c0a0, 0xfdfd936e, 0x27274e69, 0xd3d3cf1c, 0x949441d5, 0xa5a52386,
        0x7c7cf884, 0xa1a12b8a, 0x05050a0f, 0x5858b0e8, 0x2d2d5a77, 0xbdbd13ae,
        0xd9d9db02, 0xc7c7e720, 0xafaf3798, 0x6b6bd6bd, 0x5454a8fc, 0x0b0b161d,
        0xe0e0a949, 0x38387048, 0x0404080c, 0xc8c8f931, 0x9d9d53ce, 0xe7e7a740,
        0x1414283c, 0xb1b10bba, 0x878767e0, 0x9c9c51cd, 0xdfdfd708, 0x6f6fdeb1,
        0xf9f99b62, 0xdadadd07, 0x2a2a547e, 0xc4c4e125, 0x5959b2eb, 0x16162c3a,
        0x7474e89c, 0x91914bda, 0xabab3f94, 0x26264c6a, 0x6161c2a3, 0x7676ec9a,
        0x3434685c, 0x2b2b567d, 0xadad339e, 0x99995bc2, 0xfbfb9f64, 0x7272e496,
        0xececb15d, 0x33336655, 0x12122436, 0xdeded50b, 0x989859c1, 0x3b3b764d,
        0xc0c0e929, 0x9b9b5fc4, 0x3e3e7c42, 0x18183028, 0x10102030, 0x3a3a744e,
        0x5656acfa, 0xe1e1ab4a, 0x7777ee99, 0xc9c9fb32, 0x1e1e3c22, 0x9e9e55cb,
        0x959543d6, 0xa3a32f8c, 0x909049d9, 0x1919322b, 0xa8a83991, 0x6c6cd8b4,
        0x0909121b, 0xd0d0c919, 0xf0f08979, 0x868665e3};

IMB_DLL_LOCAL
DECLARE_ALIGNED(const int S2_T2[256], 32) = {
        0x254a6f25, 0x24486c24, 0x73e69573, 0x67cea967, 0xd7c710d7, 0xae359bae,
        0x5cb8e45c, 0x30605030, 0xa42185a4, 0xeeb55bee, 0x6edcb26e, 0xcbff34cb,
        0x7dfa877d, 0xb503b6b5, 0x826def82, 0xdbdf04db, 0xe4a145e4, 0x8e75fb8e,
        0x4890d848, 0x4992db49, 0x4f9ed14f, 0x5dbae75d, 0x6ad4be6a, 0x78f08878,
        0x70e09070, 0x8879f188, 0xe8b951e8, 0x5fbee15f, 0x5ebce25e, 0x8461e584,
        0x65caaf65, 0xe2ad4fe2, 0xd8d901d8, 0xe9bb52e9, 0xccf13dcc, 0xedb35eed,
        0x4080c040, 0x2f5e712f, 0x11223311, 0x28507828, 0x57aef957, 0xd2cd1fd2,
        0xac319dac, 0xe3af4ce3, 0x4a94de4a, 0x152a3f15, 0x1b362d1b, 0xb91ba2b9,
        0xb20dbfb2, 0x8069e980, 0x8563e685, 0xa62583a6, 0x2e5c722e, 0x02040602,
        0x478ec947, 0x29527b29, 0x070e0907, 0x4b96dd4b, 0x0e1c120e, 0xc1eb2ac1,
        0x51a2f351, 0xaa3d97aa, 0x897bf289, 0xd4c115d4, 0xcafd37ca, 0x01020301,
        0x468cca46, 0xb30fbcb3, 0xefb758ef, 0xddd30edd, 0x4488cc44, 0x7bf68d7b,
        0xc2ed2fc2, 0x7ffe817f, 0xbe15abbe, 0xc3ef2cc3, 0x9f57c89f, 0x20406020,
        0x4c98d44c, 0x64c8ac64, 0x836fec83, 0xa22d8fa2, 0x68d0b868, 0x4284c642,
        0x13263513, 0xb401b5b4, 0x4182c341, 0xcdf33ecd, 0xba1da7ba, 0xc6e523c6,
        0xbb1fa4bb, 0x6ddab76d, 0x4d9ad74d, 0x71e29371, 0x21426321, 0xf48175f4,
        0x8d73fe8d, 0xb009b9b0, 0xe5a346e5, 0x934fdc93, 0xfe956bfe, 0x8f77f88f,
        0xe6a543e6, 0xcff738cf, 0x4386c543, 0x458acf45, 0x31625331, 0x22446622,
        0x376e5937, 0x366c5a36, 0x9645d396, 0xfa9d67fa, 0xbc11adbc, 0x0f1e110f,
        0x08101808, 0x52a4f652, 0x1d3a271d, 0x55aaff55, 0x1a342e1a, 0xc5e326c5,
        0x4e9cd24e, 0x23466523, 0x69d2bb69, 0x7af48e7a, 0x924ddf92, 0xff9768ff,
        0x5bb6ed5b, 0x5ab4ee5a, 0xebbf54eb, 0x9a5dc79a, 0x1c38241c, 0xa93b92a9,
        0xd1cb1ad1, 0x7efc827e, 0x0d1a170d, 0xfc916dfc, 0x50a0f050, 0x8a7df78a,
        0xb605b3b6, 0x62c4a662, 0xf58376f5, 0x0a141e0a, 0xf89961f8, 0xdcd10ddc,
        0x03060503, 0x3c78443c, 0x0c18140c, 0x39724b39, 0xf18b7af1, 0xb819a1b8,
        0xf38f7cf3, 0x3d7a473d, 0xf28d7ff2, 0xd5c316d5, 0x9747d097, 0x66ccaa66,
        0x816bea81, 0x32645632, 0xa02989a0, 0x00000000, 0x060c0a06, 0xcef53bce,
        0xf68573f6, 0xeabd57ea, 0xb707b0b7, 0x172e3917, 0xf78770f7, 0x8c71fd8c,
        0x79f28b79, 0xd6c513d6, 0xa72780a7, 0xbf17a8bf, 0x8b7ff48b, 0x3f7e413f,
        0x1f3e211f, 0x53a6f553, 0x63c6a563, 0x75ea9f75, 0x356a5f35, 0x2c58742c,
        0x60c0a060, 0xfd936efd, 0x274e6927, 0xd3cf1cd3, 0x9441d594, 0xa52386a5,
        0x7cf8847c, 0xa12b8aa1, 0x050a0f05, 0x58b0e858, 0x2d5a772d, 0xbd13aebd,
        0xd9db02d9, 0xc7e720c7, 0xaf3798af, 0x6bd6bd6b, 0x54a8fc54, 0x0b161d0b,
        0xe0a949e0, 0x38704838, 0x04080c04, 0xc8f931c8, 0x9d53ce9d, 0xe7a740e7,
        0x14283c14, 0xb10bbab1, 0x8767e087, 0x9c51cd9c, 0xdfd708df, 0x6fdeb16f,
        0xf99b62f9, 0xdadd07da, 0x2a547e2a, 0xc4e125c4, 0x59b2eb59, 0x162c3a16,
        0x74e89c74, 0x914bda91, 0xab3f94ab, 0x264c6a26, 0x61c2a361, 0x76ec9a76,
        0x34685c34, 0x2b567d2b, 0xad339ead, 0x995bc299, 0xfb9f64fb, 0x72e49672,
        0xecb15dec, 0x33665533, 0x12243612, 0xded50bde, 0x9859c198, 0x3b764d3b,
        0xc0e929c0, 0x9b5fc49b, 0x3e7c423e, 0x18302818, 0x10203010, 0x3a744e3a,
        0x56acfa56, 0xe1ab4ae1, 0x77ee9977, 0xc9fb32c9, 0x1e3c221e, 0x9e55cb9e,
        0x9543d695, 0xa32f8ca3, 0x9049d990, 0x19322b19, 0xa83991a8, 0x6cd8b46c,
        0x09121b09, 0xd0c919d0, 0xf08979f0, 0x8665e386};

IMB_DLL_LOCAL
DECLARE_ALIGNED(const int S2_T3[256], 32) = {
        0x4a6f2525, 0x486c2424, 0xe6957373, 0xcea96767, 0xc710d7d7, 0x359baeae,
        0xb8e45c5c, 0x60503030, 0x2185a4a4, 0xb55beeee, 0xdcb26e6e, 0xff34cbcb,
        0xfa877d7d, 0x03b6b5b5, 0x6def8282, 0xdf04dbdb, 0xa145e4e4, 0x75fb8e8e,
        0x90d84848, 0x92db4949, 0x9ed14f4f, 0xbae75d5d, 0xd4be6a6a, 0xf0887878,
        0xe0907070, 0x79f18888, 0xb951e8e8, 0xbee15f5f, 0xbce25e5e, 0x61e58484,
        0xcaaf6565, 0xad4fe2e2, 0xd901d8d8, 0xbb52e9e9, 0xf13dcccc, 0xb35eeded,
        0x80c04040, 0x5e712f2f, 0x22331111, 0x50782828, 0xaef95757, 0xcd1fd2d2,
        0x319dacac, 0xaf4ce3e3, 0x94de4a4a, 0x2a3f1515, 0x362d1b1b, 0x1ba2b9b9,
        0x0dbfb2b2, 0x69e98080, 0x63e68585, 0x2583a6a6, 0x5c722e2e, 0x04060202,
        0x8ec94747, 0x527b2929, 0x0e090707, 0x96dd4b4b, 0x1c120e0e, 0xeb2ac1c1,
        0xa2f35151, 0x3d97aaaa, 0x7bf28989, 0xc115d4d4, 0xfd37caca, 0x02030101,
        0x8cca4646, 0x0fbcb3b3, 0xb758efef, 0xd30edddd, 0x88cc4444, 0xf68d7b7b,
        0xed2fc2c2, 0xfe817f7f, 0x15abbebe, 0xef2cc3c3, 0x57c89f9f, 0x40602020,
        0x98d44c4c, 0xc8ac6464, 0x6fec8383, 0x2d8fa2a2, 0xd0b86868, 0x84c64242,
        0x26351313, 0x01b5b4b4, 0x82c34141, 0xf33ecdcd, 0x1da7baba, 0xe523c6c6,
        0x1fa4bbbb, 0xdab76d6d, 0x9ad74d4d, 0xe2937171, 0x42632121, 0x8175f4f4,
        0x73fe8d8d, 0x09b9b0b0, 0xa346e5e5, 0x4fdc9393, 0x956bfefe, 0x77f88f8f,
        0xa543e6e6, 0xf738cfcf, 0x86c54343, 0x8acf4545, 0x62533131, 0x44662222,
        0x6e593737, 0x6c5a3636, 0x45d39696, 0x9d67fafa, 0x11adbcbc, 0x1e110f0f,
        0x10180808, 0xa4f65252, 0x3a271d1d, 0xaaff5555, 0x342e1a1a, 0xe326c5c5,
        0x9cd24e4e, 0x46652323, 0xd2bb6969, 0xf48e7a7a, 0x4ddf9292, 0x9768ffff,
        0xb6ed5b5b, 0xb4ee5a5a, 0xbf54ebeb, 0x5dc79a9a, 0x38241c1c, 0x3b92a9a9,
        0xcb1ad1d1, 0xfc827e7e, 0x1a170d0d, 0x916dfcfc, 0xa0f05050, 0x7df78a8a,
        0x05b3b6b6, 0xc4a66262, 0x8376f5f5, 0x141e0a0a, 0x9961f8f8, 0xd10ddcdc,
        0x06050303, 0x78443c3c, 0x18140c0c, 0x724b3939, 0x8b7af1f1, 0x19a1b8b8,
        0x8f7cf3f3, 0x7a473d3d, 0x8d7ff2f2, 0xc316d5d5, 0x47d09797, 0xccaa6666,
        0x6bea8181, 0x64563232, 0x2989a0a0, 0x00000000, 0x0c0a0606, 0xf53bcece,
        0x8573f6f6, 0xbd57eaea, 0x07b0b7b7, 0x2e391717, 0x8770f7f7, 0x71fd8c8c,
        0xf28b7979, 0xc513d6d6, 0x2780a7a7, 0x17a8bfbf, 0x7ff48b8b, 0x7e413f3f,
        0x3e211f1f, 0xa6f55353, 0xc6a56363, 0xea9f7575, 0x6a5f3535, 0x58742c2c,
        0xc0a06060, 0x936efdfd, 0x4e692727, 0xcf1cd3d3, 0x41d59494, 0x2386a5a5,
        0xf8847c7c, 0x2b8aa1a1, 0x0a0f0505, 0xb0e85858, 0x5a772d2d, 0x13aebdbd,
        0xdb02d9d9, 0xe720c7c7, 0x3798afaf, 0xd6bd6b6b, 0xa8fc5454, 0x161d0b0b,
        0xa949e0e0, 0x70483838, 0x080c0404, 0xf931c8c8, 0x53ce9d9d, 0xa740e7e7,
        0x283c1414, 0x0bbab1b1, 0x67e08787, 0x51cd9c9c, 0xd708dfdf, 0xdeb16f6f,
        0x9b62f9f9, 0xdd07dada, 0x547e2a2a, 0xe125c4c4, 0xb2eb5959, 0x2c3a1616,
        0xe89c7474, 0x4bda9191, 0x3f94abab, 0x4c6a2626, 0xc2a36161, 0xec9a7676,
        0x685c3434, 0x567d2b2b, 0x339eadad, 0x5bc29999, 0x9f64fbfb, 0xe4967272,
        0xb15decec, 0x66553333, 0x24361212, 0xd50bdede, 0x59c19898, 0x764d3b3b,
        0xe929c0c0, 0x5fc49b9b, 0x7c423e3e, 0x30281818, 0x20301010, 0x744e3a3a,
        0xacfa5656, 0xab4ae1e1, 0xee997777, 0xfb32c9c9, 0x3c221e1e, 0x55cb9e9e,
        0x43d69595, 0x2f8ca3a3, 0x49d99090, 0x322b1919, 0x3991a8a8, 0xd8b46c6c,
        0x121b0909, 0xc919d0d0, 0x8979f0f0, 0x65e38686};
