;;;; easyffi.scm
;
; Copyright (c) 2000-2003, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany

#{compiler debugging-chicken register-ffi-macro parse-easy-ffi ffi-include-path foreign-type-declaration
	   foreign-declarations}

(declare
  (unit easyffi)
  (compress-literals)
  ;(no-bound-checks) (no-procedure-checks)
  (export parse-easy-ffi register-ffi-macro) )

(eval-when (compile) (match-error-control #:unspecified))

(include "easyffi.l.silex")

(define pp-mode #f)
(define processed-output '())
(define macro-table '((|CHICKEN| * ())))
(define return-type #f)
(define pp-conditional-stack '())
(define pp-process #t)
(define type-map '())
(define ffi-include-path '("."))
(define export-constants #f)
(define prefix #f)
(define name-substitution-rxs '())
(define name-substitution-repls '())
(define declared-types '())
(define rename-list '())

(define (pp-token tok)
  (if pp-mode
      tok
      (quit "FFI lexer error in line ~A: preprocessor-command out of context: ~S" ##sys#read-line-counter tok) ) )

(define (pp-token2 tok tok2)
  (if pp-mode tok tok2) )

(define (lexer-error)
  (quit "FFI lexer error in line ~A: illegal character" ##sys#read-line-counter) )

(define (chunkify)
  (let rec ([scope 0])
    (let ([chunks '()])
      (let loop ([mode #f] [tokens '()])
	(let ([t (lexer)])
	  (case t
	    [(stop) (reverse chunks)]
	    [(pp-end)
	     (set! chunks (cons (reverse tokens) chunks))
	     (loop #f '()) ]
	    [(pp-define pp-include pp-if pp-ifdef pp-ifndef pp-else pp-endif pp-undef pp-pragma pp-error)
	     (loop 'pp (list t)) ]
	    [(close-curly)
	     (cond [(not (positive? scope)) (quit "`}' out of context")]
		   [(null? tokens) (reverse chunks)]
		   [else (cons (reverse tokens) chunks)] ) ]
	    [(open-curly)
	     (let ([new (rec (add1 scope))])
	       (set! chunks (cons (append-reverse tokens `((scope . ,new))) chunks))
	       (loop #f '()) ) ]
	    [(close-paren)
	     (if (eq? mode 'declare)
		 (begin
		   (set! chunks (cons (reverse (cons 'close-paren tokens)) chunks))
		   (loop #f '()) )
		 (loop mode (cons t tokens)) ) ]
	    [(declare)
	     (loop 'declare '(declare)) ]
	    [(semicolon)
	     (if mode
		 (quit "unexpected semicolon")
		 (begin
		   (set! chunks (cons (reverse tokens) chunks))
		   (loop #f '()) ) ) ]
	    [else (loop mode (cons t tokens))] ) ) ) ) ) )

(define (parse c)
  (match c
    [() #f]
    [('pp-else)
     (when (null? pp-conditional-stack)
       (quit "unbalanced preprocessor conditionals") )
     (set! pp-process (and (not (car pp-conditional-stack)) (every identity (cdr pp-conditional-stack)))) ]
    [('pp-endif)
     (when (null? pp-conditional-stack)
       (quit "unbalanced preprocessor conditionals") )
     (set! pp-conditional-stack (cdr pp-conditional-stack))
     (set! pp-process (every identity pp-conditional-stack)) ]
    [('pp-ifdef ('id name))
     (set! pp-process (and pp-process (assq (string->symbol name) macro-table)))
     (set! pp-conditional-stack (cons pp-process pp-conditional-stack)) ]
    [('pp-ifndef ('id name))
     (set! pp-process (and pp-process (not (assq (string->symbol name) macro-table))))
     (set! pp-conditional-stack (cons pp-process pp-conditional-stack)) ]
    [('pp-if . _)
     (warning "preprocessor conditional `~A' ignored (assuming false)" c)
     (set! pp-process #f)
     (set! pp-conditional-stack (cons #f pp-conditional-stack)) ]
    [_ (when pp-process
	 (match c
	   [('pp-define ('id name))
	    (let ([s (string->symbol name)])
	      (set! macro-table (cons (list s '* '()) macro-table)) ) ]
	   [('pp-define ('id name) ('num n))
	    (let ([s (string->symbol name)])
	      (set! macro-table (cons (list s (if (exact? n) 'int 'double) `((num ,n))) macro-table))
	      (process-constant-def s n) ) ]
	   [('pp-define ('id name) . more)
	    (let ([t (compute-macro-type more)]
		  [s (string->symbol name)] )
	      (set! macro-table (cons (list s t more) macro-table))
	      (process-macro-def s t) ) ]
	   [('pp-undef ('id name))
	    (set! macro-table (delete (assq (string->symbol name) macro-table) eq?)) ]
	   [('pp-error msgs ...)
	    (quit (string-intersperse (cons "(#error) " (map reparse-item msgs)) "")) ]
	   [('pp-include ((or 'string 'i-string) filename))
	    (let ([fname (resolve-ffi-include-file filename)])
	      (if fname
		  (call-with-input-file fname parse-easy-ffi-rec)
		  (quit "can not open include file `~A'" filename) ) ) ]
	   [('pp-pragma . more) #f]
	   [('declare 'open-paren ('id decl) 'comma val 'close-paren)
	    (parse-declaration decl val) ]
	   [('declare . _)
	    (quit "invalid syntax in pseudo declaration: ~S" c) ]
	   [_ (let loop ([c (subst-macros c)] [cb #f])
		(match c
		  [((or 'extern 'static 'volatile 'inline) . more)
		   (loop more cb) ]
		  [('callback . more)
		   (loop more #t) ]
		  [('const . more)
		   (parse-constant more) ]
		  [('enum ('scope more))
		   (parse-enum-def #f (subst-macros more)) ]
		  [('enum ('id name) ('scope more))
		   (parse-enum-def name (subst-macros more)) ]
		  [('class . more)
		   (parse-class-def more) ]
		  [('struct ('id name)) #f]
		  [('using . more) #f]
		  [('typedef . more)
		   (parse-typedef more) ]
		  [(and more (('id name) . _))
		   (parse-prototype more cb) ]
		  [more
		   (parse-prototype more cb)] ) ) ] ) ) ] ) )

(define parse-again parse)

(define parse-type-rec
  (match-lambda
    [('const . more) 
     (let-values ([(t0 more) (parse-type-rec more)])
       (values `(const ,t0) more) ) ]
    [('unsigned t 'star . more)
     (let-values ([(t0 more) (parse-type-rec (cons* 'unsigned t more))])
       (values `(pointer ,t0) more) ) ]
    [('signed t 'star . more)
     (let-values ([(t0 more) (parse-type-rec (cons* 'signed t more))])
       (values `(pointer ,t0) more) ) ]
    [(t ('op "<") . more)
     (let*-values ([(ts more) (parse-typelist more)]
		   [(t0 _) (parse-type-rec (list t))] )
       (values `(template ,t0 ,@ts) more) ) ]
    [('signed . more) (parse-type-rec more)]
    [`(unsigned fixnum . ,more) (values 'unsigned-int more)]
    [`(unsigned int . ,more) (values 'unsigned-integer more)]
    [`(unsigned char . ,more) (values 'unsigned-char more)]
    [`(unsigned short . ,more) (values 'unsigned-short more)]
    [`(unsigned long . ,more) (values 'unsigned-long more)]
    [`(void . ,more) (values 'void more)]
    [`(bool . ,more) (values 'bool more)]
    [`(fixnum . ,more) (values 'int more)]
    [`(int . ,more) (values 'integer more)]
    [`(char . ,more) (values 'char more)]
    [`(short . ,more) (values 'short more)]
    [`(long . ,more) (values 'long more)]
    [`(float . ,more) (values 'float more)]
    [`(double . ,more) (values 'double more)]
    [('struct ('id sname) . more) (values `(struct ,sname) more)]
    [('union ('id sname) . more) (values `(union ,sname) more)]
    [('enum ('id sname) . more) (values `(enum ,sname) more)]
    [(('id t) . more)
     (let ([st (string->symbol t)])
       (cond [(assq st type-map) => (lambda (a) (values (cdr a) more))]
	     [(memq st declared-types) (values st more)]
	     [else (values t more)] ) ) ]
    [x (quit "invalid type specifier: ~S" x)] ) )

(define (parse-type ts . rt)
  (fluid-let ([return-type (:optional rt #f)])
    (let-values ([(t0 more) (parse-type-rec ts)])
      (let loop ([t0 t0] [more more])
	(match more
	  [('star . more)
	   (loop `(pointer ,t0) more) ]
	  [(('op "&") . more)
	   (loop `(ref ,t0) more) ]
	  [_ (values
	      (if return-type
		  (match t0
		    ['(pointer char) 'c-string]
		    ['(pointer (const char)) 'c-string]
		    [_ t0] )
		  (let loop ([t1 t0])
		    (match t1
		      [`(pointer (const ,t2)) (loop `(pointer ,t2))]
		      ['(pointer unsigned-fixnum) 'u32vector]
		      ['(pointer unsigned-integer) 'u32vector]
		      ['(pointer unsigned-short) 'u16vector]
		      ['(pointer unsigned-char) 'u8vector]
		      ['(pointer unsigned-long) 'u32vector]
		      ['(pointer fixnum) 's32vector]
		      ['(pointer integer) 's32vector]
		      ['(pointer int) 's32vector]
		      ['(pointer short) 's16vector]
		      ['(pointer char) 'c-string]
		      ['(pointer long) 's32vector]
		      ['(pointer float) 'f32vector]
		      ['(pointer double) 'f64vector]
		      [_ t0] ) ) )
	      more) ] ) ) ) ) )

(define (parse-arglist ts)
  (let rec ([more ts] [args '()])
    (match more
      [('close-paren . more)
       (values (reverse args) more) ]
      [_ (let-values ([(type more) (parse-type more #f)])
	   (match more
	     [(('id str) 'comma . more)
	      (rec more (cons type args)) ]
	     [(('id str) 'close-paren . more)
	      (values (reverse (cons type args)) more) ]
	     [('comma . more) 
	      (rec more (cons type args)) ]
	     [('close-paren . more)
	      (values (reverse (cons type args)) more) ]
	     [_ (quit "bad argument list syntax: ~S" more)] ) ) ] ) ) )

(define (parse-typelist ts)
  (let rec ([more ts] [ts '()])
    (match more
      [(('op ">") . more)
       (values (reverse ts) more) ]
      [_ (let-values ([(type more) (parse-type more #f)])
	   (match more
	     [('comma . more)
	      (rec more (cons type ts)) ]
	     [(('op ">") . more)
	      (values (reverse (cons type ts)) more) ]
	     [_ (quit "bad template type list syntax: ~S" more)] ) ) ] ) ) )

(define (subst-macros chunk)
  (let loop ([c chunk])
    (match c
      [() '()]
      [((and x ('id name)) . more)
	(let ([a (assq (string->symbol name) macro-table)])
	  (if a
	      (loop (append (third a) more))
	      (cons x (loop more)) ) ) ]
      [(x . y) (cons x (loop y))] ) ) )

(define (parse-prototype ts cb)
  (let-values ([(rtype more) (parse-type ts #t)])
    (match more
      [(('id str) 'open-paren 'void 'close-paren . more)
       (process-prototype-def rtype (string->symbol str) '() cb)
       (match more
	 [(('scope . _) . more) (parse-again more)]
	 [() #f]
	 [_ (quit "unexpected tokens: ~S" more)] ) ]
      [(('id str) 'open-paren . more)
       (let-values ([(args more) (parse-arglist more)])
	 (process-prototype-def rtype (string->symbol str) args cb)
	 (match more
	   [(('scope . _) . more) (parse-again more)]
	   [() #f]
	   [_ (quit "unexpected tokens: ~S" more)] ) ) ]
      [(('id str) . (or (('op "=") . _) ()))
       (process-variable-def rtype (string->symbol str)) ]
      [else (quit "bad prototype syntax `~A'" more)] ) ) )

(define (parse-constant ts)
  (let-values ([(type more) (parse-type ts #t)])
    (match more
      [(('id name) ('op "=") (num n))
       (process-constant-def (string->symbol name) n) ]
      [(('id name) . _)
       (process-macro-def (string->symbol name) type) ] 
      [_ (quit "syntax error in constant form: ~S" ts)] ) ) )

(define (parse-enum-def ename ts)
  (let loop ([ts ts] [i 0] [items '()])
    (match ts
      [('close-curly) #f]
      [_ (let-values ([(sym val more) (parse-enum-item ts i)])
	   (let ([items (alist-cons sym val items)]
		 [i (add1 val)] )
	     (match more
	       [() (process-enum-def ename items)]
	       [('comma . more) (loop more i items)]
	       [_ (quit "syntax error in enum form `~A'" more)] ) ) ) ] ) ) )

(define (parse-enum-item ts i)
  (match ts
    [(('id name) ('op "=") ('num n) . more)
     (if (integer? n)
	 (values (string->symbol name) n more) 
	 (error "non-exact enum value for `~A'" name) ) ]
    [(('id name) . more)
     (values (string->symbol name) i more) ] 
    [_ (quit "invalid enum syntax `~A'" ts)] ) )

(define (parse-typedef ts)
  (let-values ([(type more) (parse-type ts #t)])
    (match more
      [(('id tname))
       (set! type-map (alist-cons (string->symbol tname) type type-map)) ]
      [_ (quit "invalid typedef syntax `~A'" more)] ) ) )

(define has-constructor #f)
(define defined-classes '())

(define (parse-class-def ts)
  (match ts
    [(('id name)) 
     (set! defined-classes (cons (string->symbol name) defined-classes)) ]
    [(('id name) . more)
     (set! defined-classes (cons (string->symbol name) defined-classes)) 
     (let loop ([more more] [t '(op ":")] [bases '()])
       (if (and (pair? more) (equal? t (car more)))
	   (match more
	     [(_ (or 'public 'protected 'private) ('id bname) . more)
	      (loop more 'comma 
		    (if (memq (string->symbol bname) defined-classes)
			(cons bname bases)
			bases) ) ]
	     [(_ ('id bname) . more)
	      (loop more 'comma
		    (if (memq (string->symbol bname) defined-classes)
			(cons bname bases)
			bases) ) ]
	     [_ (quit "invalid class definition for `~A': ~S" name more)] ) 
	   (match more
	     [(('scope . chunks))
	      (let ([cname (fix-cname name)])
		(process-class-def name cname bases)
		(fluid-let ([has-constructor #f])
		  (let ([exp #f])
		    (for-each
		     (lambda (chunk)
		       (let loop ([more (subst-macros chunk)])
			 (match more
			   [() #f]
			   [((or 'public 'protected) ('op ":") . more) 
			    (set! exp #t)
			    (loop more) ]
			   [('private ('op ":") . more) 
			    (set! exp #f)
			    (loop more) ]
			   [more 
			    (when exp 
			      (fluid-let ([parse-again loop])
				(parse-member-prototype name cname more #f) ) ) ] ) ) )
		     chunks)
		    (unless has-constructor
		      (process-constructor-def name cname '()) ) ) ) ) ]
	     [_ (quit "invalid class definition for `~A': ~S" name more)] ) ) ) ]
    [_ (quit "invalid class definition: ~S" ts)] ) )

(define (parse-member-prototype name cname ts cb)
  (match ts
    [('callback . more) 
     (parse-member-prototype name cname more #t) ]
    [((or 'explicit 'virtual) . more)
     (parse-member-prototype name cname more cb) ]
    [(('id name2) 'open-paren 'void 'close-paren . more)
     (if (string=? name2 name)
	 (begin
	   (process-constructor-def name cname '())
	   (set! has-constructor #t)
	   (match more
	     [(('scope . _) . more) (parse-again more)]
	     [() #f]
	     [_ (quit "unexpected tokens: ~S" more)] ) )
	 (quit "invalid constructor for `~A': ~S" name ts) ) ]
    [(('id name2) 'open-paren . more)
     (if (string=? name2 name)
	 (let-values ([(args more) (parse-arglist more)])
	   (process-constructor-def name cname args) 
	   (set! has-constructor #t)
	   (match more
	     [(('scope . _) . more) (parse-again more)]
	     [() #f]
	     [_ (quit "unexpected tokens: ~S" more)] ) )
	 (quit "invalid constructor for `~A': ~S" name ts) ) ]
    [(('op "~") ('id name2) 'open-paren . (or ('void 'close-paren . more) ('close-paren . more)))
     (if (string=? name2 name)
	 (match more
	   [(('scope . _) . more) (parse-again more)]
	   [() #f]
	   [_ (quit "unexpected tokens: ~S" more)] )
	 (quit "invalid destructor for `~A': ~S" name ts) ) ]
    [('static . more)
     (let-values ([(rtype more) (parse-type more #t)])
       (match more
	 [(('id str) 'open-paren 'void 'close-paren . more)
	    (process-prototype-def rtype (string->symbol (string-append name "::" str)) '() cb)
	    (match more
	      [(('scope . _) . more) (parse-again more)]
	      [() #f]
	      [_ (quit "unexpected tokens: ~S" more)] ) ]
	 [(('id str) 'open-paren . more)
	  (let-values ([(args more) (parse-arglist more)])
	    (process-prototype-def rtype (string->symbol (string-append name "::" str)) args cb)
	    (match more
	      [(('scope . _) . more) (parse-again more)]
	      [() #f]
	      [_ (quit "unexpected tokens: ~S" more)] ) ) ]
	 [_ (quit "bad static member prototype syntax: ~S" more)] ) ) ]
    [_ (let-values ([(rtype more) (parse-type ts #t)])
	 (match more
	   [(('id str) 'open-paren 'void 'close-paren . more)
	    (process-member-prototype-def name cname rtype (string->symbol str) '() cb)
	    (parse-member-body more) ]
	   [(('id str) 'open-paren . more)
	    (let-values ([(args more) (parse-arglist more)])
	      (process-member-prototype-def name cname rtype (string->symbol str) args cb)
	      (parse-member-body more) ) ]
	   [(('id str) . (or (('op "=") . _) ()))
	    #f]				; member variables are ignored
	   [_ (quit "bad member prototype syntax: ~S" more)] ) ) ] ) )

(define (parse-member-body ts)
  (let loop ([more ts])
    (match more
      [('const . more) (loop more)]
      [(('op "=") (num 0) . more) 
       (set! has-constructor #t)
       (loop more) ]
      [(('scope . _) . more) (parse-again more)]
      [() #f]
      [_ (quit "unexpected tokens: ~S" more)] ) ) )

(define reparse-item 
  (match-lambda 
   ['pp-define "#define"]
   ['pp-include "#include"]
   ['pp-undef "#undef"]
   ['pp-ifndef "#ifndef"]
   ['pp-ifdef "#ifdef"]
   ['pp-if "#if"]
   ['pp-pragma "#pragma"]
   ['pp-error "#error"]
   ['pp-else "#else"]
   ['pp-endif "#endif"]
   [('id str) str]
   [('num num) num]
   [('op op) op]
   ['star "*"]
   ['open-paren "("]
   ['close-paren ")"]
   ['open-bracket "["]
   ['close-bracket "]"]
   ['open-curly "{"]
   ['close-curly "}"]
   ['fixnum "int"]
   ['comma ","]
   [('string str) (string-append "\"" str "\"")]
   [('i-string str) (string-append "<" str ">")]
   ['class "class"]
   ['protected "protected"]
   ['public "public"]
   ['private "private"]
   [c c] ) )

(define (type-union t1 t2)
  (cond [(eq? '_ t2) t1]
	[(eq? t1 t2) t1]
	[(eq? 'integer t1)
	 (case t2
	   [(double) 'double]
	   [else '*] ) ]
	[(and (eq? 'double t1) (eq? 'integer t2)) 'double]
	[else '*] ) )

(define (compute-macro-type ts)
  (let rec ([ts ts])
    (if (null? ts)
	'_
	(match (car ts)
	  [('num n) (type-union (if (exact? n) 'integer 'double) (rec (cdr ts)))]
	  [('id str)
	   (let ([a (assq (string->symbol str) macro-table)])
	     (if a 
		 (type-union (second a) (rec (cdr ts)))
		 '*) ) ]
	  [_ (rec (cdr ts))] ) ) ) )

(define (emit . xs)
  (when (memq '|F| debugging-chicken)
    (for-each pretty-print xs) )
  (set! processed-output (append-reverse xs processed-output)) )

(define (process-macro-def name type)
  (if (memq type '(* _))
      (warning "can not compute macro type `~A' (ignored)" name)
      (let ([name2 (fix-name name)])
	(emit `(define-foreign-variable ,name2 ,type ,(->string name)))
	(when export-constants (emit `(define ,name2 ,name2))) ) ) )

(define (process-constant-def name val)
  (let ([name (fix-name name)])
    (emit `(define-constant ,name ,val))
    (when export-constants (emit `(define ,name ,name))) ) )

(define (process-prototype-def rtype name args cb)
  (let ([name2 (fix-name name)])
    (emit `(define ,name2
	     (,(if cb 'foreign-callback-lambda 'foreign-lambda)
	      ,rtype ,(->string name) ,@args))) ) )

(define (process-variable-def rtype name)
  (let ([tmp (gensym)]
	[var (gensym)] 
	[name2 (fix-name name)] )
    (emit 
     `(define-foreign-variable ,tmp ,rtype ,(->string name))
     `(define (,name2 . ,var)
	(if (pair? ,var)
	    (set! ,tmp (car ,var))
	    ,tmp) ) ) ) )

(define (process-enum-def ename items)
  (for-each
   (match-lambda
     [(name . val)
      (let ([name (fix-name name)])
	(emit `(define-constant ,name ,val))
	(when export-constants (emit `(define ,name ,name))) ) ] )
   (reverse items) ) )

(define (process-class-def name cname basenames)
  (let ([destr (gensym)]
	[name2 (fix-name name)] )
    (emit
     `(declare (hide ,destr))
     `(define ,destr (foreign-lambda void "delete " (pointer ,name)))
     `(define-class ,cname
	,(if (null? basenames)
	     `(<c++-object>)
	     (map (lambda (b) (fix-cname b)) (reverse basenames) ) )
	() )
     `(define-method (destroy (this ,cname))
	(,destr (slot-ref this 'this)) ) ) ) )

(define (process-constructor-def name cname args)
  (let ([constr (gensym)])
    (emit
     `(declare (hide ,constr))
     `(define ,constr (foreign-lambda (pointer ,name) ,(string-append "new " name) ,@args))
     `(define-method (initialize (this ,cname) initargs) 
	;; no CALL-NEXT-METHOD here: we don't want to invoke the base-class constructor.
	(slot-set! this 'this (apply ,constr initargs)) ) ) ) )

(define (process-member-prototype-def name cname rtype mname args cb)
  (let* ([stub (gensym)]
	 [this (gensym)] 
	 [vars (map (lambda _ (gensym)) args)] 
	 [fvars (map list args vars)] )
    (emit 
     `(declare (hide ,stub))
     `(define ,stub 
	(,(if cb 'foreign-callback-lambda* 'foreign-lambda*)
	 ,rtype (((pointer ,name) ,this) ,@fvars)
	 ,(sprintf (if (eq? 'void rtype) "~A->~A(~A);" "return(~A->~A(~A));")
		   this mname
		   (string-intersperse (map ->string vars) ",")) ) )
     `(define-method (,mname (this ,cname) . args)
	(apply ,stub (slot-ref this 'this) args) ) ) ) )

(define parse-declaration
  (match-lambda*
    [("export_constants" (or ('id "yes") ('num 1)))
     (set! export-constants #t) ]
    [("export_constants" _)
     (set! export-constants #f) ]
    [("prefix" ('string str))
     (set! prefix str) ]
    [("prefix" (or ('id "no") ('num 0)))
     (set! prefix #f) ]
    [("scheme" ('string str))
     (let ([exp (with-input-from-string str read)])
       (emit exp) ) ]
    [("type" ('string str))
     (parse-type-declaration (string-split str ";")) ]
    [("rename" ('string str))
     (match (string-split str ";")
       [(from to) 
	(set! rename-list (alist-cons (string->symbol from) (string->symbol to) rename-list)) ]
       [_ (quit "invalid rename declaration: ~S" decl)] ) ]
    [("substitute" ('string str))
     (match (string-split str ";")
       [(from to) 
	(set! name-substitution-rxs (cons from name-substitution-rxs))
	(set! name-substitution-repls (cons to name-substitution-repls)) ]
       [_ (quit "invalid name substitution string: ~S" str)] ) ]
    [(decl _)
     (quit "invalid pseudo declaration: ~S" decl) ] ) )

(define (parse-type-declaration vals)
  (define (reads str)
    (handle-exceptions ex (quit "can not parse expression: ~S" str)
      (with-input-from-string str read) ) )
  (let rec ([vals vals])
    (match vals
      [(tname stype arg ret)
       (let ([stype (reads stype)]
	     [arg (and arg (reads arg))]
	     [ret (and ret (reads ret))] 
	     [stname (string->symbol tname)] )
	 (set! foreign-declarations (cons (sprintf "#define ~A ~A~%" tname (foreign-type-declaration stype "")) foreign-declarations))
	 (emit 
	  `(define-foreign-type ,stname ,stype ,@(if arg (list arg) '()) ,@(if ret (list ret) '())) )
	 (set! declared-types (cons stname declared-types)) ) ]
      [(tname stype arg) (rec (list tname stype arg #f))]
      [(tname stype) (rec (list tname stype #f #f))]
      [_ (quit "invalid value-syntax in type declaration: ~S" vals)] ) ) )

(define (fix-name str)
  (let ([a (assq (->symbol str) rename-list)])
    (if a 
	(cdr a)
	(let ([n1 (fold string-substitute (->string str) name-substitution-rxs name-substitution-repls)])
	  (string->symbol
	   (if prefix
	       (string-append prefix n1)
	       n1) ) ) ) ) )

(define (fix-cname str)
  (let ([a (assq (->symbol str) rename-list)])
    (if a 
	(cdr a)
	(string->symbol (string-append "<" (->string (fix-name str)) ">")) ) ) )

(define (->symbol s)
  (if (symbol? s)
      s
      (string->symbol s) ) )

(define (parse-easy-ffi text)
  (lexer-init 'string text)
  (set! processed-output '())
  (set! pp-conditional-stack '())
  (set! pp-process #t)
  (let ([chunks (chunkify)])
    ;;(pretty-print chunks)		
    (for-each parse chunks)
    (reverse processed-output) ) )

(define (parse-easy-ffi-rec port)
  (lexer-init 'port port)
  (let* ([output processed-output]
	 [chunks (chunkify)] )
    (set! processed-output '())
    (for-each parse chunks)
    (set! processed-output (append output processed-output)) ) )

(define (register-ffi-macro name)
  (set! macro-table (cons (list (string->symbol name) '* '()) macro-table)) )

(define (resolve-ffi-include-file fname)
  (find file-exists? (map (cut make-pathname <> fname) ffi-include-path)) )
