/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "log.h"
#include "logadaptor.h"

#include <QtCore/QMutexLocker>
#include <QtCore/QPointer>
#include <QtCore/QFile>
#include <QtCore/QTextStream>
#include <QtCore/QString>
#include <QtCore/QDebug>

#include <Decibel/Errors>

/// @cond INCLUDE_PRIVATE

namespace
{
const QString logFile("~/.Decibel/channel.log");

QString logFileName()
{
    QString result(logFile);
    if (result.startsWith('~'))
    { result = qgetenv("HOME") + result.mid(1); }

    return result;
}

}

/**
 * @brief Private class implementing the Log.
 *
 * A private class implementing the Log.
 *
 * FIXME: Actually save the log to disc.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */

class LogPrivate
{
public:
    /**
     * @brief Constructor.
     */
    LogPrivate()
    {
        QString file_name(logFileName());
        QFile file(file_name);
        file.open(QIODevice::ReadOnly);
        QTextStream stream(&file);
        qDebug() << "Reading Logentry from file" << file_name;

        while (stream.status() == QTextStream::Ok)
        {
            QString line;
            stream >> line;
            if (line.isEmpty()) { continue; }

            qDebug() << ">>>" << line;
            Decibel::LogEntry * entry = new Decibel::LogEntry();

            QStringList line_parts(line.split(':'));

            if (line_parts.size() != 6) { continue; }

            bool ok = false;
            entry->startTime = line_parts[0].toInt(&ok);
            if (!ok)
            {
                qWarning() << "startTime is not OK.";
                delete entry; break;
            }
            entry->connectTime = line_parts[1].toInt(&ok);
            if (!ok)
            {
                qWarning() << "connectTime is not OK.";
                delete entry; break;
            }
            entry->endTime = line_parts[2].toInt(&ok);
            if (!ok)
            {
                qWarning() << "endTime is not OK.";
                delete entry; break;
            }
            entry->protocol = line_parts[3];
            if (entry->protocol.isEmpty())
            {
                qWarning() << "Protocol is empty.";
                delete entry; break;
            }
            entry->incoming = (line_parts[4].toInt(&ok) != 0);
            if (!ok)
            {
                qWarning() << "Incoming is not OK.";
                delete entry; break;
            }
            entry->contacts = line_parts[5].split(",");
            if (entry->contacts.isEmpty())
            {
                qWarning() << "Contacts are not OK.";
                delete entry; break;
            }
            entries.append(entry);
        }
        qDebug() << "Read" << entries.size() << "entries from logfile.";
    }

    /**
     * @brief Destructor.
     */
    ~LogPrivate()
    { }

    void clearLog()
    {
        entries.clear();

        // Clear file:
        QFile file(logFileName());
        file.resize(0);
        qDebug() << "Logfile" << logFileName() << "cleared.";
    }

    void addEntry(const Decibel::LogEntry * const entry)
    {
        QString file_name(logFileName());
        QFile file(file_name);
        qDebug() << "Writing LogEntry to:" << file_name;
        file.open(QIODevice::WriteOnly | QIODevice::Append);
        QTextStream stream(&file);

        stream << entry->startTime << ':'
               << entry->connectTime << ':'
               << entry->endTime << ':'
               << entry->protocol << ':'
               << ((entry->incoming) ? 1 : 0) << ':'
               << entry->contacts.join(",") << '\n';

        entries.append(entry);
    }

    QList<const Decibel::LogEntry *> entries;
    /** @brief A pointer to the D-Bus Adaptor of the AccountManager. */
    QPointer<LogAdaptor> m_adaptor;
};

/// @endcond

// ****************************************************************************

Log::Log(QObject * parent) :
    QObject(parent),
    d(new LogPrivate())
{
    Q_ASSERT(d != 0);
    d->m_adaptor = new LogAdaptor(this);
    Q_ASSERT(d->m_adaptor != 0);
}

Log::~Log()
{ delete d; }

void Log::addEntry(Decibel::LogEntry * entry)
{
    d->addEntry(entry);
    emit newEntry(d->entries.size() - 1);
}

void Log::clearLog()
{
    d->clearLog();
    emit logCleared();
}

uint Log::numberOfEntries() const
{ return d->entries.size(); }

Decibel::LogEntry Log::getEntry(uint pos) const
{
    if (pos < uint(d->entries.size()))
    {
        Decibel::LogEntry result(*(d->entries[pos]));
        return result;
    }
    else
    {
        sendErrorReply(Decibel::ErrorInvalidValue,
                       tr("Position is invalid."));
        return Decibel::LogEntry();
    }
}

#include "log.moc"
