/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by:
 *               Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

/**
 * SECTION:dee-versioned-model
 * @short_description: Abstract base class for easing implementations of
 *                     #DeeModel<!-- -->s providing a unique version number
 *                     for each row
 * @include: dee.h
 *
 * #DeeVersionedModel is an abstract base class to ease implementation of
 * #DeeModel<!-- -->s providing rows versioned by a
 * <emphasis>sequence number</emphasis>.
 *
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <memory.h>
#include <time.h>
#include <unistd.h>
#include <gobject/gvaluecollector.h>

#include "dee-model.h"
#include "dee-versioned-model.h"
#include "dee-marshal.h"
#include "trace-log.h"

static void dee_versioned_model_model_iface_init (DeeModelIface *iface);
G_DEFINE_ABSTRACT_TYPE_WITH_CODE (DeeVersionedModel,
                                  dee_versioned_model,
                                  G_TYPE_OBJECT,
                                  G_IMPLEMENT_INTERFACE (DEE_TYPE_MODEL,
                                                         dee_versioned_model_model_iface_init));

#define DEE_VERSIONED_MODEL_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, DEE_TYPE_VERSIONED_MODEL, DeeVersionedModelPrivate))

/**
 * DeeVersionedModelPrivate:
 *
 * Ignore this structure.
 */
struct _DeeVersionedModelPrivate
{
  /* Seqnum tracking */
  GArray     *seqnums;
  guint64     seqnum;

  /* Column type info */
  guint       n_columns;
  GType      *column_types;
  
};

/*
 * Public overridable DeeVersionedModel methods
 */
static DeeSeqnumState
                 dee_versioned_model_check_seqnums_real    (DeeModel *self,
                                                           GArray *seqnums);

static DeeSeqnumState
                 dee_versioned_model_check_seqnum_real     (DeeModel     *self,
                                                           guint64       seqnum);

static guint64   dee_versioned_model_inc_seqnum_real       (DeeModel     *self,
                                                           guint         pos);
                                                          
static void      dee_versioned_model_set_seqnum_real       (DeeModel     *self,
                                                           guint         pos,
                                                           guint64       seqnum);

static guint64   dee_versioned_model_append_next_seqnum_real  (DeeModel     *self);

static guint64   dee_versioned_model_prepend_next_seqnum_real (DeeModel     *self);

static guint64   dee_versioned_model_insert_next_seqnum_real  (DeeModel     *self,
                                                              guint         pos);

static guint64   dee_versioned_model_get_seqnum_real       (DeeModel     *self,
                                                           guint         pos);

static guint64   dee_versioned_model_get_last_seqnum_real  (DeeModel     *self);

static void      dee_versioned_model_set_last_seqnum_real  (DeeModel     *self,
                                                            guint64       seqnum);

static void      dee_versioned_model_remove_seqnum_real    (DeeModel     *self,
                                                           guint         pos);

/*
 * DeeModel forward declarations
 */
static void           dee_versioned_model_set_column_type (DeeModel *self,
                                                           guint     column,
                                                           GType     type);

static void           dee_versioned_model_set_n_columns  (DeeModel *self,
                                                          guint     n_columns);

static guint          dee_versioned_model_get_n_columns  (DeeModel *self);

static guint          dee_versioned_model_get_n_rows     (DeeModel *self);

static GType          dee_versioned_model_get_column_type (DeeModel *self,
                                                          guint      column);

static void           dee_versioned_model_clear          (DeeModel *self);

static DeeModelIter*  dee_versioned_model_append_valist  (DeeModel *self,
                                                         va_list    args);

static DeeModelIter*  dee_versioned_model_prepend_valist  (DeeModel *self,
                                                          va_list    args);

static DeeModelIter*  dee_versioned_model_insert_valist  (DeeModel *self,
                                                         gint       pos,
                                                         va_list    args);

static DeeModelIter*  dee_versioned_model_insert_before_valist (DeeModel     *self,
                                                              DeeModelIter *iter,
                                                              va_list    args);

static void           dee_versioned_model_remove         (DeeModel     *self,
                                                          DeeModelIter *iter);

static void           dee_versioned_model_set_valist     (DeeModel       *self,
                                                          DeeModelIter   *iter,
                                                          va_list          args);

static void           dee_versioned_model_set_value      (DeeModel       *self,
                                                         DeeModelIter   *iter,
                                                         guint            column,
                                                         const GValue    *value);

static void           dee_versioned_model_set_value_silently (DeeModel       *self,
                                                              DeeModelIter   *iter,
                                                              guint            column,
                                                              const GValue    *value);

static void           dee_versioned_model_get_valist     (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         va_list        args);

static void           dee_versioned_model_get_value      (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column,
                                                         GValue        *value);

static DeeModelIter* dee_versioned_model_get_first_iter  (DeeModel     *self);

static DeeModelIter* dee_versioned_model_get_last_iter   (DeeModel     *self);

static DeeModelIter* dee_versioned_model_get_iter_at_row (DeeModel     *self,
                                                         guint          row);

static gboolean       dee_versioned_model_get_bool       (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static guchar         dee_versioned_model_get_uchar      (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static gint           dee_versioned_model_get_int        (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static guint          dee_versioned_model_get_uint       (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static gint64         dee_versioned_model_get_int64      (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static guint64        dee_versioned_model_get_uint64     (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static gdouble        dee_versioned_model_get_double     (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static const gchar*   dee_versioned_model_get_string     (DeeModel     *self,
                                                         DeeModelIter *iter,
                                                         guint          column);

static DeeModelIter* dee_versioned_model_next            (DeeModel     *self,
                                                         DeeModelIter *iter);

static DeeModelIter* dee_versioned_model_prev            (DeeModel     *self,
                                                         DeeModelIter *iter);

static gboolean       dee_versioned_model_is_first       (DeeModel     *self,
                                                         DeeModelIter *iter);

static gboolean       dee_versioned_model_is_last        (DeeModel     *self,
                                                         DeeModelIter *iter);

static gint           dee_versioned_model_get_position   (DeeModel     *self,
                                                         DeeModelIter *iter);

static void           dee_versioned_model_freeze_signals (DeeModel     *self);

static void           dee_versioned_model_thaw_signals   (DeeModel     *self);

/* GObject Init */
static void
dee_versioned_model_finalize (GObject *object)
{
  //DeeVersionedModelPrivate *priv = DEE_VERSIONED_MODEL (object)->priv;

  G_OBJECT_CLASS (dee_versioned_model_parent_class)->finalize (object);
}

static void
dee_versioned_model_set_property (GObject      *object,
                                 guint         id,
                                 const GValue *value,
                                 GParamSpec   *pspec)
{
  switch (id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_versioned_model_get_property (GObject    *object,
                                 guint       id,
                                 GValue     *value,
                                 GParamSpec *pspec)
{
  switch (id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, id, pspec);
      break;
    }
}

static void
dee_versioned_model_class_init (DeeVersionedModelClass *klass)
{
  GObjectClass  *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize     = dee_versioned_model_finalize;  
  obj_class->set_property = dee_versioned_model_set_property;
  obj_class->get_property = dee_versioned_model_get_property;

  klass->check_seqnums   = dee_versioned_model_check_seqnums_real;
  klass->check_seqnum    = dee_versioned_model_check_seqnum_real;
  klass->inc_seqnum      = dee_versioned_model_inc_seqnum_real;
  klass->set_seqnum      = dee_versioned_model_set_seqnum_real;
  klass->append_next_seqnum  = dee_versioned_model_append_next_seqnum_real;
  klass->prepend_next_seqnum = dee_versioned_model_prepend_next_seqnum_real;
  klass->insert_next_seqnum  = dee_versioned_model_insert_next_seqnum_real;
  klass->get_seqnum      = dee_versioned_model_get_seqnum_real;
  klass->get_last_seqnum = dee_versioned_model_get_last_seqnum_real;
  klass->set_last_seqnum = dee_versioned_model_set_last_seqnum_real;
  klass->remove_seqnum   = dee_versioned_model_remove_seqnum_real;

  /* Add private data */
  g_type_class_add_private (obj_class, sizeof (DeeVersionedModelPrivate));
}

static void
dee_versioned_model_init (DeeVersionedModel *model)
{
  DeeVersionedModelPrivate *priv;

  priv = model->priv = DEE_VERSIONED_MODEL_GET_PRIVATE (model);

  priv->seqnums = g_array_new (FALSE, FALSE, sizeof (guint64));
  priv->seqnum = 0;

  priv->n_columns = 0;
  priv->column_types = NULL;
  
  
}

static DeeSeqnumState
dee_versioned_model_check_seqnums_real (DeeModel *self, GArray *seqnums)
{
  DeeVersionedModelPrivate *priv;
  guint64                  seqnum;
  gint                     i;

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), FALSE);
  g_return_val_if_fail (seqnums != NULL, FALSE);
  g_return_val_if_fail (seqnums->len > 0, FALSE);

  priv = DEE_VERSIONED_MODEL (self)->priv;

  /* Check that the incoming seqnums are consecutive */
  seqnum = g_array_index (seqnums, guint64, 0);
  for (i = 1; i < seqnums->len; i++)
    {
      if (seqnum != g_array_index (seqnums, guint64, i) - 1)
        { 
          return DEE_SEQNUM_STATE_NOT_CONSECUTIVE;
        }
      seqnum = g_array_index (seqnums, guint64, i);
    }

  /* Check how the incoming seqnums align with our own */
  seqnum = g_array_index (seqnums, guint64, 0);
  if (seqnum > priv->seqnum + 1)
    {
      return DEE_SEQNUM_STATE_IN_THE_FUTURE;
    }
  else if (seqnum < priv->seqnum + 1)
    {
      return DEE_SEQNUM_STATE_IN_THE_PAST;
    }
  
  return DEE_SEQNUM_STATE_OK;
}

/**
 * dee_versioned_model_check_seqnums:
 * @self:
 * @seqnums:
 *
 * Assert that @seqnums are consecutive and that the beggining sequence number
 * in @seqnums is exactly one higher than the last seen seqnum of this model.
 */
DeeSeqnumState
dee_versioned_model_check_seqnums (DeeModel *self, GArray *seqnums)
{
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), DEE_SEQNUM_STATE_OK);

  return DEE_VERSIONED_MODEL_GET_CLASS (self)->check_seqnums (self, seqnums);
}

static DeeSeqnumState
dee_versioned_model_check_seqnum_real (DeeModel     *self,
                                      guint64       seqnum)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), DEE_SEQNUM_STATE_OK);

  priv = DEE_VERSIONED_MODEL (self)->priv;

  if (priv->seqnum > seqnum)
    return DEE_SEQNUM_STATE_IN_THE_PAST;
  if (priv->seqnum < seqnum)
    return DEE_SEQNUM_STATE_IN_THE_FUTURE;
  else
    return DEE_SEQNUM_STATE_OK;
}

DeeSeqnumState
dee_versioned_model_check_seqnum (DeeModel     *self,
                                 guint64       seqnum)
{
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), DEE_SEQNUM_STATE_OK);

  return DEE_VERSIONED_MODEL_GET_CLASS (self)->check_seqnum (self, seqnum);
}

static guint64
dee_versioned_model_inc_seqnum_real (DeeModel     *self,
                                    guint         pos)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  priv = DEE_VERSIONED_MODEL (self)->priv;
  g_return_val_if_fail (pos < priv->seqnums->len, 0);
  
  priv->seqnum = priv->seqnum + 1;
  g_array_index (priv->seqnums, guint64, pos) = priv->seqnum;
  
  return priv->seqnum;
}

guint64
dee_versioned_model_inc_seqnum (DeeModel     *self,
                               guint         pos)
{
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  return DEE_VERSIONED_MODEL_GET_CLASS (self)->inc_seqnum (self, pos);
}

static void
dee_versioned_model_set_seqnum_real (DeeModel     *self,
                                    guint         pos,
                                    guint64       seqnum)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_if_fail (DEE_IS_VERSIONED_MODEL (self));

  priv = DEE_VERSIONED_MODEL (self)->priv;  

  g_array_index (priv->seqnums, guint64, pos) = seqnum;
}
                                                          
void
dee_versioned_model_set_seqnum (DeeModel     *self,
                               guint         pos,
                               guint64       seqnum)
{
  g_return_if_fail (DEE_IS_VERSIONED_MODEL (self));

  DEE_VERSIONED_MODEL_GET_CLASS (self)->set_seqnum (self, pos, seqnum);
}

guint64
dee_versioned_model_append_next_seqnum (DeeModel     *self)
{
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  return DEE_VERSIONED_MODEL_GET_CLASS (self)->append_next_seqnum (self);
}

static guint64
dee_versioned_model_append_next_seqnum_real (DeeModel     *self)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  priv = DEE_VERSIONED_MODEL (self)->priv;
  priv->seqnum = priv->seqnum + 1;
  g_array_append_val (priv->seqnums, (priv->seqnum));
  
  return priv->seqnum;
}

guint64
dee_versioned_model_prepend_next_seqnum (DeeModel     *self)
{
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  return DEE_VERSIONED_MODEL_GET_CLASS (self)->prepend_next_seqnum (self);
}

static guint64
dee_versioned_model_prepend_next_seqnum_real (DeeModel     *self)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  priv = DEE_VERSIONED_MODEL (self)->priv;
  priv->seqnum = priv->seqnum + 1;
  g_array_prepend_val (priv->seqnums, priv->seqnum);
  
  return priv->seqnum;
}

guint64
dee_versioned_model_insert_next_seqnum (DeeModel     *self,
                                       guint         pos)
{
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  return DEE_VERSIONED_MODEL_GET_CLASS (self)->insert_next_seqnum (self, pos);
}


static guint64
dee_versioned_model_insert_next_seqnum_real (DeeModel *self, guint pos)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);
  g_return_val_if_fail (pos >= 0, 0);

  priv = DEE_VERSIONED_MODEL (self)->priv;
  g_return_val_if_fail (pos < priv->seqnums->len, 0);
  
  priv->seqnum = priv->seqnum + 1;
  g_array_insert_val (priv->seqnums, pos, priv->seqnum);
  
  return priv->seqnum;
}

static guint64
dee_versioned_model_get_seqnum_real (DeeModel     *self,
                                    guint         pos)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  priv = DEE_VERSIONED_MODEL (self)->priv;  
  g_return_val_if_fail (pos < priv->seqnums->len, 0);

  return g_array_index (priv->seqnums, guint64, pos);
}
                                                          
guint64
dee_versioned_model_get_seqnum (DeeModel     *self,
                               guint         pos)
{
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  return DEE_VERSIONED_MODEL_GET_CLASS (self)->get_seqnum (self, pos);
}

static guint64
dee_versioned_model_get_last_seqnum_real (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);
  
  return DEE_VERSIONED_MODEL (self)->priv->seqnum;
}
                                                          
guint64
dee_versioned_model_get_last_seqnum (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  return DEE_VERSIONED_MODEL_GET_CLASS (self)->get_last_seqnum (self);
}

static void
dee_versioned_model_set_last_seqnum_real (DeeModel *self,
                                          guint64   seqnum)
{
  g_return_if_fail (DEE_IS_VERSIONED_MODEL (self));
    
  DEE_VERSIONED_MODEL (self)->priv->seqnum = seqnum;
}
                                                          
void
dee_versioned_model_set_last_seqnum (DeeModel *self,
                                     guint64   seqnum)
{
  g_return_if_fail (DEE_IS_VERSIONED_MODEL (self));

  DEE_VERSIONED_MODEL_GET_CLASS (self)->set_last_seqnum (self, seqnum);
}

static void
dee_versioned_model_remove_seqnum_real (DeeModel *self,
                                       guint       pos)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_if_fail (DEE_IS_VERSIONED_MODEL (self));

  priv = DEE_VERSIONED_MODEL (self)->priv;  
  g_return_if_fail (pos < priv->seqnums->len);
  
  g_array_remove_index (priv->seqnums, pos);
}


void
dee_versioned_model_remove_seqnum (DeeModel *self,
                                  guint       pos)
{
  g_return_if_fail (DEE_IS_VERSIONED_MODEL (self));

  DEE_VERSIONED_MODEL_GET_CLASS (self)->remove_seqnum (self, pos);
}


/*
 * DeeModel API
 */

static void
dee_versioned_model_set_column_type (DeeModel *self,
                                     guint     column,
                                     GType     type)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_if_fail (DEE_IS_VERSIONED_MODEL (self));

  priv = DEE_VERSIONED_MODEL (self)->priv;
  g_return_if_fail (column <= dee_model_get_n_columns (self));

  priv->column_types[column] = type;
}

static void
dee_versioned_model_set_n_columns (DeeModel *self,
                                   guint     n_columns)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_if_fail (DEE_IS_VERSIONED_MODEL (self));

  priv = DEE_VERSIONED_MODEL (self)->priv;

  if (priv->column_types != NULL)
    g_free (priv->column_types);

  priv->column_types = g_new (GType, n_columns);
  priv->n_columns = n_columns;
}

static guint
dee_versioned_model_get_n_columns (DeeModel *self)
{
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  return DEE_VERSIONED_MODEL (self)->priv->n_columns;
}

static guint
dee_versioned_model_get_n_rows (DeeModel *self)
{
  DeeModelIter *iter;
  guint         count;

  count = 0;
  iter = dee_model_get_first_iter (self);
  while (!dee_model_is_last (self, iter))
    {
      iter = dee_model_next (self, iter);
      count++;
    }

  return count;
}

static GType
dee_versioned_model_get_column_type (DeeModel *self,
                                    guint      column)
{
  DeeVersionedModelPrivate *priv;
  
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), G_TYPE_NONE);

  priv = DEE_VERSIONED_MODEL (self)->priv;
  g_return_val_if_fail (column < priv->n_columns, G_TYPE_NONE);

  return priv->column_types[column];
}

static void
dee_versioned_model_clear (DeeModel *self)
{
  DeeModelIter            *iter;

  g_return_if_fail (DEE_IS_VERSIONED_MODEL (self));

  iter = dee_model_get_first_iter (self);

  while (!dee_model_is_last (self, iter))
    {
      dee_model_remove (self, iter);

      iter = dee_model_get_first_iter (self);      
    }
}

static DeeModelIter*
dee_versioned_model_prepend_valist (DeeModel *self,
                                   va_list    args)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static DeeModelIter*
dee_versioned_model_append_valist (DeeModel *self,
                                  va_list    args)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static DeeModelIter*
dee_versioned_model_insert_valist (DeeModel *self,
                                  gint       pos,
                                  va_list    args)
{
  DeeModelIter *iter;

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), 0);

  if (pos > 0)
    {
      iter = dee_model_get_iter_at_row (self, pos);
      return dee_model_insert_before_valist (self, iter, args);
    }
  else if (pos == 0)
    return dee_model_prepend_valist (self, args);
  else
    return dee_model_append_valist (self, args);
}

static DeeModelIter*
dee_versioned_model_insert_before_valist (DeeModel *self,
                                         DeeModelIter *iter,
                                         va_list args)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static void
dee_versioned_model_remove (DeeModel     *self,
                           DeeModelIter *iter_)
{
  g_critical ("%s not implemented", G_STRFUNC);
}

static void
dee_versioned_model_set_valist (DeeModel       *self,
                               DeeModelIter   *iter,
                               va_list          args)
{
  g_critical ("%s not implemented", G_STRFUNC);
}

static void
dee_versioned_model_set_value (DeeModel      *self,
                              DeeModelIter  *iter,
                              guint           column,
                              const GValue   *value)
{
  g_critical ("%s not implemented", G_STRFUNC);
}


static void
dee_versioned_model_set_value_silently (DeeModel      *self,
                                        DeeModelIter  *iter,
                                        guint           column,
                                        const GValue   *value)
{
  g_critical ("%s not implemented", G_STRFUNC);
}

static void
dee_versioned_model_get_valist (DeeModel       *self,
                               DeeModelIter   *iter,
                               va_list         args)
{
  guint            column;

  g_return_if_fail (DEE_IS_VERSIONED_MODEL (self));
  g_return_if_fail (iter);

  column = va_arg (args, gint);

  while (column != -1)
    {
      GValue  value = { 0, };
      gchar  *error = NULL;

      if (column < 0 || column >= dee_model_get_n_columns (self) + 1)
        {
          g_warning ("Column '%d' is out of range for this model", column);
          break;
        }

      dee_model_get_value (self, iter, column, &value);

      G_VALUE_LCOPY (&value, args, 0, &error);
      if (error)
        {
          g_warning ("Unable to copy value: %s", error);
          g_free (error);

          break;
        }

      g_value_unset (&value);

      column = va_arg (args, gint);
    }
}

static void
dee_versioned_model_get_value (DeeModel     *self,
                              DeeModelIter *iter,
                              guint          column,
                              GValue        *value)
{
  g_critical ("%s not implemented", G_STRFUNC);
}

static gboolean
dee_versioned_model_get_bool (DeeModel      *self,
                             DeeModelIter  *iter,
                             guint          column)
{
  GValue          value = { 0 };

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), FALSE);

  dee_model_get_value (self, iter, column, &value);

  g_return_val_if_fail (G_VALUE_HOLDS_BOOLEAN (&value), FALSE);

  return g_value_get_boolean (&value);
}

static guchar
dee_versioned_model_get_uchar (DeeModel      *self,
                              DeeModelIter  *iter,
                              guint           column)
{
  GValue          value = { 0 };

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), FALSE);

  dee_model_get_value (self, iter, column, &value);

  g_return_val_if_fail (G_VALUE_HOLDS_UCHAR (&value), 0);

  return g_value_get_uchar (&value);
}

static gint
dee_versioned_model_get_int (DeeModel        *self,
                            DeeModelIter    *iter,
                            guint             column)
{
  GValue          value = { 0 };

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), FALSE);

  dee_model_get_value (self, iter, column, &value);

  g_return_val_if_fail (G_VALUE_HOLDS_INT (&value), 0);

  return g_value_get_int (&value);
}

static guint
dee_versioned_model_get_uint (DeeModel      *self,
                             DeeModelIter  *iter,
                             guint           column)
{
  GValue          value = { 0 };

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), FALSE);

  dee_model_get_value (self, iter, column, &value);

  g_return_val_if_fail (G_VALUE_HOLDS_UINT (&value), 0);

  return g_value_get_uint (&value);
}


static gint64
dee_versioned_model_get_int64 (DeeModel      *self,
                              DeeModelIter  *iter,
                              guint           column)
{
  GValue          value = { 0 };

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), FALSE);

  dee_model_get_value (self, iter, column, &value);

  g_return_val_if_fail (G_VALUE_HOLDS_INT64 (&value), 0);

  return g_value_get_int64 (&value);
}


static guint64
dee_versioned_model_get_uint64 (DeeModel      *self,
                               DeeModelIter  *iter,
                               guint           column)
{
  GValue          value = { 0 };

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), FALSE);

  dee_model_get_value (self, iter, column, &value);

  g_return_val_if_fail (G_VALUE_HOLDS_UINT64 (&value), 0);

  return g_value_get_uint64 (&value);
}

static gdouble
dee_versioned_model_get_double (DeeModel      *self,
                               DeeModelIter  *iter,
                               guint                 column)
{
  GValue          value = { 0 };

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), FALSE);

  dee_model_get_value (self, iter, column, &value);

  g_return_val_if_fail (G_VALUE_HOLDS_DOUBLE (&value), 0);

  return g_value_get_double (&value);
}

static const gchar *
dee_versioned_model_get_string (DeeModel      *self,
                               DeeModelIter  *iter,
                               guint          column)
{
  /* Caveat hackor: We can not do a generic impl here because we have to
   *                pass back a const gchar*. Each subclass must implement
   *                this method natively */
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static DeeModelIter*
dee_versioned_model_get_first_iter (DeeModel     *self)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static DeeModelIter*
dee_versioned_model_get_last_iter (DeeModel *self)
{
  DeeModelIter *iter;

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), NULL);

  iter = dee_model_get_first_iter (self);
  while (!dee_model_is_last (self, iter))
    iter = dee_model_next (self, iter);

  return iter;
}

static DeeModelIter*
dee_versioned_model_get_iter_at_row (DeeModel *self, guint row)
{
  DeeModelIter *iter;
  guint         pos;

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), NULL);

  pos = 0;
  iter = dee_model_get_first_iter (self);
  while (!dee_model_is_last (self, iter) && pos < row)
    iter = dee_model_next (self, iter);
    pos++;

  if (dee_model_is_last (self, iter))
    {
      g_critical ("Index %u is out of bounds in model of size %u",
                  row, pos);
    }

  return iter;
}

static DeeModelIter*
dee_versioned_model_next (DeeModel     *self,
                         DeeModelIter *iter)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static DeeModelIter*
dee_versioned_model_prev (DeeModel     *self,
                         DeeModelIter *iter)
{
  g_critical ("%s not implemented", G_STRFUNC);
  return NULL;
}

static gboolean
dee_versioned_model_is_first (DeeModel     *self,
                             DeeModelIter *iter)
{
  DeeModelIter *first;
  
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), FALSE);

  first = dee_model_get_first_iter (self);
  return first == iter;
}

static gboolean
dee_versioned_model_is_last (DeeModel     *self,
                            DeeModelIter *iter)
{
  DeeModelIter *last;
  
  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), FALSE);

  last = dee_model_get_last_iter (self);
  return last == iter;
}

static gint
dee_versioned_model_get_position (DeeModel     *self,
                                 DeeModelIter *iter)
{
  DeeModelIter *_iter;
  gint          pos;

  g_return_val_if_fail (DEE_IS_VERSIONED_MODEL (self), -1);

  pos = 0;
  _iter = dee_model_get_first_iter (self);
  while (!dee_model_is_last (self, iter) && iter != _iter)
    _iter = dee_model_next (self, _iter);
    pos++;

  if (iter == _iter)
    return pos;
  else
    {
      g_critical ("Can not find position of unknown iter %p", iter);
      return -1;
    }
}

static void
dee_versioned_model_freeze_signals (DeeModel     *self)
{
  g_critical ("%s not implemented", G_STRFUNC);
}

static void
dee_versioned_model_thaw_signals (DeeModel     *self)
{
  g_critical ("%s not implemented", G_STRFUNC);
}

static void
dee_versioned_model_model_iface_init (DeeModelIface *iface)
{
  iface->set_column_type      = dee_versioned_model_set_column_type;
  iface->set_n_columns        = dee_versioned_model_set_n_columns;
  iface->get_n_columns        = dee_versioned_model_get_n_columns;
  iface->get_n_rows           = dee_versioned_model_get_n_rows;
  iface->get_column_type      = dee_versioned_model_get_column_type;
  iface->clear                = dee_versioned_model_clear;
  iface->append_valist        = dee_versioned_model_append_valist;
  iface->prepend_valist       = dee_versioned_model_prepend_valist;
  iface->insert_valist        = dee_versioned_model_insert_valist;
  iface->insert_before_valist = dee_versioned_model_insert_before_valist;
  iface->remove               = dee_versioned_model_remove;
  iface->set_valist           = dee_versioned_model_set_valist;
  iface->set_value            = dee_versioned_model_set_value;
  iface->set_value_silently   = dee_versioned_model_set_value_silently;
  iface->get_valist           = dee_versioned_model_get_valist;
  iface->get_value            = dee_versioned_model_get_value;
  iface->get_first_iter       = dee_versioned_model_get_first_iter;
  iface->get_last_iter        = dee_versioned_model_get_last_iter;
  iface->get_iter_at_row      = dee_versioned_model_get_iter_at_row;
  iface->get_bool             = dee_versioned_model_get_bool;
  iface->get_uchar            = dee_versioned_model_get_uchar;
  iface->get_int              = dee_versioned_model_get_int;
  iface->get_uint             = dee_versioned_model_get_uint;
  iface->get_int64            = dee_versioned_model_get_int64;
  iface->get_uint64           = dee_versioned_model_get_uint64;
  iface->get_double           = dee_versioned_model_get_double;
  iface->get_string           = dee_versioned_model_get_string;
  iface->next                 = dee_versioned_model_next;
  iface->prev                 = dee_versioned_model_prev;
  iface->is_first             = dee_versioned_model_is_first;
  iface->is_last              = dee_versioned_model_is_last;
  iface->get_position         = dee_versioned_model_get_position;
  iface->freeze_signals       = dee_versioned_model_freeze_signals;
  iface->thaw_signals         = dee_versioned_model_thaw_signals;
}
