/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by
 *              Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <dee.h>

typedef struct
{
  DeeModel  *model;

} FilterFixture;

static void setup    (FilterFixture *fix, gconstpointer data);
static void teardown (FilterFixture *fix, gconstpointer data);

static void test_empty_orig                    (FilterFixture *fix,
                                                gconstpointer  data);
static void test_append_all                    (FilterFixture *fix,
                                                gconstpointer  data);
static void test_discard_all                   (FilterFixture *fix,
                                                gconstpointer  data);
static void test_discard_all_append_notify     (FilterFixture *fix,
                                                gconstpointer  data);

void
test_filter_model_create_suite (void)
{
#define DOMAIN "/Model/Filter"
  g_test_add (DOMAIN"/EmptyOrig", FilterFixture, 0,
              setup, test_empty_orig, teardown);
  g_test_add (DOMAIN"/AppendAll", FilterFixture, 0,
              setup, test_append_all, teardown);
  g_test_add (DOMAIN"/DiscardAll", FilterFixture, 0,
              setup, test_discard_all, teardown);
  g_test_add (DOMAIN"/DiscardAllAppendNotify", FilterFixture, 0,
              setup, test_discard_all_append_notify, teardown);
}

static void
setup (FilterFixture *fix, gconstpointer data)
{
  fix->model = dee_sequence_model_new (2,
                                       G_TYPE_INT,
                                       G_TYPE_STRING);
  
  dee_model_append (fix->model, 0, 0, 1, "Zero", -1);
  dee_model_append (fix->model, 0, 1, 1, "One", -1);
  dee_model_append (fix->model, 0, 2, 1, "Two", -1);

  g_assert (DEE_IS_SEQUENCE_MODEL (fix->model));
}

static void
teardown (FilterFixture *fix, gconstpointer data)
{
  g_object_unref (fix->model);
  fix->model = NULL;
}

static void
discard_all_model_map (DeeModel       *orig_model,
                       DeeFilterModel *mapped_model,
                       gpointer  user_data)
{
  /* Don't do anything! */
}

static void
append_all_model_map (DeeModel       *orig_model,
                      DeeFilterModel *mapped_model,
                      gpointer  user_data)
{
  DeeModelIter *iter;
  
  iter = dee_model_get_first_iter (orig_model);
  
  while (!dee_model_is_last (orig_model, iter))
    {
      dee_filter_model_append_iter (mapped_model, iter);
      iter = dee_model_next (orig_model, iter);
    }
}

static void
discard_all_model_notify (DeeModel       *orig_model,
                          DeeModelIter   *orig_iter,
                          DeeFilterModel *mapped_model,
                          gpointer        user_data)
{
  /* Do nothing */
}

static void
append_all_model_notify (DeeModel       *orig_model,
                         DeeModelIter   *orig_iter,
                         DeeFilterModel *mapped_model,
                         gpointer        user_data)
{
  /* Always say "Thank you, I am delighted",
   * and append the new row to @mapped_model */
  dee_filter_model_append_iter (mapped_model, orig_iter);
}

static void
signal_counter (DeeModel *model, DeeModelIter *iter, guint *count)
{
  *count = *count + 1;
}

/* Test behaviouor when orig_model is empty */
static void
test_empty_orig (FilterFixture *fix, gconstpointer data)
{
  DeeFilter *filter = g_new0 (DeeFilter, 1);
  filter->map_func = append_all_model_map;
  filter->map_notify = append_all_model_notify;
  filter->user_data = fix;
  
  dee_model_clear (fix->model);
  g_assert_cmpint (0, ==, dee_model_get_n_rows (fix->model));
  
  DeeModel *m = dee_filter_model_new (filter, fix->model);
  g_assert_cmpint (0, ==, dee_model_get_n_rows (fix->model));
  
  DeeModelIter *iter = dee_model_get_first_iter (m);
  g_assert (dee_model_is_first (m, iter));
  g_assert (dee_model_is_last (m, iter));
}

/* A filter model that is a complete copy of the orig */
static void
test_append_all (FilterFixture *fix, gconstpointer data)
{
  DeeFilter filter = { 0 };
  filter.map_func = append_all_model_map;
  filter.map_notify = append_all_model_notify;
  filter.user_data = fix;
  
  DeeModel *m = dee_filter_model_new (&filter, fix->model);
  g_assert_cmpint (3, ==, dee_model_get_n_rows (fix->model));
  
  DeeModelIter *iter = dee_model_get_first_iter (m);
  g_assert (dee_model_is_first (m, iter));
  g_assert (!dee_model_is_last (m, iter));
  g_assert_cmpint (0, ==, dee_model_get_int (m, iter, 0));
  g_assert_cmpstr ("Zero", ==, dee_model_get_string (m, iter, 1));
  
  iter = dee_model_next (m, iter);
  g_assert (!dee_model_is_first (m, iter));
  g_assert (!dee_model_is_last (m, iter));
  g_assert_cmpint (1, ==, dee_model_get_int (m, iter, 0));
  g_assert_cmpstr ("One", ==, dee_model_get_string (m, iter, 1));
  
  iter = dee_model_next (m, iter);
  g_assert (!dee_model_is_first (m, iter));
  g_assert (!dee_model_is_last (m, iter));
  g_assert_cmpint (2, ==, dee_model_get_int (m, iter, 0));
  g_assert_cmpstr ("Two", ==, dee_model_get_string (m, iter, 1));
  
  iter = dee_model_next (m, iter);
  g_assert (!dee_model_is_first (m, iter));
  g_assert (dee_model_is_last (m, iter));
  
  g_assert (dee_model_is_last (m, iter));
}

/* Test a filter that blocks everything */
static void
test_discard_all (FilterFixture *fix, gconstpointer data)
{
  DeeFilter filter = { 0 };
  filter.map_func = discard_all_model_map;
  filter.map_notify = discard_all_model_notify;
  filter.user_data = fix;
  
  DeeModel *m = dee_filter_model_new (&filter, fix->model);
  
  /* Check expected sizes */
  g_return_if_fail (DEE_IS_FILTER_MODEL (m));
  g_assert_cmpint (0, ==, dee_model_get_n_rows (m));
  g_assert_cmpint (3, ==, dee_model_get_n_rows (fix->model));
  
  /* Check that the begin iter is indeed also the end iter */
  DeeModelIter *iter = dee_model_get_first_iter (m);
  g_assert (dee_model_is_first (m, iter));
  g_assert (dee_model_is_last (m, iter));
  
  guint filter_add_count = 0;
  guint orig_add_count = 0;
  g_signal_connect (m, "row-added", G_CALLBACK (signal_counter), &filter_add_count);
  g_signal_connect (fix->model, "row-added", G_CALLBACK (signal_counter), &orig_add_count);
  
  dee_model_append (fix->model, 0, 3, 1, "Three", -1);
  dee_model_append (fix->model, 0, 4, 1, "Four", -1);
  
  /* Updates to the orig model should be ignored by this filter */
  g_assert_cmpint (0, ==, filter_add_count);
  g_assert_cmpint (2, ==, orig_add_count);
  g_assert_cmpint (0, ==, dee_model_get_n_rows (m));
  g_assert_cmpint (5, ==, dee_model_get_n_rows (fix->model));
  
  /* Now add stuff to the filtered model directly. This should work,
   * and should be written through to the orig model as well  */
  dee_model_append (m, 0, 27, 1, "TwentySeven", -1);
  g_assert_cmpint (1, ==, filter_add_count);
  g_assert_cmpint (3, ==, orig_add_count);
  g_assert_cmpint (1, ==, dee_model_get_n_rows (m));
  g_assert_cmpint (6, ==, dee_model_get_n_rows (fix->model));
  
  /* The first (and only) row of 'm' should be at offset 5 in fix->model */
  iter = dee_model_get_first_iter (m);
  g_assert (iter == dee_model_get_iter_at_row (fix->model, 5));
  g_assert_cmpint (27, ==, dee_model_get_int (m, iter, 0));
  g_assert_cmpint (27, ==, dee_model_get_int (fix->model, iter, 0));
  g_assert_cmpstr ("TwentySeven", ==, dee_model_get_string (m, iter, 1));
  g_assert_cmpstr ("TwentySeven", ==, dee_model_get_string (fix->model, iter, 1));
  
  g_object_unref (m);
}

/* Test a filter that blocks everything */
static void
test_discard_all_append_notify (FilterFixture *fix, gconstpointer data)
{
  DeeFilter filter = { 0 };
  filter.map_func = discard_all_model_map;
  filter.map_notify = append_all_model_notify;
  filter.user_data = fix;
  
  DeeModel *m = dee_filter_model_new (&filter, fix->model);
  
  guint filter_add_count = 0;
  guint orig_add_count = 0;
  g_signal_connect (m, "row-added", G_CALLBACK (signal_counter), &filter_add_count);
  g_signal_connect (fix->model, "row-added", G_CALLBACK (signal_counter), &orig_add_count);
  
  dee_model_append (fix->model, 0, 3, 1, "Three", -1);
  dee_model_append (fix->model, 0, 4, 1, "Four", -1);
  
  /* Updates to the orig model should be detected by this filter */
  g_assert_cmpint (2, ==, filter_add_count);
  g_assert_cmpint (2, ==, orig_add_count);
  g_assert_cmpint (2, ==, dee_model_get_n_rows (m));
  g_assert_cmpint (5, ==, dee_model_get_n_rows (fix->model));
  
  /* The first row of 'm' should be at offset 3 in fix->model */
  DeeModelIter *iter = dee_model_get_first_iter (m);
  g_assert (iter == dee_model_get_iter_at_row (fix->model, 3));
  g_assert_cmpint (3, ==, dee_model_get_int (m, iter, 0));
  g_assert_cmpint (3, ==, dee_model_get_int (fix->model, iter, 0));
  g_assert_cmpstr ("Three", ==, dee_model_get_string (m, iter, 1));
  g_assert_cmpstr ("Three", ==, dee_model_get_string (fix->model, iter, 1));
  
  /* The second row of 'm' should be at offset 4 in fix->model */
  iter = dee_model_next (m, iter);
  g_assert (iter == dee_model_get_iter_at_row (fix->model, 4));
  g_assert_cmpint (4, ==, dee_model_get_int (m, iter, 0));
  g_assert_cmpint (4, ==, dee_model_get_int (fix->model, iter, 0));
  g_assert_cmpstr ("Four", ==, dee_model_get_string (m, iter, 1));
  g_assert_cmpstr ("Four", ==, dee_model_get_string (fix->model, iter, 1));
  
  /* Assert that the next iter in 'm is the end iter */
  iter = dee_model_next (m, iter);
  g_assert (dee_model_is_last (m, iter));
  
  g_object_unref (m);
}
