/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by
 *              Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include <dee.h>

typedef struct
{
  DeeModel      *model;
  DeeIndex *index;

} Fixture;

static void setup    (Fixture *fix, gconstpointer data);
static void teardown (Fixture *fix, gconstpointer data);

static void analyze_string0 (DeeModel     *model,
                             DeeModelIter *iter,
                             DeeTermList  *out_terms,
                             gpointer      userdata);

static void
setup (Fixture *fix, gconstpointer data)
{
  DeeAnalyzer analyzer;

  fix->model = dee_sequence_model_new (2, G_TYPE_STRING, G_TYPE_INT);
  g_assert (DEE_IS_MODEL (fix->model));

  analyzer.analyze = analyze_string0;
  analyzer.destroy = NULL;
  analyzer.userdata = fix;

  fix->index = g_object_new (DEE_TYPE_HASH_INDEX,
                             "model", fix->model,
                             "analyzer", &analyzer,
                             NULL);
  g_assert (DEE_IS_INDEX (fix->index));
}

static void
teardown (Fixture *fix, gconstpointer data)
{
  g_object_unref (fix->index);
  g_object_unref (fix->model);
  fix->index = NULL;
  fix->model = NULL;
}

/* Splits string at pos 0 on whitespace */
static void
analyze_string0 (DeeModel     *model,
                 DeeModelIter *iter,
                 DeeTermList  *out_terms,
                 gpointer      userdata)
{
  gchar       **terms;
  const gchar  *s;
  int           i;

  s = dee_model_get_string(model, iter, 0);
  terms = g_strsplit (s, " ", -1);

  for (i = 0; terms[i] != NULL; i++)
    dee_term_list_add_term (out_terms, terms[i]);

  g_strfreev (terms);
}


static void
test_empty (Fixture *fix, gconstpointer data)
{
  g_assert_cmpint (dee_index_get_n_rows (fix->index), ==, 0);
  g_assert_cmpint (dee_index_get_n_terms(fix->index), ==, 0);

  dee_model_clear (fix->model);

  g_assert_cmpint (dee_index_get_n_rows (fix->index), ==, 0);
  g_assert_cmpint (dee_index_get_n_terms(fix->index), ==, 0);
}

static void
test_one (Fixture *fix, gconstpointer data)
{
  DeeModelIter *orig_iter, *iter;
  DeeResultSet *results;

  orig_iter = dee_model_append (fix->model, 0, "Hello world", 1, 27, -1);

  g_assert_cmpint (dee_index_get_n_rows (fix->index), ==, 1);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "Hello"), ==, 1);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "world"), ==, 1);
  g_assert_cmpint (dee_index_get_n_terms(fix->index), ==, 2);

  results = dee_index_lookup (fix->index, "Hello", DEE_TERM_MATCH_EXACT);
  g_assert_cmpint (dee_result_set_get_n_rows (results), ==, 1);

  iter = dee_result_set_next (results);
  g_assert (iter == orig_iter);

  dee_model_clear (fix->model);
  g_assert_cmpint (dee_index_get_n_rows (fix->index), ==, 0);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "Hello"), ==, 0);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "world"), ==, 0);
  g_assert_cmpint (dee_index_get_n_terms(fix->index), ==, 0);

  g_object_unref (results);
}

static void
test_two (Fixture *fix, gconstpointer data)
{
  DeeModelIter *orig_iter1, *orig_iter2, *iter;
  DeeResultSet *results;

  orig_iter1 = dee_model_append (fix->model, 0, "Hello world", 1, 27, -1);
  orig_iter2 = dee_model_append (fix->model, 0, "Dee world", 1, 68, -1);

  g_assert_cmpint (dee_index_get_n_rows (fix->index), ==, 2);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "Hello"), ==, 1);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "world"), ==, 2);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "Dee"), ==, 1);
  g_assert_cmpint (dee_index_get_n_terms(fix->index), ==, 3);

  results = dee_index_lookup (fix->index, "Hello", DEE_TERM_MATCH_EXACT);
  g_assert_cmpint (dee_result_set_get_n_rows(results), ==, 1);

  iter = dee_result_set_next (results);
  g_assert (iter == orig_iter1);
  g_object_unref (results);

  results = dee_index_lookup (fix->index, "world", DEE_TERM_MATCH_EXACT);
  g_assert_cmpint (dee_result_set_get_n_rows(results), ==, 2);
  g_object_unref (results);

  results = dee_index_lookup (fix->index, "Dee", DEE_TERM_MATCH_EXACT);
  g_assert_cmpint (dee_result_set_get_n_rows(results), ==, 1);
  iter = dee_result_set_next (results);
  g_assert (iter == orig_iter2);
  g_object_unref (results);

  dee_model_clear (fix->model);
  g_assert_cmpint (dee_index_get_n_rows (fix->index), ==, 0);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "Hello"), ==, 0);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "world"), ==, 0);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "Dee"), ==, 0);
  g_assert_cmpint (dee_index_get_n_terms(fix->index), ==, 0);

}

static void
test_change (Fixture *fix, gconstpointer data)
{
  DeeModelIter *orig_iter, *iter;
  DeeResultSet *results;

  dee_model_append (fix->model, 0, "Hello world", 1, 27, -1);
  dee_model_append (fix->model, 0, "Hello world", 1, 27, -1);
  dee_model_append (fix->model, 0, "Hello world", 1, 27, -1);
  orig_iter = dee_model_append (fix->model, 0, "xyz", 1, 27, -1);
  dee_model_append (fix->model, 0, "Hello world", 1, 27, -1);

  g_assert_cmpint (dee_index_get_n_rows (fix->index), ==, 5);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "Hello"), ==, 4);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "world"), ==, 4);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "xyz"), ==, 1);
  g_assert_cmpint (dee_index_get_n_terms(fix->index), ==, 3);

  results = dee_index_lookup (fix->index, "xyz", DEE_TERM_MATCH_EXACT);
  g_assert_cmpint (dee_result_set_get_n_rows(results), ==, 1);

  iter = dee_result_set_next(results);
  g_assert (iter == orig_iter);

  /* Change the model and assert that the changes propogate to the index */
  dee_model_set (fix->model, orig_iter, 0, "Hello Yoda", -1);

  g_assert_cmpint (dee_index_get_n_rows (fix->index), ==, 5);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "Hello"), ==, 5);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "world"), ==, 4);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "xyz"), ==, 0);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "Yoda"), ==, 1);
  g_assert_cmpint (dee_index_get_n_terms(fix->index), ==, 3);

  dee_model_clear (fix->model);
  g_assert_cmpint (dee_index_get_n_rows (fix->index), ==, 0);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "Hello"), ==, 0);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "world"), ==, 0);
  g_assert_cmpint (dee_index_get_n_rows_for_term(fix->index, "xyz"), ==, 0);
  g_assert_cmpint (dee_index_get_n_terms(fix->index), ==, 0);

  g_object_unref (results);
}

void
test_hash_index_create_suite (void)
{
#define DOMAIN "/Index/Hash"

  g_test_add (DOMAIN"/Empty", Fixture, 0,
              setup, test_empty, teardown);
  g_test_add (DOMAIN"/One", Fixture, 0,
                setup, test_one, teardown);
  g_test_add (DOMAIN"/Two", Fixture, 0,
                  setup, test_two, teardown);
  g_test_add (DOMAIN"/Change", Fixture, 0,
                    setup, test_change, teardown);
}
