/* prefdialog.cpp
 * functions for a preferences dialog
 *
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 1999-2005 Matthew Hiller, Adam Tee */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <denemo/denemo.h>
#include <denemo/denemo_version.h>
#include "prefops.h"
#include "utils.h"
#include "binreloc.h"

struct callbackdata
{
  struct prefinfo *prefs;
  GtkWidget *lilypathentry;
  GtkWidget *midiplayerentry;
  GtkWidget *audioplayerentry;
  GtkWidget *checkimmediateplayback;
  GtkWidget *checklilystyle;
  GtkWidget *checkclone;
  GtkWidget *checkplayback;
  GtkWidget *checkautosave;
  GtkWidget *autosaveentry;
  GtkWidget *browserentry;
};

#define COLUMN_LOADED (0)
#define COLUMN_PLUGIN (1)

/**
 * Callback to enable/disable the autosave entry when the auto save button is 
 * clicked
 */
static void
toggle_autosave (GtkToggleButton * togglebutton, GtkWidget * autosaveentry)
{
  gtk_widget_set_sensitive (autosaveentry,
									 gtk_toggle_button_get_active (togglebutton));
}

/**
 * Callback to load/unload a plugin
 */
static void
toggle_plugin (GtkCellRendererToggle * cell, gchar * path_str, gpointer data)
{
  GtkListStore *model = (GtkListStore *) data;
  GtkTreePath *path = gtk_tree_path_new_from_string (path_str);
  GtkTreeIter iter;
  gboolean enabled;

  gtk_tree_model_get_iter (GTK_TREE_MODEL (model), &iter, path);
  gtk_tree_model_get (GTK_TREE_MODEL (model), &iter, COLUMN_LOADED, &enabled,
							 -1);
  enabled ^= 1;
  if (enabled)
    {
      g_warning ("TODO: Load plugin");
    }
  else
    {
      g_warning ("TODO: Unload plugin\n");
    }
  
  gtk_list_store_set (model, &iter, COLUMN_LOADED, enabled, -1);
  
  gtk_tree_path_free (path);
}

static void
set_preferences (struct callbackdata *cbdata)
{
  struct prefinfo *prefs = cbdata->prefs;
  static GString *localrc = NULL;
  FILE *fp;
  
  g_string_assign (prefs->lilypath,
						 gtk_entry_get_text (GTK_ENTRY (cbdata->lilypathentry)));
  g_string_assign (prefs->midiplayer,
						 gtk_entry_get_text (GTK_ENTRY (cbdata->midiplayerentry)));
  g_string_assign (prefs->audioplayer,
						 gtk_entry_get_text (GTK_ENTRY (cbdata->audioplayerentry)));
  g_string_assign (prefs->browser,
						 gtk_entry_get_text (GTK_ENTRY (cbdata->browserentry)));
  prefs->immediateplayback =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
											 (cbdata->checkimmediateplayback));
  prefs->lilyentrystyle =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cbdata->checklilystyle));
  prefs->createclones =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cbdata->checkclone));

  prefs->playbackoutput =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cbdata->checkplayback));

  prefs->autosave =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cbdata->checkautosave));
  prefs->autosave_timeout =
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON
				      (cbdata->autosaveentry));
  
  g_print ("Timeout %d \n", prefs->autosave_timeout);
  /* Now write it all to denemorc */
  if (!localrc)
    {
      localrc = g_string_new (locatedotdenemo ());
      g_string_append (localrc, "/denemorc");
    }
  if ((fp = fopen (localrc->str, "w")))
    {
      fprintf (fp, "lilypath = %s\n", prefs->lilypath->str);
      fprintf (fp, "midiplayer = %s\n", prefs->midiplayer->str);
      fprintf (fp, "audioplayer = %s\n", prefs->audioplayer->str);
      fprintf (fp, "immediateplayback = %s\n",
	       prefs->immediateplayback ? "true" : "false");
      fprintf (fp, "lilyentrystyle = %s\n",
	       prefs->lilyentrystyle ? "true" : "false");
      fprintf (fp, "createclones = %s\n",
	       prefs->createclones ? "true" : "false");
      fprintf (fp, "playbackoutput = %s\n",
	       prefs->playbackoutput ? "true" : "false");
      fprintf (fp, "autosave = %s\n", prefs->autosave ? "true" : "false");
      fprintf (fp, "timeout = %d\n", prefs->autosave_timeout);
		fprintf (fp, "browser = %s\n", prefs->browser->str);
      fclose (fp);
    }
  else
    g_warning (_("set_preferences : error opening %s : %s"), localrc->str,
	       g_strerror (errno));
}

void
preferences_change (GtkAction * action, struct scoreinfo *si)
{
  GtkWidget *dialog;
  GtkWidget *label;
  GtkWidget *table;
  GtkWidget *lilypathentry;
  GtkWidget *midiplayerentry;
  GtkWidget *audioplayerentry;
  GtkWidget *browserentry;
  GtkWidget *checkimmediateplayback;
  GtkWidget *checklilystyle;
  GtkWidget *checkclone;
  GtkWidget *checkplayback;
  GtkWidget *checkautosave;
  GtkWidget *autosaveentry;
  GtkWidget *notebook;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkListStore *list_store;
  GtkWidget *tree;
  GtkTreeIter iter;
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  static struct callbackdata cbdata;

  g_assert (si != NULL);

  dialog = gtk_dialog_new_with_buttons (_("Preferences - Denemo"),
					GTK_WINDOW (si->window),
					(GtkDialogFlags) (GTK_DIALOG_MODAL |
							  GTK_DIALOG_DESTROY_WITH_PARENT),
					GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					GTK_STOCK_CANCEL, GTK_STOCK_CANCEL,
					NULL);

  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

  notebook = gtk_notebook_new ();
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), notebook, TRUE,
		      TRUE, 0);

  /*
   * Note entry settings
   */
  table = gtk_table_new (5, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 12);
  gtk_table_set_row_spacings (GTK_TABLE (table), 8);
  gtk_table_set_col_spacings (GTK_TABLE (table), 8);

  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, NULL);
  gtk_notebook_set_tab_label_text (GTK_NOTEBOOK (notebook), table,
				   _("Entry"));


  checkimmediateplayback =
    gtk_check_button_new_with_label (_("Play back entered notes immediately"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkimmediateplayback),
				si->prefs->immediateplayback);
  gtk_table_attach (GTK_TABLE (table), checkimmediateplayback, 0, 2, 0, 1,
		    (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0,
		    0);

  checklilystyle =
    gtk_check_button_new_with_label (_("Use lilypond-style note entry"));
  /* Comment that out when the feature is added */
  gtk_widget_set_sensitive (checklilystyle, FALSE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checklilystyle),
				si->prefs->lilyentrystyle);
  gtk_table_attach (GTK_TABLE (table), checklilystyle, 0, 2, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  checkclone =
    gtk_check_button_new_with_label (_("Clone chords when tying them"));
  gtk_widget_set_sensitive (checkclone, FALSE);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkclone),
				si->prefs->createclones);
  gtk_table_attach (GTK_TABLE (table), checkclone, 0, 2, 2, 3,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_table_attach (GTK_TABLE (table), hbox, 0, 2, 4, 5,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  checkautosave = gtk_check_button_new_with_label (_("Autosave every"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkautosave),
				si->prefs->autosave);
  gtk_box_pack_start (GTK_BOX (hbox), checkautosave, FALSE, FALSE, 0);

  autosaveentry = gtk_spin_button_new_with_range (1, 50, 1.0);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (autosaveentry),
			     si->prefs->autosave_timeout);
  gtk_widget_set_sensitive (autosaveentry, si->prefs->autosave);
  gtk_box_pack_start (GTK_BOX (hbox), autosaveentry, FALSE, FALSE, 0);

  label = gtk_label_new (_("minute(s)"));
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
  g_signal_connect (GTK_OBJECT (checkautosave),
		    "toggled", G_CALLBACK (toggle_autosave), autosaveentry);

  
  


  /*
   * Playback settings
   */
  table = gtk_table_new (4, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 12);
  gtk_table_set_row_spacings (GTK_TABLE (table), 8);
  gtk_table_set_col_spacings (GTK_TABLE (table), 8);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, NULL);
  gtk_notebook_set_tab_label_text (GTK_NOTEBOOK (notebook), table,
				   _("Playback"));


  label = gtk_label_new (_("Path to Lilypond:"));
  gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  lilypathentry = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (lilypathentry), si->prefs->lilypath->str);
  gtk_table_attach (GTK_TABLE (table), lilypathentry, 1, 2, 0, 1,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  label = gtk_label_new (_("External midi player:"));
  gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  midiplayerentry = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (midiplayerentry),
		      si->prefs->midiplayer->str);
  gtk_table_attach (GTK_TABLE (table), midiplayerentry, 1, 2, 1, 2,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  label = gtk_label_new (_("External audio player:"));
  gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  audioplayerentry = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (audioplayerentry),
		      si->prefs->audioplayer->str);
  gtk_table_attach (GTK_TABLE (table), audioplayerentry, 1, 2, 2, 3,
		    (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  checkplayback = gtk_check_button_new_with_label (_("Use Denemo midi export"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (checkplayback),
				si->prefs->playbackoutput);
  gtk_table_attach (GTK_TABLE (table), checkplayback, 0, 2, 3, 4,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  /*
   * Plugins settings
   */
  vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, NULL);
  gtk_notebook_set_tab_label_text (GTK_NOTEBOOK (notebook), vbox,
				   _("Plugins"));

  list_store = gtk_list_store_new (2, G_TYPE_BOOLEAN, G_TYPE_STRING);

  if (si->plugins)
    {
      for (GList * tmp = si->plugins; tmp; tmp = tmp->next)
	{
	  gtk_list_store_append (list_store, &iter);
	  gtk_list_store_set (list_store, &iter,
			      COLUMN_LOADED, TRUE,
			      COLUMN_PLUGIN,
			      (gchar *) ((PluginData *) tmp->data)->title,
			      -1);

	}
    }

  GList *plugins = NULL;
  plugins = get_plugins_list(plugins);
  if (plugins)
  {
	  g_print("list of plugins is populated\n");
	  for(GList *tmp=plugins; tmp; tmp=tmp->next)
	  {
		  gtk_list_store_append (list_store, &iter);
		  gtk_list_store_set (list_store, &iter,
			      COLUMN_LOADED, FALSE,
			      COLUMN_PLUGIN,
			      (gchar *) tmp->data,
			      -1);
	  }
  }
  
  gtk_list_store_append (list_store, &iter);
  gtk_list_store_set (list_store, &iter,
		      COLUMN_LOADED, FALSE,
		      COLUMN_PLUGIN, "Dummy plugin 1", -1);

  gtk_list_store_append (list_store, &iter);
  gtk_list_store_set (list_store, &iter,
		      COLUMN_LOADED, TRUE,
		      COLUMN_PLUGIN, "Dummy plugin 2", -1);

  tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (list_store));
  gtk_box_pack_start (GTK_BOX (vbox), tree, TRUE, TRUE, 0);
  g_object_unref (G_OBJECT (list_store));

  renderer = gtk_cell_renderer_toggle_new ();
  g_signal_connect (renderer, "toggled", G_CALLBACK (toggle_plugin),
		    list_store);
  column =
    gtk_tree_view_column_new_with_attributes ("Enabled", renderer, "active",
					      COLUMN_LOADED, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);

  column = gtk_tree_view_column_new_with_attributes ("Plugin",
						     gtk_cell_renderer_text_new
						     (), "text",
						     COLUMN_PLUGIN, NULL);

  gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);


/*
   * Help settings
   */
  table = gtk_table_new (1, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 12);
  gtk_table_set_row_spacings (GTK_TABLE (table), 8);
  gtk_table_set_col_spacings (GTK_TABLE (table), 8);
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, NULL);
  gtk_notebook_set_tab_label_text (GTK_NOTEBOOK (notebook), table,
				   _("Help Settings"));


  label = gtk_label_new (_("Help Browser:"));
  gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);
  gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
						  (GtkAttachOptions) (GTK_FILL),
						  (GtkAttachOptions) (0), 0, 0);
  browserentry = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (browserentry), si->prefs->browser->str);
  gtk_table_attach (GTK_TABLE (table), browserentry, 1, 2, 0, 1,
						  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
						  (GtkAttachOptions) (0), 0, 0);
  
  

  /* Set up the callback data */
  cbdata.prefs = si->prefs;
  cbdata.lilypathentry = lilypathentry;
  cbdata.midiplayerentry = midiplayerentry;
  cbdata.audioplayerentry = audioplayerentry;
  cbdata.checkimmediateplayback = checkimmediateplayback;
  cbdata.checklilystyle = checklilystyle;
  cbdata.checkclone = checkclone;
  cbdata.checkplayback = checkplayback;
  cbdata.checkautosave = checkautosave;
  cbdata.autosaveentry = autosaveentry;
  cbdata.browserentry = browserentry;

  gtk_widget_grab_focus (lilypathentry);
  gtk_widget_show_all (dialog);
  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      set_preferences (&cbdata);
    }
  gtk_widget_destroy (dialog);
}

GList *get_plugins_list(GList *plugins)
{
	
	DIR *dir;
   struct dirent *dirent;
   char *name;
   gchar *dirname;
	dirname = gbr_find_lib_dir(PACKAGE_PLUGIN_DIR);
   //dirname = g_strconcat(dirname, "/denemo/", NULL);
	g_print("Directory %s\n",dirname);
   dir = opendir (dirname);
	
   if (!dir)
     return NULL;

   while ((dirent = readdir (dir)) != NULL)
    {
		g_print("Filename is %s\n", dirent->d_name);
      if ((name = stripname (dirent->d_name)) != NULL)
			plugins = g_list_append(plugins, name);
	 }
	 
	 g_free(dirname);
	 return(plugins);
}
