/* articulations.c
 * Implements articulation markings which are not notes 
 *
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * A Tee  (c) 2000-2005
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "chordops.h"
#include "calculatepositions.h"
#include "commandfuncs.h"
#include "contexts.h"
#include "dialogs.h"
#include "draw.h"
#include "objops.h"
#include "staffops.h"
#include "utils.h"
#include "articulations.h"


/**
 * Insert ornament into list if not present 
 * If the ornament is present then remove as action has been
 * performed twice.
 *
 * @param orn  ornament to insert into the ornament list
 * @param list the ornament list
 * @return the new ornament list
 */
GList* insert_ornament_list(enum ornament orn, GList *list)
{
   /*
	 * Find the ornament in the list and remove
	 */
	GList *tmp;
	for(tmp = list; tmp; tmp = tmp->next)
		{
			if(*(enum ornament *)tmp->data == (enum ornament)orn)
				return (g_list_remove(list,tmp->data));
		}
	/*
	 * Insert new ornament into the list.
	 */
	enum ornament *tmporn = (enum ornament *)g_malloc0(sizeof(enum ornament));
	*tmporn = orn;
	list = g_list_append(list, tmporn);
#ifdef DEBUG
	g_print("Inserted ORNAMENT %d into list \n", *tmporn);
#endif
		
	return list;
}

/**
 * Set the relevant articulation flag on the 
 * current mudelaobject.
 *
 * @param string textual description of the articulation
 * @param obj		 the object to add the ornament to
 * @return 0 on success or -1 on failure
 */
int
set_articulation (gchar * string, mudelaobject * obj)
{
  
  
  if (obj->type == CHORD && ((chord *) obj->object)->tones)
    {
	  
      if (!strcmp (string, "staccato")) 
		  {
		    
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(STACCATO,
					  					    ((chord *)obj->object)->ornamentlist);
			 
		  }
      else if (!strcmp (string, "tenuto"))
		  {
				
			  ((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(TENUTO,
					  					    ((chord *)obj->object)->ornamentlist);
			  
		  }
      else if (!strcmp (string, "staccatissimo")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(STACCATISSIMO,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "accent")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(D_ACCENT,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "marcato")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(MARCATO,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "fermata")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(FERMATA,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "trill")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(TRILL,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "turn")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(TURN,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "mordent")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(MORDENT,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "reverse turn")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(REVERSETURN,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      /* String specific articulations */
      else if (!strcmp (string, "up bow")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(UBOW,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "down bow")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(DBOW,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      /*organ articulations */
      else if (!strcmp (string, "rheel")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(RHEEL,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "lheel")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(LHEEL,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "rtoe")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(RTOE,
					  					    ((chord *)obj->object)->ornamentlist);
		}
      else if (!strcmp (string, "ltoe")) {
	
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(LTOE,
					  					    ((chord *)obj->object)->ornamentlist);
		}
		else if(!strcmp (string, "coda")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(CODA,
					  					    ((chord *)obj->object)->ornamentlist);
	   }
		else if(!strcmp (string, "arpeggio")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(D_ARPEGGIO,
					  					    ((chord *)obj->object)->ornamentlist);
		}
		else if(!strcmp (string, "flagoelet")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(FLAGEOLET,
					  					    ((chord *)obj->object)->ornamentlist);	
		}
		else if(!strcmp (string, "open")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(OPEN,
					  					    ((chord *)obj->object)->ornamentlist);
		}
		else if(!strcmp (string, "prall")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(PRALL,
					  					    ((chord *)obj->object)->ornamentlist);
		}
		else if(!strcmp (string, "prallmordent")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(PRALLMORDENT,
					  					    ((chord *)obj->object)->ornamentlist);
		}
		else if(!strcmp (string, "prallprall")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(PRALLPRALL,
					  					    ((chord *)obj->object)->ornamentlist);
		}
		else if(!strcmp (string, "segno")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(SEGNO,
					  					    ((chord *)obj->object)->ornamentlist);
		}
		else if(!strcmp (string, "stopped")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(STOPPED,
					  					    ((chord *)obj->object)->ornamentlist);
		}
		else if(!strcmp (string, "thumb")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(THUMB,
					  					    ((chord *)obj->object)->ornamentlist);
		}
		else if(!strcmp (string, "upprall")) {
			((chord *)obj->object)->ornamentlist = 
				  insert_ornament_list(UPPRALL,
					  					    ((chord *)obj->object)->ornamentlist);
		
		}
    }
  else
    return -1;
  
  
  return 0;
}

/**
 * Insert articulation callback
 * @param widget widget which calls the callback
 * @param si the scoreinfo structure
 * @return none
 */
static void
insert_artic_cb (GtkWidget * widget, struct scoreinfo *si)
{
  mudelaobject *mudelaobj;
  gchar *tmp;
  gchar *articulation;

  g_assert (si != NULL);

  tmp =
    g_strdup ((gchar *)
	      g_object_get_data (G_OBJECT (widget), "articulation"));

  // Convert stock name to articulation name
  articulation = tmp;
  if (g_str_has_prefix (tmp, "denemo-"))
  {
    char *c;
    
    articulation += 7;
    for (c = articulation; *c != '\0'; c++)
		{
	  	if (*c == '-')
	    {
	      *c = ' ';
	    }
		}
  }

  mudelaobj = (mudelaobject *)
    (si->currentobject ? si->currentobject->data : NULL);

  if (mudelaobj)
  {
    set_articulation (articulation, mudelaobj);
  }

  gtk_widget_draw (si->scorearea, NULL);
  gtk_widget_grab_focus (si->window);
  g_free (tmp);
}

/**
 * Creates button for the articulation palette 
 * 
 * @param stock_id	textual description of articulation 
 * @param tips			tooltip for articulation
 * @param table			table to insert button into
 * @param col				column in table to insert button into
 * @param row				row in table to insert button into
 * @param si				the score info structure
 *
 */
static GtkWidget *
create_articulation_button (const gchar * stock_id, GtkTooltips * tips,
			    GtkWidget * table, gint col, gint row,
			    struct scoreinfo *si)
{
  GtkWidget *button;
  GtkStockItem stock;
  GtkWidget *image;

  button = gtk_button_new ();
  //gtk_button_set_focus_on_click(GTK_BUTTON(button), FALSE);
  g_object_set_data (G_OBJECT (button), "articulation", (void *) stock_id);

  if (gtk_stock_lookup (stock_id, &stock))
    {
      gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), button, stock.label, NULL);
      image = gtk_image_new_from_stock (stock_id, GTK_ICON_SIZE_BUTTON);
      gtk_container_add (GTK_CONTAINER (button), image);
    }

  gtk_table_attach (GTK_TABLE (table), button, col, col + 1, row, row + 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);

  g_signal_connect (button, "clicked", G_CALLBACK (insert_artic_cb), si);

  return button;
}

/**
 * Create the articulation selection widget.
 * This widget can be put into a dialog or popup.
 * @param si the scoreinfo structure
 * @return the vbox
 */
static GtkWidget *
create_articulation_widget (struct scoreinfo *si)
{
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *table;
  GtkWidget *spacer;
  GtkTooltips *tips;

  tips = gtk_tooltips_new ();

  vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 12);

  label = gtk_label_new (_("<b>General</b>"));
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  table = gtk_table_new (4, 7, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, TRUE, 0);

  spacer = gtk_label_new ("");
  gtk_table_attach (GTK_TABLE (table), spacer, 0, 1, 0, 2,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_widget_set_size_request (spacer, 8, -1);

  create_articulation_button ("denemo-staccato", tips, table, 1, 0, si);
  create_articulation_button ("denemo-staccatissimo", tips, table, 2, 0, si);
  create_articulation_button ("denemo-marcato", tips, table, 3, 0, si);
  create_articulation_button ("denemo-accent", tips, table, 4, 0, si);
  create_articulation_button ("denemo-fermata", tips, table, 5, 0, si);
  create_articulation_button ("denemo-coda", tips, table, 6, 0, si);
  
  create_articulation_button ("denemo-tenuto", tips, table, 1, 1, si);
  create_articulation_button ("denemo-turn", tips, table, 2, 1, si);
  create_articulation_button ("denemo-reverse-turn", tips, table, 3, 1, si);
  create_articulation_button ("denemo-trill", tips, table, 4, 1, si);
  create_articulation_button ("denemo-mordent", tips, table, 5, 1, si);
  create_articulation_button ("denemo-prall", tips, table, 6, 1, si);
  
  create_articulation_button ("denemo-flagoelet", tips, table, 1, 2, si);
  create_articulation_button ("denemo-open", tips, table, 2, 2, si);
  create_articulation_button ("denemo-prallmordent", tips, table, 3, 2, si);
  create_articulation_button ("denemo-prallprall", tips, table, 4, 2, si);
  create_articulation_button ("denemo-segno", tips, table, 5, 2, si);
  create_articulation_button ("denemo-stopped", tips, table, 6, 2, si);
  
  create_articulation_button ("denemo-thumb", tips, table, 1, 3, si);
  create_articulation_button ("denemo-upprall", tips, table, 2, 3, si);
  create_articulation_button ("denemo-arpeggio", tips, table, 3, 3, si);


  label = gtk_label_new (_("<b>String</b>"));
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  table = gtk_table_new (1, 3, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, TRUE, 0);

  spacer = gtk_label_new ("");
  gtk_table_attach (GTK_TABLE (table), spacer, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_widget_set_size_request (spacer, 8, -1);

  create_articulation_button ("denemo-up-bow", tips, table, 1, 0, si);
  create_articulation_button ("denemo-down-bow", tips, table, 2, 0, si);

  label = gtk_label_new (_("<b>Organ</b>"));
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  table = gtk_table_new (1, 6, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, TRUE, 0);

  spacer = gtk_label_new ("");
  gtk_table_attach (GTK_TABLE (table), spacer, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_widget_set_size_request (spacer, 12, -1);

  create_articulation_button ("denemo-rheel", tips, table, 1, 0, si);
  create_articulation_button ("denemo-lheel", tips, table, 2, 0, si);
  create_articulation_button ("denemo-rtoe", tips, table, 3, 0, si);
  create_articulation_button ("denemo-ltoe", tips, table, 4, 0, si);

#if 0
  label = gtk_label_new (_("<b>Brass</b>"));
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  table = gtk_table_new (1, 6, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, TRUE, 0);

  spacer = gtk_label_new ("");
  gtk_table_attach (GTK_TABLE (table), spacer, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_widget_set_size_request (spacer, 8, -1);

  label = gtk_label_new (_("<b>Woodwind</b>"));
  gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  table = gtk_table_new (1, 6, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, TRUE, 0);

  spacer = gtk_label_new ("");
  gtk_table_attach (GTK_TABLE (table), spacer, 0, 1, 0, 1,
		    (GtkAttachOptions) (GTK_FILL),
		    (GtkAttachOptions) (0), 0, 0);
  gtk_widget_set_size_request (spacer, 8, -1);

#endif

  return vbox;
}


/**
 * Callback to hide articulation palette
 */
static gboolean
hide_palette (GtkWidget * widget, GdkEvent * event, struct scoreinfo *si)
{
  GtkWidget *toggle_palette;

  if (si->articulation_palette)
    {
      toggle_palette = gtk_ui_manager_get_widget (si->ui_manager,
						  "/MainMenu/ViewMenu/ToggleArticulationPalette");
      gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (toggle_palette),
				      FALSE);
    }
  return TRUE;
}

/**
 * Toggle articulation palette
 * @param action the gtk action emitted by menu item
 * @param si the scoreinfo structure
 * @return none
 */
void
toggle_articulation_palette (GtkAction * action, struct scoreinfo *si)
{
  GtkWidget *window;
  GtkWidget *vbox;

  g_assert (si != NULL);

  if (si->articulation_palette)
    {
      if (GTK_WIDGET_VISIBLE (si->articulation_palette) )
	{
	  gtk_widget_hide (si->articulation_palette);
	}
	else if(si->prefs->articulation_palette == FALSE && GTK_WIDGET_VISIBLE (si->articulation_palette))
	{
		 gtk_widget_hide (si->articulation_palette);
	}
      else 
	{
	  gtk_widget_show (si->articulation_palette);
	}
      return;
    }

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  si->articulation_palette = window;
  gtk_window_set_title (GTK_WINDOW (window), _("Articulation"));
  gtk_window_set_focus_on_map (GTK_WINDOW (window), FALSE);
  //GTK_WIDGET_UNSET_FLAGS(window, GTK_CAN_FOCUS);
  gtk_window_set_transient_for (GTK_WINDOW (window), GTK_WINDOW (si->window));
  gtk_window_set_type_hint (GTK_WINDOW (window),
			    GDK_WINDOW_TYPE_HINT_UTILITY);
  gtk_window_set_resizable (GTK_WINDOW (window), FALSE);
  gtk_window_set_role (GTK_WINDOW (window), "articulation-toolbox");

  vbox = create_articulation_widget (si);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);
  gtk_container_add (GTK_CONTAINER (window), vbox);

  g_signal_connect (window, "delete-event", G_CALLBACK (hide_palette), si);

  gtk_widget_show_all (window);
}
