/* view.cpp
 * Functions to create a new view of the main window
 *
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 2003-2005  Adam Tee
 */
#include "view.h"
#include "bookmarks.h"
#include "lyparserfuncs.h"
#include "dialogs.h"
#include "binreloc.h"
#include <stdlib.h>
#include "scorewizard.h"
#include "playback.h"
#include "alsaplayback.h"
#include "midiseq.h"

#if GTK_MAJOR_VERSION > 1
#include <gtk/gtkaccelgroup.h>
#endif

#define INITIAL_WIDTH 1000
#define INITIAL_HEIGHT 500

extern GList *displays;
extern midi_seq *sq; /* global denemo sequencer FIXME: should not be global */

/**
 * close the application and shut down the sequencer if open
 *
 */
void
closeit (GtkWidget * widget, gpointer data)
{
#ifdef HAVEALSA
	midi_seq_delete(sq);
#endif
  gtk_main_quit ();
}

/**
 * close application saving the current file before exiting 
 * only called when the gtk_widget_destroy signal has been 
 * recieved. Close also the sequencer object.
 */
/* currently this function should be modified because if two windows are open and both are unedited it will not close them
 * unless users edits one or both
 * If the there is only one display unedited the user gets prompted.
 */
void
close_application (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  	  
  scoreinfo* si = (scoreinfo *) data;
  if (g_list_length (displays) == 1)
    {
      if ((si->haschanged) ? (confirmbox (si)) : 1) 
      {
  		closeit(NULL, si);
      }
      else
      {
          	g_printf("Don't quit now!\n");
      }
      
    }
  else if (g_list_length (displays) > 1)
    {
      if ((si->haschanged) ? (confirmbox (si)) : 1)
      {
          	close_view (NULL, si);
      }
      else
      {
      	  	g_printf("Don't quit now!\n");
      }
    }
}

/**
* Wrapper function to close application when the quit
* menu item has been used
* 
*
*/
void
closewrapper (GtkAction * action, gpointer data)
{
  GList *display = NULL;
  struct scoreinfo *si;
  
  si = (struct scoreinfo *) data;
  //Iterate through the views list if more than one is open
  if (g_list_length (displays) > 1)
    {
      for (display = displays; display != NULL;display = g_list_next (display))
        {
          si = (struct scoreinfo *) display->data;
          if (si->haschanged)
            {
              if (confirmbox (si))
                {
                  close_view (NULL, si);
                }
            }
          else
            {
              close_view (NULL, si);
            }
        }
    }
  //final view so propmt to save if it has changed
  //otherwise exit app
  if (si->haschanged)
    {
      if (confirmbox (si))
        {
          closeit (NULL, si);
        }
    }
  else
    {
      closeit (NULL, si);
    }
  writeHistory(si->prefs);
}

/**
 * Callback for NewWindow calls newview to actually create the view
 *
 */
void
createview (GtkAction * action, gpointer data)
{
  newview ();
}

/**
 * Open in New Window callback 
 * Creates new view then opens file in the view
 */
void
openinnew (GtkAction * action, gpointer data)
{
  struct scoreinfo *si = (struct scoreinfo *) g_list_last (displays)->data;
  newview ();
  if (si->haschanged)
    {
      if (confirmbox (si))
        {
          file_open (NULL, si);
        }
    }
  else
    {
      file_open (NULL, si);
    }
}

/**
 * Close a specific view
 * Do not close the sequencer
 */
void
close_view (GtkWidget * widget, gpointer data)
{
  displays = g_list_remove (displays, data);
  gtk_widget_destroy (((struct scoreinfo *) data)->window);
  g_free ((struct scoreinfo *) data);
}

/**
 * Close callback 
 * Delete score if one display open
 * otherwise close window
 */
void
closeview (GtkAction * action, gpointer data)
{
  struct scoreinfo *si = (struct scoreinfo *) data;
  if (g_list_length (displays) == 1)
    {
      if (confirmbox (si))
        {
          deletescore (NULL, si);
        }
    }
  else if (si->haschanged)
    {
      if (confirmbox (si))
        {
          close_view (NULL, si);
        }
    }
  else
    {
      close_view (NULL, data);
    }
}


/*
 * Note palette callbacks
 */

/**
 * Note palette callback inserts whole note into score 
 * 
 */
static void
insert_chord_0key_cb (GtkAction * action, struct scoreinfo *si)
{
  insert_chord_0key (si);
}

/**
 * Note palette callback insert half note into score
 *
 */
static void
insert_chord_1key_cb (GtkAction * action, struct scoreinfo *si)
{
  insert_chord_1key (si);
}

/**
 * Note palette callback insert quarter note into score
 *
 */
static void
insert_chord_2key_cb (GtkAction * action, struct scoreinfo *si)
{
  insert_chord_2key (si);
}

/**
 * Note palette callback insert eigth note into the score
 *
 */
static void
insert_chord_3key_cb (GtkAction * action, struct scoreinfo *si)
{
  insert_chord_3key (si);
}

/**
 * Note palette callback insert sixteenth note into the score
 *
 */
static void
insert_chord_4key_cb (GtkAction * action, struct scoreinfo *si)
{
  insert_chord_4key (si);
}

/**
 * Note palette callback insert whole note rest into the score
 *
 */
static void
insert_rest_0key_cb (GtkAction * action, struct scoreinfo *si)
{
  insert_rest_0key (si);
}

/**
 * Note palette callback insert half note rest into the score
 *
 */
static void
insert_rest_1key_cb (GtkAction * action, struct scoreinfo *si)
{
  insert_rest_1key (si);
}

/**
 * Note palette callback insert quarter note rest into the score
 *
 */

static void
insert_rest_2key_cb (GtkAction * action, struct scoreinfo *si)
{
  insert_rest_2key (si);
}

/**
 * Note palette callback insert eigth note rest into the score
 *
 */
static void
insert_rest_3key_cb (GtkAction * action, struct scoreinfo *si)
{
  insert_rest_3key (si);
}

/**
 * Note palette callback insert sixteenth note rest into the score
 *
 */
static void
insert_rest_4key_cb (GtkAction * action, struct scoreinfo *si)
{
  insert_rest_4key (si);
}

/**
 * Menu entries with shortcut keys, tooltips, and callback functions
 */
GtkActionEntry menu_entries[] = {
                                  {"FileMenu", NULL, N_("_File")},
                                  {"New", GTK_STOCK_NEW, N_("_New"), "<control>n", "Create a new file",
                                   G_CALLBACK (file_newwrapper)},
                                  {"OpenTemplate", NULL, N_("New from Template"), NULL, "Open New Score From Template",
                                   G_CALLBACK (file_open_template_wrapper)},
                                  {"NewWindow", NULL, N_("New Window"), NULL, "Create a new window",
                                   G_CALLBACK (createview)},
                                  {"Open", GTK_STOCK_OPEN, N_("_Open"), "<control>o", "Open a file",
                                   G_CALLBACK (file_openwrapper)},
                                  {"OpenNewWindow", NULL, N_("Open in New Window"), NULL,
                                   "Open in a New Window", G_CALLBACK (openinnew)},
                                  {"Save", GTK_STOCK_SAVE, N_("_Save"), "<control>s", "Save File",
                                   G_CALLBACK (file_savewrapper)},
                                  {"SaveAs", GTK_STOCK_SAVE_AS, N_("Save _As"), NULL, "Save File As..",
                                   G_CALLBACK (file_saveaswrapper)},
                                  {"ExportPDF", NULL, N_("Export PDF"), NULL, "Export PDF...",
                                   G_CALLBACK (export_pdf_action)},
#ifdef DEBUG
                                  {"ConfigureScore", NULL, N_("Score Configuration"), "<control>w", "Score Configuration",
                                   G_CALLBACK(scorewizard)},
#endif			  
                                  {"Print", GTK_STOCK_PRINT, N_("_Print"), "<control>p", "Print Score",
                                   G_CALLBACK (print)},
                                  {"Close", GTK_STOCK_CLOSE, N_("_Close"), NULL, "Close View",
                                   G_CALLBACK (closeview)},
                                  {"Quit", GTK_STOCK_QUIT, N_("_Quit"), "<control>q", "Quit Denemo",
                                   G_CALLBACK (closewrapper)},
                                  {"EditMenu", NULL, N_("_Edit")},
                                  {"Undo", GTK_STOCK_UNDO, N_("_Undo"), "<control>z", "Undo",
                                   G_CALLBACK (undowrapper)},
                                  {"Redo", GTK_STOCK_REDO, N_("_Redo"), "<control>y", "Redo",
                                   G_CALLBACK (redowrapper)},
                                  {"Copy", GTK_STOCK_COPY, N_("_Copy"), "<control>c", "Copy",
                                   G_CALLBACK (copywrapper)},
                                  {"Cut", GTK_STOCK_CUT, N_("Cu_t"), "<control>x", "Cut",
                                   G_CALLBACK (cutwrapper)},
                                  {"Paste", GTK_STOCK_PASTE, N_("_Paste"), "<control>v", "Paste",
                                   G_CALLBACK (pastewrapper)},
                                  {"ScoreProperties", GTK_STOCK_PROPERTIES, N_("_Score properties"), NULL,
                                   NULL, G_CALLBACK (score_properties_dialog)},
                                  {"SaveSelection", NULL, N_("Save Selection"), NULL, "Save Selection",
                                   G_CALLBACK (saveselwrapper)},
                                  {"Preferences", GTK_STOCK_PREFERENCES, N_("Pr_eferences"), NULL,
                                   "Preferences", G_CALLBACK (preferences_change)},
                                  //  {"Keyboard", NULL, N_("_Keyboard"), NULL, NULL,
                                  //   G_CALLBACK (configure_keyboard_dialog)},
                                  {"Keyboard", NULL, N_("Set Keybindings"), NULL, NULL,
                                   G_CALLBACK (configure_keyboard_dialog_OLD)},
                                  {"LoadPlugins", NULL, N_("Load Plugins"), NULL, "Load Plugins",
                                   G_CALLBACK (load_plugin)},
                                  {"UnloadPlugins", NULL, N_("Unload Plugins"), NULL, "Unload Plugins",
                                   G_CALLBACK (unloadplugins)},
                                  {"ListPlugins", NULL, N_("List Plugins"), NULL, "List Plugins",
                                   G_CALLBACK (list_loaded_plugins)},
                                  {"ListAvailPlugins", NULL, N_("List Available Plugins"), NULL,
                                   "List Available Plugins", G_CALLBACK (list_available_plugins)},
                                  {"ViewMenu", NULL, N_("_View")},
                                  {"EntryMenu", NULL, N_("Mo_de")},
                                  {"Insert", NULL, N_("_Insert"), "Insert", "Insert Mode",
                                   G_CALLBACK (insert_mode)},
                                  {"Replace", NULL, N_("_Replace"), NULL, "Replace Mode",
                                   G_CALLBACK (replace_mode)},
                                  {"Blank", NULL, N_("_Blank"), NULL, "Blank Mode", G_CALLBACK(blank_mode)},
                                  {"Rest", NULL, N_("_Rest"), "r", "Rest Mode", G_CALLBACK(rest_toggle_key)},
                                  {"Default", NULL, N_("_Default"), "Escape", "Default Mode",
                                   G_CALLBACK (default_mode)},
                                  {"StaffMenu", NULL, N_("_Staff")},
                                  {"AddBefore", NULL, N_("Add _Before Current Staff..."), NULL, NULL,
                                   G_CALLBACK (newstaffbefore)},
                                  {"AddAfter", NULL, N_("Add _After Current Staff..."), NULL, NULL,
                                   G_CALLBACK (newstaffafter)},
                                  {"AddInitial", NULL, N_("Add in Initial Position..."), NULL, NULL,
                                   G_CALLBACK (newstaffinitial)},
                                  {"AddLast", NULL, N_("Add in Last Position..."), NULL, NULL,
                                   G_CALLBACK (newstafflast)},
                                  {"DeleteBefore", NULL, N_("Delete Staff Before"), NULL, NULL,
                                   G_CALLBACK (delete_staff_before)},
                                  {"Delete", GTK_STOCK_DELETE, N_("_Delete Staff"), NULL, NULL,
                                   G_CALLBACK (delete_staff_current)},
                                  {"DeleteAfter", NULL, N_("Delete Staff After"), NULL, NULL,
                                   G_CALLBACK (delete_staff_after)},
                                  {"AddVoice", NULL, N_("Add _Voice to Current Staff..."), NULL, NULL,
                                   G_CALLBACK (newstaffvoice)},
                                  {"AddLyric", NULL, N_("Add _Lyric Staff..."), NULL, NULL,
                                   G_CALLBACK (newstafflyric)},
                                  {"AddFiguredBass", NULL, N_("Add Figured Bass Staff..."), NULL, NULL,
                                   G_CALLBACK (newstafffigured)},
                                  {"StaffProperties", GTK_STOCK_PROPERTIES, N_("Staff _Properties"), NULL,
                                   NULL, G_CALLBACK (staff_properties_change)},
                                  {"InsertMenu", NULL, N_("_Insert")},
                                  {"Clef", NULL, N_("_Clef")},
                                  {"InitialClef", NULL, N_("_Set Clef"), NULL, NULL,
                                   G_CALLBACK (clef_change_initial)},
                                  {"InsertClef", NULL, N_("Insert Clef _Change"), NULL, NULL,
                                   G_CALLBACK (clef_change_insert)},
                                  {"Key", NULL, N_("_Key")},
                                  {"InitialKey", NULL, N_("_Initial Key"), NULL, NULL,
                                   G_CALLBACK (key_change_initial)},
                                  {"InsertKey", NULL, N_("Insert Key _Change"), NULL, NULL,
                                   G_CALLBACK (key_change_insert)},
                                  {"TimeSig", NULL, N_("_Time Signature")},
                                  {"InitialTimeSig", NULL, N_("_Initial Time Signature"), NULL, NULL,
                                   G_CALLBACK (timesig_change_initial)},
                                  {"InsertTimeSig", NULL, N_("Insert Time Signature _Change"), NULL, NULL,
                                   G_CALLBACK (timesig_change_insert)},
											  /* {"OtherMenu", NULL, N_("_Other")},*/
                                  {"ChangeNotehead", NULL, N_("_Notehead"), NULL, NULL,
                                   G_CALLBACK (set_notehead)},
                                  {"InsertStem", NULL, N_("Insert _Stem Directive"), NULL, NULL,
                                   G_CALLBACK (stem_directive_insert)},
											  {"Lyrics", NULL, N_("_Lyrics")},
                                  {"InsertLyric", NULL, N_("Insert Lyric"), NULL, NULL,
                                   G_CALLBACK (lyric_insert)},
                                  {"EditLyric", NULL, N_("Edit _Lyric"), NULL, NULL,
                                   G_CALLBACK (lyric_insert)},
                                  {"EditFiguredBass", NULL, N_("Edit Figured Bass"), NULL, NULL,
                                   G_CALLBACK (figure_insert)},
                                  {"InsertDynamic", NULL, N_("Insert _Dynamic"), NULL, NULL,
                                   G_CALLBACK (insert_dynamic)},
                                  /* {"InsertLilyDirective", NULL, N_("Insert Lilypond Directive"), NULL, NULL,
											  G_CALLBACK (lily_directive)},*/
                                  {"InsertBarline", NULL, N_("Insert _Barline"), NULL, NULL,
                                   G_CALLBACK (insert_barline)},
                                  {"NavigationMenu", NULL, N_("_Navigation")},
                                  {"GoToMeasure", NULL, N_("Go To _Measure..."), NULL, NULL,
                                   G_CALLBACK (tomeasurenum)},
                                  {"GoToBeginning", GTK_STOCK_GOTO_FIRST, N_("Go To _Beginning"), "Home",
                                   N_("Go To Beginning"),
                                   G_CALLBACK (tohome)},
                                  {"GoToEnd", GTK_STOCK_GOTO_LAST, N_("Go To _End"), "End", N_("Go To End"),
                                   G_CALLBACK (toend)},
                                  {"GoToNextScore", NULL, N_("Go To _Next Score"), NULL, NULL,
                                   G_CALLBACK (tonextscore)},
                                  {"PlaybackMenu", NULL, N_("_Playback")},
                                  {"Play", GTK_STOCK_MEDIA_PLAY, N_("_Play"), NULL, N_("Play"),
                                   G_CALLBACK (playback)},

#ifdef HAVEALSA
                                  {"PlayALSA", NULL, N_("Play using _Alsa"), NULL, NULL, G_CALLBACK(alsaplayback)},

#endif
                                  {"PlayCSound", GTK_STOCK_MEDIA_PLAY, N_("Play using _CSound..."), NULL,
                                   N_("Play using CSound..."),
                                   G_CALLBACK (csoundplayback)},
                                  {"Properties", GTK_STOCK_PROPERTIES, N_("Playback Pr_operties"), NULL, NULL,
                                   G_CALLBACK (playback_properties_change)},
                                  {"HelpMenu", NULL, N_("_Help")},
                                  {"Help", NULL, N_("Help"), NULL, NULL, G_CALLBACK(browse_manual)},
                                  {"About", NULL, N_("_About"), NULL, NULL,
                                   G_CALLBACK (about)},
                                  {"Bookmarks", NULL, N_("Bookmarks")},
                                  {"AddBookmark", NULL, N_("Add Bookmark"), NULL, NULL, G_CALLBACK(addbookmark)},
                                  {"GotoBookMark", NULL, N_("Goto Bookmark"), NULL, NULL, G_CALLBACK(gotobookmark)},
                                  {"Stub", NULL, "", NULL, "", NULL},
                                  {"OpenRecent", NULL, N_("Open Recent")},
                                  /* Note entry */
                                  {"InsertWholeNote", "denemo-whole-note", N_("Full note"), NULL,
                                   N_("Full note"),
                                   G_CALLBACK (insert_chord_0key_cb)},
                                  {"InsertHalfNote", "denemo-half-note", N_("Half note"), NULL,
                                   N_("Half note"),
                                   G_CALLBACK (insert_chord_1key_cb)},
                                  {"InsertQuarterNote", "denemo-quarter-note", N_("Quarter note"), NULL,
                                   N_("Quarter note"),
                                   G_CALLBACK (insert_chord_2key_cb)},
                                  {"InsertEighthNote", "denemo-eight-note", N_("Eight note"), NULL,
                                   N_("Eight note"),
                                   G_CALLBACK (insert_chord_3key_cb)},
                                  {"InsertSixteenthNote", "denemo-sixteenth-note", N_("Sixteenth note"), NULL,
                                   N_("Sixteenth note"),
                                   G_CALLBACK (insert_chord_4key_cb)},
                                  {"InsertWholeRest", "denemo-whole-rest", N_("Full rest"), NULL,
                                   N_("Full rest"),
                                   G_CALLBACK (insert_rest_0key_cb)},
                                  {"InsertHalfRest", "denemo-half-rest", N_("Half rest"), NULL,
                                   N_("Half rest"),
                                   G_CALLBACK (insert_rest_1key_cb)},
                                  {"InsertQuarterRest", "denemo-quarter-rest", N_("Quarter rest"), NULL,
                                   N_("Quarter rest"),
                                   G_CALLBACK (insert_rest_2key_cb)},
                                  {"InsertEighthRest", "denemo-eight-rest", N_("Eight rest"), NULL,
                                   N_("Eight rest"),
                                   G_CALLBACK (insert_rest_3key_cb)},
                                  {"InsertSixteenthRest", "denemo-sixteenth-rest", N_("Sixteenth rest"), NULL,
                                   N_("Sixteenth rest"),
                                   G_CALLBACK (insert_rest_4key_cb)}

                                };

//Get number of menu entries
gint n_menu_items = G_N_ELEMENTS (menu_entries);

/**
 *  Function to toggle whether notation toolbar is visible 
 *
 */
static void
toggle_notation_toolbar (GtkAction * action, scoreinfo *si)
{
  GtkWidget *widget;
  widget = gtk_ui_manager_get_widget (si->ui_manager, "/EntryToolBar");
  if (GTK_WIDGET_VISIBLE (widget) && si->prefs->notation_palette == TRUE)
    {
      gtk_widget_hide (widget);
    }
  else if(GTK_WIDGET_VISIBLE (widget) && si->prefs->notation_palette == FALSE)
    {
      gtk_widget_hide (widget);
    }
  else
    {
      gtk_widget_show (widget);
    }
}

/**
 * entries for View menu
 */
GtkToggleActionEntry toggle_menu_entries[] = {
      {"ToggleNotationToolbar", NULL, N_("_Notation Toolbar"), NULL, NULL,
       G_CALLBACK (toggle_notation_toolbar), FALSE}
      ,
      {"ToggleArticulationPalette", NULL, N_("_Articulation Palette"), NULL, NULL,
       G_CALLBACK (toggle_articulation_palette), FALSE}

    };


struct cbdata
  {
    struct scoreinfo *si;
    gchar *filename;
  };

/**
 * Callback for the history menu
 * opens the selected file
 */
static void
openrecent(GtkWidget *widget, gpointer data)
{
  struct cbdata *cdata = (struct cbdata *)data;
  g_print("actioned\n");
  open_for_real(cdata->filename, cdata->si);

}



/**
 * Add history entry to the History menu 
 *
 */
void addhistorymenuitem(gpointer data, gpointer user_data)
{

  GtkWidget *item;
  struct cbd *cd = (struct cbd *) user_data;
  gchar *tmpstring = g_strconcat((gchar *)data, "\0", NULL);
  item = gtk_menu_item_new_with_label(tmpstring);
  gtk_menu_shell_insert(GTK_MENU_SHELL(cd->menu),item,0);
  struct cbdata *cdata = (struct cbdata *)g_malloc0(sizeof(struct cbdata));
  cdata->si = cd->si;
  cdata->filename = (gchar *)data;
  g_signal_connect(item, "activate", G_CALLBACK(openrecent), cdata);
  gtk_widget_show(item);
  g_free(tmpstring);
}

/**
 * Top-Level function to populate the History menu
 * with elements read from the denemohistory file
 */
void populate_opened_recent(scoreinfo *si)
{
	GtkWidget *item = gtk_ui_manager_get_widget( si->ui_manager, "/MainMenu/FileMenu/OpenRecent/Stub" );
  GtkWidget *menu = gtk_widget_get_parent( GTK_WIDGET( item ) );
 // gtk_widget_destroy( item );
  static struct cbd data;
  data.si = si;
  data.menu = menu;

  g_queue_foreach(si->prefs->history, addhistorymenuitem, &data);
}


/**
 * Creates a new view to allow editing of multiple scores during a denemo session
 *
 */
void
newview ()
{
  scoreinfo *si =
    (scoreinfo *) g_malloc0 (sizeof (scoreinfo));
	
  struct prefinfo *prefs;
  GtkWidget *main_vbox, *menubar, *score_and_scroll_hbox, *toolbar, *hbox;
  GtkActionGroup *action_group;
  GtkUIManager *ui_manager;
  GtkAccelGroup *accel_group;
  GError *error;
  GtkWidget *widget;
  gchar *data_dir;
  gchar *data_dir2;

  /* Initialize preferences */
  si->prefs = initprefs ();
  readHistory(si->prefs);
  /* Initialize the staff structure */
  init_score (si);
  //si->prefs = prefs;

  newstaff (si, INITIAL, DENEMO_NONE);
  si->pixmap = NULL;

  /* Initialize the GUI */

  si->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  gchar *dialog_title = "Denemo";
  gtk_window_set_title (GTK_WINDOW (si->window), dialog_title);
#if GTK_MAJOR_VERSION > 1

  gtk_window_set_resizable (GTK_WINDOW (si->window), TRUE);
#else

  gtk_window_set_policy (GTK_WINDOW (si->window), TRUE, TRUE, FALSE);
#endif

  main_vbox = gtk_vbox_new (FALSE, 1);
  gtk_container_border_width (GTK_CONTAINER (main_vbox), 1);
  gtk_container_add (GTK_CONTAINER (si->window), main_vbox);
  gtk_widget_show (main_vbox);

  /* This part is taken more-or-less directly from the gtk+ tutorial */
  accel_group = gtk_accel_group_new ();
  action_group = gtk_action_group_new ("MenuActions");
  /* This also sets si as the  callback data for all the functions in the
   * menubar, which is precisely what we want. */
  gtk_action_group_add_actions (action_group, menu_entries,
                                G_N_ELEMENTS (menu_entries), si);

  gtk_action_group_add_toggle_actions (action_group,
                                       toggle_menu_entries,
                                       G_N_ELEMENTS (toggle_menu_entries),
                                       si);
  ui_manager = gtk_ui_manager_new ();
  si->ui_manager = ui_manager;
  gtk_ui_manager_insert_action_group (ui_manager, action_group, 0);
  accel_group = gtk_ui_manager_get_accel_group (ui_manager);
  error = NULL;
  //item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>",
  //                           accel_group);
  //gtk_item_factory_create_items (item_factory, n_menu_items, menu_items, si);
  
#if GTK_MAJOR_VERSION > 1

  gtk_window_add_accel_group (GTK_WINDOW (si->window), accel_group);
  data_dir2 = gbr_find_data_dir(PKGDATADIR);
  data_dir = g_strconcat(data_dir2, "/denemoui.xml", NULL);
  g_free(data_dir2);
  if (!gtk_ui_manager_add_ui_from_file (ui_manager,data_dir,
                                        &error))
    {
      g_message ("building menu failed: %s", error->message);
      g_error_free (error);
      exit (EXIT_FAILURE);
    }
  g_free(data_dir);
#else
  gtk_accel_group_attach (accel_group, GTK_OBJECT (si->window));
#endif
  //menubar = gtk_item_factory_get_widget (item_factory, "<main>");
  menubar = gtk_ui_manager_get_widget (ui_manager, "/MainMenu");
  si->menubar = menubar;
  gtk_box_pack_start (GTK_BOX (main_vbox), menubar, FALSE, TRUE, 0);
  gtk_widget_show (menubar);
  populate_opened_recent(si);

  toolbar = gtk_ui_manager_get_widget (ui_manager, "/ToolBar");
  // The user should be able to decide toolbar style.
  // But without gnome, there is no (ui) to set this option.
  gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), GTK_TOOLBAR_BOTH_HORIZ);
  gtk_box_pack_start (GTK_BOX (main_vbox), toolbar, FALSE, TRUE, 0);
  gtk_widget_show (toolbar);

  widget = gtk_ui_manager_get_widget (ui_manager, "/ToolBar/Open");
  /*gtk_tool_item_set_is_important (GTK_TOOL_ITEM (widget), TRUE);*/

  widget = gtk_ui_manager_get_widget (ui_manager, "/ToolBar/Save");
  /*gtk_tool_item_set_is_important (GTK_TOOL_ITEM (widget), TRUE);*/

  toolbar = gtk_ui_manager_get_widget (ui_manager, "/EntryToolBar");
  gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), GTK_TOOLBAR_ICONS);
  gtk_box_pack_start (GTK_BOX (main_vbox), toolbar, FALSE, TRUE, 0);
  gtk_widget_show (toolbar);


  score_and_scroll_hbox = gtk_hbox_new (FALSE, 1);
  gtk_box_pack_start (GTK_BOX (main_vbox), score_and_scroll_hbox, TRUE, TRUE,
                      0);
  gtk_widget_show (score_and_scroll_hbox);

  si->scorearea = gtk_drawing_area_new ();
#if GTK_MAJOR_VERSION > 1

  gtk_window_set_default_size (GTK_WINDOW (si->window), INITIAL_WIDTH,
                               INITIAL_HEIGHT);
#else

  gtk_drawing_area_size (GTK_DRAWING_AREA (si->scorearea),
                         INITIAL_WIDTH, INITIAL_HEIGHT);
#endif

  gtk_box_pack_start (GTK_BOX (score_and_scroll_hbox), si->scorearea, TRUE,
                      TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (si->scorearea), "expose_event",
                      GTK_SIGNAL_FUNC (scorearea_expose_event), si);
  gtk_signal_connect (GTK_OBJECT (si->scorearea), "configure_event",
                      GTK_SIGNAL_FUNC (scorearea_configure_event), si);
  gtk_signal_connect (GTK_OBJECT (si->scorearea), "button_release_event",
                      GTK_SIGNAL_FUNC (scorearea_button_release), si);
  gtk_widget_set_events (si->scorearea, (GDK_EXPOSURE_MASK
                                         | GDK_LEAVE_NOTIFY_MASK
                                         | GDK_BUTTON_PRESS_MASK
                                         | GDK_BUTTON_RELEASE_MASK));
  gtk_widget_show (si->scorearea);

  si->vadjustment = gtk_adjustment_new (1.0, 1.0, 2.0, 1.0, 4.0, 1.0);
  gtk_signal_connect (GTK_OBJECT (si->vadjustment), "value_changed",
                      GTK_SIGNAL_FUNC (vertical_scroll), si);
  si->vscrollbar = gtk_vscrollbar_new (GTK_ADJUSTMENT (si->vadjustment));
  gtk_box_pack_start (GTK_BOX (score_and_scroll_hbox), si->vscrollbar, FALSE,
                      TRUE, 0);
  gtk_widget_show (si->vscrollbar);

  si->hadjustment = gtk_adjustment_new (1.0, 1.0, 2.0, 1.0, 4.0, 1.0);

  gtk_signal_connect (GTK_OBJECT (si->hadjustment), "value_changed",
                      GTK_SIGNAL_FUNC (horizontal_scroll), si);
  si->hscrollbar = gtk_hscrollbar_new (GTK_ADJUSTMENT (si->hadjustment));
  gtk_box_pack_start (GTK_BOX (main_vbox), si->hscrollbar, FALSE, TRUE, 0);
  gtk_widget_show (si->hscrollbar);

  hbox = gtk_hbox_new(FALSE, 1);
  gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, TRUE,0);
  gtk_widget_show(hbox);


  si->statusbar = gtk_statusbar_new ();
  gtk_box_pack_start (GTK_BOX (hbox), si->statusbar, TRUE,TRUE, 5);
  gtk_widget_show (si->statusbar);
  si->status_context_id =
    gtk_statusbar_get_context_id (GTK_STATUSBAR (si->statusbar), "Denemo");
  gtk_statusbar_push (GTK_STATUSBAR (si->statusbar), si->status_context_id,
                      "Denemo");

  si->progressbar = gtk_progress_bar_new();
  gtk_box_pack_start(GTK_BOX(hbox), si->progressbar,TRUE,TRUE,5);
  gtk_widget_show(si->progressbar);


  gtk_signal_connect (GTK_OBJECT (si->window), "delete_event",
		  (GtkSignalFunc) close_application, si);
  gtk_signal_connect (GTK_OBJECT (si->window), "key_press_event",
                      (GtkSignalFunc) scorearea_keypress_event, si);

  gtk_widget_show (si->window);
  gtk_widget_grab_focus (si->scorearea);

  /* Now that the window's shown, initialize the gcs */
  gcs_init (si->window->window);
  if(si->prefs->articulation_palette)
    toggle_articulation_palette(NULL, si);

  if(!si->prefs->notation_palette)
    {
      g_print("Notation palette %d\n", si->prefs->notation_palette);
      toggle_notation_toolbar(NULL, si);
    }
  /* Similarly, the keymap should be initialized after the
     only once si->window is shown, as it may pop up an advisory
     dialog. */
  si->prefs->the_keymap = init_keymap ();
  displays = g_list_append (displays, si);
  g_print ("No of Displays %d\n", g_list_length (displays));
  if (si->prefs->autosave)
    g_timeout_add (si->prefs->autosave_timeout * 1000 * 60,
                   (GSourceFunc) auto_save_document_timeout, si);

  g_free (error);
}
