# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>
#          Chad Miller <chad.miller@canonical.com>

"""testing database/contact.py module"""
import testtools
import os
import signal
import time

import desktopcouch.tests as test_environment
from desktopcouch.records.server import CouchDatabase
from desktopcouch.records.server_base import (
    row_is_deleted, NoSuchDatabase, FieldsConflict, ResourceConflict)
from desktopcouch.records.record import Record
from desktopcouch.stop_local_couchdb import stop_couchdb
from desktopcouch import find_pid

# pylint can't deal with failing imports even when they're handled
# pylint: disable-msg=F0401
try:
    from io import StringIO
except ImportError:
    from cStringIO import StringIO as StringIO
# pylint: enable-msg=F0401

FAKE_RECORD_TYPE = "http://example.org/test"

js = """
function(doc) {
    if (doc.record_type == '%s') {
        emit(doc._id, null);
    }
}""" % FAKE_RECORD_TYPE


class TestCouchDatabase(testtools.TestCase):
    """tests specific for CouchDatabase"""

    def setUp(self):
        """setup each test"""
        super(TestCouchDatabase, self).setUp()
        # Connect to CouchDB server
        self.dbname = self._testMethodName
        self.database = CouchDatabase(self.dbname, create=True,
                ctx=self.get_test_context())
        #create some records to pull out and test
        self.database.put_record(Record({
            "key1_1": "val1_1", "key1_2": "val1_2", "key1_3": "val1_3",
            "record_type": "test.com"}))
        self.database.put_record(Record({
            "key2_1": "val2_1", "key2_2": "val2_2", "key2_3": "val2_3",
            "record_type": "test.com"}))
        self.database.put_record(Record({
            "key13_1": "va31_1", "key3_2": "val3_2", "key3_3": "val3_3",
            "record_type": "test.com"}))

    def tearDown(self):
        """tear down each test"""
        this_context = self.get_test_context()
        if this_context != test_environment.test_context:
            stop_couchdb(ctx=this_context)
        super(TestCouchDatabase, self).tearDown()

    def get_test_context(self):
        return test_environment.test_context

    def maybe_die(self):
        pass

    def wait_until_server_dead(self, pid=None):
        if pid is not None:
            pid = find_pid(start_if_not_running=False, ctx=self.get_test_context())
        if pid is None:
            return
        while True:
            try:
                os.kill(pid, 0)  # Wait until exited
                time.sleep(0.1)
            except OSError:
                break

    def test_database_not_exists(self):
        self.assertRaises(
            NoSuchDatabase, CouchDatabase, "this-must-not-exist", create=False)

    def test_get_records_by_record_type_save_view(self):
        """Test getting mutliple records by type"""
        records = self.database.get_records(
            record_type="test.com",create_view=True)
        self.maybe_die()  # should be able to survive couchdb death
        self.assertEqual(3, len(records))

    def test_get_record(self):
        """Test getting a record."""
        record = Record({'record_number': 0}, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        self.maybe_die()  # should be able to survive couchdb death
        retrieved_record = self.database.get_record(record_id)
        self.assertEqual(0, retrieved_record['record_number'])

    def test_put_record(self):
        """Test putting a record."""
        record = Record({'record_number': 0}, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        self.maybe_die()  # should be able to survive couchdb death
        retrieved_record = self.database.get_record(record_id)
        self.assertEqual(
            record['record_number'], retrieved_record['record_number'])

    def test_put_records_batch(self):
        """Test putting a batch of records."""
        # create records to be put in a batch
        batch = []
        for current in (1, 2, 3, 4, 5):
            record = Record(
                {'record_number': current}, record_type="http://example.com/")
            if current % 2:
                # set bad data to get an exception
                record._data["_rev"] = "1-32323"
            batch.append(record)
        # put the batch and ensure that the records have been added
        batch_result = self.database.put_records_batch(batch)
        for current_tuple in batch_result:
            success, docid, rev_or_exc = current_tuple
            if success:
                self.assertTrue(self.database._server[self.dbname][docid])
            else:
                # make sure we do not have the record in the db
                self.assertFalse(self.database._server[self.dbname][docid])

    def test_delete_record(self):
        """Test deletion of records."""
        record = Record({'record_number': 0}, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        self.database.delete_record(record_id)
        ###self.maybe_die()  # should be able to survive couchdb death
        deleted_record = self.database._server[self.dbname][record_id]
        #deleted_record = self.database.get_record(record_id)
        self.assert_(deleted_record['application_annotations']['Ubuntu One'][
            'private_application_annotations']['deleted'])

    def test_delete_and_recreate_record(self):
        """Test deletion of records."""
        old_record = Record({'record_number': 0},
                record_type="http://example.com/", record_id="Reuse me")
        old_record_id = self.database.put_record(old_record)
        self.database.delete_record(old_record_id)

        new_record = Record({'something else': 42},
                record_type="http://example.com/", record_id=old_record_id)
        new_record_id = self.database.put_record(new_record)

        self.assertEqual(old_record_id, new_record_id)

        new_record_fetched = self.database.get_record(new_record_id)
        self.assertEqual(new_record_fetched["something else"], 42)

        # this still fails, to be sure.
        bad_record = Record({'again': 'prev not deleted'},
                record_type="http://example.com/", record_id=old_record_id)
        self.assertRaises(ResourceConflict,
                self.database.put_record, bad_record)

        self.database.delete_record(new_record_id)

    def test_get_deleted_record(self):
        """Test (not) getting a deleted record."""
        record = Record({'record_number': 0}, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        self.database.delete_record(record_id)
        self.maybe_die()  # should be able to survive couchdb death
        retrieved_record = self.database.get_record(record_id)
        self.assertEqual(None, retrieved_record)

    def test_record_exists(self):
        """Test checking whether a record exists."""
        record = Record({'record_number': 0}, record_type="http://example.com/")
        self.assert_(not self.database.record_exists("ThisMustNotExist"))
        record_id = self.database.put_record(record)
        self.maybe_die()  # should be able to survive couchdb death
        self.assert_(self.database.record_exists(record_id))
        self.database.delete_record(record_id)
        self.assert_(not self.database.record_exists(record_id))

    def test_update_fields(self):
        """Test the update_fields method."""
        dictionary = {'record_number': 0, 'field1': 1, 'field2': 2}
        record = Record(dictionary, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        self.maybe_die()  # should be able to survive couchdb death
        # manipulate the database 'out of view'
        non_working_copy = self.database.get_record(record_id)
        non_working_copy['field2'] = 22
        non_working_copy['field3'] = 3
        self.database.put_record(non_working_copy)
        self.database.update_fields(record_id, {'field1': 11})
        self.maybe_die()  # should be able to survive couchdb death
        working_copy = self.database.get_record(record_id)
        self.assertEqual(0, working_copy['record_number'])
        self.assertEqual(11, working_copy['field1'])
        self.assertEqual(22, working_copy['field2'])
        self.assertEqual(3, working_copy['field3'])

    def test_view_add_and_delete(self):
        design_doc = "design"
        view1_name = "unit_tests_are_wonderful"
        view2_name = "unit_tests_are_marvelous"

        map_js = """function(doc) { emit(doc._id, null) }"""
        reduce_js = """\
                function (key, values, rereduce) {
                    return sum(values);
                }"""

        # add two and delete two.
        self.assertRaises(
            KeyError, self.database.delete_view, view1_name, design_doc)
        self.assertRaises(
            KeyError, self.database.delete_view, view2_name, design_doc)
        self.database.add_view(view1_name, map_js, reduce_js, design_doc)
        self.maybe_die()  # should be able to survive couchdb death
        self.database.add_view(view2_name, map_js, reduce_js, design_doc)
        self.database.delete_view(view1_name, design_doc)
        self.assertRaises(
            KeyError, self.database.delete_view, view1_name, design_doc)
        self.maybe_die()  # should be able to survive couchdb death
        self.database.delete_view(view2_name, design_doc)
        self.assertRaises(
            KeyError, self.database.delete_view, view2_name, design_doc)

    def test_func_get_records(self):
        record_ids_we_care_about = set()
        good_record_type = "http://example.com/unittest/good"
        other_record_type = "http://example.com/unittest/bad"

        for i in range(7):
            record = Record({'record_number': i},
                    record_type=good_record_type)
            if i % 3 == 1:
                record = Record({'record_number': i},
                        record_type=good_record_type)
                record_ids_we_care_about.add(self.database.put_record(record))
            elif i % 3 == 2:
                record = Record({'record_number': i},
                        record_type=good_record_type)
                record_id = self.database.put_record(record)  # correct type,
                self.database.delete_record(record_id)  # but marked deleted!
            else:
                record = Record({'record_number': i},
                        record_type=other_record_type)
                self.database.put_record(record)

        results = self.database.get_records(create_view=True)

        self.assertTrue(8 <= len(results.rows))  # our 8, plus callers' data
        self.assertIn("record_type", results.rows[0].value)

        for row in results[good_record_type]:  # index notation
            self.assertTrue(row.id in record_ids_we_care_about)
            record_ids_we_care_about.remove(row.id)
            self.assertFalse(row_is_deleted(row))

        self.maybe_die()  # should be able to survive couchdb death
        self.assertTrue(len(record_ids_we_care_about) == 0, "expected zero")

        self.assertRaises(KeyError, self.database.get_records,
                design_doc="mustNotExist", create_view=False)

    def test_list_views(self):
        design_doc = "d"
        self.assertEqual(self.database.list_views(design_doc), [])

        view_name = "unit_tests_are_fantastic"
        map_js = """function(doc) { emit(doc._id, null) }"""
        self.database.add_view(view_name, map_js, None, design_doc)

        self.maybe_die()  # should be able to survive couchdb death
        self.assertEqual(self.database.list_views(design_doc), [view_name])
        self.database.delete_view(view_name, design_doc)

        self.assertEqual(self.database.list_views(design_doc), [])

    def test_get_view_by_type_new_but_already(self):
        self.database.get_records(create_view=True)
        self.maybe_die()  # should be able to survive couchdb death
        self.database.get_records(create_view=True)
        # No exceptions on second run?  Yay.

    def test_get_view_by_type_createxcl_fail(self):
        self.database.get_records(create_view=True)
        self.maybe_die()  # should be able to survive couchdb death
        self.assertRaises(KeyError, self.database.get_records, create_view=None)

    def test_get_changes(self):
        self.test_put_record()
        self.test_update_fields()
        self.test_delete_record()
        self.test_view_add_and_delete()
        self.test_func_get_records()

        changes = self.database.get_changes()
        self.assertTrue(len(changes) > 4)
        for change in changes:
            self.assertTrue(isinstance(change, dict))
            self.failUnless("changes" in change)
            self.failUnless("id" in change)

    def test_report_changes_polite(self):
        def rep(**kwargs):
            self.failUnless("changes" in kwargs)
            self.failUnless("id" in kwargs)

        # First try all operations.
        self.test_put_record()
        self.test_update_fields()
        self.test_delete_record()
        self.test_view_add_and_delete()
        self.test_func_get_records()

        self.database.report_changes(rep)  # Make sure nothing horks.

        # Too soon to try again.
        count = self.database.report_changes(lambda **kw: self.fail())
        self.assertEqual(0, count)

    def test_report_changes_exceptions(self):
        def rep(**kwargs):
            self.failUnless("changes" in kwargs)
            self.failUnless("id" in kwargs)

        # Consume pending.
        self.database.report_changes(rep)
        self.database._changes_last_used = 0

        # Store time.
        saved_time = self.database._changes_last_used
        # Store position.
        saved_position = self.database._changes_since
        # Queue new changes.  This is  1  event!
        self.test_put_record()

        # Exceptions in our callbacks do not consume an event.
        self.assertRaises(
            ZeroDivisionError, self.database.report_changes, lambda **kw: 1/0)

        # Ensure pos'n is same.
        self.assertEqual(saved_position, self.database._changes_since)
        # Ensure time is same.
        self.assertEqual(saved_time, self.database._changes_last_used)

        ###self.maybe_die()  # should be able to survive couchdb death

        # Next time we run, we get the same event again.
        # Consume queued changes.
        count = self.database.report_changes(rep)
        # Ensure position different.
        self.assertEquals(1, count)
        # Ensure position different.
        self.assertEqual(saved_position + 1, self.database._changes_since)

    def test_view_wrapper_has_all_attributes(self):
        design_doc = "test_view_wrapper_has_all_attributes"
        view1_name = "unit_tests_are_great_yeah"

        map_js = """function(doc) { emit(doc._id, null) }"""

        self.database.add_view(view1_name, map_js, None, design_doc)
        view = self.database.execute_view(view1_name, design_doc)
        
        for attr_name in dir(view.obj):
            # This works because hasattr actually calls getattr.
            self.assertTrue(hasattr(view, attr_name))

    def test_report_changes_all_ops_give_known_keys(self):
        def rep(**kwargs):
            self.failUnless("changes" in kwargs)
            self.failUnless("id" in kwargs)

        # Permit immediate run.
        self.database._changes_last_used = 0
        # Test expected kw args.
        self.database.report_changes(rep)

    def test_report_changes_nochanges(self):
        def rep(**kwargs):
            self.failUnless("changes" in kwargs)
            self.failUnless("id" in kwargs)

        # Consume queue.
        count = self.database.report_changes(rep)
        # Permit immediate run.
        self.database._changes_last_used = 0
        # Store position.
        saved_position = self.database._changes_since
        count = self.database.report_changes(rep)
        # Ensure event count is zero.
        self.assertEquals(0, count)
        # Pos'n is same.
        self.assertEqual(saved_position, self.database._changes_since)

    def test_attachments(self):
        content = StringIO("0123456789\n==========\n\n" * 5)

        constructed_record = Record(
            {'record_number': 0}, record_type="http://example.com/")

        # Before anything is attached, there are no attachments.
        self.assertEqual(constructed_record.list_attachments(), [])

        # We can add attachments before a document is put in the DB.
        # Documents can look like files or strings.
        constructed_record.attach(content, "nu/mbe/rs", "text/plain")
        constructed_record.attach("string", "another document", "text/plain")

        ###self.maybe_die()  # should be able to survive couchdb death
        constructed_record.attach("XXXXXXXXX", "never used", "text/plain")
        constructed_record.detach("never used")  # detach works before commit.

        # We can read from a document that we constructed.
        out_file, out_content_type = \
                constructed_record.attachment_data("nu/mbe/rs")
        self.assertEqual(out_content_type, "text/plain")

        # One can not put another document of the same name.
        self.assertRaises(KeyError, constructed_record.attach, content,
                "another document", "text/x-rst")

        ###self.maybe_die()  # should be able to survive couchdb death
        record_id = self.database.put_record(constructed_record)
        retrieved_record = self.database.get_record(record_id)

        # We can add attachments after a document is put in the DB.
        retrieved_record.attach(content, "Document", "text/x-rst")
        # push new version
        record_id = self.database.put_record(retrieved_record)
        # get new
        retrieved_record = self.database.get_record(record_id)

        # Remove and test that it is indeed gone.
        retrieved_record.detach("another document")
        self.assertRaises(KeyError, retrieved_record.detach,
                "another document")

        # push new version
        record_id = self.database.put_record(retrieved_record)
        # get new
        retrieved_record = self.database.get_record(record_id)

        ###self.maybe_die()  # should be able to survive couchdb death
        # We can get a list of attachments.
        self.assertEqual(set(retrieved_record.list_attachments()),
                set(["nu/mbe/rs", "Document"]))

        # Send back name.
        retrieved_record.attach(content, "another document", "text/plain")

        # push new version
        record_id = self.database.put_record(retrieved_record)
        # get new
        retrieved_record = self.database.get_record(record_id)

        # We can get a list of attachments.
        self.assertEqual(set(retrieved_record.list_attachments()),
                set(["nu/mbe/rs", "Document", "another document"]))

        # We can read from a document that we retrieved.
        out_data, out_content_type = retrieved_record.attachment_data(
            "nu/mbe/rs")
        self.assertEqual(out_data, content.getvalue())
        self.assertEqual(out_content_type, "text/plain")

        ###self.maybe_die()  # should be able to survive couchdb death
        # Asking for a named document that does not exist causes KeyError.
        self.assertRaises(KeyError, retrieved_record.attachment_data,
                "NoExist")
        self.assertRaises(KeyError, constructed_record.attachment_data,
                "No Exist")
        self.assertRaises(KeyError, retrieved_record.detach,
                "NoExist")

        for i, name in enumerate(retrieved_record.list_attachments()):
            if i != 0:
                # delete all but one.
                retrieved_record.detach(name)
        # push new version.
        record_id = self.database.put_record(retrieved_record)

        # We can remove records with attachments.
        self.database.delete_record(record_id)

    def test_view_fetch(self):
        design_doc = "test_view_fetch"
        view1_name = "unit_tests_are_great_yeah"

        map_js = """function(doc) { emit(doc._id, null) }"""

        self.database.add_view(view1_name, map_js, None, design_doc)
        data = [i.key for i in
            list(self.database.execute_view(view1_name, design_doc))]
        # ordinary requests are in key order
        self.assertEqual(data, sorted(data))

        self.maybe_die()  # should be able to survive couchdb death
        # now request descending order and confirm that it *is* descending
        descdata = [i.key for i in
            list(self.database.execute_view(view1_name, design_doc,
                descending=True))]
        self.assertEqual(descdata, list(reversed(sorted(data))))

        self.database.delete_view(view1_name, design_doc)

    def test_update_fields_success(self):
        """Test update_fields method"""
        dictionary = {
            'record_number': 0,
            'field1': 1,
            'field2': 2,
            'nested': {
                'sub1': 's1',
                'sub2': 's2'}}
        record = Record(dictionary, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        # manipulate the database 'out of view'
        non_working_copy = self.database.get_record(record_id)
        non_working_copy['field2'] = 22
        non_working_copy['field3'] = 3
        self.database.put_record(non_working_copy)
        self.database.update_fields(
            record_id, {'field1': 11,('nested', 'sub2'): 's2-changed'},
            cached_record=record)
        working_copy = self.database.get_record(record_id)
        self.assertEqual(0, working_copy['record_number'])
        self.assertEqual(11, working_copy['field1'])
        self.assertEqual(22, working_copy['field2'])
        self.assertEqual(3, working_copy['field3'])
        self.assertEqual('s2-changed', working_copy['nested']['sub2'])
        self.assertEqual('s1', working_copy['nested']['sub1'])

    def test_update_fields_failure(self):
        """Test update_fields method"""
        dictionary = {
            'record_number': 0,
            'field1': 1,
            'field2': 2,
            'nested': {
                'sub1': 's1',
                'sub2': 's2'}}
        record = Record(dictionary, record_type="http://example.com/")
        record_id = self.database.put_record(record)
        # manipulate the database 'out of view'
        non_working_copy = self.database.get_record(record_id)
        non_working_copy['field1'] = 22
        non_working_copy['field3'] = 3
        self.database.put_record(non_working_copy)
        try:
            self.database.update_fields(
                record_id, {'field1': 11, ('nested', 'sub2'): 's2-changed'},
                cached_record=record)
            # we want the exception
            self.fail()
        except FieldsConflict, e:
            self.assertEqual({('field1',): (22, 11)}, e.conflicts)


class TestServerDiesSegv(TestCouchDatabase):
    def get_test_context(self):
        try:
            return self.ctx
        except AttributeError:
            self.ctx = test_environment.create_new_test_environment()
            return self.ctx

    def maybe_die(self):
        self.database.ensure_full_commit()
        time.sleep(2)
        pid = find_pid(start_if_not_running=False, ctx=self.get_test_context())
        if pid is None:
            print "couchdb has already quit!  That's unexpected."
            return
        print "DIE, process", pid, "!"
        os.kill(pid, signal.SIGSEGV)
        self.wait_until_server_dead(pid=pid)


class TestServerDiesNormal(TestCouchDatabase):
    def get_test_context(self):
        try:
            return self.ctx
        except AttributeError:
            self.ctx = test_environment.create_new_test_environment()
            return self.ctx

    def maybe_die(self):
        self.database.ensure_full_commit()
        time.sleep(2)
        stop_couchdb(ctx=self.get_test_context())
        self.wait_until_server_dead()
