# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.

"""Test Couchdb IO"""

import pygtk
pygtk.require('2.0')

import desktopcouch.application.tests as test_environment

from desktopcouch.application.pair.couchdb_pairing import couchdb_io
from desktopcouch.application.server import DesktopDatabase
from desktopcouch.records import Record
from desktopcouch.application.replication_services import ubuntuone
import unittest
import uuid
import os

URI = None  # use autodiscovery that desktopcouch.tests permits.


class TestCouchdbIo(unittest.TestCase):
    """Test case for Couchdb IO"""

    def setUp(self):
        """setup each test"""
        self.mgt_database = DesktopDatabase('management', create=True, uri=URI,
                ctx=test_environment.test_context)
        self.foo_database = DesktopDatabase('foo', create=True, uri=URI,
                ctx=test_environment.test_context)
        #create some records to pull out and test
        self.foo_database = DesktopDatabase('bar', create=True, uri=URI,
                ctx=test_environment.test_context)
        #create some records to pull out and test
        self.foo_database.put_record(Record({
            "key1_1": "val1_1", "key1_2": "val1_2", "key1_3": "val1_3",
            "record_type": "test.com"}))
        self.foo_database.put_record(Record({
            "key2_1": "val2_1", "key2_2": "val2_2", "key2_3": "val2_3",
            "record_type": "test.com"}))
        self.foo_database.put_record(Record({
            "key13_1": "va31_1", "key3_2": "val3_2", "key3_3": "val3_3",
            "record_type": "test.com"}))

        couchdb_io.put_static_paired_service({
            "consumer_key": str("abcdef"),
            "consumer_secret": str("ghighjklm"),
            "token": str("opqrst"),
            "token_secret": str("uvwxyz")}, "ubuntuone", uri=URI,
            ctx=test_environment.test_context)
        excl = ubuntuone.ReplicationExclusion(
            ctx=test_environment.test_context)
        excl.exclude("bar")

    def tearDown(self):
        """tear down each test"""
        del self.mgt_database._server['management']  # pylint: disable=W0212
        del self.mgt_database._server['foo']         # pylint: disable=W0212

    def test_obsfuscation(self):
        """Test the obfuscation of sensitive data."""
        test_record = {
            'url':
            'https://couchdb.one.ubuntu.com/u%2Fb2%2Fc8%2F276%2Ftest',
            'auth': {
                'oauth': {
                    'consumer_secret': 'SeCrEtSe',
                    'token': '3XRjQrWX92TTTJFDTWJJ',
                    'consumer_key': 'ubuntuone',
                    'token_secret':
                        'jBmSeCrEtawkefwklefliwuregqwlkeh347wq87w4fiuq4fyu3q4f'
                        'iqwu4fqwfiqufM6xjsPwSeCrEt4'}}}
        cleaned_t = couchdb_io.obsfuscate(test_record)
        self.failIf(
            "SeCrEt" in str(cleaned_t), {
                'url':
                'https://couchdb.one.ubuntu.com/u%2Fb2%2Fc8%2F276%2Ftest',
                'auth': {'oauth': {
                    'consumer_secret': 'HiddenHidd',
                    'token': '3XRjQrWX92TTTJFDTWJJ',
                    'consumer_key': 'ubuntuone',
                    'token_secret':
                        'HiddenHiddenHiddenHiddenHiddenHiddenHiddenHiddenHidde'
                        'nHiddenHiddenHiddenHiddenHi'}}})
        self.assertEqual(couchdb_io.obsfuscate(""), "")
        self.assertEqual(couchdb_io.obsfuscate({}), {})
        self.assertEqual(couchdb_io.obsfuscate({1: {}}), {1: {}})
        self.assertEqual(couchdb_io.obsfuscate({1: 1}), {1: 1})

    def test_put_static_paired_service(self):  # pylint: disable=R0201
        """Test putting a static paired service."""
        service_name = "dummyfortest"
        oauth_data = {
                "consumer_key": str("abcdef"),
                "consumer_secret": str("ghighjklm"),
                "token": str("opqrst"),
                "token_secret": str("uvwxyz"),
            }
        couchdb_io.put_static_paired_service(
            oauth_data, service_name, uri=URI,
            ctx=test_environment.test_context)
        couchdb_io.get_pairings(ctx=test_environment.test_context)
        # Assert something?

    def test_put_dynamic_paired_host(self):
        """Test putting a dynamically paired host."""
        hostname = "host%d" % (os.getpid(),)
        remote_uuid = str(uuid.uuid4())
        oauth_data = {
                "consumer_key": str("abcdef"),
                "consumer_secret": str("ghighjklm"),
                "token": str("opqrst"),
                "token_secret": str("uvwxyz"),
            }

        couchdb_io.put_dynamic_paired_host(hostname, remote_uuid, oauth_data,
                uri=URI, ctx=test_environment.test_context)
        couchdb_io.put_dynamic_paired_host(hostname, remote_uuid, oauth_data,
                uri=URI, ctx=test_environment.test_context)
        couchdb_io.put_dynamic_paired_host(hostname, remote_uuid, oauth_data,
                uri=URI, ctx=test_environment.test_context)

        pairings = list(couchdb_io.get_pairings(
            ctx=test_environment.test_context))
        # 3, plus 1 from setUp()
        self.assertEqual(4, len(pairings))
        self.assertEqual(pairings[0].value["oauth"], oauth_data)
        self.assertEqual(pairings[0].value["server"], hostname)
        self.assertEqual(pairings[0].value["pairing_identifier"], remote_uuid)

        for i, row in enumerate(pairings):
            couchdb_io.remove_pairing(
                row.id, i == 1, ctx=test_environment.test_context)

        pairings = list(
            couchdb_io.get_pairings(ctx=test_environment.test_context))
        self.assertEqual(0, len(pairings))

    def test_get_database_names_replicatable(self):
        """Test get the names of replicatable databases."""
        names = couchdb_io.get_database_names_replicatable(
            uri=URI, ctx=test_environment.test_context)
        self.assertFalse('management' in names)
        self.assertTrue('foo' in names)
        self.assertFalse('bar' in names, names)  # is excluded

    def test_get_my_host_unique_id(self):
        """Test get the unique id of the host record."""
        got = couchdb_io.get_my_host_unique_id(
            uri=URI, ctx=test_environment.test_context)
        again = couchdb_io.get_my_host_unique_id(
            uri=URI, ctx=test_environment.test_context)
        self.assertEquals(len(got), 1)
        self.assertEquals(got, again)

    def test_mkuri(self):
        """Test creating a URI."""
        uri = couchdb_io.mkuri(
            'fnord.org', 55241, has_ssl=True, path='a/b/c',
            auth_pair=('f o o', 'b=a=r'))
        self.assertEquals(
            'https://f%20o%20o:b%3Da%3Dr@fnord.org:55241/a/b/c', uri)
