#!/usr/bin/env python
# -*- coding: UTF-8 -*-
###########################################################################
# displayconfig.py - description                                          #
# ------------------------------                                          #
# begin     : Fri Mar 02 2007                                             #
# copyright : (C) 2007 Canonical                                          #
# email     : Michael Vogt <michael.vogt@ubuntu.com>                      #
#             Sebastian Heinlein <sebi@glatzor.de>                        #
#                                                                         #
###########################################################################
#                                                                         #
#   This program is free software; you can redistribute it and/or modify  #
#   it under the terms of the GNU General Public License as published by  #
#   the Free Software Foundation; either version 2 of the License, or     #
#   (at your option) any later version.                                   #
#                                                                         #
###########################################################################

import string
import os
import select
import sys
import csv
import time
import signal
import shutil

import pygtk
pygtk.require("2.0")
import gtk
import gtk.glade
gtk.glade.textdomain("displayconfig-gtk")
import gobject
from SimpleGladeApp import SimpleGladeApp
from widgets import *

from displayconfigcommon import _findXorgConfig, testX
from displayconfigabstraction import *

from displayconfiggtk import information 

from gettext import gettext as _
import gettext
gettext.textdomain("displayconfig-gtk")

from ConfigParser import ConfigParser

# Map secondary screen orientation to the index of available options
# in the combobox
ORIENTATION_INDEX = {XSetup.POSITION_LEFTOF : 0,
                     XSetup.POSITION_RIGHTOF :1,
                     XSetup.POSITION_BELOW :2,
                     XSetup.POSITION_ABOVE : 3}

def skip_vendor(vendor, model_name):
    if isinstance(vendor, str):
        if model_name.startswith(vendor):
            return model_name[len(vendor):].lstrip().capitalize()
        else:
            return model_name
    else:
        return model_name

class DisplayConfig(SimpleGladeApp):

    def __init__(self, options):
        """Provide a dialog that allows to configure your screens and 
           graphics cards including resolution, refresh rate, orientation,
           drivers and finally dual screen mode"""
        self.options = options
        # setup the default window icon and append your local
        # data dir as fallback for icons
        icon_theme = gtk.icon_theme_get_default()
        icon_theme.append_search_path(os.path.join(self.options.datadir,
                                                   "icons"))
        gtk.window_set_default_icon_name("video-display")

        SimpleGladeApp.__init__(self, os.path.join(self.options.datadir,
                                                   "displayconfig.glade"))

        self.store_screens = gtk.ListStore(gobject.TYPE_STRING,
                                           gtk.gdk.Pixbuf,
                                           gobject.TYPE_PYOBJECT)
        self.treeview_screens.set_model(self.store_screens)
        column_screens = gtk.TreeViewColumn(_("Screens"))
        renderer_icon = gtk.CellRendererPixbuf()
        renderer_screen = gtk.CellRendererText()
        column_screens.pack_start(renderer_icon, True)
        column_screens.pack_start(renderer_screen, True)
        column_screens.add_attribute(renderer_screen, "markup", 0)
        column_screens.add_attribute(renderer_icon, "pixbuf", 1)
        self.treeview_screens.append_column(column_screens)

        self.icon_theme = gtk.icon_theme_get_default()

        self.chooser_screen = DeviceChooser()
        self.chooser_screen.set_sensitive(False)
        self.table_screen.attach(self.chooser_screen, 1, 2, 0, 1,
                                 yoptions=gtk.FILL)
        self.chooser_screen.show()
        self.chooser_screen.connect("clicked", self.on_configure_device_clicked)
        self.store_screen_resolutions = gtk.ListStore(gobject.TYPE_STRING)
        renderer = gtk.CellRendererText()
        self.combobox_resolution.set_model(self.store_screen_resolutions)
        self.combobox_resolution.pack_start(renderer)
        self.combobox_resolution.add_attribute(renderer, "text", 0)

        self.store_screen_rates = gtk.ListStore(gobject.TYPE_STRING)
        renderer = gtk.CellRendererText()
        self.combobox_rate.set_model(self.store_screen_rates)
        self.combobox_rate.pack_start(renderer)
        self.combobox_rate.add_attribute(renderer, "text", 0)

        self.store_screen_rotations = gtk.ListStore(gobject.TYPE_STRING,
                                                    gobject.TYPE_PYOBJECT)
        renderer = gtk.CellRendererText()
        self.combobox_rotation.set_model(self.store_screen_rotations)
        self.combobox_rotation.pack_start(renderer)
        self.combobox_rotation.add_attribute(renderer, "text", 0)

        #FIXME: Add support to configure colour depth and gama correction
        #self.store_screen_colours = gtk.ListStore(gobject.TYPE_STRING)
        #renderer = gtk.CellRendererText()
        #self.combobox_colours.set_model(self.store_screen_colours)
        #self.combobox_colours.pack_start(renderer)
        #self.combobox_colours.add_attribute(renderer, "markup", 0)

        self.store_locations = gtk.ListStore(gobject.TYPE_STRING,
                                             gobject.TYPE_STRING,
                                             gobject.TYPE_STRING)
        self.store_locations.set_sort_column_id(0, gtk.SORT_ASCENDING)
        self.combobox_location.set_model(self.store_locations)

        # Setup the user interface of the monitor choser
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_("Manufacturer"), renderer, markup=0)
        self.treeview_screen_vendors.append_column(column)
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_("Model"), renderer, text=0)
        self.treeview_screen_models.append_column(column)

        # Setup the user interface of the gfxcard choser
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_("Model"), renderer, text=0)
        self.treeview_gfxcards.append_column(column)
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_("Manufacturer"), renderer, text=0)
        self.treeview_gfxcard_vendors.append_column(column)
        renderer = gtk.CellRendererText()
        renderer.set_property("ellipsize", pango.ELLIPSIZE_END)
        self.combobox_gfxcard_drivers.pack_start(renderer)
        self.combobox_gfxcard_drivers.add_attribute(renderer, "text", 2)

        # Load the location profiles
        self.locations = ConfigParser()
        self.locations.read(options.locationspath)
        self.location_config_changed = False
        for loc in self.locations.sections():
            self.store_locations.append((loc,
                                         self.locations.get(loc, "xorgconfig"),
                                         self.locations.get(loc, "pcitable")))

        # Set the data dir for the backend
        SetDataFileDir("/usr/share/apps/guidance")

        # Find a config file. If none can be found, switch to
        # fallback mode
        self.fallback_mode = False
        xconfiglive = _findXorgConfig()
        if self.options.xconfigpath:
            xconfiginit = self.options.xconfigpath
            self.xconfigpath = self.options.xconfigpath
        elif xconfiglive not in (None, "") and os.path.exists(xconfiglive):
            xconfiginit = xconfiglive
            self.xconfigpath = xconfiglive
        else:
            xconfiginit = os.path.join(options.datadir,
                                       "xorg.conf.fallback")
            self.xconfigpath = "/etc/X11/xorg.conf"
            print "fallback mode"
            self.fallback_mode = True
        # Overwrite the init 
        if self.options.failsafemode:
            print "fail safe session started"
            self.fallback_mode = True
            xconfiginit = self.options.failsafemode
        # Abstract the current configuration
        self.xsetup = XSetup(xconfiginit,
                             debug_scan_pci_filename=self.options.pcitable)
        self.gfxcarddb = None
        self.monitordb = None
        # setup all handlers manually, since we need to inhibt them
        # during sync
        self.handlers = []
        for (widget, signal, handler) in \
            [(self.combobox_resolution, "changed", 
              self.on_combobox_resolution_changed),
             (self.combobox_rate, "changed", 
              self.on_combobox_rate_changed),
             (self.combobox_rotation, "changed",
              self.on_combobox_rotation_changed),
             (self.combobox_orientation, "changed", 
              self.on_combobox_orientation_changed),
             (self.radiobutton_extend, "toggled", 
              self.on_radiobutton_extend_toggled),
             (self.radiobutton_clone, "toggled", 
              self.on_radiobutton_clone_toggled)]:
            self.handlers.append((widget, widget.connect(signal, handler)))
        self.handlers.append(
            (self.radiobutton_primary,
             self.radiobutton_primary.connect("toggled",
             self.on_radiobutton_role_toggled, XSetup.ROLE_PRIMARY)))
        self.handlers.append(
            (self.radiobutton_secondary,
             self.radiobutton_secondary.connect("toggled",
             self.on_radiobutton_role_toggled, XSetup.ROLE_SECONDARY)))
        self.handlers.append(
            (self.radiobutton_disabled,
             self.radiobutton_disabled.connect("toggled",
              self.on_radiobutton_role_toggled, XSetup.ROLE_UNUSED)))
        self.handlers.append(
            (self.treeview_gfxcards,
             self.treeview_gfxcards.connect("cursor_changed",
              self.on_treeview_gfxcards_cursor_changed)))

        # define window_main as the top level window for all dialogs
        self.dialog_confirmation.set_transient_for(self.window_main)
        self.dialog_monitor.set_transient_for(self.window_main)
        self.dialog_gfxcard.set_transient_for(self.window_main)
        self.dialog_restart.set_transient_for(self.window_main)

        # Check if we are on a laptop
        self.laptop = os.system("/usr/sbin/laptop-detect")

        # Fill the user interface with some life!
        self._sync_screens()
        self.treeview_screens.set_cursor(0)

        # Show a button on each monitor that allows to move the main
        # dialog to it
        self.move_hints = []
        display = gtk.gdk.display_get_default()
        screen = display.get_screen(0)
        self.window_main.realize()
        parent = self.window_main.window
        current_monitor = screen.get_monitor_at_window(parent)
        for m in range(0, screen.get_n_monitors()):
            if not m == current_monitor:
                monitor = screen.get_monitor_geometry(m)
                window = gtk.Window()
                window.set_decorated(False)
                button = gtk.Button(_("Move Screen Preferences here"))
                button.connect("clicked", self.move_to_monitor, monitor)
                window.add(button)
                window.realize()
                window.window.set_skip_taskbar_hint(True)
                window.window.set_skip_pager_hint(True)
                window.set_keep_above(True)
                width, height = window.get_size()
                window.move(monitor.x + monitor.width/2 - width/2,
                            monitor.y + monitor.height/2 - height/2)
                window.show_all()
                self.move_hints.append(window)
        # Hide the buttons again when the changed something in the dialog
        self.window_main.connect("set-focus", self._remove_move_hints)
        # show the main window
        self.window_main.show()

        # only enable widgets that depend on the xorg.conf if we are
        # allowed to write it
        if os.access(self.xconfigpath, os.F_OK | os.W_OK) or self.fallback_mode:
            self.button_apply.set_sensitive(True)
            self.button_test_config.set_sensitive(not self._badFbRestore())
            self.vbox_cards.set_sensitive(True)
            self.vbox_role.set_sensitive(True)
            self.hbox_location.set_sensitive(True)
            self.chooser_screen.set_sensitive(True)
        else:
            msg = _("You need administrative rights to change all "
                    "screen settings")
            dia = gtk.MessageDialog(parent=self.window_main,
                                    type=gtk.MESSAGE_ERROR,
                                    buttons=gtk.BUTTONS_CLOSE,
                                    message_format=msg)
            dia.run()
            dia.destroy()

    def _remove_move_hints(self, window=None, widget=None):
        """Remove the buttons that allow to move the main window to
           another monitor"""
        for button in self.move_hints:
            button.hide()

    def move_to_monitor(self, widget, monitor):
        """Move the main window to the center of the given rectangle"""
        self._remove_move_hints()
        width, height = self.window_main.get_size()
        self.window_main.move(monitor.x + monitor.width/2 - width/2,
                              monitor.y + monitor.height/2 - height/2)

    def _populate_monitor_models(self, force=False):
        """Initialize the dialog that allows to configure screens"""
        self.monitordb = monitordb = GetMonitorModelDB(force=force)

        # Store the vendor names and a ListStore with the corresponding
        # monitor models
        store_screen_vendors = gtk.ListStore(gobject.TYPE_STRING, 
                                             gobject.TYPE_PYOBJECT)

        self.treeview_screen_vendors.set_model(store_screen_vendors)
        # Load predefined monitors with model information
        vendors = monitordb.vendordb.keys()
        vendors.sort()
        for vendor in vendors:
            monitorkeys = self.monitordb.vendordb[vendor].keys()
            store_monitors = gtk.ListStore(gobject.TYPE_STRING, 
                                           gobject.TYPE_PYOBJECT)
            store_screen_vendors.append([gobject.markup_escape_text(vendor),
                                         store_monitors])
            monitorkeys.sort()
            for monitorkey in monitorkeys:
                store_monitors.append([skip_vendor(vendor, monitorkey),
                                       monitordb.vendordb[vendor][monitorkey]])
        # Load generic monitors
        store_generic_monitors = gtk.ListStore(gobject.TYPE_STRING, 
                                               gobject.TYPE_PYOBJECT)
        generics = monitordb.genericdb.keys()
        store_screen_vendors.insert(0, ["<b>%s</b>" % _("Generic"),
                                        store_generic_monitors])
        generics.sort()
        for generic in generics:
            store_generic_monitors.append([generic,
                                           monitordb.genericdb[generic]])
        # Load custom monitors if available
        customs = monitordb.getCustomMonitors().keys()
        for custom in customs:
            store_generic_monitors.append([custom,
                                          monitordb.getCustomMonitors()[custom]])

    def on_treeview_screen_vendors_cursor_changed(self, treeview):
        """Attach the ListStore that contains the models of the selected
           vendor to the model treeview."""
        model = treeview.get_model()
        cursor_path = treeview.get_cursor()[0]
        store_monitors = model[cursor_path][1]
        self.treeview_screen_models.set_model(store_monitors)
        self.treeview_screen_models.set_cursor(0)

    def on_treeview_monitors_cursor_changed(self, treeview):
        """Update the frequence range information of the selected monitor."""
        model = treeview.get_model()
        cursor_path = treeview.get_cursor()[0]
        mon = model[cursor_path][1]
        if mon is not None:
            self.label_vrange.set_text(mon.getVerticalSync())
            self.label_hrange.set_text(mon.getHorizontalSync())

    def on_treeview_monitors_row_activated(self, treeview, path, column):
        """Choose the activated monitor and close the dialog"""
        self.on_treeview_monitors_cursor_changed(treeview)
        self.dialog_monitor.response(gtk.RESPONSE_OK)

    def _populate_gfxcards(self):
        """Setup the dialog that allows to select the driver for a gfx card."""
        self.gfxcarddb = gfxcarddb = GfxCardModelDB()
        #Store the vendor name and a ListStore with corresponding models
        self.store_gfxcard_vendors = gtk.ListStore(gobject.TYPE_STRING,
                                                   gobject.TYPE_PYOBJECT)
        vendors = gfxcarddb.vendordb.keys()
        vendors.sort()
        for vendor in vendors:
            store_vendor_cards = gtk.ListStore(gobject.TYPE_STRING,
                                               gobject.TYPE_PYOBJECT)
            self.store_gfxcard_vendors.append([vendor, store_vendor_cards])
            cardkeys = self.gfxcarddb.vendordb[vendor].keys()
            cardkeys.sort()
            for cardkey in cardkeys:
                store_vendor_cards.append([skip_vendor(vendor,cardkey),
                                           self.gfxcarddb.vendordb[vendor][cardkey]])

        # Store all available drivers
        self.store_gfxcard_drivers = gtk.ListStore(gobject.TYPE_STRING,
                                                   gobject.TYPE_PYOBJECT,
                                                   gobject.TYPE_STRING)
        self.store_gfxcard_drivers.set_sort_column_id(2,
                                                      gtk.SORT_ASCENDING)
        drivers = gfxcarddb.driverdb.keys()
        for driver in drivers:
            if information.VIDEODRIVERS.has_key(driver):
                driver_name = "%s - %s" % (driver,
                                           information.VIDEODRIVERS[driver])
            else:
                driver_name = driver
            self.store_gfxcard_drivers.append([driver,
                                               gfxcarddb.driverdb[driver],
                                               driver_name])
        self.treeview_gfxcard_vendors.set_model(self.store_gfxcard_vendors)
        self.combobox_gfxcard_drivers.set_model(self.store_gfxcard_drivers)

    def on_button_detect_monitor_clicked(self, button):
        model = self.monitordb.detect()
        if model.getType() == MonitorModel.TYPE_NORMAL and not \
           model.getManufacturer() in ["Generic LCD Monitor",
                                       "Generic CRT Monitor"]:
            self._select_in_treeview(self.treeview_screen_vendors,
                                     model.getManufacturer(),
                                     0)
        else:
            self._select_in_treeview(self.treeview_screen_vendors,
                                     "<b>%s</b>" % _("Generic"),
                                     0)
        # Select the model of the monitor
        self._select_in_treeview(self.treeview_screen_models, model, 1)

    def on_button_import_models_clicked(self, button):
        """Provide a file chooser dialog to import monitor models
           from a Windows 'driver' file"""
        import infimport
        chooser = gtk.FileChooserDialog(title=_("Choose a driver file"),
                                        parent=self.window_main,
                                        action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                        buttons=("gtk-cancel",
                                                  gtk.RESPONSE_CANCEL,
                                                 _("_Import"),
                                                  gtk.RESPONSE_OK))
        filter = gtk.FileFilter()
        filter.add_pattern("*.inf")
        filter.add_pattern("*.INF")
        filter.set_name(_("Monitor drivers"))
        chooser.set_filter(filter)
        ret = chooser.run()
        driver_path = chooser.get_filename()
        chooser.hide()
        if ret != gtk.RESPONSE_OK:
            return
        try:
            monitors = infimport.get_monitors_from_inf(driver_path)
        except:
            monitors = {}
        if len(monitors) == 0:
            msg = _("No monitors driver have been found")
            dia = gtk.MessageDialog(parent=self.window_main,
                                    type=gtk.MESSAGE_ERROR,
                                    buttons=gtk.BUTTONS_CLOSE,
                                    message_format=msg)
            dia.run()
            dia.destroy()
            return
        monstore = gtk.ListStore(gobject.TYPE_STRING)
        for m in monitors.keys():
            monstore.append([m[1]])
        self.treeview_import.insert_column_with_attributes(0, "monitors",
                                                           gtk.CellRendererText(),
                                                           text=0)
        self.treeview_import.set_model(monstore)
        ret = self.dialog_import.run()
        self.dialog_import.hide()
        if ret != gtk.RESPONSE_OK:
            return
        infimport.append_monitors_to_file(monitors,
                                          os.path.join(var_data_dir,
                                          "CustomMonitorsDB"))
        self._populate_monitor_models(force=True)
        # Select the vendor of the new monitors
        self._select_in_treeview(self.treeview_screen_vendors,
                                 monitors.keys()[0][0],
                                 0)

    def on_treeview_gfxcards_cursor_changed(self, treeview):
        """Check if a proprietary driver is available for the selected
           model and if so allow to select it. Furthermore update the 
           choose-driver-by-name combobox."""
        model = treeview.get_model()
        # FIXME: this should not happen!
        if model:
            cursor_path = treeview.get_cursor()[0]
            card = model[cursor_path][1]
            # Default to the open source driver
            self.combobox_gfxcard_freedom.set_active(0)
            self.combobox_gfxcard_freedom.set_sensitive(False)
            # Allow to choose a proprietary driver if available
            if card is not None:
                s = card.getProprietaryDriver()
                self.combobox_gfxcard_freedom.set_sensitive(s is not None)
            # select the driver of the current model in the driver chooser
            self._select_in_combobox(self.combobox_gfxcard_drivers,
                                     card.getDriver(),
                                     0)

    def on_treeview_gfxcards_row_activated(self, treeview, path, column):
        """Choose the activated card and close the dialog"""
        self.on_treeview_gfxcards_cursor_changed(treeview)
        self.dialog_gfxcard.response(gtk.RESPONSE_OK)

    def on_treeview_gfxcard_vendors_cursor_changed(self, treeview):
        """Attach the ListStore that contains the models of the selected
           vendor to the model treeview."""
        path = treeview.get_cursor()[0]
        if path is None:
            return
        model =  treeview.get_model()
        iter = model.get_iter(path)
        (name, store_cards) = model[iter]
        self.treeview_gfxcards.set_model(store_cards)
        self.treeview_gfxcards.set_cursor(0)

    def on_radiobutton_gfxcard_model_toggled(self, button):
        """Only enable widgets that are used by the selected way of 
           choosing a gfx card driver"""
        self.hbox_gfxcard_model.set_sensitive(button.get_active())
        self.combobox_gfxcard_drivers.set_sensitive(not button.get_active())
        # update the information
        self.on_treeview_gfxcards_cursor_changed(self.treeview_gfxcards)

    def block_handlers(self):
        """Block signals of all configuration widgets"""
        for (widget, handler) in self.handlers:
            widget.handler_block(handler)
    def unblock_handlers(self):
        """Unblock signals of all configuration widgets"""
        for (widget, handler) in self.handlers:
            widget.handler_unblock(handler)

    def _sync_screens(self):
        """Update the user interface to the latest configuration"""
        path, column = self.treeview_screens.get_cursor()
        self.store_screens.clear()
        # Show a frame for each gfx card on the hardware tab
        self.vbox_cards.foreach(lambda w: self.vbox_cards.remove(w))
        for gfxcard in self.xsetup.getGfxCards():
            gfx_frame = GfxCardFrame(self.xsetup, gfxcard)
            gfx_frame.connect("configure-clicked",
                              self.on_configure_device_clicked)
            self.vbox_cards.add(gfx_frame)
            for screen in gfxcard.getScreens():
                #screen_name = get_screen_name(screen)
                screen_name = _("Screen %i") % (gfxcard.getScreens().index(screen) +1)
                role = self.xsetup.getScreenRole(screen)
                # The default screen is highlighted
                if role == XSetup.ROLE_PRIMARY:
                    screen_name = "<b>%s</b>" % screen_name
                # Use an icon to represent the device type
                if screen.getMonitorAspect()==ModeLine.ASPECT_16_9:
                    icon_name = "video-display-widescreen"
                elif self.laptop == 0 and \
                     len(self.xsetup.getGfxCards()) == 1 and \
                     screen == gfxcard.getScreens()[0]:
                    # FIXME: Perhaps too much magic :)
                    icon_name = "video-display-laptop"
                elif gfxcard.getGfxCardModel().getFbTvOut() and \
                     screen == gfxcard.getScreens()[1]:
                    icon_name = "video-display-tv"
                else:
                    icon_name = "video-display-default"
                # A small hack to get insensitive icons for disabled
                # screen devices
                icon = self.icon_theme.load_icon(icon_name,
                                                 32,
                                                 gtk.ICON_LOOKUP_NO_SVG)
                icon_source = gtk.IconSource()
                icon_source.set_pixbuf(icon.copy())
                style = self.window_main.get_style()
                # FIXME: The backend should really unset the secondary
                #        screen on single layout
                if self.xsetup.getScreenRole(screen) == XSetup.ROLE_UNUSED or\
                   (self.xsetup.getScreenRole(screen) == XSetup.ROLE_SECONDARY \
                    and self.xsetup.getLayout() == XSetup.LAYOUT_SINGLE):
                    pixbuf = style.render_icon(icon_source,
                                               gtk.TEXT_DIR_NONE,
                                               gtk.STATE_INSENSITIVE,
                                               gtk.ICON_SIZE_DND)
                else:
                    pixbuf = style.render_icon(icon_source,
                                               gtk.TEXT_DIR_NONE,
                                               gtk.STATE_NORMAL,
                                               gtk.ICON_SIZE_DND)
                # Add the screen to the store
                self.store_screens.append([screen_name,
                                           pixbuf,
                                           screen])
        if path:
            self.treeview_screens.set_cursor(path)

    def on_radiobutton_role_toggled(self, button, role):
        """Set the role of the selected screen"""
        if not button.get_active():
            return
        screen = self._get_current_screen()
        if role == XSetup.ROLE_SECONDARY:
            if self.radiobutton_extend.get_active():
                self.xsetup.setLayout(XSetup.LAYOUT_DUAL)
            else:
                self.xsetup.setLayout(XSetup.LAYOUT_CLONE)
        elif role == XSetup.ROLE_UNUSED:
            # FIXME: Should be done by the backend
            secondary = self.xsetup.getSecondaryScreen()
            if secondary and \
               self.xsetup.getScreenRole(screen) == XSetup.ROLE_PRIMARY:
                self.xsetup.setScreenRole(secondary, XSetup.ROLE_PRIMARY)
            self.xsetup.setLayout(XSetup.LAYOUT_SINGLE)
            self.xsetup.setScreenRole(screen, XSetup.ROLE_UNUSED)
        self.xsetup.setScreenRole(screen, role)
        self._sync_screens()

    def on_combobox_orientation_changed(self, box):
        """Change the orientation of the secondary screen."""
        active = box.get_active()
        self.xsetup.setDualheadOrientation(ORIENTATION_INDEX[active])

    def on_radiobutton_extend_toggled(self, radio):
        """Toggle the dual mode of the secondary screen."""
        state = radio.get_active()
        self.combobox_orientation.set_sensitive(state)
        if radio.get_active() == True:
            self.xsetup.setLayout(XSetup.LAYOUT_DUAL)

    def on_radiobutton_clone_toggled(self, radio):
        """Toggle the clone mode of the secondary screen."""
        if radio.get_active() == True:
            self.xsetup.setLayout(XSetup.LAYOUT_CLONE)

    def _select_in_combobox(self, combobox, key, col):
        """Select the row with value key in the column col in a given
           combobox"""
        def find(model, path, iter):
            if model[path][col] == key:
                combobox.set_active_iter(iter)
                return True
            return False
        model = combobox.get_model()
        if model:
            model.foreach(find)

    def _select_in_treeview(self, treeview, key, col):
        """Select the row with value key in the column col in a given
           treeview"""
        def find(model, path, iter):
            if model[path][col] == key:
                treeview.expand_to_path(path)
                treeview.scroll_to_cell(path, use_align=True, row_align=0.5)
                treeview.set_cursor(path)
                return True
            return False
        model = treeview.get_model()
        if model:
            model.foreach(find)

    def on_configure_device_clicked(self, button, device=None):
        """Show a dialog that allows to configure the given device. Device
           can be a Screen or a GfxCard."""
        # Configure a screen device
        if device == None:
            device = button.get_device()
        if isinstance(device, Screen):
            if not self.monitordb:
                self._populate_monitor_models()
            # UGLY: Only allow to detetct the monitor on the first screen
            #       since we are limited here by the external script
            self.button_detect_monitor.set_sensitive(device == self.xsetup.getScreen(0))
            model = device.getMonitorModel()
            self.checkbutton_widescreen.set_active(device.getMonitorAspect() == ModeLine.ASPECT_16_9)
            # FIXME: how do we handle unknown models PnP or Generic?
            if model:
                model_type = model.getType()
                # Select the manfacturer of the monitor. handle custom and
                # generic devices separately
                if model_type == MonitorModel.TYPE_NORMAL and not \
                    model.getManufacturer() in ["Generic LCD Monitor",
                                                "Generic CRT Monitor"]:
                    self._select_in_treeview(self.treeview_screen_vendors,
                                         model.getManufacturer(),
                                         0)
            else: 
                self._select_in_treeview(self.treeview_screen_vendors,
                                         "<b>%s</b>" % _("Generic"),
                                         0)
            # Select the model of the monitor
            self._select_in_treeview(self.treeview_screen_models,
                                     model, 1)
            # Run the dialog
            res = self.dialog_monitor.run()
            self.dialog_monitor.hide()
            if res == gtk.RESPONSE_OK:
                model = self.treeview_screen_models.get_model()
                cursor_path = self.treeview_screen_models.get_cursor()[0]
                mon = model[cursor_path][1]
                if mon is not None:
                    device.setMonitorModel(mon)
                    if self.checkbutton_widescreen.get_active():
                        device.setMonitorAspect(ModeLine.ASPECT_16_9)
                    else:
                        device.setMonitorAspect(ModeLine.ASPECT_4_3)
                    self._sync_screens()
        # Configure a gfx card device
        elif isinstance(device, GfxCard):
            if not self.gfxcarddb:
                self._populate_gfxcards()
            self.block_handlers()
            model = device.getGfxCardModel()
            self.combobox_gfxcard_freedom.set_active(0)
            self.hbox_gfxcard_model.set_sensitive(False)
            self.combobox_gfxcard_freedom.set_sensitive(False)
            self.treeview_gfxcard_vendors.set_cursor(0)
            # Try to select the current model
            self._select_in_treeview(self.treeview_gfxcard_vendors,
                                     model.getVendor(), 0)
            # Searching for the GfxCardModel in the second
            # column doesn't work :/
            card = skip_vendor(model.getVendor(), model.name)
            self._select_in_treeview(self.treeview_gfxcards,
                                     card, 0)
            self._select_in_combobox(self.combobox_gfxcard_drivers,
                                     model.getDriver(),
                                     0)
            # by default select the driver browser
            self.radiobutton_gfxcard_driver.set_active(True)
            # Run the dialog
            self.unblock_handlers()
            res = self.dialog_gfxcard.run()
            self.dialog_gfxcard.hide()
            if res == gtk.RESPONSE_OK:
                if self.radiobutton_gfxcard_driver.get_active() == True:
                    # The user specified a driver
                    driver_index = self.combobox_gfxcard_drivers.get_active()
                    driver = self.store_gfxcard_drivers[driver_index][0]
                    device.getGfxCardModel().setDriver(driver)
                else:
                    # The user has chosen a model from the list
                    model = self.treeview_gfxcards.get_model()
                    cursor_path = self.treeview_gfxcards.get_cursor()[0]
                    card = model[cursor_path][1]
                    if card is not None:
                        device.setGfxCardModel(card)
                        # Check if the proprietary driver was chosen
                        isprop = (self.combobox_gfxcard_freedom.get_active()==1)
                        device.setProprietaryDriver(isprop)
                if not self.xsetup.getLayout() & device._getAvailableLayouts():
                    self.xsetup.setLayout(XSetup.LAYOUT_SINGLE)
                self._sync_screens()

    def on_treeview_screens_cursor_changed(self, treeview):
        """Update the configuration widgets of the selected screen"""
        # Block any changes
        self.block_handlers()
        # Get the currently selected screen device
        path = treeview.get_cursor()[0]
        if path is None:
            return
        model =  treeview.get_model()
        iter = model.get_iter(path)
        (name, pixbuf, device) = model[iter]

        # Connect the device chooser to the current screen
        self.chooser_screen.set_device(device)

        # Show the resolution configuration of the selected screen
        self.store_screen_resolutions.clear()
        for res in device.getAvailableResolutions():
            # TRANS: Resolutions e.g. 640x480
            res_string = _("%i×%i") % (res[0], res[1])
            self.store_screen_resolutions.append([res_string])
        self.combobox_resolution.set_active(device.getResolutionIndex())

        self.get_rates_for_current_resolution()

        # Show the rotation configuration of the selected screen
        self.store_screen_rotations.clear()
        rotations = device.getAvailableRotations()
        rotation_index = {}
        # Disable the combobox if the rotation cannot be changed
        self.combobox_rotation.set_sensitive(rotations != Screen.RR_Rotate_0)

        if (rotations & Screen.RR_Rotate_0):
            # TRANS: Screen rotation
            self.store_screen_rotations.append([_("Normal"),
                                               Screen.RR_Rotate_0])
            rotation_index[Screen.RR_Rotate_0] = len(self.store_screen_rotations)-1
        if (rotations & Screen.RR_Rotate_90):
            # TRANS: Screen rotation
            self.store_screen_rotations.append([_("Right"),
                                               Screen.RR_Rotate_90])
            rotation_index[Screen.RR_Rotate_90] = len(self.store_screen_rotations)-1
        if (rotations & Screen.RR_Rotate_180):
            # TRANS: Screen rotation
            self.store_screen_rotations.append([_("Flipped"),
                                               Screen.RR_Rotate_180])
            rotation_index[Screen.RR_Rotate_180] = len(self.store_screen_rotations)-1
        if (rotations & Screen.RR_Rotate_270):
            # TRANS: Screen rotation
            self.store_screen_rotations.append([_("Left"),
                                               Screen.RR_Rotate_270])
            rotation_index[Screen.RR_Rotate_270] = len(self.store_screen_rotations)-1
        self.combobox_rotation.set_active(rotation_index[device.getRotation()])

        # Show the aspect ratio
        #aspect = device.getMonitorAspect()
        #self.combobox_aspect.set_active({ ModeLine.ASPECT_4_3 : 0,
#                                          ModeLine.ASPECT_16_9: 1 }[aspect])
        # Dual screens setup

        # Enable the widgets that correspond to the available configuration
        # options
        available_layouts = device.gfx_card._getAvailableLayouts()
        self.radiobutton_extend.set_sensitive(available_layouts & \
                                              XSetup.LAYOUT_DUAL)
        self.radiobutton_clone.set_sensitive(available_layouts & \
                                             XSetup.LAYOUT_CLONE)

        # Select the current layout
        current_layout = self.xsetup.getLayout()
        self.radiobutton_extend.set_active(current_layout == XSetup.LAYOUT_DUAL)
        self.radiobutton_clone.set_active(current_layout == XSetup.LAYOUT_CLONE)

        # Select the current orientation
        orient = self.xsetup.getDualheadOrientation()
        self.combobox_orientation.set_active(ORIENTATION_INDEX[orient])

        # Will enable the secondary screen mode widgets if the selected
        # screen is actually the secondary screen of a dual screen layout
        self.vbox_secondary.set_sensitive(
            current_layout != XSetup.LAYOUT_SINGLE and \
            self.xsetup.getScreenRole(device) == XSetup.ROLE_SECONDARY)
        # Will disable the secondary radio button if multiple screens
        # are not available or if the selected screen is the
        # primary screen of a single screen layout
        self.radiobutton_secondary.set_sensitive(
            len(self.store_screens) > 1 and \
            not (current_layout == XSetup.LAYOUT_SINGLE and \
                 self.xsetup.getScreenRole(device) == XSetup.ROLE_PRIMARY) and \
            available_layouts != XSetup.LAYOUT_SINGLE)
        # Only allows changing the orientation if the current screen
        # is a secodary screen
        self.combobox_orientation.set_sensitive(
            self.radiobutton_extend.get_active())
        # Will disable the disable button if the selected screen is the
        # primary screen of a single screen layout
        self.radiobutton_disabled.set_sensitive(
            not(current_layout == XSetup.LAYOUT_SINGLE and \
                self.xsetup.getScreenRole(device) == XSetup.ROLE_PRIMARY))

        # Activate the radio button that represents the role of the screen
        role = self.xsetup.getScreenRole(device)
        if role == XSetup.ROLE_PRIMARY:
            self.radiobutton_primary.set_active(True)
        elif role == XSetup.ROLE_SECONDARY and \
             current_layout in (XSetup.LAYOUT_CLONE, XSetup.LAYOUT_DUAL):
            self.radiobutton_secondary.set_active(True)
        else:
            self.radiobutton_disabled.set_active(True)
        # Allow interaction again
        self.unblock_handlers()

    def _get_current_screen(self):
        """Return the currently selected screen"""
        path = self.treeview_screens.get_cursor()[0]
        return(self.store_screens[path][2])

    def on_combobox_aspect_changed(self, combobox):
        """Change the aspect ratio of the screen. Currently not used"""
        # FIXME: Need a way to test this before, since it tends to crash
        # Get current screen
        screen = self._get_current_screen()

        # Get selected ascept and apply it
        current = combobox.get_active()
        screen.setMonitorAspect([ModeLine.ASPECT_4_3,
                                 ModeLine.ASPECT_16_9][current])

    def get_rates_for_current_resolution(self):
        """Show the available refresh rates for the current resolution"""
        device = self._get_current_screen()
        self.store_screen_rates.clear()
        for rate in device.getAvailableRefreshRates():
            # TRANS: Refresh rate e.g. 60 Hz
            self.store_screen_rates.append([_("%s Hz") % rate])
        self.combobox_rate.set_active(device.getRefreshRateIndex())

    def on_combobox_rotation_changed(self, combobox):
        """Set the screen rotation to the selected value. Changes will be
           applied later."""
        # Get current screen
        path = self.treeview_screens.get_cursor()[0]
        screen = self.store_screens[path][2]
        # Get index of active rotation
        index = combobox.get_active()
        # Set rotation
        screen.setRotation([Screen.RR_Rotate_0, Screen.RR_Rotate_90,
                            Screen.RR_Rotate_180, Screen.RR_Rotate_270][index])

    def on_combobox_rate_changed(self, combobox):
        """Set the refresh rate to the selected value"""
        # get current screen
        path = self.treeview_screens.get_cursor()[0]
        screen = self.store_screens[path][2]
        #FIXME: perhaps we should use the available rates per resolution
        #       instead of the general one
        index = combobox.get_active()
        screen.setRefreshRateIndex(index)

    def on_combobox_resolution_changed(self, combobox):
        """Update the available refresh rates"""
        # get current screen
        path = self.treeview_screens.get_cursor()[0]
        screen = self.store_screens[path][2]

        # Set the resolution
        index_res = self.combobox_resolution.get_active()
        screen.setResolutionIndex(index_res)
        self.get_rates_for_current_resolution()

    def on_button_location_save_clicked(self, button):
        """Save the current layout as a new location"""
        model = self.combobox_location.get_model()
        path = self.combobox_location.get_active()
        if path != -1:
            name = model[path][0]
            self.entry_location_name.set_text(name)
        ret = self.dialog_location_name.run()
        self.dialog_location_name.hide()
        if ret == gtk.RESPONSE_OK:
            loc_name = self.entry_location_name.get_text()
            if len(loc_name) == 0:
                return
            self.create_location_from_current(loc_name)
            try:
                pass
            except:
                msg = _("Configuration for the location could not "
                        "be saved")
                dia = gtk.MessageDialog(parent=self.window_main,
                                        type=gtk.MESSAGE_ERROR,
                                        buttons=gtk.BUTTONS_CLOSE,
                                        message_format=msg)
                dia.run()
                dia.destroy()

            self.location_config_changed = True

    def on_entry_location_name_changed(self, entry):
        """Disable the safe button if an illegal name is used"""
        text = self.entry_location_name.get_text()
        self.button_location_name_save.set_sensitive(text != "")

    def on_button_location_remove_clicked(self, button):
        """Remove the selected location"""
        model = self.combobox_location.get_model()
        path = self.combobox_location.get_active()
        name = model[path][0]
        self.locations.remove_section(name)
        iter = model.get_iter(path)
        model.remove(iter)
        self.location_config_changed = True

    def on_combobox_location_changed(self, widget):
        """Change the location to the selcted one"""
        model = self.combobox_location.get_model()
        path = self.combobox_location.get_active()
        self.button_location_remove.set_sensitive(path!=-1)
        if path != -1:
            xconfig_loc = model[path][1]
            pcitable_loc = model[path][2]
            try:
                location = XSetup(xconfig_loc,
                                  debug_scan_pci_filename=pcitable_loc)
                self.apply_location(self.xsetup, location)
            except:
                msg = _("The configuration for the location '%s' could "
                        "not be applied completely") % model[path][0]
                dia = gtk.MessageDialog(parent=self.window_main,
                                        type=gtk.MESSAGE_ERROR,
                                        buttons=gtk.BUTTONS_CLOSE,
                                        message_format=msg)
                dia.run()
                dia.destroy()
        self._sync_screens()

    def create_location_from_current(self, name):
        """Create a location from the current XSetup"""
        path = os.path.join("/var/lib/displayconfig-gtk/locations",
                            os.path.normpath(name))
        self.xsetup._syncXorgConfig()
        xorgconfig = "%s.conf" % path
        pcitable = "%s.pci" % path
        self.xsetup.xorg_config.writeConfig(xorgconfig)
        pci_bus = ScanPCI.PCIBus("/usr/share/apps/guidance")
        pci_bus.detect()
        dump = open(pcitable, "w")
        dump.write(str(pci_bus))
        dump.close()
        if not self.locations.has_section(name):
            self.locations.add_section(name)
            model = self.combobox_location.get_model()
            iter = model.append((name, xorgconfig, pcitable))
            self.combobox_location.set_active(model.get_path(iter)[0])
        self.locations.set(name, "xorgconfig", xorgconfig)
        self.locations.set(name, "pcitable", pcitable)
        self.location_config_changed = True

    def apply_location(self, orig, location):
        """Apply the screen and dual head configuration from one XSetup to
           another"""
        orig.setLayout(location.getLayout())
        orig.setDualheadOrientation(location.getDualheadOrientation())
        for card in orig.getGfxCards():
            card_loc = location.getGfxCardByPCIBusID(card.pci_id)
            if card_loc and card.detected_model == card_loc.detected_model:
                # Restore screens
                screens = card.getScreens()
                screens_loc = card_loc.getScreens()
                if not len(screens_loc) == len(screens_loc):
                    #FIXME: nicer error handling
                    raise
                for i in range(len(screens)):
                    orig.setScreenRole(screens[i], 
                                       location.getScreenRole(screens_loc[i]))
                    screens[i].setMonitorModel(screens_loc[i].getMonitorModel())
                    screens[i].setResolutionIndex(screens_loc[i].getResolutionIndex())
                    screens[i].setMonitorAspect(screens_loc[i].getMonitorAspect())
                    #FIXME: tends to crash
                    #screens[i].setRefreshRateIndex(screens_loc[i].getRefreshRateIndex())
            else:
                raise

    def on_window_main_delete_event(self, window, event):
        """Interpretate a closing of the main window as a cancel event"""
        self.on_button_cancel_clicked(None)

    def on_button_cancel_clicked(self, widget):
        gtk.main_quit()

    def on_button_apply_clicked(self, widget):
        # show a confirmation dialog
        self.apply_changes()
        gtk.main_quit()

    def on_button_test_config_clicked(self, widget):
        print "on_button_test_config_clicked()"
        (res, msg) = testX(self.xsetup,"/usr/share/displayconfig-gtk/servertestdialog-gtk")
        print res
        print msg
        if (res == False):
            dia = gtk.MessageDialog(parent=self.window_main,
                                    type=gtk.MESSAGE_ERROR,
                                    buttons=gtk.BUTTONS_CLOSE,
                                    message_format=_("Configuration test failed"))
            dia.format_secondary_text(_("Please verify the selected devices "
                                        "and configuration."))
            dia.run()
            dia.hide()
            
    def _badFbRestore(self):
        bad_fb_restore = False
        for card in self.xsetup.getGfxCards():
            bad_fb_restore = bad_fb_restore or \
                             ((card.getGfxCardModel() is not None) and \
                              card.getGfxCardModel().getBadFbRestore(card.isProprietaryDriver()))
        return bad_fb_restore

    def apply_changes(self):
        ischanged = self.xsetup.isXorgConfigChanged()
        # If we are in fallback mode we should reenable some modules:
        if self.fallback_mode:
            module_sections = self.xsetup.xorg_config.getSections("Module")
            if len(module_sections) > 0:
                modules = module_sections[0]
                for m in ["dri", "dbe", "glx", "vbe"]:
                    modules.allowModule(m)
        # If the user is root and the config has changed, save it and show a 
        # hint to restart x or the computer 
        # FIXME: Find a better way to handle user and admin separation
        #        so that a non admin user can use the dialog too
        if self.xsetup.isLiveResolutionConfigChanged() and \
           not self.options.instantapply:
            if self.xsetup.applyLiveResolutionChanges():
                # show a dialog to revert the changes
                timer = gobject.timeout_add(15000, 
                                            self.dialog_confirmation.response,
                                            gtk.RESPONSE_CANCEL)
                res = self.dialog_confirmation.run()
                self.dialog_confirmation.hide()
                gobject.source_remove(timer)
                if res == gtk.RESPONSE_OK:
                    self.xsetup.acceptLiveResolutionChanges()
                else:
                    self.xsetup.rejectLiveResolutionChanges()
                    return False
            else:
                # Nothing changed
                self.xsetup.applyLiveResolutionChanges()
            self.xsetup.acceptLiveResolutionChanges()

        # Write a changed xorg.config
        if ischanged or self.fallback_mode:
            # Backup up the current config file.
            if os.path.exists(self.xconfigpath):
                i = 1
                while os.path.exists("%s.%i" % (self.xconfigpath,i)):
                    i += 1
                shutil.copyfile(self.xconfigpath,"%s.%i" % (self.xconfigpath,i))
            request = self.xsetup.getRestartHint()
            # Write out the new config
            tmpfilename = "%s.tmp" % self.xconfigpath
            self.xsetup.writeXorgConfig(tmpfilename)
            os.rename(tmpfilename, self.xconfigpath)
            # Check if we have to restart the xserver or even the whole system
            if request == XSetup.RESTART_X and not self.options.failsafemode:
                msg = "<big><b>%s</b></big>" % \
                      _("All users must log off for the changes "
                        "to take effect")
                self.label_restart.set_label(msg)
                self.dialog_restart.run()
                # Send the USR1 signal to gdm so that it restarts the
                # X server after all users have been logged off
                if os.getenv("GDMSESSION") and \
                   os.path.exists("/var/run/gdm.pid"):
                    pid_file = open("/var/run/gdm.pid")
                    pid = int(pid_file.read())
                    os.kill(pid, 10)

            elif request == XSetup.RESTART_SYSTEM:
                msg = "<big><b>%s</b></big>" % \
                      _("You must restart your computer for the "
                        "changes to take effect")
                self.label_restart.set_label(msg)
                self.dialog_restart.run()
        # Save location profiles if changed
        if os.getuid() == 0 and self.location_config_changed:
            self.locations.write(open("/var/lib/displayconfig-gtk/locations.conf", "w"))

# vim:ts=4:sw=4:et
