# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.const import *

from elisa.plugins.pigment.graph.image import Image
from pgm.timing import implicit
from pgm.timing.implicit import AnimatedObject

import pgm
import gobject

import logging


class Range(Widget):
    """
    The range widget provides the common interface and implementation to set a
    value in a range of values.

    Emit the signals:
      - index-changed: when the set value has changed

    @ivar items_number: the number of items in the range
    @type items_number: int
    @ivar current_index: the current value, from 0 to (items_number - 1)
    @type current_index: int
    """

    __gsignals__ = {
        'index-changed' : (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
                           (gobject.TYPE_INT, gobject.TYPE_INT)),
        }

    def __init__(self):
        super(Range, self).__init__()

        self._items_number = 1
        self._current_index = 0

        self._orientation = VERTICAL

        self._cursor = None
        self._cursor_size = None
        self._cursor_position = None
        self._animated_cursor = None
        self._background = None
        self._animated_background = None

        self._set_background(Image())
        self._background.bg_color = self.style.background_color

        self._set_cursor(Image())
        self._cursor.bg_color = self.style.cursor_color

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(Range, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'background-color':
                self._animated_background.bg_color = value
            elif key == 'cursor-color':
                self._animated_cursor.bg_color = value

    def _set_background(self, background):
        if self._background:
            self.remove(self._background)

        self._background = background
        self._background.visible = True
        self.add(self._background)

        self._background.connect('pressed', self._pressed)
        self._background.connect("drag-begin", self._drag_begin)
        self._background.connect("drag-motion", self._drag_motion)
        self._background.connect("drag-end", self._drag_end)

        self._update_cursor_size()
        self._update_cursor_position()

        self._animated_background = AnimatedObject(self._background)

    def _set_cursor(self, cursor):
        if self._cursor:
            self.remove(self._cursor)

        self._cursor = cursor
        self._cursor.visible = True
        self.add(self._cursor, forward_signals=False)
        self._cursor.connect('pressed', self._pressed)
        self._cursor.connect("drag-begin", self._drag_begin)
        self._cursor.connect("drag-motion", self._drag_motion)
        self._cursor.connect("drag-end", self._drag_end)

        self._update_cursor_size()
        self._update_cursor_position()

        attrs = ('current_index', 'position')
        self._animated_cursor = implicit.AnimatedObject(self._cursor, attrs)
        settings = {'duration': 300,
                    'transformation': implicit.LINEAR,
                    'resolution': 5}
        self._animated_cursor.setup_next_animations(**settings)
        self._animated_cursor.mode = implicit.REPLACE

    def items_number__get(self):
        return self._items_number

    def items_number__set(self, nb):
        """
        Set the number of items in the range. Minumum is 1. If the range has
        shrunk too much, reset the curren index

        @param nb: the number of items
        @type nb: int
        """

        if nb <= 0:
            nb = 1

        self._items_number = nb

        if self._current_index > (nb-1):
            self._current_index = nb - 1

        self._update_cursor_size()
        self._update_cursor_position()

    items_number = property(items_number__get, items_number__set)


    def current_index__get(self):
        return self._current_index

    def current_index__set(self, position):
        """
        Set the position of the cursor. The bar is automatically new painted

        @param position: the position to set to
        @type position: int
        """
        if 0 <= position < self.items_number:
            if position != self._current_index:
                prev = self._current_index
                self._current_index = position
                self.emit('index-changed', position, prev)

    current_index = property(current_index__get, current_index__set)

    def _update_cursor_size(self):
        spacing = (self.style.spacing * 2)

        if self._orientation == VERTICAL:
            width = self.style.cursor_thickness - spacing
            height = (1.0 / self.items_number) * (self._current_index + 1)
            if self.items_number == 1:
                height -= spacing
        else:
            width = (1.0 / self.items_number) * (self._current_index + 1)
            height = self.style.cursor_thickness - spacing
            if self.items_number == 1:
                width -= spacing

        size = (width, height)

        self._cursor_size = size
        if self._animated_cursor:
            self._animated_cursor.size = self._cursor_size
        elif self._cursor:
            self._cursor.size = self._cursor_size

    def _update_cursor_position(self):
        if self._background:
            if self._current_index == 0:
                spacing = self.style.spacing
            elif self._current_index == self.items_number-1:
                spacing = - self.style.spacing
            else:
                spacing = 0

            w, h = self._cursor_size
            # FIXME: we should us subclasses for HORIZONTAL/VERTICAL split,
            # like we do in the List
            if self._orientation == VERTICAL:
                y = self.absolute_y + spacing + (h * self._current_index)
                x = self.absolute_x  + self.style.spacing \
                    - ((self.style.cursor_thickness - self._background.width) / 2.0)
            else:
                x = self.absolute_x + spacing + (w * self._current_index)
                y = self.absolute_y + self.style.spacing \
                    - ((self.style.cursor_thickness - self._background.width) / 2.0)

            self._cursor_position = (x, y, 0)
            if self._animated_cursor:
                self._animated_cursor.position = self._cursor_position
            elif self._cursor:
                self._cursor.position = self._cursor_position

    def _drag_begin(self, widget, x, y, z, button, time, pressure):
        if not self.focus:
            return True

        self._original_position = (x, y, z)
        return True

    def _pressed(self, widget, x, y, z, button, time, pressure ):
        pass

    def _drag_motion(self, widget, x, y, z, button, time, pressure):
        if not self.focus:
            return True

        self.current_index = self._get_index_from_position(x, y, z)
        return True

    def _drag_end(self, widget, x, y, z, button, time):
        if not self.focus:
            return True

        self._original_position = None
        return True

    def do_index_changed(self, current, previous):
        """Default 'index-changed' signal handler"""
        self._update_cursor_position()

    def do_scrolled(self, x, y, z, direction, time):
        if direction == pgm.SCROLL_UP:
            self.current_index -= 1
        else:
            self.current_index += 1
        return True

    def do_key_press_event(self, viewport, event, widget):
        if event.keyval == pgm.keysyms.Up:
            self.emit('scrolled', 0, 0, 0, pgm.SCROLL_UP, 0)
        elif event.keyval == pgm.keysyms.Down:
            self.emit('scrolled', 0, 0, 0, pgm.SCROLL_DOWN, 0)

    def do_focus(self, value):
        if value:
            self.state = STATE_SELECTED
        else:
            self.state = STATE_NORMAL

    def do_clicked(self, x, y, z, button, time, pressure):
        self.current_index = self._get_index_from_position(x, y, z)
        return True

    def _get_index_from_position(self, x, y, z):
        if self._orientation == VERTICAL:
            new_fraction = (y - self.absolute_y) / self.absolute_height
        else:
            new_fraction = (x - self.absolute_x) / self.absolute_width
        return int(self.items_number * new_fraction)

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        widget = cls()
        widget.items_number = 12
        widget.visible = True

        return widget


if __name__ == "__main__":
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    range = Range.demo ()
    try:
        __IPYTHON__
    except NameError:
        pgm.main()

