/** -*- C++ -*-
    @file libept/filterlist.h
    @author Peter Rockai <me@mornfall.net>
*/

#include <klistview.h>
#include <qtooltip.h>
#include <kdebug.h>

#include <apt-front/predicate/matchers.h>
#include <apt-front/predicate/combinators.h>
#include <apt-front/predicate/factory.h>
#include <apt-front/cache/entity/entity.h>
#include <apt-front/cache/component/packagetags.h>

#include <libept/lister.h>
#include <libept/extendablelist.h>

#ifndef EPT_FILTERLIST_H
#define EPT_FILTERLIST_H

namespace ept {
using namespace aptFront;

struct InterfacingPredicate
{
public:
    virtual std::string summary() const = 0;
    /* virtual std::string field( const std::string &name ) const = 0;
    virtual void setField( const std::string &name, const std::string
    &val ) = 0; */
signals:
    void changed();
    /* void parseArguments( const predicate::ArgumentList &l );
    bool operator==( const InterfacingPredicate & ) const;
    void operator()( const entity::Entity & ); */
};

template< typename T >
struct TagFilter : predicate::Implementation< T, TagFilter< T > >,
    InterfacingPredicate, cache::Observer
{
    TagFilter() {
        setupPredicate();
        observeComponent< cache::component::Tags >();
    }

    void setupPredicate() {
        m_op = predicate::Factory< T >::tag( m_match );
    }

    std::string summary() const {
        return "Tag Filter: [" + m_match.fullname() + "] "
            + m_match.shortDescription();
    }

    void parseArguments( const predicate::ArgumentList & ) {}

    bool operator==( const TagFilter &o ) const {
        return o.m_match == m_match;
    }

    bool operator()( const T &p ) {
        return m_op( p );
    }

    void setTag( entity::Tag t ) {
        m_match = t;
        m_matchName = t.fullname();
        setupPredicate();
    }

    void notifyPreRebuild( cache::component::Base * ) {
        kdDebug() << "TagFilter pre-rebuild" << endl;
        m_match = entity::Tag();
    }

    void notifyPostRebuild( cache::component::Base * ) {
        Cache &c = cache::Global::get( m_cache );
        kdDebug() << "TagFilter post-rebuild, tag = " << m_matchName << endl;
        m_match = c.tags().tagByName( m_matchName );
        setupPredicate();
    }

protected:
    predicate::Predicate< T > m_op;
    entity::Tag m_match;
    std::string m_matchName;
};

class PredicateInterface: public ItemExtender
{
    Q_OBJECT
public:
    typedef predicate::Predicate< entity::Entity > Predicate;
    PredicateInterface( QWidget *w = 0, const char *n = 0 );
    virtual Predicate predicate() = 0;
signals:
    void predicateAdd( Predicate );
    void predicateDrop( Predicate );
public slots:
    virtual void setPredicate( const Predicate &p ) {
        m_pred = p;
        predicateChanged();
    }
    virtual void widgetsChanged();
    virtual void predicateChanged() = 0;
protected:
    predicate::Predicate< entity::Entity > m_pred;
};

class FilterItem;

class FilterList : public ExtendableList
{
    Q_OBJECT
public:
    typedef predicate::Predicate< entity::Entity > Predicate;
    // typedef std::map< KListViewItem *, Predicate > Map;
    FilterList( QWidget *parent = 0, const char *name = 0 );
    typedef predicate::And< entity::Entity > And;
    void appendPredicate( Predicate p );
    void plugLister( Lister *l );
signals:
    void predicateChanged( Lister::Predicate p );
public slots:
    void setPredicate( Predicate p );
    void setHiddenPredicate( Predicate p );
protected slots:
    void editorPredicateDrop( Predicate );
    void editorPredicateAdd( Predicate );
    void contextMenu( QListViewItem *, const QPoint &, int );
    void filterMenu( const QPoint & );
    void contextActivated( int );
    void filterMenuActivated( int );
protected:
    Predicate m_pred;
    Predicate m_hidden;
    FilterItem *m_context;
    bool m_changed;
    Lister *m_lister;
    std::vector< entity::Tag > m_tagMenuMap;
};

class FilterItem : public ExtendableItem
{
public:
    ItemExtender *createExtender();
    FilterItem( FilterList *l ) 
        : ExtendableItem( l ), m_delayedDone( false )
        {
        }
    void setup() {
        ExtendableItem::setup();
        if (m_delayedDone)
            return;
        m_delayedDone = true;
        showExtender();
    }
    QString text( int ) const;
    virtual bool less( const ExtendableItem *other ) const {
        const FilterItem *o = dynamic_cast< const FilterItem * >( other );
        return downcast< InterfacingPredicate >( m_pred ).summary() <
            downcast< InterfacingPredicate >( o->m_pred ).summary();
    }
    void setPredicate( predicate::Predicate< entity::Entity > p ) {
        m_pred = p;
    }
    FilterList::Predicate predicate() const {
        return m_pred;
    }
protected:
    predicate::Predicate< entity::Entity > m_pred;
    bool m_delayedDone:1;
};

}

#endif
