/** -*- C++ -*-
    @file libept/desktoplist.cpp
    @author Peter Rockai <me@mornfall.net>
*/

#include <qimage.h>
#include <qlabel.h>
#include <qcursor.h>
#include <qcheckbox.h>
#include <kstandarddirs.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <klocale.h>
#include <kdebug.h>

#include <libept/desktoplist.h>
#include <libept/utils.h>

namespace ept {

DesktopEntry::DesktopEntry( QWidget *p, const char *n )
    : DesktopEntryUi( p, n )
{
    adjustFontSize( m_description, -1 );
}

void DesktopEntry::resize( int w, int h )
{
    // kdDebug() << "DesktopEntry::resize( " << w << ", " << h << " )" << endl;
    int margin = layout()->margin() + layout()->spacing() * 2
                 + 6 /* spacer */ + m_icon->width() + m_check->width();
    if ( m_check->isVisible() ) margin += m_check->width() + layout()->spacing();
    int hFW1 = m_name->heightForWidth( w - margin );
    int hFW2 = m_description->heightForWidth( w - margin );
    /* kdDebug() << "margin = " << margin << ", hFW1 = " << hFW1 << ", hFW2 = "
       << hFW2 << endl; */
    int height = 2 * layout()->margin() + layout()->spacing() + hFW1 + hFW2;
    if ( height < 32 + 2*layout()->margin() /* icon size + margin */ )
        height = 32 + 2*layout()->margin();
    // m_description->resize( w - margin, m_description->heightForWidth( w - margin ) );
    QWidget::resize( w, height );
}

void DesktopEntry::setItem( ExtendableItem *i ) {
    m_polished = false;
    ItemExtender::setItem( i );
    m_name->setText( u8( "<b>" ) + u8( entity().name() ) + u8( "</b>" ) );
    m_description->setText( u8( entity().description() ) );
    if ( !item()->list()->displayCheckboxes() ) {
        m_check->hide();
    }
    if ( !package().valid() || !package().hasVersion() ) {
        setEnabled( false );
    } else {
        connect( m_check, SIGNAL( toggled( bool ) ),
                 this, SLOT( toggled() ) );
        notifyPostChange();
    }
}

void DesktopEntry::mousePressEvent( QMouseEvent *e ) {
    kdDebug() << "DesktopEntry::mousePressEvent" << endl;
    e->ignore();
}

void DesktopEntry::mouseReleaseEvent( QMouseEvent *e ) {
    kdDebug() << "DesktopEntry::mouseReleaseEvent" << endl;
    e->ignore();
}

void DesktopEntry::showEvent( QShowEvent *e ) {
    DesktopEntryUi::showEvent( e );
    // DesktopEntryUi::polish();
    if ( m_polished ) return;
    item()->list()->polishing();
    kdDebug() << "polish for " << entity().name() << endl;
    QImage icon( KGlobal::iconLoader()->iconPath( entity().icon(), -32, true ) );
    if ( icon.isNull() ) {
        icon = QImage( KGlobal::dirs()->findResource( "desktopicon", entity().icon() ) );
    }
    m_icon->setPixmap( icon.isNull() ? item()->list()->emptyIcon() :
                       QPixmap( icon.smoothScale( 32, 32, QImage::ScaleMin ) ) );
    item()->list()->polished();
    m_polished = true;
}

void DesktopEntry::toggled() {
    /* if ( package().isInstalled() ) {
        if ( package().markedRemove() )
            package().markKeep();
        else
            package().markRemove();
    } else if ( package().markedInstall() )
        package().markKeep();
    else
    package().markInstall(); */

    if ( package().isInstalled() ) {
        if ( package().markedRemove() )
            item()->list()->fireRequest( package(), component::State::AKeep );
        else
            item()->list()->fireRequest( package(), component::State::ARemove );
    } else if ( package().markedInstall() )
        item()->list()->fireRequest( package(), component::State::AKeep );
    else
        item()->list()->fireRequest( package(), component::State::AInstall );
    notifyPostChange();
}

void DesktopList::polishing() {
    if ( m_polishing == 0 )
        QApplication::setOverrideCursor( QCursor( Qt::BusyCursor ) );
    m_polishing++;
    if ( m_polishing % 10 == 0 )
        kapp->processEvents();
}

void DesktopList::polished() {
    if ( m_polishing == childCount() )
        QApplication::restoreOverrideCursor();
}

void DesktopList::notifyPostChange( component::Base * ) {
    utils::Range< ExtendableItem * > r = extenders();
    for ( ; r != r.end(); ++r ) {
        dynamic_cast< DesktopEntry * >( (*r)->extender() )->notifyPostChange();
    }
}

void DesktopEntry::notifyPostChange() {
    if ( !package().valid() || !package().hasVersion() )
        return;

    m_check->blockSignals( true );
    m_check->setChecked( ( package().isInstalled()
                         || entity().package().markedInstall() )
                         && !entity().package().markedRemove() );
    m_check->blockSignals( false );
}

DesktopList::DesktopList( QWidget *parent, const char *name )
    : ExtendableList( parent, name ), m_polishing( 0 ), m_displayCheckboxes( true )
{
    observeComponent< component::State >();
    addColumn( i18n( "Application" ) );
    setResizeMode( LastColumn );
    setToggleColumn( -1 ); // no toggling, thanks
    setExtenderHighlight( true );
    m_emptyIcon = QPixmap(
        QImage(
            KGlobal::iconLoader()->iconPath( u8( "exec" ), -32, false )
            ).smoothScale( 32, 32, QImage::ScaleMin ) );
    connect( this, SIGNAL( clicked( QListViewItem *, const QPoint &, int ) ),
             this, SLOT( processClick( QListViewItem *, const QPoint &, int ) ) );
}

void DesktopList::processClick( QListViewItem *it, const QPoint &, int ) {
    kdDebug() << "DesktopList::processClick..." << endl;
    DesktopItem *i = dynamic_cast< DesktopItem * >( it );
    if ( !i ) return;
    emit showDescription( i->entity() );
}

void DesktopList::insertRange( Range r ) {
    m_range = r;
    DesktopItem *last = 0;
    int i = 0;
    QApplication::setOverrideCursor( QCursor( Qt::BusyCursor ) );
    for( ; r != r.end(); r.advance() ) {
        if ( i % 20 == 0 )
            kapp->processEvents();
        if ( !r->package( entity::Package() ).valid() )
            continue;
        DesktopItem *i = last ? new DesktopItem( this, last ) : new DesktopItem( this );
        last = i;
        i->setEntity( *r );
        ++ i;
    }
    QApplication::restoreOverrideCursor();
}

ItemExtender *DesktopItem::createExtender() {
    return new DesktopEntry();
}

DesktopItem *DesktopEntry::item() const {
    return dynamic_cast< DesktopItem * >( m_item );
}

entity::Desktop DesktopEntry::entity() const {
    return item()->entity();
}

}
