/** -*- C++ -*-
	@file libept/filterwidgets.h
	@author Peter Rockai <me@mornfall.net>
*/

#include <apt-front/cache/entity/entity.h>
#include <apt-front/cache/entity/package.h>
#include <apt-front/predicate/factory.h>

#include <libept/statefilterui.h>
#include <libept/filterlist.h>
#include <libept/lister.h>

#include <klocale.h>
#include <qlayout.h>
#include <iostream>
#include <sstream>

#ifndef EPT_STATEFILTER_H
#define EPT_STATEFILTER_H

class KLineEdit;

namespace ept {

template< typename T >
struct StateFilter : predicate::Implementation< T, StateFilter< T > >,
                     InterfacingPredicate
{
    enum Mask { Installed = 1 << 0,
                NotInstalled = 1 << 1,
                Upgradable = 1 << 2,
                Install = 1 << 3,
                Remove = 1 << 4,
                Keep = 1 << 5,
                Upgrade = 1 << 6 };

    StateFilter()
        : m_mask( 0xff ) {
        setupPredicate();
    }

    void setupPredicate() {
        predicate::Predicate< T >
            p0 = not predicate::True< T >(),
            p1 = not predicate::True< T >();
        if (m_mask & Installed)
            p0 = p0 or (predicate::Factory< T >::member( &T::isInstalled )
                        and not predicate::Factory< T >::member( &T::isUpgradable ) );
        if (m_mask & NotInstalled)
            p0 = p0 or not predicate::Factory< T >::member( &T::isInstalled );
        if (m_mask & Upgradable)
            p0 = p0 or predicate::Factory< T >::member( &T::isUpgradable );
        
        if (m_mask & Install)
            p1 = p1 or predicate::Factory< T >::member( &T::markedNewInstall );
        if (m_mask & Remove)
            p1 = p1 or predicate::Factory< T >::member( &T::markedRemove );
        if (m_mask & Keep)
            p1 = p1 or predicate::Factory< T >::member( &T::markedKeep );
        if (m_mask & Upgrade)
            p1 = p1 or predicate::Factory< T >::member( &T::markedUpgrade );
        
        m_op = p0 and p1;
    }

    std::string summary() const {
        std::ostringstream s;
        std::vector< std::string > r;
        s << "State filter: ";
        if (m_mask & Installed && m_mask & NotInstalled && m_mask & Upgradable)
            r.push_back( "Any State" );
        else {
            if (m_mask & Installed)
                r.push_back( i18n( "Installed" ) );
            if (m_mask & NotInstalled)
                r.push_back( i18n( "Not Installed" ) );
            if (m_mask & Upgradable)
                r.push_back( i18n( "Upgradable" ) );
        }
        std::copy( r.begin(), r.end(),
                   std::ostream_iterator< std::string >( s, " " ) );
        s << "; ";
        r.clear();
        if (m_mask & Install && m_mask & Remove
            && m_mask & Keep && m_mask & Upgrade)
            r.push_back( i18n( "Any Action" ) );
        else {
            if (m_mask & Install)
                r.push_back( i18n( "Install" ) );
            if (m_mask & Remove)
                r.push_back( i18n( "Remove" ) );
            if (m_mask & Keep)
                r.push_back( i18n( "Keep" ) );
            if (m_mask & Upgrade)
                r.push_back( i18n( "Upgrade" ) );
        }
        std::copy( r.begin(), r.end(),
                   std::ostream_iterator< std::string >( s, " " ) );
        return s.str();
    }

    unsigned mask() const {
        return m_mask;
    }

    void setMask( unsigned m ) {
        m_mask = m;
        setupPredicate();
    }

    void parseArguments( const predicate::ArgumentList & ) {}

    bool operator==( const StateFilter &o ) const {
        return o.m_op == m_op;
    }

    bool operator()( const T &p ) {
        return m_op( p );
    }

    virtual void reset() {
        m_mask = 0xff;
        setupPredicate();
    }

protected:
    unsigned m_mask;
    predicate::Predicate< T > m_op;
};

class StateFilterWidget : public StateFilterUi
{
    Q_OBJECT
public:
    StateFilterWidget( QWidget *parent, const char *name = 0 );
    virtual Predicate predicate();
public slots:
    void predicateChanged();
};

}

#endif
