/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: message.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "fullengine.h"
#include "memman.h"
#include "message.h"
#include "engine.h"
#include "handlemgr.h"

/*
 * Global data
 */
ui_callbacks_t * ui_callbacks = NULL;

/*
 * message_buffer has some added fudge space in the buffer to allow for the
 * header we prepend and cover any possible overrun from expanding the format
 * string.
 */
char message_buffer[MAX_USER_MESSAGE_LEN + 1024];

int engine_user_message(int    * answer,
			char * * choice_text,
			char   * message_fmt,
			...) {
	int rc = 0;

	LOG_PROC_ENTRY();

	if (ui_callbacks != NULL) {
		if (ui_callbacks->user_message != NULL) {
			va_list args;

			if (engine_mode & ENGINE_DAEMON) {
				strcpy(message_buffer, "Daemon: ");
			} else {
				strcpy(message_buffer, "Engine: ");
			}

			va_start(args, message_fmt);
			vsprintf(message_buffer + strlen(message_buffer), message_fmt, args);
			va_end(args);

			/*
			 * Put the message (and later, any answer) into the log.  Use the
			 * CRITICAL level so that the message will always go to the log,
			 * even though the message itself may not be CRITICAL.
			 */
			LOG_CRITICAL("Message is: %s\n", message_buffer);
			rc = ui_callbacks->user_message(message_buffer, answer, choice_text);

			if (rc == 0) {
				if ((answer != NULL) && (choice_text != NULL)) {
					LOG_CRITICAL("Answer is: \"%s\"\n", choice_text[*answer]);
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int plugin_user_message(plugin_record_t * plugin,
			int             * answer,
			char          * * choice_text,
			char            * message_fmt,
			...) {
	int rc = 0;

	LOG_PROC_ENTRY();

	if (ui_callbacks != NULL) {
		if (ui_callbacks->user_message != NULL) {
			va_list args;

			strcpy(message_buffer, plugin->short_name);
			strcat(message_buffer, ": ");

			va_start(args, message_fmt);
			vsprintf(message_buffer + strlen(message_buffer), message_fmt, args);
			va_end(args);

			/*
			 * Put the message (and later, any answer) into the log.  Use the
			 * CRITICAL level so that the message will always go to the log,
			 * even though the message itself may not be CRITICAL.
			 */
			LOG_CRITICAL("Message is: %s\n", message_buffer);
			rc = ui_callbacks->user_message(message_buffer, answer, choice_text);

			if (rc == 0) {
				if ((answer != NULL) && (choice_text != NULL)) {
					LOG_CRITICAL("Answer is: \"%s\"\n", choice_text[*answer]);
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int plugin_user_communication(void                * object_instance,
			      char                * message_text,
			      option_desc_array_t * options){
	int rc = 0;

	LOG_PROC_ENTRY();

	if (ui_callbacks != NULL) {
		if (ui_callbacks->user_communication != NULL) {

			task_context_t * task = engine_alloc(sizeof(task_context_t));

			if (task != NULL) {
				task_handle_t task_handle;

				/*
				 * Fill in the appropriate task fields.  The rest are
				 * left 0 (NULL).
				 */

				task->plugin = NULL;
				task->object = object_instance;
				task->action = EVMS_Task_Message;

				task->option_descriptors = options;

				/* Make a handle for the task. */
				rc = create_handle(task,
						   TASK,
						   &task_handle);

				if (rc == 0) {

					/* Call the UI callback. */
					rc = ui_callbacks->user_communication(message_text, task_handle);

				} else {
					LOG_WARNING("create_handle() returned error %d.\n", rc);
				}

				engine_free(task);

			} else {
				LOG_CRITICAL("Memory allocation of task_context_t failed.\n");
				rc = ENOMEM;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int plugin_progress(progress_t * progress) {

	int rc = ENOSYS;    /* Default return code if UI didn't provide */
			    /* a progress indicator */

	LOG_PROC_ENTRY();

	LOG_DEBUG("    id:                %d\n", progress->id);
	LOG_DEBUG("    title:             %s\n", progress->title);
	LOG_DEBUG("    description:       %s\n", progress->description);
	LOG_DEBUG("    type:              %s\n", (progress->type == DISPLAY_PERCENT) ? "DISPLAY_PERCENT" :
						 (progress->type == DISPLAY_COUNT)   ? "DISPLAY_COUNT" :
						 (progress->type == INDETERMINATE)   ? "INDETERMINATE" :
			       							       "(unknown)");
	LOG_DEBUG("    count:             %"PRIu64"\n", progress->count);
	LOG_DEBUG("    total_count:       %"PRIu64"\n", progress->total_count);
	LOG_DEBUG("    remaining_seconds: %u\n", progress->remaining_seconds);
	
	/* Pass the call to the UI's progress() function, if it provided one. */
	if (ui_callbacks != NULL) {
		if (ui_callbacks->progress != NULL) {
			rc = ui_callbacks->progress(progress);
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}
