/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: remote.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <semaphore.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <pthread.h>

#include "fullengine.h"
#include "dmonmsg.h"
#include "remote.h"
#include "flatten.h"
#include "cluster.h"
#include "engine.h"
#include "daemon.h"
#include "memman.h"
#include "message.h"
#include "internalAPI.h"
#include "commit.h"
#include "common.h"
#include "object.h"
#include "option.h"


static unsigned char msg_buf[2048];

static inline void log_buffer(void * in_buff, uint len) {

	int i;
	int byte_count = 0;

	unsigned char * byte = (unsigned char *) in_buff;
	unsigned char * p = msg_buf;

	for (i = 0; i < len; i++) {
		sprintf(p, "%02x", *byte);
		p += strlen(p);	
		byte++;
		byte_count++;
		if (byte_count % 16 == 0) {
			LOG_DEBUG("%s\n", msg_buf);
			p = msg_buf;
		} else {
			if (byte_count % 4 == 0) {
				strcat(p, " ");
				p++;
			}
		}
	}

	if (p != msg_buf) {
		LOG_DEBUG("%s\n", msg_buf);
	}
}


#define SEND_MSG(msg, rc)							\
{										\
	int retry = 5;								\
	LOG_DEBUG("Send message with command %#x of size %zu to node %s\n", 	\
		  (msg)->cmd, (msg)->size, nodeid_to_string(&((msg)->node)));	\
	do {									\
		rc = ece_funcs->send_msg(msg);					\
										\
		if (rc != 0) {							\
			if (rc == EAGAIN) {					\
				usleep(1000000);				\
				retry--;					\
				LOG_DEBUG("Retry count is %d.\n", retry);	\
			} else {						\
				LOG_SERIOUS("send_msg() to node %s returned error code %d: %s\n",	\
					    nodeid_to_string(&((msg)->node)), rc, evms_strerror(rc));	\
			}							\
		}								\
	} while ((rc == EAGAIN) && (retry > 0));				\
}


#define REQUEST_TIMEOUT 600	/* in seconds */


typedef struct talk_s {
	ece_msg_t       say;	/* the corrolator in this message is the  */
				/* identifier for this discussion.        */
	ece_msg_t       hear;
	pthread_mutex_t mutex;
	pthread_cond_t  cond;
	int             rc;
	boolean         got_response;
} talk_t;

STATIC_LIST_DECL(talk_list);
static pthread_mutex_t talk_list_mutex = PTHREAD_MUTEX_INITIALIZER;


/* Send out the "say" message in the talk_t. */
static int say(talk_t * talk) {

	int rc;
	list_element_t el;

	LOG_PROC_ENTRY();

	talk->got_response = FALSE;

	pthread_mutex_lock(&talk_list_mutex);

	el = insert_thing(&talk_list, talk, INSERT_AFTER, NULL);

	pthread_mutex_unlock(&talk_list_mutex);

	SEND_MSG(&talk->say, rc);

	if (rc != 0) {
		pthread_mutex_lock(&talk_list_mutex);

		delete_element(el);

		pthread_mutex_unlock(&talk_list_mutex);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Allocate a talk_t for the message to be sent, filling in the talk_t
 * with the paramters provided.  Send the message.
 * Note: The caller is responsible for initializing the semaphore.
 */
static talk_t * new_talk(ece_nodeid_t * node,
			 u_int32_t      cmd,
			 size_t         size,
			 void         * msg) {

	talk_t * talk;

	LOG_PROC_ENTRY();

	talk = engine_alloc(sizeof(talk_t));

	if (talk != NULL) {
		memcpy(&talk->say.node, node, sizeof(talk->say.node));
		talk->say.cmd = cmd;
		talk->say.size = size;
		talk->say.msg = msg;
		pthread_mutex_init(&talk->mutex, NULL);
		pthread_cond_init(&talk->cond, NULL);
	}

	LOG_PROC_EXIT_PTR(talk);
	return talk;
}


static void handle_response(const ece_msg_t * msg) {

	talk_t * talk = NULL;
	list_element_t iter;

	LOG_PROC_ENTRY();
	
	pthread_mutex_lock(&talk_list_mutex);
	LIST_FOR_EACH(&talk_list, iter, talk) {
		if ((talk->say.corrolator == msg->corrolator)  &&
		    (memcmp(&talk->say.node, &msg->node, sizeof(msg->node)) == 0)) {
			delete_element(iter);
			break;
		}
	}
        pthread_mutex_unlock(&talk_list_mutex);

	if (talk != NULL) {
		talk->hear = *msg;

		if (msg->size != 0) {

			if (msg->cmd & COMMAND_RESPONSE) {
				/*
				 * First u_int32_t in the response buffer is always
				 * the return code (except for the MSG_GET_VERSION
				 * command which has no return code).
				 */
				if ((msg->cmd & COMMAND_MASK) != MSG_GET_VERSION) {
					talk->rc = NET_TO_HOST32(*((u_int32_t *) msg->msg));

				} else {
					talk->rc = 0;
				}

			} else {
				/* Incoming request */
				talk->rc = 0;
			}

			/*
			 * If the command failed then all that was sent back is
			 * a return code.  Don't copy the response buffer if
			 * all we got was a bad return code.
			 */

			if (talk->rc == 0) {
				talk->hear.msg = engine_alloc(msg->size);

				if (talk->hear.msg != NULL) {
					memcpy(talk->hear.msg, msg->msg, msg->size);
				} else {
					/* Couldn't allocate memory for a response. */
					LOG_CRITICAL("Could not allocate memory to copy the response data.\n");
					talk->rc = ENOMEM;

					engine_free(talk->hear.msg);
					talk->hear.msg = NULL;
				}

			} else {
				/*
				 * Clear the msg pointer so other functions
				 * don't try to use the ECE delivery buffer.
				 */
				talk->hear.msg = NULL;
			}

		} else {
			/*
			 * Remote node couldn't even allocate memory
			 * for a response.
			 */
			LOG_CRITICAL("Node %s returned an empty message.\n", nodeid_to_string(&msg->node));
                        talk->rc = ENOSYS;
		}

		pthread_mutex_lock(&talk->mutex);
		talk->got_response = TRUE;
		pthread_cond_signal(&talk->cond);
		pthread_mutex_unlock(&talk->mutex);

	} else {
		LOG_WARNING("Could not find a talk_t for corrolator %d.\n", msg->corrolator);
	}

	LOG_PROC_EXIT_VOID();
}


/*
 * Given a talk_t, wait for the condition to be signaled,
 * then return the response in the talk.
 */
static int wait_for_response(talk_t * talk) {

	int rc = 0;

	LOG_PROC_ENTRY();

	pthread_mutex_lock(&talk->mutex);
	if (!talk->got_response) {
		struct timespec timeout;
		struct timeval now;
		struct timezone tz;

		gettimeofday(&now, &tz);
		timeout.tv_sec = now.tv_sec + REQUEST_TIMEOUT;
		timeout.tv_nsec = 0;
		rc = pthread_cond_timedwait(&talk->cond, &talk->mutex, &timeout);
		if (rc != 0) {
			talk->rc = rc;
		}
	}
        pthread_mutex_unlock(&talk->mutex);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

#define FROM_PREFIX	"From node "
#define FROM_PREFIX_LEN	(sizeof(FROM_PREFIX) - 1)
#define SEPARATOR	": "
#define SEPARATOR_LEN	(sizeof(FROM_PREFIX) - 1)

static int msg_user_msg(talk_t * talk) {
	
	int rc;
	const char * node_name = nodeid_to_string(&talk->hear.node);
	char * message_text;
	char * new_message_text;
	int * answer;
	char * * choices = NULL;
	u_int32_t response[2];

	LOG_PROC_ENTRY();
	
	talk->say.node       = talk->hear.node;
	talk->say.corrolator = talk->hear.corrolator;
        talk->say.cmd        = talk->hear.cmd | COMMAND_RESPONSE;
	talk->say.msg        = response;

	rc = evms_net_to_host(talk->hear.msg, user_message_args_f,
			      &message_text,
			      &answer,
			      &choices);

	if (rc == 0) {
		/*
		 * Put the message (and later, any answer) into the log.  Use the
		 * CRITICAL level so that the message will always go to the log,
		 * even though the message itself may not be CRITICAL.
		 */
		new_message_text = engine_alloc(FROM_PREFIX_LEN +
						strlen(node_name) +
						SEPARATOR_LEN +
						strlen(message_text) +
						1);

		if (new_message_text != NULL) {
			strcpy(new_message_text, FROM_PREFIX);
			strcat(new_message_text, node_name);
			strcat(new_message_text, SEPARATOR);
			strcat(new_message_text, message_text);

			LOG_CRITICAL("Message is: %s\n", new_message_text);
			rc = ui_callbacks->user_message(new_message_text, answer, choices);

			engine_free(new_message_text);

		} else {
			/*
			 * Couln't get memory for building a new message.
			 * Use the original message.
			 */
			LOG_CRITICAL("Message is: %s\n", message_text);
			rc = ui_callbacks->user_message(message_text, answer, choices);
		}

		if (rc == 0) {
			if ((&answer != NULL) && (choices != NULL)) {
				LOG_CRITICAL("Answer is: \"%s\"\n", choices[*answer]);
			}
		}

		if (answer != NULL) {
			evms_host_to_net(response, user_message_rets_f,
					 rc,
					 *answer);
		} else {
			evms_host_to_net(response, user_message_rets_f,
					 rc,
					 0);
		}

		talk->say.size = sizeof(response);

	} else {
		evms_host_to_net(response, int_f,
				 rc);

		talk->say.size = sizeof(u_int32_t);
	}

	engine_free(talk->hear.msg);
	talk->hear.msg = NULL;

	rc = say(talk);

	engine_free(choices);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int msg_progress(talk_t * talk) {

	int rc;
	progress_t * progress = NULL;
	u_int32_t response[3];

	LOG_PROC_ENTRY();
	
	talk->say.node       = talk->hear.node;
	talk->say.corrolator = talk->hear.corrolator;
        talk->say.cmd        = talk->hear.cmd | COMMAND_RESPONSE;
	talk->say.msg        = response;

	rc = evms_net_to_host(talk->hear.msg, user_progress_args_f,
			      &progress);

	if (rc == 0) {
		rc = plugin_progress(progress);

		evms_host_to_net(response, user_progress_rets_f,
				 rc,
				 progress->id,
				 progress->ui_private_data);

		talk->say.size = sizeof(response);

	} else {
		evms_host_to_net(response, int_f,
				 rc);

		talk->say.size = sizeof(u_int32_t);
	}

	engine_free(talk->hear.msg);
	talk->hear.msg = NULL;

	rc = say(talk);

	engine_free(progress);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int msg_status(talk_t * talk) {

	int rc;
	char * status = (char *) talk->hear.msg;

	LOG_PROC_ENTRY();

	status_message(status);

	talk->say.node       = talk->hear.node;
	talk->say.corrolator = talk->hear.corrolator;
	talk->say.cmd        = talk->hear.cmd | COMMAND_RESPONSE;
	talk->say.size       = 0;
	talk->say.msg        = NULL;

	engine_free(talk->hear.msg);
	talk->hear.msg = NULL;

	rc = say(talk);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int handle_callback(talk_t * talk) {

	int rc = 0;

	LOG_PROC_ENTRY();

	switch (talk->hear.cmd) {
		case USER_MESSAGE:
			rc = msg_user_msg(talk);
			break;

		case PROGRESS:
			rc = msg_progress(talk);
			break;

		case STATUS:
			rc = msg_status(talk);
			break;

		default:
			LOG_WARNING("Node %s sent invalid command %d (%#x).\n",
				    nodeid_to_string(&talk->hear.node), talk->hear.cmd, talk->hear.cmd);

			engine_free(talk->hear.msg);
			talk->hear.msg = NULL;

			talk->say.cmd = MSG_INVALID_CMD | COMMAND_RESPONSE;
			talk->say.size = 0;

			rc = say(talk);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Send a message and wait for a reply.  Return the response.
 * Note: The caller is responsible for initializing the semaphore.
 */
static void * transact_message(ece_nodeid_t * nodeid,
			       u_int32_t      cmd,
			       u_int32_t      size,
			       void         * msg,
			       int          * p_rc) {

	int rc;
	talk_t * talk;
        void * response = NULL;

	LOG_PROC_ENTRY();

	talk = new_talk(nodeid, cmd, size, msg);

	if (talk != NULL) {
		rc = say(talk);

		if (rc == 0) {
			do {
				wait_for_response(talk);

				if (talk->rc == 0) {
					if (!(talk->hear.cmd & COMMAND_RESPONSE)) {
						rc = handle_callback(talk);
					}

				} else {
					rc = talk->rc;
				}

			} while ((rc == 0) && ((talk->hear.cmd & COMMAND_MASK) != cmd));
		}

	} else {
		LOG_CRITICAL("Error getting memory for a talk_t.\n");
		rc = ENOMEM;
	}

	*p_rc = rc;

	if (rc == 0) {
		response = talk->hear.msg;

	} else {
		response = NULL;
	}

	engine_free(talk);

	LOG_PROC_EXIT_PTR(response);
	return response;
}


static int make_user_ha(handle_array_t *   my_ha,
			handle_array_t * * user_ha) {

	int rc = 0;
	handle_array_t * ha;
	uint size;

	LOG_PROC_ENTRY();

	if (my_ha == NULL) {
		*user_ha = NULL;
		return rc;
	}

	size = sizeof(handle_array_t) + (my_ha->count - 1) * sizeof(object_handle_t);

	ha = alloc_app_struct(size, NULL);

	if (ha != NULL) {
		memcpy(ha, my_ha, size);
		*user_ha = ha;

	} else {
		LOG_CRITICAL("Error getting memory for a user handle array.\n");
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int one_handle_arg(msg_cmd_t cmd, object_handle_t thing) {

	int rc;
	u_int32_t net_thing;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_thing, object_handle_f,
			 thing);

	response = transact_message(current_nodeid, cmd, sizeof(net_thing), &net_thing, &rc);

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int two_handle_arg(msg_cmd_t cmd, object_handle_t thing1, object_handle_t thing2) {

	int rc;
	u_int32_t net_things[2];
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_things, object_handle_f object_handle_f,
			 thing1,
			 thing2);

	response = transact_message(current_nodeid, cmd, sizeof(u_int32_t) * 2, &net_things, &rc);

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


typedef struct get_version_thread_args_s {
	talk_t          * talk;
	uint            * count;
	pthread_mutex_t * mutex;
	pthread_cond_t  * cond;
} get_version_thread_args_t;


static void * get_version_thread(void * arg) {

        get_version_thread_args_t * gvta = (get_version_thread_args_t *) arg;
	talk_t * talk = gvta->talk;
	int rc;

	LOG_PROC_ENTRY();

	talk->rc = ETIMEDOUT;

	/*
	 * Use the mutex to make sure only one thread is doing a send at a
	 * time.  The HA ECE can only handle one at a time.
	 */
	pthread_mutex_lock(gvta->mutex);
	rc = say(talk);
	pthread_mutex_unlock(gvta->mutex);

	if (rc == 0) {
		wait_for_response(talk);
	} else {
		talk->rc = rc;
	}

	pthread_mutex_lock(gvta->mutex);
	*(gvta->count) -= 1;
	if (*(gvta->count) == 0) {
		pthread_cond_signal(gvta->cond);
        }
	pthread_mutex_unlock(gvta->mutex);

	LOG_PROC_EXIT_VOID();
	return NULL;
}


int remote_verify_version(void) {

	int rc = 0;
	int i;
	get_version_thread_args_t * gvta;
	STATIC_LIST_DECL(version_list);
	int response_count;
	pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
	pthread_cond_t cond = PTHREAD_COND_INITIALIZER;
	list_element_t iter;
	list_element_t iter1;
	list_element_t iter2;

	LOG_PROC_ENTRY();

	/*
	 * Make sure the other nodes are running the same
	 * daemon protocol version.
	 */
	response_count = membership->num_entries;

	for (i = 0; (i < membership->num_entries) && (rc == 0); i++) {

		gvta = engine_alloc(sizeof(*gvta));
		if (gvta != NULL) {
			gvta->count = &response_count;
			gvta->mutex = &mutex;
			gvta->cond  = &cond;

			gvta->talk = new_talk(&membership->node[i], MSG_GET_VERSION, 0, NULL);

			if (gvta->talk != NULL) {
				insert_thing(&version_list, gvta, INSERT_AFTER, NULL);

			} else {
				engine_free(gvta);
				rc = ENOMEM;
			}

		} else {
			rc = ENOMEM;
		}

	}

	if (rc != 0) {
		LIST_FOR_EACH_SAFE(&version_list, iter1, iter2, gvta) {
			engine_free(gvta->talk);
			engine_free(gvta);
			delete_element(iter);
		}
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(&version_list, iter, gvta) {
                pthread_t tid;

		pthread_create(&tid, &pthread_attr_detached, get_version_thread, gvta);
	}

	pthread_mutex_lock(&mutex);
	if (response_count != 0) {
		struct timespec timeout;
		struct timeval now;
		struct timezone tz;

		gettimeofday(&now, &tz);
		timeout.tv_sec = now.tv_sec + REQUEST_TIMEOUT;
		timeout.tv_nsec = 0;
		pthread_cond_timedwait(&cond, &mutex, &timeout);
	}
	pthread_mutex_unlock(&mutex);

	LIST_FOR_EACH(&version_list, iter, gvta) {
		int tmp_rc;
		talk_t * talk = gvta->talk;
		evms_version_t * version = (evms_version_t *) (talk->hear.msg);

		tmp_rc = talk->rc;

		LOG_DEBUG("Status from node %s is %d: %s\n",
			  nodeid_to_string(&talk->say.node), tmp_rc, evms_strerror(tmp_rc));

		if (tmp_rc == 0) {
			evms_net_to_host(version, evms_version_f,
					 &version->major,
					 &version->minor,
					 &version->patchlevel);

			tmp_rc = check_version(*version, engine_daemon_msg_version);

			if (tmp_rc != 0) {
				if (tmp_rc < 0) {
					engine_user_message(NULL, NULL, "The Engine on node %s is running daemon protocol version (%d.%d.%d) which is greater than this Engine's daemon protocol version (%d.%d.%d).\n",
							    nodeid_to_string(&talk->say.node),
							    version->major, version->minor, version->patchlevel,
							    engine_daemon_msg_version.major, engine_daemon_msg_version.minor, engine_daemon_msg_version.patchlevel);

				} else {
					engine_user_message(NULL, NULL, "The Engine on node %s is running daemon protocol version (%d.%d.%d) which is less than this Engine's daemon protocol version (%d.%d.%d).\n",
							    nodeid_to_string(&talk->say.node),
							    version->major, version->minor, version->patchlevel,
							    engine_daemon_msg_version.major, engine_daemon_msg_version.minor, engine_daemon_msg_version.patchlevel);
				}

				tmp_rc = EPROTONOSUPPORT;
			}

		} else {
			engine_user_message(NULL, NULL,
					    "There was an error when getting the version of the daemon protocol on node %s.  "
					    "The error code was %d: %s.\n",
					    nodeid_to_string(&talk->say.node), tmp_rc, evms_strerror(tmp_rc));
		}

		/* Use the first bad error code. */
		if (rc == 0) {
			rc = tmp_rc;
		}
	}

	if (rc == 0) {

		/*
		 * Make sure the other nodes are running the same
		 * Engine API version.
		 */
		response_count = list_count(&version_list);

		LIST_FOR_EACH(&version_list, iter, gvta) {
			pthread_t tid;
			engine_free(gvta->talk->hear.msg);
			gvta->talk->say.cmd = EVMS_GET_API_VERSION;
			pthread_create(&tid, &pthread_attr_detached, get_version_thread, gvta);
		}

		pthread_mutex_lock(&mutex);
		if (response_count != 0) {
			struct timespec timeout;
			struct timeval now;
			struct timezone tz;

			gettimeofday(&now, &tz);
			timeout.tv_sec = now.tv_sec + REQUEST_TIMEOUT;
			timeout.tv_nsec = 0;
			rc = pthread_cond_timedwait(&cond, &mutex, &timeout);
		}
		pthread_mutex_unlock(&mutex);

		LIST_FOR_EACH(&version_list, iter, gvta) {
			int tmp_rc;
			talk_t * talk = gvta->talk;
			evms_version_t * version = (evms_version_t *) (talk->hear.msg);

			tmp_rc = talk->rc;

			LOG_DEBUG("Status from node %s is %d: %s\n",
				  nodeid_to_string(&talk->say.node), tmp_rc, evms_strerror(tmp_rc));

			if (tmp_rc == 0) {
				evms_net_to_host(version, evms_get_api_version_rets_f,
						 &tmp_rc,
						 &version->major,
						 &version->minor,
						 &version->patchlevel);

				tmp_rc = check_version(*version, engine_api_version);

				if (tmp_rc != 0) {
					if (tmp_rc < 0) {
						engine_user_message(NULL, NULL, "The Engine on node %s is at version (%d.%d.%d) which is greater than this Engine's version (%d.%d.%d).\n",
								    nodeid_to_string(&talk->say.node),
								    version->major, version->minor, version->patchlevel,
								    engine_version.major, engine_version.minor, engine_version.patchlevel);

					} else {
						engine_user_message(NULL, NULL, "The Engine on node %s is at version (%d.%d.%d) which is less than this Engine's version (%d.%d.%d).\n",
								    nodeid_to_string(&talk->say.node),
								    version->major, version->minor, version->patchlevel,
								    engine_version.major, engine_version.minor, engine_version.patchlevel);
					}

					tmp_rc = EPROTONOSUPPORT;
				}

			} else {
				engine_user_message(NULL, NULL,
						    "There was an error when getting the version of the daemon protocol on node %s.  "
						    "The error code was %d: %s.\n",
						    nodeid_to_string(&talk->say.node), tmp_rc, evms_strerror(tmp_rc));
			}

			/* Use the first bad error code. */
			if (rc == 0) {
				rc = tmp_rc;
			}
		}
	}

	LIST_FOR_EACH_SAFE(&version_list, iter1, iter2, gvta) {
		engine_free(gvta->talk->hear.msg);
		engine_free(gvta->talk);
		engine_free(gvta);
		delete_element(iter1);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


typedef struct open_engine_thread_args_s {
	talk_t          * talk;
	uint            * count;
	list_anchor_t     callback_list;
	pthread_mutex_t * mutex;
	pthread_cond_t  * cond;
} open_engine_thread_args_t;


static void * open_engine_thread(void * arg) {

        open_engine_thread_args_t * oeta = (open_engine_thread_args_t *) arg;
	talk_t * talk = oeta->talk;
	int rc;

	LOG_PROC_ENTRY();

	talk->rc = ETIMEDOUT;

	/*
	 * Use the mutex to make sure only one thread is doing a send at a
	 * time.  The HA ECE can only handle one at a time.
	 */
	pthread_mutex_lock(oeta->mutex);
	rc = say(talk);
	pthread_mutex_unlock(oeta->mutex);

	if (rc == 0) {
		wait_for_response(talk);
	} else {
		talk->rc = rc;
	}

	while ((talk->rc == 0) && !(talk->hear.cmd & COMMAND_RESPONSE)) {
		/* It's an incoming callback. */

		if (talk->hear.cmd == STATUS) {
			/*
			 * We don't present status from the other nodes when the
			 * Engine is opening.  Status messages from multiple
			 * nodes getting mixed together would be very confusing.
			 */
			engine_free(talk->hear.msg);
			talk->hear.msg = NULL;

			talk->say.cmd  = talk->hear.cmd | COMMAND_RESPONSE;
			talk->say.size = 0;
			talk->say.msg  = NULL;

			talk->rc = say(talk);

		} else {
			/* It's an incoming callback. */

			/*
			 * Turn off the talk's got_response flag so that the
			 * wait_for_response() below does not think it got the
			 * response again and exits causing this while loop to
			 * proecess the response again.
			 */
			talk->got_response = FALSE;

			/*
			 * Put the talk on the callback list and poke the main
			 * thread to process it.
			 */
			pthread_mutex_lock(oeta->mutex);
			insert_thing(oeta->callback_list, talk, INSERT_AFTER, NULL);
			pthread_cond_signal(oeta->cond);
			pthread_mutex_unlock(oeta->mutex);
		}

		wait_for_response(talk);
	}

	pthread_mutex_lock(oeta->mutex);
	*(oeta->count) -= 1;
	if (*(oeta->count) == 0) {
		pthread_cond_signal(oeta->cond);
        }
	pthread_mutex_unlock(oeta->mutex);

	LOG_PROC_EXIT_VOID();
	return NULL;
}


int remote_open_engine(char           * node_name,
		       engine_mode_t    mode,
		       ui_callbacks_t * ui_callbacks,
		       debug_level_t    level,
                       char           * log_name) {

	int rc = 0;
	int i;
	size_t arg_size;
	void * net_args;
	STATIC_LIST_DECL(open_engine_list);
	open_engine_thread_args_t * oeta;
        int response_count = 0;
	STATIC_LIST_DECL(callback_list);
	pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
	pthread_cond_t cond = PTHREAD_COND_INITIALIZER;
	list_element_t iter;
	list_element_t iter1;
	list_element_t iter2;

	LOG_PROC_ENTRY();

	status_message("Starting EVMS on the other nodes in the cluster...\n");

	rc = evms_sizeof_host_to_net(&arg_size, evms_open_engine_args_f,	
				     node_name,
				     mode,
				     ui_callbacks,
				     level,
				     log_name);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_open_engine_args_f,
			 node_name,
			 mode,
			 ui_callbacks,
			 level,
			 log_name);

	response_count = membership->num_entries - 1;

	for (i = 0; (i < membership->num_entries) && (rc == 0); i++) {

		/* Don't send an open_engine message to myself. */
		if (memcmp(&membership->node[i], my_nodeid, sizeof(ece_nodeid_t)) != 0) {

			oeta = engine_alloc(sizeof(*oeta));
			if (oeta != NULL) {
				oeta->count         = &response_count;
				oeta->callback_list = &callback_list;
				oeta->mutex         = &mutex;
				oeta->cond          = &cond;

				oeta->talk = new_talk(&membership->node[i], EVMS_OPEN_ENGINE, arg_size, net_args);

				if (oeta->talk != NULL) {
					insert_thing(&open_engine_list, oeta, INSERT_AFTER, NULL);

				} else {
					engine_free(oeta);
					rc = ENOMEM;
				}

			} else {
				rc = ENOMEM;
			}
		}
	}

	if (rc != 0) {
		LIST_FOR_EACH_SAFE(&open_engine_list, iter1, iter2, oeta) {
			engine_free(oeta->talk);
			engine_free(oeta);
			delete_element(iter);
		}
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(&open_engine_list, iter, oeta) {
                pthread_t tid;

		pthread_create(&tid, &pthread_attr_detached, open_engine_thread, oeta);
	}

	/*
	 * The open engine threads can wakeup this main thread to process
	 * callbacks. This thread will be woken up either to process a
	 * callback that has been placed on the callbuack_list or when
	 * all of the threads have completed and the count has gone to zero.
	 */
	pthread_mutex_lock(&mutex);
	while ((response_count != 0) && (rc == 0)) {
		struct timespec timeout;
		struct timeval now;
		struct timezone tz;

		gettimeofday(&now, &tz);
		timeout.tv_sec = now.tv_sec + REQUEST_TIMEOUT;
		timeout.tv_nsec = 0;
		rc = pthread_cond_timedwait(&cond, &mutex, &timeout);
		if (rc != 0) {
			break;
		}

		if (response_count != 0) {
			/* Must have been woken up to process a callback. */

			while (!list_empty(&callback_list)) {
				talk_t * talk;
				list_element_t el;

				talk = first_thing(&callback_list, &el);
				delete_element(el);
				pthread_mutex_unlock(&mutex);

				handle_callback(talk);

				pthread_mutex_lock(&mutex);
			}

		}
	}
	pthread_mutex_unlock(&mutex);

	LIST_FOR_EACH(&open_engine_list, iter, oeta) {
		int tmp_rc;
		talk_t * talk = oeta->talk;

		tmp_rc = talk->rc;

		LOG_DEBUG("Status from node %s is %d: %s\n",
			  nodeid_to_string(&talk->say.node), tmp_rc, evms_strerror(tmp_rc));

		if (tmp_rc == 0) {
			evms_net_to_host(talk->hear.msg, evms_open_engine_rets_f,
					 &tmp_rc);
		}

		/* Use the first bad error code. */
		if (rc == 0) {
			rc = tmp_rc;
		}
	}

	LIST_FOR_EACH_SAFE(&open_engine_list, iter1, iter2, oeta) {
		engine_free(oeta->talk->hear.msg);
		engine_free(oeta->talk);
		engine_free(oeta);
		delete_element(iter1);
	}

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_plugin_list(evms_plugin_code_t    type,
			   plugin_search_flags_t flags,
			   handle_array_t    * * plugin_handle_list) {
	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_get_plugin_list_args_f,
				     type,
				     flags);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_get_plugin_list_args_f,
			 type,
			 flags);

	response = transact_message(current_nodeid, EVMS_GET_PLUGIN_LIST, arg_size, net_args, &rc);

	if (rc == 0) {
		handle_array_t * ha;

		evms_net_to_host(response, evms_get_plugin_list_rets_f,
				 &rc,
				 &ha);

		if (rc == 0) {
			rc = make_user_ha(ha, plugin_handle_list);

		} else {
			*plugin_handle_list = NULL;
		}

		engine_free(ha);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_plugin_by_ID(plugin_id_t       plugin_ID,
			    plugin_handle_t * plugin_handle) {
	int rc = 0;
	u_int32_t net_plugin_ID;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_plugin_ID, evms_get_plugin_by_ID_args_f,
			 plugin_ID);

	response = transact_message(current_nodeid, EVMS_GET_PLUGIN_BY_ID, sizeof(net_plugin_ID), &net_plugin_ID, &rc);

	if (rc == 0) {
		evms_net_to_host(response, evms_get_plugin_by_ID_rets_f,
				 &rc,
				 plugin_handle);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_plugin_by_name(char            * plugin_name,
			      plugin_handle_t * plugin_handle) {
	int rc = 0;
	void * response;

	LOG_PROC_ENTRY();

	/*
	 * Strings get passed as-is on the wire.  No need to convert
	 * the plugin_name to net format.  just send the name.
	 */
	response = transact_message(current_nodeid, EVMS_GET_PLUGIN_BY_NAME, strlen(plugin_name) + 1, plugin_name, &rc);

	if (rc == 0) {
		evms_net_to_host(response, evms_get_plugin_by_name_rets_f,
				 &rc,
				 plugin_handle);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_changes_pending(boolean                 * result,
			   change_record_array_t * * p_changes) {

	int rc = 0;
	void * response;

	LOG_PROC_ENTRY();

	response = transact_message(current_nodeid, EVMS_CHANGES_PENDING, 0, NULL, &rc);

	if (rc == 0) {
		change_record_array_t * cra;

		evms_net_to_host(response, evms_changes_pending_rets_f,
				 &rc,
				 result,
				 &cra);

		if (p_changes != NULL) {
			if ((rc == 0) && (cra != NULL)) {
				change_record_array_t * user_cra;
				uint size = sizeof(change_record_array_t) + sizeof(change_record_t) * cra->count;

				user_cra = alloc_app_struct(size, free_changes_pending_record_array_contents);

				if (user_cra != NULL) {
					memcpy(user_cra, cra, size);
					*p_changes = user_cra;

				} else {
					LOG_CRITICAL("Error getting memory for a handle_object_info_t.\n");
					rc = ENOMEM;
					*p_changes = NULL;
				}

			} else {
				*p_changes = NULL;
			}
		}

		engine_free(cra);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


typedef struct set_debug_level_thread_args_s {
	talk_t          * talk;
	uint            * count;
	pthread_mutex_t * mutex;
	pthread_cond_t  * cond;
} set_debug_level_thread_args_t;


static void * set_debug_level_thread(void * arg) {

        set_debug_level_thread_args_t * sdlta = (set_debug_level_thread_args_t *) arg;
	talk_t * talk = sdlta->talk;
	int rc;

	LOG_PROC_ENTRY();

	talk->rc = ETIMEDOUT;

	/*
	 * Use the mutex to make sure only one thread is doing a send at a
	 * time.  The HA ECE can only handle one at a time.
	 */
	pthread_mutex_lock(sdlta->mutex);
	rc = say(talk);
	pthread_mutex_unlock(sdlta->mutex);

	if (rc == 0) {
		wait_for_response(talk);
	} else {
		talk->rc = rc;
	}

	pthread_mutex_lock(sdlta->mutex);
	*(sdlta->count) -= 1;
	if (*(sdlta->count) == 0) {
		pthread_cond_signal(sdlta->cond);
        }
	pthread_mutex_unlock(sdlta->mutex);

	LOG_PROC_EXIT_VOID();
	return NULL;
}


int remote_set_debug_level(debug_level_t level) {

	int rc = 0;
	int i;
	u_int32_t net_debug_level;
        STATIC_LIST_DECL(set_debug_level_list);
	set_debug_level_thread_args_t * sdlta;
	pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
	pthread_cond_t cond = PTHREAD_COND_INITIALIZER;
	int response_count = 0;
	list_element_t iter;
	list_element_t iter1;
	list_element_t iter2;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_debug_level, evms_set_debug_level_args_f,
			 level);

	response_count = membership->num_entries - 1;

	for (i = 0; (i < membership->num_entries) && (rc == 0); i++) {

		/* Don't send a set debug level message to myself. */
		if (memcmp(&membership->node[i], my_nodeid, sizeof(ece_nodeid_t)) != 0) {

			sdlta = engine_alloc(sizeof(*sdlta));
			if (sdlta != NULL) {
				sdlta->count = &response_count;
				sdlta->mutex = &mutex;
				sdlta->cond  = &cond;

				sdlta->talk = new_talk(&membership->node[i], EVMS_SET_DEBUG_LEVEL, sizeof(net_debug_level), &net_debug_level);

				if (sdlta->talk != NULL) {
					insert_thing(&set_debug_level_list, sdlta, INSERT_AFTER, NULL);

				} else {
					engine_free(sdlta);
					rc = ENOMEM;
				}

			} else {
				rc = ENOMEM;
			}
		}
	}

	if (rc != 0) {
		LIST_FOR_EACH_SAFE(&set_debug_level_list, iter1, iter2, sdlta) {
			engine_free(sdlta->talk);
			engine_free(sdlta);
			delete_element(iter);
		}
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(&set_debug_level_list, iter, sdlta) {
                pthread_t tid;

		pthread_create(&tid, &pthread_attr_detached, set_debug_level_thread, sdlta);
	}

	pthread_mutex_lock(&mutex);
	if (response_count != 0) {
		struct timespec timeout;
		struct timeval now;
		struct timezone tz;

		gettimeofday(&now, &tz);
		timeout.tv_sec = now.tv_sec + REQUEST_TIMEOUT;
		timeout.tv_nsec = 0;
		pthread_cond_timedwait(&cond, &mutex, &timeout);
	}
	pthread_mutex_unlock(&mutex);

	LIST_FOR_EACH(&set_debug_level_list, iter, sdlta) {
		int tmp_rc;
		talk_t * talk = sdlta->talk;

		tmp_rc = talk->rc;

		LOG_DEBUG("Status from node %s is %d: %s\n",
			  nodeid_to_string(&talk->say.node), tmp_rc, evms_strerror(tmp_rc));

		if (tmp_rc == 0) {
			evms_net_to_host(talk->hear.msg, evms_set_debug_level_rets_f,
					 &tmp_rc);
		}

		/* Use the first bad error code. */
		if (rc == 0) {
			rc = tmp_rc;
		}
	}

	LIST_FOR_EACH_SAFE(&set_debug_level_list, iter1, iter2, sdlta) {
                engine_free(sdlta->talk->hear.msg);
                engine_free(sdlta->talk);
                engine_free(sdlta);
		delete_element(iter1);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_commit_changes(void) {

	int rc;
	void * response;

	response = transact_message(current_nodeid, EVMS_COMMIT_CHANGES, 0, NULL, &rc);

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


typedef struct close_engine_thread_args_s {
	talk_t          * talk;
	uint            * count;
	list_anchor_t     callback_list;
	pthread_mutex_t * mutex;
	pthread_cond_t  * cond;
} close_engine_thread_args_t;


static void * close_engine_thread(void * arg) {

        close_engine_thread_args_t * ceta = (close_engine_thread_args_t *) arg;
	talk_t * talk = ceta->talk;
	int rc;

	LOG_PROC_ENTRY();

	talk->rc = ETIMEDOUT;

	/*
	 * Use the mutex to make sure only one thread is doing a send at a
	 * time.  The HA ECE can only handle one at a time.
	 */
	pthread_mutex_lock(ceta->mutex);
	rc = say(talk);
	pthread_mutex_unlock(ceta->mutex);

	if (rc == 0) {
		wait_for_response(talk);
	} else {
		talk->rc = rc;
	}

	while ((talk->rc == 0) && !(talk->hear.cmd & COMMAND_RESPONSE)) {
		/* It's an incoming callback. */

		if (talk->hear.cmd == STATUS) {
			/*
			 * We don't present status from the other nodes when the
			 * Engine is opening.  Status messages from multiple
			 * nodes getting mixed together would be very confusing.
			 */
			engine_free(talk->hear.msg);
			talk->hear.msg = NULL;

			talk->say.cmd  = talk->hear.cmd | COMMAND_RESPONSE;
			talk->say.size = 0;
			talk->say.msg  = NULL;

			talk->rc = say(talk);

		} else {
			/* It's an incoming callback. */

			/*
			 * Turn off the talk's got_response flag so that the
			 * wait_for_response() below does not think it got the
			 * response again and exits causing this while loop to
			 * proecess the response again.
			 */
			talk->got_response = FALSE;

			/*
			 * Put the talk on the callback list and poke the main
			 * thread to process it.
			 */
			pthread_mutex_lock(ceta->mutex);
			insert_thing(ceta->callback_list, talk, INSERT_AFTER, NULL);
			pthread_cond_signal(ceta->cond);
			pthread_mutex_unlock(ceta->mutex);
		}

		wait_for_response(talk);
	}

	pthread_mutex_lock(ceta->mutex);
	*(ceta->count) -= 1;
	if (*(ceta->count) == 0) {
		pthread_cond_signal(ceta->cond);
        }
	pthread_mutex_unlock(ceta->mutex);

	LOG_PROC_EXIT_VOID();
	return NULL;
}


int remote_close_engine() {

	int rc = 0;
	int i;
	STATIC_LIST_DECL(close_engine_list);
	close_engine_thread_args_t * ceta;
	int response_count = 0;
	STATIC_LIST_DECL(callback_list);
	pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
	pthread_cond_t cond = PTHREAD_COND_INITIALIZER;
	list_element_t iter;
	list_element_t iter1;
	list_element_t iter2;

	LOG_PROC_ENTRY();

	response_count = membership->num_entries - 1;

	for (i = 0; (i < membership->num_entries) && (rc == 0); i++) {

		/* Don't send a set debug level message to myself. */
		if (memcmp(&membership->node[i], my_nodeid, sizeof(ece_nodeid_t)) != 0) {

			ceta = engine_alloc(sizeof(*ceta));
			if (ceta != NULL) {
				ceta->count = &response_count;
				ceta->callback_list = &callback_list;
				ceta->mutex = &mutex;
				ceta->cond  = &cond;

				ceta->talk = new_talk(&membership->node[i], EVMS_CLOSE_ENGINE, 0, NULL);

				if (ceta->talk != NULL) {
					insert_thing(&close_engine_list, ceta, INSERT_AFTER, NULL);

				} else {
					engine_free(ceta);
					rc = ENOMEM;
				}

			} else {
				rc = ENOMEM;
			}
		}
	}

	if (rc != 0) {
		LIST_FOR_EACH_SAFE(&close_engine_list, iter1, iter2, ceta) {
			engine_free(ceta->talk);
			engine_free(ceta);
			delete_element(iter);
		}
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(&close_engine_list, iter, ceta) {
                pthread_t tid;

		pthread_create(&tid, &pthread_attr_detached, close_engine_thread, ceta);
	}

	/*
	 * The close engine threads can wakeup this main thread to process
	 * callbacks. This thread will be woken up either to process a
	 * callback that has been placed on the callbuack_list or when
	 * all of the threads have completed and the count has gone to zero.
	 */
	pthread_mutex_lock(&mutex);
	while ((response_count != 0) && (rc == 0)) {
		struct timespec timeout;
		struct timeval now;
		struct timezone tz;

		gettimeofday(&now, &tz);
		timeout.tv_sec = now.tv_sec + REQUEST_TIMEOUT;
		timeout.tv_nsec = 0;
		rc = pthread_cond_timedwait(&cond, &mutex, &timeout);
		if (rc != 0) {
			break;
		}

		if (response_count != 0) {
			/* Must have been woken up to process a callback. */

			while (!list_empty(&callback_list)) {
				talk_t * talk;
				list_element_t el;

				talk = first_thing(&callback_list, &el);
				delete_element(el);
				pthread_mutex_unlock(&mutex);

				handle_callback(talk);

				pthread_mutex_lock(&mutex);
			}

		}
	}
	pthread_mutex_unlock(&mutex);


	LIST_FOR_EACH(&close_engine_list, iter, ceta) {
		int tmp_rc;
		talk_t * talk = ceta->talk;

		tmp_rc = talk->rc;

		LOG_DEBUG("Status from node %s is %d: %s\n",
			  nodeid_to_string(&talk->say.node), tmp_rc, evms_strerror(tmp_rc));

		if (tmp_rc == 0) {
			evms_net_to_host(talk->hear.msg, evms_close_engine_rets_f,
					 &tmp_rc);
		}

		/* Use the first bad error code. */
		if (rc == 0) {
			rc = tmp_rc;
		}
	}

	LIST_FOR_EACH_SAFE(&close_engine_list, iter1, iter2, ceta) {
                engine_free(ceta->talk->hear.msg);
                engine_free(ceta->talk);
                engine_free(ceta);
		delete_element(iter1);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/********************************************
* Functions common to several object types, *
* e.g., volumes and storage objects         *
********************************************/

int remote_can_delete(object_handle_t thing) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_DELETE, thing);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_destroy(object_handle_t thing) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_DESTROY, thing);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_expand(object_handle_t thing) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_EXPAND, thing);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_shrink(object_handle_t thing) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_SHRINK, thing);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_replace(object_handle_t source_handle,
		       object_handle_t target_handle) {
	int rc;

	LOG_PROC_ENTRY();

	rc = two_handle_arg(EVMS_CAN_REPLACE, source_handle, target_handle);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_can_set_info(object_handle_t thing) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_SET_INFO, thing);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_delete(object_handle_t thing) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_DELETE, thing);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_destroy(object_handle_t thing) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_DESTROY, thing);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_get_expand_points(object_handle_t           thing,
			     expand_handle_array_t * * expand_points) {
	int rc = 0;
	u_int32_t net_thing;
	void * response;

	LOG_PROC_ENTRY();

	*expand_points = NULL;

	evms_host_to_net(&net_thing, evms_get_expand_points_args_f,
			 thing);

	response = transact_message(current_nodeid, EVMS_GET_EXPAND_POINTS, sizeof(net_thing), &net_thing, &rc);

	if (rc == 0) {
		uint size;
		expand_handle_array_t * eha;

                evms_net_to_host(response, evms_get_expand_points_rets_f,
				 &rc,
				 &eha);

		if (rc == 0) {
			expand_handle_array_t * user_eha;

			if (eha->count > 1) {
				size = sizeof(expand_handle_array_t) + ((eha->count -1) * sizeof(expand_handle_t));
			} else {
				size = sizeof(expand_handle_array_t);
			}

			user_eha = alloc_app_struct(size, NULL);
			if (user_eha != NULL) {
				memcpy(user_eha, eha, size);
				*expand_points = user_eha;

			} else {
				rc = ENOMEM;
			}
		}
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_expand(object_handle_t  thing,
		  handle_array_t * objects,
		  option_array_t * options) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_expand_args_f,	
				     thing,
				     objects,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_expand_args_f,
			 thing,
			 objects,
			 options);

	response = transact_message(current_nodeid, EVMS_EXPAND, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}



int remote_get_shrink_points(object_handle_t           thing,
			     shrink_handle_array_t * * shrink_points) {
	int rc = 0;
	u_int32_t net_thing;
	void * response;

	LOG_PROC_ENTRY();

	*shrink_points = NULL;

	evms_host_to_net(&net_thing, evms_get_shrink_points_args_f,
			 thing);

	response = transact_message(current_nodeid, EVMS_GET_SHRINK_POINTS, sizeof(net_thing), &net_thing, &rc);

	if (rc == 0) {
		uint size;
		shrink_handle_array_t * sha;

		evms_net_to_host(response, evms_get_shrink_points_rets_f,
				 &rc,
				 &sha);

		if (rc == 0) {
			shrink_handle_array_t * user_sha;

			if (sha->count > 1) {
				size = sizeof(shrink_handle_array_t) + ((sha->count -1) * sizeof(shrink_handle_t));
			} else {
				size = sizeof(shrink_handle_array_t);
			}

			user_sha = alloc_app_struct(size, NULL);
			if (user_sha != NULL) {
				memcpy(user_sha, sha, size);
				*shrink_points = user_sha;

			} else {
				rc = ENOMEM;
			}
		}
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_shrink(object_handle_t  thing,
		  handle_array_t * objects,
		  option_array_t * options) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_shrink_args_f,	
				     thing,
				     objects,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_shrink_args_f,
			 thing,
			 objects,
			 options);

	response = transact_message(current_nodeid, EVMS_SHRINK, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_replace(object_handle_t  source,
		   object_handle_t  target) {
	int rc;

	LOG_PROC_ENTRY();

	rc = two_handle_arg(EVMS_REPLACE, source, target);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_handle_object_type(object_handle_t handle,
				  object_type_t * type) {
	int rc;
	u_int32_t net_thing;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_thing, evms_get_handle_object_type_args_f,
			 handle);

	response = transact_message(current_nodeid, EVMS_GET_HANDLE_OBJECT_TYPE, sizeof(net_thing), &net_thing, &rc);

	if (rc == 0) {
		evms_net_to_host(response, evms_get_handle_object_type_rets_f,
				 &rc,
				 type);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_info(object_handle_t          thing,
		    handle_object_info_t * * user_info) {

	int rc = 0;
	u_int32_t net_thing;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_thing, evms_get_info_args_f,
			 thing);

	response = transact_message(current_nodeid, EVMS_GET_INFO, sizeof(net_thing), &net_thing, &rc);

	if (rc == 0) {
		handle_object_info_t * hoi;

		evms_net_to_host(response, evms_get_info_rets_f,
				 &rc,
				 &hoi);

		if (rc == 0) {
			handle_object_info_t * user_hoi;

			user_hoi = alloc_app_struct(sizeof(handle_object_info_t), free_info_object_contents);

			if (user_hoi != NULL) {
				memcpy(user_hoi, hoi, sizeof(handle_object_info_t));
				*user_info = user_hoi;

			} else {
				LOG_CRITICAL("Error getting memory for a handle_object_info_t.\n");
				rc = ENOMEM;
			}

		} else {
			*user_info = NULL;
		}

		engine_free(hoi);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_extended_info(object_handle_t           thing,
			     char                    * descriptor_name,
			     extended_info_array_t * * info) {
	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_get_extended_info_args_f,	
				     thing,
				     descriptor_name);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_get_extended_info_args_f,
			 thing,
			 descriptor_name);

	response = transact_message(current_nodeid, EVMS_GET_EXTENDED_INFO, arg_size, net_args, &rc);

	if (rc == 0) {
		extended_info_array_t * eia;

		evms_net_to_host(response, evms_get_extended_info_rets_f,
				 &rc,
				 &eia);

		if (rc == 0) {
			extended_info_array_t * user_eia;
			uint size = sizeof(extended_info_array_t) + (eia->count - 1) * sizeof(extended_info_t);

			user_eia = alloc_app_struct(size, free_info_object_contents);

			if (user_eia != NULL) {
				memcpy(user_eia, eia, size);
				*info = user_eia;

			} else {
				LOG_CRITICAL("Error getting memory for a handle_object_info_t.\n");
				rc = ENOMEM;
			}

		} else {
			*info = NULL;
		}

		engine_free(eia);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_set_info(object_handle_t  object,
		    option_array_t * options) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_set_info_args_f,	
				     object,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_set_info_args_f,
			 object,
			 options);

	response = transact_message(current_nodeid, EVMS_SET_INFO, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_object_handle_for_name(object_type_t     type,
				      char            * name,
				      object_handle_t * object_handle) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_get_object_handle_for_name_args_f,	
				     type,
				     name);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_get_object_handle_for_name_args_f,
			 type,
			 name);

	response = transact_message(current_nodeid, EVMS_GET_OBJECT_HANDLE_FOR_NAME, arg_size, net_args, &rc);

	if (rc == 0) {
		evms_net_to_host(response, evms_get_object_handle_for_name_rets_f,
				 &rc,
				 object_handle);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_feature_list(object_handle_t    thing,
			    handle_array_t * * plugin_list) {

	int rc = 0;
	u_int32_t net_thing;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_thing, evms_get_feature_list_args_f,
			 thing);

	response = transact_message(current_nodeid, EVMS_GET_FEATURE_LIST, sizeof(net_thing), &net_thing, &rc);

	if (rc == 0) {
		handle_array_t * ha;

		evms_net_to_host(response, evms_get_feature_list_rets_f,
				 &rc,
				 &ha);

		if (rc == 0) {
			rc = make_user_ha(ha, plugin_list);

		} else {
			*plugin_list = NULL;
		}

		engine_free(ha);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/***************************
* Storage object functions *
***************************/

int remote_create(plugin_handle_t    plugin,
		  handle_array_t   * input_objects,
		  option_array_t   * options,
		  handle_array_t * * output_objects) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_create_args_f,
				     plugin,
				     input_objects,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_create_args_f,
			 plugin,
			 input_objects,
			 options);

	response = transact_message(current_nodeid, EVMS_CREATE, arg_size, net_args, &rc);

	if (rc == 0) {
		handle_array_t * ha;

		evms_net_to_host(response, evms_create_rets_f,
				 &rc,
				 &ha);

		if (rc == 0) {
			rc = make_user_ha(ha, output_objects);

		} else {
			*output_objects = NULL;
		}

		engine_free(ha);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_transfer(object_handle_t  object,
		    plugin_handle_t  plugin,
		    object_handle_t  container,
		    option_array_t * options) {
	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_transfer_args_f,	
				     object,
				     plugin,
				     container,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_transfer_args_f,
			 object,
			 plugin,
			 container,
			 options);

	response = transact_message(current_nodeid, EVMS_TRANSFER, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_assign(object_handle_t  object,
		  plugin_handle_t  plugin,
		  option_array_t * options) {
	
	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_assign_args_f,	
				     object,
				     plugin,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_assign_args_f,
			 object,
			 plugin,
			 options);

	response = transact_message(current_nodeid, EVMS_ASSIGN, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_can_unassign(object_handle_t object) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_UNASSIGN, object);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_unassign(object_handle_t object) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_UNASSIGN, object);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_get_object_list(object_type_t         object_type,
			   data_type_t           data_type,
			   plugin_handle_t       plugin_handle,
			   object_handle_t       disk_group_handle,
			   object_search_flags_t flags,
			   handle_array_t    * * object_handle_list) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_get_object_list_args_f,
				     object_type,
				     data_type,
				     plugin_handle,
				     disk_group_handle,
				     flags);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_get_object_list_args_f,
			 object_type,
			 data_type,
			 plugin_handle,
			 disk_group_handle,
			 flags);

	response = transact_message(current_nodeid, EVMS_GET_OBJECT_LIST, arg_size, net_args, &rc);

	if (rc == 0) {
		handle_array_t * ha;

		evms_net_to_host(response, evms_get_object_list_rets_f,
				 &rc,
				 &ha);

		if (rc == 0) {
			rc = make_user_ha(ha, object_handle_list);

		} else {
			*object_handle_list = NULL;
		}

		engine_free(ha);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_plugin_functions(engine_handle_t           thing,
				function_info_array_t * * actions) {
	int rc = 0;
	u_int32_t net_thing;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_thing, evms_get_plugin_functions_args_f,
			 thing);

	response = transact_message(current_nodeid, EVMS_GET_PLUGIN_FUNCTIONS, sizeof(net_thing), &net_thing, &rc);

	if (rc == 0) {
		function_info_array_t * fia;

		evms_net_to_host(response, evms_get_plugin_functions_rets_f,
				 &rc,
				 &fia);

		if (rc == 0) {
			uint size = sizeof(function_info_array_t);
			function_info_array_t * user_fia;

			if (fia->count > 1) {
				size += (fia->count - 1) * sizeof(function_info_t);
			}

			user_fia = alloc_app_struct(size, free_function_info_array_contents);

			if (user_fia != NULL) {
				/* We do not copy the strings, just the pointers to the strings. */
				memcpy(user_fia, fia, size);
				*actions = user_fia;

			} else {
				rc = ENOMEM;
				*actions = NULL;
			}

		} else {
			*actions = NULL;
		}

		engine_free(fia);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_do_plugin_function(engine_handle_t  thing,
                              task_action_t    action,
			      handle_array_t * objects,
			      option_array_t * options) {
	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_do_plugin_function_args_f,	
				     thing,
				     action,
				     objects,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_do_plugin_function_args_f,
			 thing,
			 action,
			 objects,
			 options);

	response = transact_message(current_nodeid, EVMS_DO_PLUGIN_FUNCTION, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*******************
* Volume functions *
*******************/

int remote_can_create_volume(object_handle_t object) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_CREATE_VOLUME, object);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_create_compatibility_volume(object_handle_t object) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_CREATE_COMPATIBILITY_VOLUME, object);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_set_volume_name(object_handle_t volume) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_SET_VOLUME_NAME, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_convert_to_evms_volume(object_handle_t volume) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_CONVERT_TO_EVMS_VOLUME, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_convert_to_compatibility_volume(object_handle_t volume){

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_CONVERT_TO_COMPATIBILITY_VOLUME, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_add_feature_to_volume(object_handle_t volume,
				     plugin_handle_t feature) {

	int rc;

	LOG_PROC_ENTRY();

	rc = two_handle_arg(EVMS_CAN_ADD_FEATURE_TO_VOLUME, volume, feature);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_create_volume(object_handle_t object,
			 char          * name) {
	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_create_volume_args_f,	
				     object,
				     name);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_create_volume_args_f,
			 object,
			 name);

	response = transact_message(current_nodeid, EVMS_CREATE_VOLUME, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_set_volume_name(object_handle_t volume,
			   char          * name) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_set_volume_name_args_f,	
				     volume,
				     name);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_set_volume_name_args_f,
			 volume,
			 name);

	response = transact_message(current_nodeid, EVMS_SET_VOLUME_NAME, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_create_compatibility_volume(object_handle_t object) {
	
	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CREATE_COMPATIBILITY_VOLUME, object);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_convert_to_evms_volume(object_handle_t volume,
				  char * name) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_convert_to_evms_volume_args_f,	
				     volume,
				     name);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_convert_to_evms_volume_args_f,
			 volume,
			 name);

	response = transact_message(current_nodeid, EVMS_CONVERT_TO_EVMS_VOLUME, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_convert_to_compatibility_volume(object_handle_t volume) {
	
	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CONVERT_TO_COMPATIBILITY_VOLUME, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_add_feature_to_volume(object_handle_t  volume,
				 plugin_handle_t  feature,
				 option_array_t * options) {
	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_add_feature_to_volume_args_f,	
				     volume,
				     feature,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_add_feature_to_volume_args_f,
			 volume,
			 feature,
			 options);

	response = transact_message(current_nodeid, EVMS_ADD_FEATURE_TO_VOLUME, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_can_mkfs(object_handle_t volume,
		    plugin_handle_t fsim) {

	int rc;

	LOG_PROC_ENTRY();

	rc = two_handle_arg(EVMS_CAN_MKFS, volume, fsim);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}

int remote_can_unmkfs(object_handle_t volume) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_UNMKFS, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_fsck(object_handle_t volume) {


	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_FSCK, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_defrag(object_handle_t volume) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_DEFRAG, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_mkfs(object_handle_t  volume,
		plugin_handle_t  fsim,
		option_array_t * options) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_mkfs_args_f,	
				     volume,
				     fsim,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_mkfs_args_f,
			 volume,
			 fsim,
			 options);

	response = transact_message(current_nodeid, EVMS_MKFS, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_unmkfs(object_handle_t volume) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_UNMKFS, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_fsck(object_handle_t  volume,
		option_array_t * options) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_fsck_args_f,	
				     volume,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_fsck_args_f,
			 volume,
			 options);

	response = transact_message(current_nodeid, EVMS_FSCK, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_defrag(object_handle_t  volume,
		  option_array_t * options) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_defrag_args_f,	
				     volume,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_defrag_args_f,
			 volume,
			 options);

	response = transact_message(current_nodeid, EVMS_DEFRAG, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_volume_list(object_handle_t       fsim_handle,
			   object_handle_t       disk_group_handle,
			   volume_search_flags_t flags,
			   handle_array_t    * * volume_handle_list) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_get_volume_list_args_f,
				     fsim_handle,
				     disk_group_handle,
				     flags);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_get_volume_list_args_f,
			 fsim_handle,
			 disk_group_handle,
			 flags);

	response = transact_message(current_nodeid, EVMS_GET_VOLUME_LIST, arg_size, net_args, &rc);

	if (rc == 0) {
		handle_array_t * ha;

		evms_net_to_host(response, evms_get_volume_list_rets_f,
				 &rc,
				 &ha);

		if (rc == 0) {
			rc = make_user_ha(ha, volume_handle_list);

		} else {
			*volume_handle_list = NULL;
		}

		engine_free(ha);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_can_mount(object_handle_t volume) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_MOUNT, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_mount(object_handle_t volume,
		 char          * mount_point,
		 char          * options) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_mount_args_f,	
				     volume,
				     mount_point,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_mount_args_f,
			 volume,
			 mount_point,
			 options);

	response = transact_message(current_nodeid, EVMS_MOUNT, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_can_unmount(object_handle_t volume) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_UNMOUNT, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_unmount(object_handle_t volume) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_UNMOUNT, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_can_remount(object_handle_t volume) {

	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_REMOUNT, volume);

	LOG_PROC_EXIT_INT(rc);
	return rc;			
}


int remote_remount(object_handle_t volume,
		   char          * options) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_remount_args_f,	
				     volume,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_remount_args_f,
			 volume,
			 options);

	response = transact_message(current_nodeid, EVMS_REMOUNT, arg_size, net_args, &rc);

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/******************************
* Storage Container functions *
******************************/

int remote_can_add_to_container(object_handle_t object,
				object_handle_t container) {
	int rc;

	LOG_PROC_ENTRY();

	rc = two_handle_arg(EVMS_CAN_ADD_TO_CONTAINER, object, container);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_can_remove_from_container(object_handle_t object) {
	
	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_CAN_REMOVE_FROM_CONTAINER, object);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_create_container(plugin_handle_t   plugin,
			    handle_array_t  * input_objects,
			    option_array_t  * options,
			    object_handle_t * output_container) {
	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_create_container_args_f,	
				     plugin,
				     input_objects,
				     options);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_create_container_args_f,
			 plugin,
			 input_objects,
			 options);

	response = transact_message(current_nodeid, EVMS_CREATE_CONTAINER, arg_size, net_args, &rc);

	if (rc == 0) {
                evms_net_to_host(response, evms_create_container_rets_f,
				 &rc,
				 output_container);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_container_list(object_handle_t          plugin_handle,
			      object_handle_t          disk_group_handle,
			      container_search_flags_t flags,
			      handle_array_t       * * container_handle_list) {

	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_get_container_list_args_f,
				     plugin_handle,
				     disk_group_handle,
				     flags);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_get_container_list_args_f,
			 plugin_handle,
			 disk_group_handle,
			 flags);

	response = transact_message(current_nodeid, EVMS_GET_CONTAINER_LIST, arg_size, net_args, &rc);

	if (rc == 0) {
		handle_array_t * ha;

		evms_net_to_host(response, evms_get_container_list_rets_f,
				 &rc,
				 &ha);

		if (rc == 0) {
			rc = make_user_ha(ha, container_handle_list);

		} else {
			*container_handle_list = NULL;
		}

		engine_free(ha);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*****************
* Task functions *
*****************/

int remote_create_task(engine_handle_t thing,
		       task_action_t   action,
		       task_handle_t * new_task_context) {
	int rc = 0;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_create_task_args_f,	
				     thing,
				     action);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_create_task_args_f,
			 thing,
			 action);

	response = transact_message(current_nodeid, EVMS_CREATE_TASK, arg_size, net_args, &rc);

	if (rc == 0) {
                evms_net_to_host(response, evms_create_task_rets_f,
				 &rc,
				 new_task_context);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_invoke_task(task_handle_t      task,
		       handle_array_t * * resulting_objects) {

	int rc;
	u_int32_t net_task;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_task, evms_invoke_task_args_f,
			 task);

	response = transact_message(current_nodeid, EVMS_INVOKE_TASK, sizeof(net_task), &net_task, &rc);

	if (rc == 0) {
		handle_array_t * ha;

		evms_net_to_host(response, evms_invoke_task_rets_f,
				 &rc,
				 &ha);

		if (rc == 0) {
			rc = make_user_ha(ha, resulting_objects);

		} else {
			*resulting_objects = NULL;
		}

		engine_free(ha);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_destroy_task(task_handle_t task) {
	
	int rc;

	LOG_PROC_ENTRY();

	rc = one_handle_arg(EVMS_DESTROY_TASK, task);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_task_action (task_handle_t   handle,
			    task_action_t * action) {

        int rc = 0;
	u_int32_t net_handle;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_handle, evms_get_task_action_args_f,
			 handle);

	response = transact_message(current_nodeid, EVMS_GET_TASK_ACTION, sizeof(net_handle), &net_handle, &rc);

	if (rc == 0) {
                evms_net_to_host(response, evms_get_task_action_rets_f,
				 &rc,
				 action);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/************************
* Task object functions *
************************/

int remote_get_acceptable_objects(task_handle_t      task,
				  handle_array_t * * acceptable_object_list) {

	int rc;
	u_int32_t net_task;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_task, evms_get_acceptable_objects_args_f,
			 task);

	response = transact_message(current_nodeid, EVMS_GET_ACCEPTABLE_OBJECTS, sizeof(net_task), &net_task, &rc);

	if (rc == 0) {
		handle_array_t * ha;

		evms_net_to_host(response, evms_get_acceptable_objects_rets_f,
				 &rc,
				 &ha);

		if (rc == 0) {
			rc = make_user_ha(ha, acceptable_object_list);

		} else {
			*acceptable_object_list = NULL;
		}

		engine_free(ha);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_selected_objects(task_handle_t      task,
				handle_array_t * * selected_object_list) {

	int rc;
	u_int32_t net_task;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_task, evms_get_selected_objects_args_f,
			 task);

	response = transact_message(current_nodeid, EVMS_GET_SELECTED_OBJECTS, sizeof(net_task), &net_task, &rc);

	if (rc == 0) {
		handle_array_t * ha;

		evms_net_to_host(response, evms_get_selected_objects_rets_f,
				 &rc,
				 &ha);

		if (rc == 0) {
			rc = make_user_ha(ha, selected_object_list);

		} else {
			*selected_object_list = NULL;
		}

		engine_free(ha);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_selected_object_limits(task_handle_t task,
				      u_int32_t   * minimum,
				      u_int32_t   * maximum) {
	int rc;
	u_int32_t net_task;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_task, evms_get_selected_object_limits_args_f,
			 task);

	response = transact_message(current_nodeid, EVMS_GET_SELECTED_OBJECT_LIMITS, sizeof(net_task), &net_task, &rc);

	if (rc == 0) {
		evms_net_to_host(response, evms_get_selected_object_limits_rets_f,
				 &rc,
				 minimum,
				 maximum);

	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_set_selected_objects(task_handle_t               task,
				handle_array_t            * selected_object_list,
				declined_handle_array_t * * declined_list,
				task_effect_t             * effect) {
	int rc;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_set_selected_objects_args_f,	
				     task,
				     selected_object_list);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_set_selected_objects_args_f,
			 task,
			 selected_object_list);

	response = transact_message(current_nodeid, EVMS_SET_SELECTED_OBJECTS, arg_size, net_args, &rc);

	if (rc == 0) {
		declined_handle_array_t * dha;
		task_effect_t eff;

		evms_net_to_host(response, evms_set_selected_objects_rets_f,
				 &rc,
				 &dha,
				 &eff);

		if (declined_list != NULL) {
			if ((rc == 0) && (dha != NULL)) {
				uint size = sizeof(declined_handle_array_t) + (dha->count - 1) * sizeof(declined_handle_t);
				declined_handle_array_t * user_dha;

				user_dha = alloc_app_struct(size, NULL);

				if (user_dha != NULL) {
					memcpy(user_dha, dha, size);
					*declined_list = user_dha;

				} else {
					LOG_CRITICAL("Error getting memory for a user handle array.\n");
					rc = ENOMEM;
				}

			} else {
				*declined_list = NULL;
			}
		}

		if (effect != NULL) {
		    *effect = eff;
		}

		engine_free(dha);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}



/************************
* Task option functions *
************************/

int remote_get_option_count(task_handle_t task,
			    int         * count) {

        int rc = 0;
	u_int32_t net_handle;
	void * response;

	LOG_PROC_ENTRY();

	evms_host_to_net(&net_handle, evms_get_option_count_args_f,
			 task);

	response = transact_message(current_nodeid, EVMS_GET_OPTION_COUNT, sizeof(net_handle), &net_handle, &rc);

	if (rc == 0) {
                evms_net_to_host(response, evms_get_option_count_rets_f,
				 &rc,
				 count);
	}

	engine_free(response);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_option_descriptor(task_handle_t           task,
				 u_int32_t               option,
				 option_descriptor_t * * descriptor) {
	int rc;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_get_option_descriptor_args_f,	
				     task,
				     option);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_get_option_descriptor_args_f,
			 task,
			 option);

	response = transact_message(current_nodeid, EVMS_GET_OPTION_DESCRIPTOR, arg_size, net_args, &rc);

	if (rc == 0) {
		option_descriptor_t * od;

		evms_net_to_host(response, evms_get_option_descriptor_rets_f,
				 &rc,
				 &od);

		if (rc == 0) {
			option_descriptor_t * user_od;

			user_od = alloc_app_struct(sizeof(option_descriptor_t), free_option_descriptor_contents);

			if (user_od != NULL) {
				memcpy(user_od, od, sizeof(option_descriptor_t));
				*descriptor = user_od;

			} else {
				LOG_CRITICAL("Error getting memory for a user handle array.\n");
				rc = ENOMEM;
			}


		} else {
			*descriptor = NULL;
		}

		engine_free(od);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_set_option_value(task_handle_t   task,
			    u_int32_t       option,
			    value_t       * value,
			    task_effect_t * effect) {
	int rc;
	option_descriptor_t * od;
	value_type_t value_type;
	boolean value_is_list;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	/*
	 * We need to know the value type to know how to send the value over
	 * the wire.  Get the option descriptor which has the type.
	 */
	rc = evms_get_option_descriptor(task, option, &od);
	if (rc != 0) {
		LOG_ERROR("Error getting option descriptor index %d.  Return code is %d: %s\n", option, rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}
	value_type = od->type;
	value_is_list = od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST;
	evms_free(od);

	rc = evms_sizeof_host_to_net(&arg_size, evms_set_option_value_args_f,	
				     task,
				     option,
				     value_type,
				     value_is_list,
				     *value);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_set_option_value_args_f,
			 task,
			 option,
			 value_type,
			 value_is_list,
			 *value);

	response = transact_message(current_nodeid, EVMS_SET_OPTION_VALUE, arg_size, net_args, &rc);

	if (rc == 0) {
		task_effect_t eff;

		evms_net_to_host(response, evms_set_option_value_rets_f,
				 &rc,
				 value,
				 &eff);

		if (effect != NULL) {
			*effect = eff;
		}
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_get_option_descriptor_by_name(task_handle_t           task,
					 const char            * option,
					 option_descriptor_t * * descriptor) {
	int rc;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_get_option_descriptor_by_name_args_f,	
				     task,
				     option);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_get_option_descriptor_by_name_args_f,
			 task,
			 option);

	response = transact_message(current_nodeid, EVMS_GET_OPTION_DESCRIPTOR_BY_NAME, arg_size, net_args, &rc);

	if (rc == 0) {
		option_descriptor_t * od;

		evms_net_to_host(response, evms_get_option_descriptor_by_name_rets_f,
				 &rc,
				 &od);

		if (rc == 0) {
			option_descriptor_t * user_od;

			user_od = alloc_app_struct(sizeof(option_descriptor_t), free_option_descriptor_contents);

			if (user_od != NULL) {
				memcpy(user_od, od, sizeof(option_descriptor_t));
				*descriptor = user_od;

			} else {
				LOG_CRITICAL("Error getting memory for a user handle array.\n");
				rc = ENOMEM;
			}


		} else {
			*descriptor = NULL;
		}

		engine_free(od);
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int remote_set_option_value_by_name(task_handle_t   task,
				    const char    * option_name,
				    value_t       * value,
				    task_effect_t * effect) {
	int rc;
	option_descriptor_t * od;
	value_type_t value_type;
	boolean value_is_list;
	size_t arg_size;
	void * net_args;
	void * response;

	LOG_PROC_ENTRY();

	/*
	 * We need to know the value type to know how to send the value over
	 * the wire.  Get the option descriptor which has the type.
	 */
	rc = evms_get_option_descriptor_by_name(task, option_name, &od);
	if (rc != 0) {
		LOG_ERROR("Error getting option descriptor named \"%s\".  Return code is %d: %s\n", option_name, rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}
	value_type = od->type;
	value_is_list = od->flags & EVMS_OPTION_FLAGS_VALUE_IS_LIST;
	evms_free(od);

	rc = evms_sizeof_host_to_net(&arg_size, evms_set_option_value_by_name_args_f,	
				     task,
				     option_name,
				     value_type,
				     value_is_list,
				     *value);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_set_option_value_by_name_args_f,
			 task,
			 option_name,
			 value_type,
			 value_is_list,
			 *value);

	response = transact_message(current_nodeid, EVMS_SET_OPTION_VALUE_BY_NAME, arg_size, net_args, &rc);

	if (rc == 0) {
		task_effect_t eff;

		evms_net_to_host(response, evms_set_option_value_by_name_rets_f,
				 &rc,
				 value,
				 &eff);

		if (effect != NULL) {
			*effect = eff;
		}
	}

	engine_free(response);

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


typedef struct mark_for_rediscover_thread_args_s {
	talk_t          * talk;
	uint            * count;
	pthread_mutex_t * mutex;
	pthread_cond_t  * cond;
} mark_for_rediscover_thread_args_t;


static void * mark_for_rediscover_thread(void * arg) {

        mark_for_rediscover_thread_args_t * mfrta = (mark_for_rediscover_thread_args_t *) arg;
	talk_t * talk = mfrta->talk;
	int rc;

	LOG_PROC_ENTRY();

	talk->rc = ETIMEDOUT;

	/*
	 * Use the mutex to make sure only one thread is doing a send at a
	 * time.  The HA ECE can only handle one at a time.
	 */
	pthread_mutex_lock(mfrta->mutex);
	rc = say(talk);
	pthread_mutex_unlock(mfrta->mutex);

	if (rc == 0) {
		wait_for_response(talk);
	} else {
		talk->rc = rc;
	}

	pthread_mutex_lock(mfrta->mutex);
	*(mfrta->count) -= 1;
	if (*(mfrta->count) == 0) {
		pthread_cond_signal(mfrta->cond);
        }
	pthread_mutex_unlock(mfrta->mutex);

	LOG_PROC_EXIT_VOID();
	return NULL;
}


int remote_mark_for_rediscover(char * name) {

	int rc = 0;
	int i;
	size_t arg_size;
	void * net_args;
	STATIC_LIST_DECL(mark_for_rediscover_list);
	mark_for_rediscover_thread_args_t * mfrta;
	int response_count = 0;
	STATIC_LIST_DECL(callback_list);
	pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
	pthread_cond_t cond = PTHREAD_COND_INITIALIZER;
	list_element_t iter;
	list_element_t iter1;
	list_element_t iter2;

	LOG_PROC_ENTRY();

	rc = evms_sizeof_host_to_net(&arg_size, evms_mark_for_rediscover_args_f,	
				     name);

	if (rc != 0) {
		LOG_SERIOUS("evms_sizeof_host_to_net() returned error code %d: %s\n", rc, evms_strerror(rc));
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	net_args = engine_alloc(arg_size);

	if (net_args == NULL) {
		LOG_CRITICAL("Error getting memory for net args.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	evms_host_to_net(net_args, evms_mark_for_rediscover_args_f,
			 name);

	response_count = membership->num_entries - 1;

	for (i = 0; (i < membership->num_entries) && (rc == 0); i++) {

		/* Don't send a set debug level message to myself. */
		if (memcmp(&membership->node[i], my_nodeid, sizeof(ece_nodeid_t)) != 0) {

			mfrta = engine_alloc(sizeof(*mfrta));
			if (mfrta != NULL) {
				mfrta->count = &response_count;
				mfrta->mutex = &mutex;
				mfrta->cond  = &cond;

				mfrta->talk = new_talk(&membership->node[i], EVMS_MARK_FOR_REDISCOVER, arg_size, net_args);

				if (mfrta->talk != NULL) {
					insert_thing(&mark_for_rediscover_list, mfrta, INSERT_AFTER, NULL);

				} else {
					engine_free(mfrta);
					rc = ENOMEM;
				}

			} else {
				rc = ENOMEM;
			}
		}
	}

	if (rc != 0) {
		LIST_FOR_EACH_SAFE(&mark_for_rediscover_list, iter1, iter2, mfrta) {
			engine_free(mfrta->talk);
			engine_free(mfrta);
			delete_element(iter);
		}
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(&mark_for_rediscover_list, iter, mfrta) {
                pthread_t tid;

		pthread_create(&tid, &pthread_attr_detached, mark_for_rediscover_thread, mfrta);
	}

	pthread_mutex_lock(&mutex);
	if (response_count > 0) {
		struct timespec timeout;
		struct timeval now;
		struct timezone tz;

		gettimeofday(&now, &tz);
		timeout.tv_sec = now.tv_sec + REQUEST_TIMEOUT;
		timeout.tv_nsec = 0;
		rc = pthread_cond_timedwait(&cond, &mutex, &timeout);
	}
	pthread_mutex_unlock(&mutex);

	LIST_FOR_EACH(&mark_for_rediscover_list, iter, mfrta) {
		int tmp_rc;
		talk_t * talk = mfrta->talk;

		tmp_rc = talk->rc;

		LOG_DEBUG("Status from node %s is %d: %s\n",
			  nodeid_to_string(&talk->say.node), tmp_rc, evms_strerror(tmp_rc));

		if (tmp_rc == 0) {
			evms_net_to_host(talk->hear.msg, evms_mark_for_rediscover_rets_f,
					 &tmp_rc);
		}

		/* Use the first bad error code. */
		if (rc == 0) {
			rc = tmp_rc;
		}
	}

	LIST_FOR_EACH_SAFE(&mark_for_rediscover_list, iter1, iter2, mfrta) {
                engine_free(mfrta->talk->hear.msg);
                engine_free(mfrta->talk);
                engine_free(mfrta);
		delete_element(iter1);
	}

	engine_free(net_args);

	LOG_PROC_EXIT_INT(rc);
        return rc;
}


typedef struct rediscover_thread_args_s {
	talk_t          * talk;
	uint            * count;
	list_anchor_t     callback_list;
	pthread_mutex_t * mutex;
	pthread_cond_t  * cond;
} rediscover_thread_args_t;


static void * rediscover_thread(void * arg) {

        rediscover_thread_args_t * rta = (rediscover_thread_args_t *) arg;
	talk_t * talk = rta->talk;
	int rc;

	LOG_PROC_ENTRY();

	talk->rc = ETIMEDOUT;

	/*
	 * Use the mutex to make sure only one thread is doing a send at a
	 * time.  The HA ECE can only handle one at a time.
	 */
	pthread_mutex_lock(rta->mutex);
	rc = say(talk);
	pthread_mutex_unlock(rta->mutex);

	if (rc == 0) {
		wait_for_response(talk);
	} else {
		talk->rc = rc;
	}

	while ((talk->rc == 0) && !(talk->hear.cmd & COMMAND_RESPONSE)) {
		/* It's an incoming callback. */

		if (talk->hear.cmd == STATUS) {
			/*
			 * We don't present status from the other nodes during
			 * rediscover.  Status messages from multiple nodes
			 * getting mixed together would be very confusing.
			 */
			engine_free(talk->hear.msg);
			talk->hear.msg = NULL;

			talk->say.cmd  = talk->hear.cmd | COMMAND_RESPONSE;
			talk->say.size = 0;
			talk->say.msg  = NULL;

			talk->rc = say(talk);

		} else {
			/* It's an incoming callback. */

			/*
			 * Turn off the talk's got_response flag so that the
			 * wait_for_response() below does not think it got the
			 * response again and exits causing this while loop to
			 * proecess the response again.
			 */
			talk->got_response = FALSE;

			/*
			 * Put the talk on the callback list and poke the main
			 * thread to process it.
			 */
			pthread_mutex_lock(rta->mutex);
			insert_thing(rta->callback_list, talk, INSERT_AFTER, NULL);
			pthread_cond_signal(rta->cond);
			pthread_mutex_unlock(rta->mutex);
		}

		wait_for_response(talk);
	}

	pthread_mutex_lock(rta->mutex);
	*(rta->count) -= 1;
	if (*(rta->count) == 0) {
		pthread_cond_signal(rta->cond);
        }
	pthread_mutex_unlock(rta->mutex);

	LOG_PROC_EXIT_VOID();
	return NULL;
}


int remote_rediscover(void) {

	int rc = 0;
	int i;
	STATIC_LIST_DECL(rediscover_list);
	rediscover_thread_args_t * rta;
	int response_count = 0;
	STATIC_LIST_DECL(callback_list);
	pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
	pthread_cond_t cond = PTHREAD_COND_INITIALIZER;
	list_element_t iter;
	list_element_t iter1;
	list_element_t iter2;

	LOG_PROC_ENTRY();

	response_count = membership->num_entries - 1;

	for (i = 0; (i < membership->num_entries) && (rc == 0); i++) {

		/* Don't send a set debug level message to myself. */
		if (memcmp(&membership->node[i], my_nodeid, sizeof(ece_nodeid_t)) != 0) {

			rta = engine_alloc(sizeof(*rta));
			if (rta != NULL) {
				rta->count = &response_count;
				rta->mutex = &mutex;
				rta->cond  = &cond;

				rta->talk = new_talk(&membership->node[i], EVMS_REDISCOVER, 0, NULL);

				if (rta->talk != NULL) {
					insert_thing(&rediscover_list, rta, INSERT_AFTER, NULL);

				} else {
					engine_free(rta);
					rc = ENOMEM;
				}

			} else {
				rc = ENOMEM;
			}
		}
	}

	if (rc != 0) {
		LIST_FOR_EACH_SAFE(&rediscover_list, iter1, iter2, rta) {
			engine_free(rta->talk);
			engine_free(rta);
			delete_element(iter);
		}
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(&rediscover_list, iter, rta) {
                pthread_t tid;

		pthread_create(&tid, &pthread_attr_detached, rediscover_thread, rta);
	}

	/*
	 * The rediscover threads can wakeup this main thread to process
	 * callbacks. This thread will be woken up either to process a
	 * callback that has been placed on the callbuack_list or when
	 * all of the threads have completed and the count has gone to zero.
	 */
	pthread_mutex_lock(&mutex);
	while ((response_count != 0) && (rc == 0)) {
		struct timespec timeout;
		struct timeval now;
		struct timezone tz;

		gettimeofday(&now, &tz);
		timeout.tv_sec = now.tv_sec + REQUEST_TIMEOUT;
		timeout.tv_nsec = 0;
		rc = pthread_cond_timedwait(&cond, &mutex, &timeout);
		if (rc != 0) {
			break;
		}

		if (response_count != 0) {
			/* Must have been woken up to process a callback. */

			while (!list_empty(&callback_list)) {
				talk_t * talk;
				list_element_t el;

				talk = first_thing(&callback_list, &el);
				delete_element(el);
				pthread_mutex_unlock(&mutex);

				handle_callback(talk);

				pthread_mutex_lock(&mutex);
			}

		}
	}
	pthread_mutex_unlock(&mutex);

	LIST_FOR_EACH(&rediscover_list, iter, rta) {
		int tmp_rc;
		talk_t * talk = rta->talk;

		tmp_rc = talk->rc;

		LOG_DEBUG("Status from node %s is %d: %s\n",
			  nodeid_to_string(&talk->say.node), tmp_rc, evms_strerror(tmp_rc));

		if (tmp_rc == 0) {
			evms_net_to_host(talk->hear.msg, evms_rediscover_rets_f,
					 &tmp_rc);
		}

		/* Use the first bad error code. */
		if (rc == 0) {
			rc = tmp_rc;
		}
	}

	LIST_FOR_EACH_SAFE(&rediscover_list, iter1, iter2, rta) {
                engine_free(rta->talk->hear.msg);
                engine_free(rta->talk);
                engine_free(rta);
		delete_element(iter1);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


typedef struct shutdown_thread_args_s {
	talk_t          * talk;
	uint            * count;
	list_anchor_t     callback_list;
	pthread_mutex_t * mutex;
	pthread_cond_t  * cond;
} shutdown_thread_args_t;


static void * shutdown_thread(void * arg) {

        shutdown_thread_args_t * sta = (shutdown_thread_args_t *) arg;
	talk_t * talk = sta->talk;
	int rc;

	LOG_PROC_ENTRY();

	talk->rc = ETIMEDOUT;

	/*
	 * Use the mutex to make sure only one thread is doing a send at a
	 * time.  The HA ECE can only handle one at a time.
	 */
	pthread_mutex_lock(sta->mutex);
	rc = say(talk);
	pthread_mutex_unlock(sta->mutex);

	if (rc == 0) {
		wait_for_response(talk);
	} else {
		talk->rc = rc;
	}

	while ((talk->rc == 0) && !(talk->hear.cmd & COMMAND_RESPONSE)) {
		/* It's an incoming callback. */

		if (talk->hear.cmd == STATUS) {
			/*
			 * We don't present status from the other nodes when the
			 * Engine is opening.  Status messages from multiple
			 * nodes getting mixed together would be very confusing.
			 */
			engine_free(talk->hear.msg);
			talk->hear.msg = NULL;

			talk->say.cmd  = talk->hear.cmd | COMMAND_RESPONSE;
			talk->say.size = 0;
			talk->say.msg  = NULL;

			talk->rc = say(talk);

		} else {
			/* It's an incoming callback. */

			/*
			 * Turn off the talk's got_response flag so that the
			 * wait_for_response() below does not think it got the
			 * response again and exits causing this while loop to
			 * proecess the response again.
			 */
			talk->got_response = FALSE;

			/*
			 * Put the talk on the callback list and poke the main
			 * thread to process it.
			 */
			pthread_mutex_lock(sta->mutex);
			insert_thing(sta->callback_list, talk, INSERT_AFTER, NULL);
			pthread_cond_signal(sta->cond);
			pthread_mutex_unlock(sta->mutex);
		}

		wait_for_response(talk);
	}

	pthread_mutex_lock(sta->mutex);
	*(sta->count) -= 1;
	if (*(sta->count) == 0) {
		pthread_cond_signal(sta->cond);
        }
	pthread_mutex_unlock(sta->mutex);

	LOG_PROC_EXIT_VOID();
	return NULL;
}


int remote_shutdown() {

	int rc = 0;
	int i;
	STATIC_LIST_DECL(shutdown_list);
	shutdown_thread_args_t * sta;
	int response_count = 0;
	STATIC_LIST_DECL(callback_list);
	pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
	pthread_cond_t cond = PTHREAD_COND_INITIALIZER;
	list_element_t iter;
	list_element_t iter1;
	list_element_t iter2;

	LOG_PROC_ENTRY();

	response_count = membership->num_entries - 1;

	for (i = 0; (i < membership->num_entries) && (rc == 0); i++) {

		/* Don't send a set debug level message to myself. */
		if (memcmp(&membership->node[i], my_nodeid, sizeof(ece_nodeid_t)) != 0) {

			sta = engine_alloc(sizeof(*sta));
			if (sta != NULL) {
				sta->count = &response_count;
				sta->callback_list = &callback_list;
				sta->mutex = &mutex;
				sta->cond  = &cond;

				sta->talk = new_talk(&membership->node[i], SHUTDOWN, 0, NULL);

				if (sta->talk != NULL) {
					insert_thing(&shutdown_list, sta, INSERT_AFTER, NULL);

				} else {
					engine_free(sta);
					rc = ENOMEM;
				}

			} else {
				rc = ENOMEM;
			}
		}
	}

	if (rc != 0) {
		LIST_FOR_EACH_SAFE(&shutdown_list, iter1, iter2, sta) {
			engine_free(sta->talk);
			engine_free(sta);
			delete_element(iter);
		}
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	LIST_FOR_EACH(&shutdown_list, iter, sta) {
                pthread_t tid;

		pthread_create(&tid, &pthread_attr_detached, shutdown_thread, sta);
	}

	/*
	 * The close engine threads can wakeup this main thread to process
	 * callbacks. This thread will be woken up either to process a
	 * callback that has been placed on the callbuack_list or when
	 * all of the threads have completed and the count has gone to zero.
	 */
	pthread_mutex_lock(&mutex);
	while ((response_count != 0) && (rc == 0)) {
		struct timespec timeout;
		struct timeval now;
		struct timezone tz;

		gettimeofday(&now, &tz);
		timeout.tv_sec = now.tv_sec + REQUEST_TIMEOUT;
		timeout.tv_nsec = 0;
		rc = pthread_cond_timedwait(&cond, &mutex, &timeout);
		if (rc != 0) {
			break;
		}

		if (response_count != 0) {
			/* Must have been woken up to process a callback. */

			while (!list_empty(&callback_list)) {
				talk_t * talk;
				list_element_t el;

				talk = first_thing(&callback_list, &el);
				delete_element(el);
				pthread_mutex_unlock(&mutex);

				handle_callback(talk);

				pthread_mutex_lock(&mutex);
			}

		}
	}
	pthread_mutex_unlock(&mutex);


	LIST_FOR_EACH(&shutdown_list, iter, sta) {
		int tmp_rc;
		talk_t * talk = sta->talk;

		tmp_rc = talk->rc;

		LOG_DEBUG("Status from node %s is %d: %s\n",
			  nodeid_to_string(&talk->say.node), tmp_rc, evms_strerror(tmp_rc));

		/* Use the first bad error code. */
		if (rc == 0) {
			rc = tmp_rc;
		}
	}

	LIST_FOR_EACH_SAFE(&shutdown_list, iter1, iter2, sta) {
                engine_free(sta->talk->hear.msg);
                engine_free(sta->talk);
                engine_free(sta);
		delete_element(iter1);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


void engine_router(ece_msg_t * msg) {

	u_int32_t cmd = msg->cmd;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Message from node %s: command %#x (%s %s)  size: %zu\n",
		  nodeid_to_string(&msg->node), cmd, msg_cmd_name(cmd), (cmd & COMMAND_RESPONSE) ? "response" : "request", msg->size);

	handle_response(msg);

	LOG_PROC_EXIT_VOID();
}
