/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libbsd.so
 *
 *   File: segments.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "bsd_plugin.h"


int bsd_can_set_volume( DISKSEG * seg, boolean flag )
{
        LOG_ENTRY();
        REQUIRE( isa_bsd_segment(seg) == TRUE );
        LOG_EXIT_INT(0);
        return 0;
}

int bsd_can_delete( DISKSEG * seg )
{
        LOG_ENTRY();

        REQUIRE( seg != NULL);
        REQUIRE( seg->object_type == SEGMENT );
        REQUIRE( seg->data_type   == DATA_TYPE );
        REQUIRE( disk_move_pending(seg) == FALSE);
        REQUIRE( isa_bsd_segment( seg ) == TRUE );

        LOG_EXIT_INT(0);
        return 0;
}


int bsd_can_expand( DISKSEG         *seg,
                    sector_count_t  *expand_limit,
                    list_anchor_t    expansion_points )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}

int bsd_can_expand_by( DISKSEG * seg, sector_count_t *size)
{
        LOG_ENTRY();
        *size = 0;
        LOG_EXIT_INT(0);
        return 0;
}


int bsd_can_shrink( DISKSEG * seg,
                    sector_count_t   * shrink_limit,
                    list_anchor_t      shrink_points )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int bsd_can_shrink_by( DISKSEG * seg, sector_count_t * size )
{
        LOG_ENTRY();
        *size=0;
        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function:  bsd_discover
 *
 *  Called to run discovery on a list of evms objects that
 *  the engine has found thus far.  We walk the list of objects,
 *  probing each object to see if it has a BSD disk label.  If so,
 *  we consume the object by removing it from the list, and place
 *  all new segment storage objects on the output object list.
 *  Any object we dont like in the input_object list must be
 *  copied to the output_object list.
 *
 */
int bsd_discover( list_anchor_t input_objects, list_anchor_t output_objects, boolean final_call)
{
        int  count = 0;
        list_element_t iter;
        storage_object_t *object;

        LOG_ENTRY();

        if ( (input_objects != NULL) &&
             (output_objects != NULL) &&
             (EngFncs->list_count(input_objects) > 0)) {
 
                LIST_FOR_EACH( input_objects, iter, object ){
                        discover_bsd_segments( object, output_objects, &count);
                }

        }

        LOG_EXIT_INT(count);
        return count;
}


int bsd_assign( storage_object_t * input_object, option_array_t * options )
{
        LOG_ENTRY();

        REQUIRE( input_object != NULL );
        REQUIRE( options != NULL);

        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


/*
 *  Function:  bsd_unassign
 *
 *  We can unassign from any logical disk storage object that
 *  has a BSD disk label on it as long as the data segments are 
 *  top-most storage objects.
 *
 */
int bsd_can_unassign( LOGICALDISK *ld )
{
        DISKSEG *seg;
        list_element_t iter;

        LOG_ENTRY();

        REQUIRE( isa_bsd_logical_disk(ld) == TRUE );

        LIST_FOR_EACH( ld->parent_objects, iter, seg ) {
                REQUIRE( EngFncs->list_count(seg->parent_objects) == 0 );
        }

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function:  bsd_unassign
 *
 *  This routine is called to remove the BSD Segment Manager
 *  from the specified storage object (logical disk).  It will do
 *  so by:
 *
 *      - removing the BSD disk label from the storage object by writing
 *        kill sectors (zero filled sectors) to appropriate locations
 *        on the logical disk ... erasing the metadata found at
 *        those locations.
 *
 */
int bsd_unassign( LOGICALDISK *ld )
{
        LOG_ENTRY();

        REQUIRE( isa_bsd_logical_disk(ld) == TRUE );
  
        // toss segment storage objects
        prune_bsd_seg_objects_from_list( ld->parent_objects );

        // toss our logical disk private data area
        delete_bsd_disk_private_data( ld );

        // blast bsd disk label 
        KILL_SECTORS(ld, 0, 1);

        LOG_EXIT_INT(0);
        return 0;
}


int bsd_create( list_anchor_t          input_objects,
                option_array_t * options,
                list_anchor_t          new_objects)
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int bsd_destroy( DISKSEG * seg, list_anchor_t child_objects )
{
        int           rc = EINVAL;
        DISKSEG      *metadata=NULL;
        DISKSEG      *md=NULL;
        LOGICALDISK  *ld=NULL;
        disk_private_data_t  *disk_pdata=NULL;
        list_element_t iter;


        LOG_ENTRY();

        LOG_DEBUG("seg: %s\n", seg->name );

        REQUIRE( seg != NULL);
        REQUIRE( seg->object_type == SEGMENT );
        REQUIRE( seg->data_type   == DATA_TYPE );
        REQUIRE( disk_move_pending(seg) == FALSE);
        REQUIRE( isa_bsd_segment( seg ) == TRUE );
 
        ld = get_logical_disk(seg);
        REQUIRE( ld != NULL);

        disk_pdata = get_bsd_disk_private_data(ld);
        REQUIRE( disk_pdata != NULL);

        LIST_FOR_EACH( ld->parent_objects, iter, md ){
                if (md->data_type == META_DATA_TYPE) {
                        metadata = md;
                        break;
                }
        }

        if (metadata) {
                
                rc = remove_bsd_segment_from_list( ld->parent_objects, seg );
                if (rc == 0) {

                        free_bsd_segment(seg);

                        //find_freespace_on_bsd_disk( ld );

                        disk_pdata->flags |= DISK_HAS_CHANGES_PENDING;
                        metadata->flags   |= SOFLAG_DIRTY;
                }
                else {
                        rc = ENODEV; 
                }

        }
        else {
                rc = EINVAL;
        }

        LOG_EXIT_INT(rc);
        return  rc;
}

/*
 * Forget about these objects.  Don't delete them.  Just clean up any
 * data structures you may have associated with them.  The Engine will
 * call to deactivate the objects durring commit.
*/
int bsd_discard(list_anchor_t objects)
{
	int rc=0;
	LOGICALDISK *ld;
	DISKSEG *seg;
	list_element_t iter;

	LOG_ENTRY();

	LIST_FOR_EACH( objects, iter, seg ) {
		if ( isa_bsd_segment( seg ) == TRUE ) {
			ld = get_logical_disk(seg);
			if (ld) {
				remove_bsd_segment_from_list( ld->parent_objects, seg );
				if (EngFncs->list_empty(ld->parent_objects) == TRUE)
			        // toss our logical disk private data area
			        delete_bsd_disk_private_data( ld );	
			}
			
			free_bsd_segment(seg);				
			
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int bsd_expand( DISKSEG          *seg,
                storage_object_t *expand_object,
                list_anchor_t           objects,
                option_array_t   *options )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int bsd_shrink( DISKSEG          * seg,
                storage_object_t * shrink_object,
                list_anchor_t            objects,
                option_array_t   * options )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int bsd_add_sectors_to_kill_list( DISKSEG       *seg,
                                  lsn_t          lsn,
                                  sector_count_t count)
{
        int rc = EINVAL;
        LOGICALDISK *ld=NULL;
        struct plugin_functions_s *funcs=NULL;

        LOG_ENTRY();

        REQUIRE( isa_bsd_segment( seg ) == TRUE );
        REQUIRE( lsn + count > seg->size );

        ld = get_logical_disk( seg );
        if (ld) {

                funcs = (struct plugin_functions_s *)ld->plugin->functions.plugin;
                rc   = funcs->add_sectors_to_kill_list( ld, seg->start+lsn, count);

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  bsd_commit_changes
 *
 *  The engine will call this plug-in API to save BSD metadata
 *  changes to disk.  The two arguments tell us which segment
 *  storage object is marked dirty and which commit phase is
 *  being processed.  We have no work for commit phase Zero. So,
 *  we only call the low level BSD commit routine for subsequent
 *  commit phases.
 */
int bsd_commit_changes( DISKSEG *seg, uint commit_phase )
{        
        int rc;
        LOGICALDISK *ld;
        disk_private_data_t *disk_pdata;


        LOG_ENTRY();
        
        REQUIRE( isa_bsd_segment(seg) == TRUE );

        ld = get_logical_disk(seg);     
        REQUIRE( ld != NULL );

        disk_pdata = get_bsd_disk_private_data(ld);
        REQUIRE( disk_pdata != NULL);
             
        if (commit_phase == MOVE) {
                if ( disk_pdata->flags & DISK_HAS_MOVE_PENDING ) {                              
                        /*
                        rc = bsd_move_segment_commit( disk_pdata->copy_job );
                        */
                        rc = ENOSYS;
                        if (disk_pdata->copy_job) free(disk_pdata->copy_job );
                        disk_pdata->copy_job = NULL;
                        disk_pdata->flags   &= ~DISK_HAS_MOVE_PENDING; 
                        seg->flags          &= ~SOFLAG_DIRTY;
                }
                else {
                        rc = 0;
                }
        }
        else if ( commit_phase==FIRST_METADATA_WRITE || commit_phase==SECOND_METADATA_WRITE ) {
                if ( ( seg->flags & SOFLAG_DIRTY) &&
                     ( disk_pdata->flags & DISK_HAS_CHANGES_PENDING) ) {
                        rc = commit_bsd_segments( seg, get_logical_disk(seg), commit_phase );
                }
                else {
                        rc = 0;
                }
        }
        else {
                rc=0;
        }
        

        LOG_EXIT_INT(0);
        return 0;   
}


int bsd_read( DISKSEG           *seg,
              lsn_t              lsn,
              sector_count_t     count,
              void              *buffer )
{
        int rc = EINVAL;
        LOGICALDISK *ld=NULL;

        LOG_ENTRY();

        REQUIRE( isa_bsd_segment(seg) == TRUE );
        REQUIRE( lsn + count <= seg->size );

        ld = get_logical_disk( seg );
        if (ld) {
                rc = READ( ld, lsn + seg->start, count, buffer);
        }

        LOG_EXIT_INT(rc);
        return rc;
}

int bsd_write( DISKSEG          *seg,
               lsn_t             lsn,
               sector_count_t    count,
               void             *buffer )
{
        int rc = EINVAL;
        LOGICALDISK *ld=NULL;

        LOG_ENTRY();

        REQUIRE( isa_bsd_segment(seg) == TRUE );
        REQUIRE( lsn + count <= seg->size );

        ld = get_logical_disk( seg );
        if (ld) {
                rc = WRITE( ld, lsn + seg->start, count, buffer);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


void bsd_set_volume(storage_object_t * object, boolean flag)
{
        return;
}


int bsd_get_info( DISKSEG *seg, char * name, extended_info_array_t  ** info_array )
{
        int rc=0;

        LOG_ENTRY();

        REQUIRE( seg!=NULL );
        REQUIRE( seg->object_type == SEGMENT );
        REQUIRE( info_array != NULL );

        if ( name==NULL ) {
                rc = bsd_get_segment_info(seg, info_array);
        }
        else if ( ( strcmp(name, "Type")==0 ) && (seg->data_type==META_DATA_TYPE) ){
                rc = bsd_get_metadata_info(seg, info_array);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  delete_all_bsd_segment_private_data
 *
 *  Only called by our plug-in cleanup API.  This routine will
 *  know how to properly free up all bsd segment private data areas.
 */
void delete_all_bsd_segment_private_data(void)
{
        list_anchor_t list;
        int rc;
        storage_object_t *object;
        list_element_t iter;

        LOG_ENTRY();

        list = EngFncs->allocate_list();
        if (list != NULL) {

                rc = EngFncs->get_object_list( SEGMENT,
                                                     0,
                                                     bsd_plugin,
                                                     NULL,
                                                     0,
                                                     &list );
                if (!rc) {

                        LIST_FOR_EACH( list, iter, object ){
                                if (object->private_data) free(object->private_data);
                        }

                }

                EngFncs->destroy_list( list );
        }
        LOG_EXIT_VOID();
}

