/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: move.c
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "move.h"
#include "dm.h"
#include "commit.h"
#include "display.h"
#include "segs.h"
#include "display.h"

/*
 *  Function: safe to move
 *
 *  Called to determine if it would be Ok to move
 *  the sepcified segment object.
 */
static inline boolean safe_to_move(DISKSEG *seg)
{
	LOGICALDISK        *ld=get_logical_disk(seg);   
	DISK_PRIVATE_DATA  *disk_pdata=NULL;

	if ( ld ) {

		disk_pdata = get_disk_private_data(ld); 
		if (disk_pdata) {

			if ( i_can_modify_seg(seg)  ==  TRUE &&
			     seg->data_type         ==  DATA_TYPE &&
			     seg_is_volitile(seg)   ==  TRUE  &&
			     !(seg->flags & SOFLAG_DIRTY) &&
			     (disk_pdata->flags & DISK_HAS_MOVE_PENDING) == 0 ) {

				return TRUE;

			}
		}

	}

	return FALSE;
}


/*
 *  Function: free a copy_job_t
 *
 *  Called to free a copy_job_t
 */
static inline void free_copy_job_t(copy_job_t *job)
{
	if (job) {
		free(job->title);
		free(job);
	}
}


/*
 *  Function: alloc copy_job_t
 *
 *  Called to allocate a copy_job_t
 */
static inline copy_job_t * alloc_copy_job_t(void)
{
	copy_job_t *job = calloc( 1, sizeof(copy_job_t) );
	return job;
}


/*
 *  Function: create_copy_job
 *
 *  Called by dos_move_start() to create an engine copy job that
 *  will be executed during a subsequent commit.
 */
static int create_copy_job( DISKSEG      *seg,	     // data segment to be moved
			    DISKSEG      *trg,	     // move target segment
			    copy_job_t  **copyjob )  // callers copy_job struct
{
	int rc=ENOMEM;
	DISK_PRIVATE_DATA   *disk_pdata=NULL;
	LOGICALDISK         *ld=NULL;
	copy_job_t          *job=NULL;
	char                *title=NULL;


	LOG_ENTRY();

	ld          = get_logical_disk(seg);
	disk_pdata  = get_disk_private_data(ld);        
	job         = alloc_copy_job_t();
	title       = malloc(EVMS_NAME_SIZE*2+2);

	if (job && title) {

		sprintf(title, "Moving segment %s\n", seg->name);
		job->title = title;
		job->description = "";

		job->src.obj     = ld;
		job->src.start   = seg->start;
		job->src.len     = seg->size;

		job->trg.obj     = ld;
		job->trg.start   = trg->start;
		job->trg.len     = seg->size;

		*copyjob         = job;

		rc               = 0;
	}

	if (rc) {
		if (job) free_copy_job_t(job);
		if (title) free(title);
	}


	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function:  create move target segment
 *
 *  Called by our private function dos_move_segment()
 *  to create a target storage_object_t that will
 *  be used to move the specified disk segment.
 */
static int create_move_target( DISKSEG  *seg,	     // source data segment
			       DISKSEG  *freespace,  // target freespace segment
			       DISKSEG **target,     // where to return move target storage object
			       boolean   testing )   // testing == TRUE if we just want to test
{						     // if a move object could be created and
	LOGICALDISK             *ld=NULL;	     // dont actually want the object returned
	DISKSEG                 *trg=NULL;
	DISKSEG                 *ebr=NULL;
	SEG_PRIVATE_DATA        *src_pdata=NULL;
	SEG_PRIVATE_DATA        *trg_pdata=NULL;
	SEG_PRIVATE_DATA        *ebr_pdata=NULL;
	DISK_PRIVATE_DATA       *disk_pdata=NULL;
	lba_t                   start=0;
	lba_t                   end=0;
	int                     rc = EINVAL;

	LOG_ENTRY();

	if (seg) {
		ld          = get_logical_disk(seg);
		disk_pdata  = get_disk_private_data(ld);
		src_pdata   = (SEG_PRIVATE_DATA *) seg->private_data;
		LOG_DEBUG("  src seg->start = %"PRIu64"\n", seg->start);
		LOG_DEBUG("            size = %"PRIu64"\n", seg->size);
	}

	if (ld == NULL || disk_pdata == NULL) {
		LOG_ERROR("error, get_logical_disk() failed\n");
		rc = ENODEV;
		LOG_EXIT_INT(rc);
		return rc;              
	}

	if (freespace) {
		LOG_DEBUG("freespace->start = %"PRIu64"\n", freespace->start);
		LOG_DEBUG("           size  = %"PRIu64"\n", freespace->size);
		if (freespace == NULL || freespace->size < seg->size) {
			LOG_ERROR("error, freespace too small for move\n");
			rc = EFBIG;
			LOG_EXIT_INT(rc);
			return rc;              
		}
	}
	else {
		LOG_ERROR("error, freespace is NULL\n");
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	rc = ENOMEM;

	trg = allocate_disk_segment( ld );
	if (trg) {
		sprintf(trg->name, "%s_move_target", seg->name );
		trg->data_type    = META_DATA_TYPE;
		trg->flags       &= ~SOFLAG_DIRTY;
		trg_pdata         = (SEG_PRIVATE_DATA *) trg->private_data;
		trg_pdata->flags |= SEG_IS_MOVE_TARGET;

		if (src_pdata->boot_ind == ACTIVE_PARTITION) {
			trg->flags |= SOFLAG_BIOS_READABLE;
		}

		if (src_pdata->flags & SEG_IS_LOGICAL_PARTITION) {

			trg_pdata->flags |= SEG_IS_LOGICAL_PARTITION;

			ebr = allocate_disk_segment( ld );
			if (ebr) {
				sprintf(ebr->name, "%s_move_target", src_pdata->ebr->name);
				ebr->data_type          = META_DATA_TYPE;
				ebr_pdata               = (SEG_PRIVATE_DATA *) ebr->private_data;
				ebr_pdata->sys_id       = DOS_EXTENDED_PARTITION ;
				ebr_pdata->flags        = SEG_IS_EBR;
				trg_pdata->ebr          = ebr;
				rc = 0;
			}

		}
		else {
			trg_pdata->flags |= SEG_IS_PRIMARY_PARTITION;
			trg_pdata->ebr    = src_pdata->ebr;
			rc = 0;
		}
	}

	if (rc==0) {

		rc = EFBIG;

		// calculate the starting lba
		if (src_pdata->flags & SEG_IS_LOGICAL_PARTITION) {

			lba_t cyl_start;
			sector_count_t cyl_offset;

			LOG_DEBUG("creating move target for logical partition ...\n");

			// a logical partition and its ebr should always start on a cyl boundary
			if (starts_on_cylinder_boundary(ld,freespace->start)==FALSE) {
				LOG_DEBUG("freespace doesnt start on cyl bdy ... rounding up\n");
				start = roundup_to_cylinder_boundary(ld,freespace->start) + 1;
			}
			else {
				LOG_DEBUG("freespace starts on cyl bdy\n");
				start = freespace->start;
			}

			LOG_DEBUG("start of logical = %"PRIu64"\n", start);

			// ebr will be in very 1st sector and occupy 1 track
			ebr->start = start;
			ebr->size  = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block; 

			LOG_DEBUG("ebr start = %"PRIu64"\n", ebr->start);
			LOG_DEBUG("ebr  size = %"PRIu64"\n", ebr->size);

			// determine the offset of the logical from the start of the cylinder
			// that it starts within ... not from the start of the ebr ... which 
			// may or may not be in the very 1st sector of the cylinder.
			cyl_start  = rounddown_to_cylinder_boundary(ld,seg->start);
			cyl_offset = seg->start - cyl_start;        

			LOG_DEBUG("src seg cyl start  = %"PRIu64"\n",cyl_start); 
			LOG_DEBUG("src seg cyl offset = %"PRIu64"\n", cyl_offset);

			if (cyl_offset == 0) { // partition is cyl alligned
				LOG_DEBUG("src seg starts on cyl allignment and so must target ... rounding start up to cyl bdy\n");
				trg->start = roundup_to_cylinder_boundary(ld,ebr->start+1) + 1;
			}
			else {
				LOG_DEBUG("src seg doesnt start on cyl allignment so calculating offset start of target\n");
				trg->start = ebr->start + cyl_offset;
			}

			LOG_DEBUG("target start = %"PRIu64"\n", trg->start);
		}
		else {
			LOG_DEBUG("creating move target for primary partition ...\n");

			// but a primary partition is complicated !!!  Complications involve:
			//
			// (1) source segment could be track alligned NOT cylinder alligned due to following
			//     an MBR track on the disk.
			// (2) a source segment might be track alligned because it has been moved before and was
			//     track alligned because it use to follow the MBR track on the disk.
			// (3) the target freespace may be track alligned because it follows the MBR
			//     track on the disk.               

			if (starts_on_cylinder_boundary(ld,seg->start)==FALSE) {
				sector_count_t sector_offset;
				lba_t          cylinder_start;

				LOG_DEBUG("src seg doesnt start on cyl bdy\n");

				cylinder_start = rounddown_to_cylinder_boundary(ld,seg->start);
				sector_offset  = seg->start - cylinder_start;

				LOG_DEBUG("cyl start= %"PRId64"  sector offset= %"PRId64"\n", cylinder_start, sector_offset);           

				if (starts_on_cylinder_boundary(ld,freespace->start)==TRUE) {
					LOG_DEBUG("freespace starts on cyl bdy so simply offsetting into freespace by same amount\n");
					start = freespace->start + sector_offset;
				}
				else {
					lba_t  freespace_cylinder_start;

					LOG_DEBUG("freespace doesnt start on cyl bdy\n");

					cylinder_start = rounddown_to_cylinder_boundary(ld,freespace->start);
					freespace_cylinder_start = freespace->start-cylinder_start;

					if (sector_offset >= freespace_cylinder_start) {
						LOG_DEBUG("seg trk offset >= freespace trk offset\n");
						start = cylinder_start + sector_offset;
					}
					else {
						LOG_DEBUG("freespace trk offset > seg trk offset\n");
						start = roundup_to_cylinder_boundary(ld,freespace->start)+1;
						start += sector_offset;
					}
				}
			}
			else {
				LOG_DEBUG("src seg starts on cyl bdy\n");
				if (starts_on_cylinder_boundary(ld,freespace->start)==TRUE) {
					LOG_DEBUG("freespace also starts on cyl bdy\n");
					start = freespace->start;
				}
				else {
					LOG_DEBUG("freespace starts on trk bdy so need to round it up to cyl bdy\n");
					start = roundup_to_cylinder_boundary(ld,freespace->start)+1;
				}
			}


			trg->start = start;
			LOG_DEBUG("target seg will start at %"PRIu64"\n", start);
		}

		// now calculate the ending lba
		end = trg->start + seg->size - 1;
		LOG_DEBUG("target seg will end at %"PRIu64"\n", end);

		// error checks ...
		if (ends_on_cylinder_boundary(ld,end)==FALSE) {
			LOG_DEBUG("error ... end lba= %"PRIu64" ... doesnt end on cyl\n", end );
			rc = EINVAL;
		}
		else if (start > freespace->start+freespace->size-1 ) {
			LOG_DEBUG("error ... would start past end of freespace segment\n");             
		}
		else if (end > freespace->start+freespace->size-1 ) {
			LOG_DEBUG("error ... would end past end of freespace segment\n");
			rc = EINVAL;
		}
		else {
			trg->size = end - trg->start + 1;
			*target=trg;
			rc = 0;                 
		}

	}

	if (seg) {
		LOG_DEBUG("Src ...\n");
		LOG_DEBUG("     start= %"PRIu64"\n", seg->start);
		LOG_DEBUG("      size= %"PRIu64"\n", seg->size);
		LOG_DEBUG("       end= %"PRIu64"\n", seg->start+seg->size-1);
	}
	if (trg) {
		LOG_DEBUG("Target ...\n");
		LOG_DEBUG("     start= %"PRIu64"\n", trg->start);
		LOG_DEBUG("      size= %"PRIu64"\n", trg->size);
		LOG_DEBUG("       end= %"PRIu64"\n", trg->start+trg->size-1);
	}

	if (rc || testing==TRUE) {
		if (ebr) free_disk_segment(ebr);
		if (trg) free_disk_segment(trg);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*  Function:  validate move target
 *
 *  Called to validate that the specified segment
 *  can be moved to the target freespace segment.
 */
int dos_validate_move_target( DISKSEG *src, DISKSEG *trg )
{
	LOGICALDISK             *ld=NULL;
	DISKSEG                 *tseg;
	SEG_PRIVATE_DATA        *src_pdata=NULL;
	SEG_PRIVATE_DATA        *trg_pdata=NULL;
	DISK_PRIVATE_DATA       *disk_pdata=NULL;
	int rc=EINVAL;

	LOG_ENTRY();

	if (src && trg) {

		if ( src->data_type == DATA_TYPE &&
		     trg->data_type == FREE_SPACE_TYPE ) {

			ld          = get_logical_disk(src);
			disk_pdata  = get_disk_private_data(ld);
			src_pdata   = (SEG_PRIVATE_DATA *) src->private_data;
			trg_pdata   = (SEG_PRIVATE_DATA *) trg->private_data;

			if (ld && disk_pdata) {
				rc = 0;
			}
		}

		if (rc==0) {
			if (src_pdata->flags & SEG_IS_LOGICAL_PARTITION) {
				if (seg_is_within_or_adjacant_to_extended_partition(ld,trg)==FALSE) {
					rc = EINVAL;
				}
			}
			else {
				if (seg_is_within_the_extended_partition(ld,trg)==TRUE) {
					rc = EINVAL;
				}
			}
		}

		if (rc==0) {
			rc = create_move_target( src, trg, &tseg, TRUE);                                        
		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function:  can move segment
 *
 *  Check if the specified segment object can be moved
 *  elsewhere on the disk. Do so by ... checking each
 *  freespace segment we find ... stopping at the 1st
 *  freespace segment that could be used for the move.
 */
int dos_can_move_segment( DISKSEG *seg )
{
	DISKSEG       *freespace=NULL;  
	LOGICALDISK   *ld=NULL; 
	int rc=EINVAL;
	list_element_t iter;

	LOG_ENTRY();

	if ( safe_to_move(seg) == TRUE ) {

		ld = get_logical_disk(seg);
		if (ld) {

			LIST_FOR_EACH( ld->parent_objects, iter, freespace ) {

				if ( (freespace->data_type == FREE_SPACE_TYPE) &&
				     (freespace->size      >= seg->size) ) {

					rc = dos_validate_move_target(seg,freespace);
					if (rc == 0) {
						break;
					}

				}

			}

		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function:  dos update segment info
 *
 *  Called when finishing a move to update the partition
 *  table metadata info. This means we need to update the
 *  source segment with target segment info. We also need
 *  to remove the (no longer needed) target segment from
 *  the segment list, owned by the logical disk.
 */
static int dos_update_segment_info( DISKSEG *src, DISKSEG *trg )
{
	int rc=EINVAL;
	LOGICALDISK        *ld=NULL;
	DISK_PRIVATE_DATA  *disk_pdata=NULL;
	SEG_PRIVATE_DATA   *src_pdata=NULL;
	SEG_PRIVATE_DATA   *trg_pdata=NULL;
	boolean             logical_partition;

	LOG_ENTRY();

	if (src && trg) {

		ld         = get_logical_disk(src);
		disk_pdata = get_disk_private_data(ld); 
		src_pdata  = (SEG_PRIVATE_DATA *) src->private_data;            
		trg_pdata  = (SEG_PRIVATE_DATA *) trg->private_data;

		logical_partition = (src_pdata->flags & SEG_IS_LOGICAL_PARTITION) ? TRUE : FALSE;

		// remove segments from the disks segment list cuz ...
		// doing so will unregister the names and create freespace
		// areas on the disk.
		remove_diskseg_from_list( ld->parent_objects, src );
		remove_diskseg_from_list( ld->parent_objects, trg );
		if (logical_partition == TRUE) {
			remove_diskseg_from_list(ld->parent_objects, src_pdata->ebr );
			remove_diskseg_from_list(ld->parent_objects, trg_pdata->ebr );
		}

		// update source segment information
		src->start = trg->start;
		src->size  = trg->size;

		if (logical_partition==TRUE) {
			src_pdata->ebr->start = trg_pdata->ebr->start;
			src_pdata->ebr->size  = trg_pdata->ebr->size;
		}

		// take care of OS/2 dlat information
		if (src_pdata->dla_entry) {
			src_pdata->dla_entry->Partition_Start = src->start;
			src_pdata->dla_entry->Partition_Size  = src->size;
		}

		// reinsert the source segment storage object because ...
		// doing so will now build and register correct names.
		if (logical_partition==TRUE) {
			insert_diskseg_into_list(ld->parent_objects, src_pdata->ebr );
		}
		insert_diskseg_into_list( ld->parent_objects, src );

		// fixup the extended partition info if needed
		if (logical_partition==TRUE) {

			// recalc the size of the extended partition
			fixup_disk_extd_partition_dimensions( ld );

			// if logical drives are offset in the extd partition we need to
			// keep an anchor EBR in the 1st sector of the extd partition.
			fixup_disk_extd_partition_anchor(ld);

			// calc sizes of each std ebr partition
			fixup_EBR_Sizes(ld);                    
		}

		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*  Function:  DOS move segment
 *
 *  Called as a seg mgr private function to move a data segment
 *  to a freespace area on the same disk.
 */
int dos_move_segment( DISKSEG *src, DISKSEG *freespace )
{
	int rc=EINVAL;
	LOGICALDISK        *ld=NULL;
	DISKSEG            *trg=NULL;
	SEG_PRIVATE_DATA   *trg_pdata=NULL;
	SEG_PRIVATE_DATA   *src_pdata=NULL;
	DISK_PRIVATE_DATA  *disk_pdata=NULL;
	copy_job_t         *job=NULL;

	LOG_ENTRY();

	if ( safe_to_move(src) == TRUE ) {

		ld         = get_logical_disk(src);
		disk_pdata = get_disk_private_data(ld);
		src_pdata  = (SEG_PRIVATE_DATA *) src->private_data;                            

		// create a move target segment
		rc = create_move_target(src, freespace, &trg, FALSE);
		if (rc==0) trg_pdata=(SEG_PRIVATE_DATA *)trg->private_data;

		// create the engine copy job
		if (rc==0) {
			rc = create_copy_job( src, trg, &job );                                         
			if (rc) {
				if (trg_pdata->flags & SEG_IS_LOGICAL_PARTITION) {
					free_disk_segment(trg_pdata->ebr);
				}
				free_disk_segment( trg );
				find_freespace_on_disk(ld);
			}
		}

		// remove the freespace segment from the disk segment
		// list and insert the new move-target data segment.
		if (rc==0) {

			remove_diskseg_from_list( ld->parent_objects, freespace );
			free_disk_segment( freespace );

			if (trg_pdata->flags & SEG_IS_LOGICAL_PARTITION) {
				insert_diskseg_into_ordered_list(ld->parent_objects, trg_pdata->ebr );
			}
			insert_diskseg_into_ordered_list( ld->parent_objects, trg );

			disk_pdata->flags     |= DISK_HAS_MOVE_PENDING;         
			disk_pdata->copy_job   = job;
			src_pdata->move_target = trg;
			src->flags |= SOFLAG_DIRTY;
			find_freespace_on_disk(ld);
		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*  Function:  move segment finish
 *
 *  Called after a move to update partition tables.
 */
static int do_move_segment_finish( DISKSEG      *src, 
				   DISKSEG      *trg, 
				   int   copy_rc,
				   boolean       offline )
{
	int rc=EINVAL;  
	LOGICALDISK        *ld=NULL;
	SEG_PRIVATE_DATA   *src_pdata=NULL;
	SEG_PRIVATE_DATA   *trg_pdata=NULL;
	SEG_PRIVATE_DATA    saved_pdata;
	DISKSEG             saved_seg;
	DISKSEG             saved_ebr;
	DISK_PRIVATE_DATA  *disk_pdata=NULL;
	boolean             logical_partition;

	LOG_ENTRY();

	if (src && trg) {

		ld         = get_logical_disk(src);
		disk_pdata = get_disk_private_data(ld); 
		src_pdata  = (SEG_PRIVATE_DATA *)src->private_data;
		trg_pdata  = (SEG_PRIVATE_DATA *)trg->private_data;                             
		logical_partition = (src_pdata->flags & SEG_IS_LOGICAL_PARTITION) ? TRUE : FALSE;

		// if sectors were copied Ok ... then try to update partition tables to
		// complete the move ... restore the source segment if any errors occur.
		if (copy_rc==0) {
			memcpy(&saved_seg, src, sizeof(DISKSEG));
			memcpy(&saved_pdata, src_pdata, sizeof(SEG_PRIVATE_DATA));      
			if (logical_partition==TRUE) {
				memcpy(&saved_ebr, src_pdata->ebr, sizeof(DISKSEG));
			}

			rc = dos_update_segment_info(src,trg);
			if (rc==0) {
				rc = Commit_Disk_Partition_Tables(ld);
			}

			if (rc) {
				memcpy(src, &saved_seg, sizeof(DISKSEG));
				memcpy(src_pdata, &saved_pdata, sizeof(SEG_PRIVATE_DATA));
				if (logical_partition==TRUE) {
					memcpy(src_pdata->ebr, &saved_ebr, sizeof(DISKSEG));
				}
				Commit_Disk_Partition_Tables(ld);
			}
		}
		else {
			rc = copy_rc;
		}

		// now toss the move target segment object(s)
		if (logical_partition==TRUE) {
			remove_diskseg_from_list( ld->parent_objects, trg_pdata->ebr);
			free_disk_segment(trg_pdata->ebr);
		}
		remove_diskseg_from_list( ld->parent_objects, trg);
		free_disk_segment( trg );
		src_pdata->move_target = NULL;
		find_freespace_on_disk(ld);
		src->flags |= SOFLAG_NEEDS_ACTIVATE;
		SEG_activate(src);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int do_online_copy(storage_object_t *src, storage_object_t *tgt, copy_job_t *job)
{
	int rc = 0;
	dm_target_t *target=NULL;

	LOG_ENTRY();

	rc = EngFncs->copy_setup(job);
	if (rc != 0) {
		LOG_SERIOUS("Error code %d when setting up a copy job: %s\n", rc, EngFncs->strerror(rc));
		LOG_EXIT_INT(rc);
		return rc;
	}

	/*
	 * Remap the source segment object to point to the mirror rather than the
	 * source object.
	 */
	rc = EngFncs->dm_get_targets(src, &target);

	if (rc == 0) {
		target->data.linear->major = job->mirror->dev_major;
		target->data.linear->minor = job->mirror->dev_minor;
		target->data.linear->start = 0;

		rc = EngFncs->dm_load_targets(src, target);

		EngFncs->dm_deallocate_targets(target);

		if (rc == 0) {

			EngFncs->dm_set_suspended_flag(TRUE);

			rc = EngFncs->dm_suspend(src, TRUE);
			if (rc == 0) {
				rc = EngFncs->copy_start(job);
				if (rc) {
					EngFncs->dm_clear_targets(src);
				}
				EngFncs->dm_suspend(src, FALSE);
			}
			else {
				LOG_SERIOUS("Error code %d when resuming object %s: %s\n", 
					    rc, src->name, EngFncs->strerror(rc));
			}

			EngFncs->dm_set_suspended_flag(FALSE);

		}
		else {
			LOG_SERIOUS("Error code %d when loading the new mirror target for the object %s: %s\n", 
				    rc, src->name, EngFncs->strerror(rc));
		}

	}
	else {
		LOG_SERIOUS("Error code %d when getting the target for the object %s: %s\n", 
			    rc, src->name, EngFncs->strerror(rc));
	}

	if (rc == 0) {
		rc = EngFncs->copy_wait(job);
	}

	rc = do_move_segment_finish( src, tgt, rc, FALSE);

	EngFncs->copy_cleanup(job);

	LOG_EXIT_INT(rc);
	return rc;
}


static int do_offline_copy( storage_object_t *src, storage_object_t *tgt, copy_job_t *job )
{
	int rc = 0;

	LOG_ENTRY();

	rc = EngFncs->offline_copy( job);

	rc = do_move_segment_finish( src, tgt, rc, TRUE);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Function: dos move commit
 *
 *  Called during MOVE commit phase to commit a segment
 *  move to disk. This is done by calling the engine copy
 *  service and afterwards calling our own move finish
 *  routine that takes care of committing metadata changes
 *  or backing out changes.
 */
int dos_move_segment_commit( DISKSEG *src, DISKSEG *tgt, copy_job_t *job )
{
	int rc=0;       
	char * choices[] = {"Yes", "No", NULL};        
	int answer = 0;    
	logical_volume_t *vol=NULL;

	LOG_ENTRY();

	if (EngFncs->can_online_copy()) {
		rc = do_online_copy(src, tgt, job);
	}
	else {
		if (EngFncs->is_offline(src,&vol)==FALSE) {
			LOG_DEBUG("Segment %s appears to be part of mounted volume %s\n", src->name, vol->name);
			QUESTION( &answer, choices,
				  "Segment %s appears to be part of a mounted volume.  The volume is %s.\n\n"
				  "Offline move can safely be used only with unmounted volumes.  The volume may "
				  "become unuseable if you continue with the move.\n\n"
				  "Question: Would you like to continue and move segment %s?\n",
				  src->name, vol->name, src->name);

			if (answer != 0) {
				rc=EPERM;
				LOG_DEBUG("User is cancelling move\n");
			}
		}
		if (rc==0) {
			rc = do_offline_copy(src, tgt, job);
		}
	}

	src->flags &= ~SOFLAG_DIRTY;  // always exit with dirty flag cleared

	LOG_EXIT_INT(rc);
	return rc;
}
