/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: linear_mgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the Linear MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#include "md.h"
#include "linear_mgr.h"
#include "linear_discover.h"

#define my_plugin_record linear_plugin

static boolean linear_can_change_region_configuration( storage_object_t *region )
{
	boolean rc = TRUE;
	md_volume_t * volume = (md_volume_t *)region->private_data;

	if (volume->region_mgr_flags & MD_LINEAR_CONFIG_CHANGE_PENDING) {
		rc = FALSE;
	}
	
	LOG_EXIT_BOOL(rc);
	return rc;
}

static void linear_calculate_and_update_size(storage_object_t *region)
{
	md_volume_t * volume = (md_volume_t *)region->private_data;
	int i, count;

	/* Recalculate size */
	region->size = 0;
	for (i = 0, count = 0; count < volume->nr_disks; i++ ) {
		if (volume->child_object[i] && volume->super_array[i]) {
			region->size += MD_NEW_SIZE_SECTORS(volume->child_object[i]->size);
			++count;
		}
	}

	/* Update master superblock and children's superblock */
	volume->super_block->size = region->size / 2;		
	for (i = 0, count = 0; count < volume->nr_disks; i++ ) {
		if (volume->child_object[i] && volume->super_array[i]) {
			volume->super_array[i]->size = volume->super_block->size;
			++count;
		}
	}
	
}


/*
 * Look for this object in the list, return TRUE if found
 */
static boolean linear_find_object_in_list( list_anchor_t list, storage_object_t *obj )
{
	if (EngFncs->find_in_list(list, obj, NULL))
		return TRUE;
	else
		return FALSE;
}

static int linear_add_new_disk(md_volume_t * volume, storage_object_t *new_disk) {

	int       rc= 0,i;
	mdp_disk_t disk;

	LOG_ENTRY();

	rc = md_clone_superblock(volume, volume->nr_disks);
	if (rc) {
		LOG_EXIT_INT(rc);
		return rc;
	}
	i = volume->nr_disks;

	volume->child_object[i] = new_disk;
	md_append_region_to_object(volume->region, new_disk);

	disk.major = new_disk->dev_major;
	disk.minor = new_disk->dev_minor;
	disk.number = i;
	disk.raid_disk = disk.number;
	disk.state = (1 << MD_DISK_ACTIVE) | (1 << MD_DISK_SYNC);
	memcpy(&volume->super_block->disks[i], &disk, sizeof(mdp_disk_t));
	volume->super_block->active_disks++;
	volume->super_block->working_disks++;
	volume->super_block->raid_disks++;
	volume->super_block->nr_disks++;
	volume->nr_disks++;
	
	linear_calculate_and_update_size(volume->region);

	LOG_EXIT_INT(rc);
	return rc;
}

static int linear_remove_last_disk(md_volume_t * volume, storage_object_t *child, boolean kill_sects) {

	int rc = 0, i;

	LOG_ENTRY();
	i = volume->nr_disks - 1;
	if (volume->child_object[i] == child) {

		md_remove_region_from_object(volume->region, child);

		/* If we are unwinding expansion, don't clear the superblock */
		if (kill_sects)
			KILL_SECTORS(child, MD_NEW_SIZE_SECTORS(child->size),MD_RESERVED_SECTORS);
		
		EngFncs->engine_free(volume->super_array[i]);
		volume->super_array[i] = NULL;
		volume->child_object[i] = NULL;

		memset(&volume->super_block->disks[i], 0, sizeof(mdp_disk_t));  //NULL out now empty disk entry

		volume->super_block->raid_disks--;
		volume->super_block->active_disks--;
		volume->super_block->working_disks--;
		volume->super_block->nr_disks--;
		volume->nr_disks--;

		linear_calculate_and_update_size(volume->region);
	} else {
		LOG_ERROR(" %s is not the last disk of the %s region\n", child->name, volume->region->name);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_setup_evms_plugin
 *
 *	This function gets called shortly after the plugin is loaded by the
 *	Engine. It performs all tasks that are necessary before the initial
 *	discovery pass.
 */
static int linear_setup_evms_plugin(engine_functions_t	* functions)
{
	int rc = 0;

	// Parameter check
	if (!functions) {
		return EINVAL;
	}

	EngFncs = functions;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	rc = md_register_name_space();

	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Region Checking Functions ******/


/* All of the following md_can_ functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: linear_can_delete
 *
 *	Can we remove the specified MD logical volume, and consolidate the
 *	space with the freespace volume?
 */
static int linear_can_delete( storage_object_t * region )
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/* Function: linear_can_last_child_expand
 *
 * Notes: This function returns error code not boolean
 */
static int linear_can_last_child_expand(storage_object_t *region,
					u_int64_t        *expand_limit,
					list_anchor_t           expansion_points )
{
	int rc = 0;
	storage_object_t *child;
	md_volume_t * volume = (md_volume_t *)region->private_data;
	LOG_ENTRY();

	if (volume->super_block->nr_disks < 1) {
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}
	
	child = volume->child_object[volume->super_block->nr_disks-1];
	if (!child) {
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}

	rc = child->plugin->functions.plugin->can_expand(child, expand_limit, expansion_points);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_can_last_child_expand
 *
 * Notes: This function returns error code not boolean
 */
static int linear_can_last_child_shrink(storage_object_t *region,
					u_int64_t        *shrink_limit,
					list_anchor_t           shrink_points )
{
	int rc = 0;
	storage_object_t *child;
	md_volume_t * volume = (md_volume_t *)region->private_data;
	LOG_ENTRY();

	if (volume->super_block->nr_disks < 1) {
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}

	child = volume->child_object[volume->super_block->nr_disks-1];
	if (!child) {
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}

	rc = child->plugin->functions.plugin->can_shrink(child, shrink_limit, shrink_points);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_can_expand
 *
 * This region can be expanded if:
 *	The last child can be expanded -OR-
 *	There is at least 1 available object
 */
static int linear_can_expand(storage_object_t *region,
			     u_int64_t *expand_limit,
			     list_anchor_t expansion_points)
{
	int rc = 0;
	uint count;
	list_anchor_t tmp_list;
	list_element_t li=NULL;
	expand_object_info_t * expand_object;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	if (region->flags & SOFLAG_CORRUPT) {
		LOG_EXIT_INT(EPERM);
		return EPERM;
	}

	if (!linear_can_change_region_configuration(region)) {
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	/* Allow the last child to add expansion point */
	linear_can_last_child_expand(region, expand_limit, expansion_points);

	/* get a list of all valid input disks, segments, and regions. */
	EngFncs->get_object_list(DISK | SEGMENT | REGION,
				DATA_TYPE,
				NULL,
				NULL,
				VALID_INPUT_OBJECT,
				&tmp_list);
	
	/* Remove this MD object from the list */
	EngFncs->remove_thing(tmp_list, region);

	count = EngFncs->list_count(tmp_list);

	EngFncs->destroy_list(tmp_list);

	if (count) {
		expand_object = (expand_object_info_t *) EngFncs->engine_alloc( sizeof(expand_object_info_t) );
		if (expand_object) {
			expand_object->object          = region;
			expand_object->max_expand_size = -1;

			li = EngFncs->insert_thing(expansion_points,
						   expand_object,
						   INSERT_AFTER,
						   NULL);
			if (!li) {
				EngFncs->engine_free( expand_object );
				rc = ENOMEM;
			}
		} else {
			rc = ENOMEM;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_can_expand_by
 *
 *	Can we expand the specified region, by the specified size, using space
 *	from the freespace volume in the same group?
 */

static int linear_can_expand_by(storage_object_t	* child_object,
				u_int64_t		* size )
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: linear_can_shrink
 *
 * This region can be expanded if:
 *	The last child can be shrunk -OR-
 *	This region has more than 1 disk
 */
static int linear_can_shrink(storage_object_t *region,
			     u_int64_t *shrink_limit,
			     list_anchor_t shrink_points )
{
	int rc = 0;
	list_element_t li = NULL;
	md_volume_t *volume = (md_volume_t *)region->private_data;
	shrink_object_info_t * shrink_object;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	/*
	 * Note: In the future, we may consider shrinking a corrupt linear region.
	 * Especially the case of the missing child(ren) is the last one.
	 */
	if (region->flags & SOFLAG_CORRUPT) {
		LOG_EXIT_INT(EPERM);
		return EPERM;
	}

	if (!linear_can_change_region_configuration(region)) {
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	/* Allow the last child to add shrink point */
	linear_can_last_child_shrink(region, shrink_limit, shrink_points);

	if (volume->nr_disks > 1) {
		shrink_object = (shrink_object_info_t *) EngFncs->engine_alloc( sizeof(shrink_object_info_t) );
		if (shrink_object) {
			shrink_object->object          = region;
			shrink_object->max_shrink_size = region->size - MD_NEW_SIZE_SECTORS(volume->child_object[0]->size);
			li = EngFncs->insert_thing(shrink_points,
						   shrink_object,
						   INSERT_AFTER,
						   NULL);
			if (!li) {
				EngFncs->engine_free( shrink_object );
				rc = ENOMEM;
			}
		} else {
			rc = ENOMEM;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_can_shrink_by
 *
 *	Can we shrink the specified logical volume and consolidate the removed
 *	space into the freespace volume? If it can be shrunk, but not to
 *	exactly new_size, reset new_size to the next lower possible size.
 */
static int linear_can_shrink_by(storage_object_t	* region,
				u_int64_t		* size )
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


static int linear_can_replace_child(storage_object_t *region,
				    storage_object_t *child,
				    storage_object_t *new_child)
{
	int rc;
	my_plugin = linear_plugin;
	LOG_ENTRY();
	rc = md_can_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_discover
 *
 *	Examine all disk segments and find MD PVs. Assemble volume groups
 *	and export all MD logical volumes as EVMS regions.
 *
 *	All newly created regions must be added to the output list, and all
 *	segments from the input list must either be claimed or moved to the
 *	output list.
 */
static int linear_discover(list_anchor_t	input_list,
           		   list_anchor_t	output_list,
			   boolean	final_call )
{
	int count = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!input_list || !output_list) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (final_call) {
		md_discover_final_call(input_list, output_list, &count);
	} else {
		md_discover_volumes(input_list, output_list);
		LOG_DETAILS("Object discovery complete.\n");

		// LV discovery and exporting
		linear_discover_regions(output_list, &count, final_call);
		LOG_DETAILS("Object creation complete.\n");
	}

	LOG_EXIT_INT(count);
	return count;
}



/****** Region Functions ******/



/* Function: linear_create
 *
 *	Create a new MD volume.
 */
static int linear_create(list_anchor_t       	objects,
			option_array_t	* options,
			list_anchor_t		new_region_list )
{
	md_volume_t	* volume = NULL;
	storage_object_t * object;
	int nr_disks;
	unsigned long size = -1;
	int i, index = 0;
	int rc = 0;
	mdp_disk_t disk;
	list_element_t iter1, iter2;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	if ( !(volume = EngFncs->engine_alloc( sizeof(md_volume_t) ) )){
		LOG_CRITICAL("Memory error new volume structure.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	nr_disks = EngFncs->list_count(objects);
	if (nr_disks > MAX_MD_DEVICES) {
		LOG_CRITICAL("Too many objects for MD Linear create %d.\n",nr_disks);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	
	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		size = min(size, object->size);  // track smallest object for super block
		volume->child_object[index] = object;
		index ++;
		EngFncs->remove_element(iter1);
	}

	disk.number = 0;
	disk.raid_disk = 0;
	disk.state = (1 << MD_DISK_ACTIVE) | (1 << MD_DISK_SYNC);

	size = MD_NEW_SIZE_BLOCKS(size/2); // first convert sectors to blocks

	rc = md_create_first_superblock(volume, disk, pers_to_level(LINEAR), 4, size, nr_disks, 0, (1 << MD_SB_CLEAN));
	if (rc) {
		EngFncs->engine_free(volume);
		LOG_EXIT_INT(rc);
		return rc;
	}

	for (i = 0; i < nr_disks; i++) {
		rc = md_clone_superblock(volume, i);
		if (rc) {
			for (i--; i>=0; i--) {
				EngFncs->engine_free(volume->super_array[i]);
			}
			EngFncs->engine_free(volume->super_block);
			EngFncs->engine_free(volume);
			LOG_EXIT_INT(rc);
			return rc;
		}
	}

	volume->personality = LINEAR;
	volume->nr_disks = nr_disks;
	volume->next = volume_list_head;
	volume_list_head = volume;

	rc = linear_create_region(volume, new_region_list, TRUE);
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: w_delete
 *
 * Worker function for linear_delete and linear_discard
 */
static int w_delete(storage_object_t *region, list_anchor_t	children, boolean tear_down)
{
	md_volume_t	* volume;
	int		rc = 0;

	LOG_ENTRY();

	volume = region->private_data;

	// Remove the parent/child associations with the PVs
	md_clear_child_list(region, children);

	// Delete the volume.
	md_delete_volume(volume, tear_down);
	region->private_data = NULL;
	EngFncs->free_region(region);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_delete
 *
 *	Remove the specified region
 */
static int linear_delete(storage_object_t *region, list_anchor_t children)
{
	int		rc = 0;

	LOG_ENTRY();
	rc = w_delete(region, children, TRUE);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: linear_discard
 *
 * This function is similar to delete.  Just call delete to free all
 * data structures related to the regions.
 */
static int linear_discard(list_anchor_t regions)
{
	storage_object_t * region;
	list_element_t le;

	LOG_ENTRY();

	LIST_FOR_EACH(regions, le, region) {
		w_delete(region, NULL, FALSE);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/* Function: linear_expand_last_child
 *
 * Call expand() on the last child of the MD linear array to expand.
 */
static int linear_expand_last_child(storage_object_t	*region,
				    storage_object_t	*expand_object,
				    list_anchor_t		input_objects,
				    option_array_t	*options )
{
	int rc=0;
	md_volume_t *volume = (md_volume_t *)region->private_data;
	storage_object_t *child;
	u_int64_t cur_size;
	LOG_ENTRY();

	child = volume->child_object[volume->nr_disks-1];
	if (child != expand_object) {
		LOG_ERROR(" Error, expand obj (%s) is not the last child!\n", expand_object->name);
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	cur_size = child->size;
	LOG_DEBUG(" %s's current size = %"PRIu64" sectors.\n", child->name, child->size);
	KILL_SECTORS(child, MD_NEW_SIZE_SECTORS(child->size), MD_RESERVED_SECTORS);

	rc = child->plugin->functions.plugin->expand(child, expand_object, input_objects, options);
	if (!rc) {
		LOG_DEBUG(" %s's new size = %"PRIu64" sectors.\n", child->name, child->size);
		if (child->size > cur_size) {
			linear_calculate_and_update_size(region);
			volume->flags |= MD_DIRTY;
			region->flags |= SOFLAG_DIRTY;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_shrink_last_child
 *
 * Call shrink() on the last child of the MD linear array to shrink.
 */
static int linear_shrink_last_child(storage_object_t	*region,
				    storage_object_t	*shrink_object,
				    list_anchor_t		input_objects,
                                    option_array_t	*options )
{
	int rc=0;
	md_volume_t *volume = (md_volume_t *)region->private_data;
	storage_object_t *child;
	u_int64_t cur_size;
	LOG_ENTRY();

	child = volume->child_object[volume->nr_disks-1];
	if (child != shrink_object) {
		LOG_ERROR(" Error, shrink obj (%s) is not the last child!\n", shrink_object->name);
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	cur_size = child->size;
	LOG_DEBUG(" %s's current size = %"PRIu64" sectors.\n", child->name, child->size);
	KILL_SECTORS(child, MD_NEW_SIZE_SECTORS(child->size), MD_RESERVED_SECTORS);

	rc = child->plugin->functions.plugin->shrink(child, shrink_object, input_objects, options);
	if (!rc) {
		LOG_DEBUG(" %s's new size = %"PRIu64" sectors.\n", child->name, child->size);
		if (child->size < cur_size) {
			linear_calculate_and_update_size(region);
			volume->flags |= MD_DIRTY;
			region->flags |= SOFLAG_DIRTY;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Function: linear_expand
 *
 * If the last child is the target for this call,
 * let it handle the expansion.  Otherwise, add one disk at a time.
 */

static int linear_expand(storage_object_t	*region,
			 storage_object_t	*expand_object,
			 list_anchor_t       	input_objects,
			 option_array_t		*options )
{
	int rc = 0;
	md_volume_t *volume;
	int saved_nr_disks;
	storage_object_t *candidate;
	list_element_t li;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	if (region != expand_object) {
		rc = linear_expand_last_child(region, expand_object, input_objects, options);
	} else {

		/* We're about to perform the expansion */

		volume = (md_volume_t *)region->private_data;

		saved_nr_disks = volume->nr_disks; /* just in case... */

		LIST_FOR_EACH(input_objects, li, candidate) {
			rc = linear_add_new_disk(volume, candidate);
			if (rc) {
				break;
			}
		}

		if (!rc) {
			/* Expansion was successful */
			volume->flags |= MD_DIRTY;
			region->flags |= SOFLAG_DIRTY;
		} else {
			/* There was an error, unwind the expand */
			int last_idx;
			for (last_idx = volume->nr_disks-1;
				 last_idx >= saved_nr_disks; last_idx--) {
				if (volume->child_object[last_idx]) {
					linear_remove_last_disk(volume, volume->child_object[last_idx], FALSE);
				}
			}
		}
	}

	if (!rc) {
		region->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_shrink
 */
static int linear_shrink(storage_object_t	* region,
			storage_object_t	* shrink_object,
			list_anchor_t		input_objects,
			option_array_t		* options )
{
	int rc = 0;
	md_volume_t *volume;
	storage_object_t *candidate;
	int i, removed_count, selected_count;
	list_anchor_t removed_list;
	list_element_t li = NULL;
	
	my_plugin = linear_plugin;
	LOG_ENTRY();

	if (region != shrink_object) {
		rc = linear_shrink_last_child(region, shrink_object, input_objects, options);
	} else {
		
		/* We're about to perform the shrinking */

		volume = (md_volume_t *)region->private_data;

		selected_count = EngFncs->list_count(input_objects);
		
		i = volume->nr_disks - 1;
		removed_count = 0;
		removed_list = EngFncs->allocate_list();
		if (!removed_list) {
			LOG_ERROR("Can't allocate removed list.\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}


		while (!rc && (removed_count < selected_count)) {
			candidate = volume->child_object[i];
			if (linear_find_object_in_list(input_objects, candidate)) {
				rc = linear_remove_last_disk(volume, candidate, TRUE);
				if (!rc) {
					removed_count++;
					i--;
					li = EngFncs->insert_thing(removed_list,
								   candidate,
								   INSERT_AFTER,
								   NULL);
					if (!li) {
						rc = ENOMEM;
						LOG_ERROR("Could not insert to removed list.\n");
					}
				}
			} else {
				LOG_ERROR("%s does not exist in selected object list.\n",
					  candidate->name);
				rc = EINVAL;
				break;
			}
		}
		

		if (!rc && (removed_count == selected_count)) {
			/* The shrink was successful */
			volume->flags |= MD_DIRTY;
			region->flags |= SOFLAG_DIRTY;
		} else {

			/* There was an error, unwind the shrink */
			LIST_FOR_EACH(removed_list, li, candidate) {
				rc = linear_add_new_disk(volume, candidate);
				if (rc) {
					LOG_ERROR("Could not re-add %s back to %s region.\n",
						  candidate->name, volume->name);
					break;
				}
			}
		}
		
		EngFncs->destroy_list(removed_list);
	}

	if (!rc) {
		region->flags |= SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int linear_replace_child(storage_object_t *region,
				storage_object_t *child,
				storage_object_t *new_child)
{
	int rc;
	my_plugin = linear_plugin;
	LOG_ENTRY();
	rc = md_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_add_sectors_to_kill_list
 *
 *	The kill_sectors list contains a list of sectors that need to be zeroed
 *	during the next commit. This function is very similar to read/write.
 */
static int linear_add_sectors_to_kill_list( storage_object_t	* region,
					lsn_t			lsn,
					sector_count_t		count )
{

	lsn_t                      io_lsn = lsn;
	lsn_t                      link_lsn;
	sector_count_t             io_sector_count;
	sector_count_t             sectors_left_to_kill = count;
	sector_count_t             max_sector_count;
	sector_count_t             current_end = 0;
	sector_count_t             current_start = 0;
	md_volume_t 		  *volume = (md_volume_t *)region->private_data;
	int 			i;
	int			rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();


	if (volume->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, data is suspect \n",volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

        for (i=0; i < volume->nr_disks; i++) {

		current_end += MD_NEW_SIZE_SECTORS(volume->child_object[i]->size);
                if ( current_end >= io_lsn ) {

                    max_sector_count = current_end - io_lsn + 1;

                    if ( max_sector_count >= sectors_left_to_kill ) {
                        io_sector_count = sectors_left_to_kill;
                    }
                    else {
                        io_sector_count = max_sector_count;
                    }

                    link_lsn = io_lsn - current_start;

                    rc = KILL_SECTORS( volume->child_object[i],
                               link_lsn,
                               io_sector_count);

                    io_lsn               += io_sector_count;
                    sectors_left_to_kill -= io_sector_count;

                    if ((sectors_left_to_kill == 0) || (rc)) break;
                }
		current_start = current_end;

        }
	LOG_EXIT_INT(rc);
	return rc;
}

static int linear_commit_changes( storage_object_t * region, uint phase )
{
	md_volume_t	* volume;
	int		rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	if (!region ||  !(volume = (md_volume_t *)region->private_data)) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if ( ! (region->flags & SOFLAG_DIRTY) ) {
		LOG_WARNING("Region %s is not dirty - not committing.\n", region->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	switch (phase) {
	case FIRST_METADATA_WRITE:
		volume->flags |= MD_DIRTY;
		volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
		rc = md_write_sbs_to_disk(volume);   // write super blocks
		region->flags &= ~SOFLAG_DIRTY;  // mark clean
		break;
	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int linear_activate_region(storage_object_t * region)
{
	int rc = 0, i;
	md_volume_t   * vol = (md_volume_t *)region->private_data;
	dm_target_t *targets = NULL, *target=NULL;
	dm_device_t *dev;
	u_int64_t offset = 0;
	
	my_plugin = linear_plugin;
	LOG_ENTRY();
	
	for (i=0, rc=0; (!rc) && (i<MAX_MD_DEVICES); i++) {
        		
		/* TODO : create DM_TARGET_ERROR for stale objects ??? */
			
		if (vol->child_object[i]) {
		
			target = EngFncs->dm_allocate_target(
					DM_TARGET_LINEAR, offset,
					MD_NEW_SIZE_SECTORS(vol->child_object[i]->size), 0, 0);
			offset += MD_NEW_SIZE_SECTORS(vol->child_object[i]->size);
			if (target) {
				dev = target->data.linear;
				dev->major = vol->child_object[i]->dev_major;
				dev->minor = vol->child_object[i]->dev_minor;
				dev->start = 0;
				EngFncs->dm_add_target(target, &targets);
			} else {
				rc = ENOMEM;
			}
		}
	}

	if (!rc) {
		rc = EngFncs->dm_activate(region, targets);
		if (!rc) {
			region->flags &= ~SOFLAG_NEEDS_ACTIVATE;
			LOG_DEBUG("Region %s has been activated, DM device(%d, %d)\n",
				  region->name, region->dev_major, region->dev_minor);
		}
	}

	if (targets) EngFncs->dm_deallocate_targets(targets);

	LOG_EXIT_INT(rc);
	return rc;

}

static int linear_deactivate_region(storage_object_t * region)
{
	int rc;
	mdu_array_info_t info;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	rc = md_ioctl_get_array_info(region, &info);
	if (rc == 0) {
		/*
		 * This MD array was started by another tool.
		 * Stop the array via ioctl to the kernel MD driver.
		 */
		 rc = md_deactivate_region(region);
	} else {
		rc = EngFncs->dm_deactivate(region);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: linear_commit_changes
 *
 */


/* Function: linear_get_option_count
 *
 *	Determine the type of Task that is being performed, and return
 *	the number of options that are available for that Task.
 */
static int linear_get_option_count( task_context_t * task )
{
	int count = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	switch(task->action) {
	case EVMS_Task_Create:
		count = 0;
		break;
	case EVMS_Task_Expand:
		count = 0;
		break;
	case EVMS_Task_Shrink:
		count = 0;
		break;
	
	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}


/* Function: linear_init_task
 *
 *	Determine the type of Task that is being performed, and set up the
 *	context structure with the appropriate initial values.
 */
static int linear_init_task( task_context_t * context )
{
	int i, rc = 0;
	list_anchor_t tmp_list;
	list_element_t li=NULL;
	md_volume_t * volume;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	switch(context->action) {

	case EVMS_Task_Create:

		context->option_descriptors->count = 0;

		// get a list of all valid input disks, segments, and regions.
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					DATA_TYPE,
					NULL,
					NULL,
					VALID_INPUT_OBJECT,
					&tmp_list);

		// move these items to the acceptable objects list.
		md_transfer_list(tmp_list, context->acceptable_objects);
		EngFncs->destroy_list(tmp_list);
       		context->min_selected_objects = 1;
       		context->max_selected_objects = MAX_MD_DEVICES;
		break;

	case EVMS_Task_Expand:
		
		volume = (md_volume_t *) context->object->private_data;
		context->option_descriptors->count = 0;

		/* get a list of all valid input disks, segments, and regions. */
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					DATA_TYPE,
					NULL,
					NULL,
					VALID_INPUT_OBJECT,
					&tmp_list);

		/* Remove this MD object from the list */
		EngFncs->remove_thing(tmp_list, context->object);
		
		/* move these items to the acceptable objects list. */
		md_transfer_list(tmp_list, context->acceptable_objects);
		EngFncs->destroy_list(tmp_list);
		context->min_selected_objects = 1;
		context->max_selected_objects = MAX_MD_DEVICES - volume->super_block->nr_disks;
		break;

	case EVMS_Task_Shrink:
		
		volume = (md_volume_t *) context->object->private_data;
		context->option_descriptors->count = 0;

		if (volume->nr_disks <= 1) {
			rc = EINVAL;
			break;
		}

		/* Clear out the spare_disks list. */
		EngFncs->delete_all_elements(context->acceptable_objects);

		/* start from the last disk */
		for (i = volume->nr_disks-1; (rc == 0) && (i > 0); i-- ) {
			/* Check for null object, if missing, skip. */
			if (volume->child_object[i]) {

				li = EngFncs->insert_thing(context->acceptable_objects,
							   volume->child_object[i],
							   INSERT_AFTER,
							   NULL);
				if (!li) {
					LOG_ERROR("Could not insert MD child into acceptable object list.\n");
					rc = ENOMEM;
				}
			}
		}

       	context->min_selected_objects = 1;
       	context->max_selected_objects = volume->nr_disks - 1;
		break;

	case MD_LINEAR_FUNCTION_FIX:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		context->option_descriptors->count = 0;
		break;

	case MD_LINEAR_RESTORE_SUPERBLOCK:
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		context->option_descriptors->count = 0;
		break;
	
	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_set_option
 *
 *	Determine the type of Task that is being performed. Then examine the
 *	desired option (using the index), and verify that the given value is
 *	appropriate. Reset the value if necessary and possible. Adjust other
 *	options as appropriate.
 */
static int linear_set_option(task_context_t	* context,
			u_int32_t		index,
			value_t			* value,
			task_effect_t		* effect )
{
	int rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	switch (context->action) {

	case EVMS_Task_Create:
		// no options, just return 0
		break;
	
	case EVMS_Task_Expand:
		// no options, just return 0
		break;
	
	case EVMS_Task_Shrink:
		// no options, just return 0
		break;

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int linear_set_expand_object( task_context_t * context,
				     list_anchor_t declined_objects,
				     task_effect_t  * effect )
{
	int                      rc=0;
	list_element_t		li=NULL;
	list_element_t		li_declined=NULL;
	storage_object_t        *obj=NULL;
	declined_object_t       *declined_object=NULL;
	int                      selected_objects_count=0;
	int                      declined_object_count=0;
	int                      max_objects_allowed=0;
	md_volume_t *volume = (md_volume_t *)context->object->private_data;


	LOG_ENTRY();

	/* determine how many disks we can add */
	if (context) {
		max_objects_allowed = MAX_MD_DEVICES - volume->super_block->nr_disks;
		if (!max_objects_allowed) {
			LOG_EXIT_INT(EOVERFLOW);
			return EOVERFLOW;
		}
	} else {
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LIST_FOR_EACH(context->selected_objects, li, obj) {

		if ( max_objects_allowed > selected_objects_count ) {

			++selected_objects_count;
			*effect |=  EVMS_Effect_Reload_Options;

		} else {

			LOG_WARNING("Overflow : declining object (%s)\n", obj->name);

			++declined_object_count;

			declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

			if (declined_object) {

				declined_object->object = obj;
				declined_object->reason = EOVERFLOW;

				li_declined = EngFncs->insert_thing(declined_objects,
								    declined_object,
								    INSERT_AFTER,
								    NULL);
				if (li_declined) {
					*effect |=  EVMS_Effect_Reload_Objects;
				} else {
					EngFncs->engine_free(declined_object);
					LOG_ERROR("Could not insert declined object into declined object list\n");
					rc = ENOMEM;
				}

			} else {
				LOG_ERROR("Could not allocate memory for a declined object.\n");
				rc = ENOMEM;
			}

		}
		if (rc) {
			break;
		}
	}


	if (declined_object_count) {
		*effect |= EVMS_Effect_Reload_Objects;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int linear_set_shrink_object( task_context_t * context,
				     list_anchor_t declined_objects,
				     task_effect_t  * effect )
{
	int                      rc=0;
	list_element_t li_declined = NULL;
	declined_object_t       *declined_object=NULL;
	int declined_object_count=0;
	int max_objects_allowed=0;
	int i, verified_count, selected_count;
	md_volume_t *volume = (md_volume_t *)context->object->private_data;


	LOG_ENTRY();

	/* determine how many disks we can remove */
	if (context) {
		max_objects_allowed = volume->nr_disks - 1;
	} else {
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	selected_count = EngFncs->list_count(context->selected_objects);
	if (selected_count > max_objects_allowed) {
		LOG_ERROR(" Error, too many selected objects!\n");
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* All selected disks must at the end of MD linear array */

	verified_count = 0;
	i = volume->nr_disks-1;
	while ( verified_count < selected_count ) {
		if (linear_find_object_in_list(context->selected_objects, volume->child_object[i])) {
			verified_count++;
			i--;
		} else {
			LOG_WARNING(" could not find %s in the selected obj list. Stop\n", volume->child_object[i]->name);
			rc = EINVAL;
			break;
		}
	}

	if (verified_count == selected_count) {
		*effect |=  EVMS_Effect_Reload_Options;
		LOG_EXIT_INT(0);
		return 0;
	}

	/* Build declined objects */

	for (; i >= 0 && !rc; i--) {
		if (linear_find_object_in_list(context->selected_objects, volume->child_object[i])) {

			LOG_WARNING(" declining object (%s)\n", volume->child_object[i]->name);
			++declined_object_count;

			declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));


			if (declined_object) {

				declined_object->object = volume->child_object[i];
				declined_object->reason = EINVAL;

				li_declined = EngFncs->insert_thing(declined_objects,
								    declined_object,
								    INSERT_AFTER,
								    NULL);

				if (li_declined) {
					*effect |=  EVMS_Effect_Reload_Objects;
				} else {
					EngFncs->engine_free(declined_object);
					LOG_ERROR("Could not insert declined object to declined object list.\n");
					rc = ENOMEM;
				}
			} else {
				LOG_ERROR("Could not allocate memory for a declined object.\n");
				rc = ENOMEM;
			}
		}
	} /* for() */

	if (declined_object_count) {
		*effect |= EVMS_Effect_Reload_Objects;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_set_objects
 *
 * Validate the objects in the selected_objects dlist in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects dlist.
 * Modify the accepatble_objects dlist in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
static int linear_set_objects(	task_context_t	* context,
				list_anchor_t		declined_objects,
				task_effect_t	* effect )
{
	int rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	switch(context->action) {

	case EVMS_Task_Create:
		// since this is within task context, and we provided the initial list
		// do we need any further validation here?
		rc = 0;
		break;
	
	case EVMS_Task_Expand:
		rc = linear_set_expand_object( context, declined_objects, effect );
		break;
	
	case EVMS_Task_Shrink:
		rc = linear_set_shrink_object( context, declined_objects, effect );
		break;

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_get_info
 *
 *	Return MD-specific information about the specified region. If the
 *	name field is set, only return the "extra" information pertaining
 *	to that name.
 */
static int linear_get_info(	storage_object_t	* region,
				char			* name,
				extended_info_array_t	** info_array )
{
	md_volume_t * volume = NULL;
	int           rc= 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	volume = region->private_data;

	rc = md_get_info(volume, name, info_array);

        LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_get_plugin_info
 *
 *	Return information about the MD plugin. There is no "extra"
 *	information about MD, so "name" should always be NULL.
 */
static int linear_get_plugin_info(char			* name,
				  extended_info_array_t	** info_array )
{
	extended_info_array_t	* info = NULL;
	char			buffer[50] = {0};
	int			i = 0;
	
	my_plugin = linear_plugin;
	LOG_ENTRY();

	// Parameter check
	if ( ! info_array ) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if ( ! name ) {
		// Get memory for the info array
		if ( ! (info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*5)) ) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		// Short Name
		SET_STRING(info->info[i].name, "ShortName");
		SET_STRING(info->info[i].title, "Short Name");
		SET_STRING(info->info[i].desc, "A short name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, linear_plugin->short_name);
		i++;

		// Long Name
		SET_STRING(info->info[i].name, "LongName");
		SET_STRING(info->info[i].title, "Long Name");
		SET_STRING(info->info[i].desc, "A long name given to this plugin");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, linear_plugin->long_name);
		i++;

		// Plugin Type
		SET_STRING(info->info[i].name, "Type");
		SET_STRING(info->info[i].title, "Plugin Type");
		SET_STRING(info->info[i].desc, "There are various types of plugins; each responsible for some kind of storage object.");
		info->info[i].type = EVMS_Type_String;
		SET_STRING(info->info[i].value.s, "Region Manager");
		i++;

		// Plugin Version
		SET_STRING(info->info[i].name, "Version");
		SET_STRING(info->info[i].title, "Plugin Version");
		SET_STRING(info->info[i].desc, "This is the version number of the plugin.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required Engine Services Version
		SET_STRING(info->info[i].name, "Required_Engine_Version");
		SET_STRING(info->info[i].title, "Required Engine Services Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine services that this plug-in requires.  It will not run on older versions of the Engine services.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", linear_plugin->required_engine_api_version.major, linear_plugin->required_engine_api_version.minor, linear_plugin->required_engine_api_version.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

		// Required Plug-in API Version
		SET_STRING(info->info[i].name, "Required_Plugin_Version");
		SET_STRING(info->info[i].title, "Required Plug-in API Version");
		SET_STRING(info->info[i].desc, "This is the version of the Engine plug-in API that this plug-in requires.  It will not run on older versions of the Engine plug-in API.");
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", linear_plugin->required_plugin_api_version.plugin.major, linear_plugin->required_plugin_api_version.plugin.minor, linear_plugin->required_plugin_api_version.plugin.patchlevel);
		SET_STRING(info->info[i].value.s, buffer);
		i++;

	}
	else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	info->count = i;
	*info_array = info;
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: linear_read
 *
 *	Perform a logical-to-physical remapping, and send the read down to
 *	the next plugin.
 */
static int linear_read(	storage_object_t	* region,
			lsn_t			lsn,
			sector_count_t		count,
			void			* buffer )
{
	char                      *io_buffer_ptr = (char *)buffer;
	lsn_t                      io_lsn = lsn;
	lsn_t                      link_lsn;
	sector_count_t             io_sector_count;
	sector_count_t             sectors_left_to_read = count;
	sector_count_t             max_sector_count;
	sector_count_t             current_end = 0;
	sector_count_t             current_start = 0;
	md_volume_t 		  *volume = (md_volume_t *)region->private_data;
	int 			i;
	int			rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	// to be in sync with the kernel MD driver we are not supporting
	// partial exports at this time.  Use EVMS Drive linking instead.
	if (volume->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, returning zero filled buffer.\n",volume->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	rc = md_region_rw(region, lsn, count, buffer, 0);
	if (rc) {
		rc = 0;
		for (i=0; i < volume->nr_disks; i++) {
	
			current_end += MD_NEW_SIZE_SECTORS(volume->child_object[i]->size);
			if ( current_end >= io_lsn ) {
	
			    max_sector_count = current_end - io_lsn + 1;
	
			    if ( max_sector_count >= sectors_left_to_read ) {
				io_sector_count = sectors_left_to_read;
			    }
			    else {
				io_sector_count = max_sector_count;
			    }
	
			    link_lsn = io_lsn - current_start;
	
			    rc = READ( volume->child_object[i],
				       link_lsn,
				       io_sector_count,
				       (void *) io_buffer_ptr );
	
			    io_lsn               += io_sector_count;
			    io_buffer_ptr        += io_sector_count*EVMS_VSECTOR_SIZE;
			    sectors_left_to_read -= io_sector_count;
	
			    if ((sectors_left_to_read == 0) || (rc)) break;
			}
			current_start = current_end;
	
		}

	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: linear_write
 *
 *	Perform a logical-to-physical remapping, and send the write down to
 *	the next plugin.
 */
static int linear_write(storage_object_t	* region,
			lsn_t			lsn,
			sector_count_t		count,
			void			* buffer )
{
	char                      *io_buffer_ptr = (char *)buffer;
	lsn_t                      io_lsn = lsn;
	lsn_t                      link_lsn;
	sector_count_t             io_sector_count;
	sector_count_t             sectors_left_to_write = count;
	sector_count_t             max_sector_count;
	sector_count_t             current_end = 0;
	sector_count_t             current_start = 0;
	md_volume_t 		  *volume = (md_volume_t *)region->private_data;
	int 		 	  i;
	int			  rc = 0;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	if (volume->flags & MD_CORRUPT) {
		MESSAGE("MD region %s is corrupt, writing data is not allowed.\n",volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	rc = md_region_rw(region, lsn, count, buffer, 1);
	if (rc) {
		rc = 0;
		for (i=0; i < volume->nr_disks; i++) {

			current_end += MD_NEW_SIZE_SECTORS(volume->child_object[i]->size);
			if ( current_end >= io_lsn ) {

			    max_sector_count = current_end - io_lsn + 1;

			    if ( max_sector_count >= sectors_left_to_write ) {
				io_sector_count = sectors_left_to_write;
			    }
			    else {
				io_sector_count = max_sector_count;
			    }

			    link_lsn = io_lsn - current_start;

			    rc = WRITE( volume->child_object[i],
				       link_lsn,
				       io_sector_count,
				       (void *) io_buffer_ptr );

			    io_lsn               += io_sector_count;
			    io_buffer_ptr        += io_sector_count*EVMS_VSECTOR_SIZE;
			    sectors_left_to_write -= io_sector_count;

			    if ((sectors_left_to_write == 0) || (rc)) break;
			}
			current_start = current_end;

		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function:  linear_get_plugin_functions
 */
static int linear_get_plugin_functions(storage_object_t        *region,
				      function_info_array_t * * functions) {

	int rc = 0;
	function_info_array_t * fia;
	int function_count;
	md_volume_t * md_volume;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	/*
	 * If object is NULL, that means the user is asking for plug-in
	 * functions on the plug-in.  We don't have any plug-in functions that
	 * are global for the plug-in.
	 */
	if (region == NULL) {
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	md_volume = (md_volume_t *) region->private_data;

	if (md_volume->flags & MD_CORRUPT) {
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) + sizeof(function_info_t) * (MD_LINEAR_FUNCTION_COUNT - 1));

	if (fia != NULL) {
		function_count = 0;

		/*
		 * Our functions are only allowed if the Engine is opened for
		 * writing.
		 */
		if (EngFncs->get_engine_mode() & ENGINE_WRITE) {

			/*
			 * If the array needs fixing, then that is the only function
			 * available.
			 */
			if (linear_verify_and_fix_array(md_volume, 0, 0)) {

				fia->info[function_count].function = MD_LINEAR_FUNCTION_FIX;
				SET_STRING(fia->info[function_count].name, "fix");
				SET_STRING(fia->info[function_count].title, "Fix");
				SET_STRING(fia->info[function_count].verb, "Fix");
				SET_STRING(fia->info[function_count].help, "The RAID array has inconsistent metadata.  Use this function to fix the metadata.");

				function_count++;

			} else {

				if (md_can_restore_saved_sb(region)) {
					fia->info[function_count].function = MD_LINEAR_RESTORE_SUPERBLOCK;
					SET_STRING(fia->info[function_count].name, "ressuperblock");
					SET_STRING(fia->info[function_count].title, "Restore orginal major/minor");
					SET_STRING(fia->info[function_count].verb, "Restore");
					SET_STRING(fia->info[function_count].help, "Use this function to restore the original major and minor of all devices made up the MD array.");
					function_count++;
				}
			}
		}

		fia->count = function_count;
		*functions = fia;

	} else {
		LOG_CRITICAL("Error allocating memory for an action info array.\n");
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function:  linear_plugin_function
 */
static int linear_plugin_function(storage_object_t * object,
				 task_action_t      action,
				 list_anchor_t            objects,
				 option_array_t   * options) {

	int rc = 0;
	md_volume_t * volume = (md_volume_t *) object->private_data;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	if ((action < EVMS_Task_Plugin_Function) ||
	    (action >= EVMS_Task_Plugin_Function + MD_LINEAR_FUNCTION_COUNT)) {
		LOG_ERROR("Action code 0x%x is out of range.\n", action);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	switch (action) {
		case MD_LINEAR_FUNCTION_FIX:
			linear_verify_and_fix_array(volume, 1, 0);
			break;
		case MD_LINEAR_RESTORE_SUPERBLOCK:
			md_restore_saved_sb(volume);
			break;
		default:
			rc = EINVAL;
	}	

	if (rc == 0) {
		volume->region->flags |= SOFLAG_DIRTY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static void linear_plugin_cleanup(void) {

	list_anchor_t regions_list;
	list_element_t li;
	storage_object_t * region;
	md_volume_t * volume;

	my_plugin = linear_plugin;
	LOG_ENTRY();

	EngFncs->get_object_list(REGION, DATA_TYPE, linear_plugin, NULL, 0, &regions_list);

	LIST_FOR_EACH(regions_list, li, region) {
		volume = (md_volume_t *)region->private_data;
		md_free_volume(volume);
	}

	EngFncs->destroy_list(regions_list);

	LOG_EXIT_VOID();
	return;
}


/* Function tables for the MD Region Manager */
static plugin_functions_t linear_functions = {
	setup_evms_plugin		: linear_setup_evms_plugin,
	cleanup_evms_plugin		: linear_plugin_cleanup,
	can_delete			: linear_can_delete,
	can_expand			: linear_can_expand,
	can_expand_by			: linear_can_expand_by,
	can_shrink			: linear_can_shrink,
	can_shrink_by			: linear_can_shrink_by,
	can_replace_child		: linear_can_replace_child,
	discover			: linear_discover,
	create				: linear_create,
	delete				: linear_delete,
	discard				: linear_discard,
	expand				: linear_expand,
	shrink				: linear_shrink,
	replace_child			: linear_replace_child,
	add_sectors_to_kill_list	: linear_add_sectors_to_kill_list,
	commit_changes			: linear_commit_changes,
	activate			: linear_activate_region,
	deactivate			: linear_deactivate_region,
	get_option_count		: linear_get_option_count,
	init_task			: linear_init_task,
	set_option			: linear_set_option,
	set_objects			: linear_set_objects,
	get_info			: linear_get_info,
	get_plugin_info			: linear_get_plugin_info,
	read				: linear_read,
	write				: linear_write,
	get_plugin_functions		: linear_get_plugin_functions,
	plugin_function			: linear_plugin_function
};



/* Function: PluginInit
 *
 *	Initializes the local plugin record
 */

plugin_record_t linear_plugin_record = {
    id:                    SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 4),

    version:              {major:      MAJOR_VERSION,
                           minor:      MINOR_VERSION,
                           patchlevel: PATCH_LEVEL},

    required_engine_api_version: {major:      13,
				  minor:      0,
				  patchlevel: 0},
    required_plugin_api_version: {plugin: {major:      12,
					   minor:      0,
					   patchlevel: 0} },

    short_name:            "MDLinearRegMgr",
    long_name:             "MD Linear Raid Region Manager",
    oem_name:              "IBM",

    functions:             {plugin: &linear_functions},

    container_functions:   NULL
};
