/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libs390.so
 *
 *   File: helpers.c
 */
/*
 *  Special thanks to dasdfmt and fdasd code.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#include "vtoc.h"
#include "390segmgr.h"
#include "dm.h"
#include "helpers.h"


static unsigned char EBCtoASC[256] =
{
/* 0x00  NUL   SOH   STX   ETX  *SEL    HT  *RNL   DEL */
        0x00, 0x01, 0x02, 0x03, 0x07, 0x09, 0x07, 0x7F,
/* 0x08  -GE  -SPS  -RPT    VT    FF    CR    SO    SI */
        0x07, 0x07, 0x07, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
/* 0x10  DLE   DC1   DC2   DC3  -RES   -NL    BS  -POC
                                -ENP  ->LF             */
        0x10, 0x11, 0x12, 0x13, 0x07, 0x0A, 0x08, 0x07,
/* 0x18  CAN    EM  -UBS  -CU1  -IFS  -IGS  -IRS  -ITB
                                                  -IUS */
        0x18, 0x19, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
/* 0x20  -DS  -SOS    FS  -WUS  -BYP    LF   ETB   ESC
                                -INP                   */
        0x07, 0x07, 0x1C, 0x07, 0x07, 0x0A, 0x17, 0x1B,
/* 0x28  -SA  -SFE   -SM  -CSP  -MFA   ENQ   ACK   BEL
                     -SW                               */
        0x07, 0x07, 0x07, 0x07, 0x07, 0x05, 0x06, 0x07,
/* 0x30 ----  ----   SYN   -IR   -PP  -TRN  -NBS   EOT */
        0x07, 0x07, 0x16, 0x07, 0x07, 0x07, 0x07, 0x04,
/* 0x38 -SBS   -IT  -RFF  -CU3   DC4   NAK  ----   SUB */
        0x07, 0x07, 0x07, 0x07, 0x14, 0x15, 0x07, 0x1A,
/* 0x40   SP   RSP                         ----       */
        0x20, 0xFF, 0x83, 0x84, 0x85, 0xA0, 0x07, 0x86,
/* 0x48                      .     <     (     +     | */
        0x87, 0xA4, 0x9B, 0x2E, 0x3C, 0x28, 0x2B, 0x7C,
/* 0x50    &                                      ---- */
        0x26, 0x82, 0x88, 0x89, 0x8A, 0xA1, 0x8C, 0x07,
/* 0x58               !     $     *     )     ;       */
        0x8D, 0xE1, 0x21, 0x24, 0x2A, 0x29, 0x3B, 0xAA,
/* 0x60    -     /  ----       ----  ----  ----       */
        0x2D, 0x2F, 0x07, 0x8E, 0x07, 0x07, 0x07, 0x8F,
/* 0x68             ----     ,     %     _     >     ? */
        0x80, 0xA5, 0x07, 0x2C, 0x25, 0x5F, 0x3E, 0x3F,
/* 0x70  ---        ----  ----  ----  ----  ----  ---- */
        0x07, 0x90, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
/* 0x78    *     `     :     #     @     '     =     " */
        0x70, 0x60, 0x3A, 0x23, 0x40, 0x27, 0x3D, 0x22,
/* 0x80    *     a     b     c     d     e     f     g */
        0x07, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
/* 0x88    h     i              ----  ----  ----       */
        0x68, 0x69, 0xAE, 0xAF, 0x07, 0x07, 0x07, 0xF1,
/* 0x90         j     k     l     m     n     o     p */
        0xF8, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70,
/* 0x98    q     r                    ----        ---- */
        0x71, 0x72, 0xA6, 0xA7, 0x91, 0x07, 0x92, 0x07,
/* 0xA0          ~     s     t     u     v     w     x */
        0xE6, 0x7E, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
/* 0xA8    y     z              ----  ----  ----  ---- */
        0x79, 0x7A, 0xAD, 0xAB, 0x07, 0x07, 0x07, 0x07,
/* 0xB0    ^                    ----       ----       */
        0x5E, 0x9C, 0x9D, 0xFA, 0x07, 0x07, 0x07, 0xAC,
/* 0xB8       ----     [     ]  ----  ----  ----  ---- */
        0xAB, 0x07, 0x5B, 0x5D, 0x07, 0x07, 0x07, 0x07,
/* 0xC0    {     A     B     C     D     E     F     G */
        0x7B, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
/* 0xC8    H     I  ----                         ---- */
        0x48, 0x49, 0x07, 0x93, 0x94, 0x95, 0xA2, 0x07,
/* 0xD0    }     J     K     L     M     N     O     P */
        0x7D, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50,
/* 0xD8    Q     R  ----                              */
        0x51, 0x52, 0x07, 0x96, 0x81, 0x97, 0xA3, 0x98,
/* 0xE0    \           S     T     U     V     W     X */
        0x5C, 0xF6, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
/* 0xE8    Y     Z        ----       ----  ----  ---- */
        0x59, 0x5A, 0xFD, 0x07, 0x99, 0x07, 0x07, 0x07,
/* 0xF0    0     1     2     3     4     5     6     7 */
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
/* 0xF8    8     9  ----  ----       ----  ----  ---- */
        0x38, 0x39, 0x07, 0x07, 0x9A, 0x07, 0x07, 0x07
};

static unsigned char ASCtoEBC[256] =
{
        /*00  NL    SH    SX    EX    ET    NQ    AK    BL */
        0x00, 0x01, 0x02, 0x03, 0x37, 0x2D, 0x2E, 0x2F,
        /*08  BS    HT    LF    VT    FF    CR    SO    SI */
        0x16, 0x05, 0x15, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        /*10  DL    D1    D2    D3    D4    NK    SN    EB */
        0x10, 0x11, 0x12, 0x13, 0x3C, 0x15, 0x32, 0x26,
        /*18  CN    EM    SB    EC    FS    GS    RS    US */
        0x18, 0x19, 0x3F, 0x27, 0x1C, 0x1D, 0x1E, 0x1F,
        /*20  SP     !     "     #     $     %     &     ' */
        0x40, 0x5A, 0x7F, 0x7B, 0x5B, 0x6C, 0x50, 0x7D,
        /*28   (     )     *     +     ,     -    .      / */
        0x4D, 0x5D, 0x5C, 0x4E, 0x6B, 0x60, 0x4B, 0x61,
        /*30   0     1     2     3     4     5     6     7 */
        0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
        /*38   8     9     :     ;     <     =     >     ? */
        0xF8, 0xF9, 0x7A, 0x5E, 0x4C, 0x7E, 0x6E, 0x6F,
        /*40   @     A     B     C     D     E     F     G */
        0x7C, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
        /*48   H     I     J     K     L     M     N     O */
        0xC8, 0xC9, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6,
        /*50   P     Q     R     S     T     U     V     W */
        0xD7, 0xD8, 0xD9, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6,
        /*58   X     Y     Z     [     \     ]     ^     _ */
        0xE7, 0xE8, 0xE9, 0xAD, 0xE0, 0xBD, 0x5F, 0x6D,
        /*60   `     a     b     c     d     e     f     g */
        0x79, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
        /*68   h     i     j     k     l     m     n     o */
        0x88, 0x89, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96,
        /*70   p     q     r     s     t     u     v     w */
        0x97, 0x98, 0x99, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6,
        /*78   x     y     z     {     |     }     ~    DL */
        0xA7, 0xA8, 0xA9, 0xC0, 0x4F, 0xD0, 0xA1, 0x07,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0xFF
};







// list for keeping disk private data areas
list_anchor_t  Disk_PrivateData_List=NULL;
#define DISK_PDATA_TAG  0x08080808





/*
 *  A freespace has a name that has 3 parts to it.
 *  (1) a parent object name like hda
 *  (2) the word freespace
 *  (3) a number used to keep freespace seg names different
 *
 *  Examples:  hda_freespace1
 *             hda_freespace2
 *             dasdh3_freespace4
 *
 *  This routine returns the number found at the end
 *  of the freespace segment name, ignoring any differences
 *  that may be found in other parts of the name.
 *
 *  Returns: success: integer > 0
 *           failure: -1
 *
 */
static int get_freespace_number(DISKSEG *freespace)
{
        int number=-1;
        int i;

        if (freespace) {

                if (freespace->name) {

                        // pretty liberal here...
                        // minimum string has length of 2 ... freespac[eN]
                        if (strlen(freespace->name)>=2) {

                                // walk backwards
                                for (i=strlen(freespace->name)-1; i>0; i--) {

                                        // first e we find ends the search
                                        if ( freespace->name[i-1] == 'e' ) {

                                                // convert ascii number to integer
                                                return(  atoi( &freespace->name[i] ) );

                                        }

                                }

                        }

                }

        }

        return number;
}


/*
 *  Called by the get_name_for_disk_segment() routine to come up
 *  with the next available number to use when naming freespace
 *  segments on this drive.
 */
static int  get_next_avail_freespace_number(LOGICALDISK *ld)
{
        int freespace_number=0;
        DISKSEG *seg;
        list_element_t iter;

        LOG_ENTRY();

        LIST_FOR_EACH( ld->parent_objects, iter, seg ) {

                if (seg->data_type == FREE_SPACE_TYPE) {

                        if ( get_freespace_number(seg) > freespace_number ) {
                                freespace_number = get_freespace_number(seg);
                        }

                }

        }

        ++freespace_number;

        LOG_EXIT_INT(freespace_number);
        return freespace_number;
}


/*
 *  Called by insert_DiskSeg_Into_List() to get a name
 *  for the segment storage object being created.
 */
static int  get_name_for_disk_segment( DISKSEG *seg )
{
        int                 rc = EINVAL;
        LOGICALDISK        *ld  = get_logical_disk(seg);
        DISK_PRIVATE_DATA  *disk_pdata;
        SEG_PRIVATE_DATA   *pdata;

        LOG_ENTRY();

        if (ld) {

                disk_pdata = get_s390_disk_private_data(ld);
                pdata      = (SEG_PRIVATE_DATA *)seg->private_data;

                if (seg->data_type == DATA_TYPE) {

                        sprintf(seg->name, "%s%d", ld->name, pdata->minor );
                        rc = 0;

                }
                else if (seg->data_type == FREE_SPACE_TYPE) {

                        sprintf(seg->name, "%s_freespace%d", ld->name, get_next_avail_freespace_number(ld) );
                        rc = 0;

                }
                else if (seg->data_type == META_DATA_TYPE) {

                        if (seg->size == ld->size) {
                                sprintf(seg->name, "%s_unformatted_space", ld->name );
                        }
                        else {
                                sprintf(seg->name, "%s_metadata", ld->name );
                        }
                        rc = 0;

                }
                else {
                        LOG_ERROR("segment has unknown data type (type=%d)\n", seg->data_type );
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to allocate a segment storage object
 */
DISKSEG *  allocate_s390_disk_segment( LOGICALDISK *ld )
{
        int   rc;
        DISKSEG  *seg=NULL;
        list_element_t e;

        LOG_ENTRY();

        rc = EngFncs->allocate_segment( NULL, &seg );
        if (rc==0) {

                e=EngFncs->insert_thing( seg->child_objects,
                                         ld,
                                         INSERT_AFTER | EXCLUSIVE_INSERT,
                                         NULL );
                if (e!=NULL) {
                        rc = 0;

                        seg->plugin      = s390_plugin_rec;
                        seg->object_type = SEGMENT;
                        seg->data_type   = DATA_TYPE;

                        memcpy(&seg->geometry, &ld->geometry, sizeof(geometry_t));

                        seg->private_data = calloc(1, sizeof(SEG_PRIVATE_DATA));
                        if (seg->private_data) {

                                ((SEG_PRIVATE_DATA *)seg->private_data)->signature    = S390_SEG_MGR_PDATA_SIGNATURE;
                                ((SEG_PRIVATE_DATA *)seg->private_data)->logical_disk = ld;

                        }
                        else {
                                LOG_ERROR("call to malloc segment private storage area failed\n");
                                EngFncs->free_segment( seg );
                                seg = NULL;
                        }
                }
                else {
                        rc = EPERM;             
                }

        }
        else {
                LOG_ERROR("call to engine_allocate_segment failed, RC= %d\n", rc);
                seg = NULL;
        }

        LOG_EXIT_PTR(seg);

        return seg;
}


/*
 * Called to free a segment storage object
 */
void free_s390_disk_segment( DISKSEG *seg )
{
        SEG_PRIVATE_DATA  *pdata=NULL;

        LOG_ENTRY();

        if (seg) {

                LOG_DEBUG("segment name= %s\n", seg->name);

                pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                EngFncs->free_segment( seg );

                if (pdata) free(pdata);
        }

        LOG_EXIT_VOID();

}





/*
 *  Called to allocate a DISK_PRIVATE_DATA struct and link it into the
 *  list we maintain for disks we manage disk segments on.
 */
int create_s390_disk_private_data( LOGICALDISK *ld )
{
        int rc=ENOMEM;
        DISK_PRIVATE_DATA *disk_pdata;
        list_element_t e;

        LOG_ENTRY();

        // make sure list is created
        if (Disk_PrivateData_List == NULL) {

                Disk_PrivateData_List = EngFncs->allocate_list();

                if (Disk_PrivateData_List == NULL) {
                        LOG_EXIT_INT(rc);
                        return rc;
                }

        }

        if ( get_s390_disk_private_data(ld ) == NULL ) {

                disk_pdata = (DISK_PRIVATE_DATA *) calloc(1, sizeof(DISK_PRIVATE_DATA) );

                if ( disk_pdata ) {

                        disk_pdata->signature          = S390_SEG_MGR_PDATA_SIGNATURE;
                        disk_pdata->key                = ld;

                        disk_pdata->vsectors_per_block = ld->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;
                        disk_pdata->geometry.cylinders = ld->geometry.cylinders;
                        disk_pdata->geometry.heads     = ld->geometry.heads;
                        disk_pdata->geometry.sectors   = ld->geometry.sectors_per_track;
                        disk_pdata->geometry.start     = 0;

                        e=EngFncs->insert_thing( Disk_PrivateData_List,
                                                 disk_pdata,
                                                 INSERT_AFTER,
                                                 NULL );
                        if (e!=NULL) {
                                rc = 0;
                        }
                        else {
                                rc = EPERM;
                                free(disk_pdata);
                        }

                }
                else {
                        rc = ENOMEM;
                }
        }
        else {
                rc = 0;
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Called to delete the specified disk private data by pruning it from the double
 *  linked list.
 */
int delete_s390_disk_private_data( LOGICALDISK *ld )
{
        int                rc = EINVAL;
        DISK_PRIVATE_DATA *disk_pdata = get_s390_disk_private_data(ld);

        LOG_ENTRY();

        // make sure list exists
        if ( Disk_PrivateData_List != NULL ) {

                // delete object from list and free its memory
                if (disk_pdata) {
                        EngFncs->remove_thing( Disk_PrivateData_List, disk_pdata );
                        free(disk_pdata);
                        rc = 0;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Returns the private data area for the specified disk.
 */
DISK_PRIVATE_DATA * get_s390_disk_private_data( LOGICALDISK *ld )
{
        DISK_PRIVATE_DATA  *disk_pdata = NULL;
        list_element_t      iter;

        LOG_ENTRY();

        // make sure list is created
        if ( Disk_PrivateData_List ) {

                LIST_FOR_EACH( Disk_PrivateData_List, iter, disk_pdata ) {

                        // compare storage object ptr to key field in private data area
                        if ( disk_pdata->key == ld ) {
                                LOG_EXIT_PTR(disk_pdata);
                                return disk_pdata;
                        }

                }

        }

	LOG_EXIT_PTR(NULL);
        return NULL;
}



/*
 *   Returns the DISKSEG struct ptr to the freespace seg following the
 *   specified DISKSEG.  Returns NULL if freespace is not found.
 *
 *  Consider the following seglist ...
 *
 *    Seg_1     Seg2       Seg3       Seg4      Seg5
 *    MBR_SEG   DATA_SEG   Free_Seg   EBR_SEG   DATA_SEG
 *
 *   If we were asked for the freespace seg following Seg2 we would
 *   respond by returning Seg3.  The freespace seg must immediately
 *   follow the specified DISKSEG.
 *
 */
DISKSEG * get_freespace_following_s390_segment(DISKSEG *seg)
{
        DISKSEG     *prev = NULL;
        DISKSEG     *this = NULL;
        LOGICALDISK  *ld=NULL;
        list_element_t  iter;

        // get logical disk and the entire segment list for the disk
        ld = get_logical_disk( seg );   
        if ( ld ) {

                prev = EngFncs->first_thing( ld->parent_objects, &iter );
                if (prev==NULL) return NULL;

                do {
                        this = EngFncs->next_thing( &iter );
                        if (this != NULL) {

                                if (prev == seg) {

                                        if ( this->data_type==FREE_SPACE_TYPE ) {
                                                return this;
                                        }
                                        else {
                                                return NULL;
                                        }

                                }

                                prev=this;
                        }

                } while (this != NULL);

                return NULL;
        }
        else {
                return NULL;
        }

}



/*
 *   Called to try and find free space between disk data segments, returning a
 *   new free space DISKSEG struct if a gap between existing segments is found.
 */

DISKSEG * find_freespace_in_seglist( list_anchor_t seglist )
{
        DISKSEG            *prev = NULL;
        DISKSEG            *this = NULL;
        DISKSEG            *freeseg  = NULL;
        LOGICALDISK        *ld;
        int64_t             gap;
        list_element_t      iter;

        LOG_ENTRY();

        if (seglist) {

                prev = EngFncs->first_thing( seglist, &iter );
                if ( prev != NULL ) {

                        // need logical disk
                        ld = get_logical_disk( prev );
                        if (ld==NULL) return NULL;

                        do {
                                this = EngFncs->next_thing( &iter );
                                if (this != NULL) {


                                        gap = this->start - ( prev->start + prev->size );

                                        if ( gap > 0 ) {

                                                // allocate a segment storage object from the engine
                                                freeseg = allocate_s390_disk_segment(ld);

                                                if (freeseg ) {

                                                        freeseg->data_type    = FREE_SPACE_TYPE;
                                                        freeseg->size         = gap;
                                                        freeseg->start        = prev->start + prev->size;

                                                        freeseg->flags       &= ~SOFLAG_DIRTY;

                                                        LOG_EXIT_PTR(freeseg);
                                                        return freeseg;

                                                }
                                                else {
                                                        LOG_EXIT_PTR(NULL);
                                                        return NULL;
                                                }

                                        }
                                        else {
                                                prev = this;
                                        }
                                }

                        } while ( this != NULL );
                }
        }


        LOG_EXIT_PTR(NULL);
        return NULL;
}


/*
 *  Called to look for free space on the logical disk and to create
 *  segments which will expose the free space on the disk.
 */
int  find_freespace_on_s390_disk( LOGICALDISK *ld )
{
        DISKSEG            *freeseg              = NULL;
        DISKSEG            *seg                  = NULL;
        DISKSEG            *segaddr              = NULL;
        sector_count_t      sectors_left_on_disk = 0;
        int                 rc;
        lba_t               freespace_start_lba;
        DISK_PRIVATE_DATA  *disk_pdata;
        list_anchor_t             seglist = ld->parent_objects;



        LOG_ENTRY();

        // need disk private data area
        disk_pdata = get_s390_disk_private_data( ld );
        if (disk_pdata==NULL) return EINVAL;

        if ( EngFncs->list_count( seglist ) > 0 ) {                    /* IF ... we have disk segments */

                do {                                                   /* THEN ... look for gaps between them */
                        seg = find_freespace_in_seglist( seglist );

                        if (seg != NULL) {

                                segaddr = insert_s390_segment_into_list( seglist, seg);
                                if (segaddr==NULL) {
                                        free_s390_disk_segment( seg );
                                        rc = ENOMEM;
                                        LOG_EXIT_INT(rc);
                                        return rc;                                      
                                }

                        }

                } while (seg != NULL);

                seg = EngFncs->last_thing( seglist, NULL );

                sectors_left_on_disk = ld->size - ( seg->start + seg->size );

                if ( sectors_left_on_disk > 0 ) {
                        freespace_start_lba  =  seg->start + seg->size;
                }
                else {
                        sectors_left_on_disk = 0;
                        freespace_start_lba  = 0;
                }
        }
        else {  // there are no segments on the disk at all so just create a
                // single free space segment for the disk!
                freespace_start_lba  = 0;
                sectors_left_on_disk = ld->size;
        }

        if ( sectors_left_on_disk > 0 ) {

                freeseg = allocate_s390_disk_segment(ld);
                if (freeseg) {

                        freeseg->data_type  = FREE_SPACE_TYPE;
                        freeseg->size       = sectors_left_on_disk;
                        freeseg->start      = freespace_start_lba;

                        freeseg->flags     &= ~SOFLAG_DIRTY;

                        segaddr = insert_s390_segment_into_list( seglist, freeseg);
                        if (segaddr==NULL) {

                                int i;

                                for (i=1; i<10; i++) {

                                        segaddr = insert_s390_segment_into_list( seglist, freeseg);
                                        if (segaddr==NULL) {
                                                LOG_DEBUG("error, insert_DiskSeg_Into_List returned an error\n");
                                                free_s390_disk_segment(freeseg);
                                                LOG_EXIT_INT(EIO);
                                                return EIO;
                                        }
                                        else {
                                                break;
                                        }
                                }

                        }

                }
                else {
                        LOG_EXIT_INT(EIO);
                        return EIO;
                }
        }

        // lastly, merge adjacent free areas
        merge_adjacent_freedisksegs_in_list( ld->parent_objects );

        LOG_EXIT_INT(0);
        return 0;
}



/*
 *   Called to try and join neighboring free segments together.
 *
 *   NOTE:  Assumes that the DISKSEG structs are kept in an ordered list
 *
 */
static int  merge_freespace_segments( list_anchor_t seglist )
{
        DISKSEG            *prev;
        DISKSEG            *this;
        LOGICALDISK        *ld;
        list_element_t      iter;

        LOG_ENTRY();

        prev = EngFncs->first_thing( seglist, &iter );
        if ( prev != NULL ) {

                // need logical disk
                ld = get_logical_disk(prev);
                if (ld) {

                        do {
                                this = EngFncs->next_thing( &iter );
                                if (this != NULL) {

                                        if ( ( prev==NULL ) ||
                                             ( this->data_type != FREE_SPACE_TYPE )||
                                             ( prev->data_type != FREE_SPACE_TYPE )) {
                                                prev = this;
                                                continue;
                                        }

                                        if (get_freespace_number(prev) > get_freespace_number(this) ) {
                                                EngFncs->remove_thing( seglist, prev );                                         
                                                this->start -= prev->size;
                                                this->size  += prev->size;
                                                free_s390_disk_segment( prev );
                                                LOG_DEBUG("        kept seg: %s  start: %"PRIu64"  size: %"PRIu64"\n", this->name, this->start, this->size );
                                                return 0;
                                        }
                                        else {
                                                EngFncs->remove_thing( seglist, this );                                         
                                                prev->size  += this->size;
                                                free_s390_disk_segment( this );
                                                LOG_DEBUG("        kept seg: %s  start: %"PRIu64"  size: %"PRIu64"\n", prev->name, prev->start, prev->size );
                                                return 0;
                                        }

                                }

                        } while (this != NULL);  

                }
        }


        LOG_EXIT_INT(ENODATA);
        return ENODATA;
}





/*
 *   Called to try and join neighboring free segments together.
 *
 *   NOTE:  Assumes that the DISKSEG structs are kept in an ordered list
 *
 */
int  merge_adjacent_freedisksegs_in_list( list_anchor_t seglist )
{
        int  rc;

        LOG_ENTRY();

        do {
                rc = merge_freespace_segments(seglist);
        } while (rc==0);  /* continue while there are segs to merge */


        rc = 0;

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Wrapper routine that is called to remove a segment storage object from a list.
 */
int  remove_s390_segment_from_list( list_anchor_t seglist, storage_object_t *seg )
{
        LOG_ENTRY();
        LOG_DEBUG("segment name= %s\n",  seg->name );

        EngFncs->remove_thing( seglist, seg );

        EngFncs->unregister_name( seg->name );

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *   Called to name a new disk segment and insert it into the
 *   ordered segment list for the logical disk.
 */
void *  insert_s390_segment_into_list( list_anchor_t seglist, storage_object_t *seg)
{
        int    rc=EINVAL;
        void  *result=NULL;


        LOG_ENTRY();
        LOG_DEBUG("seg start= %"PRIu64"   size= %"PRIu64"\n", seg->start, seg->size);


        rc = get_name_for_disk_segment( seg );
        if (rc) {
                LOG_EXIT_PTR(NULL);
                return NULL;
        }

        rc = EngFncs->register_name( seg->name );
        if (rc == 0) {
                result = insert_s390_segment_into_ordered_list( seglist, seg );
        }
        else {
                LOG_ERROR("error, get_name_for_disk_segment failed, RC= %d\n", rc);
                LOG_EXIT_PTR(NULL);
                return NULL;
        }


        LOG_DEBUG("returning %p\n", result);
        LOG_EXIT_PTR(result);
        return result;
}


/*
 *   Called to insert a disk segment into a segment list that we
 *   are trying to keep ordered by the segments starting LBA
 *   number.
 */
void *  insert_s390_segment_into_ordered_list( list_anchor_t seglist, storage_object_t *seg )
{
        int                rc;
        storage_object_t  *seg2;
        lba_t              seg2_end_lba;
        boolean            overlapping=FALSE;
        list_element_t     iter,e;

        LOG_ENTRY();
        LOG_DEBUG("seg name= %s   seg start= %"PRIu64"  ends= %"PRIu64"  size= %"PRIu64"\n", seg->name, seg->start, seg->start+seg->size-1, seg->size );

        rc = -1;

        LIST_FOR_EACH( seglist, iter, seg2 ) {

                seg2_end_lba = seg2->start + seg2->size - 1;

                // test and set ... overlapping segments flag
                if ( (  seg->start >= seg2->start )&&
                     (  seg->start <= seg2_end_lba)) {
                        overlapping = TRUE;
                }
                else if ( ( seg->start  <  seg2->start ) &&
                          ( seg2->start <= (seg->start + seg->size - 1)) ) {
                        overlapping = TRUE;
                }
                else {
                        overlapping = FALSE;
                }

                if ( overlapping == TRUE ) {

                        LOG_DEBUG("Error ... Overlapping Segments ...\n");
                        LOG_DEBUG("seg2:   name: %s\n", seg2->name );
                        LOG_DEBUG("       start: %"PRIu64"\n", seg2->start );
                        LOG_DEBUG("        size: %"PRIu64"\n", seg2->size );
                        LOG_DEBUG("         end: %"PRIu64"\n", seg2_end_lba );
                        LOG_DEBUG(" overlap lba: %"PRIu64"\n", seg->start );

                        rc = EINVAL;    // must be genuine partition overlap
                        break;          // break out of loop ... looking for insertion pt
                }

                // test for ... valid insertion point
                if (seg2->start > seg->start ) {
                        rc=0;
                        break;
                }
        }


        switch (rc) {
        
        case 0:  /* Ok, found a segment we should insert in front of */
                e = EngFncs->insert_thing( seglist, 
                                           seg,
                                           INSERT_BEFORE|EXCLUSIVE_INSERT,
                                           EngFncs->find_in_list(seglist, seg2, NULL) );
                if (e != NULL) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }
                break;

        case -1: /* This new segment lays out on the disk at a higher */
                /* LBA than any other existing segment.  So, just    */
                /* insert at end of the segment list.                */
                e = EngFncs->insert_thing( seglist, 
                                           seg,
                                           INSERT_AFTER|EXCLUSIVE_INSERT,
                                           NULL );
                if (e != NULL) {
                        rc = 0;
                }
                else {
                        rc = EPERM;
                }
                break;

        default:     /* REAL ERROR ... return NULL ptr */
                LOG_ERROR("error, insertion failed ... RC= %d\n",  rc);
                break;
        }

        if (rc) {
                LOG_EXIT_PTR(NULL);
                return NULL;
        }
        else {
                LOG_EXIT_PTR(seg);
                return seg;
        }
}



/*
 *  Called to add a segment to the logical disk's segment list. First, convert the
 *  partition record to a DISKSEG struct.  Then, insert the DISKSEG struct into
 *  the list hanging off the logical disk.
 */
DISKSEG * create_s390_metadata_segment( LOGICALDISK  *ld,  lba_t start, sector_count_t size )
{
        DISKSEG *metadata=NULL;

        LOG_ENTRY();

        metadata = allocate_s390_disk_segment( ld );
        if (metadata) {

                metadata->size        = size;
                metadata->start       = start;
                metadata->data_type   = META_DATA_TYPE;

        }

        LOG_EXIT_PTR(metadata);
        return metadata;
}


/*
 *  Convert from ascii string to ebcdic string
 */
void  atoe( char *buffer, int length )
{
        int i;

        if ( buffer ) {

                for (i=0; i<length; i++) {

                        buffer[i] = (char) ASCtoEBC[ (int) buffer[i] ];

                }

        }

}


/*
 *  Convert from ebcdic string to ascii string
 */
void  etoa( char *buffer, int length )
{
        int i;

        if ( buffer ) {

                for (i=0; i<length; i++) {

                        buffer[i] = (char) EBCtoASC[ (int) buffer[i] ];

                }

        }

}


/*
 *  Called to build an ebcdic dataset name for the specified
 *  evms segment. The name will be stored in the format 1
 *  dscb that hangs off the segment private data area.
 */
static int  get_dataset_name_for_disk_segment( LOGICALDISK *ld, DISKSEG *seg )
{
        int                rc=EINVAL;
        SEG_PRIVATE_DATA  *pdata=NULL;
        char               dsname[MAX_DSNAME_LENGTH+1];
        char               volid[VOLSER_LENGTH+1];
        DISK_PRIVATE_DATA *disk_pdata=NULL;


        LOG_ENTRY();

        if (ld && seg) {

                disk_pdata = get_s390_disk_private_data( ld );
                pdata      = (SEG_PRIVATE_DATA *)seg->private_data;

                if (disk_pdata && pdata) {

                        // need volume id field from the dasd volume label
                        memcpy(volid, &disk_pdata->vlabel.volid, VOLSER_LENGTH);
                        etoa(volid,VOLSER_LENGTH);
                        volid[VOLSER_LENGTH]=0x00;

                        // use same format as fdasd
                        sprintf( dsname, "%s%s%s%04X%s","LINUX.V", volid, ".PART", pdata->minor, ".NEW" );
                        memset( &pdata->f1.DS1DSNAM, 0x20, MAX_DSNAME_LENGTH );
                        memcpy( &pdata->f1.DS1DSNAM, dsname, strlen(dsname) );
                        atoe( pdata->f1.DS1DSNAM, MAX_DSNAME_LENGTH );
                        pdata->f1.DS1DSNAM[MAX_DSNAME_LENGTH]=0x00;

                        rc = 0;

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}




/*
 *  Initialize a new format 1 dscb for callers data segment
 */
int init_format1_dscb( LOGICALDISK *ld, DISKSEG *seg )
{
        int                 rc=EINVAL;
        time_t              current_time = time(NULL);
        struct tm          *today=NULL;
        DISK_PRIVATE_DATA  *disk_pdata = NULL;
        SEG_PRIVATE_DATA   *pdata=NULL;


        LOG_ENTRY();

        if (ld && seg) {

                disk_pdata = get_s390_disk_private_data(ld);
                pdata      = (SEG_PRIVATE_DATA *) seg->private_data;

                if (disk_pdata && pdata) {

                        memset( &pdata->f1, 0, sizeof(format1_label_t) );

                        get_dataset_name_for_disk_segment( ld, seg );

                        memcpy( pdata->f1.DS1SYSCD, S390_SYSTEM_CODE, S390_SYSTEM_CODE_LENGTH );
                        atoe( pdata->f1.DS1SYSCD, S390_SYSTEM_CODE_LENGTH );

                        memcpy(&pdata->f1.DS1DSSN, &disk_pdata->vlabel.volid, S390_DATA_SET_SN_LENGTH);

                        today = gmtime(&current_time);                 // get time

                        pdata->f1.DS1CREDT.year  = today->tm_year;     // creation date
                        pdata->f1.DS1CREDT.day   = today->tm_yday;

                        pdata->f1.DS1EXPDT.year  = 99;                 // expiration date
                        pdata->f1.DS1EXPDT.day   = 365;

                        memcpy( &pdata->f1.DS1REFD, &pdata->f1.DS1CREDT, sizeof( labeldate_t) );  // last referenced date

                        pdata->f1.DS1FMTID = 0xf1;
                        pdata->f1.DS1VOLSQ = 1;
                        pdata->f1.DS1NOEPV = 1;
                        pdata->f1.DS1RECFM = 0x88;
                        pdata->f1.DS1BLKL  = disk_pdata->vsectors_per_block;
                        pdata->f1.DS1LRECL = disk_pdata->vsectors_per_block;
                        pdata->f1.DS1DSIND = 0x80;
                        pdata->f1.DS1SCAL1 = 0x80;

                        pdata->f1.DS1EXT1.typeind = 0x81;

                        rc = 0;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Initialize a format 5 dscb for callers logical disk
 */
int init_format5_dscb( LOGICALDISK *ld )
{
        int                 rc=EINVAL;
        DISK_PRIVATE_DATA  *disk_pdata = NULL;
        DISKSEG            *seg;
        format5_label_t    *f5;
        int                 extent_number = 0;
        u_int16_t           rta=0;
        u_int16_t           fc=0;
        u_int16_t           ft=0;
        list_element_t      iter;

        LOG_ENTRY();

        if (ld) {

                disk_pdata = get_s390_disk_private_data(ld);

                if (disk_pdata) {

                        f5 = &disk_pdata->f5;

                        memset(f5, 0, sizeof(format5_label_t));      // clear it
                        memset(&f5->DS5KEYID, 0x05, 4 );             // set key field
                        f5->DS5FMTID = 0xF5;                         // set format 5 identifier

                        LIST_FOR_EACH( ld->parent_objects, iter, seg ) {  // walk disk segment list finding all freespace segs
                                                                          // and add them as extents
                                if (seg->data_type == FREE_SPACE_TYPE) {

                                        LOG_DEBUG("adding segment %s to descriptor\n", seg->name);

                                        rta = (seg->start/disk_pdata->vsectors_per_block) / disk_pdata->geometry.sectors;
                                        fc  = (seg->size/disk_pdata->vsectors_per_block) / (disk_pdata->geometry.heads*disk_pdata->geometry.sectors);
                                        ft  = (seg->size/disk_pdata->vsectors_per_block) % (disk_pdata->geometry.heads*disk_pdata->geometry.sectors);
                                        ft  = ft / disk_pdata->geometry.sectors;

                                        if (extent_number == 0) {        // use 1st available extent
                                                f5->DS5AVEXT.t  = rta;
                                                f5->DS5AVEXT.fc = fc;
                                                f5->DS5AVEXT.ft = ft;
                                        }
                                        else if (extent_number < 8) {    // use next 7 extents
                                                f5->DS5EXTAV[extent_number-1].t  = rta;
                                                f5->DS5EXTAV[extent_number-1].fc = fc;
                                                f5->DS5EXTAV[extent_number-1].ft = ft;
                                        }
                                        else if (extent_number < 26) {   // use next 18 extents
                                                f5->DS5MAVET[extent_number-8].t  = rta;
                                                f5->DS5MAVET[extent_number-8].fc = fc;
                                                f5->DS5MAVET[extent_number-8].ft = ft;
                                        }
                                        else {
                                                LOG_ERROR("error, no more freespace extents available in F5 dscb\n");
                                        }
                                        ++extent_number;
                                }

                        }

                        rc = 0;
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to fixup names of logial and embedded partitions because their names will
 *  change if the EBR chain is modified by a delete or create.
 */
int   fixup_390_partition_names( LOGICALDISK *ld )
{
        DISKSEG           *seg=NULL;
        SEG_PRIVATE_DATA  *pdata=NULL;
        int                minor=1;
        char               old_name[EVMS_NAME_SIZE+1];
        list_element_t     iter;


        LOG_ENTRY();

        // unregister all partition names and build new names
        LIST_FOR_EACH( ld->parent_objects, iter, seg ) {

                pdata = (SEG_PRIVATE_DATA *) seg->private_data;

                if ( seg->data_type == DATA_TYPE ) {

                        // unregister segment name
                        if (strlen(seg->name)) EngFncs->unregister_name( seg->name );

                        // update minor info
                        pdata->minor = minor;
                        ++minor;

                        strncpy(old_name, seg->name, EVMS_NAME_SIZE);

                        // build new evms segment name
                        get_name_for_disk_segment( seg );

                        // test if name has changed and then mark the object so we
                        // can do a dm rename of the kernel object at commit time.
                        if (  (seg->flags & SOFLAG_ACTIVE) &&
                              strlen(old_name) > 0  &&
                              strncmp(seg->name, old_name, EVMS_NAME_SIZE)!=0) {

                                s390_schedule_dm_rename(old_name,seg); 

                        }

                        // build new dsname in format 1 dscb
                        get_dataset_name_for_disk_segment( ld, seg);

                }

        }

        // new register all new partition names
        LIST_FOR_EACH( ld->parent_objects, iter, seg ) {
                if ( seg->data_type == DATA_TYPE ) {
                        EngFncs->register_name( seg->name );
                }
        }

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Called by private function code to see if a format operation is
 *  allowed on the disk upon which the segment object resides.
 *
 *  Check each segment object to see if it is being consumed by
 *  another storage object.  Also, check to see if it is part of
 *  a volume.  Deny the format request till the user cleans up
 *  these situations.
 */
boolean i_can_format_disk( LOGICALDISK *ld )
{
        DISKSEG  *seg;
        uint      count;
        list_element_t iter;

        LOG_ENTRY();

        if (ld) {
                // test if any of the segs are being consumed
                LIST_FOR_EACH( ld->parent_objects, iter, seg)  {
                        count=EngFncs->list_count(seg->parent_objects);
                        if (count!=0 || seg->volume ) {
                                LOG_EXIT_BOOL(FALSE);
                                return FALSE;
                        }
                }
                LOG_EXIT_BOOL(FALSE);
                return TRUE;
        }

        LOG_EXIT_BOOL(FALSE);
        return FALSE;
}

/*
 *  Called when we are aborting discovery.  Called with a list
 *  of segment objects we have created, the purpose of this routine
 *  is simply to provide a convenient means of discarding the work
 *  done prior to aborting discovery.  By returning TRUE for any
 *  object we created, the object will be pruned from the list.
 */
void prune_s390_segments_from_list( list_anchor_t list )
{
        list_element_t iter,iter2;
        storage_object_t *seg;

        LIST_FOR_EACH_SAFE( list, iter, iter2, seg ) {
                if ( seg->plugin == s390_plugin_rec ) {
                        if ( seg->flags & SOFLAG_ACTIVE ) {
                                EngFncs->dm_deactivate(seg);
                        }
                        free_s390_disk_segment(seg);
                        EngFncs->delete_element(iter);
                }
        }
}


/*
 *  Called to get the segment storage object corresponding to the specified
 *  device minor number.  This information is kept in the private data area
 *  of gpt segment storage objects.
 *
 *  Returns: DISKSEG * if successful
 *           NULL ptr if minor is not found
 */
DISKSEG * get_s390_segment_from_minor( LOGICALDISK *ld, int minor )
{
        DISKSEG    *seg=NULL;
        list_element_t iter;

        LOG_ENTRY();

        LIST_FOR_EACH( ld->parent_objects, iter, seg ) {

                if (seg->data_type == DATA_TYPE) {
                        if ( ((SEG_PRIVATE_DATA *)seg->private_data)->minor == minor ) {
                                LOG_EXIT_PTR(seg);
                                return seg;
                        }
                }

        }

        LOG_EXIT_PTR(NULL);
        return NULL;
}


/*
 *  Called to obtain the next available device minor
 *  number for the specified drive. This is an UNUSED
 *  minor.
 *
 *  Note: this routine should not be called during discovery
 *        because it expects all segments to be created for
 *        the drive and not in the middle of adding existing
 *        drive partitions.
 *
 *  Returns: device minor number if successful
 *           -1 if not successful
 */
int get_next_s390_minor( LOGICALDISK *ld )
{
        int                 i;
        DISK_PRIVATE_DATA  *disk_pdata=NULL;


        LOG_ENTRY();

        disk_pdata = get_s390_disk_private_data(ld);
        if (disk_pdata) {

                for (i=1; i<=disk_pdata->max_partitions_allowed; i++) {

                        if ( get_s390_segment_from_minor( ld, i)==NULL ) {
                                LOG_EXIT_INT(i);
                                return i;
                        }

                }

        }


        LOG_EXIT_INT(-1);
        return -1;
}


