/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * plugins/xfs/fsimxfs.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/wait.h>
#include <plugin.h>
#include "fsimxfs.h"

extern char xfsutils_version[10];
extern u_int32_t xfsutils_support;

/**
 * rw_diskblocks
 *
 * @volume:
 * @fd:			File handle of an opened device to read/write
 * @disk_offset:	Byte offset from beginning of device for start of disk
 *			block read/write
 * @disk_count:		Number of bytes to read/write
 * @data_buffer:	On read this will be filled in with data read from
 *			disk; on write this contains data to be written
 * @mode:		GET (read) or PUT (write)
 *
 * Read or write specific number of bytes for an opened device.
 */
static int rw_diskblocks(logical_volume_t * volume,
			 int fd,
			 int64_t start,
			 int32_t count,
			 void * buffer,
			 int mode)
{
	size_t bytes = 0;
	int rc = 0;

	LOG_ENTRY();

	switch (mode) {
	case GET:
		bytes = EngFncs->read_volume(volume, fd, buffer, count, start);
		break;
	case PUT:
		bytes = EngFncs->write_volume(volume, fd, buffer, count, start);
		break;
	default:
		break;
	}

	if (bytes != count) {
		rc = EIO;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_remove
 *
 * Remove the XFS filesystem from this volume.
 **/
int xfs_remove(logical_volume_t * volume)
{
	xfs_volume_t * xfs_vol = volume->private_data;
	int fd, fd2, rc;

	LOG_ENTRY();

	fd = EngFncs->open_volume(volume, O_RDWR|O_EXCL, 0);
	if (fd < 0) {
		rc = -1;
		goto out;
	}

	if (xfs_vol->sb) {
		/* Zero primary superblock */
		memset(xfs_vol->sb, 0, SIZE_OF_SUPER);
		rw_diskblocks(volume, fd, XFS_SUPER1_OFF,
			      SIZE_OF_SUPER, xfs_vol->sb, PUT);

		/* If volume has external log, delete it as well */
		if (xfs_vol->log_vol) {
			fd2 = EngFncs->open_volume(xfs_vol->log_vol, O_RDWR|O_EXCL, 0);
			if (fd2 > 0) {
				rw_diskblocks(volume, fd2, XFS_SUPER1_OFF,
					      SIZE_OF_SUPER, xfs_vol->sb, PUT);
				EngFncs->close_volume(volume, fd2);
			}
		}
		EngFncs->engine_free(xfs_vol->sb);
		EngFncs->engine_free(volume->private_data);
		volume->private_data = NULL;
	} else if (xfs_vol->log_sb && !xfs_vol->fs_vol) {
		/* Orphaned log */
		memset(xfs_vol->log_sb, 0, sizeof(xlog_rec_header_t));
		fd2 = EngFncs->open_volume(volume, O_RDWR|O_EXCL, 0);
		if (fd2 < 0) {
			rw_diskblocks(volume, fd2, XFS_SUPER1_OFF,	
				      SIZE_OF_SUPER, xfs_vol->log_sb, PUT);
			EngFncs->close_volume(volume, fd2);
		}
		EngFncs->engine_free(xfs_vol->log_sb);
		EngFncs->engine_free(volume->private_data);
		volume->private_data = NULL;
	} else {
		rc = -EINVAL;
	}

	EngFncs->close_volume(volume, fd);
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

static inline int set_arg(char **argv, int index, char *str)
{
        argv[index] = NULL;
	SET_STRING(argv[index], str);
	return 0;
}

/**
 * set_mkfs_options
 *
 * @options:	Options array passed from EVMS engine
 * @argv:	mkfs options array
 * @volume:	Volume on which program will be executed
 *
 * Build options array (argv) for mkfs.xfs.
 * Returns the number of args used.
 **/
static int set_mkfs_options(option_array_t * options,
			    char ** argv,
			    logical_volume_t * volume)
{
	int rc, i = 0, opt_count = 0;
	char buffer[EVMS_VOLUME_NAME_SIZE + 10];

	LOG_ENTRY();

        rc = set_arg(argv, opt_count++ , "mkfs.xfs");

	if (rc == 0) {
		for (i = 0; i < options->count && rc == 0; i++) {
			if (!options->option[i].is_number_based) {
				if (!strcmp(options->option[i].name, "vollabel")) {
					options->option[i].number = MKFS_SETVOL_INDEX;
				} else if (!strcmp(options->option[i].name, "journalvol")) {
					options->option[i].number = MKFS_JOURNAL_VOL_INDEX;
				} else if (!strcmp(options->option[i].name, "logsize")) {
					options->option[i].number = MKFS_SETLOGSIZE_INDEX;
				} else if (!strcmp(options->option[i].name, "force")) {
					options->option[i].number = MKFS_FORCE_INDEX;
				} else {
					/* unknown. ignore. */
					continue;
				}
			}

			switch (options->option[i].number) {
			case MKFS_FORCE_INDEX:
				if (options->option[i].value.b) {
					strcpy(buffer, "-f");
					break;
				} else {
					continue;
				}
			case MKFS_SETVOL_INDEX:
				if (options->option[i].value.s) {
					sprintf(buffer, "-L%s",
						options->option[i].value.s);
				}
				break;
			case MKFS_JOURNAL_VOL_INDEX:
				if (options->option[i].value.s &&
				    strcmp(options->option[i].value.s, NO_SELECTION)) {
					sprintf(buffer, "-llogdev=%s",
						options->option[i].value.s);
				}
				break;
			case MKFS_SETLOGSIZE_INDEX:
				if (options->option[i].value.r32) {
					u_int32_t tmp = (u_int32_t)(options->option[i].value.r32*1024*1024);
					tmp = tmp & ~(4095); // 4k multiple
					sprintf(buffer, "-lsize=%d", tmp);
				}
				break;
			default:
				/* unknown. ignore. */
				continue;
				break;
			}

			rc = set_arg(argv, opt_count++, buffer);
		}

        	if (rc == 0) {
	        	rc = set_arg(argv, opt_count++, volume->dev_node);
                        if (rc == 0) {
                                /* NULL terminate the argument array */
                                argv[opt_count] = NULL;
                        }
	        }
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_create
 *
 * Create an XFS filesystem on this volume.
 **/
int xfs_create(logical_volume_t * volume, option_array_t * options)
{
	char * argv[MKFS_XFS_OPTIONS_COUNT_TOTAL];
	char * buffer;
	pid_t pidm;
	int fds2[2];
	int bytes_read = 0;
	int status, i, rc;

	LOG_ENTRY();

        argv[0] = NULL;

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (!buffer) {
		rc = ENOMEM;
		goto out;
	}

	rc = pipe(fds2);
	if (rc) {
		goto out;
	}

	rc = set_mkfs_options(options, argv, volume);
	if (rc) {
		goto out;
	}

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		waitpid(pidm, &status, 0);
		if (WIFEXITED(status)) {
			bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
			/* Get mkfs.xfs exit code */
			rc = WEXITSTATUS(status);
			if (rc == 0) {
				if (bytes_read > 0) {
					LOG_DETAILS("mkfs output: \n%s",buffer);
					memset(buffer, 0, bytes_read);
				}
				LOG_DETAILS("mkfs.xfs completed with exit code %d \n", status);
			} else {
				if (bytes_read > 0) {
					MESSAGE("mkfs output: \n%s", buffer);
					memset(buffer, 0, bytes_read);
				}
				LOG_ERROR("mkfs.xfs completed with exit code %d \n", status);
			}
		} else {
			rc = EINTR;
		}
	} else {
		rc = EIO;
	}

	close(fds2[0]);
	close(fds2[1]);

out:
	for (i = 0; argv[i] != NULL; i++) {
		EngFncs->engine_free(argv[i]);
	}
	EngFncs->engine_free(buffer);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * set_fsck_options
 *
 * @options:	Options array passed from EVMS engine.
 * @argv:	fsck options array.
 * @volume:	Volume on which program will be executed.
 *
 * Build options array (argv) for fsck.xfs
 **/
static int set_fsck_options(option_array_t * options,
			    char ** argv,
			    logical_volume_t * volume)
{
	int rc, i = 0, opt_count = 0;
        xfs_volume_t * xfs_vol = volume->private_data;

	LOG_ENTRY();

        rc = set_arg(argv, opt_count++, "xfs_repair");
	if (rc == 0) {
	        for (i = 0; i < options->count && rc == 0; i++) {
		        if (!options->option[i].is_number_based) {
			        /* 'check read only' option selected or mounted */
			        if ((!strcmp(options->option[i].name, "readonly") &&
        			     options->option[i].value.b == TRUE) ||
	        		    EngFncs->is_mounted(volume->dev_node, NULL)) {
                                        options->option[i].number = FSCK_READONLY_INDEX;

                                /* 'verbose output' option selected */
        			} else if (!strcmp(options->option[i].name, "verbose") &&
		        	    options->option[i].value.b == TRUE) {
        				options->option[i].number = FSCK_VERBOSE_INDEX;
	        		} else {
					/* unknown. ignore. */
					continue;
				}
                        }

			/* 'check read only' option or mounted */
			if ((options->option[i].number == FSCK_READONLY_INDEX &&
			     options->option[i].value.b == TRUE) ||
			    EngFncs->is_mounted(volume->dev_node, NULL)) {
				/* set fsck.xfs -n option */
				rc = set_arg(argv, opt_count++, "-n");

			/* 'verbose output' option */
			} else if (options->option[i].number == FSCK_VERBOSE_INDEX &&
			    options->option[i].value.b == TRUE) {
				/* set fsck.xfs -v option */
				rc = set_arg(argv, opt_count++, "-v");
			}
                }

		if (rc == 0) {
                        if (xfs_vol->log_vol) {
                                /* Have external log, need to add -l logdev parm */
                                rc = set_arg(argv, opt_count++, "-l");
                                if (rc == 0) {
                                        rc = set_arg(argv, opt_count++,
                                                     xfs_vol->log_vol->dev_node);
                                }
                        }

                        if (rc == 0) {
                                rc = set_arg(argv, opt_count++, volume->dev_node);
                                if (rc == 0) {
                                        /* NULL terminate the argument array */
                                        argv[opt_count] = NULL;
                                }
                        }
                }
        }

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_check
 *
 * Check the XFS filesystem on this volume.
 **/
int xfs_check(logical_volume_t * volume, option_array_t * options)
{
	char * argv[FSCK_XFS_OPTIONS_COUNT_TOTAL];
	char * buffer;
	pid_t pidf;
	int fds2[2];
	int status, bytes_read;
	int i, rc;

	LOG_ENTRY();

        argv[0] = NULL;

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (!buffer) {
		rc = ENOMEM;
		goto out;
	}

	rc = pipe(fds2);
	if (rc) {
		goto out;
	}

	rc = set_fsck_options(options, argv, volume);
        if (rc) {
                goto out;
        }

	pidf = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidf != -1) {
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (!(waitpid(pidf, &status, WNOHANG))) {
			bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				MESSAGE("check output: \n%s", buffer);
				memset(buffer, 0, bytes_read);
			}
			usleep(10000);
		}
		if (WIFEXITED(status) && (WEXITSTATUS(status) != ENOENT)) {
			do {
				bytes_read = read(fds2[0], buffer, MAX_USER_MESSAGE_LEN);
				if (bytes_read > 0) {
					MESSAGE("check output1: \n%s", buffer);
				}
			} while (bytes_read == MAX_USER_MESSAGE_LEN);
			rc = WEXITSTATUS(status);
			if (rc == 0) {
				LOG_DETAILS("check completed with rc = %d \n", status);
			} else {
				LOG_ERROR("check completed with rc = %d \n", status);
			}
		} else {
			rc = EINTR;
		}
	} else {
		rc = EIO;
	}

	close(fds2[0]);
	close(fds2[1]);
out:
	for (i = 0; argv[i] != NULL; i++) {
		EngFncs->engine_free(argv[i]);
	}
	EngFncs->engine_free(buffer);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_swap_superblock
 *
 * Endian swap a XFS superblock
 **/
static void xfs_swap_superblock(xfs_sb_t * sblk)
{
	LOG_ENTRY();
	sblk->sb_versionnum	= XDISK_TO_CPU16(sblk->sb_versionnum);
	sblk->sb_blocksize	= XDISK_TO_CPU32(sblk->sb_blocksize);
	sblk->sb_magicnum	= XDISK_TO_CPU32(sblk->sb_magicnum);
	sblk->sb_dblocks	= XDISK_TO_CPU64(sblk->sb_dblocks);
	sblk->sb_rblocks	= XDISK_TO_CPU64(sblk->sb_rblocks);
	sblk->sb_logblocks	= XDISK_TO_CPU32(sblk->sb_logblocks);
	sblk->sb_agcount	= XDISK_TO_CPU32(sblk->sb_agcount);
	sblk->sb_agblocks	= XDISK_TO_CPU32(sblk->sb_agblocks);
	sblk->sb_unit		= XDISK_TO_CPU32(sblk->sb_unit);
	sblk->sb_width		= XDISK_TO_CPU32(sblk->sb_width);
	LOG_EXIT_VOID();
}

/**
 * xfs_get_superblock
 *
 * @volume:	Pointer to volume from which to get the superblock.
 * @sb_ptr:	Pointer to superblock.
 *
 * Get and validate a XFS superblock
 **/
int xfs_get_superblock(logical_volume_t * volume, xfs_sb_t * sb_ptr)
{
	int fd, rc;

	LOG_ENTRY();

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) {
		rc = EIO;
		goto out1;
	}

	/* Get primary superblock */
	rc = rw_diskblocks(volume, fd, XFS_SUPER1_OFF,
			   SIZE_OF_SUPER, sb_ptr, GET);
	if (rc) {
		goto out2;
	}

	xfs_swap_superblock(sb_ptr);

	/* See if primary superblock is XFS */
	if ((sb_ptr->sb_versionnum & XFS_SB_VERSION_NUMBITS) < XFS_SB_VERSION_1 ||
	    sb_ptr->sb_magicnum != XFS_SB_MAGIC) {
		rc = -1;
	}

out2:
	EngFncs->close_volume(volume, fd);
out1:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_swap_log_superblock
 *
 * Endian swaps a XFS external log superblock
 **/
static void xfs_swap_log_superblock(xlog_rec_header_t * sblk)
{
	LOG_ENTRY();
	sblk->h_magicno = XDISK_TO_CPU32(sblk->h_magicno);
	sblk->h_version = XDISK_TO_CPU32(sblk->h_version);
	LOG_EXIT_VOID();
}

/**
 * xfs_get_log_superblock
 *
 * @dev_node:	Pointer to volume from which to get the superblock.
 * @log_sb_ptr:	Pointer to superblock.
 *
 * Get and validate an XFS external-log superblock
 **/
int xfs_get_log_superblock(logical_volume_t * volume,
			   xlog_rec_header_t * log_sb_ptr)
{
	int  fd, rc;

	LOG_ENTRY();

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) {
		rc = EIO;
		goto out;
	}

	rc = rw_diskblocks(volume, fd, 0,
			   sizeof(xlog_rec_header_t), log_sb_ptr, GET);
	if (rc) {
		goto out2;
	}

	xfs_swap_log_superblock(log_sb_ptr);

	/* See if superblock is XFS log superblock */
	if (log_sb_ptr->h_magicno != XLOG_HEADER_MAGIC_NUM) {
		rc = -1;
	}

out2:
	EngFncs->close_volume(volume, fd);
out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * xfs_test_version
 *
 * Test mkfs.xfs version.
 *
 * xfsutils must be at version 1.0.9 or greater to function properly
 * with this FSIM.  That also happens to be the first version in which
 * mkfs.xfs supported the -V (version) option.  So, run mkfs.xfs with
 *  the -V option,and if it succeeds, the installed version is OK.  If
 * it fails, pass back the exit code (should be EINVAL) of the failure.
 **/
int xfs_test_version(void)
{
	pid_t pidm;
	char * argv[3];
	char * buffer, * buf_ptr;
	int fds2[2];
	int status, bytes_read;
	int ver_string_size, rc;

	LOG_ENTRY();

	buffer = EngFncs->engine_alloc(4096);
	if (!buffer) {
		rc = ENOMEM;
		goto out;
	}

	rc = pipe(fds2);
	if (rc) {
		goto out;
	}

	argv[0] = "mkfs.xfs";
	argv[1] = "-V";
	argv[2] = NULL;

	pidm = EngFncs->fork_and_execvp(NULL, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		waitpid(pidm, &status, 0);
		if (WIFEXITED(status)) {
			bytes_read = read(fds2[0], buffer, 4096);
			if (bytes_read > 0) {
				/* Find start and end of version number.
				 * Store in xfsutils_version.
				 */
				buf_ptr = strstr(buffer, "version");
				if (buf_ptr) {
					buf_ptr = strstr(buf_ptr, " ") + 1;
					ver_string_size = strstr(buf_ptr, "\n") - buf_ptr;
					if (ver_string_size > sizeof(xfsutils_version)) {
						ver_string_size = sizeof(xfsutils_version);
					}
					xfsutils_support = XFS_VALID_UTILS;
					strncpy(xfsutils_version, buf_ptr, ver_string_size);
					if (strcmp(xfsutils_version, "2.1.0") >= 0) {
						xfsutils_support |= XFS_V2_LOG;
					}
				} else {
					memset(xfsutils_version, 0, 9);
				}
			}
			/* Get mkfs.xfs exit code */
			rc = WEXITSTATUS(status);
			if (rc == 1) {
				/* Old xfs exits with rc=1 on version check. */
				rc = 0;
			} else if (rc == ENOENT) {
				/* exec did not find mkfs.xfs so indicate we don't have support */
				xfsutils_support = 0;
				rc = 0;
			}
			LOG_DETAILS("mkfs.xfs test version completed with exit code %d \n", rc);
		}
	} else {
		rc = EIO;
	}

	close(fds2[0]);
	close(fds2[1]);
out:
	EngFncs->engine_free(buffer);
	LOG_EXIT_INT(rc);
	return rc;
}

