# (C) Copyright IBM Corp. 2004
#
# This program is free software;  you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY;  without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
# the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program;  if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Perl module with common subroutines for creating volumes.

package Evms::Volume;

use strict;
use warnings;

use Evms::Common;
use Evms::Log;

BEGIN {
	use Exporter();
	our (@ISA, @EXPORT);
        @ISA = qw(Exporter);
        @EXPORT = qw(&create_compatibility_volume
		     &create_evms_volume
		     &query_volume

		     &rename_volume

		     &convert_to_evms_volume
		     &convert_to_compatibility_volume

		     &add_feature_to_volume

		     &mkfs_volume
		     &fsck_volume
		     &unmkfs_volume

		     &get_volume_filesystem);
}

# create_compatibility_volume
# arguments:
#    object: Object to be made into a volume
sub create_compatibility_volume($)
{
        my $object = $_[0];
        my $command = "create:volume,$object,compatibility";
        my $rc;

        $rc = run_evms_command($command);

        return $rc;
}

# create_evms_volume
# arguments:
#    object: Object to be made into a volume
#    name:   Name for the EVMS volume
sub create_evms_volume($ $)
{
	my $object = $_[0];
        my $name = $_[1];
        my $command = "create:volume,$object,name=$name";
        my $rc;

        $rc = run_evms_command($command);

        return $rc;
}

# query_volume
#
# arguments:
#    volume: Name of volume to query
#    output: Optional reference of array to pass back query output.
sub query_volume($)
{
	my $volume = $_[0];
	my @output;
	my $command = "query:volumes,volume=$volume";
	my $rc;

	$rc = run_evms_command($command, \@output);
	if ($rc) {
		return;
	}

	return @output;
}

# rename_volume
#
# arguments:
#    volume: Name of EVMS volume to rename
#    name:   New name for the EVMS volume
sub rename_volume($ $)
{
	my $volume = $_[0];
	my $name = $_[1];
	my $command = "rename:$volume,name=$name";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# convert_to_evms_volume
#
# arguments:
#    volume: Name of compatibility volume to convert
#    name:   Name for the EVMS volume
sub convert_to_evms_volume($ $)
{
	my $volume = $_[0];
	my $name = $_[1];
	my $command = "convert:$volume,name=$name";
	my $rc;
	my @output;

	$rc = run_evms_command($command, \@output);

	return $rc;
}

# convert_to_compatibility_volume
#
# arguments:
#    volume: Name of EVMS volume to convert
sub convert_to_compatibility_volume($)
{
	my $volume = $_[0];
	my $command = "convert:$volume,compatibility";
	my $rc;

	$rc = run_evms_command($command);

	return $rc;
}

# add_feature_to_volume
#
# arguments:
#    volume:  Name of EVMS volume
#    feature: Name of feature
#    options: Optional reference to hash of options
sub add_feature_to_volume
{
	my $volume = $_[0];
	my $feature = $_[1];
	my $options_ref = $_[2];
	my $rc;
        my $command = "Add Feature:$feature={";

	if (ref($options_ref)) {
		my %options = %{$options_ref};
		if (%options) {
			my @names = keys(%options);
			my $i;
			my $name;

			for ($i = 0; $i < $#names; $i++) {
				$name = $names[$i];
				$command .= "$name=$options{$name},";
			}
			$name = $names[$#names];
			$command .= "$name=$options{$name}";
		}
	}

	$command .= "},$volume";

	log_debug("Command is: $command\n");
	$rc = run_evms_command($command);

	return $rc;
}

# mkfs_volume
#
# arguments:
#    volume:  Name of volume
#    fsim:    Name of FSIM
#    options: Optional reference to hash of options
sub mkfs_volume
{
	my $volume = $_[0];
	my $fsim = $_[1];
	my $options_ref = $_[2];
	my $rc;
        my $command = "mkfs:$fsim={";

	if (ref($options_ref)) {
		my %options = %{$options_ref};
		if (%options) {
			my @names = keys(%options);
			my $i;
			my $name;

			for ($i = 0; $i < $#names; $i++) {
				$name = $names[$i];
				$command .= "$name=$options{$name},";
			}
			$name = $names[$#names];
			$command .= "$name=$options{$name}";
		}
	}

	$command .= "},$volume";

	log_debug("Command is: $command\n");
	$rc = run_evms_command($command);

	return $rc;
}

# fsck_volume
#
# arguments:
#    volume:  Name of volume
#    options: Optional reference to hash of options
sub fsck_volume
{
	my $volume = $_[0];
	my $options_ref = $_[1];
	my $rc;
        my $command = "check:$volume";

	if (ref($options_ref)) {
		my %options = %{$options_ref};
		if (%options) {
			my @names = keys(%options);
			my $i;
			my $name;

			$command .= ",";

			for ($i = 0; $i < $#names; $i++) {
				$name = $names[$i];
				$command .= "$name=$options{$name},";
			}
			$name = $names[$#names];
			$command .= "$name=$options{$name}";
		}
	}

	log_debug("Command is: $command\n");
	$rc = run_evms_command($command);

	return $rc;
}

# unmkfs_volume
#
# arguments:
#    volume: Name of volume
sub unmkfs_volume($)
{
	my $volume = $_[0];
	my $rc;
        my $command = "unmkfs:$volume";

	$rc = run_evms_command($command);

	return $rc;
}

# get_volume_filesystem
#
# arguments:
#    volume: Name of volume
# returns
#    name of filesystem, "" if error or volume has no filesystem
sub get_volume_filesystem($)
{
	my $volume = $_[0];
	my $rc;
	my @query_output;
	my $line;
	my $filesystem = "";

	@query_output = query_volume($volume);
	unless (@query_output) {
		log_error("Error querying volume $volume.\n");
		return $filesystem;
	}

	($line) = grep(/Filesystem:/, @query_output);
	if ((defined($line)) && ($line =~ /Filesystem:\s+(\S+)/)) {
		$filesystem = $1;
	}

	return $filesystem;
}

1;

