/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdrivelink.so
 *
 *   File: dl_discovery.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "dl_common.h"


static list_anchor_t    dl_output_list;


/*
 *  Function: dl_read_metadata
 *
 *  Called to read the meta data for a feature into the specified
 *  buffer. If the feature header ptr is NULL it means we need to
 *  get the feature header for the storage object by reading it off
 *  the disk. We need the feature header because it tells us where
 *  the feature data is located in the storage object.
 */
static int dl_read_metadata( storage_object_t          * object,
                             drivelink_metadata_t      * md,
                             evms_feature_header_t     * fh )
{
        int rc1,rc2,rc=ENODATA;
        drivelink_metadata_t  md2;
        u_int32_t   crc;
        u_int32_t   calculated_crc;
        lsn_t       lsn1=0, lsn2=0;

        LOG_ENTRY();

        REQUIRE(object != NULL);
        REQUIRE(md != NULL);
        REQUIRE(fh != NULL);

        lsn1 = fh->feature_data1_start_lsn;
        lsn2 = fh->feature_data2_start_lsn;

        rc1 = READ( object, lsn1, DRIVELINK_METADATA_SECTOR_COUNT, md );

        if (rc1==0) {
                if ( DISK_TO_CPU32(md->signature) == EVMS_DRIVELINK_SIGNATURE ) {

                        crc = DISK_TO_CPU32(md->crc);

                        md->crc = 0;

                        calculated_crc = EngFncs->calculate_CRC( EVMS_INITIAL_CRC,
                                                                        (void *)md,
                                                                        sizeof(drivelink_metadata_t) );

                        md->crc = CPU_TO_DISK32(crc);

                        if ( crc == calculated_crc ) {
                                dl_disk_metadata_to_cpu( md );
                                rc1 = 0;
                        }
                        else {
                                rc1 = ENODATA;
                        }
                }
                else {
                        rc1 = ENODATA;
                }
        }

        // look for second copy of metadata if feature header says it exists ...
        if ( fh->feature_data2_size != 0 ) {

                rc2 = READ( object, lsn2, DRIVELINK_METADATA_SECTOR_COUNT, &md2 );

                if (rc2 ==0 ) {

                        if ( DISK_TO_CPU32(md2.signature) == EVMS_DRIVELINK_SIGNATURE) {

                                crc = DISK_TO_CPU32(md2.crc);

                                md2.crc = 0;

                                calculated_crc = EngFncs->calculate_CRC( EVMS_INITIAL_CRC,
                                                                                (void *)&md2,
                                                                                sizeof(drivelink_metadata_t) );

                                md2.crc = CPU_TO_DISK32(crc);

                                if ( crc == calculated_crc ) {
                                        dl_disk_metadata_to_cpu( &md2 );
                                        rc2 = 0;
                                }
                                else {
                                        rc2 = ENODATA;
                                }
                        }
                        else {
                                rc2 = ENODATA;
                        }

                }

        }
        else {
                rc2 = ENODATA;
        }

        if ( rc1==0 && rc2==0 ) {
                if ( md->sequence_number >= md2.sequence_number) {
                        LOG_DEBUG("\tusing 1st copy cuz seq numbers are same or 1st is > 2nd\n");
                }
                else {
                        LOG_DEBUG("\tusing 2nd copy of metadata cuz of seq numbers\n");
                        memcpy(md, &md2, sizeof(drivelink_metadata_t ));
                }
                rc = 0;
        }
        else if ( rc1 == 0 ) {
                LOG_DEBUG("2nd read failed or metadata is bad/missing\n");
                rc = 0;
        }
        else if ( rc2 == 0 ) {
                LOG_DEBUG("1st read failed or metadata is bad/missing\n");
                memcpy(md, &md2, sizeof(drivelink_metadata_t) );
                rc = 0;
        }
        else {
                rc = ENODATA;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: test_and_remove_missing_child
 *
 *  It is possible during discovery that we will find a missing child
 *  object already occupying the ordering table. When this happens, we
 *  simply need to remove the missing child and continue with discovery.
 *  This routine is called to test the ordering table for a missing
 *  child and to free it if one is found.
 */
static int  dl_test_and_remove_missing_child( storage_object_t *drivelink,
                                              u_int32_t         serial_number)
{
        int rc=EINVAL, index=-1;
        drivelink_private_data_t *pdata;


        LOG_ENTRY();

        pdata = (drivelink_private_data_t *) drivelink->private_data;

        index = dl_get_child_index_by_sn( drivelink, serial_number );

        if ( index == BAD_DOT_INDEX ) {
                LOG_ERROR("error, child serial number not found in link table\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        if ( pdata->drive_link[index].object != NULL &&
             dl_isa_missing_child(pdata->drive_link[index].object)==TRUE){
                dl_free_missing_child_object(pdata->drive_link[index].object);
                pdata->drive_link[index].object = NULL;
		pdata->drive_link[index].flags &= ~DL_FLAG_MISSING;
        }

        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_verify_metadata
 *
 *  Called to validate a child object's drivelink metadata before
 *  allowing it to be added to the drivelink.
 */
static int dl_verify_metadata( storage_object_t *drivelink,
                               storage_object_t *child,
                               drivelink_metadata_t *md )
{
        int rc=EINVAL, links=0, index=-1;
        drivelink_private_data_t *pdata;


        LOG_ENTRY();

        pdata = (drivelink_private_data_t *) drivelink->private_data;
        links = pdata->drive_link_count;

        // find the object in the ordering table
        index = dl_get_child_index_by_sn( drivelink, md->child_serial_number );

        /*
         *  TEST 1:  a childs serial number must actually occur in the ordering table
         */
        if ( index == BAD_DOT_INDEX ) {
                LOG_ERROR("error, child serial number not found in link table\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        /*
         * TEST 2: a child object may already have been found ... meaning there is an
         *         object with old metadata on it.  happens when old disk is added back
         *         to the system.
         */
        if ( pdata->drive_link[index].object != NULL ){
                LOG_ERROR("error, another child object is already present in the ordering table\n");
                if (child->feature_header->sequence_number > pdata->drive_link[index].object->feature_header->sequence_number) {
                        LOG_DEBUG("will replace child object found in ordering table\n");
                }
                else {
                        LOG_ERROR("keeping child object already in ordering table\n");
                        LOG_EXIT_INT(rc);
                        return rc;
                }
        }

        /*
         *  TEST 3: children must all report the same number of links in the drivelink
         */
        if ( md->child_count != links) {
                LOG_ERROR("error, drive link child reports diff number of links\n");
                rc = EINVAL;
        }

        /*
         *  TEST 4: all children must have matching ordering tables
         */
        else if ( memcmp( &pdata->ordering_table[0],
                          &md->ordering_table[0],
                          (sizeof(dot_entry_t)*links) ) !=0 ) {
                LOG_ERROR("error, drive link child has diff ordering table for drive link\n");
                rc = EINVAL;
        }

        /*
         *  TEST 5:  all children must agree on the parent drivelink object name
         */
        else if ( strncmp( pdata->parent_object_name,
                           child->feature_header->object_name,
                           strlen(drivelink->name) )!=0) {
                LOG_ERROR("error, child doesnt have same parent storage object name\n");
                rc = EINVAL;
        }

        /*
         *  TEST 6:  cant span cluster containers
         */
        else if ( drivelink->disk_group != child->disk_group) {
                LOG_ERROR("error, child doesnt belong to the same disk group as the drivelink.\n");
                rc = EINVAL;
        }

	/*
	 * TEST 7:  Must have the same metadata sequence number
	 */
	else if ( md->sequence_number != pdata->sequence_number) {
                LOG_ERROR("error, child metadata sequence number %"PRIu64" does not match the drivelink sequence number %"PRIu64".\n",
			  md->sequence_number, pdata->sequence_number);
                rc = EINVAL;
	}

        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  dl_add_child_object_to_drivelink
 *
 *  Called to insert a drive link child object into the parent drive
 *  link object's list ... and also into the in-memory linear map.
 *
 *  Information about each child object is placed in the private data
 *  area of the parent drive link object.  The child objects are links
 *  and the links are kept ordered by their link sequence number.  This
 *  number is found in the child object meta data area.
 */
int dl_add_child_object_to_drivelink( storage_object_t       * drivelink,
                                      storage_object_t       * child,
                                      u_int32_t                child_serial_number,
                                      evms_feature_header_t  * fh )
{
        int rc=0, index;
        u_int32_t padding=0;
        u_int64_t child_useable_size=0;
        dot_entry_t  *dot=NULL;
        drivelink_private_data_t *pdata = (drivelink_private_data_t *) drivelink->private_data;
        sector_count_t evms_sectors = (DRIVELINK_METADATA_SECTOR_COUNT*2) + (FEATURE_HEADER_SECTOR_COUNT*2);
        list_element_t iter1=NULL, iter2=NULL;

        LOG_ENTRY();

        LOG_DEBUG("child= %s  child sn= 0x%X  size= %"PRIu64"\n", child->name, child_serial_number, child->size);

        // find the object in the ordering table
        index = dl_get_child_index_by_sn( drivelink, child_serial_number );
        if (index == BAD_DOT_INDEX){
                LOG_ERROR("error, child not found in drivelink metadata\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        dot = &pdata->ordering_table[index];

        if (dot->child_vsize == 0) {

                // useable area = size - 2 feature header sectors - 2 copies of metadata
                child_useable_size = child->size - evms_sectors;

                // useable area must an 8k (16 512 byte sectors) multiple, remaining sectors will
                // be placed in the padding area.
                padding = child_useable_size % 16;

                child_useable_size -= padding;

                dot->child_vsize = child_useable_size;
        }
        else {
                child_useable_size = dot->child_vsize;

                if ( dl_isa_missing_child(child) == TRUE ) {
                        padding = 0;
                }
                else {
                        if (child->size >= (child_useable_size + evms_sectors)) {
                                padding = child->size - child_useable_size - evms_sectors;
                        }
                        else {
                                LOG_DEBUG("error, child object doesnt match metadata ... too small for ordering table info\n");
                                rc = EINVAL;
                        }
                }
        }

        if (rc==0) {
                iter1 = EngFncs->insert_thing(drivelink->child_objects, child,
                                              INSERT_AFTER|EXCLUSIVE_INSERT, NULL);
                if (!iter1) {
                        rc = ENOMEM;
                }
        }

        if (rc==0) {
                iter2 = EngFncs->insert_thing(child->parent_objects, drivelink,
                                              INSERT_AFTER|EXCLUSIVE_INSERT, NULL);
                if (!iter2) {
                        EngFncs->delete_element(iter1);
                        rc = ENOMEM;
                }
        }

        if (rc) {
                LOG_ERROR("error, drivelink list is unable to consume child object\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        // fill in the information for this link in the drivelink linear map
        pdata->drive_link[index].padding         = padding;
        pdata->drive_link[index].sector_count    = child_useable_size;
        pdata->drive_link[index].serial_number   = child_serial_number;
        pdata->drive_link[index].index           = index;
        pdata->drive_link[index].object          = child;

	if ( dl_isa_missing_child(child) == TRUE ) {
		pdata->drive_link[index].flags |= DL_FLAG_MISSING;
	}
	else {
		pdata->drive_link[index].flags &= ~DL_FLAG_MISSING;
	}

        LOG_DEBUG("Added child to:       drive link = %s\n", drivelink->name);
        LOG_DEBUG("                           index = %d\n", index);
        LOG_DEBUG("                        child sn = 0x%X\n", child_serial_number);
        LOG_DEBUG("                      child size = %"PRIu64"\n", child->size);
        LOG_DEBUG("                         padding = %d\n", padding);
        LOG_DEBUG("                    useable size = %"PRIu64"\n", child_useable_size );
        LOG_DEBUG("                    evms sectors = %"PRIu64"\n", evms_sectors);

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_find_drivelink_object
 *
 *  This is a helper routine that is used to locate and return
 *  a drivelink storage object.
 */
static int  dl_find_drivelink_object( char *name, storage_object_t **drivelink )
{
        int rc=ENODEV;
        storage_object_t *object;
        list_anchor_t object_list = NULL;
        list_element_t iter;
        drivelink_private_data_t *pdata;

        LOG_ENTRY();

        REQUIRE(name != NULL);
        REQUIRE(drivelink != NULL);

        *drivelink=NULL;

        rc = dl_get_drivelink_list(&object_list);
        if (rc==0) {
                LIST_FOR_EACH( object_list, iter, object ) {
                        pdata = (drivelink_private_data_t *) object->private_data;
                        if (strncmp(name,pdata->parent_object_name,EVMS_NAME_SIZE)==0){
                                *drivelink = object;
                                break;
                        }
                }
                EngFncs->destroy_list(object_list);
        }

        if (*drivelink == NULL) rc=ENODEV;

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_create_new_drivelink_object
 *
 *  Called to malloc a storage object and setup drivelink private
 *  data, registering names and serial numbers appropriate for a
 *  drivelink storage object.
 */
static int dl_create_new_drivelink_object(char                  *drivelink_name,
                                          drivelink_metadata_t  *md,
                                          storage_object_t      *child,
                                          storage_object_t     **drivelink )
{
        int rc=ENOMEM;
        storage_object_t *object;
        drivelink_private_data_t *pdata;

        LOG_ENTRY();

        object = dl_malloc_drivelink_object();
        if (object) {
                pdata = (drivelink_private_data_t *)object->private_data;

                object->disk_group = child->disk_group;

                // build cluster ready object name
                if (object->disk_group) {
                        strncpy(object->name, object->disk_group->name, EVMS_NAME_SIZE);
                        strncat(object->name, "/", EVMS_NAME_SIZE-strlen(object->name));
                }
                strncat(object->name, drivelink_name, EVMS_NAME_SIZE-strlen(object->name));

                // save drivelink name in pdata
                strncpy(pdata->parent_object_name, drivelink_name, EVMS_NAME_SIZE);

                pdata->parent_serial_number = md->parent_serial_number;
		pdata->sequence_number      = md->sequence_number;
                pdata->drive_link_count     = md->child_count;

                memcpy( &pdata->ordering_table[0],
                        &md->ordering_table[0],
                        sizeof(dot_entry_t)*md->child_count );

                rc = EngFncs->register_name( object->name );
                if (rc) {
                        LOG_ERROR("failed to register new parent drivelink storage object name\n");
                }

                // register drive link parent serial number
                if (rc==0) {
                        rc = dl_register_serial_number( md->parent_serial_number);
                        if (rc) {
                                LOG_ERROR("failed to register new drivelink parent serial number\n");
                        }
                }

                if (rc==0) {
                        *drivelink = object;
                }
                else {
                        dl_free_drivelink_object(object);
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  dl_discover_child_objects
 *
 *  Called by the engine to discover drive link child objects. The objects
 *  in the list will be validated as drivelink objects and added to
 *  drivelink storage objects.  If any object is not part of a drivelink it
 *  will simply be moved to the output list.
 */
static int dl_discover_child_objects( storage_object_t * obj )
{
        int                         rc=EINVAL;
        storage_object_t           *drivelink=NULL;
        storage_object_t           *tmp_obj;
        boolean                     drivelink_object_created=FALSE;
        drivelink_metadata_t        md;
        drivelink_private_data_t   *pdata;
        list_element_t              iter1, iter2;


        LOG_ENTRY();

        LOG_DEBUG("examining object %s\n", obj->name);

        rc = dl_read_metadata( obj, &md, obj->feature_header);

        if (rc==0) {

                dl_display_feature_header(obj->feature_header);

                // find the drivelink object this child object belongs to ...
                rc = dl_find_drivelink_object(obj->feature_header->object_name, &drivelink);

                if (rc) {  // drivelink storage object doesnt exist yet so we need to create it
                        rc = dl_create_new_drivelink_object( obj->feature_header->object_name,
                                                             &md,
                                                             obj,
                                                             &drivelink );

                        if (rc==0) {
                                drivelink_object_created=TRUE;
                        }
                }

                if (rc==0) { // validate the childs metadata by comparing it against the
                             // drivelink private data.
                        pdata = (drivelink_private_data_t *)drivelink->private_data;

                        rc = dl_verify_metadata(drivelink, obj, &md);

                        if (rc==0) { // if we suddenly found a child object for which we had a missing
                                     // child .... it means that somehow the engine didnt give us all
                                     // the child objects with which to construct the drivelink ... depth
                                     // discovery problem in the engine ... no problem ... just replace
                                     // the missing child object with the link object we just got.
                                rc = dl_test_and_remove_missing_child(drivelink, md.child_serial_number);
                        }

                        if (rc==0) {  // metadata is confirmed so add the child object to
                                      // the drivelink
                                rc = dl_add_child_object_to_drivelink( drivelink,
                                                                       obj,
                                                                       md.child_serial_number,
                                                                       obj->feature_header );
                                if (rc==0) { // always clear the DISCOVERY COMPLETE flag when we
                                             // succeed in adding a child object ... to force the
                                             // dl_discover() to rerun the mapping table work AND
                                             // refresh the device mapper kernel mapping. This prevents
                                             // completing the DISCOVERY work too soon ... if the engine
                                             // didn't give us all the child objects the first time :)
                                        pdata->flags &= ~DL_DISCOVERY_COMPLETE;
                                }
                        }
                }

                if (rc) {
                        if (drivelink!=NULL) EngFncs->remove_thing(drivelink->child_objects, obj);
                        LIST_FOR_EACH_SAFE(obj->parent_objects, iter1, iter2, tmp_obj) {
                                if (dl_isa_drivelink(tmp_obj)) {
                                        EngFncs->delete_element(iter1);
                                }
                        }
                        EngFncs->insert_thing(dl_output_list, obj,
                                              INSERT_AFTER, NULL);
                }
        }

        if (drivelink_object_created) {
                if (rc==0) {
                        EngFncs->insert_thing(dl_output_list, drivelink,
                                              INSERT_AFTER, NULL);
                }
                else {
                        dl_free_drivelink_object(drivelink);
                }
        }

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function:  dl_discover
 *
 *  Called to run discovery on a list of evms objects that
 *  the engine has found thus far.  We walk the list of objects,
 *  probing each object to see if it has drivelink metadata.  If so,
 *  we consume the object by removing it from the list, and place
 *  all new storage objects on the output object list.
 *  Any object we dont like in the input_object list must be
 *  copied to the output_object list.
 *
 */
int dl_discover( list_anchor_t input_list, list_anchor_t output_list, boolean final_call)
{
        int  rc,i,count;
        storage_object_t *drivelink, *object;
        list_anchor_t object_list;
        list_element_t iter;
        drivelink_private_data_t *pdata;

        LOG_ENTRY();

        REQUIRE(input_list != NULL && output_list != NULL);

        dl_output_list = output_list;

        { // debug section
                i=0;
                count = EngFncs->list_count(input_list);
                LOG_DEBUG("input object count= %d  final_call= %d\n", count, final_call);
                LIST_FOR_EACH(input_list, iter, object){
                        LOG_DEBUG("object[%d]= %s\n", i, object->name);
                        i++;
                }
                rc=0;
        }

        // check each item in the input list to see if it may be a drivelink child object
        LIST_FOR_EACH(input_list, iter, object) {
                dl_discover_child_objects(object);
        }

        // if this is the last time we will be called for discovery at this depth then ...
        if (final_call == TRUE) {
                rc = dl_get_drivelink_list(&object_list);
                if (rc==0) {
                        LIST_FOR_EACH(object_list, iter, drivelink){
                                LOG_DEBUG("drivelink: %s\n",drivelink->name);
                                pdata = (drivelink_private_data_t *)drivelink->private_data;
                                if ( !(pdata->flags & DL_DISCOVERY_COMPLETE) ) {
                                        LOG_DEBUG("...into final call processing for this drivelink.\n");
                                        dl_table_fixup(drivelink);      // looks at the object link table and builds a
                                                                        // missing child object for any NULL spot in
                                                                        // the ordered link table.

                                        dl_build_linear_mapping(drivelink); // creates a linear mapping of the child storage
                                                                        // objects, i.e. a relationship between drivelink
                                                                        // logical sectors and child object logical sectors.

                                        dl_setup_geometry(drivelink);   // creates a best-fit geometry for the drivelink
                                                                        // that is appropriate for all the child objects.

                                        dl_get_devmap_info(drivelink);  // gets Device Mapper Info from kernel

                                        pdata->flags |= DL_DISCOVERY_COMPLETE;
                                }
                                else {
                                        LOG_DEBUG("...final call was already processed for this drivelink.\n");
                                }
                        }
                        EngFncs->destroy_list(object_list);
                }
        }

        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}
