/*
 *   (C) Copyright IBM Corp. 2001, 2005
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid1_mgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the Raid1 MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <sys/ioctl.h>

#include "md.h"
#include "raid1_mgr.h"

#define my_plugin_record raid1_plugin

static boolean raid1_can_change_region_configuration( storage_object_t *region )
{
	logical_volume_t *evms_volume;
	boolean rc = TRUE;
	md_volume_t * volume = (md_volume_t *)region->private_data;

	LOG_ENTRY();

	if (!volume)
		rc = FALSE;
	else if (volume->flags & MD_NEW_REGION)
		rc = FALSE;
	else if (raid1_region_config_change_pending(volume))
		rc = FALSE;
	else if ((volume->flags & (MD_CORRUPT | MD_DEGRADED)))
		rc = FALSE;
	else if (EngFncs->is_offline(region, &evms_volume) == FALSE)
		rc = FALSE;
	else if (md_is_recovery_running(region) == TRUE)
		rc = FALSE;

	LOG_EXIT_INT(rc);
	return rc;
}


static int raid1_can_children_shrink(
	storage_object_t *region,
	u_int64_t shrink_limit,
	u_int64_t * max_size)
{
	int rc = 0;
	list_anchor_t temp_list = NULL;
	list_element_t li;
	shrink_object_info_t * shrink_object;
	md_volume_t * volume = (md_volume_t *)region->private_data;
	int count = 0;
	u_int64_t shrink_size;
	struct plugin_functions_s * fncs;
	md_member_t *member;
	
	LOG_ENTRY();
	
	temp_list  = EngFncs->allocate_list();
	if (temp_list == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	LIST_FOR_EACH(volume->members, li, member) {
		fncs = (struct plugin_functions_s *)member->obj->plugin->functions.plugin;
		fncs->can_shrink(member->obj, shrink_limit, temp_list);
		count++;
	}

	if (EngFncs->list_count(temp_list) != count) {
		/* Return error to indicate that not all child objects can shrink */
		rc = EINVAL;
		goto out;
	}
	
	shrink_size = -1;
	LIST_FOR_EACH(temp_list, li, shrink_object) {
		LOG_DEBUG(" object %s said its max shrink size is %"PRIu64"\n",
			  shrink_object->object->name, shrink_object->max_shrink_size);
		shrink_size = min(shrink_size, shrink_object->max_shrink_size);
	}
	
	if (shrink_size > shrink_limit) {
		LOG_WARNING(" Can't shrink more than the limit %"PRIu64"\n", shrink_limit);
		rc = EINVAL;
		goto out;
	}

	if (shrink_size < RAID1_MINIMUM_SHRINK_SIZE) {
		LOG_WARNING(" shrinkable size is too small (%"PRIu64" sectors)\n",
			    shrink_size);
		rc = EINVAL;
		goto out;
	} else {
		*max_size = shrink_size;
		if (*max_size > (region->size * RAID1_PERCENT_SHRINK_THRESHOLD) / 100) {
			*max_size = (region->size * RAID1_PERCENT_SHRINK_THRESHOLD) / 100;
			LOG_WARNING("Adjust max shrink size down to %"PRIu64" (%d%% threshold)\n",
						*max_size, RAID1_PERCENT_SHRINK_THRESHOLD);
		}
	}

out:
	if (temp_list) {
		EngFncs->destroy_list(temp_list);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_can_children_expand
 *
 * Call can_expand() on each child to find out if they can expand.
 * If all child objects can expand, the region can be expanded.
 */
static int raid1_can_children_expand (
	storage_object_t *region,
	u_int64_t expand_limit,
	u_int64_t * max_size)
{
	int rc = 0;
	list_anchor_t temp_list = NULL;
	md_volume_t * volume = (md_volume_t *)region->private_data;
	expand_object_info_t * expand_object;
	u_int64_t expand_size;
	list_element_t li;
	int count = 0;
	struct plugin_functions_s * fncs;
	md_member_t *member;

	LOG_ENTRY();

	temp_list  = EngFncs->allocate_list();
	if (temp_list == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	LIST_FOR_EACH(volume->members, li, member) {
		fncs = (struct plugin_functions_s *)member->obj->plugin->functions.plugin;
		fncs->can_expand(member->obj, expand_limit, temp_list);
		count++;
	}

	if (EngFncs->list_count(temp_list) != count) {
		/* Return error to indicate that not all child objects can shrink */
		rc = EINVAL;
		goto out;
	}

	expand_size = -1;
	LIST_FOR_EACH(temp_list, li, expand_object) {
		LOG_DEBUG(" object %s said its max expand size is %"PRIu64"\n",
			  expand_object->object->name, expand_object->max_expand_size);
		expand_size = min(expand_size, expand_object->max_expand_size);
	}

	if (expand_size > expand_limit) {
		LOG_WARNING(" Can't expand more than the limit %"PRIu64"\n", expand_limit);
		rc = EINVAL;
		goto out;
	}

	if (expand_size < RAID1_MINIMUM_EXPAND_SIZE) {
		LOG_WARNING(" Expandable size is too small (%"PRIu64" sectors)\n",
					expand_size);
		rc = EINVAL;
		goto out;
	}
	
	*max_size = expand_size;
	//*max_size = MD_NEW_SIZE_SECTORS(*max_size);

out:
	if (temp_list) {
		EngFncs->destroy_list(temp_list);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_setup_evms_plugin
 *
 *  This function gets called shortly after the plugin is loaded by the
 *  Engine. It performs all tasks that are necessary before the initial
 *  discovery pass.
 */
static int raid1_setup_evms_plugin(engine_functions_t * functions)
{
	int rc = 0;

	// Parameter check
	if (!functions) {
		return EINVAL;
	}

	EngFncs = functions;

	my_plugin = raid1_plugin;
	LOG_ENTRY();
	rc = md_register_name_space();

	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Region Checking Functions ******/


/* All of the following md_can_ functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: raid1_can_delete
 *
 *  Can we remove the specified MD logical volume
 */
static int raid1_can_delete( storage_object_t * region )
{
	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check
	if (! region) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	// The region must be owned by MD,
	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s is not owned by MD.\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	LOG_EXIT_INT(0);
	return 0;
}

/*
 *  Called by raid1_init_task() for Task_Expand to allocate and initialize
 *  the expand option descriptor array.
 */
static int raid1_init_expand_option_descriptors( task_context_t * context )
{
	int rc = EINVAL;
	storage_object_t *region;

	LOG_ENTRY();

	context->option_descriptors->count = 0;

	region = context->object;
	if ( region ) {

		if (region->object_type == REGION) {

			if (region->data_type == DATA_TYPE) {

				u_int64_t max_expand_size = 0;

				rc = raid1_can_children_expand(region, -1, &max_expand_size);
				if (!rc) {

					context->option_descriptors->count = RAID1_EXPAND_OPTION_COUNT;

					// Expanded RAID1 region size delta
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.list = NULL;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
					if (context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range==NULL) {
						LOG_EXIT_INT(ENOMEM);
						return ENOMEM;
					}
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].flags = 0;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].help = NULL;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].name = EngFncs->engine_strdup( RAID1_EXPAND_OPTION_SIZE_NAME );
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].tip =
						EngFncs->engine_strdup( _("Use this option to specify how much space to add to the region.") );
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].title = EngFncs->engine_strdup( _("Additional Size") );
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range->min.ui64 = RAID1_MINIMUM_EXPAND_SIZE;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range->max.ui64 = max_expand_size;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = 1;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].value.ui64 = max_expand_size;

				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Called by raid1_init_task() for Task_Expand to allocate and initialize
 *  the shrink option descriptor array.
 */
static int raid1_init_shrink_option_descriptors( task_context_t * context )
{
	int rc = EINVAL;
	storage_object_t *region;

	LOG_ENTRY();

	context->option_descriptors->count = 0;

	region = context->object;
	if ( region ) {

		if (region->object_type == REGION) {

			if (region->data_type == DATA_TYPE) {

				u_int64_t max_shrink_size = 0;

				rc = raid1_can_children_shrink(region, -1, &max_shrink_size);
				if (!rc) {

					context->option_descriptors->count = RAID1_SHRINK_OPTION_COUNT;

					// Expanded RAID1 region size delta
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.list = NULL;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
					if (context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range==NULL) {
						LOG_EXIT_INT(ENOMEM);
						return ENOMEM;
					}
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].flags = 0;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].help = NULL;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].name = EngFncs->engine_strdup( RAID1_SHRINK_OPTION_SIZE_NAME );
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].tip =
						EngFncs->engine_strdup( _("Use this option to specify how much space to reduce from the region.") );
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].title = EngFncs->engine_strdup( _("Shrink by Size") );
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range->min.ui64 = RAID1_MINIMUM_SHRINK_SIZE;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range->max.ui64 = max_shrink_size;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = 1;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].value.ui64 = max_shrink_size;

				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: set_expand_option_descriptors
 *
 * Called by raid1_set_option() for Task_Expand
 *
 */
static int raid1_set_expand_option( task_context_t * context,
				    u_int32_t        index,
				    value_t        * value,
				    task_effect_t  * effect )

{
	int rc = EINVAL;
	storage_object_t *region;
	u_int64_t expand_sectors = value->ui64;

	LOG_ENTRY();

	region = context->object;
	if ( region ) {

		if (region->object_type == REGION) {

			if (region->data_type == DATA_TYPE) {

				u_int64_t max_expand_size = 0;

				rc = raid1_can_children_expand(region, -1, &max_expand_size);
				if (!rc) {
					if (expand_sectors > max_expand_size) {
						expand_sectors = max_expand_size;
						*effect |= EVMS_Effect_Inexact;
					} else if (expand_sectors < RAID1_MINIMUM_EXPAND_SIZE) {
						expand_sectors = RAID1_MINIMUM_EXPAND_SIZE;
						*effect |= EVMS_Effect_Inexact;
					}
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range->min.ui64 = RAID1_MINIMUM_EXPAND_SIZE;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].constraint.range->max.ui64 = max_expand_size;
					context->option_descriptors->option[RAID1_EXPAND_OPTION_SIZE_INDEX].value.ui64 = expand_sectors;
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: set_shrink_option_descriptors
 *
 * Called by raid1_set_option() for Task_Expand
 *
 */
static int raid1_set_shrink_option( task_context_t * context,
				    u_int32_t        index,
				    value_t        * value,
				    task_effect_t  * effect )

{
	int rc = EINVAL;
	storage_object_t *region;
	u_int64_t shrink_sectors = value->ui64;

	LOG_ENTRY();

	region = context->object;
	if ( region ) {

		if (region->object_type == REGION) {

			if (region->data_type == DATA_TYPE) {

				u_int64_t max_shrink_size = 0;

				rc = raid1_can_children_shrink(region, -1, &max_shrink_size);
				if (!rc) {
					if (shrink_sectors > max_shrink_size) {
						shrink_sectors = max_shrink_size;
						*effect |= EVMS_Effect_Inexact;
					} else if (shrink_sectors < RAID1_MINIMUM_SHRINK_SIZE) {
						shrink_sectors = RAID1_MINIMUM_SHRINK_SIZE;
						*effect |= EVMS_Effect_Inexact;
					}
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range->min.ui64 = RAID1_MINIMUM_SHRINK_SIZE;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].constraint.range->max.ui64 = max_shrink_size;
					context->option_descriptors->option[RAID1_SHRINK_OPTION_SIZE_INDEX].value.ui64 = shrink_sectors;
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static void raid1_get_expand_options( option_array_t * options, sector_count_t  *size)
{
    int i;

    LOG_ENTRY();

    for (i = 0; i < options->count; i++) {

        if (options->option[i].is_number_based) {

            if (options->option[i].number == RAID1_EXPAND_OPTION_SIZE_INDEX) {
                *size = options->option[i].value.ui64;
            }

        }
        else {

            if (strcmp(options->option[i].name, RAID1_EXPAND_OPTION_SIZE_NAME) == 0) {
                *size = options->option[i].value.ui64;
            }

        }
    }
	LOG_DEBUG(" expand size %"PRIu64" sectors\n", *size);

    LOG_EXIT_VOID();
}

static void raid1_get_shrink_options( option_array_t * options, sector_count_t  *size)
{
    int i;

    LOG_ENTRY();

    for (i = 0; i < options->count; i++) {

        if (options->option[i].is_number_based) {

            if (options->option[i].number == RAID1_SHRINK_OPTION_SIZE_INDEX) {
                *size = options->option[i].value.ui64;
            }

        }
        else {

            if (strcmp(options->option[i].name, RAID1_SHRINK_OPTION_SIZE_NAME) == 0) {
                *size = options->option[i].value.ui64;
            }

        }
    }
	LOG_DEBUG(" shrink size %"PRIu64" sectors\n", *size);

    LOG_EXIT_VOID();
}

/* Function: raid1_can_expand
 *
 *	Can this region be expanded? If so,
 *	add the region to the expansion_points output list. else, just
 *      return 0 to allow those above us to do as they please
 */
static int raid1_can_expand(
	storage_object_t * region,
	u_int64_t expand_limit,
	list_anchor_t expansion_points )
{
	int rc = 0;
	expand_object_info_t * expand_object;
	u_int64_t max_expand_size = 0;
	list_element_t li=NULL;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (!raid1_can_change_region_configuration(region)) {
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	/* This region can expand if all children can, else return 0 */
	rc = raid1_can_children_expand(region, expand_limit, &max_expand_size);
	if (rc) {
		LOG_EXIT_INT(0);
		return 0;
	}

	expand_object = (expand_object_info_t *) EngFncs->engine_alloc( sizeof(expand_object_info_t) );
	if (expand_object) {
		expand_object->object          = region;
		expand_object->max_expand_size = max_expand_size;

		li = EngFncs->insert_thing(expansion_points,
					   expand_object,
					   INSERT_AFTER,
					   NULL);

		if (!li) {
			EngFncs->engine_free( expand_object );
			rc = ENOMEM;
		}
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid1_can_expand_by( storage_object_t * child_object,
				u_int64_t        * size )
{
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: raid1_can_shrink
 *
 *	Just like can_expand, but in the opposite direction.
 */
static int raid1_can_shrink(
	storage_object_t * region,
	u_int64_t shrink_limit,
	list_anchor_t shrink_points )
{
	int rc = 0;
	shrink_object_info_t * shrink_object;
	u_int64_t max_shrink_size = 0;
	list_element_t li = NULL;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (!raid1_can_change_region_configuration(region)) {
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	/* This region can be shrunk if all children can, else return 0 */
	rc = raid1_can_children_shrink(region, shrink_limit, &max_shrink_size);
	if (rc) {
		LOG_EXIT_INT(0);
		return 0;
	}

	shrink_object = (shrink_object_info_t *) EngFncs->engine_alloc( sizeof(shrink_object_info_t) );
	if (shrink_object) {
		shrink_object->object          = region;
		shrink_object->max_shrink_size = max_shrink_size;

		li = EngFncs->insert_thing(shrink_points, shrink_object, INSERT_AFTER, NULL);

		if (!li) {
			EngFncs->engine_free( shrink_object );
			rc = ENOMEM;
		}
	} else {
		rc = ENOMEM;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_can_shrink_by
 *
 *  Can we shrink the specified region? If it can be shrunk, but not to
 *  exactly new_size, reset new_size to the next lower possible size.
 */
static int raid1_can_shrink_by( storage_object_t * region,
				u_int64_t        * size )
{
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


static int raid1_can_replace_child(storage_object_t *region,
				   storage_object_t *child,
				   storage_object_t *new_child)
{
	int rc;
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	rc = md_can_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}

static void raid1_show_degraded(md_volume_t *vol)
{
	if (vol->flags & MD_DEGRADED) {
		int missing = vol->raid_disks - vol->active_disks;
		char * devices = "devices";
		char * device = "device";
		MESSAGE(_("Region %s is currently in degraded mode.  "
			  "To bring it back to normal state, add %d new spare %s"
			  " to replace the faulty or missing %s.\n"),
			vol->name, missing,
			(missing > 1) ? devices : device,
			(missing > 1) ? devices : device);
	}

}

static void raid1_show_stale_disks(md_volume_t *vol)
{
	int stale_disks;
	md_member_t *member;
	list_element_t iter;

	stale_disks = md_volume_count_stale_disks(vol);
	if (stale_disks == 0) {
		return;
	}

	message_buffer[0] = '\0';
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->flags & MD_MEMBER_STALE) {
			strcat(message_buffer, member->obj->name);
			strcat(message_buffer, " ");
		}
	}

	MESSAGE(_("Region %s : MD superblocks found in object(s) [%s] are not valid.  "
		  "[%s] will not be activated and should be removed from the region.\n"),
		vol->name, message_buffer, message_buffer);
}

static int raid1_init_region(
	md_volume_t *vol, 
	storage_object_t *region, 
	boolean final_call)
{
	int rc = 0;
	md_member_t *member;
	list_element_t iter;
	mdu_array_info_t info;

	LOG_ENTRY();

	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj) {
			md_append_region_to_object(region, member->obj);
		} else {
			LOG_MD_BUG();
		}
	}
	
	region->size = md_volume_calc_size(vol);
	region->data_type = DATA_TYPE;
	region->plugin = raid1_plugin;
	region->private_data = (void *)vol;
	region->dev_major = MD_MAJOR;
	region->dev_minor = vol->md_minor;
	
	vol->flags |= MD_DISCOVERED;
	vol->region = region;
	
	md_get_kernel_info(region, &info);
	if (final_call) {
		if (region->flags & SOFLAG_ACTIVE) {
			rc = md_analyze_active_region(vol);
		} else {
			md_analyze_volume(vol);
			md_fix_dev_major_minor(vol, TRUE);
		}
	} else {
		// Needs to validate later
		vol->flags |= MD_NEEDS_VALIDATE;
	}

	if (vol->flags & MD_CORRUPT) {
		region->flags |= SOFLAG_CORRUPT;
	}
	
	LOG_DETAILS("Region [%s] has been created (%s, %s, %s)\n",
		    region->name,
		    (vol->flags & MD_DISCOVERED) ? "discovered" : "BUG: not discovered",
		    (region->flags & SOFLAG_ACTIVE) ? "active" : "inactive",
		    (vol->flags & MD_DEGRADED) ? "degraded" : ((vol->flags & MD_CORRUPT) ? "corrupt" : "normal"));
	
	LOG_EXIT_INT(rc);
	return rc;
}

static int raid1_create_region(md_volume_t * vol, list_anchor_t output_list, boolean final_call)
{
	int rc = 0;
	storage_object_t * region;

	LOG_ENTRY();

	if (!vol->sb) {
		LOG_MD_BUG();
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (!vol->active_disks) {
		if (final_call) {
			vol->flags |= MD_CORRUPT;
			LOG_CRITICAL("Volume %s does not have any active disks."
				    "  This is final discovery call, the volume is corrupt.\n",
				    vol->name);
		} else {
			LOG_DEBUG("Volume %s does not have any active disks, delaying discovery.\n",
				  vol->name);
			LOG_EXIT_INT(0);
			return 0;
		}
	}

	rc = EngFncs->allocate_region(vol->name, &region);
	if (rc) {
		LOG_WARNING("Region %s is already created, try new name.\n", vol->name);
		rc = md_volume_get_alternative_name(vol, 255);
		if (!rc) {
			LOG_WARNING("Trying tnew region name: %s...\n", vol->name);
			rc = EngFncs->allocate_region(vol->name, &region);
			if (!rc) {
				LOG_WARNING("OK. got it.\n");
			} else {
				LOG_ERROR("Give up.\n");
			}
		} else {
			LOG_CRITICAL("The MD region name space has been exausted.\n");
			LOG_EXIT_INT(rc);
			return rc;
		}
	}
	
	if (!rc) {
		rc = raid1_init_region(vol, region, final_call);
		md_add_object_to_list(region, output_list);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid1_rediscover_region(storage_object_t * region, boolean final_call)
{
	int rc = 0;
	md_volume_t   *vol;
	list_anchor_t children;
	list_anchor_t output_list;
	md_member_t *member;
	list_element_t iter;
	storage_object_t *obj;
	int md_minor;

	LOG_ENTRY();
	
	if (region == NULL) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	vol = region->private_data;
	
	if (md_is_recovery_running(region)) {
		/*
		 * If array is syncing, don't rediscover
		 * Set flag to rediscover the array when it's done.
		 */
		LOG_DEBUG("MD array %s is syncing, skipping rediscovery.\n", vol->name);
		vol->flags |= MD_ARRAY_SYNCING;
		LOG_EXIT_INT(0);
		return 0;
	}

	LOG_DEBUG("About to rediscover volume %s.\n", vol->name);

	md_minor = vol->md_minor; // Save the minor for later comparison

	children = EngFncs->allocate_list();
	output_list = EngFncs->allocate_list();
	md_clear_child_list(region, children);

	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj) {
			EngFncs->insert_thing(children, member->obj, INSERT_AFTER | EXCLUSIVE_INSERT, NULL);
		}
	}
	md_free_volume(vol);
	LIST_FOR_EACH(children, iter, obj) {
		LOG_DEBUG("   Rediscover on this object: %s.\n", obj->name);
	}
	md_discover_volumes(children, output_list);
	region->private_data = NULL;

	vol = volume_list_head;
	while (vol != NULL) {
		if ((!(vol->flags & MD_DISCOVERED)) && (vol->personality == RAID1) && (vol->md_minor == md_minor)) {
			region->flags &= ~(SOFLAG_DIRTY | SOFLAG_CORRUPT | SOFLAG_NEEDS_ACTIVATE | SOFLAG_NEEDS_DEACTIVATE | SOFLAG_ACTIVE);
			rc = raid1_init_region(vol, region, final_call);
		}
		vol = vol->next;
	}

	EngFncs->destroy_list(children);
	EngFncs->destroy_list(output_list);

	if (region->private_data == NULL) {
		LOG_MD_BUG();
		EngFncs->free_region(region);
		rc = ENODEV;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int raid1_discover_regions( list_anchor_t output_list, int *count, boolean final_call)
{
	int rc = 0;
	md_volume_t *vol;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

retry:
	for (vol=volume_list_head; vol; vol=vol->next) {
		if (vol->personality != RAID1) {
			continue;
		}
		
		/* Create raid1 regions */
		if (!(vol->flags & MD_DISCOVERED)) {
			rc = raid1_create_region(vol, output_list, final_call);
			if (!rc && (vol->flags & MD_DISCOVERED)) {
				*count = *count + 1;
			}
		}
		
		/* Rediscover raid1 regions */
		if ((vol->flags & MD_DISCOVERED) && (vol->flags & MD_NEEDS_REDISCOVER)) {
			vol->flags &= ~MD_NEEDS_REDISCOVER;
			rc = raid1_rediscover_region(vol->region, final_call);
			if (!(vol->flags & MD_NEEDS_REDISCOVER)) {
				/* The region was rediscovered, 
				 * go back to check everything again.
				 */
				goto retry;
			}
		}		
	}

	if (final_call) {
		for (vol=volume_list_head; vol; vol=vol->next) {
			if (vol->personality != RAID1) {
				continue;
			}

			if (vol->flags & MD_NEEDS_VALIDATE) {
				if (vol->region->flags & SOFLAG_ACTIVE) {
					rc = md_analyze_active_region(vol);
				} else {
					md_analyze_volume(vol);
					md_fix_dev_major_minor(vol, TRUE);
				}
				vol->flags &= ~MD_NEEDS_VALIDATE;
			}
			
			if ((vol->flags & MD_DEGRADED) && !(vol->flags & MD_ARRAY_SYNCING)) {
				raid1_show_degraded(vol);
			}
			if (vol->stale_disks) {
				raid1_show_stale_disks(vol);
			}
			md_display_corrupt_messages(RAID1);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_discover
 *
 *  Examine all disk segments and find MD PVs. Assemble volume groups
 *  and export all MD logical volumes as EVMS regions.
 *
 *  All newly created regions must be added to the output list, and all
 *  segments from the input list must either be claimed or moved to the
 *  output list.
 */
static int raid1_discover( list_anchor_t input_list,
			   list_anchor_t output_list,
			   boolean final_call )
{
	int count = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (final_call) {
		md_discover_final_call(input_list, output_list, &count);
	} else {
		md_discover_volumes(input_list, output_list);
		LOG_DETAILS("PV discovery complete.\n");

		// LV discovery and exporting
		raid1_discover_regions(output_list, &count, final_call);
		LOG_DETAILS("RAID1 volume discovery complete.\n");
	}

	LOG_EXIT_INT(count);
	return count;
}



/****** Region Functions ******/


static int raid1_get_create_options(
	option_array_t * options,
	char ** spare_disk,
	md_sb_ver_t *sb_ver)
{
	int i;
	int rc = 0;
	boolean ver1_superblock = FALSE;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			switch (options->option[i].number) {
			
			case RAID1_CREATE_OPTION_SB1_INDEX:
				ver1_superblock = options->option[i].value.b;
				break;
			
			case RAID1_OPTION_SPARE_DISK_INDEX:
				// Not worth validation, will catch when we try to find the original
				*spare_disk = options->option[i].value.s;
				break;
			default:
				break;

			}

		} else {

			if (strcmp(options->option[i].name, RAID1_OPTION_SPARE_DISK_NAME) == 0) {
				*spare_disk = options->option[i].value.s;
			} else if ((strcmp(options->option[i].name, RAID1_CREATE_OPTION_SB1_NAME) == 0) ) {
				ver1_superblock = options->option[i].value.b;
			}
		}
	}
	
	if (ver1_superblock == TRUE) {
		sb_ver->major_version = MD_SB_VER_1;
		sb_ver->minor_version = 0;
		sb_ver->patchlevel = 0;
	} else {
		sb_ver->major_version = MD_SB_VER_0;
		sb_ver->minor_version = 90;
		sb_ver->patchlevel = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid1_create_new_region(md_volume_t * vol, list_anchor_t output_list)
{
	int rc=0;
	md_member_t *member;
	list_element_t iter;
	storage_object_t *region;

	LOG_ENTRY();
	rc = EngFncs->allocate_region(vol->name, &region);
	if (rc) {
		LOG_ERROR("Region %s is already created (rc=%d).\n",
			  vol->name, rc);
	}

	if (!rc) {
		LOG_DEBUG("Creating new region %s: nr_disks=%d, raid_disks=%d,"
			  " spares=%d, actives=%d, working=%d\n",
			  vol->name, vol->nr_disks, vol->raid_disks,
			  vol->spare_disks, vol->active_disks, vol->working_disks);
	
		LIST_FOR_EACH(vol->members, iter, member) {
			md_append_region_to_object(region, member->obj);
		}
		region->size = md_volume_calc_size(vol);
		region->data_type = DATA_TYPE;
		region->plugin = raid1_plugin;
		region->private_data = (void *)vol;
		region->dev_major = MD_MAJOR;
		region->dev_minor = vol->md_minor;
		vol->region = region;
		region->flags |= SOFLAG_DIRTY;
		md_add_object_to_list(region, output_list);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * raid1_create
 *
 *  Create a new MD volume using the specified freespace.
 */
static int raid1_create(
	list_anchor_t objects,
	option_array_t *options,
	list_anchor_t new_region_list )
{
	md_volume_t * volume = NULL;
	storage_object_t * object;
	unsigned long size = -1;
	storage_object_t * spare=NULL;
	char * spare_disk = NULL;
	list_element_t iter1, iter2;
	md_sb_ver_t sb_ver = {MD_SB_VER_0, 90, 0};
	md_member_t *member;
	int rc = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!objects || !options || !new_region_list) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	// Must have at least 1 disk
	if (EngFncs->list_count(objects) < 1) {
		LOG_CRITICAL("Must have at least 1 object.\n");
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (!(volume = md_allocate_volume())) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	rc = md_volume_get_available_name(volume, 256);
	if (rc) {
		goto error_free;
	}

	raid1_get_create_options(options, &spare_disk, &sb_ver);

	LIST_FOR_EACH(objects, iter1, object) {
		size = min(size, md_object_usable_size(object, &sb_ver, 0));
	}

	if (spare_disk) {
		spare = md_find_valid_input_object( spare_disk);
		if (spare) {
			size = min(size, md_object_usable_size(spare, &sb_ver, 0));
		}
	}

	rc = md_init_sb(volume, &sb_ver, MD_LEVEL_RAID1, 0, size, 0);
	if (rc) {
		goto error_free;
	}

	// Add raid members
	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		member = md_allocate_member(object);
		if (member) {
			// This will add the member and update the MD superblock.
			member->data_size = size;
			member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_SYNC);
			rc = md_volume_add_new_member(volume, member);
			if (rc) {
				md_free_member(member);
				goto error_free;
			}
		} else {
			rc = ENOMEM;
		}
		if (rc) {
			goto error_free;
		}
		EngFncs->delete_element(iter1);
	}

	// Add spare member
	if (spare) {
		member = md_allocate_member(spare);
		if (member) {
			// This will add the member and update the MD superblock.
			member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_SPARE);
			member->data_size = size;
			rc = md_volume_add_new_member(volume, member);
			if (rc) {
				md_free_member(member);
				goto error_free;
			}
		} else {
			rc = ENOMEM;
		}
		if (rc) {
			goto error_free;
		}
	}

	rc = raid1_create_new_region(volume, new_region_list);
	if (rc) {
		goto error_free;
	} else {
		volume->flags |= MD_DIRTY;
	}
	
	LOG_EXIT_INT(rc);
	return rc;

error_free:
	md_free_volume(volume);
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: w_delete
 *
 * Worker function for raid1_delete and raid1_discard
 */
static int w_delete(storage_object_t *region, list_anchor_t children, boolean tear_down)
{
	md_volume_t   * volume;
	int     rc;

	LOG_ENTRY();

	// Check that this region can be removed.
	if ((rc = raid1_can_delete(region))) {
		LOG_EXIT_INT(rc);
		return rc;
	}
	volume = region->private_data;

	// Remove the parent/child associations with the PVs
	md_clear_child_list(region, children);

	// Delete the volume.
	md_delete_volume(volume, tear_down);
	region->private_data = NULL;
	EngFncs->free_region(region);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_delete
 *
 *  Remove the specified region and consolidate all of its space into
 *  the appropriate freespace region.
 */
static int raid1_delete(storage_object_t * region, list_anchor_t children)
{
	int rc;
	
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	rc = w_delete(region, children, TRUE);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: raid1_discard
 *
 * This function is similar to delete.  Just call delete to free all
 * data structures related to the regions.
 */
static int raid1_discard(list_anchor_t regions)
{
	storage_object_t * region;
	list_element_t le;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	LIST_FOR_EACH(regions, le, region) {
		w_delete(region, NULL, FALSE);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/* Function: raid1_expand_shrink_children
 *
 * Tell all children to expand
 */
static int raid1_expand_shrink_children(int cmd, storage_object_t *region, u_int64_t sectors)
{
	int rc = 0;
	md_volume_t *vol = (md_volume_t *)region->private_data;
	option_array_t option_array;
	u_int64_t size = -1;
	list_element_t iter;
	md_member_t *member;
	storage_object_t *child;

	LOG_ENTRY();

	option_array.count = 1;
	option_array.option[0].is_number_based = FALSE;
	option_array.option[0].name = RAID1_EXPAND_OPTION_SIZE_NAME;
	option_array.option[0].value.ui64 = sectors;

	LOG_DEBUG(" %s region %s. current size = %"PRIu64" sectors\n",
		  (cmd == RAID1_EXPAND) ? "expanding" : "shrinking",
		region->name, region->size);

	LIST_FOR_EACH(vol->members, iter, member) {
		child = member->obj;
		vol->sb_func->zero_superblock(member, FALSE);
		LOG_DEBUG(" %s (current size=%"PRIu64")\n", child->name, child->size);
		if (cmd == RAID1_EXPAND)
			rc = child->plugin->functions.plugin->expand(child, child, NULL, &option_array);
		else
			rc = child->plugin->functions.plugin->shrink(child, child, NULL, &option_array);
		if (!rc) {
			LOG_DEBUG(" %s (new size=%"PRIu64")\n", child->name, child->size);
			size = min(size, md_object_usable_size(child, &vol->sb_ver, 0));
		} else {
			break;
		}
	}

	LIST_FOR_EACH(vol->members, iter, member) {
		member->data_size = size;
	}
	vol->flags |= MD_NEEDS_UPDATE_SIZE;
	region->size = md_volume_calc_size(vol);

	if (!rc) {
		LOG_DEBUG(" region %s now has new size = %"PRIu64" sectors.\n",
			  region->name, region->size);
	} else {
		LOG_CRITICAL(" [%s] Error! region size is now %"PRIu64" sectors.\n",
			  region->name, region->size);
	}

	vol->flags |= MD_DIRTY;
	region->flags |= SOFLAG_DIRTY;
	if (region->flags & SOFLAG_ACTIVE) {
		region->flags |= (SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_expand
 */
static int raid1_expand( storage_object_t    * region,
			 storage_object_t    * expand_object,
			 list_anchor_t         input_objects,
			 option_array_t      * options )
{
	int rc = 0;
	u_int64_t expand_size = 0;
	u_int64_t max_expand_size = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	/* check to make sure */
	if (region &&
		region == expand_object &&
		region->object_type == REGION &&
		region->data_type == DATA_TYPE) {

		raid1_get_expand_options(options, &expand_size);
		rc = EngFncs->can_expand_by(region, &expand_size);
		if (rc) {
			LOG_ERROR("Expand of region %s rejected by the engine.\n",
				  region->name);
			LOG_EXIT_INT(rc);
			return rc;
		}
		

		/* confirm one more time */
		rc = raid1_can_children_expand(region, expand_size, &max_expand_size);
		if (!rc) {
			if (expand_size > max_expand_size) {
				LOG_WARNING(" requested expand_size=%"PRIu64" max_expand_size=%"PRIu64"\n", expand_size, max_expand_size);
				expand_size = max_expand_size;
			}
			rc = raid1_expand_shrink_children(RAID1_EXPAND, region, expand_size);
		}


	} else {
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_shrink
 */
static int raid1_shrink( storage_object_t * region,
			 storage_object_t * shrink_object,
			 list_anchor_t      input_objects,
			 option_array_t   * options )
{
	int rc = 0;
	u_int64_t shrink_size = 0;
	u_int64_t max_shrink_size = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	/* check to make sure */
	if (region &&
		region == shrink_object &&
		region->object_type == REGION &&
		region->data_type == DATA_TYPE) {

		raid1_get_shrink_options(options, &shrink_size);		
		rc = EngFncs->can_shrink_by(region, &shrink_size);
		if (rc) {
			LOG_ERROR("Shrink of region %s rejected by the engine.\n",
				  region->name);
			LOG_EXIT_INT(rc);
			return rc;
		}

		/* confirm one more time */
		rc = raid1_can_children_shrink(region, shrink_size, &max_shrink_size);
		if (!rc) {
			if (shrink_size > max_shrink_size) {
				LOG_WARNING(" requested shrink_size=%"PRIu64" max_shrink_size=%"PRIu64"\n",
					    shrink_size, max_shrink_size);
				shrink_size = max_shrink_size;
			}
			rc = raid1_expand_shrink_children(RAID1_SHRINK, region, shrink_size);
		}


	} else {
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}




static int raid1_replace_child(storage_object_t *region,
			       storage_object_t *child,
			       storage_object_t *new_child)
{
	int rc;
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	rc = md_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * raid1_add_sectors_to_kill_list
 *
 *  The kill_sectors list contains a list of sectors that need to be zeroed
 *  during the next commit. This function is very similar to read/write.
 */
static int raid1_add_sectors_to_kill_list( storage_object_t * region,
					   lsn_t              lsn,
					   sector_count_t     count )
{

	md_volume_t * vol;
	int rc = 0;
	int rc2;
	md_member_t *member;
	list_element_t iter;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (!region) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	vol = (md_volume_t *)region->private_data;
	if (!vol) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (vol->flags & MD_CORRUPT) {
		MESSAGE(_("MD Object %s is corrupt.  Writing data is not allowed.\n"),
			vol->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}
	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to write past end of region %s sector=%"PRIu64"\n",
			  vol->name,lsn+count);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj && md_member_is_raid_disk(member)) {
			rc2 = KILL_SECTORS(member->obj, lsn, count);
			if (!rc && rc2) {
				rc = rc2; //save and return first error code.
			}
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid1_commit_changes
 *
 * The phases:
 *	SETUP
 *	  - for actions required prior to superblocks update
 *	FIRST_METADATA_WRITE
 *	  - write MD superblocks, must have enabled MD_DIRTY flag
 *	SECOND_METADATA_WRITE
 *	  - not used
 *	POST_ACTIVATE
 *	  -  for actions required after superblocks update, _or_
 *	     for queued IOCTLs
 *	  -  reload superblocks via raid1_rediscover_region
 *	
 *	NOTE:  In order to get invoked for all phases of commit process,
 *		leave SOFLAG_DIRTY until the last phase (POST_ACTIVATE)
 */
static int raid1_commit_changes( storage_object_t * region,
				 uint               phase )
{
	md_volume_t * vol;
	int rc = 0;
	int saved_rc;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (! region) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s does not belong to MD.\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	if (! (region->flags & SOFLAG_DIRTY)) {
		LOG_WARNING("Region %s is not dirty - not committing.\n", region->name);
		LOG_EXIT_INT(0);
		return 0;
	}
	
	vol = (md_volume_t *)region->private_data;
	if (!vol) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}


	switch (phase) {
	case SETUP:
		rc = process_setup_funcs(region);
		break;
	case FIRST_METADATA_WRITE:
		if (vol->flags & MD_DIRTY) {
			rc = md_write_sbs_to_disk(vol);
			vol->flags &= ~MD_NEW_REGION;
		}
		break;
	case SECOND_METADATA_WRITE:
		break;
	case POST_ACTIVATE:
		rc = process_md_ioctl_pkgs(region);
		free_ioctl_pkgs(vol);

		/* Despite rc, we will rediscover the MD array, save the return code */
		saved_rc = rc;
		rc = raid1_rediscover_region(region, TRUE);
		if (!rc) {
			/* 
			 * The region was rediscovered, volume pointer is now invalid.
			 * The correct volume pointer is region->private_data.
			 */
			vol = (md_volume_t *)region->private_data;
			region->flags &= ~SOFLAG_DIRTY;
		} else {
			/* Rediscovery failed, display any queued corrupt messages. */
			md_display_corrupt_messages(RAID1);
		}

		if (saved_rc != 0)
			rc = saved_rc;
		break;
	default	:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid1_can_activate_region(storage_object_t * region)
{
	md_volume_t   * volume = (md_volume_t *)region->private_data;
	my_plugin = raid0_plugin;
	LOG_ENTRY();

	if (volume->flags & MD_CORRUPT) {
		LOG_WARNING("MD region %s is corrupt.\n", volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}

static int raid1_activate_region(storage_object_t * region)
{
	int rc;
	md_volume_t *vol;
	my_plugin = raid1_plugin;
	LOG_ENTRY();
	
	if (!region || !(vol = (md_volume_t *)region->private_data)) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	rc = md_activate_region (region);

	LOG_EXIT_INT(rc);
	return rc;

}

static int raid1_can_deactivate_region(storage_object_t * region)
{
	LOG_ENTRY();
	
	LOG_EXIT_INT(0);
	return 0;
}

static int raid1_deactivate_region(storage_object_t * region)
{
	int rc=0;
	my_plugin = raid1_plugin;
	LOG_ENTRY();

	rc = md_deactivate_region (region);

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_get_option_count
 *
 *  Determine the type of Task that is being performed, and return
 *  the number of options that are available for that Task.
 */
static int raid1_get_option_count( task_context_t * task )
{
	int count = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	switch (task->action) {
	case EVMS_Task_Create:
		count = RAID1_CREATE_OPTION_COUNT;
		break;
	case EVMS_Task_Expand:
		count = RAID1_EXPAND_OPTION_COUNT;
		break;
	case EVMS_Task_Shrink:
		count = RAID1_SHRINK_OPTION_COUNT;
		break;
	
	case MD_RAID1_FUNCTION_ADD_SPARE:
	case MD_RAID1_FUNCTION_ADD_ACTIVE:
	case MD_RAID1_FUNCTION_ACTIVATE_SPARE:
	case MD_RAID1_FUNCTION_REMOVE_SPARE:
	case MD_RAID1_FUNCTION_REMOVE_ACTIVE:
	case MD_RAID1_FUNCTION_REMOVE_FAULTY:
	case MD_RAID1_FUNCTION_MARK_FAULTY:
	case MD_RAID1_FUNCTION_REMOVE_STALE:
		count = 0;
		break;
	default:
		count = 0;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}


// Get the list of volumes on the system that we can use as spares
static int raid1_create_selectable_spare_list(
	value_list_t **value_list,
	list_anchor_t selected_objects,
	u_int64_t min_size)
{
	int rc = 0;
	storage_object_t * object;
	list_anchor_t tmp_list;
	int count, i;
	list_element_t li;

	LOG_ENTRY();

	//Clean up the existing list if it exists
	if (*value_list) {
		for (i = 0; i < (*value_list)->count; i++) {
			if ((*value_list)->value[i].s) {
				EngFncs->engine_free((*value_list)->value[i].s);
			}
		}
		EngFncs->engine_free(*value_list);
	}

	rc = EngFncs->get_object_list(DISK | SEGMENT | REGION, // all types
				      DATA_TYPE,
				      NULL,
				      NULL,
				      VALID_INPUT_OBJECT,
				      &tmp_list);

	if (rc) {
		LOG_ERROR("Could not get available objects.\n");
		LOG_EXIT_INT(rc);
		return rc;
	}

	// loop through selected object, removing objects from tmp_list
	LIST_FOR_EACH(selected_objects, li, object) {
		LOG_DETAILS("Object %s selected, removing from spare list\n",object->name);
		EngFncs->remove_thing(tmp_list, object);
	}

	count = EngFncs->list_count(tmp_list);
	count++; // increment count to holed the 'None' selection
	
	*value_list = EngFncs->engine_alloc(count * sizeof(value_t) + sizeof(value_list_t));

	if (*value_list == NULL) {
		LOG_ERROR("No memory\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	(*value_list)->count = count;
	i = 0;
	(*value_list)->value[i].s = EngFncs->engine_strdup(RAID1_NO_SELECTION);
	i++;
	LIST_FOR_EACH(tmp_list, li, object) {
		if (object->size >= min_size) {
			(*value_list)->value[i].s = EngFncs->engine_alloc(strlen(object->name) + 1);
			strcpy((*value_list)->value[i].s, object->name);
			i++;
		} else {
			(*value_list)->count--;
		}
	}
	EngFncs->destroy_list(tmp_list);

	LOG_EXIT_INT(0);
	return 0;
}


static int raid1_get_spare_disks(md_volume_t * vol, list_anchor_t spare_disks)
{
	int rc = 0;
	list_element_t iter;
	md_member_t *member;
	list_element_t li=NULL;

	LOG_ENTRY();

	EngFncs->delete_all_elements(spare_disks);
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj &&
		    (member->flags & MD_MEMBER_DISK_SPARE)) {
			li = EngFncs->insert_thing(spare_disks, member->obj,
						   INSERT_AFTER, NULL);
			if (!li) {
				/* Log error, but continue */
				LOG_ERROR("Could not insert object to list.\n");
				rc = ENOMEM;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid1_get_faulty_disks(md_volume_t *vol, list_anchor_t faulty_disks)
{
	int rc = 0;
	list_element_t iter;
	md_member_t *member;
	list_element_t li=NULL;

	LOG_ENTRY();

	EngFncs->delete_all_elements(faulty_disks);
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj &&
		    (member->flags & MD_MEMBER_DISK_FAULTY)) {
			li = EngFncs->insert_thing(faulty_disks, member->obj,
						   INSERT_AFTER, NULL);
			if (!li) {
				/* Log error, but continue */
				LOG_ERROR("Could not insert object to list.\n");
				rc = ENOMEM;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid1_get_stale_disks(md_volume_t *vol, list_anchor_t stale_disks)
{
	int rc = 0;
	list_element_t iter;
	md_member_t *member;
	list_element_t li=NULL;

	LOG_ENTRY();

	EngFncs->delete_all_elements(stale_disks);
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj &&
		    (member->flags & MD_MEMBER_STALE)) {
			li = EngFncs->insert_thing(stale_disks, member->obj,
						   INSERT_AFTER, NULL);
			if (!li) {
				/* Log error, but continue */
				LOG_ERROR("Could not insert object to list.\n");
				rc = ENOMEM;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int raid1_get_active_disks(md_volume_t *vol, list_anchor_t active_disks)
{
	int rc = 0;
	list_element_t iter;
	md_member_t *member;
	list_element_t li=NULL;

	LOG_ENTRY();

	EngFncs->delete_all_elements(active_disks);
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj && md_member_is_raid_disk(member)) {
			li = EngFncs->insert_thing(active_disks, member->obj,
						   INSERT_AFTER, NULL);
			if (!li) {
				/* Log error, but continue */
				LOG_ERROR("Could not insert object to list.\n");
				rc = ENOMEM;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


typedef struct prune_small_object_parms_s {
	sector_count_t min_size;
	sector_count_t chunk_size;
} prune_small_object_parms_t;

static void prune_small_objects(list_anchor_t objects,
				prune_small_object_parms_t * parms)
{
	list_element_t iter1, iter2;
	storage_object_t * obj;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE(objects, iter1, iter2, obj) {
		if (MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(parms->chunk_size, obj->size) < parms->min_size) {
			EngFncs->delete_element(iter1);
		}
	}

	LOG_EXIT_VOID();
}


/* Function: raid1_init_task
 *
 *  Determine the type of Task that is being performed, and set up the
 *  context structure with the appropriate initial values.
 */
static int raid1_init_task( task_context_t * context )
{
	int rc = 0;
	list_anchor_t tmp_list;
	md_volume_t * vol;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	switch (context->action) {
	
	case EVMS_Task_Create:

		context->option_descriptors->count = RAID1_CREATE_OPTION_COUNT;

		// Version 1 Superblock Option
		if (md_can_create_sb_1() == TRUE) {
			context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].flags = 0;
			context->min_selected_objects = 1;
			context->max_selected_objects = MD_SB_1_DISKS;
		} else {
			context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
			context->min_selected_objects = 1;
			context->max_selected_objects = MD_SB_DISKS;
		}
		context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].constraint.list = NULL;
		context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].help = NULL;
		context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].name =
			EngFncs->engine_strdup( RAID1_CREATE_OPTION_SB1_NAME );
		context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].tip =
			EngFncs->engine_strdup( _("Choose Yes if you want to create MD version 1 super block.") );
		context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].title = EngFncs->engine_strdup( _("Version 1 Super Block") );
		context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].type = EVMS_Type_Boolean;
		context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].unit = EVMS_Unit_None;
		context->option_descriptors->option[RAID1_CREATE_OPTION_SB1_INDEX].value.b = FALSE;

		// Spare Disk Option
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
		// get the list of disks that can be spares.
		raid1_create_selectable_spare_list(
			(value_list_t **)&context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].constraint.list,
			context->selected_objects,0);
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].help = NULL;
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].name = 
			EngFncs->engine_strdup(RAID1_OPTION_SPARE_DISK_NAME );
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].min_len = 1;
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].max_len = EVMS_VOLUME_NAME_SIZE;
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].tip = 
			EngFncs->engine_strdup(_("Object to use as a spare disk in the array") );
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].title = 
			EngFncs->engine_strdup(_("Spare Disk") );
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].type = EVMS_Type_String;
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].unit = EVMS_Unit_None;
		context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].value.s = 
			EngFncs->engine_alloc(EVMS_VOLUME_NAME_SIZE+1);
		strcpy(context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].value.s, RAID1_NO_SELECTION);

		// get a list of all valid input  disks, segments, and regions.
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					 DATA_TYPE,
					 NULL,
					 NULL,
					 VALID_INPUT_OBJECT,
					 &tmp_list);

		// move these items to the acceptable objects list.
		md_transfer_list(tmp_list, context->acceptable_objects);
		EngFncs->destroy_list(tmp_list);

		//Option 1 is a boolean value for version 1 super block


		break;

	case EVMS_Task_Expand:

		vol = (md_volume_t *) context->object->private_data;
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		rc = raid1_init_expand_option_descriptors(context);
		EngFncs->delete_all_elements(context->acceptable_objects);
		break;

	case EVMS_Task_Shrink:

		vol = (md_volume_t *) context->object->private_data;
		context->min_selected_objects = 0;
		context->max_selected_objects = 0;
		rc = raid1_init_shrink_option_descriptors(context);
		EngFncs->delete_all_elements(context->acceptable_objects);
		break;

	case MD_RAID1_FUNCTION_ADD_ACTIVE:
	case MD_RAID1_FUNCTION_ADD_SPARE:
		vol = (md_volume_t *) context->object->private_data;
		{

			context->min_selected_objects = 1;
			if ((context->action == MD_RAID1_FUNCTION_ADD_SPARE) &&
			    (vol->flags & MD_DEGRADED)) {
				context->max_selected_objects =
					vol->raid_disks - md_volume_count_active_disks(vol);

			} else {
				context->max_selected_objects = MAX_DISKS(vol) - vol->nr_disks;
			}
			context->option_descriptors->count = 0;

			rc = EngFncs->get_object_list(DISK | SEGMENT | REGION,
						      DATA_TYPE,
						      NULL,
						      context->object->disk_group,
						      VALID_INPUT_OBJECT | NO_DISK_GROUP,
						      &tmp_list);

			if (rc == 0) {
				prune_small_object_parms_t parms;

				/*
				 * If this MD region is available, it will appear in the list.
				 * Remove this MD region if it is in the list.
				 */
				EngFncs->remove_thing(tmp_list, context->object);

				parms.min_size = md_volume_calc_size(vol);
				parms.chunk_size = 0;

				prune_small_objects(tmp_list, &parms);
				
				/* Remove all parents of this MD region from acceptable list */
				remove_parent_regions_from_list(tmp_list, context->object);
				
				md_transfer_list(tmp_list, context->acceptable_objects);
				EngFncs->destroy_list(tmp_list);
			}
		}
		break;

	case MD_RAID1_FUNCTION_ACTIVATE_SPARE:
	case MD_RAID1_FUNCTION_REMOVE_SPARE:
		vol = (md_volume_t *) context->object->private_data;
		context->min_selected_objects = 1;
		context->max_selected_objects = -1;
		context->option_descriptors->count = 0;
		rc = raid1_get_spare_disks(vol, context->acceptable_objects);
		break;

	case MD_RAID1_FUNCTION_REMOVE_ACTIVE:
	case MD_RAID1_FUNCTION_MARK_FAULTY:
		vol = (md_volume_t *) context->object->private_data;
		context->min_selected_objects = 1;
		context->max_selected_objects = 1;
		context->option_descriptors->count = 0;
		rc = raid1_get_active_disks(vol, context->acceptable_objects);
		break;

	case MD_RAID1_FUNCTION_REMOVE_FAULTY:
		vol = (md_volume_t *) context->object->private_data;
		context->min_selected_objects = 1;
		context->max_selected_objects = -1;	
		context->option_descriptors->count = 0;
		rc = raid1_get_faulty_disks(vol, context->acceptable_objects);
		break;
	
	case MD_RAID1_FUNCTION_REMOVE_STALE:
		vol = (md_volume_t *) context->object->private_data;
		context->min_selected_objects = 1;
		context->max_selected_objects = -1;	
		context->option_descriptors->count = 0;
		rc = raid1_get_stale_disks(vol, context->acceptable_objects);
		break;
	
	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


#define PERCENT_WARN_THRESHOLD  5

static void issue_warning_big_obj(storage_object_t *obj, u_int64_t diffsize)
{
	int answer = 0;
	char * choice_text[2] = { _("OK"), NULL };
	char number_buffer[64];

	sprintf(number_buffer, "%"PRIu64, diffsize * EVMS_VSECTOR_SIZE / (1024*1024) );
	
	QUESTION(&answer, choice_text,
		_("The %s object is %s MB larger than the smallest object in the RAID array.  "
		  "The extra space will not be used.\n"),
		obj->name, number_buffer);
}

static void warn_if_big_objects( task_context_t * context )
{
	storage_object_t * obj;
	storage_object_t * spare = NULL;
	u_int64_t smallest_size = -1;
	list_element_t li;
	md_volume_t *vol = NULL;

	LOG_ENTRY();

	if ( EngFncs->list_empty(context->selected_objects) ) {
		LOG_WARNING("Selected objects list is empty!!!.\n");
		LOG_EXIT_VOID();
	}

	if (context->object) {
		vol = context->object->private_data;
	}

	/* Find the smallest object. */
	LIST_FOR_EACH(context->selected_objects, li, obj) {
		if (vol) {
			smallest_size = min(smallest_size, md_object_usable_size(obj, &vol->sb_ver, 0));
		} else {
			smallest_size = min(smallest_size, obj->size);
		}
	}

	/*
	 * If we got a smallest size, then check the size of the spare, if one
	 * is specified and see if it is the smallest.
	 */
	if (smallest_size != -1) {
		if (context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].value.s != NULL) {
			spare = md_find_valid_input_object(
				context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].value.s);

			if (spare != NULL) {
				if (vol) {
					smallest_size = min(smallest_size, md_object_usable_size(spare, &vol->sb_ver, 0));
				} else {
					smallest_size = min(smallest_size, spare->size);
				}
			}
		}
	}

	/*
	 * Now go through the objects again and issue a warning message for
	 * any object whose size exceeds the threshold over the smallest
	 * object size.
	 */
	if (smallest_size != -1) {
		u_int64_t diffsize;

		LIST_FOR_EACH(context->selected_objects, li, obj) {
			if (vol) {
				diffsize = md_object_usable_size(obj, &vol->sb_ver, 0) - smallest_size;
			} else {
				diffsize = obj->size - smallest_size;
			}

			if (diffsize > (smallest_size * PERCENT_WARN_THRESHOLD) / 100) {
				issue_warning_big_obj(obj, diffsize);
			}
		}

		/*
		 * If we have a spare, check its size too.
		 */
		if (spare != NULL) {
			if (vol) {
				diffsize = md_object_usable_size(spare, &vol->sb_ver, 0) - smallest_size;
			} else {
				diffsize = spare->size - smallest_size;
			}

			if (diffsize > (smallest_size * PERCENT_WARN_THRESHOLD) / 100) {
				issue_warning_big_obj(spare, diffsize);
			}
		}
	}

	LOG_EXIT_VOID();
}

/*
 * FUNCTION: warn_if_big_new_objects
 *
 * This function is similiar to warn_if_big_objects().  This function should be
 * used when adding new objects to an existing RAID1 region.
 */
static void warn_if_big_new_objects( md_volume_t *vol, list_anchor_t objects)
{
	u_int64_t smallest_size;
	u_int64_t diffsize;
	list_element_t li;
	storage_object_t *obj;
	
	LOG_ENTRY();

	smallest_size = md_volume_smallest_data_size(vol);
	
	LIST_FOR_EACH(objects, li, obj) {
		diffsize = md_object_usable_size(obj, &vol->sb_ver, 0) - smallest_size;

		if (diffsize > (smallest_size * PERCENT_WARN_THRESHOLD) / 100) {
			issue_warning_big_obj(obj, diffsize);
		}
	}
	LOG_EXIT_VOID();
}


/* Function: raid1_set_option
 *
 *  Determine the type of Task that is being performed. Then examine the
 *  desired option (using the index), and verify that the given value is
 *  appropriate. Reset the value if necessary and possible. Adjust other
 *  options as appropriate.
 */
static int raid1_set_option( task_context_t * context,
			     u_int32_t        index,
			     value_t        * value,
			     task_effect_t  * effect )
{
	int rc = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!context || !value || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	

	switch (context->action) {
	
	case EVMS_Task_Create:
		switch (index) {
		
		case RAID1_CREATE_OPTION_SB1_INDEX:
			context->option_descriptors->option[index].value.b = value->b;
			if (value->b == TRUE) {
				context->max_selected_objects = MD_SB_1_DISKS;
			} else {
				context->max_selected_objects = MD_SB_DISKS;
			}
			break;
		
		case RAID1_OPTION_SPARE_DISK_INDEX:
			// Not worth validation, will catch when we try to find the original
			strcpy(context->option_descriptors->option[index].value.s, value->s);
			warn_if_big_objects(context);
			break;
		default:
			break;

		}
		break;
	case EVMS_Task_Expand:
		rc = raid1_set_expand_option(context, index, value, effect);
		break;
	case EVMS_Task_Shrink:
		rc = raid1_set_shrink_option(context, index, value, effect);
		break;

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_set_objects
 *
 *  Determine the type of task, and then validate that the objects on the
 *  "selected" list are valid for that task. If so, adjust the option
 *  descriptor as appropriate.
 */
static int raid1_set_objects( task_context_t * context,
			      list_anchor_t declined_objects,
			      task_effect_t * effect )
{
	int rc = 0;
	uint count = 0;
	md_volume_t * volume = NULL;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!context || !declined_objects || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	count = EngFncs->list_count(context->selected_objects);
	if (context->object != NULL) {
		volume = (md_volume_t *) context->object->private_data;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		raid1_create_selectable_spare_list(
			(value_list_t **)&context->option_descriptors->option[RAID1_OPTION_SPARE_DISK_INDEX].constraint.list,
			context->selected_objects, 0);
		warn_if_big_objects(context);
		*effect |= EVMS_Effect_Reload_Options;
		break;
	
	case EVMS_Task_Expand:
		/* Make sure too many objects are not selected. */
		LOG_DEBUG(" Task_Expand: selected objects count = %u\n", count);
		if (count > (volume->nr_disks)) {
			LOG_ERROR("Can only specify up to %d object(s) to expand\n",
				  volume->nr_disks);
			rc = EINVAL;
		}
		break;

	// The Engine makes sure that only available objects appear in the
	// selected_objects list.  Verify that only one object is selected.
	case MD_RAID1_FUNCTION_ADD_SPARE:
	case MD_RAID1_FUNCTION_ADD_ACTIVE:
		/* Make sure too many objects are not selected. */
		if (count > (MAX_DISKS(volume) - volume->nr_disks)) {
			LOG_ERROR("Can only specify up to %d object(s) to added.\n",
				  MAX_DISKS(volume) - volume->nr_disks);
			rc = EINVAL;
		}
		warn_if_big_new_objects(volume, context->selected_objects);
		break;

	case MD_RAID1_FUNCTION_ACTIVATE_SPARE:
	case MD_RAID1_FUNCTION_REMOVE_SPARE:
		if (count > md_volume_count_active_disks(volume)) {
			LOG_ERROR("Can only specify at most %d spare object(s).\n",
				  md_volume_count_active_disks(volume));
			rc = EINVAL;
		}
		break;

	case MD_RAID1_FUNCTION_REMOVE_ACTIVE:
		if (count > md_volume_count_active_disks(volume)) {
			LOG_ERROR("Can only specify at most %d active object(s).\n",
				  md_volume_count_active_disks(volume));
			rc = EINVAL;
		}
		break;

	case MD_RAID1_FUNCTION_REMOVE_FAULTY:
		if (count > md_volume_count_faulty_disks(volume)) {
			LOG_ERROR("Can only specify at most %d faulty object(s).\n",
				  md_volume_count_faulty_disks(volume));
			rc = EINVAL;
		}
		break;
	
	case MD_RAID1_FUNCTION_MARK_FAULTY:
		if (count > 1) {
			LOG_ERROR("Can mark only 1 faulty object.\n");
			rc = EINVAL;
		}
		break;
	
	case MD_RAID1_FUNCTION_REMOVE_STALE:
		if (count > volume->stale_disks) {
			LOG_ERROR("Can only specify at most %d stale object(s).\n", volume->stale_disks);
			rc = EINVAL;
		}
		break;

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_get_info
 *
 *  Return MD-specific information about the specified region. If the
 *  name field is set, only return the "extra" information pertaining
 *  to that name.
 */
static int raid1_get_info( storage_object_t       * region,
			   char                   * name,
			   extended_info_array_t ** info_array )
{

	md_volume_t * volume = NULL;
	int           rc= 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!region || !info_array) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	// Make sure this is an MD region
	if (region->plugin != raid1_plugin) {
		LOG_ERROR("Region %s is not owned by MD RAID1\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	volume = region->private_data;

	rc = md_get_info(volume, name, info_array);

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_get_plugin_info
 *
 *  Return information about the MD plugin. There is no "extra"
 *  information about MD, so "name" should always be NULL.
 */
static int raid1_get_plugin_info( char                     * name,
				  extended_info_array_t   ** info_array )
{

	extended_info_array_t   * info = NULL;
	char buffer[50] = {0};
	int i = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check
	if (! info_array) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (! name) {
		// Get memory for the info array
		if (! (info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*6))) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		// Short Name
		info->info[i].name = EngFncs->engine_strdup("ShortName");
		info->info[i].title = EngFncs->engine_strdup(_("Short Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(raid1_plugin->short_name);
		i++;

		// Long Name
		info->info[i].name = EngFncs->engine_strdup("LongName");
		info->info[i].title = EngFncs->engine_strdup(_("Long Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(raid1_plugin->long_name);
		i++;

		// Plugin Type
		info->info[i].name = EngFncs->engine_strdup("Type");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Type"));
		info->info[i].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(_("Region Manager"));
		i++;

		// Plugin Version
		info->info[i].name = EngFncs->engine_strdup("Version");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		// Required Engine Services Version
		info->info[i].name = EngFncs->engine_strdup("Required_Engine_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
							      "It will not run on older versions of the Engine services."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid1_plugin->required_engine_api_version.major, raid1_plugin->required_engine_api_version.minor, raid1_plugin->required_engine_api_version.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		// Required Plug-in API Version
		info->info[i].name = EngFncs->engine_strdup("Required_Plugin_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Plug-in API Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine plug-in API that this plug-in requires.  "
							      "It will not run on older versions of the Engine plug-in API."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid1_plugin->required_plugin_api_version.plugin.major, raid1_plugin->required_plugin_api_version.plugin.minor, raid1_plugin->required_plugin_api_version.plugin.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;
	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	info->count = i;
	*info_array = info;
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: raid1_read
 *
 *  Perform a logical-to-physical remapping, and send the read down to
 *  the next plugin.
 */
static int raid1_read( storage_object_t * region,
		       lsn_t              lsn,
		       sector_count_t     count,
		       void             * buffer )
{
	md_volume_t * vol;
	md_member_t *member;
	list_element_t iter;
	int rc = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (!region || !buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	vol = (md_volume_t *)region->private_data;
	if (!vol) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (vol->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, returning zero filled buffer.\n",
			  vol->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to read past end of region %s sector=%"PRIu64"\n",
			  vol->name, lsn+count);
	}

	rc = md_region_rw(region, lsn, count, buffer, 0);
	if (rc) {
		rc = 0;
		LIST_FOR_EACH(vol->members, iter, member) {
			if (member->obj && md_member_is_raid_disk(member)) {
				rc = READ(member->obj, lsn+member->data_offset, count, buffer);
				if (!rc) {
					break;
				} else {
					char number_buffer[64];
					sprintf(number_buffer, "%"PRIu64, lsn+count);
					MESSAGE(_("Error reading from mirror %s"
						  " of region %s sector=%s,"
						  " Mirror disabled.\n"),
						member->obj->name,
						vol->name, number_buffer);
					member->raid_disk = -1;
				}
			}
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid1_write
 *
 *  Perform a logical-to-physical remapping, and send the write down to
 *  the next plugin.
 */
static int raid1_write( storage_object_t * region,
			lsn_t              lsn,
			sector_count_t     count,
			void             * buffer )
{
	md_volume_t * vol;
	md_member_t *member;
	list_element_t iter;
	int rc = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if (!region || !buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	vol = (md_volume_t *)region->private_data;
	if (!vol) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (vol->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, returning zero filled buffer.\n",
			  vol->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to read past end of region %s sector=%"PRIu64"\n",
			  vol->name, lsn+count);
	}

	rc = md_region_rw(region, lsn, count, buffer, 1);
	if (rc) {
		rc = 0;
		LIST_FOR_EACH(vol->members, iter, member) {
			if (member->obj && md_member_is_raid_disk(member)) {
				rc = WRITE(member->obj, lsn+member->data_offset, count, buffer);
				if (rc) {
					char number_buffer[64];
					sprintf(number_buffer, "%"PRIu64, lsn+count);
					MESSAGE(_("Error writing to mirror %s"
						  " of region %s sector=%s,"
						  " Mirror disabled.\n"),
						member->obj->name,
						vol->name, number_buffer);
					member->raid_disk = -1;
				}
			}
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}

static int raid1_enable_add_disk_function(
	md_volume_t * volume,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	int rc;
	list_anchor_t available_objects = NULL;
	uint count;
	prune_small_object_parms_t parms;

	LOG_ENTRY();
	
	if (volume->flags & MD_CORRUPT) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	rc = EngFncs->get_object_list(DISK | SEGMENT| REGION,
				      DATA_TYPE,
                                      NULL,
				      volume->region->disk_group,
				      VALID_INPUT_OBJECT | NO_DISK_GROUP,
				      &available_objects);

	if (rc == 0) {

		/*
		 * If this MD region is available, it
		 * will appear in the list.  Bad things
		 * happen if this region is made a
		 * child of itself.  Remove this MD
		 * region if it is in the list.
		 */
		EngFncs->remove_thing(available_objects, volume->region);

		parms.min_size = md_volume_calc_size(volume);
		parms.chunk_size = 0;

		prune_small_objects(available_objects, &parms);
		
		/* Remove all parents of this MD region from available list */
		remove_parent_regions_from_list(available_objects, volume->region);

		count = EngFncs->list_count(available_objects);

		if (count > 0) {

			/* Add spare disk */
			fia->info[*function_count].function = MD_RAID1_FUNCTION_ADD_SPARE;
			fia->info[*function_count].name = EngFncs->engine_strdup("addspare");
			fia->info[*function_count].verb = EngFncs->engine_strdup(_("Add"));
			if (volume->flags & MD_DEGRADED && (volume->nr_disks < volume->raid_disks)) {
				fia->info[*function_count].title = EngFncs->engine_strdup(_("Add spare to fix degraded array"));
				fia->info[*function_count].help =
					EngFncs->engine_strdup(_("Use this function to add a spare object to replace a missing or faulty entry of this degraded region."));
			} else {
				fia->info[*function_count].title = EngFncs->engine_strdup("Add spare object");
				fia->info[*function_count].help =
					EngFncs->engine_strdup(_("Use this function to add an object as a spare object for this RAID array."));
			}

			if ((volume->flags & MD_ARRAY_SYNCING) ||
			    (volume->region_mgr_flags & MD_RAID1_CONFIG_CHANGE_PENDING)) {
				/* temporarily disable this function */
				LOG_DEBUG("%s in array %s temporarily disable Add Spare function\n",
					  (volume->flags & MD_ARRAY_SYNCING) ? "syncing" : "configuration pending",
					volume->name);
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}

			++*function_count;
			enable = 1;

			/* Add active disk option is available only if
			 * - there is no spare
			 * - we can reconfig the array
			 */
			if (!md_volume_count_spare_disks(volume) &&
			    raid1_can_change_region_configuration(volume->region) ) {
				fia->info[*function_count].function = MD_RAID1_FUNCTION_ADD_ACTIVE;
				fia->info[*function_count].name = EngFncs->engine_strdup("addactive");
				fia->info[*function_count].title = EngFncs->engine_strdup(_("Add active object"));
				fia->info[*function_count].verb = EngFncs->engine_strdup(_("Add"));
				fia->info[*function_count].help =
					EngFncs->engine_strdup(_("Use this function to increase the number of mirrors for this RAID array."));
				if (volume->flags & MD_ARRAY_SYNCING) {
					fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
				}
				++*function_count;
			}
		}
		EngFncs->destroy_list(available_objects);
	}

	LOG_EXIT_INT(enable);
	return enable;
}

static int raid1_enable_remove_spare_function(
	md_volume_t * vol,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();

	if (vol->flags & MD_CORRUPT) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	/*
	 * If the RAID array has a spare disk then
	 * Remove Spare is available.
	 */
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj &&
		    (member->flags & MD_MEMBER_DISK_SPARE) &&
		    !(member->flags & MD_MEMBER_NEW) ) {
			fia->info[*function_count].function = MD_RAID1_FUNCTION_REMOVE_SPARE;
			fia->info[*function_count].name = EngFncs->engine_strdup("remspare");
			fia->info[*function_count].title = EngFncs->engine_strdup(_("Remove spare object"));
			fia->info[*function_count].verb = EngFncs->engine_strdup(_("Remove"));
			fia->info[*function_count].help =
				EngFncs->engine_strdup(_("Use this function to remove a spare object from this RAID array."));
			if ((vol->flags & MD_ARRAY_SYNCING) ||
			    (vol->region_mgr_flags & MD_RAID1_CONFIG_CHANGE_PENDING)) {
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}

			++*function_count;
			enable = 1;
			break;
		}
	}

	LOG_EXIT_INT(enable);
	return enable;
}


static int raid1_enable_activate_spare_function(
	md_volume_t * vol,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();
	
	/*
	 * Activate Spare will reconfigure the array.
	 * Therefore, make sure that we can change the array configuration.
	 */
	if (!raid1_can_change_region_configuration(vol->region)) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	/*
	 * If the RAID array has a spare disk then
	 * Activate Spare is available.
	 */
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj &&
		    (member->flags & MD_MEMBER_DISK_SPARE) &&
		    !(member->flags & MD_MEMBER_NEW) ) {
			fia->info[*function_count].function = MD_RAID1_FUNCTION_ACTIVATE_SPARE;
			fia->info[*function_count].name = EngFncs->engine_strdup("activatespare");
			fia->info[*function_count].title = EngFncs->engine_strdup(_("Activate spare object"));
			fia->info[*function_count].verb = EngFncs->engine_strdup(_("Activate"));
			fia->info[*function_count].help =
				EngFncs->engine_strdup(_("Use this function to activate a spare object to reconfigure this RAID1 region.  "
							 "For example, the current RAID1 region is a 2-way mirror with a spare.  "
							 "You may reconfigure it to become a 3-way mirror with no spare.\n"));
			if (vol->flags & MD_ARRAY_SYNCING) {
				fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
			}
			++*function_count;
			enable = 1;
			break;
		}
	}

	LOG_EXIT_INT(enable);
	return enable;
}

static int raid1_enable_remove_active_function(
	md_volume_t * vol,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;

	LOG_ENTRY();

	/*
	 * Remove active will reconfigure the array.
	 * Therefore, make sure that we can change the array configuration.
	 */
	if (!raid1_can_change_region_configuration(vol->region)) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	/*
	 * If the array has at least two active members
	 * then Remove Active is available.
	 */
	if (md_volume_count_active_disks(vol) > 1) {
		fia->info[*function_count].function = MD_RAID1_FUNCTION_REMOVE_ACTIVE;
		fia->info[*function_count].name = EngFncs->engine_strdup("remactive");
		fia->info[*function_count].title = EngFncs->engine_strdup(_("Remove active object"));
		fia->info[*function_count].verb = EngFncs->engine_strdup(_("Remove"));
		fia->info[*function_count].help =
			EngFncs->engine_strdup(_("Use this function to remove an active object from this RAID array."));
		if (vol->flags & MD_ARRAY_SYNCING) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}
		++*function_count;
		enable = 1;
	}

	LOG_EXIT_INT(enable);
	return enable;
}


static int raid1_enable_remove_faulty_function(
	md_volume_t * vol,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;

	LOG_ENTRY();

	/*
	 * If the RAID array has a faulty disk then
	 * Remove Faulty is available.
	 */
	if (md_volume_count_faulty_disks(vol) > 0) {
		fia->info[*function_count].function = MD_RAID1_FUNCTION_REMOVE_FAULTY;
		fia->info[*function_count].name = EngFncs->engine_strdup("remfaulty");
		fia->info[*function_count].title = EngFncs->engine_strdup(_("Remove a faulty object"));
		fia->info[*function_count].verb = EngFncs->engine_strdup(_("Remove"));
		fia->info[*function_count].help =
		EngFncs->engine_strdup(_("Use this function to permanently remove a faulty object from this RAID array."));
		if (vol->flags & MD_ARRAY_SYNCING) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}
		++*function_count;
		enable = 1;
	}
	LOG_EXIT_INT(enable);
	return enable;
}

static int raid1_enable_remove_stale_disk_function(
	md_volume_t * vol,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;

	LOG_ENTRY();

	/*
	 * If the RAID array has a stale disk then
	 * Remove Stale is available.
	 */
	if (md_volume_count_stale_disks(vol) > 0) {
		fia->info[*function_count].function = MD_RAID1_FUNCTION_REMOVE_STALE;
		fia->info[*function_count].name = EngFncs->engine_strdup("remstale");
		fia->info[*function_count].title = EngFncs->engine_strdup(_("Remove a stale object"));
		fia->info[*function_count].verb = EngFncs->engine_strdup(_("Remove"));
		fia->info[*function_count].help =
		EngFncs->engine_strdup(_("Use this function to permanently remove a stale (possibly faulty) object from this RAID array."));
		if (vol->flags & MD_ARRAY_SYNCING) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}
		++*function_count;
		enable = 1;
	}
	LOG_EXIT_INT(enable);
	return enable;
}

static int raid1_enable_mark_disk_faulty_function(
	md_volume_t * vol,
	function_info_array_t * fia,
	int *function_count)
{
	int enable = 0;

	LOG_ENTRY();

	if ((vol->flags & MD_NEW_REGION) ||
	    !md_is_region_active(vol->region) ) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	if (vol->flags & MD_CORRUPT) {
		LOG_EXIT_INT(enable);
		return enable;
	}

	/*
	 * As long as the array is not 1-way mirror, Mark Faulty is available.
	 */
	if (md_volume_count_active_disks(vol) > 1) {
		fia->info[*function_count].function = MD_RAID1_FUNCTION_MARK_FAULTY;
		fia->info[*function_count].name = EngFncs->engine_strdup("markfaulty");
		fia->info[*function_count].title = EngFncs->engine_strdup(_("Mark object faulty"));
		fia->info[*function_count].verb = EngFncs->engine_strdup(_("Mark faulty"));
		fia->info[*function_count].help = EngFncs->engine_strdup(_("Use this function to mark an object faulty in this RAID array."));
		if ((vol->flags & MD_ARRAY_SYNCING) ||
		    (vol->region_mgr_flags & MD_RAID1_CONFIG_CHANGE_PENDING)) {
			fia->info[*function_count].flags |= EVMS_FUNCTION_FLAGS_INACTIVE;
		}
		++*function_count;
		enable = 1;
	}
	LOG_EXIT_INT(enable);
	return enable;
}

/* Function:  raid1_get_plugin_functions
 */
static int raid1_get_plugin_functions(storage_object_t        * region,
				      function_info_array_t * * functions) {

	function_info_array_t * fia;
	int rc, function_count = 0;
	md_volume_t * volume;

	my_plugin = raid1_plugin;
	LOG_ENTRY();
	
	/*
	 * If region is NULL, that means the user is asking for plug-in
	 * functions on the plug-in.  We don't have any plug-in functions that
	 * are global for the plug-in.
	 */
	if (region == NULL) {
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	volume = (md_volume_t *) region->private_data;

	if (volume->flags & MD_CORRUPT) {
		LOG_WARNING("MD region %s is corrupt.\n", volume->name);
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	fia = EngFncs->engine_alloc(sizeof(function_info_array_t) + sizeof(function_info_t) * MD_RAID1_FUNCTION_COUNT);
	if (!fia) {
		LOG_CRITICAL("Error allocating memory for an action info array.\n");
		LOG_EXIT_INT (ENOMEM);
		return ENOMEM;
	}

	/*
	 * Our functions are only allowed if the Engine is opened for
	 * writing.
	 */
	if (!(EngFncs->get_engine_mode() & ENGINE_WRITE)) {
		fia->count = function_count;
		*functions = fia;
		LOG_EXIT_INT(0);
		return 0;
	}
	
	if (md_is_recovery_running(region)) {
		volume->flags |= MD_ARRAY_SYNCING;
		LOG_DEBUG("%s : Resync/recovery is running\n", region->name);
	} else {
		if (volume->flags & MD_ARRAY_SYNCING) {
			rc = raid1_rediscover_region(region, TRUE);
			/* 
			 * If the region was rediscovered,
			 * volume is invalid, must reset!!!
			 */
			volume = (md_volume_t*)region->private_data;
		}
		volume->flags &= ~MD_ARRAY_SYNCING;
	}

	raid1_enable_add_disk_function(volume, fia, &function_count);
	raid1_enable_remove_spare_function(volume, fia, &function_count);
	raid1_enable_activate_spare_function(volume, fia, &function_count);
	raid1_enable_remove_active_function(volume, fia, &function_count);
	raid1_enable_remove_faulty_function(volume, fia, &function_count);
	raid1_enable_remove_stale_disk_function(volume, fia, &function_count);
	raid1_enable_mark_disk_faulty_function(volume, fia, &function_count);

	fia->count = function_count;
	*functions = fia;

	LOG_EXIT_INT(0);
	return 0;
}


static int raid1_can_be_added(md_volume_t * volume, storage_object_t * spare_candidate)
{
	LOG_ENTRY();

	/* The spare must be a disk, segment, or region. */
	if ((spare_candidate->object_type != DISK) &&
	    (spare_candidate->object_type != SEGMENT) &&
	    (spare_candidate->object_type != REGION)) {
		LOG_ERROR("The type of object %s is not data.\n",
			  spare_candidate->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The spare must not be too small. */
	if (MD_CHUNK_ALIGN_NEW_SIZE_SECTORS(0, spare_candidate->size) 
	    < md_volume_calc_size(volume)) {
		LOG_ERROR("Object %s is too small"
			  " to be a spare object for array %s.\n",
			  spare_candidate->name, volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The region cannot be a spare of itself. */
	if (spare_candidate == volume->region) {
		LOG_ERROR("Region %s cannot be a spare object for itself.\n",
			  spare_candidate->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* The spare must be in the same disk group as the region. */
	if (spare_candidate->disk_group != volume->region->disk_group) {
		LOG_ERROR("Object %s is not in the same disk group as region %s.\n",
			  spare_candidate->name, volume->region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}


static int raid1_is_spare(md_volume_t *vol, storage_object_t *obj)
{
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj == obj) {
			if (member->flags & MD_MEMBER_DISK_SPARE) {
				LOG_EXIT_INT(0);
				return 0;
			} else {
				LOG_ERROR("Object %s is in array %s"
					  " but is not a spare disk.\n",
					  obj->name,
					  vol->name);
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n",
		  obj->name, vol->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}


static int raid1_is_faulty(md_volume_t * vol, storage_object_t *obj)
{
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj == obj) {
			if (member->flags & MD_MEMBER_DISK_FAULTY) {
				LOG_EXIT_INT(0);
				return 0;
			} else {
				LOG_ERROR("Object %s is in array %s"
					  " but is not a faulty disk.\n",
					  obj->name,
					  vol->name);
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n",
		  obj->name, vol->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}

static int raid1_is_stale(md_volume_t * vol, storage_object_t * obj)
{
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj == obj) {
			if (member->flags & MD_MEMBER_STALE) {
				LOG_EXIT_INT(0);
				return 0;
			} else {
				LOG_ERROR("Object %s is in array %s"
					  " but is not a stale disk.\n",
					  obj->name,
					  vol->name);
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n",
		  obj->name, vol->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}

static int raid1_is_active(md_volume_t * vol, storage_object_t *obj)
{
	md_member_t *member;
	list_element_t iter;

	LOG_ENTRY();
	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj == obj) {
			if (md_member_is_raid_disk(member)) {
				LOG_EXIT_INT(0);
				return 0;
			} else {
				LOG_ERROR("Object %s is in array %s"
					  " but is not a active disk.\n",
					  obj->name,
					  vol->name);
				LOG_EXIT_INT(EINVAL);
				return EINVAL;
			}
		}
	}

	/* The object was not found in the array. */
	LOG_ERROR("Object %s is not part of array %s.\n",
		  obj->name, vol->name);
	LOG_EXIT_INT(EINVAL);
	return EINVAL;
}


/* 
 * raid1_plugin_function
 */
static int raid1_plugin_function(storage_object_t * region,
				 task_action_t action,
				 list_anchor_t objects,
				 option_array_t * options)
{
	int rc = 0;
	md_volume_t * vol = (md_volume_t *) region->private_data;
	storage_object_t * object;
	list_element_t li;
	uint count;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	if ((action < EVMS_Task_Plugin_Function) ||
	    (action >= EVMS_Task_Plugin_Function + MD_RAID1_FUNCTION_COUNT)) {
		LOG_ERROR("Action code 0x%x is out of range.\n", action);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	count = EngFncs->list_count(objects);
	
	switch (action) {
	case MD_RAID1_FUNCTION_ADD_SPARE:
	case MD_RAID1_FUNCTION_ADD_ACTIVE:
		if (count == 0) {
			LOG_ERROR("Must specify at least one object to be added.\n");
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
		if (count > (MAX_DISKS(vol) - vol->nr_disks)) {
			LOG_ERROR("Can only specify up to %d object(s) to added.\n",
				  MAX_DISKS(vol) - vol->nr_disks);
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
	
		LIST_FOR_EACH(objects, li, object) {
			rc |= raid1_can_be_added(vol, object);
		}
		if (rc != 0) {
			LOG_EXIT_INT(rc);
			return rc;
		}
		break;
	
	case MD_RAID1_FUNCTION_ACTIVATE_SPARE:
	case MD_RAID1_FUNCTION_REMOVE_SPARE:
		if (count == 0) {
			LOG_ERROR("Must specify at least one spare object.\n");
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
		LIST_FOR_EACH(objects, li, object) {
			rc |= raid1_is_spare(vol, object);
		}
		if (rc != 0) {
			LOG_EXIT_INT(rc);
			return rc;
		}
	
		break;
	
	case MD_RAID1_FUNCTION_REMOVE_ACTIVE:
	case MD_RAID1_FUNCTION_MARK_FAULTY:
		if (count == 0) {
			LOG_ERROR("Must specify at least one active object.\n");
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
		LIST_FOR_EACH(objects, li, object) {
			rc |= raid1_is_active(vol, object);
		}
		if (rc != 0) {
			LOG_EXIT_INT(rc);
			return rc;
		}
		break;
	
	case MD_RAID1_FUNCTION_REMOVE_FAULTY:
		if (count == 0) {
			LOG_ERROR("Must specify at least one faulty object.\n");
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
		LIST_FOR_EACH(objects, li, object) {
			rc |= raid1_is_faulty(vol, object);
		}
		if (rc != 0) {
			LOG_EXIT_INT(rc);
			return rc;
		}
		break;
	
	case MD_RAID1_FUNCTION_REMOVE_STALE:
		if (count == 0) {
			LOG_ERROR("Must specify at least one stale object.\n");
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
		LIST_FOR_EACH(objects, li, object) {
			rc |= raid1_is_stale(vol, object);
		}
		if (rc != 0) {
			LOG_EXIT_INT(rc);
			return rc;
		}
		break;
	
	default:
		LOG_ERROR("0x%x is not a valid action code.\n", action);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LIST_FOR_EACH(objects, li, object) {
		switch (action) {
		case MD_RAID1_FUNCTION_ADD_SPARE:
			rc = raid1_add_spare_disk(vol, object);
			break;
		case MD_RAID1_FUNCTION_ADD_ACTIVE:
			rc = raid1_add_active_disk(vol, object);
			break;
		case MD_RAID1_FUNCTION_ACTIVATE_SPARE:
			rc = raid1_activate_spare_disk(vol, object);
			break;
		case MD_RAID1_FUNCTION_REMOVE_SPARE:
			rc = raid1_remove_spare_disk(vol, object);
			break;
		case MD_RAID1_FUNCTION_REMOVE_ACTIVE:
			rc = raid1_remove_active_disk(vol, object);
			break;
		case MD_RAID1_FUNCTION_REMOVE_FAULTY:
			rc = raid1_remove_faulty_disk(vol, object);
			break;
		case MD_RAID1_FUNCTION_MARK_FAULTY:
			rc = raid1_mark_disk_faulty(vol, object);
			if (!rc) {
				if (follow_up_mark_faulty(vol, object) == TRUE) {
					rc = raid1_remove_faulty_disk(vol, object);
				}
			}
	
			break;
		case MD_RAID1_FUNCTION_REMOVE_STALE:
			rc = raid1_remove_stale_disk(vol, object);
			break;
		
		default:
			/*
			 * Shouldn't get here if the validiation
			 * code above did its job.
			 */
			LOG_WARNING("Action code 0x%x slipped past validation.\n", action);
			rc = EINVAL;
			break;
		}
		
		if (rc) {
			break;
		}
	}
	if (rc == 0) {
		vol->region->flags |= SOFLAG_DIRTY;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static void free_region(storage_object_t *region)
{
	md_volume_t * volume = (md_volume_t *)region->private_data;

	md_free_volume(volume);

	LOG_EXIT_VOID();
}

/*
 * raid1_backup_metadata
 *
 * Called to write metadata backup.
 */
int raid1_backup_metadata(storage_object_t *region)
{
	md_volume_t *volume;
	int rc=0;
	
	my_plugin = linear_plugin;
	LOG_ENTRY();


	volume = region->private_data;
	if (volume->flags & MD_CORRUPT) {
		rc = ENOSYS;
		goto out;
	}
	volume->commit_flag |= MD_COMMIT_BACKUP_METADATA;
	volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
	volume->flags |= MD_DIRTY;
	rc = md_write_sbs_to_disk(volume); 
	volume->commit_flag &= ~MD_COMMIT_BACKUP_METADATA;
	volume->commit_flag &= ~MD_COMMIT_DONT_CHECK_ACTIVE;

out:
	LOG_EXIT_INT(rc);
	return rc;
}


static void raid1_plugin_cleanup(void) {

	list_anchor_t regions_list;
	list_element_t li;
	storage_object_t *region;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	EngFncs->get_object_list(REGION, DATA_TYPE, raid1_plugin, NULL, 0, &regions_list);

	LIST_FOR_EACH(regions_list, li, region) {
		free_region(region);
	}

	EngFncs->destroy_list(regions_list);

	LOG_EXIT_VOID();
}


/* Function tables for the MD Region Manager */
static plugin_functions_t raid1_functions = {
	setup_evms_plugin       : raid1_setup_evms_plugin,
	cleanup_evms_plugin     : raid1_plugin_cleanup,
	can_delete              : raid1_can_delete,
	can_expand              : raid1_can_expand,
	can_expand_by           : raid1_can_expand_by,
	can_shrink              : raid1_can_shrink,
	can_shrink_by           : raid1_can_shrink_by,
	can_replace_child	: raid1_can_replace_child,
	discover                : raid1_discover,
	create                  : raid1_create,
	delete                  : raid1_delete,
	discard                 : raid1_discard,
	expand                  : raid1_expand,
	shrink                  : raid1_shrink,
	replace_child		: raid1_replace_child,
	add_sectors_to_kill_list: raid1_add_sectors_to_kill_list,
	commit_changes          : raid1_commit_changes,
	can_activate		: raid1_can_activate_region,
	activate		: raid1_activate_region,
	can_deactivate		: raid1_can_deactivate_region,
	deactivate		: raid1_deactivate_region,
	get_option_count        : raid1_get_option_count,
	init_task               : raid1_init_task,
	set_option              : raid1_set_option,
	set_objects             : raid1_set_objects,
	get_info                : raid1_get_info,
	get_plugin_info         : raid1_get_plugin_info,
	read                    : raid1_read,
	write                   : raid1_write,
	get_plugin_functions    : raid1_get_plugin_functions,
	plugin_function         : raid1_plugin_function,
	backup_metadata         : raid1_backup_metadata
};



/* Function: PluginInit
 *
 *  Initializes the local plugin record
 */

plugin_record_t raid1_plugin_record = {
	id:                     SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 5),

	version:               {major:      MAJOR_VERSION,
				minor:      MINOR_VERSION,
				patchlevel: PATCH_LEVEL},

	required_engine_api_version: {major:      15,
				      minor:      0,
				      patchlevel: 0},
	required_plugin_api_version: {plugin: {major:      13,
					       minor:      0,
					       patchlevel: 0} },

	short_name:             "MDRaid1RegMgr",
	long_name:              "MD Raid 1 Region Manager",
	oem_name:               "IBM",

	functions:              {plugin: &raid1_functions},

	container_functions:    NULL
};

