/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: volume.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <mntent.h>
#include <fcntl.h>
#include <wait.h>
#include <signal.h>

#include "fullengine.h"
#include "volume.h"
#include "engine.h"
#include "handlemgr.h"
#include "discover.h"
#include "common.h"
#include "internalAPI.h"
#include "message.h"
#include "memman.h"
#include "remote.h"
#include "dm.h"
#include "option.h"


boolean is_kernel_volume_mounted(logical_volume_t * vol,
				 debug_level_t      log_level) {

	boolean result = FALSE;
	char vol_name[EVMS_VOLUME_NAME_SIZE+1];
	char * mount_point = NULL;

	LOG_PROC_ENTRY();

	strcpy(vol_name, "/dev/");
	strcat(vol_name, vol->object->name);

	result = is_mounted(vol_name, 0, 0, &mount_point);
	if (result) {
		LOG(log_level, "Kernel volume %s is mounted on %s.  Operations on volume %s are not allowed.\n",
		    vol_name, mount_point, vol->name);

		engine_free(mount_point);
	}

	LOG_PROC_EXIT_BOOLEAN(result);
	return result;
}


static boolean is_mounted_as_swap(char * volume_name) {
	boolean result = FALSE;
	FILE *  proc_swaps;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Checking if %s is mounted as swap.\n", volume_name);

	proc_swaps = fopen("/proc/swaps", "r");
	if (proc_swaps != NULL) {
		char buffer[EVMS_VOLUME_NAME_SIZE+1];
		char * tmp;

		while (!result && (fgets(buffer, EVMS_VOLUME_NAME_SIZE+1, proc_swaps) != NULL)) {

			/*
			 * Find first tab which is after type column and
			 * terminate the string there.
			 */
			tmp = strchr(buffer, '\t');

			if (tmp) {
				*tmp = '\0';
			}

			/*
			 * Find last occurrence of a space and then remove
			 * trailing white space.
			 */
			tmp = strrchr(buffer, ' ');

			if (tmp) {
				while (isspace(*tmp)) tmp--;
				*(tmp+1) = '\0';
			}

			/*
			 * Compare the result in the buffer which should be
			 * only the device name to our volume name.
			 */
			if (strcmp(buffer, volume_name) == 0) {
				result = TRUE;
			}
		}

		fclose(proc_swaps);

	} else {
		LOG_WARNING("Could not open /proc/swaps.\n");
	}

	LOG_PROC_EXIT_BOOLEAN(result);
	return result;
}

/*
 * Check if a volume is mounted.  Return the name of the mount point if
 * requested (mount_name != NULL).
 */
boolean is_mounted(char * volume_name, int dev_major, int dev_minor, char * * mount_name) {

	boolean result = FALSE;
	FILE *          mount_records;	/* Pointer for system's mount records */
	struct mntent * mount_entry;	/* Holds data for entry in mount list */
	dev_t vol_devt = makedev(dev_major, dev_minor);

	LOG_PROC_ENTRY();

	LOG_DEBUG("Checking if %s is mounted.\n", volume_name);

	/*
	 * If the caller wants a mount name, initialize it to NULL in case we
	 * don't find a mount point for the volume.
	 */
	if (mount_name != NULL) {
		*mount_name = NULL;
	}

	/* Make sure the caller specified a volume name. */
	if ((volume_name != NULL) &&
	    (*volume_name != '\0')) {

		mount_records = setmntent(MOUNTED, "r");

		if (mount_records == NULL) {

			/*
			 * Unable to access list of mounted devices in /etc/mtab !
			 * Attempt to open /proc/mounts for access.
			 */
			mount_records = setmntent("/proc/mounts", "r");
		}

		if (mount_records != NULL) {

			/* Scan the mount entries to see if this volume is mounted. */
			while ((!result) && ((mount_entry = getmntent(mount_records)) != NULL)) {
				if (strcmp(mount_entry->mnt_fsname, volume_name) == 0) {
					result = TRUE;

				} else {
					int status;
					struct stat statbuf;

					status = stat(mount_entry->mnt_fsname, &statbuf);
					if (status == 0) {
						if ((vol_devt != 0) &&
						    (statbuf.st_rdev == vol_devt)) {
							LOG_DEBUG("%s has same device number as volume %s (%u:%u)",
								  mount_entry->mnt_fsname, volume_name, dev_major, dev_minor);
							result = TRUE;
						}
					}
				}

				if (result && (mount_name != NULL)) {
					*mount_name = engine_strdup(mount_entry->mnt_dir);
					LOG_DEBUG("%s is mounted on %s.\n", volume_name, mount_entry->mnt_dir);
				}
			}

			/* Close the stream. */
			endmntent(mount_records);

		} else {
			LOG_WARNING("Could not obtain a list of mounted devices from neither /proc/mounts nor " MOUNTED ".\n");
		}

		/* Check if the volume is mounted as swap. */
		if (!result) {
			result = is_mounted_as_swap(volume_name);
			if (result) {
				/*
				 * If the user wants to know the mount point,
				 * make a copy
				 */
				if (mount_name != NULL) {
					*mount_name = engine_strdup("swap");
					LOG_DEBUG("%s is mounted as swap.\n", volume_name);
				}
			}
		}
	}

	if (!result) {
		LOG_DEBUG("%s is not mounted.\n", volume_name);
	}

	LOG_PROC_EXIT_BOOLEAN(result);
	return result;
}


/*
 * Utility to check if a volume is mounted.  Update the mount_point in the
 * volume structure.
 */
boolean is_volume_mounted(logical_volume_t * vol) {

	boolean result = FALSE;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Checking if volume %s (dev node %s) is mounted.\n",
		  vol->name, vol->dev_node);

	if (vol->mount_point != NULL) {
		engine_free(vol->mount_point);
		vol->mount_point = NULL;
	}

	if (vol->flags & VOLFLAG_ACTIVE) {
		result = is_mounted(vol->dev_node,
				    vol->dev_major,
				    vol->dev_minor,
				    &vol->mount_point);
	}

	LOG_PROC_EXIT_BOOLEAN(result);
	return result;
}


/*
 * Utility to check if a volume is opened.  Update the mount_point in the volume
 * structure.
 */
boolean is_volume_opened(logical_volume_t * vol) {

	boolean result = FALSE;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Checking if %s is opened.\n", vol->name);

	if (vol->flags & VOLFLAG_ACTIVE) {
		result = is_volume_mounted(vol);
	}

	LOG_PROC_EXIT_BOOLEAN(result);
	return result;
}


static int can_create_volume(object_handle_t object_handle,
			     debug_level_t   log_level) {
	int rc = 0;
	void * thing;
	object_type_t type;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	rc = translate_handle(object_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!((type == DISK) ||
	      (type == SEGMENT) ||
	      (type == REGION) ||
	      (type == EVMS_OBJECT))) {
		LOG(log_level, "Handle %d is not for a storage object.\n", object_handle);
		rc = EINVAL;
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	obj = (storage_object_t *) thing;

	/* Object must be a data object. */
	if (obj->data_type != DATA_TYPE) {
		LOG(log_level, "Object %s is not a data object.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* Object must not be corrupt. */
	if (obj->flags & SOFLAG_CORRUPT) {
		LOG(log_level, "Object %s is corrupt.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* Only top level objects can be made into volumes. */
	if (!is_top_object(obj)) {
		LOG(log_level, "Object %s is not a top level object.\n", obj->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/*
	 * Ask the object if it can handle being made into a volume.  Right now
	 * we know of no reason why any plug-in should fail this call.
	 */
	rc = obj->plugin->functions.plugin->can_set_volume(obj, TRUE);
	if (rc != 0) {
		LOG(log_level, "Plug-in %s refused to allow object %s to be made into a volume.\n",
		    obj->plugin->short_name, obj->name);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this object be made into an EVMS volume?
 */
int evms_can_create_volume(object_handle_t object_handle) {
	
	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_create_volume(object_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_create_volume(object_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_create_compatibility_volume(object_handle_t object_handle,
					   debug_level_t   log_level) {
	int rc = 0;
	void * thing;
	object_type_t type;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	rc = translate_handle(object_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if ((type == DISK) ||
	    (type == SEGMENT) ||
	    (type == REGION) ||
	    (type == EVMS_OBJECT)) {
		obj = (storage_object_t *) thing;

		if (type == EVMS_OBJECT) {
			LOG(log_level, "Feature object %s cannot be made into a compatibility volume.\n", obj->name);
			rc = EINVAL;
		}

		/*
		 * Object must be a data object, i.e., not metadata, not free
		 * space.
		 */
		if (obj->data_type != DATA_TYPE) {
			LOG(log_level, "Object %s is not a data object.\n", obj->name);
			rc = EINVAL;
		}

		/* Object must not be corrupt. */
		if (obj->flags & SOFLAG_CORRUPT) {
			LOG(log_level, "Object %s is corrupt.\n", obj->name);
			rc = EINVAL;
		}

		/* Object must be a top object. */
		if (!is_top_object(obj)) {
			LOG(log_level, "Object %s is not a top level object.\n", obj->name);
			rc = EINVAL;
		}

		if (rc == 0) {
			/*
			 * Ask the object if it can handle being made into a
			 * volume.  Right now we know of no reason why any
			 * plug-in should fail this call.
			 */
			rc = obj->plugin->functions.plugin->can_set_volume(obj, TRUE);
			if (rc != 0) {
				LOG(log_level, "Plug-in %s refused to allow object %s to be made into a volume.\n",
				    obj->plugin->short_name, obj->name);
			}
		}

	} else {
		LOG(log_level, "Handle %d is not for a storage object.\n", object_handle);
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this object be made into a compatibility volume?
 */
int evms_can_create_compatibility_volume(object_handle_t object_handle) {
	
	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_create_compatibility_volume(object_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_create_compatibility_volume(object_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_set_volume_name(engine_handle_t volume_handle,
			       debug_level_t   log_level) {
	int rc = 0;
	void * thing;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = translate_handle(volume_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type == VOLUME) {
		logical_volume_t * vol = (logical_volume_t *) thing;

		/* Only EVMS volumes can have their name changed. */
		if (!(vol->flags & VOLFLAG_COMPATIBILITY)) {

			/* A volume that is opened cannot have its name changed. */
			if (!is_volume_opened(vol)) {
				/*
				 * Ask the object below the volume if it can
				 * have the volume changed.
				 */
				storage_object_t * vol_obj = (storage_object_t *) vol->object;

				rc = vol_obj->plugin->functions.plugin->can_set_volume(vol_obj, TRUE);

				if (rc != 0) {
					LOG(log_level, "Object %s will not let the name of volume %s be changed.\n", vol_obj->name, vol->name);
				}

			} else {
				LOG(log_level, "Volume \"%s\" is currently opened and cannot have its name changed.\n", vol->name);
				if (vol->mount_point != NULL) {
					LOG(log_level, "Volume \"%s\" is currently mounted on %s.\n", vol->name, vol->mount_point);
				}
				rc = EBUSY;
			}

		} else {
			LOG(log_level, "Volume \"%s\" is not an EVMS volume.  Only EVMS volumes can have their names changed.\n", vol->name);
			rc = EINVAL;
		}

	} else {
		/* The handle is not for a logical volume. */
		LOG(log_level, "Handle %d is not for a volume.\n", volume_handle);
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this volume have its name changed?
 */
int evms_can_set_volume_name(engine_handle_t volume_handle) {
	
	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_set_volume_name(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_set_volume_name(volume_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Make a new volume structure for an object.  Register the volume name.  Put
 * the volume into the volumes_list.  Set the volume pointer in all the child
 * objects of the volume.
 */
int make_volume(storage_object_t * obj,
		char             * name) {

	int rc = 0;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	LOG_DEBUG("Request to make volume %s.\n", name);

	/* Try to register the volume name. */
	rc = engine_register_name(name);
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	vol = (logical_volume_t *) engine_alloc(sizeof(logical_volume_t));

	if (vol != NULL) {
		list_element_t el;

		vol->type = VOLUME;

		/*
		 * Initial volume size is the object size, minus any feature
		 * headers that may be on the object.  Some FSIM's need to know
		 * how big the volume is so they know where to look to see if
		 * their file system is installed.
		 */
		if (obj->feature_header == NULL) {
			vol->vol_size = obj->size;
		} else {
			vol->vol_size = obj->size - (EVMS_FEATURE_HEADER_SECTORS * 2);
		}

		vol->vol_size = round_down_to_hard_sector(vol->vol_size, obj);

		vol->original_vol_size = vol->vol_size;
		vol->shrink_vol_size = vol->vol_size;

		/* Default - assume the volume can be shrunk to oblivion. */
		vol->min_fs_size = 0;

		/* Default - assume the volume can grow to be as big as possible. */
		vol->max_fs_size = round_down_to_hard_sector(-1, obj);
		vol->max_vol_size = round_down_to_hard_sector(-1, obj);

		vol->object = obj;
		if (obj->flags & SOFLAG_READ_ONLY) {
			vol->flags |= VOLFLAG_READ_ONLY;
		}

		strncpy(vol->name, name, sizeof(vol->name) - 1);

		vol->disk_group = obj->disk_group;
		if (obj->flags & SOFLAG_CLUSTER_PRIVATE) {
			vol->flags |= VOLFLAG_CLUSTER_PRIVATE;
		}
		if (obj->flags & SOFLAG_CLUSTER_SHARED) {
			vol->flags |= VOLFLAG_CLUSTER_SHARED;
		}
		if (obj->flags & SOFLAG_CLUSTER_DEPORTED) {
			vol->flags |= VOLFLAG_CLUSTER_DEPORTED;
		}

		el = insert_thing(&volumes_list,
				  vol,
				  INSERT_AFTER,
				  NULL);

		if (el != NULL) {
			sort_list(&volumes_list, compare_volumes, NULL);

			set_volume_in_object(obj, vol);
		}

	} else {
		LOG_CRITICAL("Failed to get memory for a new logical volume structure.\n");
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Generate a 64-bit volume serial number.  The random number generator is used
 * to make up a serial number.  Zero is not a valid serial number.
 */
static u_int64_t generate_volume_serial() {

	u_int64_t serial = 0;
	u_int32_t * p_int32 = (u_int32_t *) &serial;

	LOG_PROC_ENTRY();

	while (serial == 0) {
		p_int32[0] = rand();
		p_int32[1] = rand();
	}

	LOG_DEBUG("Recommended serial number is %016"PRIu64".\n", serial);

	LOG_PROC_EXIT_VOID();
	return serial;
}


/*
 * Add a volume feature header to an object.
 */
int add_volume_feature_header_to_object(storage_object_t * obj) {

	int rc = 0;
	evms_feature_header_t * fh;

	LOG_PROC_ENTRY();

	fh = engine_alloc(sizeof(evms_feature_header_t));
	if (fh != NULL) {
		fh->feature_id = EVMS_VOLUME_FEATURE_ID;
		fh->sequence_number = 1;

		obj->feature_header = fh;
		obj->flags |= SOFLAG_FEATURE_HEADER_DIRTY;

	} else {
		LOG_CRITICAL("Error allocating memory for a feature header for object %s.\n", obj->name);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


char * get_volume_prefix(storage_container_t * disk_group) {

	char * prefix;

	LOG_PROC_ENTRY();
	if (disk_group == NULL) {
		prefix = engine_strdup(EVMS_DEV_NODE_PATH);

	} else {
		prefix = engine_alloc(EVMS_DEV_NODE_PATH_LEN + strlen(disk_group->name) + 2);
		if (prefix != NULL) {
			strcpy(prefix, EVMS_DEV_NODE_PATH);
			strcat(prefix, disk_group->name);
			strcat(prefix, "/");
		}
	}

	LOG_PROC_EXIT_PTR(prefix);
	return prefix;
}

/*
 * make_evms_volume_name checks to make sure the volume name has the
 * EVMS_DEV_NODE_PATH prefix.  If not, it builds a volume name with the
 * EVMS_DEV_NODE_PATH prefix prepended to the name given and changes the
 * original name pointer to point to the buffer.  The buffer must be at least
 * EVMS_VOLUME_NAME_SIZE+1 in size.
 */
static int make_evms_volume_name(char * name,
				 storage_container_t * disk_group,
				 char new_name[EVMS_VOLUME_NAME_SIZE + 1]) {

	int   rc = 0;
	char * prefix;
	char * work_buf;
	char * end;

	LOG_PROC_ENTRY();

	memset(new_name, '\0', EVMS_VOLUME_NAME_SIZE + 1);

	/* Remove any leading spaces. */
	while (isspace(*name)) name++;

	if (*name == '\0') {
		LOG_ERROR("The name is empty.\n");
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	prefix = get_volume_prefix(disk_group);

	if (prefix == NULL) {
		LOG_CRITICAL("Error getting memory for the volume name prefix.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	work_buf = engine_strdup(name);
	if (work_buf == NULL) {
		engine_free(prefix);
		LOG_CRITICAL("Error getting memory for a work buffer.\n");
		LOG_PROC_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/*
	 * Trim off any trailing spaces.
	 * The code above guarantees that name starts with a non-space,
	 * so we know that "end" will not be backed up beyond "name".
	 */
	end = work_buf + strlen(work_buf);
	while ((isspace(*(end - 1)))) end--;
	*end = '\0';

	/*
	 * Prepend the EVMS_DEV_NODE_PATH to the name if the caller didn't do it
	 * already.
	 */
	if (strncmp(work_buf, prefix, strlen(prefix)) != 0) {
		if (strlen(prefix) + strlen(work_buf) <= EVMS_VOLUME_NAME_SIZE) {
			strcpy(new_name, prefix);
			strcat(new_name, work_buf);

		} else {
			LOG_ERROR("Volume name \"%s%s\" is too long.  It must be %d bytes or fewer.\n", prefix, work_buf, EVMS_VOLUME_NAME_SIZE);
			rc = EOVERFLOW;
		}

	} else {
		/* Check to see that the name isn't just the prefix. */
		uint work_buf_len = strlen(work_buf);

		if (work_buf_len == strlen(prefix)) {
			LOG_ERROR("Volume name is made of just the %s prefix.\n", prefix);
			rc = EINVAL;

		} else {
			if (work_buf_len <= EVMS_VOLUME_NAME_SIZE) {
				strcpy(new_name, work_buf);

			} else {
				LOG_ERROR("Volume name \"%s\" is too long.  It must be %d bytes or fewer.\n", work_buf, EVMS_VOLUME_NAME_SIZE);
				rc = EOVERFLOW;
			}
		}
	}

	engine_free(work_buf);
	engine_free(prefix);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int make_dm_map_for_volume(logical_volume_t * vol) {

	int rc = 0;
	dm_target_t target;
	dm_device_t linear;

	LOG_PROC_ENTRY();

	target.start = 0;
	target.length = vol->vol_size;
	target.type = DM_TARGET_LINEAR;
	target.data.linear = &linear;
	target.params = NULL;
	target.next = NULL;
	linear.major = vol->object->dev_major;
	linear.minor = vol->object->dev_minor;
	linear.start = 0;

	dm_activate_volume(vol, &target);

	/* Get the new major:minor for the volume. */
	dm_update_volume_status(vol);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


int make_evms_volume_for_object(storage_object_t * obj,
				char             * vol_name,
				u_int64_t          serial) {

	int rc = 0;
	char name_buf[EVMS_VOLUME_NAME_SIZE + 1];
	char * base_name = name_buf + EVMS_DEV_NODE_PATH_LEN;
	storage_object_t * working_top_object;
	boolean has_own_device;

	LOG_PROC_ENTRY();

	/*
	 * A raw EVMS volume, a non-EVMS storage object made into an EVMS
	 * volume, needs a separate mapping for the volume.  It will have its
	 * own device-mapper device.
	 */
	working_top_object = get_working_top_object(obj);
	has_own_device = (working_top_object->feature_header != NULL);

	rc = make_evms_volume_name(vol_name, obj->disk_group, name_buf);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/* Make sure this volume name does not already exist. */
	rc = engine_validate_name(name_buf);
	if (rc != 0) {
		LOG_ERROR("Name \"%s\" is already in use.\n", name_buf);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * If the volume will have its own device, the base name will be used
	 * for the device name.  Make sure the base name is not in use.
	 */
	if (has_own_device) {
		rc = engine_validate_name(base_name);
		if (rc != 0) {
			LOG_ERROR("Name \"%s\" is already in use.\n", base_name);
			LOG_PROC_EXIT_INT(rc);
			return rc;
		}
	}

	rc = make_volume(obj, name_buf);

	if (rc == 0) {
		logical_volume_t * vol = obj->volume;

		vol->serial_number = serial;

		if (has_own_device) {
			/*
			 * We know this will work since we validated the name
			 * above.
			 */
			engine_register_name(base_name);

			vol->flags |= VOLFLAG_HAS_OWN_DEVICE;

			/* Check if DM already has a mapping for this volume. */
			dm_update_volume_status(vol);

			if (vol->flags & VOLFLAG_ACTIVE) {
				/*
				 * Set the volume size to match the size of the
				 * DM device for the volume.
				 */
				dm_target_t * target;

				if (dm_get_volume_targets(vol, &target) == 0) {
					vol->vol_size = target->length;
					vol->original_vol_size = target->length;

					dm_deallocate_targets(target);
				}
			}

		} else {
			/*
			 * The volume is made from an EVMS object.  The volume
			 * has the same major:minor as the object.
			 */
			vol->dev_major = obj->dev_major;
			vol->dev_minor = obj->dev_minor;

			if (obj->flags & SOFLAG_ACTIVE) {
				vol->flags |= VOLFLAG_ACTIVE;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Make an EVMS volume for an object.
 */
int evms_create_volume(object_handle_t object_handle,
		       char          * name) {
	int rc = 0;
	object_type_t type;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_create_volume(object_handle, name);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_create_volume(object_handle, ERROR);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * evms_can_create_volume() did a translate_handle() and found that the
	 * handle is for a valid storage object.  Therefore we know that the
	 * following translate_handle() will work and will get a storage object.
	 */
	translate_handle(object_handle,
			 (void *) &obj,
			 &type);

	LOG_DEBUG("Request to make object %s into volume \"%s\".\n", obj->name, name);

	/*
	 * If this is not an EVMS object it must have a feature header applied
	 * to it in order to become an EVMS volume.
	 */
	if (type != EVMS_OBJECT) {
		rc = add_volume_feature_header_to_object(obj);
	}

	if (rc == 0) {
		/* Get a serial number for the volume. */
		list_element_t iter;
		logical_volume_t * vol;
		u_int64_t new_serial;

		do {
			new_serial = generate_volume_serial();
			LIST_FOR_EACH(&volumes_list, iter, vol) {
				if (vol->serial_number == new_serial) {
					rc = EINVAL;
					break;
				}
			}
		} while (rc != 0);

		rc = make_evms_volume_for_object(obj, name,new_serial);

		if (rc == 0) {
			obj->volume->flags |= (VOLFLAG_NEW | VOLFLAG_FEATURE_HEADER_DIRTY);
			if ((obj->flags & SOFLAG_NEEDS_ACTIVATE) ||
			    ((obj->flags & SOFLAG_ACTIVE) &&
			     !(obj->flags & SOFLAG_NEEDS_DEACTIVATE))) {
				obj->volume->flags |= VOLFLAG_NEEDS_ACTIVATE;
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int add_volume_to_rename_list(logical_volume_t * vol, char * new_name) {

	int rc = 0;
	storage_object_t * working_top_object;
	rename_volume_t * rv;

	LOG_PROC_ENTRY();

	/*
	 * Compatibility volumes don't need to have their device-mapper
	 * device renamed.
	 */
	if (vol->flags & VOLFLAG_COMPATIBILITY) {
		LOG_PROC_EXIT_INT(0);
		return 0;
	}

	working_top_object = get_working_top_object(vol->object);
	if (working_top_object == NULL) {
		LOG_WARNING("Could not find the working top object for volume %s.\n", vol->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (working_top_object->object_type != EVMS_OBJECT) {
		rv = engine_alloc(sizeof(rename_volume_t));
		if (rv != NULL) {
			rv->volume = vol;
			memcpy(rv->old_vol_name, vol->name, EVMS_VOLUME_NAME_SIZE+1);
			strcpy(rv->new_vol_name, new_name);
			insert_element(&rename_volume_list, &rv->element, INSERT_AFTER, NULL);

		} else {
			LOG_CRITICAL("Error getting memory for a rename volume structure.\n");
			rc = ENOMEM;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Change the name of an EVMS volume.
 */
int evms_set_volume_name(engine_handle_t volume_handle,
			 char          * name) {
	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	char new_name[EVMS_VOLUME_NAME_SIZE+1];

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_set_volume_name(volume_handle, name);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_set_volume_name(volume_handle, ERROR);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * evms_can_set_volume_name() did a translate_handle() and got a valid
	 * volume.  We can safely do a translate_handle and know that it will
	 * return a valid volume.
	 */
	translate_handle(volume_handle,
			 &thing,
			 &type);

	vol = (logical_volume_t *) thing;

	rc = make_evms_volume_name(name, vol->disk_group, new_name);

	if (rc == 0) {
		/* Make sure this volume name does not already exist. */
		rc = engine_validate_name(new_name);
		if ((rc == 0) && (vol->flags & VOLFLAG_HAS_OWN_DEVICE)) {
			rc = engine_validate_name(new_name + EVMS_DEV_NODE_PATH_LEN);
		}

		if (rc == 0) {
			/*
			 * If the volume is active, we must schedule
			 * a rename with device-mapper.
			 */
			if (vol->flags & VOLFLAG_ACTIVE) {
				rc = add_volume_to_rename_list(vol, new_name);
			}

			if (rc == 0) {
				engine_unregister_name(vol->name);
				engine_unregister_name(vol->name + EVMS_DEV_NODE_PATH_LEN);
				memset(&vol->name, 0, EVMS_VOLUME_NAME_SIZE + 1);
				strcpy(vol->name, new_name);
				engine_register_name(new_name);
				engine_register_name(new_name + EVMS_DEV_NODE_PATH_LEN);
				vol->flags |= VOLFLAG_FEATURE_HEADER_DIRTY;
				if (vol->flags & VOLFLAG_ACTIVE) {
					vol->flags |= VOLFLAG_NEEDS_ACTIVATE;
				}
				sort_list(&volumes_list, compare_volumes, NULL);
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Make a volume for a compatibility object.
 */
int make_compatibility_volume_for_object(storage_object_t * obj) {

	int rc = 0;
	char vol_name[EVMS_DEV_NODE_PATH_LEN + EVMS_NAME_SIZE + 1];

	LOG_PROC_ENTRY();

	/* Prepend the EVMS_DEV_NODE_PATH to the object name. */
	strcpy(vol_name, EVMS_DEV_NODE_PATH);
	strcat(vol_name, obj->name);

	rc = make_volume(obj, vol_name);

	if (rc == 0) {
		obj->volume->flags |= VOLFLAG_COMPATIBILITY;
		obj->volume->dev_major = obj->dev_major;
		obj->volume->dev_minor = obj->dev_minor;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Make a compatibility volume for an object.
 */
int evms_create_compatibility_volume(object_handle_t object_handle) {
	
	int rc = 0;
	void * thing;
	object_type_t type;
	storage_object_t * obj;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_create_compatibility_volume(object_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_create_compatibility_volume(object_handle, ERROR);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * can_create_compatibility_volume() did a translate_handle() and found
	 * that the handle is for a valid storage object.  Therefore we know
	 * that the following translate_handle() will work and will get a
	 * storage object.
	 */
	translate_handle(object_handle,
			 &thing,
			 &type);

	obj = (storage_object_t *) thing;

	rc = make_compatibility_volume_for_object(obj);

	if (rc == 0) {
		obj->volume->flags |= VOLFLAG_NEW;
		if ((obj->flags & SOFLAG_NEEDS_ACTIVATE) ||
		    ((obj->flags & SOFLAG_ACTIVE) &&
		     !(obj->flags & SOFLAG_NEEDS_DEACTIVATE))) {
			obj->volume->flags |= VOLFLAG_NEEDS_ACTIVATE;
		}
		/*
		 * Kill any stop data sectors
		 * that might be on the object.
		 */
		rc = obj->plugin->functions.plugin->add_sectors_to_kill_list(obj, obj->size - (EVMS_FEATURE_HEADER_SECTORS * 2), EVMS_FEATURE_HEADER_SECTORS * 2);

		if (rc != 0) {
			LOG_WARNING("Wipe out stop data.  Return code from add_sectors_to_kill_list() was %d.\n", rc);
			LOG_PROC_EXIT_INT(rc);
			return rc;
		}

		obj->flags &= ~SOFLAG_HAS_STOP_DATA;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_mkfs(object_handle_t volume_handle,
		    plugin_handle_t fsim_handle,
		    debug_level_t   log_level) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	plugin_record_t * fsim;

	LOG_PROC_ENTRY();

	rc = translate_handle(volume_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG(log_level, "%d is not a valid handle.\n", volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type != VOLUME) {
		LOG(log_level, "Handle %d is not for a volume.\n", volume_handle);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = (logical_volume_t *) thing;

	rc = translate_handle(fsim_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG(log_level, "%d is not a valid handle.\n", fsim_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type != PLUGIN) {
		LOG(log_level, "Handle %d is not for a plug-in.\n", fsim_handle);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	fsim = (plugin_record_t *) thing;

	if (vol->flags & VOLFLAG_READ_ONLY) {
		LOG(log_level, "Volume \"%s\" is read only.\n", vol->name);
		rc = EINVAL;
	}

	if (is_kernel_volume_mounted(vol, log_level)) {
		rc = EBUSY;
	}

	if (!(vol->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW | VOLFLAG_NEEDS_ACTIVATE))) {
		LOG(log_level, "Volume \"%s\" is not active.\n", vol->name);
		rc = EINVAL;
	}

	if (vol->file_system_manager != NULL) {
		LOG(log_level, "Volume \"%s\" already has file system %s installed on it.  The file system must be removed (unmkfs) before a new file system can be installed.\n", vol->name, vol->file_system_manager->short_name);
		rc = EINVAL;
	}

	if (rc == 0) {
		if (GetPluginType(fsim->id) == EVMS_FILESYSTEM_INTERFACE_MODULE) {

			rc = fsim->functions.fsim->can_mkfs(vol);

		} else {
			LOG(log_level, "Handle %d is for a plug-in %s which is not a File System Interface Module.\n", fsim_handle, fsim->short_name);
			rc = EINVAL;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this file system be installed on this volume?
 */
int evms_can_mkfs(object_handle_t volume_handle,
		  plugin_handle_t fsim_handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_mkfs(volume_handle, fsim_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_mkfs(volume_handle, fsim_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_unmkfs(object_handle_t volume_handle,
		      debug_level_t   log_level) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	plugin_record_t * fsim;

	LOG_PROC_ENTRY();

	rc = translate_handle(volume_handle,
			      &thing,
			      &type);
        if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type != VOLUME) {
		LOG(log_level, "Handle %d is not for a volume.\n", volume_handle);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = (logical_volume_t *) thing;

	if (is_kernel_volume_mounted(vol, log_level)) {
		LOG_PROC_EXIT_INT(EBUSY);
		return EBUSY;
	}

	if (!(vol->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW | VOLFLAG_NEEDS_ACTIVATE))) {
		LOG(log_level, "Volume \"%s\" is not active.\n", vol->name);
		rc = EINVAL;
	}

	if (vol->flags & VOLFLAG_READ_ONLY) {
		LOG(log_level, "Volume \"%s\" is read only.\n", vol->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	fsim = vol->file_system_manager;

	if (fsim == NULL) {
		LOG(log_level, "Volume \"%s\" does not have a File System Interface Module associated with it.\n", vol->name);
		LOG_PROC_EXIT_INT(ENOSYS);
		return ENOSYS;
	}
		
	rc = fsim->functions.fsim->can_unmkfs(vol);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can the file system be removed from the volume?
 */
int evms_can_unmkfs(object_handle_t volume_handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_unmkfs(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_unmkfs(volume_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_fsck(object_handle_t volume_handle,
		    debug_level_t   log_level) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	plugin_record_t * fsim;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(volume_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}
	if (type != VOLUME) {
		LOG(log_level, "Handle %d is not for a volume.\n", volume_handle);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = (logical_volume_t *) thing;

	if (is_kernel_volume_mounted(vol, log_level)) {
		LOG_PROC_EXIT_INT(EBUSY);
		return EBUSY;
	}

	if (!(vol->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW | VOLFLAG_NEEDS_ACTIVATE))) {
		LOG(log_level, "Volume \"%s\" is not active.\n", vol->name);
		rc = EINVAL;
	}

	fsim = vol->file_system_manager;

	if (fsim == NULL) {
		LOG(log_level, "Volume \"%s\" does not have a File System Interface Module associated with it.\n", vol->name);
		LOG_PROC_EXIT_INT(ENOSYS);
		return ENOSYS;
	}
	rc = fsim->functions.fsim->can_fsck(vol);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can fsck be run on the volume?
 */
int evms_can_fsck(object_handle_t volume_handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_fsck(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_fsck(volume_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/* Make a deep copy of a value list. */
static value_list_t * dup_value_list(value_list_t * vl, value_type_t type) {

	uint size;
	value_list_t * dup_vl;

	LOG_PROC_ENTRY()

	size = sizeof(value_list_t) + vl->count * sizeof(value_t);

	dup_vl = engine_alloc(size);

	if (dup_vl != NULL) {

		if (type == EVMS_Type_String) {
			/* Must replace the srings with copies of the strings.*/
			int i;

			dup_vl->count = vl->count;

			for (i = 0; i < vl->count; i++) {
				dup_vl->value[i].s = engine_strdup(vl->value[i].s);
			}

		} else {
			/* The values can be copied as-is. */
			memcpy(dup_vl,vl,size);
		}
	}

	LOG_PROC_EXIT_PTR(dup_vl);
	return dup_vl;
}


/* Make a deep copy of an option array. */
static option_array_t * dup_option_array(option_array_t * oa) {

	uint size;
	option_array_t * dup_oa;

	LOG_PROC_ENTRY()

	size = sizeof(option_array_t) + oa->count * sizeof(key_value_pair_t);

	dup_oa = engine_alloc(size);

	if (dup_oa != NULL) {
		int i;

		memcpy(dup_oa, oa, size);

		for (i = 0; i < oa->count; i++) {

			if (oa->option[i].name != NULL) {
				dup_oa->option[i].name = engine_strdup(oa->option[i].name);
			}

			if (oa->option[i].flags & EVMS_KEY_VALUE_IS_LIST) {
				dup_oa->option[i].value.list = dup_value_list(oa->option[i].value.list,
									      oa->option[i].type);
			} else {
				if (oa->option[i].type == EVMS_Type_String) {
					dup_oa->option[i].value.s = engine_strdup(oa->option[i].value.s);
				}
			}
		}
	}

	LOG_PROC_EXIT_PTR(dup_oa);
	return dup_oa;
}


/*
 * Install a file system on a volume.
 */
int evms_mkfs(object_handle_t  volume_handle,
	      plugin_handle_t  fsim_handle,
	      option_array_t * options) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	plugin_record_t * fsim;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_mkfs(volume_handle, fsim_handle, options);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_mkfs(volume_handle, fsim_handle, ERROR);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	translate_handle(volume_handle,
			 &thing,
			 &type);

	vol = (logical_volume_t *) thing;

	translate_handle(fsim_handle,
			 &thing,
			 &type);

	fsim = (plugin_record_t *) thing;

	rc = fsim->functions.fsim->mkfs_setup(vol, options);
	if (rc == 0) {
		vol->mkfs_options = dup_option_array(options);
		if (vol->mkfs_options != NULL) {
			vol->file_system_manager = fsim;
			vol->flags |= VOLFLAG_MKFS;
		} else {
			LOG_CRITICAL("Error allocating memory for a copy of the options.\n");
			rc = ENOMEM;
		}

	} else {
		LOG_ERROR("The %s FSIM failed to setup for mkfs on volume %s.  Error code is %d.\n", fsim->short_name, vol->name, rc);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Remove a file system from a volume.
 */
int evms_unmkfs(object_handle_t volume_handle) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	plugin_record_t * fsim;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_unmkfs(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_unmkfs(volume_handle, ERROR);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	translate_handle(volume_handle,
			 &thing,
			 &type);

	vol = (logical_volume_t *) thing;

	fsim = vol->file_system_manager;

	/* Tell the FSIM to do any setup for an unmkfs. */
	rc = fsim->functions.fsim->unmkfs_setup(vol);

	if (rc == 0) {
		/* Mark that this volume has no FSIM. */
		vol->file_system_manager = NULL;

		/*
		 * Since it has no FSIM any pending mkfs, or fsck can't
		 * be run.
		 */
		vol->flags &= ~(VOLFLAG_MKFS | VOLFLAG_FSCK);
		if (vol->mkfs_options != NULL) {
			free_option_array_contents(vol->mkfs_options);
			engine_free(vol->mkfs_options);
			vol->mkfs_options = NULL;
		}
		if (vol->fsck_options != NULL) {
			free_option_array_contents(vol->fsck_options);
			engine_free(vol->fsck_options);
			vol->fsck_options = NULL;
		}

		/* If this FSIM claimed the volume during discovery ... */
		if (fsim == vol->original_fsim) {

			/*
			 * Turn on the VOLFLAG_UNMKFS flag so that unmkfs will
			 * be run at commit time.
			 */
			vol->flags |= VOLFLAG_UNMKFS;
		}

		/* Reinitialize the file system and volume limits. */
		get_volume_sizes_and_limits(vol);

	} else {
		LOG_ERROR("The %s FSIM failed to setup for unmkfs on volume %s.  Error code is %d.\n", fsim->short_name, vol->name, rc);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Run fsck on a volume.
 */
int evms_fsck(object_handle_t  volume_handle,
	      option_array_t * options) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	plugin_record_t * fsim;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_fsck(volume_handle, options);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_fsck(volume_handle, ERROR);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	translate_handle(volume_handle,
			 &thing,
			 &type);

	vol = (logical_volume_t *) thing;

	fsim = vol->file_system_manager;

	vol->fsck_options = dup_option_array(options);
	if (vol->fsck_options != NULL) {
		vol->flags |= VOLFLAG_FSCK;

	} else {
		LOG_CRITICAL("Error allocating memory for a copy of the options.\n");
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * evms_get_volume_list returns a pointer to a handle_array_t with handles for
 * logical volumes, optionally filtering on the FSIM that manages the volume.
 * If the object handle for the FSIM is 0, handles for all of the
 * logical volumes will be returned.
 */
int evms_get_volume_list(object_handle_t       fsim_handle,
			 object_handle_t       disk_group_handle,
			 volume_search_flags_t flags,
			 handle_array_t    * * volume_handle_list) {
	int rc = 0;
	void * thing = NULL;
	object_type_t type;
	plugin_record_t * fsim = NULL;
	storage_container_t * disk_group = NULL;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}
		
	if (!local_focus) {
		rc = remote_get_volume_list(fsim_handle, disk_group_handle, flags, volume_handle_list);

		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (fsim_handle != 0) {
		/*
		 * Translate the handle for the FSIM to make sure it is valid
		 * and to get the plugin_record_t for the FSIM.
		 */
		rc = translate_handle(fsim_handle,
				      &thing,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {
			if (type == PLUGIN) {
				fsim = (plugin_record_t *) thing;
				if (GetPluginType(fsim->id) != EVMS_FILESYSTEM_INTERFACE_MODULE) {
					LOG_ERROR("Handle %d is not for a FSIM plug-in.\n", fsim_handle);
					rc = EINVAL;
				}

			} else {
				LOG_ERROR("Handle %d is not for a plug-in.\n", fsim_handle);
				rc = EINVAL;
			}
		}
	}

	if (rc == 0) {
		if (disk_group_handle != 0) {
			/*
			 * Translate the handle for the disk group to make sure
			 * it is valid and to get the disk group
			 * storage_container_t.
			 */
			rc = translate_handle(disk_group_handle,
					      &thing,
					      &type);

			if (rc == HANDLE_MANAGER_NO_ERROR) {
				if (type == CONTAINER) {
					disk_group = (storage_container_t *) thing;

				} else {
					LOG_ERROR("Handle %d is not for a disk group.\n", disk_group_handle);
					rc = EINVAL;
				}
			}
		}
	}

	if (rc == 0) {
		list_anchor_t volume_list;

		/*
		 * Call the internal version of get_volumes_list.  "FSIM" will
		 * be NULL if the caller did not specify a FSIM, else it will be
		 * a pointer to the FSIM's plugin_record_t.
		 */
		rc = engine_get_volume_list(fsim, disk_group, flags, &volume_list);

		if (rc == 0) {
			rc = make_user_handle_array(volume_list, volume_handle_list);

			/*
			 * We are finished with the list that was returned by
			 * engine_get_volume_list.
			 */
			destroy_list(volume_list);
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this volume be converted an EVMS volume?
 */
static int can_convert_to_evms_volume(object_handle_t volume_handle,
				      debug_level_t   log_level) {

	int rc = 0;
	void * thing;
	logical_volume_t * vol;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_convert_to_evms_volume(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(volume_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/* The handle must be for a volume. */
	if (type != VOLUME) {
		LOG(log_level, "Handle %d is not for a volume.\n", volume_handle);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = (logical_volume_t *) thing;

	/* The volume must be active or new. */
	if (!(vol->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW))) {
		LOG(log_level, "Volume \"%s\" is not active.\n", vol->name);
		rc = EINVAL;
	}

	/* The volume must not be opened. */
	if (is_volume_opened(vol)) {
		LOG(log_level, "Volume \"%s\" is currently opened.\n", vol->name);
		if (vol->mount_point != NULL) {
			LOG_DETAILS("Volume \"%s\" is currently mounted on %s.\n", vol->name, vol->mount_point);
		}
		rc = EBUSY;
	}

	/* The volume must be a compatibility volume. */
	if (!(vol->flags & VOLFLAG_COMPATIBILITY)) {
		LOG(log_level, "Volume %s is already an EVMS volume.\n", vol->name);
		rc = EINVAL;
	}

	/*
	 * The kernel volume that corresponds to the compatibilty volume
	 * must not be mounted.
	 */
	if (is_kernel_volume_mounted(vol, log_level)) {
		rc = EBUSY;
	}

	/*
	 * The volume must be active or new since we may have to ask the
	 * file system if it can shrink.
	 */
	if (!(vol->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW))) {
		LOG(log_level, "Volume %s is not activee.\n", vol->name);
		rc = EINVAL;
	}

	if (rc == 0) {
		/*
		 * The volume's object must allow the volume name to be
		 * changed.
		 */
		rc = vol->object->plugin->functions.plugin->can_set_volume(vol->object, TRUE);

		if (rc == 0) {
			/*
			 * If the volume has an FSIM and unmkfs is not
			 * scheduled to be run on the volume, ask the FSIM
			 * if it can shrink the file system to allow room
			 * for the EVMS feature headers.
			 */
			if ((vol->original_fsim != NULL) &&
			    !(vol->flags & VOLFLAG_UNMKFS)) {
				sector_count_t shrink_size = EVMS_FEATURE_HEADER_SECTORS * 2;

				rc = vol->original_fsim->functions.fsim->can_shrink_by(vol, &shrink_size);

				if (rc != 0) {
					LOG(log_level, "Volume %s cannot be shrunk to make room for the EVMS metadata.  FSIM return code was %d.\n", vol->name, rc);
				}
			}

		} else {
			LOG(log_level, "Object %s will not allow the volume's name to be changed.\n", vol->object->name);
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this volume be converted an EVMS volume?
 */
int evms_can_convert_to_evms_volume(object_handle_t volume_handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_convert_to_evms_volume(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_convert_to_evms_volume(volume_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Convert a compatibility volume to an EVMS volume.
 */
int evms_convert_to_evms_volume(object_handle_t volume_handle, char * name) {

	int rc = 0;
	void * thing;
	logical_volume_t * vol;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
        if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_convert_to_evms_volume(volume_handle, name);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_convert_to_evms_volume(volume_handle, ERROR);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * No need to check the return codes from translate_handle() since
	 * evms_can_convert_to_evms_volume() translated the handle and got good
	 * return codes.
	 */
	translate_handle(volume_handle,
			 &thing,
			 &type);

	vol = (logical_volume_t *) thing;

	/*
	 * If the volume currently has no FSIM, warn the user that
	 * we can't perform a shrink on it without an FSIM.
	 */
	if (vol->original_fsim == NULL) {
		char * choices[] = {"Continue", "Cancel", NULL};
		int answer = 0;	    /* Default is "Continue" */

		rc = engine_user_message(&answer, choices,
					 _("Volume %s does not have a File System Interface Module (FSIM) associated with it.  "
					   "The file system (if any) on the volume cannot be shrunk to make space at the end of the volume for the metadata necessary to make an EVMS volume.\n"),
					 vol->name);

		if (answer == 1) {
			/* User wants to cancel. */
			rc = E_CANCELED;
		}
	}

	if (rc == 0) {

		/*
		 * Remove the volume pointer in the volume's object so that
		 * evms_create_volume() below will think the object is a top
		 * level object.
		 */
		vol->object->volume = NULL;

		/*
		 * Use evms_create_volume() to make an EVMS volume for this
		 * volume's object.  It's a little extra overhead, but
		 * evms_create_volume() does all the necessary checks and work
		 * to make an EVMS volume.
		 */
		ensure_app_handle(vol->object);
		rc = evms_create_volume(vol->object->app_handle, name);

		if (rc == 0) {
			/*
			 * The EVMS volume was created.  Copy information
			 * from the old volume to the new volume.  Remove
			 * the old volume from the volumes_list and put it on
			 * the volume_delete_list to be deleted in the
			 * kernel.
			 */
			list_element_t el;
			logical_volume_t * new_vol = vol->object->volume;

			new_vol->file_system_manager        = vol->file_system_manager;
			new_vol->original_fsim              = vol->original_fsim;
			new_vol->fs_size                    = vol->fs_size;
			new_vol->min_fs_size                = vol->min_fs_size;
			new_vol->max_fs_size                = vol->max_fs_size;
			new_vol->original_vol_size          = vol->original_vol_size;
			new_vol->max_vol_size               = vol->max_vol_size;
			new_vol->shrink_vol_size            = min(vol->shrink_vol_size, new_vol->vol_size);
			new_vol->mkfs_options               = vol->mkfs_options;
			new_vol->fsck_options               = vol->fsck_options;
			new_vol->flags                     |= vol->flags &
							      ~(VOLFLAG_ACTIVE | VOLFLAG_COMPATIBILITY);
			new_vol->private_data               = vol->private_data;
			new_vol->original_fsim_private_data = vol->original_fsim_private_data;
			memcpy(new_vol->dev_node, vol->dev_node, sizeof(vol->dev_node));

			/*
			 * Unmark the old volume for unmkfs.  The commit
			 * code that does unmkfs processes both the
			 * volumes_list and the volume_delete_list looking
			 * for volumes that need unmkfs.  unmkfs should only
			 * be run once on this volume.  We will run it on
			 * the new volume, which means it should not be run
			 * on the old volume.
			 */
			vol->flags &= ~VOLFLAG_UNMKFS;

			/* Delete the old volume. */
			if (vol->flags & VOLFLAG_ACTIVE) {
				vol->flags |= VOLFLAG_NEEDS_DEACTIVATE;
			}
			el = insert_thing(&volume_delete_list,
					  vol,
					  INSERT_AFTER,
					  NULL);

			if (el != NULL) {
				remove_thing(&volumes_list, vol);
				engine_unregister_name(vol->name);

			} else {
				/*
				 * The old volume was not inserted into the
				 * volume_delete_list.  The old volume
				 * will not be deleted.  We can't leave the
				 * old volume and the new volume lying around
				 * at the same time.  Destroy the new
				 * volume and return the error.
				 */
				vol->object->volume = vol;
				remove_thing(&volumes_list, new_vol);
				engine_unregister_name(new_vol->name);
				engine_unregister_name(new_vol->name + EVMS_DEV_NODE_PATH_LEN);
				engine_free(new_vol);
			}

		} else {
			/*
			 * The create failed.  Put the original volume
			 * pointer back into the object.
			 */
			vol->object->volume = vol;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this volume be converted back to a compatibility volume?
 */
int evms_can_convert_to_compatibility_volume(object_handle_t volume_handle) {

	int rc = 0;
	void * thing;
	logical_volume_t * vol;
	object_type_t type;
	char new_vol_name[EVMS_VOLUME_NAME_SIZE+1];

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_convert_to_compatibility_volume(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(volume_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/* The handle must be for a volume. */
	if (type != VOLUME) {
		LOG_DETAILS("Handle %d is not for a volume.\n", volume_handle);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = (logical_volume_t *) thing;

	/* The volume must be active or new. */
	if (!(vol->flags & (VOLFLAG_ACTIVE | VOLFLAG_NEW))) {
		LOG_DETAILS("Volume \"%s\" is not active.\n", vol->name);
		rc = EINVAL;
	}

	/* The volume must not be opened. */
	if (is_volume_opened(vol)) {
		LOG_DETAILS("Volume \"%s\" is currently opened.\n", vol->name);
		if (vol->mount_point != NULL) {
			LOG_DETAILS("Volume \"%s\" is currently mounted on %s.\n", vol->name, vol->mount_point);
		}
		rc = EBUSY;
	}

	/* The volume must not already be a compatibility volume. */
	if (vol->flags & VOLFLAG_COMPATIBILITY) {
		LOG_DETAILS("Volume %s is already a compatibility volume.\n", vol->name);
		rc = EINVAL;
	}

	/* The volume's object must not be an EVMS object. */
	if (vol->object->object_type == EVMS_OBJECT) {
		LOG_DETAILS("Volume %s cannot be made into a compatibility volume because its object, %s, is an EVMS object.\n", vol->name, vol->object->name);
		rc = EINVAL;
	}

	if (rc == 0) {
		/*
		 * The resulting compatibility volume name must not
		 * already be taken.
		 */
		strcpy(new_vol_name, EVMS_DEV_NODE_PATH);
		strcat(new_vol_name, vol->object->name);
		rc = engine_validate_name(new_vol_name);

		if (rc == 0) {
			/*
			 * The plug-in must allow the volume to be
			 * changed.
			 */
			rc = vol->object->plugin->functions.plugin->can_set_volume(vol->object, TRUE);
			if (rc != 0) {
				LOG_DETAILS("Plug-in %s will not allow the volume name to be changed.  Return code is %d.\n", vol->object->plugin->short_name, rc);
			}

		} else {
			LOG_DETAILS("The name of the resulting compatibility volume, %s, is already in use.\n", new_vol_name);
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Convert an EVMS volume to a compatibility volume.
 */
int evms_convert_to_compatibility_volume(object_handle_t volume_handle) {

	int rc = 0;
	void * thing;
	logical_volume_t * vol;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_convert_to_compatibility_volume(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = evms_can_convert_to_compatibility_volume(volume_handle);

	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * No need to check the return code from translate_handle() since
	 * evms_can_convert_to_compatibility_volume() translated the handle and
	 * got good return codes.
	 */
	translate_handle(volume_handle,
			 &thing,
			 &type);

	vol = (logical_volume_t *) thing;

	/*
	 * Remove the volume pointer in the volume's object so that
	 * evms_create_compatibility_volume() below will think the object is a
	 * top level object.
	 */
	vol->object->volume = NULL;

	/*
	 * Use evms_create_compatibility_volume() to make an EVMS volume for
	 * this volume's object.  It's a little extra overhead, but
	 * evms_create_compatibility_volume() does all the necessary checks and
	 * work to make an EVMS volume.
	 */
	ensure_app_handle(vol->object);
	rc = evms_create_compatibility_volume(vol->object->app_handle);

	if (rc == 0) {
		list_element_t el;
		logical_volume_t * new_vol = vol->object->volume;

		/*
		 * If the volume was a raw EVMS volume the object has a feature
		 * header on it.  Remove the feature header.  engine_free()
		 * handles NULL.
		 */
		engine_free(vol->object->feature_header);
		vol->object->feature_header = NULL;

		/*
		 * The compatibility volume was created.  Copy information from
		 * the old volume to the new volume.  Remove the old volume from
		 * the volumes_list and put it on the volume_delete_list.
		 */
		new_vol->file_system_manager        = vol->file_system_manager;
		new_vol->original_fsim              = vol->original_fsim;
		new_vol->fs_size                    = vol->fs_size;
		new_vol->min_fs_size                = vol->min_fs_size;
		new_vol->max_fs_size                = vol->max_fs_size;
		new_vol->original_vol_size          = vol->original_vol_size;
		new_vol->max_vol_size               = vol->max_vol_size;
		new_vol->shrink_vol_size            = min(vol->shrink_vol_size, new_vol->vol_size);
		new_vol->mkfs_options               = vol->mkfs_options;
		new_vol->fsck_options               = vol->fsck_options;
		new_vol->flags                      |= vol->flags &
						       ~(VOLFLAG_ACTIVE | VOLFLAG_HAS_OWN_DEVICE);
		new_vol->private_data               = vol->private_data;
		new_vol->original_fsim_private_data = vol->original_fsim_private_data;
		memcpy(new_vol->dev_node, vol->dev_node, sizeof(vol->dev_node));


		remove_thing(&volumes_list, vol);
		engine_unregister_name(vol->name);
		if (vol->flags & VOLFLAG_HAS_OWN_DEVICE) {
			engine_unregister_name(vol->name + EVMS_DEV_NODE_PATH_LEN);
		}

		/* Delete the old volume. */
		if (vol->flags & VOLFLAG_NEW) {
			/* It's a new volume. Just toss it. */
			engine_free(vol);

		} else {
			el = insert_thing(&volume_delete_list,
					  vol,
					  INSERT_AFTER,
					  NULL);

			if (el != NULL) {
				/*
				 * Unmark the old volume for unmkfs.  The commit
				 * code that does unmkfs processes both the
				 * volumes_list and the volume_delete_list
				 * looking for volumes that need unmkfs.  unmkfs
				 * should only be run once on this volume.  We
				 * will run it on the new volume, which means it
				 * should not be run on the old volume.
				 */
				vol->flags &= ~VOLFLAG_UNMKFS;
				/*
				 * Wipe out the EVMS metadata at the end of the
				 * volume's object.
				 */
				rc = vol->object->plugin->functions.plugin->add_sectors_to_kill_list(vol->object, vol->object->size - (EVMS_FEATURE_HEADER_SECTORS * 2), EVMS_FEATURE_HEADER_SECTORS * 2);

				/* Deactivate the volume if it is active. */
				if (vol->flags & VOLFLAG_ACTIVE) {
					vol->flags |= VOLFLAG_NEEDS_DEACTIVATE;
				}

			} else {
				/*
				 * The old volume was not inserted into the
				 * volume_delete_list.  The old volume
				 * will not be deleted.  We can't leave the
				 * old volume and the new volume lying around
				 * at the same time.  Destroy the new
				 * volume and return the error.
				 */
				vol->object->volume = vol;
				remove_thing(&volumes_list, new_vol);
				engine_unregister_name(new_vol->name);
				engine_free(new_vol);
				insert_thing(&volumes_list,
					     vol,
					     INSERT_AFTER,
					     NULL);
			}
		}

	} else {
		/*
		 * The create failed.  Put the original volume pointer back into
		 * the object.
		 */
		vol->object->volume = vol;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_add_feature_to_volume(object_handle_t volume_handle,
				     plugin_handle_t feature_handle,
				     debug_level_t   log_level) {
	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	plugin_record_t * feature;
	sector_count_t new_object_size;

	LOG_PROC_ENTRY();

	rc = translate_handle(volume_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * The handle must be for a volume, the volume must be active or new, it
	 * must not be mounted, the volume must be an EVMS volume, and
	 * the volume's object must not insist on being a top object.
	 */
	if (type != VOLUME) {
		LOG(log_level, "Handle %d is not for a volume.\n", volume_handle);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = (logical_volume_t *) thing;

	rc = translate_handle(feature_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type != PLUGIN) {
		LOG_DETAILS("Handle %d is not for a plug-in.\n", feature_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	feature = (plugin_record_t *) thing;

	if (!(vol->flags & VOLFLAG_ACTIVE)) {
		LOG(log_level, "Volume %s is not active.\n", vol->name);
		rc = EINVAL;
	}

	if (is_volume_opened(vol)) {
		LOG(log_level, "Volume \"%s\" is currently opened.\n", vol->name);
		if (vol->mount_point != NULL) {
			LOG(log_level, "Volume \"%s\" is currently mounted on %s.\n", vol->name, vol->mount_point);
		}
		rc = EBUSY;
	}

	if (vol->flags & VOLFLAG_COMPATIBILITY) {
		LOG(log_level, "Volume %s is not an EVMS volume.\n", vol->name);
		rc = EINVAL;
	}

	if (vol->object->flags & SOFLAG_MUST_BE_TOP) {
		LOG(log_level, "Object %s insists on being the top object in the volume.\n", vol->object->name);
		rc = EINVAL;
	}

	if (GetPluginType(feature->id) != EVMS_FEATURE) {
		LOG(log_level, "Plug-in %s is not an EVMS feature.\n", feature->short_name);
		rc = EINVAL;
	}

	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * Ask the feature if it can add itself and if so, what the size of the
	 * its object would be.
	 */
	rc = feature->functions.plugin->can_add_feature(vol->object, &new_object_size);
	if (rc == 0) {

		/*
		 * If the volume has an FSIM and unmkfs is not scheduled to be
		 * run on the volume and the file system can't fit in the new
		 * object size, ask the FSIM if it can shrink the file system to
		 * allow room for the EVMS feature headers and feature metadata.
		 */
		if ((vol->original_fsim != NULL) &&
		    !(vol->flags & VOLFLAG_UNMKFS) &&
		    (vol->fs_size > new_object_size)) {

			sector_count_t shrink_by_size = vol->fs_size - new_object_size;

			rc = vol->original_fsim->functions.fsim->can_shrink_by(vol, &shrink_by_size);

			if (rc != 0) {
				LOG(log_level, "FSIM %s cannot shrink volume \"%s\" by %"PRIu64" sectors.\n", vol->original_fsim->short_name, vol->name, shrink_by_size);
			}
		}

	} else {
		LOG(log_level, "Feature %s returned error %d from the call to can_add_feature().\n", feature->short_name, rc);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this EVMS feature be added to this EVMS volume?
 */
int evms_can_add_feature_to_volume(object_handle_t volume_handle,
				   plugin_handle_t feature_handle) {
	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_add_feature_to_volume(volume_handle, feature_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_add_feature_to_volume(volume_handle, feature_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Add an EVMS feature to an EVMS volume.
 */
int evms_add_feature_to_volume(object_handle_t  volume_handle,
			       plugin_handle_t  feature_handle,
			       option_array_t * options) {
	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	plugin_record_t * feature;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_add_feature_to_volume(volume_handle, feature_handle, options);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_add_feature_to_volume(volume_handle, feature_handle, ERROR);
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	translate_handle(volume_handle,
			 &thing,
			 &type);

	vol = (logical_volume_t *) thing;

	translate_handle(feature_handle,
			 &thing,
			 &type);

	feature = (plugin_record_t *) thing;

	/*
	 * If the volume currently has no FSIM, warn the user that we can't
	 * perform a shrink on it without an FSIM.
	 */
	if (vol->original_fsim == NULL) {
		char * choices[] = {"Continue", "Cancel", NULL};
		int answer = 0;	    /* Default is "Continue" */

		rc = engine_user_message(&answer, choices,
					 _("Volume %s does not have a File System Interface Module (FSIM) associated with it.  "
					   "The file system (if any) on the volume cannot be shrunk to make space at the end of the volume for the metadata necessary to make an EVMS volume.\n"),
					 vol->name);

		if (answer == 1) {
			/* User wants to cancel. */
			rc = E_CANCELED;
		}

	} else {
		/*
		 * The volume currently has an FSIM.  If the volume is marked to
		 * have unmkfs run on it, then we don't have to ask the FSIM to
		 * shrink as there will be no file system to shrink.
		 */
		if (!(vol->flags & VOLFLAG_UNMKFS)) {
			/*
			 * Ask the FSIM if it can shrink by the amount needed
			 * for the EVMS metadata and feature data.
			 */
			sector_count_t new_object_size;

			/*
			 * Ask the feature for the size of the new object it
			 * would create.
			 */
			rc = feature->functions.plugin->can_add_feature(vol->object, &new_object_size);

			if (rc == 0) {
				/*
				 * If the file system can fit in the new object
				 * we don't have to ask the FSIM to shrink.
				 */
				if (vol->fs_size > new_object_size) {
					sector_count_t shrink_by_size = vol->fs_size - new_object_size;

					rc = vol->original_fsim->functions.fsim->can_shrink_by(vol, &shrink_by_size);

					if (rc != 0) {
						LOG_WARNING("FSIM %s cannot shrink volume %s by %"PRIu64" sectors.\n", vol->original_fsim->short_name, vol->name, shrink_by_size);
					}
				}

			} else {
				LOG_WARNING("Feature %s returned error %d from the call to can_add_feature().\n", feature->short_name, rc);
			}
		}
	}

	if (rc == 0) {
		storage_object_t * prev_vol_obj = vol->object;
		u_int8_t ha[sizeof(handle_array_t) + sizeof(object_handle_t)];
		handle_array_t * input_ha = (handle_array_t *) ha;
		handle_array_t * output_ha;

		/* Build a handle_array for the call to evms_create(). */
		rc = ensure_app_handle(vol->object);

		if (rc == 0) {
			input_ha->count = 1;
			input_ha->handle[0] = vol->object->app_handle;

			/*
			 * If this is a featureless EVMS volume, remove the
			 * feature headers from the volume's object.
			 */
			if (vol->object->feature_header != NULL) {
				remove_feature_headers(vol->object);

				vol->flags |= VOLFLAG_NEEDS_DEACTIVATE;
			}

			/*
			 * Temporarily remove the volume pointer from the
			 * volume's object so that the feature will think it is
			 * a top level object and will install on it.
			 */
			prev_vol_obj->volume = NULL;

			rc = evms_create(feature_handle, input_ha, options, &output_ha);

			if (rc == 0) {

				rc = translate_handle(output_ha->handle[0],
						      &thing,
						      &type);
				if (rc == 0) {
					storage_object_t * feature_object = (storage_object_t *) thing;

					/*
					 * Set the volume pointer in the new
					 * feature object and set the feature
					 * object as the volume's object.
					 */
					feature_object->volume = vol;
					vol->object = feature_object;

					/*
					 * Set the new volume size.  This will
					 * also trigger the shrink code during
					 * commit.
					 */
					vol->vol_size = round_down_to_hard_sector(feature_object->size, feature_object);
					vol->shrink_vol_size = min(vol->shrink_vol_size,
								   vol->vol_size);

					/*
					 * Mark the volume dirty so that
					 * the new object will be
					 * committed and the depth in
					 * the feature headers is
					 * updated.
					 */
					vol->flags |= VOLFLAG_FEATURE_HEADER_DIRTY;
				}

			} else {
				LOG_WARNING("Error code %d when creating feature object from object %s.\n", rc, prev_vol_obj->name);
			}

			/*
			 * Reset the volume pointer in the original volume
			 * object.
			 */
			prev_vol_obj->volume = vol;
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_mount(object_handle_t volume_handle,
		     debug_level_t   log_level) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	rc = translate_handle(volume_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type != VOLUME) {
		LOG(log_level, "Handle %d is not for a volume.\n", volume_handle);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = (logical_volume_t *) thing;

	if (is_volume_mounted(vol)) {
		LOG(log_level, "Volume \"%s\" is already mounted.\n", vol->name);
		rc = EBUSY;
	}

	if (is_kernel_volume_mounted(vol, log_level)) {
		rc = EBUSY;
	}

	if (vol->flags & VOLFLAG_MKFS) {
		LOG(log_level, "Volume %s is scheduled to have a file system put on it.\n", vol->name);
		rc = EINVAL;
	}
	if (vol->flags & VOLFLAG_UNMKFS) {
		LOG(log_level, "Volume %s is scheduled to have the file system removed from it.\n", vol->name);
		rc = EINVAL;
	}
	if (vol->flags & VOLFLAG_NEW) {
		LOG(log_level, "Volume %s has not yet been created.\n", vol->name);
		rc = EINVAL;
	}
	if (vol->flags & VOLFLAG_NOT_MOUNTABLE) {
		LOG(log_level, "The file system says that volume %s cannot be mounted.\n", vol->name);
		rc = EINVAL;
	}
	if (!(vol->flags & VOLFLAG_ACTIVE)) {
		LOG(log_level, "Volume %s is not active.\n", vol->name);
		rc = EINVAL;
	}
	if (vol->vol_size != vol->original_vol_size) {
		LOG(log_level, "Volume %s is scheduled to be expanded.\n", vol->name);
		rc = EINVAL;
	}
	if (vol->shrink_vol_size != vol->original_vol_size) {
		LOG(log_level, "Volume %s is scheduled to be shrunk.\n", vol->name);
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this volume be mounted?
 */
int evms_can_mount(object_handle_t volume_handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_mount(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_mount(volume_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Append the given options to the argv array.
 * This function parses the options string into its individual components
 * separated by white space.  The options string will be modified as part
 * of the parsing.
 */
static void append_options(char * * argv, int * argc, char * options) {

	/* Skip over any leading white space. */
	char * pch = options + strspn(options, " \t");

	while (*pch != '\0') {
		if (*pch == '\"') {
			pch++;
			argv[*argc] = pch;

			/* Skip over stuff until the next '\"'. */
			pch += strcspn(pch, "\"");

			/*
			 * If we hit the closing '\"', replace it with a nul to
			 * terminate the string.  If we hit the end of the
			 * string, consider the rest of the string to be the
			 * option.  Leave pch pointing at the terminating nul.
			 */
			if (*pch != '\0') {
				*pch = '\0';
				pch++;
			}
			
		} else if (*pch == '\'') {
			pch++;
			argv[*argc] = pch;

			/* Skip over stuff until the next '\''. */
			pch += strcspn(pch, "\'");

			/*
			 * If we hit the closing '\'', replace it with a nul to
			 * terminate the string.  If we hit the end of the
			 * string, consider the rest of the string to be the
			 * option.  Leave pch pointing at the terminating nul.
			 */
			if (*pch != '\0') {
				*pch = '\0';
				pch++;
			}

		} else {
			argv[*argc] = pch;

			/* Skip to the next white space. */
			pch += strcspn(pch, " \t");
		}
		(*argc)++;

		/* Any more stuff after the option? */
		if (*pch != '\0') {
			/* Terminate this option. */
			*pch = '\0';

			/* Skip over any following white space. */
			pch++;
			pch += strspn(pch, " \t");
		}
	}
}


/*
 * Mount the volume on the given mount point using the specified options.
 * Options can be NULL if there are no options.
 */
static char mount_output[MAX_USER_MESSAGE_LEN];

int evms_mount(object_handle_t volume_handle, char * mount_point, char * mount_options) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	char * argv[16];
	char * mount_options_copy = NULL;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_mount(volume_handle, mount_point, mount_options);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_mount(volume_handle, ERROR);

	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	/*
	 * Make the dirctory if it does not exist.  The mode parameter gets
	 * anded with ~umask.  So, passing a mode of 0777 gives any new
	 * directories created the perms of umask.
	 */
	rc = make_directory(mount_point, 0777);

	if (rc != 0) {
		LOG_WARNING("Unable to make directory %s.\n", mount_point);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (mount_options != NULL) {
		if (strlen(mount_options) > 0) {
			mount_options_copy = engine_strdup(mount_options);
			if (mount_options_copy == NULL) {
				LOG_CRITICAL("Error getting memory for a copy of the mount options.\n");
				LOG_PROC_EXIT_INT(ENOMEM);
				return ENOMEM;
			}
		}
	}

	if (rc == 0) {
		int argc;
		int fds2[2];
		pid_t pidm;
		int status;

		/*
		 * evms_can_mount() succeeded.  Therefore we know that
		 * translate_handle() will succeed, so we don't have to check
		 * for an error return code.
		 */
		translate_handle(volume_handle,
				 &thing,
				 &type);

		vol = (logical_volume_t *) thing;

		status = pipe(fds2);
		if (status == 0) {
			argv[0] = "mount";
			argc = 1;
			if (mount_options_copy != NULL) {
				append_options(argv, &argc, mount_options_copy);
			}
			argv[argc] = vol->dev_node;
			argc++;
			argv[argc] = mount_point;
			argc++;
			argv[argc] = NULL;

			pidm = engine_fork_and_execvp(vol, argv, NULL, fds2, fds2);
			if (pidm != -1) {
				fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
				waitpid( pidm, &status, 0 );
				if (WIFEXITED(status)) {
					int bytes_read;

					bytes_read = read(fds2[0], mount_output, sizeof(mount_output) - 1);
					if (bytes_read > 0) {
						mount_output[bytes_read] = '\0';
						engine_user_message(NULL, NULL,
								    _("mount: %s"), mount_output);
					}

					rc = WEXITSTATUS(status);

				} else {
					if (WIFSIGNALED(status)) {
						LOG_SERIOUS("mount was terminated by signal %d: %s\n", WTERMSIG(status), sys_siglist[WTERMSIG(status)]);
						rc = EINTR;

					} else {
						/*
						 * We should not get here.
						 * The child process should either
						 * exit or get signaled.
						 */
						LOG_SERIOUS("Child process neither exited nor was signaled.  status is %#x.\n", status);
						rc = ENOSYS;
					}
				}

			} else {
				rc = errno;
				LOG_WARNING("fork() failed with error code %d: %s\n", rc, strerror(rc));
			}

		} else {
			rc = errno;
			LOG_SERIOUS("Pipe creation failed with error code %d: %s\n", rc, strerror(rc));
		}

		/* engine_free() handles NULL pointers. */
		engine_free(mount_options_copy);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_unmount(object_handle_t volume_handle,
		       debug_level_t   log_level) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	rc = translate_handle(volume_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type != VOLUME) {
		LOG(log_level, "Handle %d is not for a volume.\n", volume_handle);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = (logical_volume_t *) thing;

	if (!is_volume_mounted(vol)) {
		LOG(log_level, "Volume \"%s\" is not mounted.\n", vol->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this volume be unmounted?
 */
int evms_can_unmount(object_handle_t volume_handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_unmount(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_unmount(volume_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Unmount the volume.
 */
int evms_unmount(object_handle_t volume_handle) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	char * argv[3];

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_unmount(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_unmount(volume_handle, ERROR);

	if (rc == 0) {
		int fds2[2];
		pid_t pidm;
		int status;

		/*
		 * evms_can_unmount() succeeded.  Therefore we know that
		 * translate_handle() will succeed, so we don't have to
		 * check for an error return code.
		 */
		translate_handle(volume_handle,
				 &thing,
				 &type);

		vol = (logical_volume_t *) thing;

		status = pipe(fds2);
		if (status == 0) {
			argv[0] = "umount";
			argv[1] = vol->mount_point;
			argv[2] = NULL;

			pidm = engine_fork_and_execvp(vol, argv, NULL, fds2, fds2);
			if (pidm != -1) {
				fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
				waitpid( pidm, &status, 0 );
				if (WIFEXITED(status)) {
					int bytes_read;

					bytes_read = read(fds2[0], mount_output, sizeof(mount_output) - 1);
					if (bytes_read > 0) {
						mount_output[bytes_read] = '\0';
						engine_user_message(NULL, NULL,
								    _("umount: %s"), mount_output);
					}

					rc = WEXITSTATUS(status);

				} else {
					if (WIFSIGNALED(status)) {
						LOG_SERIOUS("umount was terminated by signal %d: %s\n", WTERMSIG(status), sys_siglist[WTERMSIG(status)]);
						rc = EINTR;

					} else {
						/*
						 * We should not get here.
						 * The child process should either
						 * exit or get signaled.
						 */
						LOG_SERIOUS("Child process neither exited nor was signaled.  status is %#x.\n", status);
						rc = ENOSYS;
					}
				}

			} else {
				rc = errno;
				LOG_WARNING("fork() failed with error code %d: %s\n", rc, strerror(rc));
			}

		} else {
			rc = errno;
			LOG_SERIOUS("Pipe creation failed with error code %d: %s\n", rc, strerror(rc));
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


static int can_remount(object_handle_t volume_handle,
		       debug_level_t   log_level) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(volume_handle,
			      &thing,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type != VOLUME) {
		LOG(log_level, "Handle %d is not for a volume.\n", volume_handle);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	vol = (logical_volume_t *) thing;

	if (!is_volume_mounted(vol)) {
		LOG(log_level, "Volume \"%s\" is not mounted.\n", vol->name);
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (strcmp(vol->mount_point, "swap") == 0) {
		LOG(log_level, "Remounting a swap volume is not supported.\n");
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (is_kernel_volume_mounted(vol, log_level)) {
		LOG_PROC_EXIT_INT(EBUSY);
		return EBUSY;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Can this volume be re-mounted?
 */
int evms_can_remount(object_handle_t volume_handle) {

	int rc = 0;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_can_remount(volume_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_remount(volume_handle, DETAILS);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * Remount the volume using the specified options.
 * Options can be NULL if there are no options.
 */
int evms_remount(object_handle_t volume_handle, char * mount_options) {

	int rc = 0;
	void * thing;
	object_type_t type;
	logical_volume_t * vol;
	char * argv[16];
	char * mount_options_copy = NULL;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_remount(volume_handle, mount_options);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = can_remount(volume_handle, ERROR);

	if (rc == 0) {
		if (mount_options != NULL) {
			if (strlen(mount_options) > 0) {
				mount_options_copy = engine_strdup(mount_options);
				if (mount_options_copy == NULL) {
					LOG_CRITICAL("Error getting memory for a copy of the mount options.\n");
					rc = ENOMEM;
				}
			}
		}

		if (rc == 0) {
			int argc;
			int fds2[2];
			pid_t pidm;
			int status;

			/*
			 * evms_can_remount() succeeded.  Therefore we
			 * know that translate_handle() will succeed, so
			 * we don't have to check for an error return
			 * code.
			 */
			translate_handle(volume_handle,
					 &thing,
					 &type);

			vol = (logical_volume_t *) thing;

			status = pipe(fds2);
			if (status == 0) {
				argv[0] = "mount";
				argv[1] = "-o";
				argv[2] = "remount";
				argc = 3;
				if (mount_options_copy != NULL) {
					append_options(argv, &argc, mount_options_copy);
				}
				argv[argc] = vol->dev_node;
				argc++;
				argv[argc] = NULL;

				pidm = engine_fork_and_execvp(vol, argv, NULL, fds2, fds2);
				if (pidm != -1) {
					fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
					waitpid( pidm, &status, 0 );
					if (WIFEXITED(status)) {
						int bytes_read;

						bytes_read = read(fds2[0], mount_output, sizeof(mount_output) - 1);
						if (bytes_read > 0) {
							mount_output[bytes_read] = '\0';
							engine_user_message(NULL, NULL,
									    _("mount: %s"), mount_output);
						}

						rc = WEXITSTATUS(status);

					} else {
						if (WIFSIGNALED(status)) {
							LOG_SERIOUS("mount was terminated by signal %d: %s\n", WTERMSIG(status), sys_siglist[WTERMSIG(status)]);
							rc = EINTR;

						} else {
							/*
							 * We should not get here.
							 * The child process should either
							 * exit or get signaled.
							 */
							LOG_SERIOUS("Child process neither exited nor was signaled.  status is %#x.\n", status);
							rc = ENOSYS;
						}
					}

				} else {
					rc = errno;
					LOG_WARNING("fork() failed with error code %d: %s\n", rc, strerror(rc));
				}

			} else {
				rc = errno;
				LOG_SERIOUS("Pipe creation failed with error code %d: %s\n", rc, strerror(rc));
			}

			/* engine_free() handles NULL pointers. */
			engine_free(mount_options_copy);
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

