/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libbsd.so
 *
 *   File: segoptions.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "bsd_plugin.h"





/*
 * Returns segment specific information
 */
int bsd_get_segment_info( storage_object_t  * object, extended_info_array_t * * info)
{
        int rc = EINVAL;
        extended_info_array_t   *Info;


        LOG_ENTRY();

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( SEG_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

                Info->count = SEG_INFO_COUNT;

                Info->info[SEG_INFO_NAME_INDEX].name = EngFncs->engine_strdup( "Name" );
                Info->info[SEG_INFO_NAME_INDEX].title = EngFncs->engine_strdup( _("Name") );
                Info->info[SEG_INFO_NAME_INDEX].desc = EngFncs->engine_strdup( _("This is the partition name. It must be unique on the system.") );
                Info->info[SEG_INFO_NAME_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_NAME_INDEX].value.s = EngFncs->engine_strdup( object->name );
                Info->info[SEG_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

                Info->info[SEG_INFO_SIZE_INDEX].name = EngFncs->engine_strdup( "Size" );
                Info->info[SEG_INFO_SIZE_INDEX].title = EngFncs->engine_strdup( _("Size") );
                Info->info[SEG_INFO_SIZE_INDEX].desc = EngFncs->engine_strdup( _("This is the size of the partition in sectors.") );
                Info->info[SEG_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[SEG_INFO_SIZE_INDEX].unit               = EVMS_Unit_Sectors;
                Info->info[SEG_INFO_SIZE_INDEX].value.ui64         = object->size;
                Info->info[SEG_INFO_SIZE_INDEX].format             = EVMS_Format_Normal;
                Info->info[SEG_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));
                Info->info[SEG_INFO_SIZE_INDEX].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

                Info->info[SEG_INFO_START_INDEX].name = EngFncs->engine_strdup( "Start" );
                Info->info[SEG_INFO_START_INDEX].title = EngFncs->engine_strdup( _("Start Logical Block Address") );
                Info->info[SEG_INFO_START_INDEX].desc = EngFncs->engine_strdup( _("This is the sector offset of the partition on the disk, i.e., the logical block address of the first sector of the partition.") );
                Info->info[SEG_INFO_START_INDEX].type               = EVMS_Type_Unsigned_Int64;
                Info->info[SEG_INFO_START_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_START_INDEX].value.ui64         = object->start;
                Info->info[SEG_INFO_START_INDEX].format             = EVMS_Format_Normal;
                Info->info[SEG_INFO_START_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_START_INDEX].group, 0, sizeof(group_info_t));


                Info->info[SEG_INFO_TYPE_INDEX].name = EngFncs->engine_strdup( "Type" );
                Info->info[SEG_INFO_TYPE_INDEX].title = EngFncs->engine_strdup( _("Type") );
                Info->info[SEG_INFO_TYPE_INDEX].desc = EngFncs->engine_strdup( _("Partition type - data, metadata, free.") );
                Info->info[SEG_INFO_TYPE_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                if ( object->data_type == DATA_TYPE ) {
                        Info->info[SEG_INFO_TYPE_INDEX].value.s = EngFncs->engine_strdup( _("data") );
                }
                else if ( object->data_type == META_DATA_TYPE ) {
                        Info->info[SEG_INFO_TYPE_INDEX].value.s = EngFncs->engine_strdup( _("metadata") );
                        Info->info[SEG_INFO_TYPE_INDEX].flags           = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
                }
                else {
                        Info->info[SEG_INFO_TYPE_INDEX].value.s = EngFncs->engine_strdup( _("unknown") );
                }

                *info = Info;

                rc = 0;
        }
        else {
                LOG_ERROR("unable to malloc memory for extended info array\n");
        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Returns BSD Disk Label (metadata) information
 */
int bsd_get_metadata_info(storage_object_t * object, extended_info_array_t **info)
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        LOGICALDISK             *ld;
        bsd_disklabel_t         *disk_label;
        char                     geometry_string[128];

        LOG_ENTRY();

        ld         = get_logical_disk( object );
        disk_label = get_bsd_disk_label( ld );

        REQUIRE(ld != NULL);
        REQUIRE(disk_label != NULL);

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( BSD_LABEL_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

                Info->count = BSD_LABEL_INFO_COUNT;

                sprintf(geometry_string,"C= %d   H= %d  S= %d",
                        DISK_TO_CPU32(disk_label->d_ncylinders),
                        DISK_TO_CPU32(disk_label->d_ntracks),
                        DISK_TO_CPU32(disk_label->d_nsectors));

                Info->info[BSD_INFO_GEO_INDEX].name = EngFncs->engine_strdup( "Geometry" );
                Info->info[BSD_INFO_GEO_INDEX].title = EngFncs->engine_strdup( _("Disk Geometry") );
                Info->info[BSD_INFO_GEO_INDEX].desc = EngFncs->engine_strdup( _("This is the disk geometry reported in the label.") );
                Info->info[BSD_INFO_GEO_INDEX].type               = EVMS_Type_String;
                Info->info[BSD_INFO_GEO_INDEX].unit               = EVMS_Unit_None;
                Info->info[BSD_INFO_GEO_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[BSD_INFO_GEO_INDEX].group, 0, sizeof(group_info_t));
                Info->info[BSD_INFO_GEO_INDEX].value.s = EngFncs->engine_strdup( geometry_string );

                Info->info[BSD_INFO_SECTORSIZE_INDEX].name = EngFncs->engine_strdup( "Sector Size" );
                Info->info[BSD_INFO_SECTORSIZE_INDEX].title = EngFncs->engine_strdup( _("Sector Size") );
                Info->info[BSD_INFO_SECTORSIZE_INDEX].desc = EngFncs->engine_strdup( _("Sector size reported by disk label.") );
                Info->info[BSD_INFO_SECTORSIZE_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[BSD_INFO_SECTORSIZE_INDEX].unit               = EVMS_Unit_None;
                Info->info[BSD_INFO_SECTORSIZE_INDEX].value.ui64         = DISK_TO_CPU32(disk_label->d_secsize);
                Info->info[BSD_INFO_SECTORSIZE_INDEX].format             = EVMS_Format_Normal;
                Info->info[BSD_INFO_SECTORSIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[BSD_INFO_SECTORSIZE_INDEX].group, 0, sizeof(group_info_t));

                Info->info[BSD_INFO_PCOUNT_INDEX].name = EngFncs->engine_strdup( "Partitions" );
                Info->info[BSD_INFO_PCOUNT_INDEX].title = EngFncs->engine_strdup( _("Partition Count") );
                Info->info[BSD_INFO_PCOUNT_INDEX].desc = EngFncs->engine_strdup( _("The number of partitions reported by the disk label.") );
                Info->info[BSD_INFO_PCOUNT_INDEX].type               = EVMS_Type_Unsigned_Int16;
                Info->info[BSD_INFO_PCOUNT_INDEX].unit               = EVMS_Unit_None;
                Info->info[BSD_INFO_PCOUNT_INDEX].value.ui64         = DISK_TO_CPU16(disk_label->d_npartitions);
                Info->info[BSD_INFO_PCOUNT_INDEX].format             = EVMS_Format_Normal;
                Info->info[BSD_INFO_PCOUNT_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[BSD_INFO_PCOUNT_INDEX].group, 0, sizeof(group_info_t));

                Info->info[BSD_INFO_BOOTSIZE_INDEX].name = EngFncs->engine_strdup( "Boot Size" );
                Info->info[BSD_INFO_BOOTSIZE_INDEX].title = EngFncs->engine_strdup( _("Boot Area Size") );
                Info->info[BSD_INFO_BOOTSIZE_INDEX].desc = EngFncs->engine_strdup( _("Size of boot area that is reported by the disk label.") );
                Info->info[BSD_INFO_BOOTSIZE_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[BSD_INFO_BOOTSIZE_INDEX].unit               = EVMS_Unit_None;
                Info->info[BSD_INFO_BOOTSIZE_INDEX].value.ui64         = DISK_TO_CPU32(disk_label->d_bbsize);
                Info->info[BSD_INFO_BOOTSIZE_INDEX].format             = EVMS_Format_Normal;
                Info->info[BSD_INFO_BOOTSIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[BSD_INFO_BOOTSIZE_INDEX].group, 0, sizeof(group_info_t));

                Info->info[BSD_INFO_SBSIZE_INDEX].name = EngFncs->engine_strdup( "SB Size" );
                Info->info[BSD_INFO_SBSIZE_INDEX].title = EngFncs->engine_strdup( _("Superblock Size") );
                Info->info[BSD_INFO_SBSIZE_INDEX].desc = EngFncs->engine_strdup( _("Size of file system super block that is reported by the disk label.") );
                Info->info[BSD_INFO_SBSIZE_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[BSD_INFO_SBSIZE_INDEX].unit               = EVMS_Unit_None;
                Info->info[BSD_INFO_SBSIZE_INDEX].value.ui64         = DISK_TO_CPU32(disk_label->d_sbsize);
                Info->info[BSD_INFO_SBSIZE_INDEX].format             = EVMS_Format_Normal;
                Info->info[BSD_INFO_SBSIZE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[BSD_INFO_SBSIZE_INDEX].group, 0, sizeof(group_info_t));

                *info = Info;

                rc = 0;

        }
        else {
                LOG_ERROR("unable to malloc memory for extended info array\n");
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Returns plugin specific information
 */
int bsd_get_plugin_info( char * descriptor_name, extended_info_array_t * * info )
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        char                     version_string[64];
        char                     required_engine_api_version_string[64];
        char                     required_plugin_api_version_string[64];


        LOG_ENTRY();

        // a measure of protection ...
        if (info == NULL) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (PLUGIN_INFO_COUNT*sizeof(extended_info_t))  );
        if (Info) {

                Info->count = PLUGIN_INFO_COUNT;

                sprintf(version_string, "%d.%d.%d",
                        MAJOR_VERSION,
                        MINOR_VERSION,
                        PATCH_LEVEL );

                sprintf(required_engine_api_version_string, "%d.%d.%d",
                        bsd_plugin->required_engine_api_version.major,
                        bsd_plugin->required_engine_api_version.minor,
                        bsd_plugin->required_engine_api_version.patchlevel );

                sprintf(required_plugin_api_version_string, "%d.%d.%d",
                        bsd_plugin->required_plugin_api_version.plugin.major,
                        bsd_plugin->required_plugin_api_version.plugin.minor,
                        bsd_plugin->required_plugin_api_version.plugin.patchlevel );

                Info->info[PLUGIN_INFO_SNAME_INDEX].name = EngFncs->engine_strdup( "ShortName" );
                Info->info[PLUGIN_INFO_SNAME_INDEX].title = EngFncs->engine_strdup( _("Short Name") );
                Info->info[PLUGIN_INFO_SNAME_INDEX].desc = EngFncs->engine_strdup( _("A short name given to this plug-in") );
                Info->info[PLUGIN_INFO_SNAME_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_SNAME_INDEX].unit               = EVMS_Unit_None;
                Info->info[PLUGIN_INFO_SNAME_INDEX].value.s = EngFncs->engine_strdup( bsd_plugin->short_name );
                Info->info[PLUGIN_INFO_SNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_SNAME_INDEX].group, 0, sizeof(group_info_t));

                Info->info[PLUGIN_INFO_LNAME_INDEX].name = EngFncs->engine_strdup( "LongName" );
                Info->info[PLUGIN_INFO_LNAME_INDEX].title = EngFncs->engine_strdup( _("Long Name") );
                Info->info[PLUGIN_INFO_LNAME_INDEX].desc = EngFncs->engine_strdup( _("A longer, more descriptive name for this plug-in") );
                Info->info[PLUGIN_INFO_LNAME_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_LNAME_INDEX].unit               = EVMS_Unit_None;
                Info->info[PLUGIN_INFO_LNAME_INDEX].value.s = EngFncs->engine_strdup( bsd_plugin->long_name );
                Info->info[PLUGIN_INFO_LNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_LNAME_INDEX].group, 0, sizeof(group_info_t));

                Info->info[PLUGIN_INFO_TYPE_INDEX].name = EngFncs->engine_strdup( "Type" );
                Info->info[PLUGIN_INFO_TYPE_INDEX].title = EngFncs->engine_strdup( _("Plug-in Type") );
                Info->info[PLUGIN_INFO_TYPE_INDEX].desc = EngFncs->engine_strdup( _("There are various types of plug-ins, each responsible for some kind of storage object or logical volume.") );
                Info->info[PLUGIN_INFO_TYPE_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                Info->info[PLUGIN_INFO_TYPE_INDEX].value.s = EngFncs->engine_strdup( _("Segment Manager") );
                Info->info[PLUGIN_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                Info->info[PLUGIN_INFO_VERSION_INDEX].name = EngFncs->engine_strdup( "Version" );
                Info->info[PLUGIN_INFO_VERSION_INDEX].title = EngFncs->engine_strdup( _("Plug-in Version") );
                Info->info[PLUGIN_INFO_VERSION_INDEX].desc = EngFncs->engine_strdup( _("This is the version number of the plugin.") );
                Info->info[PLUGIN_INFO_VERSION_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
                Info->info[PLUGIN_INFO_VERSION_INDEX].value.s = EngFncs->engine_strdup( version_string );
                Info->info[PLUGIN_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

                Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].name = EngFncs->engine_strdup( "Required Engine Services Version" );
                Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].title = EngFncs->engine_strdup( _("Required Engine Services Version") );
                Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].desc = EngFncs->engine_strdup( _("This is the version of the Engine services that this plugin requires. "
												" It will not run on older versions of the Engine services.") );
                Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].unit               = EVMS_Unit_None;
                Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].value.s = EngFncs->engine_strdup( required_engine_api_version_string );
                Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_REQENGSVCVERSION_INDEX].group, 0, sizeof(group_info_t));

                Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].name = EngFncs->engine_strdup( "Required Engine Plug-in API Version" );
                Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].title = EngFncs->engine_strdup( _("Required Engine Plug-in API Version") );
                Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].desc = EngFncs->engine_strdup( _("This is the version of the Engine plug-in API that this plugin requires.  "
												 "It will not run on older versions of the Engine plug-in API.") );
                Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].type               = EVMS_Type_String;
                Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].unit               = EVMS_Unit_None;
                Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].value.s = EngFncs->engine_strdup( required_plugin_api_version_string );
                Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[PLUGIN_INFO_REQPLUGAPIVERSION_INDEX].group, 0, sizeof(group_info_t));

                *info = Info;

                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}

