/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <stdio.h>
#include <string.h>
#include <ncurses.h>
#include <panel.h>
#include <glib.h>

#include "common.h"
#include "window.h"

/**
 *	get_center_coordinates - generate start x and y for a centered window
 *	@height: the number of rows in the window
 *	@width: the number of columns in the window
 *	@starty: address of where to return starting row coordinate
 *	@startx: address of where to return starting column coordinate
 *
 *	This routine calculates the start x and y coordinates to allow
 *	a window of the given height and width to be centered on the 
 *	screen.
 */
inline void get_center_coordinates(int height, int width, int *starty, int *startx)
{
	*starty = (getmaxy(stdscr) / 2) - (height / 2);
	*startx = (getmaxx(stdscr) / 2) - (width / 2);
}

/**
 *	print_centered - print a string centered in a window
 *	@win: the window id
 *	@starty: the row/line to print the string at
 *	@string: the string to print
 *
 * 	This routine prints a string centered starting at the given
 * 	line in the window.
 */
void print_centered(WINDOW *win, int starty, char *string)
{
 	int length, height, width, startx;

	getmaxyx(win, height, width);
	length = strlen(string);
	if (length >= width)
		startx = 0;
	else
		startx = (width - length) / 2;
	mvwaddnstr(win, starty, startx, string, length);
	wrefresh(win);
}

/**
 *	read_string - read a string from the keyboard within a specific window
 *	@win: the window id
 *	@starty: the row/line to place the cursor at
 *	@startx: the column to place the cursor at
 *	@string: the buffer to contain the string
 *	@max: the maximum characters that can be transferred to the string buffer
 *
 *	This routine accepts keyboard input from a user and places the contents
 *	of the characters into the supplied buffer. It returns OK if the read
 *	operation was successful.
 */
int read_string(WINDOW *win, int starty, int startx, char *string, int max)
{
	int rc;

	curs_set(1);
	echo();
	rc = mvwgetnstr(win, starty, startx, string, max);
	noecho();
	curs_set(0);

	return rc;
}

/**
 *	panel_getch - read a character for a given panel window only if topmost panel
 *	@panel: the panel/window to read from
 *
 *	This routine issues a wgetch() for the window associated with the
 *	given panel. If the wgetch() returns and the panel is not the topmost
 *	panel, we requeue the input and loop back to block on a wgetch(). 
 *	This wrapper routine helps give keyboard focus to the topmost window
 *	for the case where a second thread is displaying a popup dialog window.
 */
int panel_getch(PANEL *panel)
{
	int key = 0;
	gboolean retry = TRUE;

	while (retry) {
		key = wgetch(panel_window(panel));
		if (panel != panel_below((PANEL *)0)) {
			ungetch(key);
		} else {
			retry = FALSE;
		}
	}
	return key;
}

/**
 *	adjust_start_x_and_y - adjusts start x or/and y as necessary to fit on screen
 *	@height: the number of rows in the window
 *	@width: the number of columns in the window
 *	@starty: the row coordinate for the upper left of the window
 *	@startx: the column coordinate for the upper left of the window
 *
 *	This routine attempts to adjust the startx and starty if the bottom
 *	or right edge of the window look like they will be placed outside
 *	the viewable screen.
 */
void adjust_start_x_and_y(int height, int width, int *starty, int *startx)
{
	int x, y;

	x = *startx + width;
	y = *starty + height;

	if (x > getmaxx(stdscr)) {
		*startx -= (x - getmaxx(stdscr));
	}
	if (y > getmaxy(stdscr)) {
		*starty -= (y - getmaxy(stdscr));
	}
}

/**
 *	create_window - creates a new window on the screen
 *	@height: the number of rows in the window
 *	@width: the number of columns in the window
 *	@starty: the row coordinate for the upper left of the window
 *	@startx: the column coordinate for the upper left of the window
 *	@drawbox: non-zero value if window should be surrounded by a box
 *	@color: the background/foreground attributes for the window
 *
 *	This routine creates a new window and optionally draws a box
 *	around it.
 */
WINDOW *create_window(int height, int width, int starty, int startx, int drawbox, int color)
{
	WINDOW *win;

	adjust_start_x_and_y(height, width, &starty, &startx);
	win = newwin(height, width, starty, startx);
	if (drawbox != 0) {
		box(win, 0, 0);
	}
	keypad(win, TRUE);
	wbkgd(win, color);
	wrefresh(win);
	return win;
}

/**
 *	delete_window - deletes a window resource
 *	@win: the window id
 *
 *	This routine simply ends the use of a window and frees
 *	any resources associated with it.
 **/
inline void delete_window(WINDOW *win)
{
	delwin(win);
}

/**
 *	create_centered_window - create a window centered on the screen
 *	@height: the number of rows in the window
 *	@width: the number of columns in the window
 *	@drawbox: non-zero value if window should be surrounded by a box
 *	@color: the background/foreground color of the window
 *
 *	This routine essentially determines the start x and y coordinates
 *	to create a window of the given height and window as centered on
 *	the screen.
 */
WINDOW *create_centered_window(int height, int width, int drawbox, int color)
{
	int starty, startx;

	get_center_coordinates(height, width, &starty, &startx);
	return create_window(height, width, starty, startx, drawbox, color);
}

/**
 *	create_panel_window - creates a stackable window
 *	@height: the number of rows in the window
 *	@width: the number of columns in the window
 *	@starty: the row coordinate for the upper left of the window
 *	@startx: the column coordinate for the upper left of the window
 *	@drawbox: non-zero value if window should be surrounded by a box
 *	@color: the background/foreground color of the window
 *
 *	This routine creates a new window and optionally draws a box
 *	around it. The window is associated with a panel that allows
 *	the window to be stacked on top of others, hidden, moved, etc.
 */
PANEL *create_panel_window(int height, int width, int starty, int startx, int drawbox, int color)
{
	WINDOW *win;
	PANEL *panel;

	win = create_window(height, width, starty, startx, drawbox, color);
	panel = new_panel(win);

	return panel;
}

/**
 *	create_centered_panel_window - same as create_panel_window but auto-centered
 *	@height: the number of rows in the window
 *	@width: the number of columns in the window
 *	@drawbox: non-zero value if the window should be surrounded by a box
 *	@color: the background/foreground color of the window
 *
 *	This routine essentially determines the start x and y coordinates to
 *	create a centered panel window.
 */
PANEL *create_centered_panel_window(int height, int width, int drawbox, int color)
{
	int starty, startx;

	get_center_coordinates(height, width, &starty, &startx);
	return create_panel_window(height, width, starty, startx, drawbox, color);
}

/**
 *	show_panel_window - displays a panel window
 *	@panel: the panel window id
 *	@maketop: non-zero if the panel to display show be topmost
 *
 *	This routine displays a panel window. If optionally allows
 *	the panel to be displayed at the top of window stack.
 */
inline void show_panel_window(PANEL *panel, int maketop)
{
	show_panel(panel);
	if (maketop != 0) {
		top_panel(panel);
	}
	update_panels();
	doupdate();
}

/**
 *	hide_panel_window - hide a panel window from view
 *	@panel: the panel window id
 *
 *	This routine removes a window from view. The window still
 *	exists and can be redisplayed with a call to show_panel_window.
 */
inline void hide_panel_window(PANEL *panel)
{
	if (!panel_hidden(panel)) {
		hide_panel(panel);
		update_panels();
		doupdate();
	}
}

/**
 *	delete_panel_window - deletes a panel window from the screen
 *	@panel: the panel window id
 *
 *	This routine deletes the panel and the associated window
 *	from the screen.
 */
inline void delete_panel_window(PANEL *panel)
{
	delete_window(panel_window(panel));
	del_panel(panel);
	update_panels();
	doupdate();
}

/**
 *	create_popup_window - creates a window that allows wrapped text
 *	@height: the number of rows in the window
 *	@width: the number of columns in the window
 *	@starty: the row coordinate for the upper left of the window
 *	@startx: the column coordinate for the upper left of the window
 *	@color: the background/foreground color of the window
 *
 *	This routine create a window with a border that contains a subwindow
 *	that allows text to be wrapped within the subwindow without touching
 *	the window border.
 */
PANEL *create_popup_window(int height, int width, int starty, int startx, int color)
{
	PANEL *parent;
	PANEL *child;
	
	parent = create_panel_window(height, width, starty, startx, TRUE, color);
	child = create_panel_window(height - 2, width - 2, starty + 1, startx + 1, FALSE, color);
	set_panel_userptr(child, parent);

	return child;
}

/**
 *	show_popup_window - display a popup window (border and center panels)
 *	@panel: the (inner) panel window id
 *
 *	This routine places to the top of displayed set of windows, the border
 *	panel for a popup window along with the inner center panel.
 */
void show_popup_window(PANEL *panel)
{
	PANEL *parent;

	parent = (PANEL *)panel_userptr(panel);

	show_panel_window(parent, TRUE);
	show_panel_window(panel, TRUE);
}

/**
 *	hide_popup_window - hide a popup window (border and center panels)
 *	@panel: the (inner) panel window id
 *
 *	This routine hides the two panel windows that make a bordered
 *	popup window. The outer panel containing the window border and
 *	the inner panel containing the window contents.
 */
void hide_popup_window(PANEL *panel)
{
	PANEL *parent;

	parent = (PANEL *)panel_userptr(panel);

	hide_panel_window(panel);
	hide_panel_window(parent);
}

/**
 *	delete_popup_window - deletes the text and border windows
 *	panel: the panel window id
 *
 *	This routine deletes the subwindow used for text and the outer
 *	parent window for the border.
 */
void delete_popup_window(PANEL *panel)
{
	PANEL *parent;

	parent = (PANEL *)panel_userptr(panel);

	hide_popup_window(panel);
	delete_panel_window(panel);
	delete_panel_window(parent);
}

/**
 *	create_centered_popup_window - same as create_popup_window but auto-centered
 *	@height: the number of rows in the window
 *	@width: the number of columns in the window
 *	@color: the background/foreground color of the window
 *	
 *	This routine essentially determines the start x and y coordinates to
 *	allow creating a popup window centered on the screen.
 */
PANEL *create_centered_popup_window(int height, int width, int color)
{
	int starty, startx;

	get_center_coordinates(height, width, &starty, &startx);
	return create_popup_window(height, width, starty, startx, color);
}

/**
 *	init_window_system - initialize the ncurses library
 *	@void
 *
 *	This routine calls the various ncurses initialization routines.
 */
void init_window_system(void)
{
	initscr();
	curs_set(0);
	start_color();
	flushinp();
	nonl();
	noecho();
	keypad(stdscr, TRUE);

	init_pair(STD_COLOR, COLOR_WHITE, COLOR_BLUE);
	init_pair(MENU_COLOR, COLOR_BLACK, COLOR_WHITE);
	init_pair(ACCEL_COLOR, COLOR_RED, COLOR_WHITE);
	init_pair(MENUSEL_COLOR, COLOR_WHITE, COLOR_BLACK);
	init_pair(FOCUS_COLOR, COLOR_BLACK, COLOR_WHITE);
	init_pair(LISTSEL_COLOR, COLOR_CYAN, COLOR_BLUE);
	init_pair(SELECTED_ACCEL_COLOR, COLOR_RED, COLOR_BLACK);

	/*
	 * Remap stderr to /dev/null so we get no accidental output
	 */
	freopen("/dev/null", "w", stderr);
}

/**
 *	shutdown_window_system - stops the use of the ncurses library
 *	@void
 *
 *	This routines ends the use of the ncurses library and frees
 *	associated resources. Prior to doing so, it clears the screen.
 */
void shutdown_window_system(void)
{
	clear();
	endwin();
}

