/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: dm-targets.c
 *
 * Implementation of the routines specific to each Device-Mapper target.
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <mntent.h>
#include <pthread.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "fullengine.h"
#include "engine.h"
#include "memman.h"
#include "dm.h"

/*
 * String allocation routines. To prevent needing to allocate memory while
 * a device is suspended, we keep a list of all the string buffers that have
 * been allocated by the DM services, and reuse these buffers whenever
 * possible.
 */

typedef struct dm_string {
	struct dm_string *next;
	char *string;
	unsigned long len;
	int in_use;
} dm_string_t;

static dm_string_t *dm_string_list = NULL;
static pthread_mutex_t dm_string_mutex = PTHREAD_MUTEX_INITIALIZER;

/**
 * get_string
 *
 * See if an appropriately-sized buffer is available for use. If not, allocate
 * a new one. Each buffer on the list will be at least 100 bytes.
 **/
static char *get_string(unsigned long len)
{
	dm_string_t *entry;
	char *string = NULL;

	LOG_PROC_ENTRY();

	len = (len < 100) ? 100 : len;

	pthread_mutex_lock(&dm_string_mutex);
	for (entry = dm_string_list; entry; entry = entry->next) {
		if (!entry->in_use && len <= entry->len) {
			entry->in_use = TRUE;
			string = entry->string;
			memset(string, 0, entry->len);
			break;
		}
	}
	pthread_mutex_unlock(&dm_string_mutex);

	if (!string) {
		entry = engine_alloc(sizeof(*entry));
		if (entry) {
			string = engine_alloc(len);
			if (string) {
				entry->string = string;
				entry->len = len;
				entry->in_use = TRUE;
				pthread_mutex_lock(&dm_string_mutex);
				entry->next = dm_string_list;
				dm_string_list = entry;
				pthread_mutex_unlock(&dm_string_mutex);
			} else {
				engine_free(entry);
			}
		}
	}

	LOG_PROC_EXIT_PTR(string);
	return string;
}

/**
 * put_string
 *
 * Release a string buffer when it's no longer in use.
 **/
void put_string(char *string)
{
	dm_string_t *entry;

	LOG_PROC_ENTRY();

	pthread_mutex_lock(&dm_string_mutex);
	for (entry = dm_string_list; entry; entry = entry->next) {
		if (entry->string == string) {
			entry->in_use = FALSE;
			break;
		}
	}
	pthread_mutex_unlock(&dm_string_mutex);

	LOG_PROC_EXIT_VOID();
}

/**
 * next_token
 *
 * Return a pointer to the next white-space-delimited token in the specified
 * string.
 * - Skip any leading whitespace.
 * - Skip one token's worth of non-whitespace characters.
 * - Skip any trailing whitespace.
 *
 * If it encounters a NULL character before finding the next token, it will
 * return a pointer to the NULL character.
 **/
static char *next_token(char *string)
{
	char *str = string;
	LOG_PROC_ENTRY();
	while (isspace(*str)) { str++; }
	while (!isspace(*str) && *str != '\0') { str++; }
	while (isspace(*str)) { str++; }
	LOG_PROC_EXIT_PTR(str);
	return str;
}

/**
 * get_sysfs_mountpoint
 *
 * Return the name of the directory where sysfs is mounted, or NULL if it cannot
 * be found. The caller needs to engine_free() the returned pointer when
 * finished using it.
 **/
static char *get_sysfs_mountpoint(void)
{
	FILE *mount_records;
	struct mntent *mount_entry;
	char *mount_name = NULL;

	LOG_PROC_ENTRY();

	mount_records = setmntent(MOUNTED, "r");
	if (!mount_records) {
		mount_records = setmntent("/proc/mounts", "r");
	}
	if (!mount_records) {
		LOG_ERROR("Could not get list of mounted devices.\n");
		goto out;
	}

	while ((mount_entry = getmntent(mount_records))) {
		if (strcmp(mount_entry->mnt_type, "sysfs") == 0) {
			mount_name = engine_strdup(mount_entry->mnt_dir);
			break;
		}
	}

	endmntent(mount_records);

out:
	if (!mount_name) {
		LOG_WARNING("Sysfs filesystem not mounted.\n");
	}
	LOG_PROC_EXIT_PTR(mount_name);
	return mount_name;
}

/**
 * get_dev_num_for_name
 *
 * Use sysfs to determine the major:minor for the given device name.
 **/
static int get_dev_num_for_name(char *device_name,
				int32_t *major,
				int32_t *minor)
{
	char *sysfs_dir;
	char filename[EVMS_NAME_SIZE];
	FILE *dev;
	int rc;

	LOG_PROC_ENTRY();

	sysfs_dir = get_sysfs_mountpoint();
	if (!sysfs_dir) {
		rc = ENOENT;
		goto out;
	}

	snprintf(filename, EVMS_NAME_SIZE, "%s/block/%s/dev",
		 sysfs_dir, device_name);
	dev = fopen(filename, "r");
	if (!dev) {
		rc = errno ;
		LOG_ERROR("Unable to open %s: %s\n", filename, strerror(rc));
		goto out;
	}

	rc = fscanf(dev, "%u:%u", major, minor);
	if (rc != 2) {
		rc = fscanf(dev, "%2x%2x", major, minor);
	}

	rc = (rc != 2) ? EINVAL : 0;

	fclose(dev);

out:
	engine_free(sysfs_dir);
	if (!rc) {
		LOG_DEBUG("%s has device-number %x:%x\n",
			  device_name, *major, *minor);
	}
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * translate_device
 *
 * Parse a string representing a device. This string can be one of the
 * following three forms:
 *   %x:%x
 *   name(%d,%d)
 *   name
 * Determine the major:minor of the device, and reset the string pointer to
 * point at the token following the device string.
 **/
static int translate_device(char **string,
			    int32_t *major,
			    int32_t *minor)
{
	char name[EVMS_NAME_SIZE];
	int rc;

	LOG_PROC_ENTRY();

	switch (dm_get_version()) {
	case 3:
		rc = sscanf(*string, "%x:%x", major, minor);
		break;
	case 4:
		rc = sscanf(*string, "%d:%d", major, minor);
		break;
	default:
		rc = EINVAL;
	}
	if (rc == 2) {
		rc = 0;
		goto out;
	}

	rc = sscanf(*string, "%*[^(](%d,%d)", major, minor);
	if (rc == 2) {
		rc = 0;
		goto out;
	}

	rc = sscanf(*string, "%s", name);
	if (rc == 1) {
		rc = get_dev_num_for_name(name, major, minor);
	} else {
		rc = EINVAL;
	}

out:
	if (!rc) {
		*string = next_token(*string);
	} else {
		LOG_ERROR("Invalid device string: %s\n", *string);
	}
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * deallocate_target_type_params
 *
 * Walk a list of targets and deallocate the memory for the params string.
 **/
void deallocate_target_type_params(dm_target_t *target_list)
{
	dm_target_t *target;

	LOG_PROC_ENTRY();
	for (target = target_list; target; target = target->next) {
		if (target->params) {
			put_string(target->params);
			target->params = NULL;
		}
	}
	LOG_PROC_EXIT_VOID();
}

/**
 * build_target_type_params
 *
 * @target_list: List of targets to process.
 *
 * Walk a list of targets and build the appropriate ASCII string in the
 * params fields.
 **/
int build_target_type_params(dm_target_t *target_list)
{
	dm_target_t *target;
	int rc = 0;

	LOG_PROC_ENTRY();

	for (target = target_list; target; target = target->next) {
		rc = dm_target_type_info[target->type].build_params(target);
		if (rc) {
			LOG_ERROR("Error allocating memory for target parameter string.\n");
			LOG_ERROR("   Target Type: %d, Start: %"PRIu64", Length: %"PRIu64"\n",
				  target->type, target->start, target->length);
			deallocate_target_type_params(target_list);
			break;
		} else {
			LOG_DEBUG("Built %s parameter string: %s\n",
				  dm_target_type_info[target->type].name,
				  target->params);
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * load_dm_target_module
 *
 * Attempt to load the kernel module for the specified DM target.
 **/
static int load_dm_target_module(dm_target_type type)
{
	char *argv[3], buffer[64];
	int status, rc = 0;
	pid_t pid;

	LOG_PROC_ENTRY();

	snprintf(buffer, 64, "dm-%s", dm_target_type_info[type].name);

	argv[0] = "modprobe";
	argv[1] = buffer;
	argv[2] = NULL;

        pid = fork();

	switch (pid) {
	case -1:
		rc = errno;
		LOG_WARNING("fork() to run \"%s %s\" returned error %d: %s\n",
			    argv[0], argv[1], rc, strerror(rc));
		break;

	case 0:
		/* Child process */
		execvp(argv[0], argv);

		/* Should not get here unless execvp() fails. */
		rc = errno;
		LOG_WARNING("execvp() to run \"%s %s\" returned error %d: %s\n",
			    argv[0], argv[1], rc, strerror(rc));

		/* exit() kills the GUI.  Use _exit() instead. */
		_exit(rc);
		break;

	default:
		/* Parent process */
		waitpid(pid, &status, 0);
	}

	if (!rc) {
		if (WIFSIGNALED(status)) {
			LOG_WARNING("\"%s %s\" was terminated by signal %s\n",
				    argv[0], argv[1],
				    sys_siglist[WTERMSIG(status)]);
			rc = EINTR;
		} else {
			rc = WEXITSTATUS(status);
			LOG_DEBUG("\"%s %s\" exited with error code %d: %s\n",
				  argv[0], argv[1], rc, strerror(rc));
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * get_module_version
 *
 * Get the list of modules and versions from DM, find the desired target-type,
 * and return the version of the currently loaded module.
 **/
static int get_module_version(dm_target_type type,
			      evms_version_t *version)
{
	dm_module_list_t *module_list, *module;
	int rc, module_loaded = FALSE;

	LOG_PROC_ENTRY();

start:
	rc = dm_get_modules(&module_list);
	if (rc) {
		goto out;
	}

	for (module = module_list; module; module = module->next) {
		rc = strncmp(dm_target_type_info[type].name,
			     module->name, EVMS_NAME_SIZE);
		if (!rc) {
			break;
		}
	}

	if (module) {
		*version = module->version;
		rc = 0;
	} else if (!module_loaded) {
		module_loaded = TRUE;
		rc = load_dm_target_module(type);
		if (!rc) {
			goto start;
		}
	} else {
		rc = ENOENT;
	}

	dm_deallocate_module_list(module_list);

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/**************************
 * Generic target routines.
 *
 * Some DM target types don't need a full routine for a particular target API.
 * These generic routines can be used when we just want the default behavior
 * (which usually is just to do nothing).
 */

/**
 * generic_build_params
 *
 * Some target types have no constructor string. Simply allocate an
 * empty string.
 **/
static int generic_build_params(dm_target_t *target)
{
	int rc = ENOMEM;

	LOG_PROC_ENTRY();

	target->params = get_string(1);
	if (target->params) {
		*target->params = '\0';
		rc = 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * generic_translate_params
 *
 * Some target types have no table string to translate.
 **/
static int generic_translate_params(dm_target_t *target)
{
	LOG_PROC_ENTRY();
	LOG_PROC_EXIT_INT(0);
	return 0;
}

/**
 * generic_pretranslate_params
 *
 * Some target types don't have multiple devices or groups.
 **/
static int generic_pretranslate_params(char *params,
				       unsigned int *num_devs,
				       unsigned int *num_groups)
{
	LOG_PROC_ENTRY();
	*num_devs = 0;
	*num_groups = 0;
	LOG_PROC_EXIT_INT(0);
	return 0;
}

/**
 * generic_allocate_target
 *
 * Some target types don't have any additional structures to attach to their
 * target structure.
 **/
static int generic_allocate_target(dm_target_t *target,
				   u_int32_t num_devs, u_int32_t num_groups)
{
	LOG_PROC_ENTRY();
	LOG_PROC_EXIT_INT(0);
	return 0;
}

/**
 * generic_deallocate_target
 *
 * Some target types don't have any additional structures attached to their
 * target structure.
 **/
static void generic_deallocate_target(dm_target_t *target)
{
	LOG_PROC_ENTRY();
	LOG_PROC_EXIT_VOID();
}


/*************************
 * Linear target routines.
 */

/**
 * linear_build_params
 *
 * Generate an ASCII constructor string for a linear target.
 * A linear string has the form:
 *   <major>:<minor> <start_lba>
 **/
static int linear_build_params(dm_target_t *target)
{
	dm_device_t *dev = target->data.linear;
	char *format = (dm_get_version() == 3) ?
			"%x:%x %"PRIu64 : "%u:%u %"PRIu64;
	int rc = ENOMEM;

	LOG_PROC_ENTRY();

	target->params = get_string(40);
	if (target->params) {
		snprintf(target->params, 40, format,
			 dev->major, dev->minor, dev->start);
		rc = 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * linear_translate_params
 *
 * Fill in a linear target structure based on an ASCII table string of the form:
 *   <device> <start_lba>
 **/
static int linear_translate_params(dm_target_t *target)
{
	dm_device_t *dev = target->data.linear;
	char *params = target->params;
	int rc;

	LOG_PROC_ENTRY();

	rc = translate_device(&params, &dev->major, &dev->minor);
	if (!rc) {
		rc = sscanf(params, "%"SCNu64, &dev->start);
		rc = (rc != 1) ? EINVAL : 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*************************
 * Stripe target routines.
 */

/**
 * stripe_build_params
 *
 * Generate an ASCII constructor string for a stripe target.
 * A stripe string has the form:
 *   <stripes> <chunk_size> [<major>:<minor> <start_lba>]+
 **/
static int stripe_build_params(dm_target_t *target)
{
	dm_target_stripe_t *stripe = target->data.stripe;
	char *format = (dm_get_version() == 3) ?
			" %x:%x %"PRIu64 : " %u:%u %"PRIu64;
	int max_len = 25 + stripe->num_stripes * 40;
	int i, sz = 0, rc = ENOMEM;

	LOG_PROC_ENTRY();

	target->params = get_string(max_len);
	if (target->params) {
		sz += snprintf(target->params + sz, max_len - sz, "%u %u",
			       stripe->num_stripes, stripe->chunk_size);
		for (i = 0; i < stripe->num_stripes; i++) {
			sz += snprintf(target->params + sz, max_len - sz,
				       format,
				       stripe->devices[i].major,
				       stripe->devices[i].minor,
				       stripe->devices[i].start);
		}
		rc = 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * stripe_translate_params
 *
 * Fill in a stripe target structure based on an ASCII table string of the form:
 *   <stripes> <chunk_size> [<device> <start_lba>]+
 **/
static int stripe_translate_params(dm_target_t *target)
{
	dm_target_stripe_t *stripe = target->data.stripe;
	char *params = target->params;
	int i, rc;

	LOG_PROC_ENTRY();

	rc = sscanf(params, "%u %u", &stripe->num_stripes, &stripe->chunk_size);
	if (rc != 2) {
		rc = EINVAL;
		goto out;
	}

	params = next_token(params);
	params = next_token(params);

	for (i = 0; i < stripe->num_stripes; i++) {
		rc = translate_device(&params, &(stripe->devices[i].major),
				      &(stripe->devices[i].minor));
		if (rc) {
			goto out;
		}

		rc = sscanf(params, "%"SCNu64, &(stripe->devices[i].start));
		if (rc != 1) {
			rc = EINVAL;
			goto out;
		}

		params = next_token(params);
	}

	rc = 0;
out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * stripe_pretranslate_params
 *
 * Parse an ASCII table string to determine how many stripe devices are
 * represented. Same format as used for stripe_translate_params().
 **/
static int stripe_pretranslate_params(char *params, u_int32_t *num_devs,
				      u_int32_t *num_groups)
{
	int rc;

	LOG_PROC_ENTRY();

	rc = sscanf(params, "%u", num_devs);
	rc = (rc != 1) ? EINVAL : 0;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * stripe_allocate_target
 *
 * Allocate an array of devices to attach to the stripe target structure.
 **/
static int stripe_allocate_target(dm_target_t *target, u_int32_t num_devs,
				  u_int32_t num_groups)
{
	dm_target_stripe_t *stripe = target->data.stripe;
	int rc = 0;

	LOG_PROC_ENTRY();

	if (num_devs < 1) {
		rc = EINVAL;
		goto out;
	}

	stripe->num_stripes = num_devs;
	stripe->devices = engine_alloc(num_devs * sizeof(dm_device_t));
	if (!stripe->devices) {
		rc = ENOMEM;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * stripe_deallocate_target
 *
 * Deallocate the array of devices attached to the stripe target structure.
 **/
static void stripe_deallocate_target(dm_target_t *target)
{
	dm_target_stripe_t *stripe = target->data.stripe;

	LOG_PROC_ENTRY();

	engine_free(stripe->devices);

	LOG_PROC_EXIT_VOID();
}


/*************************
 * Mirror target routines.
 */

/**
 * mirror_build_params
 *
 * Generate an ASCII constructor string for a mirror target.
 * A mirror string has the form:
 *   <log_type> <num_log_params> [<log_params>]* <num_mirrors> [<major>:<minor> <start_lba>]{2,}
 *
 * Currently, log_type will always be "core", num_log_params is 1, and
 * log_params is chunk-size (in sectors).
 **/
static int mirror_build_params(dm_target_t *target)
{
	dm_target_mirror_t *mirror = target->data.mirror;
	char *format = (dm_get_version() == 3) ?
			" %x:%x %"PRIu64 : " %u:%u %"PRIu64;
	int max_len = 30 + mirror->num_mirrors * 40;
	int i, sz = 0, rc = ENOMEM;

	LOG_PROC_ENTRY();

	target->params = get_string(max_len);
	if (target->params) {
		sz += snprintf(target->params + sz, max_len - sz,
			       "core 1 %u %u",
			       mirror->chunk_size, mirror->num_mirrors);
		for (i = 0; i < mirror->num_mirrors; i++) {
			sz += snprintf(target->params + sz, max_len - sz,
				       format,
				       mirror->devices[i].major,
				       mirror->devices[i].minor,
				       mirror->devices[i].start);
		}
		rc = 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * mirror_translate_params
 *
 * Fill in a mirror target structure based on an ASCII table string of the form:
 *   <log_type> <num_log_params> [<log_params>]* <num_mirrors> [<major>:<minor> <start_lba>]{2,}
 *
 * Currently, log_type will always be "core", num_log_params is 1, and
 * log_params is chunk-size (in sectors).
 **/
static int mirror_translate_params(dm_target_t *target)
{
	dm_target_mirror_t *mirror = target->data.mirror;
	char *params = target->params;
	int i, rc;

	LOG_PROC_ENTRY();

	/* Skip "core 1" at the start of the string. */
	params = next_token(params);
	params = next_token(params);

	rc = sscanf(params, "%u %u", &mirror->chunk_size, &mirror->num_mirrors);
	if (rc != 2) {
		rc = EINVAL;
		goto out;
	}

	params = next_token(params);
	params = next_token(params);

	for (i = 0; i < mirror->num_mirrors; i++) {
		rc = translate_device(&params, &(mirror->devices[i].major),
				      &(mirror->devices[i].minor));
		if (rc) {
			goto out;
		}

		rc = sscanf(params, "%"SCNu64, &(mirror->devices[i].start));
		if (rc != 1) {
			rc = EINVAL;
			goto out;
		}

		params = next_token(params);
	}

	rc = 0;
out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * mirror_pretranslate_params
 *
 * Parse an ASCII table string to determine how many mirror devices are
 * represented. Same format as used for mirror_translate_params().
 **/
static int mirror_pretranslate_params(char *params, u_int32_t *num_devs,
				      u_int32_t *num_groups)
{
	int rc;

	LOG_PROC_ENTRY();

	rc = sscanf(params, "%*s %*d %*u %u", num_devs);
	rc = (rc != 1) ? EINVAL : 0;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * mirror_allocate_target
 *
 * Allocate an array of devices to attach to the mirror target structure.
 **/
static int mirror_allocate_target(dm_target_t *target,
				  u_int32_t num_devs, u_int32_t num_groups)
{
	dm_target_mirror_t *mirror = target->data.mirror;
	int rc = 0;

	LOG_PROC_ENTRY();

	if (num_devs < 1) {
		rc = EINVAL;
		goto out;
	}

	mirror->num_mirrors = num_devs;
	mirror->devices = engine_alloc(num_devs * sizeof(dm_device_t));
	if (!mirror->devices) {
		rc = ENOMEM;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * mirror_deallocate_target
 *
 * Deallocate the array of devices attached to the mirror target structure.
 **/
static void mirror_deallocate_target(dm_target_t *target)
{
	dm_target_mirror_t *mirror = target->data.mirror;

	LOG_PROC_ENTRY();

	engine_free(mirror->devices);

	LOG_PROC_EXIT_VOID();
}


/***************************
 * Snapshot target routines.
 */

/**
 * snapshot_build_params
 *
 * Generate an ASCII constructor string for a snapshot target.
 * A snapshot string has the form:
 *   <org_major>:<org_minor> <snap_major>:<snap_minor> <p|n> <chunk_size> <org_parent_major>:<org_parent_minor>
 **/
static int snapshot_build_params(dm_target_t *target)
{
	dm_target_snapshot_t *snapshot = target->data.snapshot;
	char *format = (dm_get_version() == 3) ?
			"%x:%x %x:%x %c %u %x:%x" :
			"%u:%u %u:%u %c %u %u:%u";
	int rc = ENOMEM;

	LOG_PROC_ENTRY();

	target->params = get_string(50);
	if (target->params) {
		snprintf(target->params, 50, format,
			 snapshot->origin.major, snapshot->origin.minor,
			 snapshot->snapshot.major, snapshot->snapshot.minor,
			 (snapshot->persistent) ? 'p' : 'n', snapshot->chunk_size,
			 snapshot->origin_parent.major,
			 snapshot->origin_parent.minor);
		rc = 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * snapshot_translate_params
 *
 * Fill in a snapshot target structure based on an ASCII string of the form:
 *   <org_device> <snap_device> <p|n> <chunk_size>
 **/
static int snapshot_translate_params(dm_target_t *target)
{
	dm_target_snapshot_t *snapshot = target->data.snapshot;
	char *params = target->params;
	char persistent;
	int rc;

	LOG_PROC_ENTRY();

	rc = translate_device(&params, &snapshot->origin.major,
			      &snapshot->origin.minor);
	if (rc) {
		goto out;
	}

	rc = translate_device(&params, &snapshot->snapshot.major,
			      &snapshot->snapshot.minor);
	if (rc) {
		goto out;
	}

	rc = sscanf(params, "%c %u", &persistent, &snapshot->chunk_size);
	rc = (rc != 2) ? EINVAL : 0;
	if (persistent == 'p' || persistent == 'P') {
		snapshot->persistent = 1;
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/**********************************
 * Snapshot-origin target routines.
 */

/**
 * origin_build_params
 *
 * Generate an ASCII constructor string for a snapshot origin target.
 * A snapshot origin string has the form:
 *   <org_major>:<org_minor>
 **/
static int origin_build_params(dm_target_t *target)
{
	dm_device_t *org = target->data.linear;
	char *format = (dm_get_version() == 3) ? "%x:%x" : "%u:%u";
	int rc = ENOMEM;

	LOG_PROC_ENTRY();

	target->params = get_string(25);
	if (target->params) {
		snprintf(target->params, 25, format, org->major, org->minor);
		rc = 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * origin_translate_params
 *
 * Fill in a snapshot origin target structure based on an ASCII table string
 * of the form:
 *   <org_device>
 **/
static int origin_translate_params(dm_target_t *target)
{
	dm_device_t *org = target->data.linear;
	char *params = target->params;
	int rc;

	LOG_PROC_ENTRY();

	rc = translate_device(&params, &org->major, &org->minor);

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/****************************
 * Multipath target routines.
 */

/*
 * There are now two versions of the parameter string for the multipath target.
 * Version 1.0.3 and earlier of the module use a string of the form:
 *
 *   <num_groups> [ GROUP_SPEC ]+
 *
 *   GROUP_SPEC
 *     <selector_name> <num_paths> <num_path_args> [ PATH_SPEC ]+
 *
 *     PATH_SPEC
 *       <major>:<minor> [ <path_arg> ]*
 *
 * Versions of the module above 1.0.3 use a string of the form:
 *
 *   [ FEATURE_SPEC ] [ HW_HANDLER_SPEC ] [ GROUPING_SPEC ]
 *
 *   FEATURE_SPEC
 *     <num_feature_args> [ <feature_arg> ]*
 *
 *   HW_HANDLER_SPEC
 *     <num_hw_handler_args> [ <hw_handler_name> [ <hw_handler_arg> ]* ]*
 *
 *   GROUPING_SPEC
 *     <num_groups> <initial_group> [ GROUP_SPEC ]+
 *
 *     GROUP_SPEC
 *       [ SELECTOR_SPEC ] <num_paths> <num_per_path_args> [ PATH_SPEC ]+
 *
 *       SELECTOR_SPEC
 *         <selector_name> <num_selector_args> [ <selector_arg> ]*
 *
 *       PATH_SPEC
 *         <major>:<minor> [ <path_arg> ]*
 */

/**
 * multipath_build_params
 *
 * Generate an ASCII constructor string for a multipath target.
 **/

static int multipath_build_params_v1(dm_target_t *target)
{
	dm_target_multipath_t *mp = target->data.multipath;
	char *format = (dm_get_version() == 3) ? " %x:%x" : " %u:%u";
	int i, j, sz = 0, maxlen, rc = ENOMEM;

	LOG_PROC_ENTRY();

	maxlen = 11 + mp->num_groups * 66;
	for (i = 0; i < mp->num_groups; i++) {
		maxlen += mp->group[i].num_paths *
			  (22 + mp->group[i].num_path_args * 20);
	}

	target->params = get_string(maxlen);
	if (target->params) {
		rc = 0;

		sz += snprintf(target->params + sz, maxlen - sz,
			       "%u", mp->num_groups);

		for (i = 0; i < mp->num_groups; i++) {
			sz += snprintf(target->params + sz, maxlen - sz,
				       " %s %u %u",
				       mp->group[i].selector,
				       mp->group[i].num_paths,
				       mp->group[i].num_path_args);

			for (j = 0; j < mp->group[i].num_paths; j++) {
				sz += snprintf(target->params + sz, maxlen - sz,
					       format,
					       mp->group[i].path[j].device.major,
					       mp->group[i].path[j].device.minor);

				if (mp->group[i].num_path_args) {
					sz += snprintf(target->params + sz,
						       maxlen - sz, " %s",
						       mp->group[i].path[j].path_args);
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

static int multipath_build_params_v2(dm_target_t *target)
{
	dm_target_multipath_t *mp = target->data.multipath;
	char *format = (dm_get_version() == 3) ? " %x:%x" : " %u:%u";
	int i, j, sz = 0, maxlen, rc = ENOMEM;

	LOG_PROC_ENTRY();

	maxlen = 17 + mp->num_groups * 57;
	for (i = 0; i < mp->num_groups; i++) {
		maxlen += mp->group[i].num_paths *
			  (22 + mp->group[i].num_path_args * 20);
	}

	target->params = get_string(maxlen);
	if (target->params) {
		rc = 0;

		sz += snprintf(target->params + sz, maxlen - sz,
			       "0 0 %u 1", mp->num_groups);

		for (i = 0; i < mp->num_groups; i++) {
			sz += snprintf(target->params + sz, maxlen - sz,
				       " %s 0 %u %u",
				       mp->group[i].selector,
				       mp->group[i].num_paths,
				       mp->group[i].num_path_args);

			for (j = 0; j < mp->group[i].num_paths; j++) {
				sz += snprintf(target->params + sz, maxlen - sz,
					       format,
					       mp->group[i].path[j].device.major,
					       mp->group[i].path[j].device.minor);

				if (mp->group[i].num_path_args) {
					sz += snprintf(target->params + sz,
						       maxlen - sz, " %s",
						       mp->group[i].path[j].path_args);
				}
			}
		}
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

static int multipath_build_params(dm_target_t *target)
{
	evms_version_t version;
	int rc;

	LOG_PROC_ENTRY();

	rc = get_module_version(DM_TARGET_MULTIPATH, &version);
	if (rc) {
		goto out;
	}

	if (version.major == 1 &&
	    version.minor == 0 &&
	    version.patchlevel <= 3) {
		rc = multipath_build_params_v1(target);
	} else {
		rc = multipath_build_params_v2(target);
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * multipath_translate_params
 *
 * Fill in a multipath target structure based on the ASCII table string.
 **/

static int multipath_translate_params_v1(dm_target_t *target)
{
	dm_target_multipath_t *mp = target->data.multipath;
	dm_path_t *current_path = mp->group[0].path;
	char *params = target->params;
	int i, j, k, sz, rc;

	LOG_PROC_ENTRY();

	rc = sscanf(params, "%u", &mp->num_groups);
	if (rc != 1) {
		rc = EINVAL;
		goto out;
	}

	params = next_token(params);

	for (i = 0; i < mp->num_groups; i++) {
		rc = sscanf(params, "%s %u %u", mp->group[i].selector,
			    &mp->group[i].num_paths, &mp->group[i].num_path_args);
		if (rc != 3) {
			rc = EINVAL;
			goto out;
		}

		params = next_token(params);
		params = next_token(params);
		params = next_token(params);

		mp->group[i].path = current_path;
		current_path += mp->group[i].num_paths;

		for (j = 0; j < mp->group[i].num_paths; j++) {
			rc = translate_device(&params,
					      &mp->group[i].path[j].device.major,
					      &mp->group[i].path[j].device.minor);
			if (rc) {
				goto out;
			}

			for (k = 0, sz = 0; k < mp->group[i].num_path_args; k++) {
				rc = sscanf(params, "%s",
					    mp->group[i].path[j].path_args + sz);
				if (rc != 1) {
					rc = EINVAL;
					goto out;
				}

				strncat(mp->group[i].path[j].path_args,
					" ", DM_PATH_ARGS_SIZE);
				sz = strlen(mp->group[i].path[j].path_args);
				params = next_token(params);
			}
		}
	}

	rc = 0;

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

static int multipath_translate_params_v2(dm_target_t *target)
{
	dm_target_multipath_t *mp = target->data.multipath;
	dm_path_t *current_path = mp->group[0].path;
	char *params = target->params;
	u_int32_t num, i, j, k;
	int sz, rc;

	LOG_PROC_ENTRY();

	/* We don't care about the "feature" arguments. */
	rc = sscanf(params, "%u", &num);
	if (rc != 1) {
		rc = EINVAL;
		goto out;
	}
	params = next_token(params);

	for (i = 0; i < num; i++) {
		params = next_token(params);
	}

	/* We don't care about the "hw-handler" arguments. */
	rc = sscanf(params, "%u", &num);
	if (rc != 1) {
		rc = EINVAL;
		goto out;
	}
	params = next_token(params);

	for (i = 0; i < num; i++) {
		params = next_token(params);
	}

	/* Number of priority-groups. */
	rc = sscanf(params, "%u", &mp->num_groups);
	if (rc != 1) {
		rc = EINVAL;
		goto out;
	}

	params = next_token(params);
	/* We don't care about the initial-priority-group. */
	params = next_token(params);

	for (i = 0; i < mp->num_groups; i++) {
		rc = sscanf(params, "%s %u", mp->group[i].selector, &num);
		if (rc != 2) {
			rc = EINVAL;
			goto out;
		}
		params = next_token(params);
		params = next_token(params);

		/* We don't care about the selector arguments. */
		for (j = 0; j < num; j++) {
			params = next_token(params);
		}

		rc = sscanf(params, "%u %u", &mp->group[i].num_paths,
			    &mp->group[i].num_path_args);
		if (rc != 2) {
			rc = EINVAL;
			goto out;
		}
		params = next_token(params);
		params = next_token(params);

		mp->group[i].path = current_path;
		current_path += mp->group[i].num_paths;

		for (j = 0; j < mp->group[i].num_paths; j++) {
			rc = translate_device(&params,
					      &mp->group[i].path[j].device.major,
					      &mp->group[i].path[j].device.minor);
			if (rc) {
				goto out;
			}

			for (k = 0, sz = 0; k < mp->group[i].num_path_args; k++) {
				rc = sscanf(params, "%s",
					    mp->group[i].path[j].path_args + sz);
				if (rc != 1) {
					rc = EINVAL;
					goto out;
				}

				strncat(mp->group[i].path[j].path_args,
					" ", DM_PATH_ARGS_SIZE);
				sz = strlen(mp->group[i].path[j].path_args);
				params = next_token(params);
			}
		}
	}

	rc = 0;

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

static int multipath_translate_params(dm_target_t *target)
{
	evms_version_t version;
	int rc;

	LOG_PROC_ENTRY();

	rc = get_module_version(DM_TARGET_MULTIPATH, &version);
	if (rc) {
		goto out;
	}

	if (version.major == 1 &&
	    version.minor == 0 &&
	    version.patchlevel <= 3) {
		rc = multipath_translate_params_v1(target);
	} else {
		rc = multipath_translate_params_v2(target);
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * multipath_pretranslate_params
 *
 * Parse the ASCII table string to determine how many multipath devices and
 * groups are represented. Same format as used for multipath_translate_params().
 **/

static int multipath_pretranslate_params_v1(char *params, u_int32_t *num_devs,
					    u_int32_t *num_groups)
{
	int i, j, k, rc, num_paths, num_path_args;

	LOG_PROC_ENTRY();

	*num_devs = 0;

	rc = sscanf(params, "%u", num_groups);
	if (rc != 1) {
		rc = EINVAL;
		goto out;
	}

	params = next_token(params);

	for (i = 0; i < *num_groups; i++) {
		num_paths = num_path_args = 0;

		rc = sscanf(params, "%*s %u %u",
			    &num_paths, &num_path_args);
		if (rc != 2) {
			rc = EINVAL;
			goto out;
		}

		params = next_token(params);
		params = next_token(params);
		params = next_token(params);

		for (j = 0; j < num_paths; j++) {
			params = next_token(params);
			for (k = 0; k < num_path_args; k++) {
				params = next_token(params);
			}
		}

		*num_devs += num_paths;
	}

	rc = 0;

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

static int multipath_pretranslate_params_v2(char *params, u_int32_t *num_devs,
					    u_int32_t *num_groups)
{
	u_int32_t i, j, k, num, num_paths, num_path_args;
	int rc;

	LOG_PROC_ENTRY();

	*num_devs = 0;

	/* We don't care about the "feature" arguments. */
	rc = sscanf(params, "%u", &num);
	if (rc != 1) {
		rc = EINVAL;
		goto out;
	}
	params = next_token(params);

	for (i = 0; i < num; i++) {
		params = next_token(params);
	}

	/* We don't care about the "hw-handler" arguments. */
	rc = sscanf(params, "%u", &num);
	if (rc != 1) {
		rc = EINVAL;
		goto out;
	}
	params = next_token(params);

	for (i = 0; i < num; i++) {
		params = next_token(params);
	}

	/* Number of priority-groups. */
	rc = sscanf(params, "%u", num_groups);
	if (rc != 1) {
		rc = EINVAL;
		goto out;
	}

	params = next_token(params);
	/* We don't care about the initial-priority-group. */
	params = next_token(params);

	for (i = 0; i < *num_groups; i++) {
		num_paths = num_path_args = 0;

		rc = sscanf(params, "%*s %u", &num);
		if (rc != 1) {
			rc = EINVAL;
			goto out;
		}
		params = next_token(params);
		params = next_token(params);

		/* We don't care about the selector arguments. */
		for (j = 0; j < num; j++) {
			params = next_token(params);
		}

		rc = sscanf(params, "%u %u",
			    &num_paths, &num_path_args);
		if (rc != 2) {
			rc = EINVAL;
			goto out;
		}
		params = next_token(params);
		params = next_token(params);

		for (j = 0; j < num_paths; j++) {
			params = next_token(params);
			for (k = 0; k < num_path_args; k++) {
				params = next_token(params);
			}
		}

		*num_devs += num_paths;
	}

	rc = 0;

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

static int multipath_pretranslate_params(char *params, u_int32_t *num_devs,
					 u_int32_t *num_groups)
{
	evms_version_t version;
	int rc;

	LOG_PROC_ENTRY();

	rc = get_module_version(DM_TARGET_MULTIPATH, &version);
	if (rc) {
		goto out;
	}

	if (version.major == 1 &&
	    version.minor == 0 &&
	    version.patchlevel <= 3) {
		rc = multipath_pretranslate_params_v1(params, num_devs,
						      num_groups);
	} else {
		rc = multipath_pretranslate_params_v2(params, num_devs,
						      num_groups);
	}

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * multipath_allocate_target
 *
 * Allocate arrays of groups and paths to attach to the multipath target
 * structure. num_devs is the *total* number of paths in the device, *not*
 * the number of paths per group.
 *
 * This routine has no way to know how many paths are in each group, so only
 * the first group points at the array of paths. It's up to the caller of
 * dm_allocate_target() to walk through the group array and set up the remaining
 * path fields to point at the appropriate places in the array of paths.
 **/
static int multipath_allocate_target(dm_target_t *target,
				     u_int32_t num_devs, u_int32_t num_groups)
{
	dm_target_multipath_t *multipath = target->data.multipath;
	int rc = ENOMEM;

	LOG_PROC_ENTRY();

	if (num_groups < 1 || num_devs < 1) {
		rc = EINVAL;
		goto out;
	}

	multipath->num_groups = num_groups;
	multipath->group = engine_alloc(num_groups *
					sizeof(dm_priority_group_t));
	if (multipath->group) {
		multipath->group[0].path = engine_alloc(num_devs *
							sizeof(dm_path_t));
		if (multipath->group[0].path) {
			rc = 0;
		} else {
			engine_free(multipath->group);
		}
	}
	
out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * multipath_deallocate_target
 *
 * Deallocate the arrays of groups and paths attached to the multipath target
 * structure.
 **/
static void multipath_deallocate_target(dm_target_t *target)
{
	dm_target_multipath_t *multipath = target->data.multipath;

	LOG_PROC_ENTRY();

	engine_free(multipath->group[0].path);
	engine_free(multipath->group);

	LOG_PROC_EXIT_VOID();
}


/*************************
 * Crypt target routines.
 */

/**
 * crypt_build_params
 *
 * Generate an ASCII constructor string for a crypt target.
 * A crypt string has the form:
 *   <cypher> <key> <iv_offset> <major>:<minor> <start>
 **/
static int crypt_build_params(dm_target_t *target)
{
	dm_target_crypt_t *crypt = target->data.crypt;
	char *format = (dm_get_version() == 3) ?
			"%s %s %"PRIu64" %x:%x %"PRIu64 :
			"%s %s %"PRIu64" %u:%u %"PRIu64;
	int rc = ENOMEM;

	LOG_PROC_ENTRY();

	target->params = get_string(200);
	if (target->params) {
		snprintf(target->params, 200, format,
			 crypt->cypher, crypt->key, crypt->iv_offset,
			 crypt->device.major, crypt->device.minor,
			 crypt->device.start);
		rc = 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;

}

/**
 * crypt_translate_params
 *
 * Fill in a crypt target structure based on an ASCII table string of the form:
 *   <cypher> <key> <iv_offset> <major>:<minor> <start>
 **/
static int crypt_translate_params(dm_target_t *target)
{
	dm_target_crypt_t *crypt = target->data.crypt;
	char *params = target->params;
	int rc;

	LOG_PROC_ENTRY();

	rc = sscanf(params, "%s %s %"SCNu64, crypt->cypher,
		    crypt->key, &crypt->iv_offset);
	if (rc != 3) {
		rc = EINVAL;
		goto out;
	}

	params = next_token(params);
	params = next_token(params);
	params = next_token(params);

	rc = translate_device(&params, &crypt->device.major,
			      &crypt->device.minor);
	if (rc) {
		goto out;
	}

	rc = sscanf(params, "%"SCNu64, &crypt->device.start);
	rc = (rc != 1) ? EINVAL : 0;

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/**********************
 * BBR target routines.
 */

/**
 * bbr_build_params
 *
 * Generate an ASCII constructor string for a bbr target.
 * A bbr string has the form:
 *   <major>:<minor> <start> <table1_lba> <table2_lba> <table_size> <replacement_blks_lba> <num_replacement_blks> <blk_size>
 **/
static int bbr_build_params(dm_target_t *target)
{
	dm_target_bbr_t *bbr = target->data.bbr;
	char *format = (dm_get_version() == 3) ?
			"%x:%x %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %u" :
			"%u:%u %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %"PRIu64" %u";
	int rc = ENOMEM;

	LOG_PROC_ENTRY();

	target->params = get_string(150);
	if (target->params) {
		snprintf(target->params, 150, format,
			 bbr->device.major, bbr->device.minor,
			 bbr->device.start, bbr->table1_lba, bbr->table2_lba,
			 bbr->table_size, bbr->replacement_blocks_lba,
			 bbr->num_replacement_blocks, bbr->block_size);
		rc = 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * bbr_translate_params
 *
 * Fill in a bbr target structure based on an ASCII table string of the form:
 *   <device> <table1_lba> <table2_lba> <table_size> <replacement_blks_lba> <num_replacement_blks> <blk_size>
 **/
static int bbr_translate_params(dm_target_t *target)
{
	dm_target_bbr_t *bbr = target->data.bbr;
	char *params = target->params;
	int rc;

	LOG_PROC_ENTRY();

	rc = translate_device(&params, &bbr->device.major, &bbr->device.minor);
	if (rc) {
		goto out;
	}

	rc = sscanf(params, "%"SCNu64" %"SCNu64" %"SCNu64" %"SCNu64" %"SCNu64
		    " %"SCNu64" %u", &bbr->device.start, &bbr->table1_lba,
		    &bbr->table2_lba, &bbr->table_size,
		    &bbr->replacement_blocks_lba, &bbr->num_replacement_blocks,
		    &bbr->block_size);
	rc = (rc != 7) ? EINVAL : 0;

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*************************
 * Sparse target routines.
 */

/**
 * sparse_build_params
 *
 * Generate an ASCII constructor string for a sparse target.
 * A sparse string has the form:
 *   <major>:<minor> <start_lba> <chunk_size> <num_chunks>
 **/
static int sparse_build_params(dm_target_t *target)
{
	dm_target_sparse_t *sparse = target->data.sparse;
	char *format = (dm_get_version() == 3) ?
			"%x:%x %"PRIu64" %u %u" : "%u:%u %"PRIu64" %u %u";
	int rc = ENOMEM;

	LOG_PROC_ENTRY();

	target->params = get_string(65);
	if (target->params) {
		snprintf(target->params, 65, format,
			 sparse->device.major, sparse->device.minor,
			 sparse->device.start, sparse->chunk_size,
			 sparse->num_chunks);
		rc = 0;
		
	}

	LOG_PROC_EXIT_INT(rc)
	return rc;
}

/**
 * sparse_translate_params
 *
 * Fill in a sparse target structure based on an ASCII table string of the form:
 *   <device> <start_lba> <chunk_size> <num_chunks>
 **/
static int sparse_translate_params(dm_target_t *target)
{
	dm_target_sparse_t *sparse = target->data.sparse;
	char *params = target->params;
	int rc;

	LOG_PROC_ENTRY();

	rc = translate_device(&params, &sparse->device.major,
			      &sparse->device.minor);
	if (rc) {
		goto out;
	}

	rc = sscanf(params, "%"SCNu64" %u %u",
		    &sparse->device.start, &sparse->chunk_size,
		    &sparse->num_chunks);
	rc = (rc != 3) ? EINVAL : 0;

out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*************************
 * RAID target routines.
 */

/**
 * raid_build_params
 *
 * Generate an ASCII constructor string for a raid target.
 * A raid string has the form:
 *   <log_type> <num_custom_log_params> [<custom_log_params>]*
 *      <chunk_size> <shared_flag> <num_elements>
 *      [<logmajor>:<logminor> <logstart_lba>] array of log elements
 *      [<major>:<minor> <start_lba>] array of data elements
 *
 * A NULL log_type will always be "core", num_custom_log_params is 0.
 * chunk_size is in sectors
 *
 * This is only available using DM version 4.
 **/
static int raid_build_params(dm_target_t *target)
{
	dm_target_raid_t *raid = target->data.raid;
	int max_len;
	int i, sz, rc = ENOMEM;

	LOG_PROC_ENTRY();

	if (dm_get_version() == 3) {
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (raid->log) {
		sz = strlen(raid->log);
	} else {
		sz = strlen("core 0");
	}

	max_len = sz + 20 + (raid->num_elements * 50);

	target->params = get_string(max_len);
	if (target->params) {
		if (raid->log) {
			strcpy(target->params, raid->log);
		} else {
			strcpy(target->params, "core 0");
		}

		sz += snprintf(target->params + sz, max_len - sz,
			       " %u %u %u",
			       raid->chunk_size,
			       raid->shared_flag,
			       raid->num_elements);

		for (i = 0; i < raid->num_elements; i++) {
			/* If log == null, zero out the log devices. */
			if (raid->log == NULL) {
				raid->logdevices[i].major = 0;
				raid->logdevices[i].minor = 0;
				raid->logdevices[i].start = 0;
			}
			sz += snprintf(target->params + sz, max_len - sz,
				       " %u:%u %"PRIu64,
				       raid->logdevices[i].major,
				       raid->logdevices[i].minor,
				       raid->logdevices[i].start);
		}

		for (i = 0; i < raid->num_elements; i++) {
			sz += snprintf(target->params + sz, max_len - sz,
				       " %u:%u %"PRIu64,
				       raid->devices[i].major,
				       raid->devices[i].minor,
				       raid->devices[i].start);
		}
		LOG_DEBUG("Target parameters: %s\n", target->params);
		rc = 0;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * raid_translate_params
 *
 * Fill in a raid target structure based on an ASCII table string
 * of the form shown above.
 *
 * This is only available using DM version 4.
 **/
static int raid_translate_params(dm_target_t *target)
{
	dm_target_raid_t *raid = target->data.raid;
	char *params = target->params, *log;
	int i, rc;

	LOG_PROC_ENTRY();

	if (dm_get_version() == 3) {
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* Find the log string */
	log = next_token(params);

	rc = sscanf(log, "%u", &i);

	/* Skip the count parameter */
	log = next_token(log);

	/* Skip all the log param */
	while (i) {
		log = next_token(log);
		i--;
	}

	/* Set a 0 at the end of the log string */
	log--;
	*log = 0;
	log++;

	/* Allocate a buffer and copy the log data */
	raid->log = engine_strdup(params);

	params = log;

	rc = sscanf(params, "%u %u %u", &raid->chunk_size,
		    &raid->shared_flag, &raid->num_elements);
	if (rc != 3) {
		rc = EINVAL;
		goto out;
	}

	params = next_token(params);
	params = next_token(params);
	params = next_token(params);

	for (i = 0; i < raid->num_elements; i++) {
		rc = translate_device(&params, &(raid->logdevices[i].major),
				      &(raid->logdevices[i].minor));
		if (rc) {
			goto out;
		}

		rc = sscanf(params, "%"SCNu64, &(raid->logdevices[i].start));
		if (rc != 1) {
			rc = EINVAL;
			goto out;
		}

		params = next_token(params);
	}

	for (i = 0; i < raid->num_elements; i++) {
		rc = translate_device(&params, &(raid->devices[i].major),
				      &(raid->devices[i].minor));
		if (rc) {
			goto out;
		}

		rc = sscanf(params, "%"SCNu64, &(raid->devices[i].start));
		if (rc != 1) {
			rc = EINVAL;
			goto out;
		}

		params = next_token(params);
	}

	rc = 0;
out:
	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * raid_pretranslate_params
 *
 * Parse an ASCII table string to determine how many raid devices are
 * represented. Same format as used for raid_translate_params().
 **/
static int raid_pretranslate_params(char *params, u_int32_t *num_devs,
				    u_int32_t *num_groups)
{
	int i, rc;

	LOG_PROC_ENTRY();

	if (dm_get_version() != 4) {
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* Skip log type at the start of the string. */
	params = next_token(params);

	rc = sscanf(params, "%u", &i);

	/* Skip the count parameter */
	params = next_token(params);

	/* Skip all the log params. */
	while (i) {
		params = next_token(params);
		i--;
	}

	/* Skip chunk size and shared flag. */
	params = next_token(params);
	params = next_token(params);

	rc = sscanf(params, "%u", num_devs);
	rc = (rc != 1) ? EINVAL : 0;

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * raid_allocate_target
 *
 * Allocate an array of log devices and devices to attach to the
 * raid target structure.
 **/
static int raid_allocate_target(dm_target_t *target,
				u_int32_t num_devs, u_int32_t num_groups)
{
	dm_target_raid_t *raid = target->data.raid;
	int rc = 0;

	LOG_PROC_ENTRY();

	if (num_devs < 1) {
		LOG_PROC_EXIT_INT(EINVAL);
		return EINVAL;
	}

	raid->num_elements = num_devs;
	raid->logdevices = engine_alloc(num_devs * sizeof(dm_device_t));
	raid->devices = engine_alloc(num_devs * sizeof(dm_device_t));
	if (!raid->devices || !raid->logdevices) {
		engine_free(raid->logdevices);
		engine_free(raid->devices);
		rc = ENOMEM;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

/**
 * raid_deallocate_target
 *
 * Deallocate the array of logdevices and devices attached to the
 * raid target structure.
 **/
static void raid_deallocate_target(dm_target_t *target)
{
	dm_target_raid_t *raid = target->data.raid;

	LOG_PROC_ENTRY();

	engine_free(raid->logdevices);
	engine_free(raid->devices);
	engine_free(raid->log);

	LOG_PROC_EXIT_VOID();
}

/**
 * dm_target_type_info
 *
 * This structure is indexed based on dm_target_type value, and can be used
 * to retrieve target-type-specific information without needing switch()
 * statements. Keep this structure initialization in the same order as the
 * dm_target_type definition in include/devmapper.h. The definition of a
 * struct dm_target_type_info_t is located in dm.h.
 **/
struct dm_target_type_info_t dm_target_type_info[] = {
	{ "linear",
	  linear_build_params,
	  linear_translate_params,
	  generic_pretranslate_params,
	  generic_allocate_target,
	  generic_deallocate_target,
	  sizeof(dm_device_t) },

	{ "striped",
	  stripe_build_params,
	  stripe_translate_params,
	  stripe_pretranslate_params,
	  stripe_allocate_target,
	  stripe_deallocate_target,
	  sizeof(dm_target_stripe_t) },

	{ "mirror",
	  mirror_build_params,
	  mirror_translate_params,
	  mirror_pretranslate_params,
	  mirror_allocate_target,
	  mirror_deallocate_target,
	  sizeof(dm_target_mirror_t) },

	{ "snapshot",
	  snapshot_build_params,
	  snapshot_translate_params,
	  generic_pretranslate_params,
	  generic_allocate_target,
	  generic_deallocate_target,
	  sizeof(dm_target_snapshot_t) },

	{ "snapshot-origin",
	  origin_build_params,
	  origin_translate_params,
	  generic_pretranslate_params,
	  generic_allocate_target,
	  generic_deallocate_target,
	  sizeof(dm_device_t) },

	{ "multipath",
	  multipath_build_params,
	  multipath_translate_params,
	  multipath_pretranslate_params,
	  multipath_allocate_target,
	  multipath_deallocate_target,
	  sizeof(dm_target_multipath_t) },

	{ "crypt",
	  crypt_build_params,
	  crypt_translate_params,
	  generic_pretranslate_params,
	  generic_allocate_target,
	  generic_deallocate_target,
	  sizeof(dm_target_crypt_t) },

	{ "bbr",
	  bbr_build_params,
	  bbr_translate_params,
	  generic_pretranslate_params,
	  generic_allocate_target,
	  generic_deallocate_target,
	  sizeof(dm_target_bbr_t) },

	{ "sparse",
	  sparse_build_params,
	  sparse_translate_params,
	  generic_pretranslate_params,
	  generic_allocate_target,
	  generic_deallocate_target,
	  sizeof(dm_target_sparse_t) },

	{ "error",
	  generic_build_params,
	  generic_translate_params,
	  generic_pretranslate_params,
	  generic_allocate_target,
	  generic_deallocate_target,
	  0 },

	{ "zero",
	  generic_build_params,
	  generic_translate_params,
	  generic_pretranslate_params,
	  generic_allocate_target,
	  generic_deallocate_target,
	  0 },

	{ "raid0",
	  raid_build_params,
	  raid_translate_params,
	  raid_pretranslate_params,
	  raid_allocate_target,
	  raid_deallocate_target,
	  sizeof(dm_target_raid_t) },

	{ "raid1",
	  raid_build_params,
	  raid_translate_params,
	  raid_pretranslate_params,
	  raid_allocate_target,
	  raid_deallocate_target,
	  sizeof(dm_target_raid_t) },

	{ "raid5",
	  raid_build_params,
	  raid_translate_params,
	  raid_pretranslate_params,
	  raid_allocate_target,
	  raid_deallocate_target,
	  sizeof(dm_target_raid_t) },
};

