/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: lvm2
 * File: evms2/engine/plugins/lvm2/metadata.h
 *
 * Metadata definitions and structures for LVM2 volumes. Most of this header
 * file is derived from information in the lib/label/ and lib/format_text/
 * directories in the LVM2 source package.
 */

#define LVM2_LABEL_ID		"LABELONE"
#define LVM2_LABEL_SIZE		EVMS_VSECTOR_SIZE
#define LVM2_LABEL_SCAN_SECTORS	4
#define LVM2_LABEL_SCAN_SIZE	(LVM2_LABEL_SCAN_SECTORS << EVMS_VSECTOR_SIZE_SHIFT)

#define LVM2_FMTT_MAGIC		"\040\114\126\115\062\040\170\133\065\101\045\162\060\116\052\076"
#define LVM2_FMTT_VERSION	1
#define LVM2_MDA_HEADER_SIZE	EVMS_VSECTOR_SIZE
#define LVM2_LABEL_TYPE		"LVM2 001"
#define LVM2_DEFAULT_MDA_SIZE	380

#define LVM2_UUID_LEN		32
#define LVM2_INITIAL_CRC	0xf597a6cf

#define LVM2_MIN_PV_SIZE(pe_size)					\
				(pe_size + LVM2_LABEL_SCAN_SECTORS +	\
				 LVM2_DEFAULT_MDA_SIZE)

typedef enum {
	VG_FLAGS = 0,
	PV_FLAGS,
	LV_FLAGS
} flag_type_t;

/**
 * struct disk_location
 *
 * Specifies an area of the disk, given by absolute starting offset and size
 * of the area. Values are in bytes, not sectors.
 **/
typedef struct disk_location {
	u_int64_t	offset;
	u_int64_t	size;
} __attribute__ ((packed)) disk_location_t;

/**
 * struct raw_location
 *
 * Like disk_location, but offset is now a relative value instead of absolute.
 * Also includes a checksum of the specified location. Values are in bytes,
 * not sectors.
 **/
typedef struct raw_location {
	u_int64_t	offset;
	u_int64_t	size;
	u_int32_t	checksum;
	u_int32_t	filler;
} __attribute__ ((packed)) raw_location_t;

/**
 * struct label_header
 * @id:		"LABELONE". Identifies this device as belonging to LVM2.
 * @sector:	Sector location of this structure, to verify with the actual
 *		sector where this structure is found.
 * @crc:	CRC of the *rest* of this sector (not including the id, sector,
 *		and crc fields of this structure). CRC is taken when sector is
 *		in disk format.
 * @offset:	Offset (in bytes) from the start of this structure to the
 *		label-type-specific structure, which is located in the same
 *		sector as this label.
 * @type:	Type of label. "LVM2 001" for LVM2 text-format.
 *
 * Structure identifying the device as belonging to LVM2.
 **/
typedef struct label_header {
	u_int8_t	id[8];
	u_int64_t	sector;
	u_int32_t	crc;
	u_int32_t	offset;
	u_int8_t	type[8];
} __attribute__ ((packed)) label_header_t;

/**
 * struct pv_header
 * @pv_uuid:		32-byte PV UUID.
 * @device_size:	Size of data-area, in bytes. *Not* the full
 *			size of the object.
 * @disk_areas:		Two "NULL-terminated" lists.
 *			First list: Data areas
 *			Second list: Metadata areas
 *
 * "Label-type-specific" structure for LVM2 text-format.
 **/
typedef struct pv_header {
	u_int8_t		pv_uuid[LVM2_UUID_LEN];
	u_int64_t		device_size;
	struct disk_location	disk_areas[0];
} __attribute__ ((packed)) pv_header_t;

/**
 * struct mda_header
 * @checksum:		CRC for the *rest* of the sector containing the
 *			mda_header (not including the checksum field). 
 *			CRC is taken when structure is in disk format.
 * @magic:		LVM2_FMTT_MAGIC
 * @version:		LVM2_FMTT_VERSION
 * @start:		Absolute start of this mda_header, in bytes.
 * @size:		Size of this metadata area, in bytes.
 * @raw_locations:	"NULL-terminated" list of VGDA locations. An "orphaned"
 *			PV will have an empty list here.
 *
 * This structure identifies the start of the VGDA section.
 **/
typedef struct mda_header {
	u_int32_t		checksum;
	u_int8_t		magic[16];
	u_int32_t		version;
	u_int64_t		start;
	u_int64_t		size;
	struct raw_location	raw_locations[0];
} __attribute__ ((packed)) mda_header_t;

/**
 * Routines exported from metadata.c
 **/
extern void read_flags(key_value_t *node,
		       flag_type_t type, unsigned long *flags);
extern int write_flags(unsigned long flags, flag_type_t type, char **buffer,
		       unsigned long *offset, unsigned long *size);
extern u_int32_t lvm2_calc_crc(u_int32_t initial, void *buffer, u_int32_t size);
extern int create_new_uuid(char *new_uuid);
extern int read_pv_metadata(storage_object_t *object);
extern void format_uuid(char *unformatted, char *formatted);
extern void unformat_uuid(char *formatted, char *unformatted);
extern int parse_pv_info(key_value_t *pv_entry, char *pv_uuid,
			 u_int32_t *pv_index, u_int64_t *pe_start,
			 u_int64_t *pe_count, unsigned long *pv_flags);
extern int write_mda_header(storage_object_t *object, boolean new_header,
			    boolean resized_headers_only, boolean backup);
extern int write_pv_label_and_header(storage_object_t *object, boolean backup);
extern void erase_metadata(storage_object_t *object);

