/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: utils.c
 */

#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/wait.h>
#include <plugin.h>

#include "utils.h"
#include "ntfsfsim.h"

#define LLONG_MAX	0x8fffffffffffffffLL
#define NTFS_BLOCK_SIZE	512

/* The NTFS oem_id "NTFS    " */
#define magicNTFS	CPU_TO_DISK64(0x202020205346544eULL)

/*
 * BIOS parameter block (BPB) structure.
 */
typedef struct {
	u_int16_t bytes_per_sector;	/* Size of a sector in bytes. */
	u_int8_t  sectors_per_cluster;	/* Size of a cluster in sectors. */
	u_int16_t reserved_sectors;	/* zero */
	u_int8_t  fats;			/* zero */
	u_int16_t root_entries;		/* zero */
	u_int16_t sectors;		/* zero */
	u_int8_t  media_type;		/* 0xf8 = hard disk */
	u_int16_t sectors_per_fat;	/* zero */
	u_int16_t sectors_per_track;	/* irrelevant */
	u_int16_t heads;		/* irrelevant */
	u_int32_t hidden_sectors;	/* zero */
	u_int32_t large_sectors;	/* zero */
} __attribute__ ((__packed__)) BIOS_PARAMETER_BLOCK;

/*
 * NTFS boot sector structure.
 */
typedef struct {
	u_int8_t  jump[3];		/* Irrelevant (jump to boot up code).*/
	u_int64_t oem_id;		/* Magic "NTFS    ". */
	BIOS_PARAMETER_BLOCK bpb;	/* See BIOS_PARAMETER_BLOCK. */
	u_int8_t  unused[4];		/* zero, NTFS diskedit.exe states that
					   this is actually:
						__u_int8_t physical_drive;	// 0x80
						__u_int8_t current_head;	// zero
						__u_int8_t extended_boot_signature;
										// 0x80
						__u_int8_t unused;		// zero
					 */
/*0x28*/int64_t   number_of_sectors;	/* Number of sectors in volume. Gives
					   maximum volume size of 2^63 sectors. */
	int64_t   mft_lcn;		/* Cluster location of MFT data */
	int64_t   mftmirr_lcn;		/* Cluster location of copy of MFT */
	int8_t    clusters_per_mft_record;/* MFT record size in clusters */
	u_int8_t  reserved0[3];		/* zero */
	int8_t    clusters_per_index_record;/* Index block size in clusters. */
	u_int8_t  reserved1[3];		/* zero */
	u_int64_t volume_serial_number;	/* Serial number */
	u_int32_t checksum;		/* Boot sector checksum */
/*0x54*/u_int8_t  bootstrap[426];	/* Boot up code */
	u_int16_t end_of_sector_marker;	/* End of boot sector magic. Always is
					   0xaa55 in little endian. */
/* sizeof() = 512 (0x200) bytes */
} __attribute__ ((__packed__)) NTFS_BOOT_SECTOR;


/*
 * is_boot_sector_ntfs() checks whether the buffer at boot_sector is a valid
 * NTFS boot sector. Returns TRUE if it is valid and FALSE if not.
 */
static boolean is_boot_sector_ntfs(const NTFS_BOOT_SECTOR * boot_sector) {

	u_int32_t cluster_size;

	LOG_ENTRY();

	/* Check for NTFS ID. */
	if (boot_sector->oem_id != magicNTFS) {
		LOG_DETAILS("Volume doesn't have NTFS identifier.\n");
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}
	
	/*
	 * Validate the checksum before we continue.  The checksum is done from
	 * the start of the sector up to, but not including, the checksum field.
	 * If the checksum field is zero, then there is no checksum to validate..
	 */
	if (boot_sector->checksum != 0) {
		int sum;
		u_int32_t * p_32;

		for (sum = 0, p_32 = (u_int32_t *) boot_sector;
		     p_32 < &boot_sector->checksum;
		      ++p_32) {
			sum += DISK_TO_CPU32(*p_32);
		}

		if (DISK_TO_CPU32(boot_sector->checksum) != sum) {
			LOG_DETAILS("Not a valid checksum.\n");
			LOG_EXIT_BOOL(FALSE);
			return FALSE;
		}
	}

	/* bytes_per_sector must be between 256 and 4096, inclusive. */
	if (DISK_TO_CPU16(boot_sector->bpb.bytes_per_sector) <  256 ||
	    DISK_TO_CPU16(boot_sector->bpb.bytes_per_sector) > 4096) {
		LOG_DETAILS("Bytes per sector value of %u is not between 256 and 4096.\n",
			    boot_sector->bpb.bytes_per_sector);
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}

	/* sectors_per_cluster must be a power of 2 no greater than 128. */
	if ((boot_sector->bpb.sectors_per_cluster != 1) &&
	    (boot_sector->bpb.sectors_per_cluster != 2) &&
	    (boot_sector->bpb.sectors_per_cluster != 4) &&
	    (boot_sector->bpb.sectors_per_cluster != 8) &&
	    (boot_sector->bpb.sectors_per_cluster != 16) &&
	    (boot_sector->bpb.sectors_per_cluster != 32) &&
	    (boot_sector->bpb.sectors_per_cluster != 64) &&
	    (boot_sector->bpb.sectors_per_cluster != 128)) {
		LOG_DETAILS("Sectors per cluster value of %u is not valid.\n",
			    boot_sector->bpb.sectors_per_cluster);
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}
	
	/* The cluster size cannot be greater than 64KB. */
	cluster_size = (u_int32_t) DISK_TO_CPU16(boot_sector->bpb.bytes_per_sector) *
			boot_sector->bpb.sectors_per_cluster;

	if (cluster_size > 0x10000) {
		LOG_DETAILS("Cluster size of %u is greater than 65536.\n",
			    cluster_size);
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}
	
	/* Reserved/unused fields in BPB must be zero. */
	if (DISK_TO_CPU16(boot_sector->bpb.reserved_sectors) != 0) {
		LOG_DETAILS("BIOS Parameter Block reserved_sectors field is not zero.\n");
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}

	if (DISK_TO_CPU16(boot_sector->bpb.root_entries) != 0) {
		LOG_DETAILS("BIOS Parameter Block root_entries field is not zero.\n");
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}
	if (DISK_TO_CPU16(boot_sector->bpb.sectors) != 0) {
		LOG_DETAILS("BIOS Parameter Block sectors field is not zero.\n");
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}
	if (DISK_TO_CPU16(boot_sector->bpb.sectors_per_fat) != 0) {
		LOG_DETAILS("BIOS Parameter Block sectors_per_fat field is not zero.\n");
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}
	if (DISK_TO_CPU32(boot_sector->bpb.large_sectors) != 0) {
		LOG_DETAILS("BIOS Parameter Block large_sectors field is not zero.\n");
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}
	if (boot_sector->bpb.fats != 0) {
		LOG_DETAILS("BIOS Parameter Block fats field is not zero.\n");
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}
	
	/*
	 * Validate clusters per file MFT record value.
	 * I don't know what's magic about the 0xe1 and 0xf7 values.
	 * They appear in the ntfsprogs boot sector validation code,
	 * so I replicate them here.
	 */
	if ((u_int8_t) boot_sector->clusters_per_mft_record < 0xe1 ||
	    (u_int8_t) boot_sector->clusters_per_mft_record > 0xf7) {
		if ((boot_sector->clusters_per_mft_record != 1) &&
		    (boot_sector->clusters_per_mft_record != 2) &&
		    (boot_sector->clusters_per_mft_record != 4) &&
		    (boot_sector->clusters_per_mft_record != 8) &&
		    (boot_sector->clusters_per_mft_record != 16) &&
		    (boot_sector->clusters_per_mft_record != 32) &&
		    (boot_sector->clusters_per_mft_record != 64)) {
			LOG_DETAILS("Clusters per file MFT record of %#x is not valid.\n",
				    boot_sector->clusters_per_mft_record);
			LOG_EXIT_BOOL(FALSE);
			return FALSE;
		}
	}
	
	/* Check clusters per index record value is valid. */
	if ((u_int8_t) boot_sector->clusters_per_index_record < 0xe1 ||
	    (u_int8_t) boot_sector->clusters_per_index_record > 0xf7) {
		if ((boot_sector->clusters_per_index_record != 1) &&
		    (boot_sector->clusters_per_index_record != 2) &&
		    (boot_sector->clusters_per_index_record != 4) &&
		    (boot_sector->clusters_per_index_record != 8) &&
		    (boot_sector->clusters_per_index_record != 16) &&
		    (boot_sector->clusters_per_index_record != 32) &&
		    (boot_sector->clusters_per_index_record != 64)) {
			LOG_DETAILS("Clusters per index record of %#x is not valid.\n",
				    boot_sector->clusters_per_index_record);
			LOG_EXIT_BOOL(FALSE);
			return FALSE;
		}
	}
	
	LOG_EXIT_BOOL(TRUE);
	return TRUE;
}

/*
 * Read the boot sector from the volume and validate it. If that fails, try
 * to read the backup boot sector, first from the end of the device a-la NT4 and
 * later and then from the middle of the device a-la NT3.51 and before.
 */
int has_ntfs_boot_sector(logical_volume_t * ev)
{
	int rc = 0;
	NTFS_BOOT_SECTOR * boot_sector;
	int fd;

	LOG_ENTRY();

	boot_sector = EngFncs->engine_alloc(NTFS_BLOCK_SIZE);

	if (boot_sector == NULL) {
		LOG_CRITICAL("Failed to allocate memory for a boot sector.\n");

		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	fd = EngFncs->open_volume(ev, O_RDONLY);
	if (fd < 0) {
		LOG_SERIOUS("Failed to open volume %s.  Error code is %d: %s\n",
			    ev->name, -fd, EngFncs->strerror(-fd));
		EngFncs->engine_free(boot_sector);
		LOG_EXIT_INT(-fd);
		return -fd;
	}

	/* Try to read primary boot sector. */
	if (EngFncs->read_volume(ev, fd, boot_sector, NTFS_BLOCK_SIZE, 0) == NTFS_BLOCK_SIZE) {
		if (is_boot_sector_ntfs(boot_sector)) {
			goto finished;
		} else {
			LOG_DETAILS("Primary boot sector on volume %s is not valid.\n",
                        ev->name);
		}
	} else
		LOG_WARNING("Unable to read primary boot sector on volume %s.\n",
                    ev->name);

	/* Try to read NT4+ backup boot sector at the end of the volume. */
	if (EngFncs->read_volume(ev, fd, boot_sector, NTFS_BLOCK_SIZE, (ev->vol_size << EVMS_VSECTOR_SIZE_SHIFT) - NTFS_BLOCK_SIZE) == NTFS_BLOCK_SIZE) {
		if (is_boot_sector_ntfs(boot_sector)) {
			goto finished;
		}
	} else
		LOG_WARNING("Unable to read NT4+ backup boot sector at offset %"PRIu64" on volume %s.\n",
                    (ev->vol_size << EVMS_VSECTOR_SIZE_SHIFT) - NTFS_BLOCK_SIZE,
                    ev->name);
	/* Try to read NT3.51- backup boot sector in the middle of the volume. */
	if (EngFncs->read_volume(ev, fd, boot_sector, NTFS_BLOCK_SIZE, (ev->vol_size << (EVMS_VSECTOR_SIZE_SHIFT - 1))) == NTFS_BLOCK_SIZE) {
		if (is_boot_sector_ntfs(boot_sector)) {
			goto finished;
		} else {
			LOG_DETAILS("Could not find a valid backup boot sector.\n");
		}
	} else
		LOG_WARNING("Unable to read NT3.51- backup boot sector at offset %"PRIu64" on volume %s.\n",
                    ev->vol_size << (EVMS_VSECTOR_SIZE_SHIFT - 1),
                    ev->name);

	rc = EINVAL;
finished:
	EngFncs->close_volume(ev, fd);

	EngFncs->engine_free(boot_sector);

	LOG_EXIT_INT(rc);
	return rc;
}


int clear_ntfs_boot_sectors(logical_volume_t * ev)
{
	int rc = 0;
	int fd;
	void * block;
	int32_t bytes_written;

	LOG_ENTRY();

	block = EngFncs->engine_alloc(NTFS_BLOCK_SIZE);
	if (block == NULL) {
		LOG_CRITICAL("Can't get a buffer for writing.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	fd = EngFncs->open_volume(ev, O_WRONLY);
	if (fd < 0) {
		LOG_SERIOUS("Failed to open volume %s.  Error code is %d: %s\n",
			    ev->name, -fd, EngFncs->strerror(-fd));
		EngFncs->engine_free(block);
		LOG_EXIT_INT(-fd);
		return -fd;
	}

	/* Wipe out the primary boot sector. */
	bytes_written = EngFncs->write_volume(ev, fd, block, NTFS_BLOCK_SIZE, 0);
	if (bytes_written != NTFS_BLOCK_SIZE) {
		LOG_WARNING("Failed to clear the primary boot sector on volume %s.  "
			    "Only %d bytes were written.\n",
			    ev->name, bytes_written);
		rc = -bytes_written;
	}

	/* Wipe out the NT4+ backup boot sector. */
	bytes_written = EngFncs->write_volume(ev, fd, block, NTFS_BLOCK_SIZE, (ev->vol_size << EVMS_VSECTOR_SIZE_SHIFT) - NTFS_BLOCK_SIZE);
	if (bytes_written != NTFS_BLOCK_SIZE) {
		LOG_WARNING("Failed to clear the primary boot sector on volume %s.  "
			    "Only %d bytes were written.\n",
			    ev->name, bytes_written);
		rc = -bytes_written;
	}

	/* Wipe out the NT3.51- backup boot sector. */
	bytes_written = EngFncs->write_volume(ev, fd, block, NTFS_BLOCK_SIZE, (ev->vol_size << (EVMS_VSECTOR_SIZE_SHIFT - 1)));
	if (bytes_written != NTFS_BLOCK_SIZE) {
		LOG_WARNING("Failed to clear the primary boot sector on volume %s.  "
			    "Only %d bytes were written.\n",
			    ev->name, bytes_written);
		rc = -bytes_written;
	}

	EngFncs->close_volume(ev, fd);

	EngFncs->engine_free(block);

	LOG_EXIT_INT(rc);
	return rc;
}


int try_run(char * prog_name) {

	int     rc = 0;
	char  * argv[3];
	pid_t   pidm;
	int     status;
	int     fds[2];

	LOG_ENTRY();

	/* Open a pipe to catch the program output that we don't care about. */
	rc = pipe(fds);
	if (rc) {
		LOG_SERIOUS("Could not opening a pipe.  Error code is %d: %s\n",
			    errno, strerror(errno));

		LOG_EXIT_INT(errno);
		return(errno);
	}

	/*
	 * Run with "-V" even though we are not getting the version.
	 * The program may have some default behavior if no options are
	 * given.  We don't want the program to do anything.
	 */
	argv[0] = prog_name;
	argv[1] = "-V";
	argv[2] = NULL;

	pidm = EngFncs->fork_and_execvp(NULL, argv, NULL, NULL, NULL);

	if (pidm != -1) {

		waitpid(pidm, &status, 0);

		if (WIFEXITED(status)) {

			LOG_DEFAULT("\"%s -V\" completed with exit code %d.\n",
				    prog_name, WEXITSTATUS(status));

		} else {
			LOG_WARNING("%s did not exit normally.\n", prog_name);
			rc = EINTR;
		}

	} else {
		rc = errno;
		LOG_DEFAULT("Unable to run %s.  Error code is %d: %s\n",
			    prog_name, rc, EngFncs->strerror(rc));
	}

	close(fds[0]);
	close(fds[1]);

	LOG_EXIT_INT(rc);
	return(rc);
}


int get_version_from_fd(int fd, char * version) {

	char * buffer;
	int    bytes_read;
	char * ver_start;
	char * ver_end;
	char * line_start;
	char * line_end;
	char * pch;

	LOG_ENTRY();

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (buffer == NULL) {
		LOG_CRITICAL("Unable to get memory for a buffer.\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	line_end = buffer;
	do {
		bytes_read = read(fd, line_end, MAX_USER_MESSAGE_LEN - 1 - (line_end - buffer));
		if (bytes_read > 0) {
			line_end[bytes_read] = '\0';

			line_start = buffer;

			/* Process the output line by line. */
			do {
				pch = strchr(line_start, '\n');

				if (pch != NULL) {
					*pch = '\0';

					/*
					 * NTFS utilities version strings appear
					 * as " v[0-9].[0-9].[0-9]".
					 */
					ver_start = strstr(line_start, " v");
					if (ver_start != NULL) {
						ver_start += 2;

						for (ver_end = ver_start;
						    (*ver_end != ' ') && (*ver_end != '\t') &&
						    (*ver_end != '\n') && (*ver_end != '\0');
						    ver_end++);

						*ver_end = '\0';
						strcpy(version, ver_start);
					}

					if (version[0] == '\0') {
						/* Skip over the newline we zeroed out. */
						line_start = pch + 1;
					}
				}

			} while ((pch != NULL) && (version[0] == '\0'));

			if (version[0] == '\0') {
				/*
				 * If there is text left in the buffer, it is
				 * because it did not end with a newline.
				 * Move the text to the front of the buffer
				 * and set line_end to the end of the text so
				 * that the next read() will put the new data
				 * right after the string that was moved to
				 * the front of the buffer.
				 */

				/*
				 * Must do the move by hand since the remaining
				 * text might overlap its destination.
				 */
				for (line_end = buffer; *line_start != '\0'; line_start++) {
					*line_end = *line_start;
				}
			}
		}

	} while ((version[0] == '\0') && (bytes_read > 0));

	EngFncs->engine_free(buffer);

	LOG_EXIT_INT(0);
	return 0;
}


int try_run_get_version(char * prog_name, char * version) {

	int     rc = 0;
	char  * argv[3];
	pid_t   pidm;
	int     status;
	int     fds[2];

	LOG_ENTRY();

	version[0] = '\0';

	rc = pipe(fds);
	if (rc) {
		LOG_SERIOUS("Could not opening a pipe.  Error code is %d: %s\n",
			    errno, strerror(errno));

		LOG_EXIT_INT(errno);
		return(errno);
	}

	argv[0] = prog_name;
	argv[1] = "-V";
	argv[2] = NULL;

	fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL, 0) | O_NONBLOCK);
	fcntl(fds[1], F_SETFL, fcntl(fds[1], F_GETFL, 0) | O_NONBLOCK);

	pidm = EngFncs->fork_and_execvp(NULL, argv, NULL, fds, fds);

	if (pidm != -1) {

		waitpid(pidm, &status, 0);

		if (WIFEXITED(status)) {

			LOG_DEFAULT("\"%s -V\" completed with exit code %d.\n",
				    prog_name, WEXITSTATUS(status));

			fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL,0) | O_NONBLOCK);

			rc = get_version_from_fd(fds[0], version);

			LOG_DEFAULT("%s version is %s\n",
				    prog_name, (version[0] != '\0') ? version : "(none)");

		} else {
			LOG_WARNING("%s did not exit normally.\n", prog_name);
			rc = EINTR;
		}

	} else {
		rc = errno;
		LOG_DEFAULT("Unable to run %s.  Error code is %d: %s\n",
			    prog_name, rc, EngFncs->strerror(rc));
	}

	close(fds[0]);
	close(fds[1]);

	LOG_EXIT_INT(rc);
	return(rc);
}


/*
 * Search a buffer of output for a field name.  If the name is found,
 * pass back the number that follows the field name on the line
 * and return 0.  If the name is not found, leave *number alone and
 * return ENOENT.  Leaving *number alone allows the caller to pre-load
 * number with a default value.
 */
int get_field_number_value(char * buffer, char * field_name,
			   u_int64_t * number) {

	char * pch;

	LOG_ENTRY();

	pch = strstr(buffer, field_name);
	if (pch == NULL) {
		LOG_WARNING("Field \"%s\" was not found in the buffer.\n", field_name);
		LOG_EXIT_INT(ENOENT);
		return ENOENT;
	}

	/* Skip over field and the following white space and delimiters. */
	pch += strlen(field_name);
	pch += strspn(pch,":= \t");
	*number = strtoul(pch, &pch, 10);

	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Search a buffer of output for a field name.  If the name is found,
 * pass back a copy of the string that follows the field name on the
 * line * and return 0.  If the name is not found, leave *string alone
 * and return ENOENT.
 */
int get_field_string_value(char * buffer, char * field_name,
			   char * * string) {

	char * start;
	char * end;
	char   save;

	LOG_ENTRY();

	start = strstr(buffer, field_name);
	if (start == NULL) {
		LOG_WARNING("Field \"%s\" was not found in the buffer.\n", field_name);
		LOG_EXIT_INT(ENOENT);
		return ENOENT;
	}

	/* Skip over field and the following white space and delimiters. */
	start += strlen(field_name);
	start += strspn(start,":= \t");
	end = strchr(start, '\n');
	if (end != NULL) {
		save = *end;
		*end = '\0';
		*string = EngFncs->engine_strdup(start);
		*end = save;

	} else {
		*string = EngFncs->engine_strdup(start);
	}

	LOG_EXIT_INT(0);
	return 0;
}


sector_count_t get_min_fs_size(logical_volume_t * vol) {

	int    rc = 0;
	char * argv[5];
	pid_t  pidm;
	int    status;
	int    fds[2];
	char * buffer = NULL;
	u_int64_t      min_fs_size_bytes = 0;
	sector_count_t min_fs_size = vol->fs_size;

	LOG_ENTRY();

	if (!have_ntfsresize) {
		LOG_DETAILS("The ntfsresize utility is not installed.\n");
		LOG_EXIT_U64(min_fs_size);
		return min_fs_size;
	}

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (buffer == NULL) {
		LOG_EXIT_U64(min_fs_size);
		return min_fs_size;
	}

	status = pipe(fds);
	if (status < 0) {
		EngFncs->engine_free(buffer);
		LOG_EXIT_U64(min_fs_size);
		return min_fs_size;
	}

	/* Run ntfsresize -i to get the smallest volume size. */
	argv[0] = "ntfsresize";
	argv[1] = "-i";
	argv[2] = "-f";
	argv[3] = vol->dev_node;
	argv[4] = NULL;

	fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL, 0) | O_NONBLOCK);
	fcntl(fds[1], F_SETFL, fcntl(fds[1], F_GETFL, 0) | O_NONBLOCK);

	pidm = EngFncs->fork_and_execvp(vol, argv, NULL, fds, fds);
	if (pidm != -1) {
		
		waitpid( pidm, &status, 0);

		if (WIFEXITED(status)) {
			read(fds[0], buffer, MAX_USER_MESSAGE_LEN);

			rc = WEXITSTATUS(status);
			LOG_DETAILS("%s completed with exit code %d.\n", argv[0], rc);

		} else {
			/*
			 * The process didn't exit. It must have been
			 * interrupted by a signal.
			 */
			rc = EINTR;
		}

	} else {
		LOG_SERIOUS("Failed to fork and exec %s.  Error code is %d: %s\n",
			    argv[0], rc, EngFncs->strerror(rc));
		rc = errno;
	}

	get_field_number_value(buffer, "resize at", &min_fs_size_bytes);
	if (min_fs_size_bytes != 0) {
		/* ntfsresize reports the size in bytes. */
		min_fs_size = min_fs_size_bytes >> EVMS_VSECTOR_SIZE_SHIFT;

	}

	EngFncs->engine_free(buffer);
	
	close(fds[0]);
	close(fds[1]);

	LOG_EXIT_U64(min_fs_size);
	return min_fs_size;
}


int fill_private_data(logical_volume_t * vol) {

	int rc = 0;
	private_data_t * pd = (private_data_t *) vol->private_data;
	char * argv[6];
	pid_t  pidm;
	int    status;
	int    fds[2];
	char * buffer = NULL;

	LOG_ENTRY();

	if (!have_ntfsinfo) {
		LOG_DETAILS("The ntfsinfo utility is not installed.\n");
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (buffer == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	status = pipe(fds);
	if (status < 0) {
		EngFncs->engine_free(buffer);
		LOG_EXIT_INT(errno);
		return errno;
	}

	/* Run ntfsinfo to get information about the volume. */
	argv[0] = "ntfsinfo";
	argv[1] = "-m";
	argv[2] = "-f";
	argv[3] = "-d";
	if (pd->flags & PDFLAG_NTFSCLONE_TARGET) {
		argv[4] = pd->clone_source->dev_node;

	} else {
		argv[4] = vol->dev_node;
	}
	argv[5] = NULL;

	fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL, 0) | O_NONBLOCK);
	fcntl(fds[1], F_SETFL, fcntl(fds[1], F_GETFL, 0) | O_NONBLOCK);

	pidm = EngFncs->fork_and_execvp(vol, argv, NULL, fds, fds);
	if (pidm != -1) {
		
		waitpid(pidm, &status, 0);

		if (WIFEXITED(status)) {
			read(fds[0], buffer, MAX_USER_MESSAGE_LEN);

			rc = WEXITSTATUS(status);
			LOG_DETAILS("%s completed with exit code %d.\n", argv[0], rc);

		} else {
			/*
			 * The process didn't exit. It must have been
			 * interrupted by a signal.
			 */
			rc = EINTR;
		}

	} else {
		LOG_SERIOUS("Failed to fork and exec %s.  Error code is %d: %s\n",
			    argv[0], rc, EngFncs->strerror(rc));
		rc = errno;
	}

	/* Initialize default values. */
	pd->ntfs_cluster_size = EVMS_VSECTOR_SIZE;
	pd->ntfs_nr_clusters  = vol->vol_size;

	get_field_number_value(buffer, "Cluster Size", &pd->ntfs_cluster_size);
	get_field_number_value(buffer, "Volume Size in Clusters", &pd->ntfs_nr_clusters);
	get_field_string_value(buffer, "Volume Name", &pd->ntfs_vol_name);
	get_field_string_value(buffer, "Volume Version", &pd->ntfs_version);

	if (pd->ntfs_cluster_size == EVMS_VSECTOR_SIZE) {
		pd->fs_size = pd->ntfs_nr_clusters;
		pd->max_fs_size = LLONG_MAX;

	} else if (pd->ntfs_cluster_size > EVMS_VSECTOR_SIZE) {
		pd->fs_size = pd->ntfs_nr_clusters * (pd->ntfs_cluster_size / EVMS_VSECTOR_SIZE);
		pd->max_fs_size = LLONG_MAX * (pd->ntfs_cluster_size / EVMS_VSECTOR_SIZE);

	} else {
		pd->fs_size = pd->ntfs_nr_clusters / (EVMS_VSECTOR_SIZE - pd->ntfs_cluster_size);
		pd->max_fs_size = LLONG_MAX / (EVMS_VSECTOR_SIZE - pd->ntfs_cluster_size);
	}

	EngFncs->engine_free(buffer);
	
	LOG_DEBUG("On volume %s:\n", vol->name);
	LOG_DEBUG("Volume size:\t%"PRIu64"\n", vol->vol_size);
	LOG_DEBUG("File system size:\t%"PRIu64"\n", pd->fs_size);
	LOG_DEBUG("Maximum file system size:\t%"PRIu64"\n", pd->max_fs_size);

	close(fds[0]);
	close(fds[1]);

	LOG_EXIT_INT(0);
	return 0;
}


void free_private_data(logical_volume_t * vol) {

	LOG_ENTRY();

	if (vol->private_data != NULL) {
		private_data_t * pd = (private_data_t *) vol->private_data;

		if (pd->ntfs_vol_name != NULL) {
			EngFncs->engine_free(pd->ntfs_vol_name);
		}

		EngFncs->engine_free(vol->private_data);
		vol->private_data = NULL;
	}

	LOG_EXIT_VOID();
}


logical_volume_t * find_volume(char * name) {

	list_anchor_t volume_list;
	list_element_t iter;
	logical_volume_t * vol = NULL;

	LOG_ENTRY();

	if (EngFncs->get_volume_list(NULL,NULL,0,&volume_list) == 0) {
		LIST_FOR_EACH(volume_list,iter,vol) {
			if (strcmp(vol->name,name) == 0) {
				break;
			}
		}

		EngFncs->destroy_list(volume_list);
	}

	LOG_EXIT_PTR(vol);
	return vol;
}


/*
 * Move the text following a \r to follow immediately after the first \n that
 * occurs before the \r, or to the beginning of the buffer if no \n precedes
 * the \r.
 */
static void handle_carriage_returns(char * buffer) {

	char * p_cr;
	char * p_nl;

	/* No entry/exit logs. */

	while ((p_cr = strrchr(buffer, '\r')) != NULL) {
		if (p_cr == buffer) {
			*p_cr = '\0';

		} else {
			p_nl = p_cr;
			while ((p_nl != buffer) && (*(p_nl - 1) != '\n')) {
				p_nl--;
			}

			/*
			 * p_cr points to the last \r in the buffer.  p_nl
			 * points to the character after \n before the \r or the
			 * start of the buffer.  Move the characters after the
			 * \r to the location at p_nl.
			 */
			for (p_cr++; *(p_cr - 1) != '\0'; p_cr++, p_nl++) {
				*p_nl = *p_cr;
			}
		}
	}
}


/* ntfsresize prompts with "Are you sure you want to proceed (y/[n])?" */
#define RESIZE_PROMPT		"(y/[n])?"

int resize_ntfs(logical_volume_t * volume, sector_count_t * new_size) {

	int    rc = 0;
	char * argv[6];
	char   ascii_new_size[16];
	pid_t  pidm;
	int    status;
	int    fds[2];
	char * buffer = NULL;
	int    bytes_read = 0;
	private_data_t * pd = volume->private_data;

	LOG_ENTRY();

	if (!have_ntfsresize) {
		MESSAGE("The ntfsresize utility is not installed on this machine.  "
			"The NTFS FSIM uses ntfsresize to expand the NTFS file system on the volume.  "
			"Get the latest version of the NTFS utilities from http://sourceforge.net/projects/linux-ntfs/\n");
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (buffer == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	status = pipe(fds);
	if (status < 0) {
		EngFncs->engine_free(buffer);
		LOG_EXIT_INT(errno);
		return errno;
	}

	argv[0] = "ntfsresize";
	argv[1] = "-ff";
	argv[2] = "-s";

	/* Get the ASCII version of the new size in KB. */
	sprintf(ascii_new_size, "%"PRIu64, (*new_size) >> 1);
	strcat(ascii_new_size, "k");
	argv[3] = ascii_new_size;

	argv[4] = volume->dev_node;
	argv[5] = NULL;

	fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL, 0) | O_NONBLOCK);
	fcntl(fds[1], F_SETFL, fcntl(fds[1], F_GETFL, 0) | O_NONBLOCK);

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds, fds);
	if (pidm != -1) {

		while (!(waitpid( pidm, &status, WNOHANG ))) {

			bytes_read = read(fds[0], buffer, MAX_USER_MESSAGE_LEN - 1);
			if (bytes_read > 0) {
				/* Make the output into a string. */
				buffer[bytes_read] = '\0';

				/*
				 * ntfsresize prints out a % progress line.
				 * Each new line starts with a \r so that it
				 * overwrites the previously printed line.
				 * We don't want to display all those progress
				 * status lines as user messages.  It causes a
				 * lot of beeping in the GUI.  So we handle
				 * them in our buffer like it was a display.
				 * When encountering a \r, overwrite the
				 * previous text.
				 */
				handle_carriage_returns(buffer);
				if (buffer[0] != '\0') {
					MESSAGE("%s output: \n%s\n", argv[0], buffer);
				}
			}
			usleep(10000); // Don't hog all the CPU.
		}

		if (WIFEXITED(status)) {
			do {
				bytes_read = read(fds[0], buffer, MAX_USER_MESSAGE_LEN - 1);
				if (bytes_read > 0) {
					/* Make the output into a string. */
					buffer[bytes_read] = '\0';

					MESSAGE("%s output: \n%s\n", argv[0], buffer);
				}

			} while (bytes_read > 0);

			rc = WEXITSTATUS(status);
			if (rc == 0) {
				LOG_DETAILS("%s completed with exit code %d.\n", argv[0], rc);
			} else {
				LOG_WARNING("%s completed with exit code %d.\n", argv[0], rc);
			}

		} else {
			/*
			 * The process didn't exit. It must have been
			 * interrupted by a signal.
			 */
			rc = EINTR;
		}

	} else {
		LOG_SERIOUS("Failed to fork and exec %s.  Error code is %d: %s\n",
			    argv[0], rc, EngFncs->strerror(rc));
		rc = errno;
	}

	close(fds[0]);
	close(fds[1]);

	EngFncs->engine_free(buffer);
	
	if (rc == 0) {
		/* Get the new sizes into the private data. */
		memset(pd, 0, sizeof(private_data_t));
		fill_private_data(volume);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Run ntfsfix on the volume.
 */
int run_ntfsfix(logical_volume_t * volume) {
	
	int    rc = 0;
	char * argv[3];
	pid_t  pidm;
	int    status;
	int    fds[2];
	char * buffer = NULL;
	int    bytes_read = 0;

	LOG_ENTRY();

	if (volume->file_system_manager != my_plugin_record) {
		/* It's not my volume. */
		LOG_ERROR("Volume %s does not have NTFS on it.\n", volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (EngFncs->is_mounted(volume->name, NULL)) {
		LOG_ERROR("Volume %s is mounted.  It must be unmounted in order to run ntfsfix.\n",
			  volume->name);
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	if (!have_ntfsfix) {
		MESSAGE("The ntfsfix utility is not installed on this machine.  "
			"The NTFS FSIM uses ntfsfix to fix the NTFS file system on the volume.  "
			"Get the latest version of the NTFS utilities from http://sourceforge.net/projects/linux-ntfs/\n");
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (buffer == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	status = pipe(fds);
	if (status < 0) {
		EngFncs->engine_free(buffer);
		LOG_EXIT_INT(errno);
		return errno;
	}

	argv[0] = "ntfsfix";
	argv[1] = volume->dev_node;
	argv[2] = NULL;

	fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL, 0) | O_NONBLOCK);
	fcntl(fds[1], F_SETFL, fcntl(fds[1], F_GETFL, 0) | O_NONBLOCK);

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds, fds);
	if (pidm != -1) {

		while (!(waitpid( pidm, &status, WNOHANG ))) {
			
			bytes_read = read(fds[0], buffer, MAX_USER_MESSAGE_LEN - 1);
			if (bytes_read > 0) {
				/* Make the output into a string. */
				buffer[bytes_read] = '\0';

				MESSAGE("%s output: \n%s\n", argv[0], buffer);
			}

			usleep(10000); // Don't hog all the CPU.
		}

		if (WIFEXITED(status)) {
			do {
				bytes_read = read(fds[0], buffer, MAX_USER_MESSAGE_LEN - 1);
				if (bytes_read > 0) {
					/* Make the output into a string. */
					buffer[bytes_read] = '\0';

					MESSAGE("%s output: \n%s\n", argv[0], buffer);
				}
			} while (bytes_read > 0);

			rc = WEXITSTATUS(status);
			if (rc == 0) {
				LOG_DETAILS("%s completed with exit code %d.\n", argv[0], rc);
			} else {
				LOG_WARNING("%s completed with exit code %d.\n", argv[0], rc);
			}

		} else {
			/*
			 * The process didn't exit. It must have been
			 * interrupted by a signal.
			 */
			rc = EINTR;
		}

	} else {
		LOG_SERIOUS("Failed to fork and exec %s.  Error code is %d: %s\n",
			    argv[0], rc, EngFncs->strerror(rc));
		rc = errno;
	}

	EngFncs->engine_free(buffer);
	
	close(fds[0]);
	close(fds[1]);

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Run ntfsfix on the volume.
 */
int run_ntfsclone(logical_volume_t * source,
		  logical_volume_t * target,
		  boolean            force) {
	
	int    rc = 0;
	char * argv[6];
	int    i;
	pid_t  pidm;
	int    status;
	int    fds[2];
	char * buffer = NULL;
	int    bytes_read = 0;

	LOG_ENTRY();

	if (source->file_system_manager != my_plugin_record) {
		/* It's not my volume. */
		LOG_ERROR("Volume %s does not have NTFS on it.\n", source->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (EngFncs->is_mounted(source->name, NULL)) {
		LOG_ERROR("Volume %s is mounted.  It must be unmounted in order to run ntfsclone.\n",
			  source->name);
		LOG_EXIT_INT(EBUSY);
		return EBUSY;
	}

	if (!have_ntfsclone) {
		MESSAGE("The ntfsclone utility is not installed on this machine.  "
			"The NTFS FSIM uses ntfsclone to make a clone of a volume to another volume.  "
			"Get the latest version of the NTFS utilities from http://sourceforge.net/projects/linux-ntfs/\n");
		LOG_EXIT_INT(ENOSYS);
		return ENOSYS;
	}

	buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN);
	if (buffer == NULL) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	status = pipe(fds);
	if (status < 0) {
		EngFncs->engine_free(buffer);
		LOG_EXIT_INT(errno);
		return errno;
	}

	i = 0;
	argv[i++] = "ntfsclone";
	if (force) {
		argv[i++] = "-f";
	}
	argv[i++] = "-O";
	argv[i++] = target->dev_node;
	argv[i++] = source->dev_node;
	argv[i++] = NULL;

	fcntl(fds[0], F_SETFL, fcntl(fds[0], F_GETFL, 0) | O_NONBLOCK);
	fcntl(fds[1], F_SETFL, fcntl(fds[1], F_GETFL, 0) | O_NONBLOCK);

	pidm = EngFncs->fork_and_execvp(source, argv, NULL, fds, fds);
	if (pidm != -1) {
		
		while (!(waitpid( pidm, &status, WNOHANG ))) {
			
			bytes_read = read(fds[0], buffer, MAX_USER_MESSAGE_LEN - 1);
			if (bytes_read > 0) {
				/* Make the output into a string. */
				buffer[bytes_read] = '\0';

				/*
				 * ntfsclone prints out a % progress line.
				 * Each new line starts with a \r so that it
				 * overwrites the previously printed line.
				 * We don't want to display all those progress
				 * status lines as user messages.  It causes a
				 * lot of beeping in the GUI.  So we handle
				 * them in our buffer like it was a display.
				 * When encountering a \r, overwrite the
				 * previous text.
				 */
				handle_carriage_returns(buffer);
				if (buffer[0] != '\0') {
					MESSAGE("%s output: \n%s\n", argv[0], buffer);
				}
			}
			usleep(10000); // Don't hog all the CPU.
		}

		if (WIFEXITED(status)) {
			do {
				bytes_read = read(fds[0], buffer, MAX_USER_MESSAGE_LEN - 1);
				if (bytes_read > 0) {
					/* Make the output into a string. */
					buffer[bytes_read] = '\0';

					MESSAGE("%s output: \n%s\n", argv[0], buffer);
				}
			} while (bytes_read > 0);

			/*
			 * ntfsclone's last message is "Syncing..." That's nice
			 * at a command prompt because you will get a new prompt
			 * when the command is finished.  But in our display of
			 * UI messages there is no indication that the command
			 * completed.  So put up a final message saying how
			 * ntfsclone finished.
			 */
			rc = WEXITSTATUS(status);
			if (rc == 0) {
				MESSAGE("%s completed successfully.\n", argv[0]);
			} else {
				MESSAGE("%s completed with exit code %d.\n", argv[0], rc);
			}

		} else {
			/*
			 * The process didn't exit. It must have been
			 * interrupted by a signal.
			 */
			rc = EINTR;
		}

	} else {
		LOG_SERIOUS("Failed to fork and exec %s.  Error code is %d: %s\n",
			    argv[0], rc, EngFncs->strerror(rc));
		rc = errno;
	}

	EngFncs->engine_free(buffer);
	
	close(fds[0]);
	close(fds[1]);

	LOG_EXIT_INT(rc);
	return rc;
}


int is_acceptable_clone_target(logical_volume_t * source,
			       logical_volume_t * target) {

	LOG_ENTRY();

	if (target->file_system_manager != NULL) {
		LOG_DETAILS("Target volume %s is already managed by %s.\n",
			    target->name, target->file_system_manager->short_name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (!(target->flags & VOLFLAG_ACTIVE)) {
		LOG_DETAILS("Target volume %s is not active.\n",
			    target->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (EngFncs->is_mounted(target->dev_node, NULL)) {
		LOG_DETAILS("Target volume %s is mounted on %s.\n",
			    target->name, target->mount_point);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (target->disk_group != source->disk_group) {
		LOG_DETAILS("Target volume %s in disk group %s is not in the same disk group as source volume %s in disk group %s.\n",
			    target->name, (target->disk_group != NULL) ? target->disk_group->name : "(local)",
			    source->name, (source->disk_group != NULL) ? source->disk_group->name : "(local)");
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (target->vol_size < source->vol_size) {
		LOG_DETAILS("Volume %s is too small to be a clone of volume %s.\n",
			    target->name, source->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}


