/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: delete.c
 */

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "thing.h"
#include "delete.h"
#include "readable.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   void on_deletion_window_clist_select_row (GtkCList *, gint, gint, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine is called whenever a row is selected. 
 *      All we do is ensure that the "Delete" and "Recursive Delete"
 *      buttons are sensitive to allow the operation to be initiated.
 * 
 *   Entry:
 *      clist     - address of GtkCList object that received select-row signal
 *      row       - index of row selected
 *      column    - index of column selected on current row
 *      event     - address of structure to obtain additional context information
 *      user_data - address of user data bound to clist (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_deletion_window_clist_select_row(GtkCList * clist, gint row, gint column,
					 GdkEventButton * event, gpointer user_data)
{
	GtkWidget *delete_button = lookup_widget(GTK_WIDGET(clist), "selection_window_next_button");
	GtkWidget *recursive_delete_button =
	    lookup_widget(GTK_WIDGET(clist), "selection_window_prev_button");

	if (!(GTK_WIDGET_IS_SENSITIVE(delete_button))) {
		gtk_widget_set_sensitive(delete_button, TRUE);
		gtk_widget_grab_default(delete_button);
	}

	if (!(GTK_WIDGET_IS_SENSITIVE(recursive_delete_button)))
		gtk_widget_set_sensitive(recursive_delete_button, TRUE);
}

/*
 *
 *   void on_deletion_window_clist_unselect_row (GtkCList *, gint, gint, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine is called whenever a row is unselected. 
 *      All we do is ensure that the "Delete" and "Recursive Delete"
 *      buttons are not sensitive to disallow the operation from being 
 *      initiated until a choice is made.
 * 
 *   Entry:
 *      clist     - address of GtkCList object that received unselect-row signal
 *      row       - index of row deselected
 *      column    - index of column deselected on current row
 *      event     - address of structure to obtain additional context information
 *      user_data - address of user data bound to clist (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_deletion_window_clist_unselect_row(GtkCList * clist, gint row, gint column,
					   GdkEventButton * event, gpointer user_data)
{
	GtkWidget *delete_button = lookup_widget(GTK_WIDGET(clist), "selection_window_next_button");
	GtkWidget *recursive_delete_button =
	    lookup_widget(GTK_WIDGET(clist), "selection_window_prev_button");

	if (g_list_length(clist->selection) == 0 && GTK_WIDGET_IS_SENSITIVE(delete_button))
		gtk_widget_set_sensitive(delete_button, FALSE);

	if (g_list_length(clist->selection) == 0
	    && GTK_WIDGET_IS_SENSITIVE(recursive_delete_button))
		gtk_widget_set_sensitive(recursive_delete_button, FALSE);
}

/*
 *
 *   GtkWidget *create_deletion_window (gchar *, gchar *, GtkSignalFunc,
 *                                      GtkSignalFunc, GtkSignalFunc, gpointer)
 *
 *   Description:
 *      This creates the window used for deleting "things". It makes some minor
 *      modifications to a standard selection window to reuse the "Prev" button
 *      to mean "Recursive Delete".
 * 
 *   Entry:
 *      title - the text for the window title
 *      clist_realize_handler - the handler to populate the selection list
 *      delete_button_handler - the handler to act upon clicking "Delete"
 *      recursive_delete_button_handler - the handler to act upon "Recursive Delete"
 *      user_data - whatever is needed by the signal handlers
 *
 *   Exit:
 *      Returns the id of the deletion window.
 *
 */
GtkWidget *create_deletion_window(gchar * title, gchar * help,
				  GtkSignalFunc clist_realize_handler,
				  GtkSignalFunc delete_button_handler,
				  GtkSignalFunc recursive_delete_button_handler, gpointer user_data)
{
	GtkWidget *window;
	GtkWidget *tooltips;
	GtkWidget *button_label;
	GtkWidget *delete_button;
	GtkWidget *instruction_label;
	GtkWidget *recursive_delete_button;

	window = create_standard_selection_window(title, _("Delete"), help,
						  clist_realize_handler,
						  delete_button_handler,
						  recursive_delete_button_handler,
						  NULL,
						  on_deletion_window_clist_select_row,
						  on_deletion_window_clist_unselect_row, user_data);

	tooltips = gtk_object_get_data(GTK_OBJECT(window), "tooltips");
	instruction_label =
	    gtk_object_get_data(GTK_OBJECT(window), "selection_window_instruction_label");
	delete_button = gtk_object_get_data(GTK_OBJECT(window), "selection_window_next_button");
	recursive_delete_button =
	    gtk_object_get_data(GTK_OBJECT(window), "selection_window_prev_button");

	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), delete_button,
			     _("Delete the selected objects"), NULL);
	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), recursive_delete_button,
			     _
			     ("Delete the selected objects and delete all objects that comprise them"),
			     NULL);

	gtk_label_set_text(GTK_LABEL(instruction_label),
			   _
			   ("Make a selection from the list above and then choose one of the Delete buttons"));

	button_label = gtk_label_new(_("Recursive Delete"));
	gtk_misc_set_alignment(GTK_MISC(button_label), 0.5, 0.5);

	gtk_container_remove(GTK_CONTAINER(recursive_delete_button),
			     GTK_BIN(recursive_delete_button)->child);
	gtk_container_add(GTK_CONTAINER(recursive_delete_button), button_label);

	gtk_object_set_data(GTK_OBJECT(window), "selection_window_prev_button_label", button_label);
	gtk_widget_set_sensitive(recursive_delete_button, FALSE);

	gtk_widget_show(button_label);

	return window;
}

/*
 *
 *   inline static void get_delete_result_messages (gboolean, object_type_t, gchar **, gchar **)
 *   
 *   Description:
 *      This routine returns the messages displayed to the user as the
 *      result of a delete operation.
 * 
 *   Entry:
 *      recursive   - TRUE if recursive delete
 *      type        - the object type being deleted
 *      error_msg   - address of string to point to error message
 *      success_msg - address of string to point to success message
 *
 *   Exit:
 *      
 *
 */
inline static void get_delete_result_messages(gboolean recursive, object_type_t type,
					      gchar ** error_msg, gchar ** success_msg)
{
	switch (type) {
	case VOLUME:
		if (recursive)
			*error_msg =
			    _
			    ("An error was encountered attempting to recursively delete the volume.");
		else
			*error_msg = _("An error was encountered attempting to delete the volume.");
		*success_msg = _("The volume was successfully deleted.");
		break;

	case CONTAINER:
		if (recursive)
			*error_msg =
			    _
			    ("An error was encountered attempting to recursively delete the storage container.");
		else
			*error_msg =
			    _
			    ("An error was encountered attempting to delete the storage container.");
		*success_msg = _("The storage container was successfully deleted.");
		break;

	default:
		if (recursive)
			*error_msg =
			    _
			    ("An error was encountered attempting to recursively delete the storage object.");
		else
			*error_msg =
			    _("An error was encountered attempting to delete the storage object.");
		*success_msg = _("The storage object was successfully deleted.");
		break;
	}
}

/*
 *
 *   void delete_thing_button_clicked (GtkButton *, gboolean, object_type_t)
 *
 *   Description:
 *      This routine initiates a single or recursive delete using the handle of the 
 *      objects associated with rows selected. It then creates and 
 *      displays the results popup which destroys both windows when dismissed.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      recursive - TRUE if recursive delete wanted
 *      type      - the type of thing to delete (volume, container, storage object)
 *
 *   Exit:
 *      Deletes one or more things and results window is displayed 
 *
 */
void delete_thing_button_clicked(GtkButton * button, gboolean recursive, object_type_t type)
{
	gchar *error_msg;
	gchar *success_msg;
	GtkCList *clist;
	GSList *slist;
	object_handle_t handle;

	get_delete_result_messages(recursive, type, &error_msg, &success_msg);

	clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button), "selection_window_clist"));

	slist = get_clist_selection_data(clist);

	if (slist != NULL) {
		gint count = g_slist_length(slist);

		if (count > 0) {
			gint rc;
			gint row = 0;
			gboolean errors = FALSE;
			GSList *item = slist;
			GtkCList *rclist;
			GtkWidget *window = NULL;

			/*
			 * If we have multiple objects, create the multi-results window
			 * and populate it with object info. We must do it now as opposed
			 * to later due the operation potentially deleting the object
			 * or altering its name (such as with name shifts when deleting
			 * segments).
			 */

			if (count > 1) {
				window = create_multi_results_window(GTK_WIDGET(button),
								     _
								     ("Results of Multiple Delete Operations"),
								     &rclist);

				while (item != NULL) {
					handle = GPOINTER_TO_UINT(item->data);
					append_row_to_results_clist(rclist, handle);
					item = g_slist_next(item);
				}
				item = slist;
			}

			while (item != NULL) {
				handle = GPOINTER_TO_UINT(item->data);

				if (recursive)
					rc = evms_destroy(handle);
				else
					rc = evms_delete(handle);

				if (rc != SUCCESS)
					errors = TRUE;

				if (count == 1)
					display_selection_window_results(GTK_WIDGET(button), rc,
									 error_msg, success_msg);
				else
					set_results_clist_row(rclist, row, rc, success_msg);

				row++;
				item = g_slist_next(item);
			}

			if (count > 1) {
				gchar *status;

				if (errors)
					status = _("One or more Delete operations failed.");
				else
					status = _("All Delete operations completed successfully.");

				display_multi_results_window(window, status);
			}
		}

		g_slist_free(slist);
	}
}

/*
 *
 *   void on_delete_thing_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the deletion of a thing. It uses the
 *      handle of the object associated with the selected rows.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - 
 *
 *   Exit:
 *      See description.
 *
 */
void on_delete_thing_button_clicked(GtkButton * button, gpointer user_data)
{
	delete_thing_button_clicked(button, FALSE, GPOINTER_TO_UINT(user_data));
}

/*
 *
 *   void on_recursive_delete_thing_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine initiates the deletion of a thing and its deletable
 *      children. It uses the handle of the object associated with the 
 *      selected rows.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - 
 *
 *   Exit:
 *      See description.
 *
 */
void on_recursive_delete_thing_button_clicked(GtkButton * button, gpointer user_data)
{
	delete_thing_button_clicked(button, TRUE, GPOINTER_TO_UINT(user_data));
}

/*
 *
 *   void on_delete_thing_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of things that can be used on a delete operation.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - type of thing to delete (volume, container, object)
 *
 *   Exit:
 *      Selection list populated with acceptable things
 *
 */
void on_delete_thing_clist_realize(GtkWidget * widget, gpointer user_data)
{
	gint rc = 0;
	gchar *col_title;
	GtkCList *clist = GTK_CLIST(widget);
	object_type_t type = GPOINTER_TO_UINT(user_data);
	handle_array_t *things;

	switch (type) {
	case VOLUME:
		col_title = _("Logical Volume");
		rc = evms_get_volume_list(0, 0, 0, &things);
		break;

	case CONTAINER:
		col_title = _("Storage Container");
		rc = evms_get_container_list(0, 0, 0, &things);
		break;

	default:
		col_title = _("Storage Object");
		rc = evms_get_object_list(0, DATA_TYPE, 0, 0, TOPMOST, &things);
		break;
	}

	if (rc != SUCCESS) {
		log_error("%s: evms_get_xxxx_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		guint i;
		gboolean is_selected = (things->count == 1);

		set_selection_window_clist_column_titles(clist, _("Size"), col_title, NULL);

		for (i = 0; i < things->count; i++) {
			if (evms_can_delete(things->handle[i]) == 0)
				add_thing_to_selection_list(clist, things->handle[i], is_selected);
		}

		gtk_clist_set_selection_mode(clist, GTK_SELECTION_MULTIPLE);

		if (clist->rows == 1)
			gtk_clist_select_row(clist, 0, 0);

		evms_free(things);
	}
}

/*
 *
 *   void on_delete_thing_generic_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine handles initiating the destruction of a "thing".
 *      The handle to the thing is received as the user data. We 
 *      select the proper messages for the results screen and invoke 
 *      the routine that actually delete the thing.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_delete_thing_generic_button_clicked(GtkButton * button, gpointer user_data)
{
	object_type_t type;

	if (evms_get_handle_object_type(GPOINTER_TO_UINT(user_data), &type) == SUCCESS)
		delete_thing_button_clicked(button, FALSE, type);
}

/*
 *
 *   void on_recursive_delete_thing_generic_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine handles initiating the destruction of a "thing"
 *      and all its deletable children. The handle to the thing 
 *      to start with is received as the user data. We select the
 *      proper messages for the results screen and invoke 
 *      the routine that actually delete the thing and the children.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_recursive_delete_thing_generic_button_clicked(GtkButton * button, gpointer user_data)
{
	object_type_t type;

	if (evms_get_handle_object_type(GPOINTER_TO_UINT(user_data), &type) == SUCCESS)
		delete_thing_button_clicked(button, TRUE, type);
}

/*
 *
 *   void on_delete_thing_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the thing we intend to delete and
 *      allow the user to confirm the deletion of the thing.
 * 
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_delete_thing_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	object_type_t type;

	if (evms_get_handle_object_type(GPOINTER_TO_UINT(user_data), &type) == SUCCESS) {
		gchar *window_title;
		gchar *help_text;
		GtkWidget *window;

		switch (type) {
		case VOLUME:
			window_title = _("Delete Logical Volume");
			help_text = get_delete_volume_help_text();
			break;

		case EVMS_OBJECT:
			window_title = _("Delete Feature Object");
			help_text = get_delete_object_help_text();
			break;

		case REGION:
			window_title = _("Delete Storage Region");
			help_text = get_delete_region_help_text();
			break;

		case CONTAINER:
			window_title = _("Delete Storage Container");
			help_text = get_delete_container_help_text();
			break;

		case SEGMENT:
			window_title = _("Delete Disk Segment");
			help_text = get_delete_segment_help_text();
			break;

		default:
			window_title = _("Delete");
			help_text = NULL;
			break;
		}

		window = create_deletion_window(window_title,
						help_text,
						add_thing_as_selected_list_item,
						on_delete_thing_generic_button_clicked,
						on_recursive_delete_thing_generic_button_clicked,
						user_data);

		gtk_widget_show(window);
	}
}
