/*
 * GkSu -- Gtk+ Frontend to su
 * Copyright (C) 2002 Gustavo Noronha Silva
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <sys/types.h>
#include <gtk/gtk.h>
#include <locale.h>

#include <gksu.h>

#include "../config.h"
#include "defines.h"

#include "util.h"

#define GKSU_CMD (PREFIX "/bin/gksu")

void
run_gksu (GksuContext conf)
{
  gchar **cmd = g_malloc (sizeof(gchar*)*7);
  gint i = 0;

  cmd[i] = g_strdup (GKSU_CMD); i++;
  cmd[i] = g_strdup ("--user"); i++;
  cmd[i] = g_strdup (conf.user); i++;

  if (conf.login_shell)
    {
      cmd[i] = g_strdup ("--login");
      i++;
    }

  if (conf.keep_env)
    {
      cmd[i] = g_strdup ("--preserve-env");
      i++;
    }

  cmd[i] = g_strdup_printf("%s", conf.command); i++;
  cmd[i] = NULL;

  /* executes the command */
  if (execv (cmd[0], cmd) == -1)
    {
      gk_dialog (GTK_MESSAGE_ERROR, 
		 _("Unable to run %s: %s"),
		 cmd[0], strerror(errno));
    }

  for (i = 0 ; cmd[i] != NULL ; i++)
    g_free (cmd[i]);
  g_free(cmd);
}

void
response_ok_cb (GtkWidget *w, gpointer data)
{
  GtkWidget *dialog = (GtkWidget*)data;

  gtk_dialog_response (GTK_DIALOG(dialog),
		       GTK_RESPONSE_OK);
}

void
show_hide_advanced (GtkWidget *button, gpointer data)
{
  GtkWidget *parent, *tmp;

  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *check_login;
  GtkWidget *check_presenv;

  GksuContext *conf = (GksuContext*)data;
  gint response;

  parent = gtk_widget_get_parent (button);
  while ((tmp = gtk_widget_get_parent (parent)) != NULL)
    parent = tmp;

  dialog = gtk_dialog_new_with_buttons (_("Advanced options"), 
					GTK_WINDOW(parent), 
					GTK_DIALOG_MODAL,
					GTK_STOCK_CLOSE,
					GTK_RESPONSE_CLOSE,
					NULL);
  gtk_dialog_set_has_separator (GTK_DIALOG(dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER(dialog), 4);

  /* vbox points to the dialog's vbox */
  vbox = GTK_DIALOG(dialog)->vbox;
  gtk_box_set_spacing (GTK_BOX(vbox), 3);

  /* label */
  label = gtk_label_new ("");
  gtk_label_set_markup (GTK_LABEL(label), 
			_("<b>Options to use when changing user</b>"));
  gtk_box_pack_start (GTK_BOX(vbox), label, TRUE, TRUE, 5);
  gtk_widget_show (label);

  /* login shell? (--login) */
  check_login = gtk_check_button_new_with_mnemonic (_("_login shell"));
  if (conf->login_shell == TRUE) /* window may have been opened before */
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(check_login), TRUE);
  gtk_box_pack_start (GTK_BOX(vbox), check_login, TRUE, TRUE, 0);
  gtk_widget_show (check_login);

  /* preserve environment (--preserve-env) */
  check_presenv = 
    gtk_check_button_new_with_mnemonic (_("_preserve environment"));
  if (conf->keep_env == TRUE)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(check_presenv), TRUE);
  gtk_box_pack_start (GTK_BOX(vbox), check_presenv, TRUE, TRUE, 0);
  gtk_widget_show (check_presenv);

  response = gtk_dialog_run (GTK_DIALOG(dialog));

  if (response == GTK_RESPONSE_NONE)
    return;

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(check_login)))
    conf->login_shell = TRUE;
  else
    conf->login_shell = FALSE;

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(check_presenv)))
    conf->keep_env = TRUE;
  else
    conf->keep_env = FALSE;

  gtk_widget_destroy (dialog);
}

typedef struct {
  char *username;
  uid_t userid;
} TmpUser;

/*
 * Comparison function for g_list_sort()
 */
static int fill_with_user_list_cmp(gconstpointer a, gconstpointer b)
{
  return strcmp(((TmpUser *) a)->username, ((TmpUser *) b)->username);
}

/*
 * Fill combobox with an alphabetically sorted list of all users on the system
 */
static GList *fill_with_user_list(GtkWidget *combobox)
{
  GList *tmp = NULL, *list;
  TmpUser *tu;
  struct passwd *pw;

  setpwent();

  /* Get all the users info and store it temporary */
  while ((pw = getpwent())) {
    tu = g_new(TmpUser, 1);

    tu->username = g_strdup(pw->pw_name);
    tu->userid = pw->pw_uid;

    tmp = g_list_prepend(tmp, tu);
  }

  /* Sort it! */
  tmp = g_list_sort(tmp, fill_with_user_list_cmp);

  /* Add only the usernames */
  for (list = tmp; list; list = g_list_next(list)) {
    tu = list->data;

    gtk_combo_box_append_text (GTK_COMBO_BOX(combobox), tu->username);

    if (!strcmp (tu->username, "root"))
      gtk_combo_box_set_active (GTK_COMBO_BOX(combobox),
				g_list_position(tmp, list));

    g_free(tu);
  }

  g_list_free(tmp);
  endpwent();
}

int 
main (int argc, char **argv)
{
  GtkWidget *dialog;
  GtkWidget *hbox;
  GtkWidget *lvbox;
  GtkWidget *rvbox;
  GtkWidget *image;

  GtkWidget *label_cmd;
  GtkWidget *entry_cmd;

  GtkWidget *label_user;
  GtkWidget *combo_user;

  /* advanced stuff */
  GtkWidget *advanced_button;

  gint response;

  GksuContext conf;

  /* init the default values */
  conf.login_shell = FALSE;
  conf.keep_env = FALSE;

  setlocale (LC_ALL, "");
  bindtextdomain(PACKAGE_NAME, LOCALEDIR);  
  bind_textdomain_codeset (PACKAGE_NAME, "UTF-8");
  textdomain(PACKAGE_NAME);

  gtk_init (&argc, &argv);

  gtk_window_set_default_icon_from_file (DATA_DIR "/pixmaps/gksu-icon.png", NULL);

  dialog = gtk_dialog_new_with_buttons (_("Run program"), NULL, 0,
					GTK_STOCK_CANCEL,
					GTK_RESPONSE_CANCEL,
					GTK_STOCK_OK,
					GTK_RESPONSE_OK,
					NULL);
  gtk_dialog_set_has_separator (GTK_DIALOG(dialog), FALSE);

  /* horizontal box */
  hbox = gtk_hbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox),
		      hbox, TRUE, TRUE, 2);

  /* left vertical box */
  lvbox = gtk_vbox_new (FALSE, 2);
  gtk_box_pack_start (GTK_BOX(hbox), lvbox, TRUE, TRUE, 0);

  /* command */
  label_cmd = gtk_label_new (_("Run:"));
  gtk_label_set_justify (GTK_LABEL(label_cmd), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start (GTK_BOX(lvbox), label_cmd, TRUE, TRUE, 0);

  entry_cmd = gtk_entry_new ();
  gtk_signal_connect (GTK_OBJECT(entry_cmd), "activate",
		      GTK_SIGNAL_FUNC(response_ok_cb),
		      dialog);
  gtk_box_pack_start (GTK_BOX(lvbox), entry_cmd, TRUE, TRUE, 0);

  /* user name */
  label_user = gtk_label_new (_("As user:"));
  gtk_label_set_justify (GTK_LABEL(label_user), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start (GTK_BOX(lvbox), label_user, TRUE, TRUE, 0);

  combo_user = gtk_combo_box_new_text ();
  fill_with_user_list (combo_user);

  gtk_box_pack_start (GTK_BOX(lvbox), combo_user, TRUE, TRUE, 0);

  /* right vertical box */
  rvbox = gtk_vbox_new (FALSE, 2);
  gtk_box_pack_start (GTK_BOX(hbox), rvbox, TRUE, TRUE, 0);

  /* image */
  image = gtk_image_new_from_file (DATA_DIR"/pixmaps/gksu-icon.png");
  gtk_box_pack_start (GTK_BOX(rvbox), image, TRUE, TRUE, 0);

  /* advanced button */
  advanced_button = gtk_button_new_with_mnemonic (_("_Advanced"));
  g_signal_connect (G_OBJECT(advanced_button), "clicked",
		    G_CALLBACK(show_hide_advanced), &conf);
  gtk_box_pack_start (GTK_BOX(rvbox), advanced_button, TRUE, FALSE, 0);

  /* let the magic begin! */
  gtk_widget_show_all (dialog);

  while (TRUE)
    {
      response = gtk_dialog_run (GTK_DIALOG(dialog));

      switch (response)
	{
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
	case GTK_RESPONSE_NONE:
	  exit (0);
	}

      conf.command = gtk_editable_get_chars (GTK_EDITABLE(entry_cmd), 0, -1);
      conf.user = gtk_combo_box_get_active_text (GTK_COMBO_BOX(combo_user));

      if (!strcmp (conf.command, ""))
	{
	  gk_dialog (GTK_MESSAGE_ERROR, _("Missing command to run."));
	}
      else
	{
	  gtk_widget_destroy (dialog);
	  run_gksu (conf);

	  break;
	}
    }

  return 0;
}
