/**
 * GMyth Library
 *
 * @file gmyth/gmyth_common.c
 * 
 * @brief <p> This file contains basic common functions for the gmyth library.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Leonardo Sobral Cunha <leonardo.cunha@indt.org.br>
 * @author Rosfran Borges <rosfran.borges@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gmyth_common.h"
#include "gmyth_debug.h"
#include "gmyth_util.h"

static void     free_channel_data(gpointer data, gpointer user_data);
static void     free_program_data(gpointer data, gpointer user_data);

/** 
* Frees the memory allocated to the GMythChannelInfo objects inside list.
* The list memory is also released by g_list_free(). If LIST is NULL it
* simply returns.
* 
* @param list the GList containing a list of GMythChannelInfo to free.
*/
void
gmyth_free_channel_list(GList * list)
{
    g_return_if_fail(list != NULL);

    g_list_foreach(list, free_channel_data, NULL);
    g_list_free(list);
}

/** 
 * Frees the memory allocated to the GMythProgramInfo objects inside list.
 * The list memory is also released by g_list_free(). If list is NULL it
 * simply returns.
 * 
 * @param list the GList containing a list of GMythProgramInfo to free.
 */
void
gmyth_free_program_list(GList * list)
{
    g_return_if_fail(list != NULL);

    g_list_foreach(list, free_program_data, NULL);
    g_list_free(list);
}

void
gmyth_channel_info_free(GMythChannelInfo * channel)
{
    g_return_if_fail(channel != NULL);

    if (channel->channel_num)
        g_string_free(channel->channel_num, TRUE);

    if (channel->channel_name)
        g_string_free(channel->channel_name, TRUE);

    if (channel->channel_icon)
        g_string_free(channel->channel_icon, TRUE);

    g_free(channel);
}

/**
 * Prints the channel info to the standard output. The gmyth debug must be enabled.
 * @param channel_info the GMythChannelInfo instance
 */
void
gmyth_channel_info_print(GMythChannelInfo * channel_info)
{
#ifdef GMYTH_USE_DEBUG
    if (channel_info != NULL) {
        g_return_if_fail(channel_info->channel_name != NULL);
        g_return_if_fail(channel_info->channel_num != NULL);

        gmyth_debug("ChannelInfo (Name, Num, ID) = (%s, %s, %d)",
                    channel_info->channel_name->str,
                    channel_info->channel_num->str,
                    channel_info->channel_ID);

    }
#endif
}

/**
 * Prints the program info to the standard output. The gmyth debug must be enabled.
 * @param channel_info the GMythProgramInfo instance
 */
void
gmyth_program_info_print(GMythProgramInfo * program_info)
{
#ifdef GMYTH_USE_DEBUG
    g_return_if_fail(program_info);

    gmyth_debug("ProgramInfo\n\tTitle = %s\n\t"
                "Description = %s\n\t"
                "Start time= %s\t"
                "End time = %s\n"
                "Path name = %s\n"
                "File size = %lld\n",
                program_info->title ? program_info->title->str : "NULL",
                program_info->description ? program_info->description->
                str : "NULL",
                gmyth_util_time_to_string_from_time_val(program_info->
                                                        startts),
                gmyth_util_time_to_string_from_time_val(program_info->
                                                        endts),
                program_info->pathname ? program_info->pathname->
                str : "NULL", program_info->filesize);
#endif
}

static void
free_channel_data(gpointer data, gpointer user_data)
{
    /*
     * Frees the GMythChannelInfo structure 
     */
    GMythChannelInfo *channel = (GMythChannelInfo *) data;

    gmyth_channel_info_free(channel);
}

static void
free_program_data(gpointer data, gpointer user_data)
{
    g_return_if_fail(data != NULL);

    g_object_unref((GMythProgramInfo *) data);
}
