/**
 * GMyth Library
 *
 * @file gmyth/gmyth_recorder.c
 * 
 * @brief <p> GMythRecorder defines functions for playing live tv.
 *
 * The remote encoder is used by gmyth_tvplayer to setup livetv. 
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Hallyson Luiz de Morais Melo <hallyson.melo@indt.org.br>
 * @author Rosfran Borges <rosfran.borges@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gmyth_recorder.h"

#include <assert.h>

#include "gmyth_stringlist.h"
#include "gmyth_util.h"
#include "gmyth_common.h"
#include "gmyth_debug.h"

#define	 GMYTHTV_RECORDER_HEADER			"QUERY_RECORDER"

static void     gmyth_recorder_class_init(GMythRecorderClass * klass);
static void     gmyth_recorder_init(GMythRecorder * object);

static void     gmyth_recorder_dispose(GObject * object);
static void     gmyth_recorder_finalize(GObject * object);

G_DEFINE_TYPE(GMythRecorder, gmyth_recorder, G_TYPE_OBJECT)
    static void     gmyth_recorder_class_init(GMythRecorderClass * klass)
{
    GObjectClass   *gobject_class;

    gobject_class = (GObjectClass *) klass;

    gobject_class->dispose = gmyth_recorder_dispose;
    gobject_class->finalize = gmyth_recorder_finalize;
}

static void
gmyth_recorder_init(GMythRecorder * gmyth_remote_encoder)
{
}

static void
gmyth_recorder_dispose(GObject * object)
{
    GMythRecorder  *recorder = GMYTH_RECORDER(object);

    gmyth_recorder_close(recorder);

    if (recorder->mutex != NULL) {
        g_mutex_free(recorder->mutex);
        recorder->mutex = NULL;
    }

    if (recorder->myth_socket != NULL) {
        g_object_unref(recorder->myth_socket);
        recorder->myth_socket = NULL;
    }

    if (recorder->progs_info_list != NULL)
        gmyth_free_program_list(recorder->progs_info_list);

    if (recorder->hostname != NULL)
        g_string_free(recorder->hostname, TRUE);

    G_OBJECT_CLASS(gmyth_recorder_parent_class)->dispose(object);
}

static void
gmyth_recorder_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);

    G_OBJECT_CLASS(gmyth_recorder_parent_class)->finalize(object);
}

void
gmyth_recorder_close(GMythRecorder * recorder)
{
    if (recorder != NULL && recorder->recorder_num != -1) {
        g_mutex_lock(recorder->mutex);

        gmyth_recorder_stop_playing(recorder);
        gmyth_recorder_stop_livetv(recorder);
        gmyth_recorder_finish_recording(recorder);
        gmyth_recorder_free_tuner(recorder);

        g_mutex_unlock(recorder->mutex);
    }
}

/** Creates a new instance of GMythRecorder.
 * 
 * @return a new instance of GMythRecorder.
 */
GMythRecorder  *
gmyth_recorder_new(int num, GString * hostname, gshort port)
{
    GMythRecorder  *encoder =
        GMYTH_RECORDER(g_object_new(GMYTH_RECORDER_TYPE, FALSE));

    encoder->recorder_num = num;
    encoder->hostname = g_string_new(hostname->str);
    encoder->port = port;

    encoder->mutex = g_mutex_new();

    encoder->progs_info_list = NULL;

    return encoder;
}

/** Configures the remote encoder instance connecting it to Mythtv backend.
 * 
 * @param recorder the GMythRecorder instance.
 * 
 * @return TRUE if successfull, FALSE if any error happens.
 */
gboolean
gmyth_recorder_setup(GMythRecorder * recorder)
{
    assert(recorder);
    gmyth_debug("[%s] Creating socket and connecting to backend",
                __FUNCTION__);

    if (recorder->myth_socket == NULL) {
        recorder->myth_socket = gmyth_socket_new();

        if (!gmyth_socket_connect_to_backend(recorder->myth_socket,
                                             recorder->hostname->str,
                                             recorder->port, TRUE)) {
            gmyth_debug
                ("GMythRemoteEncoder: Connection to backend failed");
            return FALSE;
        }
    } else {
        gmyth_debug("Remote encoder socket already created\n");
    }

    return TRUE;
}

/** Sends the SPAWN_LIVETV command through Mythtv protocol. This command
 * requests the backend to start capturing TV content.
 * 
 * @param recorder The GMythRecorder instance.
 * @param tvchain_id The tvchain unique id.
 * @return true if success, false if any error happens.
 */
gboolean
gmyth_recorder_spawntv(GMythRecorder * recorder, GString * tvchain_id)
{
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);
    gboolean        ret = TRUE;

    gmyth_debug("[%s] Spawntv with tvchain_id = %s", __FUNCTION__,
                tvchain_id->str);

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, tmp_str);
    g_string_free(tmp_str, TRUE);

    gmyth_string_list_append_char_array(str_list, "SPAWN_LIVETV");

    gmyth_string_list_append_string(str_list, tvchain_id);
    gmyth_string_list_append_int(str_list, 0);  // PIP = FALSE (0)

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    tmp_str = gmyth_string_list_get_string(str_list, 0);

    if (tmp_str == NULL) {
        gmyth_debug("[%s] Spawntv request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

    if (g_ascii_strncasecmp(tmp_str->str, "ok", 2)) {
        gmyth_debug("[%s] Spawntv request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

  cleanup:
    g_mutex_unlock(recorder->mutex);

    g_string_free(tmp_str, TRUE);
    g_object_unref(str_list);

    return ret;
}

/** 
 * Sends the SPAWN_LIVETV command through Mythtv protocol. This command
 * requests the backend to start capturing TV content, but it doesn't need
 * the TV chain ID.
 * 
 * @param recorder The GMythRecorder instance.
 * @return true if success, false if any error happens.
 */
gboolean
gmyth_recorder_spawntv_no_tvchain(GMythRecorder * recorder)
{
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);
    gboolean        ret = TRUE;

    gmyth_debug("[%s] Spawntv, no TV chain!", __FUNCTION__);

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, tmp_str);
    g_string_free(tmp_str, TRUE);

    gmyth_string_list_append_char_array(str_list, "SPAWN_LIVETV");

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    tmp_str = gmyth_string_list_get_string(str_list, 0);

    if (tmp_str == NULL) {
        gmyth_debug("[%s] Spawntv request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

    if (g_ascii_strncasecmp(tmp_str->str, "ok", 2)) {
        gmyth_debug("[%s] Spawntv request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

  cleanup:
    g_mutex_unlock(recorder->mutex);

    g_string_free(tmp_str, TRUE);
    g_object_unref(str_list);

    return ret;
}

/** Sends the command STOP_LIVETV to Mythtv backend.
 * 
 * @param recorder the GMythRecorder instance.
 * @return true if success, false if any error happens.
 */
gboolean
gmyth_recorder_stop_livetv(GMythRecorder * recorder)
{
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);
    gboolean        ret = TRUE;

    gmyth_debug("[%s]", __FUNCTION__);

    str_list = gmyth_string_list_new();

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, tmp_str);
    g_string_free(tmp_str, TRUE);

    gmyth_string_list_append_char_array(str_list, "STOP_LIVETV");

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    tmp_str = gmyth_string_list_get_string(str_list, 0);

    if (g_ascii_strncasecmp(tmp_str->str, "ok", 2)) {
        gmyth_debug("[%s] Stop livetv request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

  cleanup:
    g_string_free(tmp_str, TRUE);
    g_object_unref(str_list);

    return ret;
}

/** Sends the FRONTEND_READY command through Mythtv protocol. This command
 * advertises the backend to start capturing TV content.
 * 
 * @param recorder The GMythRecorder instance.
 * @return TRUE if success, FALSE if any error happens.
 */
gboolean
gmyth_recorder_send_frontend_ready_command(GMythRecorder * recorder)
{
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);
    gboolean        ret = TRUE;

    gmyth_debug("[%s] FRONTEND_READY with recorder id = %d", __FUNCTION__,
                recorder->recorder_num);

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, tmp_str);
    g_string_free(tmp_str, TRUE);

    gmyth_string_list_append_char_array(str_list, "FRONTEND_READY");

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    tmp_str = gmyth_string_list_get_string(str_list, 0);

    if (tmp_str == NULL) {
        gmyth_debug
            ("[%s] FRONTEND_READY command request couldn't returns, reason: %s",
             __FUNCTION__, tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

    if (g_ascii_strncasecmp(tmp_str->str, "ok", 2)) {
        gmyth_debug("[%s] FRONTEND_READY request returned %s",
                    __FUNCTION__, tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

  cleanup:
    g_mutex_unlock(recorder->mutex);
    g_string_free(tmp_str, TRUE);
    g_object_unref(str_list);

    return ret;
}

/** Send a CHECK_CHANNEL command request to the backend, in order to find if a 
 * certain channel actually exists.
 * 
 * @param recorder The GMythRecorder instance.
 * @param channel	 The new channel to be checked (string format).
 * @return true if success, false if any error happens.
 */
gboolean
gmyth_recorder_check_channel_name(GMythRecorder * recorder,
                                  gchar * channel)
{
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);
    gboolean        ret = TRUE;

    gmyth_debug("[%s] CHECK_CHANNEL with channel = %s", __FUNCTION__,
                channel);

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, tmp_str);
    g_string_free(tmp_str, TRUE);

    gmyth_string_list_append_char_array(str_list, "CHECK_CHANNEL");

    gmyth_string_list_append_char_array(str_list, channel);

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    tmp_str = gmyth_string_list_get_string(str_list, 0);

    if (tmp_str == NULL) {
        gmyth_debug("[%s] CHECK_CHANNEL request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

    if (g_ascii_strncasecmp(tmp_str->str, "ok", 2) == 0
        || g_ascii_strncasecmp(tmp_str->str, "0", 1) == 0) {
        gmyth_debug("[%s] CHECK_CHANNEL request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

  cleanup:
    g_mutex_unlock(recorder->mutex);
    g_string_free(tmp_str, TRUE);
    g_object_unref(str_list);

    return ret;
}

/** Send a CHECK_CHANNEL command request to the backend, in order to find if a 
 * certain channel actually exists.
 * 
 * @param recorder The GMythRecorder instance.
 * @param channel	 The new channel to be checked (decimal integer value).
 * @return true if success, false if any error happens.
 */
gboolean
gmyth_recorder_check_channel(GMythRecorder * recorder, gint channel)
{
    return gmyth_recorder_check_channel_name(recorder,
                                             g_strdup_printf("%d",
                                                             channel));
}

/** Send a SET_CHANNEL command request to the backend, to start streaming on another 
 * TV content channel.
 * 
 * @param recorder The GMythRecorder instance.
 * @param channel	 The new channel to be loaded.
 * @return true if success, false if any error happens.
 */
gboolean
gmyth_recorder_set_channel(GMythRecorder * recorder, gint channel)
{
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);
    gboolean        ret = TRUE;

    gmyth_debug("[%s] SET_CHANNEL with channel = %d", __FUNCTION__,
                channel);

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, tmp_str);
    g_string_free(tmp_str, TRUE);

    gmyth_string_list_append_char_array(str_list, "SET_CHANNEL");

    gmyth_string_list_append_int(str_list, channel);

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    tmp_str = gmyth_string_list_get_string(str_list, 0);

    if (tmp_str == NULL) {
        gmyth_debug("[%s] SET_CHANNEL request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

    if (g_ascii_strncasecmp(tmp_str->str, "ok", 2)) {
        gmyth_debug("[%s] SET_CHANNEL request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

  cleanup:
    g_mutex_unlock(recorder->mutex);
    g_string_free(tmp_str, TRUE);
    g_object_unref(str_list);

    return ret;
}

/** Send a SET_CHANNEL command request to the backend, to start streaming on another 
 * TV content channel.
 * 
 * @param recorder The GMythRecorder instance.
 * @param channel	 The new channel to be loaded.
 * @return true if success, false if any error happens.
 */
gboolean
gmyth_recorder_set_channel_name(GMythRecorder * recorder,
                                const gchar * channel)
{
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);
    gboolean        ret = TRUE;

    gmyth_debug("[%s] SET_CHANNEL with channel name = %s", __FUNCTION__,
                channel);

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, tmp_str);
    g_string_free(tmp_str, TRUE);

    gmyth_string_list_append_char_array(str_list, "SET_CHANNEL");
    gmyth_string_list_append_char_array(str_list, channel);

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    tmp_str = gmyth_string_list_get_string(str_list, 0);

    if (tmp_str == NULL) {
        gmyth_debug("[%s] SET_CHANNEL name request returned NULL!",
                    __FUNCTION__);
        ret = FALSE;
        goto cleanup;
    }

    if (tmp_str != NULL && g_ascii_strncasecmp(tmp_str->str, "ok", 2)
        /*
         * || g_ascii_strtoull( tmp_str->str, NULL, 10 ) == 0 
         */
        ) {
        g_warning("[%s] SET_CHANNEL name request returned not ok",
                  __FUNCTION__);
        ret = FALSE;
        goto cleanup;
    }

  cleanup:
    g_mutex_unlock(recorder->mutex);
    g_string_free(tmp_str, TRUE);
    g_object_unref(str_list);

    return ret;
}

/**
 * Changes the channel of the actual Recorder.
 * 
 * CHANNEL_DIRECTION_UP       - Go up one channel in the listing
 *
 * CHANNEL_DIRECTION_DOWN     - Go down one channel in the listing
 *
 * CHANNEL_DIRECTION_FAVORITE - Go to the next favorite channel
 *
 * CHANNEL_DIRECTION_SAME     - Stay
 * 
 * @param recorder 	 The GMythRecorder instance.
 * @param direction	 The new channel direction where to move to.
 * @return true if success, false if any error happens.
 */
gboolean
gmyth_recorder_change_channel(GMythRecorder * recorder,
                              const GMythRecorderChannelChangeDirection
                              direction)
{
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);
    gboolean        ret = TRUE;

    gmyth_debug("[%s] CHANGE_CHANNEL to the channel direction = %u",
                __FUNCTION__, direction);

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, tmp_str);
    g_string_free(tmp_str, TRUE);

    gmyth_string_list_append_char_array(str_list, "CHANGE_CHANNEL");
    gmyth_string_list_append_int(str_list, direction);

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    tmp_str = gmyth_string_list_get_string(str_list, 0);

    if (tmp_str == NULL) {
        gmyth_debug("[%s] CHANGE_CHANNEL name request returned %s",
                    __FUNCTION__, tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

    if (g_ascii_strncasecmp(tmp_str->str, "ok", 2)
        || g_ascii_strtoull(tmp_str->str, NULL, 10) == 0) {
        gmyth_debug("[%s] CHANGE_CHANNEL name request returned %s",
                    __FUNCTION__, tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

  cleanup:
    g_mutex_unlock(recorder->mutex);
    g_string_free(tmp_str, TRUE);
    g_object_unref(str_list);

    return ret;
}

/** 
 * Gets the channel's list from the MythTV backend server.
 * 
 * @param recorder The GMythRecorder instance.
 * 
 * @return a GList* instance with all the channel names.
 */
GList          *
gmyth_recorder_get_channel_list(GMythRecorder * recorder)
{

    GList          *channel_list = NULL;
    gchar          *channel = NULL;
    guint           i;

    for (i = 0; i < 1000; i++) {
        channel = g_strdup_printf("%u", i);

        if (gmyth_recorder_check_channel_name(recorder, channel)) {
            channel_list = g_list_append(channel_list, g_strdup(channel));
        }

    }                           /* for - channel list */

    g_free(channel);

    return channel_list;

}

/** Send a PAUSE command request to the backend, to pause streaming on another 
 * TV content channel.
 * 
 * @param recorder The GMythRecorder instance.
 * @return true if success, false if any error happens.
 */
gboolean
gmyth_recorder_pause_recording(GMythRecorder * recorder)
{
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);
    gboolean        ret = TRUE;

    gmyth_debug("[%s] PAUSE", __FUNCTION__);

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, tmp_str);
    g_string_free(tmp_str, TRUE);

    gmyth_string_list_append_char_array(str_list, "PAUSE");

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    tmp_str = gmyth_string_list_get_string(str_list, 0);

    if (tmp_str == NULL) {
        gmyth_debug("[%s] PAUSE name request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

    if (g_ascii_strncasecmp(tmp_str->str, "ok", 2)) {
        gmyth_debug("[%s] PAUSE name request returned %s", __FUNCTION__,
                    tmp_str->str);
        ret = FALSE;
        goto cleanup;
    }

  cleanup:
    g_mutex_unlock(recorder->mutex);
    g_string_free(tmp_str, TRUE);
    g_object_unref(str_list);

    return ret;
}

static          gboolean
gmyth_recorder_find_if_program_exists(GMythRecorder * recorder,
                                      GMythProgramInfo * prog)
{
    GList          *lst = NULL;

    g_return_val_if_fail(recorder != NULL
                         && recorder->progs_info_list != NULL, FALSE);

    for (lst = recorder->progs_info_list; lst != NULL;
         lst = g_list_next(lst)) {
        gmyth_debug("Got program info from list = [%s]",
                    gmyth_program_info_to_string((GMythProgramInfo *)
                                                 lst->data));
        if (gmyth_program_info_is_equals
            (prog, (GMythProgramInfo *) lst->data))
            return TRUE;
    }

    return FALSE;
}

/**
 * Requests the actual program info from the MythTV backend server.
 * 
 * @param recorder The GMythRecorder instance.
 * @return The actual program info.
 */
GMythProgramInfo *
gmyth_recorder_get_current_program_info(GMythRecorder * recorder)
{
    GMythStringList *str_list = NULL;
    GMythProgramInfo *program_info = NULL;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, tmp_str);

    if (recorder->myth_socket->mythtv_version >= 26)
        gmyth_string_list_append_char_array(str_list,
                                            "GET_CURRENT_RECORDING");
    else
        gmyth_string_list_append_char_array(str_list, "GET_PROGRAM_INFO");

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    if (str_list == NULL) {
        gmyth_debug
            ("[%s] GET_PROGRAM_INFO request returned. Error getting program info, string list equals to NULL!",
             __FUNCTION__);
        goto cleanup;
    }

    program_info = gmyth_program_info_from_string_list(str_list);

    if (NULL == program_info || NULL == program_info->pathname
        || program_info->pathname->len <= 0) {
        gmyth_debug
            ("GET_PROGRAM_INFO request returned. Error getting program info, it is equals to NULL!!!");

        if (program_info)
            g_object_unref(program_info);

        program_info = NULL;

        goto cleanup;
    }

    if (!gmyth_recorder_find_if_program_exists(recorder, program_info))
        recorder->progs_info_list =
            g_list_append(recorder->progs_info_list,
                          g_object_ref(program_info));
  cleanup:
    g_mutex_unlock(recorder->mutex);
    g_string_free(tmp_str, TRUE);
    g_object_unref(str_list);

    return program_info;
}

/**
 * Requests the actual program info from the MythTV backend server.
 * 
 * @param rec_id The GMythRecorder record number.
 * @return The GMythRecorder instance.
 */
GMythRecorder  *
gmyth_recorder_get_recorder_from_num(gint rec_id)
{
    GMythRecorder  *recorder = NULL;
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new("GET_RECORDER_FROM_NUM");
    gint            command_size = 0;

    gchar          *recorder_host = NULL;
    gint            recorder_port;

    str_list = gmyth_string_list_new();

    /*
     * g_string_append_printf ( tmp_str, " %d", recorder->recorder_num ); 
     */

    g_mutex_lock(recorder->mutex);

    gmyth_string_list_append_string(str_list, tmp_str);

    gmyth_string_list_append_int(str_list, rec_id);

    command_size =
        gmyth_socket_sendreceive_stringlist(recorder->myth_socket,
                                            str_list);

    if (str_list == NULL) {
        gmyth_debug
            ("[%s] GET_RECORDER_FROM_NUM request returned. Error getting recorder number %d, it is equals to NULL!!!",
             __FUNCTION__, rec_id);
        return NULL;
    }

    if (command_size > 0) {
        recorder_host = gmyth_string_list_get_char_array(str_list, 0);
        recorder_port = gmyth_string_list_get_int(str_list, 1);

        if (g_strstr_len(recorder_host, strlen(recorder_host), "nohost")
            != NULL) {
            gmyth_debug
                ("No available recorder with the recorder ID number %d!",
                 rec_id);
        } else {

            recorder = gmyth_recorder_new(rec_id,
                                          g_string_new(recorder_host),
                                          (gshort) recorder_port);

            if (NULL == recorder) {
                gmyth_debug
                    ("[%s] GET_RECORDER_FROM_NUM request returned. Error getting recorder number %d, it is equals to NULL!!!",
                     __FUNCTION__, rec_id);
                g_object_unref(recorder);
                return NULL;
            }

        }

    } else {
        gmyth_debug
            ("Cannot find a valuable recorder with the recorder ID number %d, backend server error!",
             rec_id);
    }

    g_mutex_unlock(recorder->mutex);

    g_object_unref(str_list);

    g_string_free(tmp_str, TRUE);

    g_free(recorder_host);

    return recorder;

}

/**
 * Requests the actual program info from the MythTV backend server.
 * 
 * @param recorder The GMythRecorder instance.
 * @param direction The direction to move based on the current channel (forward, backward,
 *            up, down).
 * 
 * @return The GMythProgramInfo next program info instance.
 */
GMythProgramInfo *
gmyth_recorder_get_next_program_info(GMythRecorder * recorder,
                                     const GMythRecorderBrowseDirection
                                     direction)
{
    GMythProgramInfo *actual_proginfo = NULL;
    GMythProgramInfo *program_info = NULL;
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);

    gchar          *date = NULL;
    struct tm      *tm = NULL;
    time_t          t;

    actual_proginfo = gmyth_recorder_get_current_program_info(recorder);

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    t = time(NULL);
    tm = localtime(&t);
    date = g_strdup_printf("%.4d%.2d%.2d%.2d%.2d%.2d", tm->tm_year + 1900,
                           tm->tm_mon + 1, tm->tm_mday, tm->tm_hour,
                           tm->tm_min, tm->tm_sec);

    gmyth_string_list_append_string(str_list, tmp_str);
    gmyth_string_list_append_char_array(str_list, "GET_NEXT_PROGRAM_INFO");
    gmyth_string_list_append_string(str_list, actual_proginfo->channame);
    gmyth_string_list_append_int(str_list, actual_proginfo->channel_id);
    gmyth_string_list_append_int(str_list, direction);
    gmyth_string_list_append_char_array(str_list, date);

    if (gmyth_socket_sendreceive_stringlist
        (recorder->myth_socket, str_list)
        > 0) {

        if (str_list == NULL) {
            gmyth_debug
                ("[%s] GET_NEXT_PROGRAM_INFO request returned. Error getting program info, it is equals to NULL!!!",
                 __FUNCTION__);
            goto done;
        }
        program_info =
            gmyth_program_info_from_string_list_next_prog(str_list);

        if (NULL == program_info) {
            gmyth_debug
                ("[%s] GET_NEXT_PROGRAM_INFO request returned. Error getting next program info, it is equals to NULL!!!",
                 __FUNCTION__);
            g_object_unref(program_info);
            goto done;
        }

        if (                    /* ( program_info->chanid != NULL &&
                                 * strlen( program_info->chanid->str ) > 0 
                                 * * * * * * ) && */
               (program_info->chansign != NULL
                && strlen(program_info->chansign->str) > 0)) {
            gmyth_debug("OK!!! Got the next program info... [%s].",
                        program_info->chansign->str);
        } else {
            gmyth_debug
                ("GET_NEXT_PROGRAM_INFO request returned. Error getting next program info, it is equals to NULL!!!");
            g_object_unref(program_info);
            program_info = NULL;
        }

    }
    /*
     * if 
     */
  done:

    g_mutex_unlock(recorder->mutex);

    if (actual_proginfo != NULL)
        g_object_unref(actual_proginfo);

    if (str_list != NULL)
        g_object_unref(str_list);

    if (tmp_str != NULL)
        g_string_free(tmp_str, TRUE);

    if (date != NULL)
        g_free(date);
    // if ( tm != NULL)
    // g_free (tm);

    return program_info;
}

/**
 * Requests the program info from the MythTV backend server, based on its 
 * channel name.
 * 
 * @param recorder The GMythRecorder instance.
 * @return The GMythProgramInfo next program info instance.
 */
GMythProgramInfo *
gmyth_recorder_get_program_info_from_channel_name(GMythRecorder * recorder,
                                                  const gchar * channel)
{
    // GMythProgramInfo* actual_proginfo= NULL;
    GMythProgramInfo *program_info = NULL;
    GMythStringList *str_list;
    GString        *tmp_str = g_string_new(GMYTHTV_RECORDER_HEADER);

    /*
     * gchar *date = NULL; struct tm *tm = NULL; time_t t;
     * 
     * actual_proginfo =
     * gmyth_recorder_get_current_program_info(recorder); 
     */

    str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(tmp_str, " %d", recorder->recorder_num);

    /*
     * t = time(NULL); tm = localtime(&t); date =
     * g_strdup_printf("%.4d%.2d%.2d%.2d%.2d%.2d", tm->tm_year + 1900,
     * tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec); 
     */

    gmyth_string_list_append_string(str_list, tmp_str);
    gmyth_string_list_append_char_array(str_list, "GET_NEXT_PROGRAM_INFO");
    gmyth_string_list_append_char_array(str_list, channel);
    gmyth_string_list_append_char_array(str_list, "0");
    gmyth_string_list_append_int(str_list, BROWSE_DIRECTION_UP);
    gmyth_string_list_append_char_array(str_list, "0");

    do {

        if (str_list != NULL &&
            gmyth_socket_sendreceive_stringlist(recorder->myth_socket,
                                                str_list) > 0) {

            if (str_list == NULL) {
                gmyth_debug
                    ("[%s] GET_NEXT_PROGRAM_INFO request returned. Error getting program info, it is equals to NULL!!!",
                     __FUNCTION__);
                goto done;
            }
            program_info =
                gmyth_program_info_from_string_list_next_prog(str_list);

            if (NULL == program_info) {
                gmyth_debug
                    ("[%s] GET_NEXT_PROGRAM_INFO request returned. Error getting next program info, it is equals to NULL!!!",
                     __FUNCTION__);
                g_object_unref(program_info);
                goto done;
            }

            if (                /* ( program_info->chanid != NULL &&
                                 * strlen( program_info->chanid->str ) > 0 
                                 * * * * * * ) && */
                   (program_info->chansign != NULL
                    && strlen(program_info->chansign->str) > 0)) {
                gmyth_debug("OK!!! Got the next program info... [%s].",
                            program_info->chansign->str);
            } else {
                gmyth_debug
                    ("GET_NEXT_PROGRAM_INFO request returned. Error getting "
                     "next program info, it is equals to NULL!!!");
                g_object_unref(program_info);
                program_info = NULL;
            }

        }
        /*
         * if 
         */
    }
    while (str_list != NULL);

  done:

    g_mutex_unlock(recorder->mutex);

    if (str_list != NULL)
        g_object_unref(str_list);

    if (tmp_str != NULL)
        g_string_free(tmp_str, TRUE);

    return program_info;
}

/**
 * Requests the actual remote file position on a LiveTV instance.
 * 
 * @param recorder The GMythRecorder instance.
 * 
 * @return The position, in bytes, of the offset to the read header.
 */
gint64
gmyth_recorder_get_file_position(GMythRecorder * recorder)
{
    gint64          pos = 0;
    GString        *query = g_string_new(GMYTHTV_RECORDER_HEADER);

    GMythStringList *str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(query, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, query);
    gmyth_string_list_append_char_array(str_list, "GET_FILE_POSITION");

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    if (str_list != NULL && gmyth_string_list_length(str_list) > 0) {
        GString        *str = NULL;

        if ((str = gmyth_string_list_get_string(str_list, 0)) != NULL
            && strstr(str->str, "bad") == NULL)
            pos = gmyth_string_list_get_int64(str_list, 0);
        g_string_free(str, TRUE);
    }
#ifndef GMYTHTV_ENABLE_DEBUG
    gmyth_debug("[%s] Got file position = %lld\n", __FUNCTION__, pos);
#endif

    g_mutex_unlock(recorder->mutex);

    if (str_list != NULL)
        g_object_unref(str_list);

    g_string_free(query, TRUE);

    return pos;
}

/**
 * Asks MythTV backend server about if it started to record the remote file.
 * 
 * @param recorder The GMythRecorder instance.
 * 
 * @return <code>true</code>, if the actual remote file is bein recorded.
 */
gboolean
gmyth_recorder_is_recording(GMythRecorder * recorder)
{
    gboolean        ret = TRUE;

    g_return_val_if_fail(recorder != NULL, FALSE);

    GMythStringList *str_list = gmyth_string_list_new();
    GString        *message = g_string_new("");

    g_mutex_lock(recorder->mutex);

    g_string_printf(message, "%s %d", GMYTHTV_RECORDER_HEADER,
                    recorder->recorder_num);
    gmyth_string_list_append_string(str_list, message);
    gmyth_string_list_append_char_array(str_list, "IS_RECORDING");

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    if (str_list != NULL && gmyth_string_list_length(str_list) > 0) {
        GString        *str = NULL;

        if ((str = gmyth_string_list_get_string(str_list, 0)) != NULL
            && strcmp(str->str, "bad") != 0) {
            gint            is_rec =
                gmyth_string_list_get_int(str_list, 0);

            if (is_rec != 0)
                ret = TRUE;
            else
                ret = FALSE;
        }
        g_string_free(str, TRUE);
    }

    gmyth_debug("%s, stream is %s being recorded!\n", ret ? "YES" : "NO",
                ret ? "" : "NOT");
    // g_static_mutex_unlock (&mutex);

    g_mutex_unlock(recorder->mutex);

    if (str_list != NULL)
        g_object_unref(str_list);

    g_string_free(message, TRUE);

    return ret;

}

/**
 * Finish remote file recording process.
 * 
 * @param recorder The GMythRecorder instance.
 * 
 * @return <code>true</code>, if the recording had been actually closed.
 */
gboolean
gmyth_recorder_finish_recording(GMythRecorder * recorder)
{
    gboolean        ret = TRUE;

    g_return_val_if_fail(recorder != NULL, FALSE);

    GMythStringList *str_list = gmyth_string_list_new();
    GString        *message = g_string_new("");

    g_string_printf(message, "%s %d", GMYTHTV_RECORDER_HEADER,
                    recorder->recorder_num);
    gmyth_string_list_append_string(str_list, message);
    gmyth_string_list_append_char_array(str_list, "FINISH_RECORDING");

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    if (str_list != NULL && gmyth_string_list_length(str_list) > 0) {
        GString        *str = NULL;

        if ((str = gmyth_string_list_get_string(str_list, 0)) != NULL &&
            strcmp(str->str, "ok") != 0) {
            gint            is_rec =
                gmyth_string_list_get_int(str_list, 0);

            if (is_rec != 0)
                ret = TRUE;
            else
                ret = FALSE;
        }
        g_string_free(str, TRUE);
    }

    gmyth_debug("%s, stream is %s finished!\n", ret ? "YES" : "NO",
                ret ? "" : "NOT");
    // g_static_mutex_unlock (&mutex);

    if (str_list != NULL)
        g_object_unref(str_list);

    g_string_free(message, TRUE);

    return ret;
}


/**
 * Stops playing the remote file.
 * 
 * @param recorder The GMythRecorder instance.
 * 
 * @return <code>true</code>, if the recording had been actually stopped.
 */
gboolean
gmyth_recorder_stop_playing(GMythRecorder * recorder)
{
    gboolean        ret = TRUE;

    g_return_val_if_fail(recorder != NULL, FALSE);

    GMythStringList *str_list = gmyth_string_list_new();
    GString        *message = g_string_new("");

    g_string_printf(message, "%s %d", GMYTHTV_RECORDER_HEADER,
                    recorder->recorder_num);
    gmyth_string_list_append_string(str_list, message);
    gmyth_string_list_append_char_array(str_list, "STOP_PLAYING");

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    if (str_list != NULL && gmyth_string_list_length(str_list) > 0) {
        GString        *str = NULL;

        if ((str = gmyth_string_list_get_string(str_list, 0)) != NULL &&
            strcmp(str->str, "ok") != 0) {
            gint            is_rec =
                gmyth_string_list_get_int(str_list, 0);

            if (is_rec != 0)
                ret = TRUE;
            else
                ret = FALSE;
        }
        g_string_free(str, TRUE);
    }

    gmyth_debug("%s, stream is %s stopped!\n", ret ? "YES" : "NO",
                ret ? "" : "NOT");

    if (str_list != NULL)
        g_object_unref(str_list);

    g_string_free(message, TRUE);

    return ret;
}

/**
 * Free the tuner responsible for recording this channel.
 * 
 * @param recorder The GMythRecorder instance.
 * 
 * @return <code>true</code>, if the tuner had been freed.
 */
gboolean
gmyth_recorder_free_tuner(GMythRecorder * recorder)
{
    gboolean        ret = TRUE;

    g_return_val_if_fail(recorder != NULL, FALSE);

    GMythStringList *str_list = gmyth_string_list_new();
    GString        *message = g_string_new("");

    g_string_printf(message, "%s %d", "FREE_TUNER",
                    recorder->recorder_num);
    gmyth_string_list_append_string(str_list, message);

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    if (str_list != NULL && gmyth_string_list_length(str_list) > 0) {
        GString        *str = NULL;

        if ((str = gmyth_string_list_get_string(str_list, 0)) != NULL &&
            g_ascii_strncasecmp(str->str, "ok", 2) != 0) {
            gint            is_rec =
                gmyth_string_list_get_int(str_list, 0);

            if (is_rec != 0)
                ret = TRUE;
            else
                ret = FALSE;
        }
        g_string_free(str, TRUE);
    }

    gmyth_debug("%s, tuner is %s freed!\n", ret ? "YES" : "NO",
                ret ? "" : "NOT");

    if (str_list != NULL)
        g_object_unref(str_list);

    g_string_free(message, TRUE);

    return ret;
}

/**
 * Asks the MythTV backend server about the frame rate 
 * of this LiveTV instance.
 * 
 * @param recorder The GMythRecorder instance.
 * 
 * @return The framerate (double value) of the current video.
 */
gdouble
gmyth_recorder_get_framerate(GMythRecorder * recorder)
{
    gdouble         fr = 0.0f;
    GString        *query = g_string_new(GMYTHTV_RECORDER_HEADER);

    GMythStringList *str_list = gmyth_string_list_new();

    g_mutex_lock(recorder->mutex);

    g_string_append_printf(query, " %d", recorder->recorder_num);

    gmyth_string_list_append_string(str_list, query);
    gmyth_string_list_append_char_array(str_list, "GET_FRAMERATE");

    gmyth_socket_sendreceive_stringlist(recorder->myth_socket, str_list);

    if (str_list != NULL && gmyth_string_list_length(str_list) > 0) {
        GString        *str = NULL;

        if ((str = gmyth_string_list_get_string(str_list, 0)) != NULL
            && strstr(str->str, "bad") == NULL)
            fr = g_ascii_strtod(str->str, NULL);

        g_string_free(str, TRUE);
    }
#ifndef GMYTHTV_ENABLE_DEBUG
    gmyth_debug("[%s] Got file position = %f\n", __FUNCTION__, fr);
#endif

    g_mutex_unlock(recorder->mutex);

    if (str_list != NULL)
        g_object_unref(str_list);

    g_string_free(query, TRUE);

    return fr;

}
