# Copyright (C) 2004-2005 Ross Burton <ross@burtonini.com>
#               2005 Canonical
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place, Suite 330, Boston, MA 02111-1307 USA


import pygtk; pygtk.require("2.0")
import gtk
import gtk.glade
import gtk.gdk
import gobject
import gconf

import LaunchpadIntegration
import gettext
from gettext import gettext as _

#setup gettext
app="gnome-app-install"
gettext.textdomain(app)
gettext.bindtextdomain(app)
gtk.glade.textdomain(app)
gtk.glade.bindtextdomain(app)


import time
import stat
import glob
import re
import subprocess
import tempfile
import warnings
import os
import sys
import shutil
from datetime import datetime

import dbus
import dbus.service
import dbus.glib

from warnings import warn
warnings.filterwarnings("ignore", "ICON:.*", UserWarning)
warnings.filterwarnings("ignore", "apt API not stable yet", FutureWarning)
import apt
import apt_pkg

# from update-manager, needs to be factored out
from SoftwareProperties import SoftwareProperties
from SoftwareProperties.aptsources import SourcesList, is_mirror

# internal imports
from DialogNewlyInstalled import DialogNewlyInstalled
from DialogPendingChanges import DialogPendingChanges
from DialogMultipleApps import DialogMultipleApps
from DialogUnavailable import DialogUnavailable
from DialogProprietary import DialogProprietary
from PackageWorker import PackageWorker
from Menu import MenuItem, Application, ApplicationMenu

# we support both moz and gtkhtml2 right now
from BrowserView import GtkHtml2BrowserView as BrowserView
#from BrowserView import MozEmbedBrowserView as BrowserView
#from BrowserView import DumpBrowserView as BrowserView
from SimpleGladeApp import SimpleGladeApp
from Progress import GtkOpProgressWindow
from Util import *
import common

# this const must DIE DIE DIE
DISTRO_VERSION = "dapper"

# this is used to guess the desktop environment that the application
# was written for
desktop_environment_mapping = {
    ("kdelibs4c2a","python-kde3") :
    _("This application is designed for the KDE desktop environment"),
    ("libgnome2-0","python-gnome2") :
    _("This application is designed for the GNOME desktop environment"),
    ("libgnustep-base1.11",):
    _("This application is designed for the Gnustep desktop environment"),
    ("libxfce4util-1",) :
    _("This application is designed for the XFCE desktop environment"),
}
    

class MyCache(apt.Cache):
    def pkgDependsOn(self, pkgname, depends_name):
        """ check if a given pkg depends on a given dependencie """
        if not self.has_key(pkgname):
            return False
        pkg = self[pkgname]
        candver = self._depcache.GetCandidateVer(pkg._pkg)
	if candver == None:
		return False
	dependslist = candver.DependsList
	for dep in dependslist.keys():
            if dep == "Depends" or dep == "PreDepends":
                # get the list of each dependency object
                for depVerList in dependslist[dep]:
                    for z in depVerList:
                        # get all TargetVersions of
                        # the dependency object
                        for tpkg in z.AllTargets():
                            if depends_name == tpkg.ParentPkg.Name:
                                return True
        return False
        

class AppInstallDbusControler(dbus.service.Object):
    """ this is a helper to provide the AppInstallIFace """
    def __init__(self, parent, bus_name,
                 object_path='/org/freedesktop/AppInstallObject'):
        dbus.service.Object.__init__(self, bus_name, object_path)
        self.parent = parent

    @dbus.service.method('org.freedesktop.AppInstallIFace')
    def bringToFront(self):
        self.parent.window_main.present()
        return True
    
class AppInstall(SimpleGladeApp):

    def __init__(self, datadir, desktopdir, arguments=None):
        self.setupDbus()

        # setup a default icon
        icons = common.ToughIconTheme()
        gtk.window_set_default_icon(icons.load_icon("gnome-settings-default-applications", 32, 0))

        SimpleGladeApp.__init__(self, domain="gnome-app-install",
                                path=datadir+"/gnome-app-install.glade")

        self.channelsdir = desktopdir+"/channels"       
        self.datadir = datadir
        self.desktopdir = desktopdir

        # this holds the filenames of the generated html pages
        self.info_pages = {}

        # sensitive stuff
        self.button_apply.set_sensitive(False)

        # Connect search widgets
        self.left_side = self.left_vbox
                
        # create the treeview
        self.setupTreeview()


        # now show the main window ...
        self.window_main.show()

        # setup the gconf backend
        self.config = gconf.client_get_default()
        self.config.add_dir ("/apps/gnome-app-install", gconf.CLIENT_PRELOAD_NONE)

        # ... and open the cache
        self.updateCache()

        # this is a set() of packagenames that contain multiple applications
        # if a pkgname is in the set, a dialog was already displayed to the
        # user about this (ugly ...)
        self.multiple_pkgs_seen = set()
        
        # handle arguments
        if arguments is not None and len(arguments) > 1:
            for arg in arguments[1:]:
                if arg.startswith("--mime-type="):
                    self.search_entry.set_text("mime-type:%s" % arg[len("--mime-type="):])
                    self.searchClicked(None)
        # create a worker that does the actual installing etc
        self.packageWorker = PackageWorker()
        self.search_timeout_id = 0

        # used for restore_state
        self.last_toggle = None

        # now check if the cache is up-to-date
        time_cache = 0
        for f in glob.glob("/var/lib/apt/lists/*Packages"):
            mt = os.stat(f)[stat.ST_MTIME]
            ct = os.stat(f)[stat.ST_CTIME] 
            if mt > time_cache:
               time_cache = mt
            if ct > time_cache:
               time_cache = ct
        time_source = os.stat("/etc/apt/sources.list")[stat.ST_MTIME]
        for f in glob.glob("/etc/apt/sources.list.d/*.list"):
            mt = os.stat(f)[stat.ST_MTIME]
            ct = os.stat(f)[stat.ST_CTIME]
            if mt > time_source:
                time_source = mt
            if ct > time_source:
                time_source = ct
        # FIXME: problem: what can happen is that the sources.list is modified
        #        but we only get I-M-S hits and the mtime of the Packages
        #        files do not change
        #print "cache:  ", time_cache
        #print "source: ", time_source
        if time_cache < time_source:
            self.dialog_cache_outdated.set_transient_for(self.window_main)
            self.dialog_cache_outdated.realize()
            self.dialog_cache_outdated.window.set_functions(gtk.gdk.FUNC_MOVE)
            res = self.dialog_cache_outdated.run()
            self.dialog_cache_outdated.hide()
            if res == gtk.RESPONSE_YES:
                self.reloadSources()

        # generate and show page
        self.intro_page = self.generateIntro()
        self.browser.loadUri("file://" + self.intro_page)
        # generate failedSearch page for later
        self.failedSearchPage = self.generateFailedSearch()

        self.search_entry.grab_focus()


    def setupDbus(self):
        """ this sets up a dbus listener if none is installed alread """
        # check if there is another g-a-i already and if not setup one
        # listening on dbus
        try:
            bus = dbus.SessionBus()
        except:
            print "warning: could not initiate dbus"
            return
        proxy_obj = bus.get_object('org.freedesktop.AppInstall', '/org/freedesktop/AppInstallObject')
        iface = dbus.Interface(proxy_obj, 'org.freedesktop.AppInstallIFace')
        try:
            iface.bringToFront()
            #print "send bringToFront"
            sys.exit(0)
        except dbus.DBusException, e:
            print "no listening object (%s) "% e
            bus_name = dbus.service.BusName('org.freedesktop.AppInstall',bus)
            self.dbusControler = AppInstallDbusControler(self, bus_name)

     
    def create_custom_browser_view(self, s1, s2, i1, i2):
        self.browser = BrowserView()
        self.browser.connect("submit", self.on_browser_submit)
        self.browser.show()
        return self.browser

    def on_window_main_key_press_event(self, widget, event):
        #print "on_window_main_key_press_event()"
        # from /usr/include/gtk-2.0/gdk/gdkkeysyms.h
        GDK_q = 0x071
        if (event.state & gtk.gdk.CONTROL_MASK) and event.keyval == GDK_q:
            self.on_window_main_delete_event(self.window_main, None)

    def on_checkbutton_show_proprietary_toggled(self, button):
        #print "on_checkbutton_show_proprietary_toggled()"
        value = button.get_active()
        self.menu.show_proprietary = value
        self.config.set_bool("/apps/gnome-app-install/show_proprietary", value)

    def on_checkbutton_show_unsupported_toggled(self, button):
        #print "on_checkbutton_show_unsupported_toggled()"
        value = button.get_active()
        self.menu.show_unsupported = value
        self.config.set_bool("/apps/gnome-app-install/show_unsupported", value)

    # install toggle on the treeview
    def on_install_toggle(self, renderer, path):
        #print "on_install_toggle: %s %s" % (renderer, path)
        model = self.treeview_packages.get_model()
        (type, name, item) = model[path]
        #print "on_install_toggle(): %s %s %s" % (type, name, item)
        # first check if the operation is save
        pkg = item.pkgname
        if not (self.cache.has_key(pkg) and self.cache[pkg].candidateDownloadable):
            for it in self.cache._cache.FileList:
                # FIXME: we need to exclude cdroms here. the problem is
                # how to detect if a PkgFileIterator is pointing to a cdrom
                if it.Component != "" and it.Component == item.component:
                    # warn that this app is not available on this plattform
                    header = _("'%s' is not available in any software "
                               "channel") % pkg
                    msg = _("The application might not support your system "
                            "architecture.")
                    d = gtk.MessageDialog(parent=self.window_main,
                                          flags=gtk.DIALOG_MODAL,
                                          type=gtk.MESSAGE_ERROR,
                                          buttons=gtk.BUTTONS_CLOSE)
                    d.set_title("")
                    d.set_markup("<big><b>%s</b></big>\n\n%s" % (header, msg))
                    d.realize()
                    d.window.set_functions(gtk.gdk.FUNC_MOVE)
                    d.run()
                    d.destroy()
                    return
            self.saveState()
            if self.addChannel(item):
                self.last_toggle = name
                self.restoreState()
            return
        if self.cache[pkg].isInstalled:
            # check if it can be removed savly
            # FIXME: Provide a list of the corresponding packages or
            #        apps
            self.cache[pkg].markDelete(autoFix=False)
            if self.cache._depcache.BrokenCount > 0:
                d = gtk.MessageDialog(parent=self.window_main,
                                      flags=gtk.DIALOG_MODAL,
                                      type=gtk.MESSAGE_ERROR,
                                      buttons=gtk.BUTTONS_CLOSE)
                d.set_title("")
                d.set_markup("<big><b>%s</b></big>\n\n%s" % \
                             ((_("Cannot remove '%s'") % pkg),
                              (_("One or more applications depend on '%s'. "
                                "To remove '%s' and the dependent applications,"
                                " please switch to the advanced software "
                                "manager.") % (pkg, pkg))))
                d.realize()
                d.window.set_functions(gtk.gdk.FUNC_MOVE)
                d.run()
                d.destroy()
                self.cache[pkg].markKeep()
                assert self.cache._depcache.BrokenCount == 0
                assert self.cache._depcache.DelCount == 0
                return
            self.cache[pkg].markKeep()
            # FIXME: those assert may be a bit too strong,
            # we may just rebuild the cache if something is
            # wrong
            assert self.cache._depcache.BrokenCount == 0
            assert self.cache._depcache.DelCount == 0
        else:
            # check if it can be installed savely
            apt_error = False
            try:
                self.cache[pkg].markInstall(autoFix=True)
            except SystemError:
                apt_error = True
            if self.cache._depcache.BrokenCount > 0 or \
               self.cache._depcache.DelCount > 0 or apt_error:
                # FIXME: Resolve conflicts
                d = gtk.MessageDialog(parent=self.window_main,
                                      flags=gtk.DIALOG_MODAL,
                                      type=gtk.MESSAGE_ERROR,
                                      buttons=gtk.BUTTONS_CLOSE)
                d.set_title("")
                d.set_markup("<big><b>%s</b></big>\n\n%s" % (
                             (_("Cannot install '%s'") % pkg),
                             (_("This application conflicts with other "
                                "installed software. To install '%s' "
                                "the conflicting software "
                                "must be removed before.\n\n"
                                "Switch to the advanced mode to resolve this "
                                "conflict.") % pkg)))
                d.realize()
                d.window.set_functions(gtk.gdk.FUNC_MOVE)
                d.run()
                d.destroy()
                # reset the cache
                # FIXME: a "pkgSimulateInstall,remove"  thing would
                # be nice
                for p in self.cache.keys():
                    self.cache[p].markKeep()
                # FIXME: those assert may be a bit too strong,
                # we may just rebuild the cache if something is
                # wrong
                assert self.cache._depcache.BrokenCount == 0
                assert self.cache._depcache.DelCount == 0
                return
        # invert the current selection
        item.toInstall = not item.toInstall
        # check if the package provides multiple desktop applications
        if len(self.menu.pkg_to_app[item.pkgname]) > 1:
            apps = self.menu.pkg_to_app[item.pkgname]
            # update the install-status of the other apps
            for app in apps:
                app.toInstall = item.toInstall
            # hack: redraw the treeview (to update the toggle icons after the
            #       tree-model was changed)
            self.treeview_packages.queue_draw()
            # show something to the user (if he hasn't already seen it)
            if not item.pkgname in self.multiple_pkgs_seen:
                dia = DialogMultipleApps(self.datadir, self.window_main, \
                                         apps, item.name)
                dia.run()
                dia.hide()
                self.multiple_pkgs_seen.add(item.pkgname)

        self.button_apply.set_sensitive(self.menu.isChanged())

    def addChannel(self, item):
        """Ask for confirmation to add the missing channel or
           component of the current selected application"""
        if item.proprietary and item.channel:
            dia = DialogProprietary(self.datadir, self.window_main, item)
        else:
            dia = DialogUnavailable(self.datadir, self.window_main, item)
        res = dia.run()
        dia.hide()
        # the user canceld
        if res != gtk.RESPONSE_OK:
            return False
        # let go
        if item.component:
            self.enableComponent(item.component)
            # FIXME: make sure to fix this after release in a more elegant
            #        and generic way
            # we check if it is multiverse and if it is and we don't
            # have universe already, we add universe too (because
            # multiverse depends on universe)
            if item.component == "multiverse":
                for it in self.cache._cache.FileList:
                    if it.Component != "" and it.Component == "universe":
                        break
                else:
                    self.enableComponent("universe")
        elif item.channel:
            self.enableChannel(item.channel)
        else:
            # should never happen
            print "ERROR: addChannel() called without channel or component"
            return False
        # now do the reload
        self.reloadSources()
        return True


    def setupTreeview(self):
        def visibility_magic(cell_layout, renderer, model,iter, visible_types):
            row_type = model.get_value(iter, COL_TYPE)
            renderer.set_property("visible", row_type in visible_types)

        def package_view_func(cell_layout, renderer, model, iter):
            visibility_magic(cell_layout, renderer, model, iter,
                             (TYPE_PACKAGE,))
            row_type = model.get_value(iter, COL_TYPE)
            if row_type != TYPE_PACKAGE:
                return
            app = model.get_value(iter, COL_ITEM)
            name = app.name
            desc = app.description
            current = app.isInstalled
            future = app.toInstall
            available = app.available
            if current != future:
                markup = "<b>%s</b>\n<small>%s</small>" % (name, desc)
            else:
                markup = "%s\n<small>%s</small>" % (name, desc)
            renderer.set_property("markup", markup)

        def toggle_cell_func(column, cell, model, iter):
            row_type = model.get_value(iter, COL_TYPE)
            if row_type != TYPE_PACKAGE:
                cell.set_property("visible", False)
                return
            menuitem = model.get_value(iter, COL_ITEM)
            cell.set_property("active", menuitem.toInstall)
            cell.set_property("visible", True)

        def icon_cell_func(column, cell, model, iter):
            (type, menuitem) = model.get(iter, COL_TYPE, COL_ITEM)
            if menuitem == None or menuitem.icon == None:
                cell.set_property("pixbuf", None)
                cell.set_property("visible", False)
                return
            cell.set_property("pixbuf", menuitem.icon)
            cell.set_property("visible", True)

        # we have two columns, one that shows some text (header, application)
        # and one that displays if the application is going to be
        # installed/removed
        column = gtk.TreeViewColumn("")
        check_column = gtk.TreeViewColumn(_("Installed"))

        self.treeview_packages.set_search_column(COL_NAME)

        # check boxes
        self.toggle_render = gtk.CellRendererToggle()
        self.store_toggle_id = self.toggle_render.connect('toggled', self.on_install_toggle)
        #self.search_toggle_id = self.toggle_render.connect('toggled', on_install_toggle, self.search_store, self)
        #self.toggle_render.handler_block(self.search_toggle_id)
        self.toggle_render.set_property("xalign", 0.3)
        check_column.pack_start(self.toggle_render, False)
        #check_column.add_attribute(self.toggle_render, "active", COL_TO_INSTALL)
        #check_column.set_cell_data_func (self.toggle_render, visibility_magic, TYPE_PACKAGE)
        check_column.set_cell_data_func (self.toggle_render, toggle_cell_func)
        
        
        # program icons
        render = gtk.CellRendererPixbuf()
        column.pack_start(render, False)
        #column.add_attribute(render, "pixbuf", COL_ICON)
        column.set_cell_data_func (render, icon_cell_func)

        # menu group names
        render = gtk.CellRendererText()
        render.set_property("scale", 1.0)
        render.set_property("weight", 700)
        column.pack_start(render, True)
        column.add_attribute(render, "markup", COL_NAME)
        column.set_cell_data_func (render, visibility_magic, (TYPE_GROUP,))

        # package names
        render = gtk.CellRendererText()
        column.pack_start(render, True)
        column.add_attribute(render, "markup", COL_NAME)
        column.set_cell_data_func (render, package_view_func)
        
        # headers
        render = gtk.CellRendererText()
        render.set_property("xpad", 4)
        render.set_property("ypad", 4)        
        render.set_property("scale", 1.3)
        render.set_property("weight", 700)
        #self.connect("style-set", lambda parent, old, widget: widget.set_property ("foreground-gdk", parent.get_style().fg[gtk.STATE_SELECTED]), render)
        #self.connect("style-set", lambda parent, old, widget: widget.set_property ("background-gdk", parent.get_style().bg[gtk.STATE_SELECTED]), render)
        column.pack_start(render, False)
        column.add_attribute(render, "text", COL_NAME)
        column.set_cell_data_func (render, visibility_magic, (TYPE_HEADER,))

        # indent headers (i think)
        render = gtk.CellRendererText()
        render.set_property("width", self.treeview_packages.style_get_property("expander-size") + self.treeview_packages.style_get_property("horizontal-separator"))
        column.pack_start(render, False)
        column.set_cell_data_func (render, visibility_magic, (TYPE_HEADER,))

        self.treeview_packages.append_column(check_column)
        self.treeview_packages.append_column(column)

        # categories
        column = gtk.TreeViewColumn("")
        # program icons
        render = gtk.CellRendererPixbuf()
        column.pack_start(render, False)
        #column.add_attribute(render, "pixbuf", COL_ICON)
        column.set_cell_data_func (render, icon_cell_func)

        # menu group names
        self.treeview_categories.set_search_column(COL_NAME)
        render = gtk.CellRendererText()
        render.set_property("scale", 1.0)
        column.pack_start(render, True)
        column.add_attribute(render, "markup", COL_NAME)
        column.set_cell_data_func (render, visibility_magic, (TYPE_GROUP,))
        self.treeview_categories.append_column(column)

        
        
    def on_browser_submit(self, browser, action, method, value):
        """Callback for the signal "submit" from the html form of the
           application description"""
        if method == "add_repo":
            self.saveState()
            if self.addChannel():
                # FUN: gtk *must* return from on_browser_submit with the same
                #      page loaded in the gtkhtml view as before the signal was
                #      send. we queue the reload here therefore and it will be
                #      run on the next gtk event-loop run
                gobject.timeout_add(1,lambda: self.restoreState())

    def saveState(self):
        """ save the current state of the app """
        # store the pkgs that are marked for removal or installation
        (self.to_add, self.to_rm) = self.menu.getChanges()
        (self.cursor_categories_path,x) = self.treeview_categories.get_cursor()
        model = self.treeview_packages.get_model()
        (packages_path, x) = self.treeview_packages.get_cursor()
        if packages_path:
            it = model.get_iter(packages_path)
            self.cursor_pkgname = model.get_value(it, COL_NAME)
        else:
            self.cursor_pkgname = None

    def restoreState(self):
        """ restore the current state of the app """
        # set category
        self.treeview_categories.set_cursor(self.cursor_categories_path)
        model = self.treeview_packages.get_model()
        # reapply search
        query = self.search_entry.get_text()
        if query:
            self.on_search_timeout()
        # remark all packages that were marked for installation
        for item in self.to_add:
            if self.cache.has_key(item.pkgname):
                try:
                    self.cache[item.pkgname].markInstall(autoFix=True)
                except SystemError:
                    continue
                # set the state of the corresponing apps
                apps = self.menu.pkg_to_app[item.pkgname]
                for app in apps:
                    app.toInstall = item.toInstall
        # remark all packages that were marked for removal
        for item in self.to_rm:
            if self.cache.has_key(item.pkgname):
                try:
                    self.cache[item.pkgname].markDelete(autoFix=True)
                except SystemError:
                    continue
                # set the state of the corresponing apps
                apps = self.menu.pkg_to_app[item.pkgname]
                for app in apps:
                    app.toInstall = item.toInstall
        # redraw the treeview so that all check buttons are updated
        self.treeview_packages.queue_draw()

        # find package
        for it in iterate_tree_model(model,model.get_iter_first()):
            name = model.get_value(it, COL_NAME)
            # if the app corresponds to the one toggled before toggle it again
            if name == self.last_toggle:
                self.last_toggle = None
                path = model.get_path(it)
                self.treeview_packages.set_cursor(path)
                self.on_install_toggle(None, path)
                break
            # if the app correpsonds to the one selected before select it again
            if name == self.cursor_pkgname and self.last_toggle == None:
                path = model.get_path(it)
                self.treeview_packages.set_cursor(path)
                break

    def updateCache(self):
        self.window_main.set_sensitive(False)

        self.window_main.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))
        while gtk.events_pending():
            gtk.main_iteration()
            
        progress = GtkOpProgressWindow(self.glade,self.window_main)
        try:
            self.cache = MyCache(progress)
        except:
            # show an error dialog if something went wrong with the cache
            header = _("Failed to check for installed and available applications")
            msg = _("This is a major failure of your software " \
                    "management system. Check the file permissions and "\
                    "correctness of the file '/etc/apt/sources.list' and "\
                    "reload the software information: 'sudo apt-get update'.")
            d = gtk.MessageDialog(parent=self.window_main,
                                  flags=gtk.DIALOG_MODAL,
                                  type=gtk.MESSAGE_ERROR,
                                  buttons=gtk.BUTTONS_CLOSE)
            d.set_title("")
            d.set_markup("<big><b>%s</b></big>\n\n%s" % (header, msg))
            d.realize()
            d.window.set_functions(gtk.gdk.FUNC_MOVE)
            d.run()
            d.destory()
            sys.exit(1)

        # Setup the treeview
        self.menu = ApplicationMenu(self.desktopdir, self.cache,
                                    self.treeview_categories,
                                    self.treeview_packages,
                                    progress)

        # move to "All" category per default
        self.treeview_categories.set_cursor((0,))

        # clean the info pages
        for file in self.info_pages.values():
            os.remove(file)
        self.info_pages.clear()
        
        adj = self.scrolled_window.get_vadjustment()
        adj.set_value(0)

        # set the toggle buttons
        val = self.config.get_bool("/apps/gnome-app-install/show_unsupported")
        if val == True or val == False:
            self.checkbutton_show_unsupported.set_active(val)
            self.menu.show_unsupported = val
        val = self.config.get_bool("/apps/gnome-app-install/show_proprietary")
        if val == True or val == False:
            self.checkbutton_show_proprietary.set_active(val)
            self.menu.show_proprietary = val
       
        self.window_main.window.set_cursor(None)
        self.window_main.set_sensitive(True)
    
    
    def ignoreChanges(self):
        """
        If any changes have been made, ask the user to apply them and return
        a value based on the status.
        Returns True if the changes should be thrown away and False otherwise
        """
        if not self.menu.isChanged():
            return True
        (to_add, to_rm) = self.menu.getChanges()
        # FIXME: move this set_markup into the dialog itself
        dia = DialogPendingChanges(self.datadir, self.window_main,
                                   to_add, to_rm)
        header =_("Apply changes to installed applications before closing?")
        msg = _("If you do not apply your changes they will be lost "\
                "permanently.")
        dia.label_pending.set_markup("<big><b>%s</b></big>\n\n%s" % \
                                     (header, msg))
        dia.button_ignore_changes.set_label(_("_Close Without Applying"))
        dia.button_ignore_changes.show()
        dia.dialog_pending_changes.realize()
        dia.dialog_pending_changes.window.set_functions(gtk.gdk.FUNC_MOVE)
        res = dia.run()
        dia.hide()
        return res

    
    # ----------------------------
    # Main window button callbacks
    # ----------------------------
    
    def on_button_help_clicked(self, widget):
        subprocess.Popen(["/usr/bin/yelp", "ghelp:gnome-app-install"])

    def on_button_advanced_clicked(self, widget):
        if self.menu.isChanged():
            ret = self.ignoreChanges()
            if ret == gtk.RESPONSE_APPLY:
                if not self.applyChanges(final=True):
                    return
            elif ret == gtk.RESPONSE_CANCEL:
                return False

        # Here we are either ignoring or have applied, so run Synaptic
        # TODO: activate startup notification (needs wrapping)
        os.execl("/usr/bin/gksu","gksu", "--desktop", 
                 "/usr/share/applications/synaptic.desktop",
                 "synaptic")

    def on_repositories_item_activate(self, widget):
        """ start gnome-software preferences """
        dia = SoftwareProperties(parent=self.window_main)
        dia.run()
        dia.hide()
        if dia.modified:
            # FIXME: ask about reloading here
            self.reloadSources()

    def applyChanges(self, final=False):
        #print "do_apply()"
        (to_add, to_rm) = self.menu.getChanges()
        # Set a busy cursor
        self.window_main.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))
        while gtk.events_pending():
            gtk.main_iteration()
        # Get the selections delta for the changes and apply them
        ret = self.packageWorker.perform_action(self.window_main, to_add, to_rm)

        # error from gksu
        if ret != 0:
            self.window_main.window.set_cursor(None)
            return False
        
        # Reload the APT cache and treeview
        if final != True:
            self.updateCache()

        self.button_apply.set_sensitive(self.menu.isChanged())
        
        # Show window with newly installed programs
        #self.checkNewStore() # only show things that successfully installed
        if len(to_add) > 0:
            dia = DialogNewlyInstalled(self.datadir, self.window_main,
                                       to_add, self.cache)
            dia.run()
            dia.hide()
        
        # And reset the cursor
        self.window_main.window.set_cursor(None)
        return True

    def on_button_ok_clicked(self, button):
        # nothing changed, exit
        if not self.menu.isChanged():
            self.quit()
        # something changed, only exit if the changes have
        # succesfully been applied (otherwise cancel)
        if self.confirmChanges():
            if self.applyChanges(final=True):
                self.quit()

    def confirmChanges(self):
        (to_add, to_rm) = self.menu.getChanges()
        dia = DialogPendingChanges(self.datadir, self.window_main,
                                   to_add, to_rm)
        # FIXME: move this inside the dialog class, we show a different
        # text for a quit dialog and a approve dialog
        header = _("Apply the following changes?")
        msg = _("Please take a final look through the list of "\
                "applications that will be installed or removed.")
        dia.label_pending.set_markup("<big><b>%s</b></big>\n\n%s" % \
                                     (header, msg))
        res = dia.run()
        dia.hide()
        if res != gtk.RESPONSE_APPLY:
            # anything but ok makes us leave here
            return False
        else:
            return True

    def on_button_apply_clicked(self, button):
        ret = self.confirmChanges()
        if ret == True :
            self.applyChanges()

    def on_search_timeout(self):
        query = self.search_entry.get_text()
        if query.lstrip() != "":
            self.menu.searchTerms = query.lower().split(" ")
        else:
            self.menu.searchTerms = []
        self.menu._refilter()
        if len(self.menu.treeview_packages.get_model()) == 0:
            print "len is zero"
            self.browser.loadUri("file://"+self.failedSearchPage)

    def on_search_entry_changed(self, widget):
        #print "on_search_entry_changed()"
        if self.search_timeout_id > 0:
            gobject.source_remove(self.search_timeout_id)
        self.search_timeout_id = gobject.timeout_add(500,self.on_search_timeout)
            
    def on_button_clear_clicked(self, button):
        self.search_entry.set_text("")
        # reset the search
        self.menu.search(None)
        # Point the treeview back to the original store
        #self.treeview_packages.set_model(self.menu.store)
        #self.treeview_packages.set_rules_hint(False)
        self.button_clear.set_sensitive(False)

    def on_item_about_activate(self, button):
        from Version import VERSION
        self.dialog_about.set_version(VERSION)
        self.dialog_about.run()
        self.dialog_about.hide()

    def on_reload_activate(self, item):
        self.reloadSources()
        
    def on_button_cancel_clicked(self, item):
        self.quit()

                
    def reloadSources(self):
        self.window_main.set_sensitive(False)
        ret = self.packageWorker.perform_action(self.window_main,
                                                action=PackageWorker.UPDATE)
        self.updateCache()
        self.window_main.set_sensitive(True)
        return ret

    def enableChannel(self, channel):
        """ enables a channel with 3rd party software """
        # enabling a channel right now is very easy, just copy it in place
        channelpath = "%s/%s.list" % (self.channelsdir,channel)
        channelkey = "%s/%s.key" % (self.channelsdir,channel)
        if not os.path.exists(channelpath):
            print "WARNING: channel '%s' not found" % channelpath
            return
        #shutil.copy(channelpath,
        #            apt_pkg.Config.FindDir("Dir::Etc::sourceparts"))
        cmd = ["gksu",
               "--desktop", "/usr/share/applications/gnome-app-install.desktop",
               "--",
               "cp", channelpath,
               apt_pkg.Config.FindDir("Dir::Etc::sourceparts")]
        subprocess.call(cmd)
                
    def enableComponent(self, component):
        """ Enables a component of the current distribution
            (in a seperate file in /etc/apt/sources.list.d/$dist-$comp)
        """
        # sanity check
        if component == "":
            print "no repo found in enableRepository"
            return

        # first find the master mirror, FIXME: something we should
        # shove into aptsources.py?
        mirror = "http://archive.ubuntu.com/ubuntu"
        sources = SourcesList()
        newentry_sec = ""
        newentry_updates = ""
        for source in sources:
            if source.invalid or source.disabled:
                continue
            #print "checking: %s" % source.dist
            # check if the security updates are enabled
            # if yes add the components to the security updates
            if source.dist == ("%s-updates" % DISTRO_VERSION):
                if component in source.comps:
                    newentry_updates = ""
                else:
                    newentry_updates = "deb %s %s-updates %s\n" % (source.uri,
                                                                   DISTRO_VERSION,
                                                                   component)
            if source.dist == ("%s-security" % DISTRO_VERSION):
                #print "need to enable security as well"
                if component in source.comps:
                    newentry_sec = ""
                else:
                    newentry_sec = "deb http://security.ubuntu.com/ubuntu "\
                                   "%s-security %s\n" % (DISTRO_VERSION, 
                                                         component)
            if source.uri != mirror and is_mirror(mirror,source.uri):
                mirror = source.uri

        newentry = "# automatically added by gnome-app-install on %s\n" % \
                   datetime.today()
        newentry += "deb %s %s %s\n" % (mirror, DISTRO_VERSION, component)
        if newentry_sec != "":
            newentry += newentry_sec
        if newentry_updates != "":
            newentry += newentry_updates
        channel_dir = apt_pkg.Config.FindDir("Dir::Etc::sourceparts")
        channel_file = "%s-%s.list" % (DISTRO_VERSION, component)

        channel = tempfile.NamedTemporaryFile()
        channel.write(newentry)
        channel.flush()
        #print "copy: %s %s" % (channel.name, channel_dir+channel_file)
        cmd = ["gksu", "--desktop",
               "/usr/share/applications/gnome-app-install.desktop",
               "--",
               "install","-m","644","-o","0",
               channel.name, channel_dir+channel_file]
        #print cmd
        subprocess.call(cmd)
            
        
# FIXME: we want this back at some point    
#     def mimeSearch(self, mime_type, resultstore):
        
#         def match(row):
#             matched = []
#             patterns = row[COL_MIME]
#             if patterns is not None:
#                 for repattern in patterns:
#                     # mvo: we get a list of regexp from
#                     # pyxdg.DesktopEntry.getMimeType, but it does not
#                     # use any special pattern at all, so we use the plain
#                     # pattern (e.g. text/html, audio/mp3 here)
#                     pattern = repattern.pattern
#                     if mime_type in pattern:
#                         resultstore.append(list(row))
#         def recurse(row):
#             match(row)
#             for child in row.iterchildren():
#                 recurse(child)
#         for row in self.store:
#             recurse(row)
    
    
    # ---------------------------
    # Window management functions
    # ---------------------------
    

    def on_window_main_delete_event(self, window, event):
        if window.get_property("sensitive") == False:
            return True
        if self.menu.isChanged():
            ret = self.ignoreChanges()
            if ret == gtk.RESPONSE_APPLY:
                if not self.applyChanges(final=True):
                    return True
            elif ret == gtk.RESPONSE_CANCEL:
                return True
            elif ret == gtk.RESPONSE_CLOSE:
                self.quit()
        self.quit()

    def on_window_main_destroy_event(self, data=None):
        #if self.window_installed.get_property("visible") == False:
        #    self.quit()
        self.quit()
            
    def quit(self):
        # FIXME: GARRRR, this tempfiles should be done automatically
        #        *and* they shouldn't be created on the FS at all!
        for file in self.info_pages.values():
            os.remove(file)
        os.remove(self.intro_page)
        os.remove(self.failedSearchPage)
        gtk.main_quit()
        sys.exit(0)
            
    # -------------------
    # Info page functions
    # -------------------
    
    def generateInfoPage(self, item):
        program_name = item.name
        package_name = item.pkgname
        icon_filename = item.iconfile
        summary = item.description
        available = item.available
        component = item.component
        channel = item.channel
        desktop_environment = ""
        version = ""
        if self.cache.has_key(item.pkgname):
            version = self.cache[item.pkgname].candidateVersion
            # try to guess the used desktop environment
            for dependencies in desktop_environment_mapping:
                for dep in dependencies:
                    if self.cache.pkgDependsOn(item.pkgname, dep):
                        desktop_environment = desktop_environment_mapping[dependencies]
                        break
        data = {"name" : program_name or "",
                "summary" : summary or "",
                "bgcolor" : "white",
                "fgcolor" : "black",
                "icon" : "",
                "version" : version,
                "desktop_environment" : desktop_environment
                }

        if version != "":
            data["version"] = (_("Version: %s") % version)

        if icon_filename != None and icon_filename != "": 
            data["icon"] = '<img src=\"' + icon_filename + '\" align=\"right\" />'
        if available:
            pkg = self.cache[package_name]
            rough_desc = pkg.description.rstrip(" \n\t")
            
            # Replace URLs with links
            # Uncomment this when the gtkmozembed bug is fixed.
            if os.getuid() > 0:
                rough_desc = re.sub(r'(^|[\s.:;?\-\]\(<])' + 
                      r'((http://|www\.)[-\w;/?:@&=+$.!~*\'()%,#]+[\w/])' +
                      r'(?=$|[\s.:;?\-\[\]>\)])',
                      r'\1<a href="\2">\2</a>', rough_desc)
                    
            # so some regular expression magic on the description
            #print "\n\nAdd a newline before each bullet:\n"
            p = re.compile(r'^(\s|\t)*(\*|0|-)',re.MULTILINE)
            rough_desc = p.sub('\n*', rough_desc)
            #print rough_desc

            #print "\n\nreplace all newlines by spaces\n"
            p = re.compile(r'\n', re.MULTILINE)
            rough_desc = p.sub(" ", rough_desc)
            #print rough_desc

            #print "\n\nreplace all multiple spaces by newlines:\n"
            p = re.compile(r'\s\s+', re.MULTILINE)
            rough_desc = p.sub("\n", rough_desc)

            lines = rough_desc.split('\n')
            #print "\n\nrough: \n"
            #print rough_desc

            clean_desc = ""
            for i in range(len(lines)):
                if lines[i].split() == []:
                    continue
                first_chunk = lines[i].split()[0]
                if first_chunk == "*":
                    p = re.compile(r'\*\s*', re.MULTILINE)
                    lines[i] = p.sub("", lines[i])
                    clean_desc += "<div class=\"item\">"\
                                  + lines[i] +\
                                  "</div>"
                else:
                    clean_desc += "<div class=\"block\">"\
                                  + lines[i] +\
                                  "</div>"
            #print "\n\nclean: \n"
            #print clean_desc
            data["description"] = clean_desc
        else:
            # FIXME: strings + code sucks
            # If the app is not available show an information 
            # and a button to add the missing repository
            msg = _("\"%s\" is not installable" % program_name)
            # check if we have seen the component
            in_repo = True
            for it in self.cache._cache.FileList:
                # FIXME: we need to exclude cdroms here. the problem is
                # how to detect if a PkgFileIterator is pointing to a cdrom
                if it.Component != "" and it.Component == item.component:
                    # warn that this app is not available on this plattform
                    msg = "<div class='block'>\n"
                    msg += (_("'%s' is not available in any software "
                              "channel.") % program_name)
                    msg += "</div><div class='block'>"
                    msg += _("The application might not support "
                             "your system architecture.")
                    msg += "</div>"
                    in_repo = False
            if component != "" and in_repo == True:
                msg = "<div class=\"block\">"
                msg += (_("The '%s' component of the Ubuntu software channel "\
                         "that includes '%s' is "\
                         "not enabled.") % (component, program_name))
                msg += "</div>"
                msg += "<div class=\"block\">"
                msg += (_("If you choose to install '%s' the component will be "\
                         " enabled.") % (program_name))
                msg += "</div>"
            elif channel != "":
                msg = "<div class=\"block\">"
                msg = (_("The '%s' software channel that includes '%s' is "\
                         "not enabled.") % (channel, program_name))
                msg += "</div>"
                msg += "<div class=\"block\">"
                msg += (_("If you choose to install '%s' the channel will be"\
                         " enabled.") % (program_name))
                msg += "</div>"
            data['description'] = "<hr>%s" % msg


        # mutliple apps per pkg check
        s = ""
        if self.menu.pkg_to_app.has_key(package_name) and \
               len(self.menu.pkg_to_app[package_name]) > 1:
            s = _("This application is bundled with "
                  "the following applications: ")
            apps = self.menu.pkg_to_app[package_name]
            s += ", ".join([pkg.name for pkg in apps])
            data["multiple_info"] = "<hr>%s" % s
        else:
            data["multiple_info"] = ""
        
        template_file = open(os.path.join(self.datadir, "template.html"))
        template = template_file.read()
        template_file.close()
        for key in data.keys():
            template = template.replace('$' + key, data[key])
        
        filename = tempfile.NamedTemporaryFile().name
        info_page = open(filename, 'w')
        info_page.write(template)
        info_page.close()
        self.info_pages[program_name] = filename
        
        return filename
    
    def clearInfo(self):
        self.browser.loadUri("about:blank")

    def on_treeview_packages_row_activated(self, treeview, path, view_column):
        iter = treeview.get_model().get_iter(path)
        type = treeview.get_model().get_value(iter, COL_TYPE)
        if type == TYPE_GROUP:
            if treeview.row_expanded(path):
                treeview.collapse_row(path)
            else:
                treeview.expand_row(path, False)
        elif type == TYPE_PACKAGE:
            # FIXME: *garrr* ?!?
            # emitting the toggled signal with a tuple path doesn't work
            # this works around that bug
            path_str = str(path)[1:-1]
            parts = path_str.split(",")
            stripped = []
            for part in parts:
                stripped.append(part.strip())
            path_str = ":".join(stripped)
            self.toggle_render.emit("toggled", path_str)

    def on_treeview_categories_cursor_changed(self, treeview):
        path = treeview.get_cursor()[0]
        iter = treeview.get_model().get_iter(path)
        (type, name, item) = treeview.get_model()[iter]
        self.treeview_packages.set_model(item.applications)
        self.menu._refilter()
            
    def on_treeview_packages_cursor_changed(self, treeview):
        path = treeview.get_cursor()[0]
        iter = treeview.get_model().get_iter(path)
        
        if treeview.get_model().get_value(iter, COL_TYPE) != TYPE_PACKAGE:
            filename = self.intro_page
        else:
            (type, name, item) = treeview.get_model()[iter]
            #program_name, package_name, icon_filename, summary, available, section, channel = treeview.get_model().get(iter, COL_NAME, COL_PACKAGE, COL_ICON_FILE, COL_DESC, COL_AVAILABLE, COL_SECTION, COL_CHANNEL)
            if not self.info_pages.has_key(item.name):
                filename = self.generateInfoPage(item)
            else:
                filename = self.info_pages[item.name]
        
        #self.isInfoPage = True
        self.browser.loadUri("file://" + filename)
        #self.isInfoPage = False

    def generateFailedSearch(self):
        text = """
<html>
<head>
    <meta http-equiv="content-type" content="text/html; charset=UTF-8">
    <style type="text/css">
    body {
        background: white;
        color: black;
        font-size: 10pt;
        font-family: Sans;
    }
    </style>
</head>
<body>"""
        # FIXME: portme
        #icon_filename = self.icon("gnome-settings-default-applications", 24)[1]
        #text += '<img src=\"' + icon_filename + '\" align=\"right\" />\n'
        text += '<h2>' + _("Could not find a matching application") + '</h2>\n'
        text += '<p>'+ _("Perhaps the sought-after application "
                        "is not fully supported by Ubuntu or has "
                        "restrictions on use. "
                        "Check the 'Show unsupported applications' and "
                        "'Show proprietary applications' buttons to extend "
                        "the search.") + '<p>'
        text += '<p>' + _("To perform an advanced search "
                          "use the Synaptic package manager. Click on the "
                          "'Advanced' button to launch Synaptic.") + '</p>'
        text += "</body></html>"
        
        filename = tempfile.NamedTemporaryFile().name
        file = open(filename, 'w')
        file.write(text)
        file.flush()
        return filename
        
    def generateIntro(self):
        text = """
<html>
<head>
    <meta http-equiv="content-type" content="text/html; charset=UTF-8">
    <style type="text/css">
    body {
        background: white;
        color: black;
        font-size: 10pt;
        font-family: Sans;
    }
    </style>
</head>
<body>"""
        # FIXME: portme
        #icon_filename = self.icon("gnome-settings-default-applications", 24)[1]
        #text += '<img src=\"' + icon_filename + '\" align=\"right\" />\n'
        text += '<h2>' + _("Quick Introduction") + '</h2>\n'
        text += '<p>' + _("To install an application check the box next to the "
                          "application. Uncheck the box to remove "
                          "the application.") + '</p>'
        text += '<p>' + _("To perform advanced tasks use the "
                          "Synaptic package manager. Click on the button "
                          "'Advanced' to launch Synaptic.") + '</p>'
        text += "</body></html>"
        
        filename = tempfile.NamedTemporaryFile().name
        file = open(filename, 'w')
        file.write(text)
        file.flush()
        return filename
        

        
      

# Entry point for testing in source tree
if __name__ == '__main__':
    app = AppInstall(os.path.abspath("menu-data"),
                     os.path.abspath("data"),
                     sys.argv)
    gtk.main()
